"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceFleet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const version_1 = require("./version");
const worker_configuration_1 = require("./worker-configuration");
/**
 *  A new or imported Deadline Worker Fleet.
 */
class WorkerInstanceFleetBase extends core_1.Construct {
}
/**
 * This construct reperesents a fleet of Deadline Workers.
 *
 * The construct consists of an Auto Scaling Group (ASG) of instances using a provided AMI which has Deadline and any number
 * of render applications installed.  Whenever an instance in the ASG start it will connect Deadline to the desired render queue.
 *
 * When the worker fleet is deployed if it has been provided a HealthMonitor the Worker fleet will register itself against the Monitor
 * to ensure that the fleet remains healthy.
 *
 * Resources Deployed
 * ------------------------
 * - An EC2 Auto Scaling Group to maintain the number of instances.
 * - An Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *    in the fleet.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *    that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *    the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *    patching strategy to keep this software current with the latest security patches. Please see
 *    https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 *
 * @stability stable
 */
class WorkerInstanceFleet extends WorkerInstanceFleetBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.env = {
            account: this.stack.account,
            region: this.stack.region,
        };
        this.validateProps(props);
        // Launching the fleet with deadline workers.
        this.fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'Default', {
            vpc: props.vpc,
            instanceType: (props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE)),
            machineImage: props.workerMachineImage,
            keyName: props.keyName,
            vpcSubnets: props.vpcSubnets ? props.vpcSubnets : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            securityGroup: props.securityGroup,
            minCapacity: props.minCapacity,
            maxCapacity: props.maxCapacity,
            desiredCapacity: props.desiredCapacity,
            resourceSignalTimeout: WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT,
            healthCheck: aws_autoscaling_1.HealthCheck.elb({
                grace: WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL,
            }),
            role: props.role,
            spotPrice: (_b = props.spotPrice) === null || _b === void 0 ? void 0 : _b.toString(),
            blockDevices: props.blockDevices,
        });
        this.targetCapacity = parseInt(this.fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = this.fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensions: {
                AutoScalingGroupName: this.fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [this.fleet.autoScalingGroupArn],
                })],
        });
        this.fleet.node.defaultChild.metricsCollection = [{
                granularity: '1Minute',
                metrics: ['GroupDesiredCapacity'],
            }];
        this.grantPrincipal = this.fleet.grantPrincipal;
        this.connections = this.fleet.connections;
        // Configure the health monitoring if provided.
        // Note: This must be done *BEFORE* configuring the worker. We rely on the worker configuration
        // script restarting the launcher.
        this.configureHealthMonitor(props);
        const workerConfig = new worker_configuration_1.WorkerInstanceConfiguration(this, id, {
            worker: this.fleet,
            cloudWatchLogSettings: {
                logGroupPrefix: WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX,
                ...props.logGroupProps,
            },
            renderQueue: props.renderQueue,
            workerSettings: props,
            userDataProvider: props.userDataProvider,
        });
        this.listeningPorts = aws_ec2_1.Port.tcpRange(workerConfig.listenerPort, workerConfig.listenerPort + WorkerInstanceFleet.MAX_WORKERS_PER_HOST);
        // Updating the user data with successful cfn-signal commands.
        this.fleet.userData.addSignalOnExitCommand(this.fleet);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Add the security group to all workers.
     *
     * @param securityGroup : The security group to add.
     * @stability stable
     */
    addSecurityGroup(securityGroup) {
        this.fleet.addSecurityGroup(securityGroup);
    }
    /**
     * Allow access to the worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that depends on this stack.
     *
     * If this stack depends on the other stack, use allowListenerPortTo().
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowListenerPortFrom(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowListenerPortFrom(Peer.ipv4('10.0.0.0/24').connections)`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowListenerPortFrom(other) {
        this.connections.allowFrom(other.connections, this.listeningPorts, 'Worker remote command listening port');
    }
    /**
     * Allow access to the worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that this stack depends on.
     *
     * If the other stack depends on this stack, use allowListenerPortFrom().
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowListenerPortTo(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowListenerPortTo(Peer.ipv4('10.0.0.0/24').connections)`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowListenerPortTo(other) {
        other.connections.allowTo(this.connections, this.listeningPorts, 'Worker remote command listening port');
    }
    validateProps(props) {
        this.validateSpotPrice(props.spotPrice);
        this.validateArrayGroupsPoolsSyntax(props.groups, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'groups');
        this.validateArrayGroupsPoolsSyntax(props.pools, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'pools');
        this.validateRegion(props.region, /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i);
        this.validateBlockDevices(props.blockDevices);
    }
    validateSpotPrice(spotPrice) {
        if (spotPrice && !(spotPrice >= WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT && spotPrice <= WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT)) {
            throw new Error(`Invalid value: ${spotPrice} for property 'spotPrice'. Valid values can be any decimal between ${WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT} and ${WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT}.`);
        }
    }
    validateRegion(region, regex) {
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property 'region'. Valid characters are A-Z, a-z, 0-9, - and _. ‘All’, ‘none’ and ‘unrecognized’ are reserved names that cannot be used.`);
        }
    }
    validateArrayGroupsPoolsSyntax(array, regex, property) {
        if (array) {
            array.forEach(value => {
                if (!regex.test(value)) {
                    throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
                }
            });
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            this.node.addWarning(`The worker-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    this.node.addWarning(`The BlockDevice "${device.deviceName}" on the worker-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
    configureHealthMonitor(props) {
        var _b, _c;
        if (props.healthMonitor) {
            const healthCheckPort = (_c = (_b = props.healthCheckConfig) === null || _b === void 0 ? void 0 : _b.port) !== null && _c !== void 0 ? _c : core_2.HealthMonitor.DEFAULT_HEALTH_CHECK_PORT;
            const configureHealthMonitorScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'WorkerConfigurationScript', {
                osType: this.fleet.osType,
                baseName: 'configureWorkerHealthCheck',
                rootDir: path.join(__dirname, '..', 'scripts/'),
            });
            configureHealthMonitorScriptAsset.executeOn({
                host: this.fleet,
                args: [
                    `'${healthCheckPort}'`,
                    `'${version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`,
                ],
            });
            props.healthMonitor.registerFleet(this, props.healthCheckConfig || {
                port: healthCheckPort,
            });
        }
        else {
            this.node.addWarning(`The worker-fleet ${this.node.id} is being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy.`);
        }
    }
}
exports.WorkerInstanceFleet = WorkerInstanceFleet;
_a = JSII_RTTI_SYMBOL_1;
WorkerInstanceFleet[_a] = { fqn: "aws-rfdk.deadline.WorkerInstanceFleet", version: "0.28.0" };
/**
 * The min limit for spot price.
 *
 * @stability stable
 */
WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT = 0.001;
/**
 * The max limit for spot price.
 *
 * @stability stable
 */
WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT = 255;
/**
 * This determines worker's health based on any hardware or software issues of EC2 instance.
 * Resource Tracker does deep ping every 5 minutes. These checks should be more frequent so
 * that any EC2 level issues are identified ASAP. Hence setting it to 1 minute.
 */
WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL = core_1.Duration.minutes(1);
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * This is the current maximum for number of workers that can be started on a single host. Currently the
 * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
 * but only the first 8 will have their ports opened in the workers' security group.
 */
WorkerInstanceFleet.MAX_WORKERS_PER_HOST = 8;
/**
 * Setting the default signal timeout to 15 min. This is the max time, a single instance is expected
 * to take for launch and execute the user-data for deadline worker configuration. As we are setting
 * failure signals in the user-data, any failure will terminate deployment immediately.
 */
WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT = core_1.Duration.minutes(15);
//# sourceMappingURL=data:application/json;base64,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