"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueue = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_ecs_patterns_1 = require("@aws-cdk/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const rq_connection_1 = require("./rq-connection");
const wait_for_stable_service_1 = require("./wait-for-stable-service");
/**
 * Base class for Render Queue providers
 */
class RenderQueueBase extends core_1.Construct {
}
/**
 * The RenderQueue construct deploys an Elastic Container Service (ECS) service that serves Deadline's REST HTTP API to Deadline Clients.
 *
 * Most Deadline clients will connect to a Deadline render farm via the the RenderQueue. The API provides Deadline
 * clients access to Deadline's database and repository file-system in a way that is secure, performant, and scalable.
 *
 * Resources Deployed
 * ------------------------
 * - An Amazon Elastic Container Service (ECS) cluster.
 * - An AWS EC2 auto-scaling group that provides the instances that host the ECS service.
 * - An ECS service with a task definition that deploys the Deadline Remote Connetion Server (RCS) in a container.
 * - A Amazon CloudWatch log group for streaming logs from the Deadline RCS.
 * - An application load balancer, listener and target group that balance incoming traffic among the RCS containers.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Care must be taken to secure what can connect to the RenderQueue. The RenderQueue does not authenticate API
 *    requests made against it. You must limit access to the RenderQueue endpoint to only trusted hosts. Those hosts
 *    should be governed carefully, as malicious software could use the API to remotely execute code across the entire render farm.
 * - The RenderQueue can be deployed with network encryption through Transport Layer Security (TLS) or without it. Unencrypted
 *    network communications can be eavesdropped upon or modified in transit. We strongly recommend deploying the RenderQueue
 *    with TLS enabled in production environments.
 *
 * @stability stable
 */
class RenderQueue extends RenderQueueBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
        super(scope, id);
        /**
         * Whether SEP policies have been added
         */
        this.haveAddedSEPPolicies = false;
        /**
         * Whether Resource Tracker policies have been added
         */
        this.haveAddedResourceTrackerPolicies = false;
        // The RCS does not currently support horizontal scaling behind a load-balancer, so we limit to at most one instance
        if (props.renderQueueSize && props.renderQueueSize.min !== undefined && props.renderQueueSize.min > 1) {
            throw new Error(`renderQueueSize.min cannot be greater than 1 - got ${props.renderQueueSize.min}`);
        }
        if (props.renderQueueSize && props.renderQueueSize.desired !== undefined && props.renderQueueSize.desired > 1) {
            throw new Error(`renderQueueSize.desired cannot be greater than 1 - got ${props.renderQueueSize.desired}`);
        }
        this.version = props === null || props === void 0 ? void 0 : props.version;
        let externalProtocol;
        if ((_b = props.trafficEncryption) === null || _b === void 0 ? void 0 : _b.externalTLS) {
            externalProtocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
            if ((props.trafficEncryption.externalTLS.acmCertificate === undefined) ===
                (props.trafficEncryption.externalTLS.rfdkCertificate === undefined)) {
                throw new Error('Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS.');
            }
            else if (props.trafficEncryption.externalTLS.rfdkCertificate) {
                if (props.trafficEncryption.externalTLS.rfdkCertificate.certChain === undefined) {
                    throw new Error('Provided rfdkCertificate does not contain a certificate chain.');
                }
                this.clientCert = new core_2.ImportedAcmCertificate(this, 'AcmCert', props.trafficEncryption.externalTLS.rfdkCertificate);
                this.certChain = props.trafficEncryption.externalTLS.rfdkCertificate.certChain;
            }
            else {
                if (props.trafficEncryption.externalTLS.acmCertificateChain === undefined) {
                    throw new Error('externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate.');
                }
                this.clientCert = props.trafficEncryption.externalTLS.acmCertificate;
                this.certChain = props.trafficEncryption.externalTLS.acmCertificateChain;
            }
        }
        else {
            externalProtocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP;
        }
        this.version = props.version;
        const internalProtocol = (_d = (_c = props.trafficEncryption) === null || _c === void 0 ? void 0 : _c.internalProtocol) !== null && _d !== void 0 ? _d : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS && !props.hostname) {
            throw new Error('A hostname must be provided when the external protocol is HTTPS');
        }
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            vpc: props.vpc,
        });
        const minCapacity = (_f = (_e = props.renderQueueSize) === null || _e === void 0 ? void 0 : _e.min) !== null && _f !== void 0 ? _f : 1;
        if (minCapacity < 1) {
            throw new Error(`renderQueueSize.min capacity must be at least 1: got ${minCapacity}`);
        }
        this.asg = this.cluster.addCapacity('RCS Capacity', {
            vpcSubnets: (_g = props.vpcSubnets) !== null && _g !== void 0 ? _g : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            instanceType: (_h = props.instanceType) !== null && _h !== void 0 ? _h : new aws_ec2_1.InstanceType('c5.large'),
            minCapacity,
            desiredCapacity: (_j = props.renderQueueSize) === null || _j === void 0 ? void 0 : _j.desired,
            maxCapacity: 1,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    // See: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-ami-storage-config.html
                    // We want the volume to be encrypted. The default AMI size is 30-GiB.
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            updateType: aws_autoscaling_1.UpdateType.ROLLING_UPDATE,
            // @ts-ignore
            securityGroup: (_k = props.securityGroups) === null || _k === void 0 ? void 0 : _k.backend,
        });
        /**
         * The ECS-optimized AMI that is defaulted to when adding capacity to a cluster does not include the awscli or unzip
         * packages as is the case with the standard Amazon Linux AMI. These are required by RFDK scripts to configure the
         * direct connection on the host container instances.
         */
        this.asg.userData.addCommands('yum install -yq awscli unzip');
        const externalPortNumber = RenderQueue.RCS_PROTO_PORTS[externalProtocol];
        const internalPortNumber = RenderQueue.RCS_PROTO_PORTS[internalProtocol];
        this.logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, {
            logGroupPrefix: '/renderfarm/',
            ...props.logGroupProps,
        });
        this.logGroup.grantWrite(this.asg);
        const taskDefinition = this.createTaskDefinition({
            image: props.images.remoteConnectionServer,
            portNumber: internalPortNumber,
            protocol: internalProtocol,
            repository: props.repository,
        });
        this.taskDefinition = taskDefinition;
        // The fully-qualified domain name to use for the ALB
        let loadBalancerFQDN;
        if (props.hostname) {
            const label = (_l = props.hostname.hostname) !== null && _l !== void 0 ? _l : 'renderqueue';
            if (props.hostname.hostname && !RenderQueue.RE_VALID_HOSTNAME.test(label)) {
                throw new Error(`Invalid RenderQueue hostname: ${label}`);
            }
            loadBalancerFQDN = `${label}.${props.hostname.zone.zoneName}`;
        }
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', {
            vpc: this.cluster.vpc,
            vpcSubnets: (_m = props.vpcSubnetsAlb) !== null && _m !== void 0 ? _m : { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true },
            internetFacing: false,
            deletionProtection: (_o = props.deletionProtection) !== null && _o !== void 0 ? _o : true,
            securityGroup: (_p = props.securityGroups) === null || _p === void 0 ? void 0 : _p.frontend,
        });
        this.pattern = new aws_ecs_patterns_1.ApplicationLoadBalancedEc2Service(this, 'AlbEc2ServicePattern', {
            certificate: this.clientCert,
            cluster: this.cluster,
            desiredCount: (_q = props.renderQueueSize) === null || _q === void 0 ? void 0 : _q.desired,
            domainZone: (_r = props.hostname) === null || _r === void 0 ? void 0 : _r.zone,
            domainName: loadBalancerFQDN,
            listenerPort: externalPortNumber,
            loadBalancer,
            protocol: externalProtocol,
            taskDefinition,
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            // This is required to ensure that the ALB listener's security group does not allow any ingress by default.
            openListener: false,
        });
        // An explicit dependency is required from the Service to the Client certificate
        // Otherwise cloud formation will try to remove the cert before the ALB using it is disposed.
        if (this.clientCert) {
            this.pattern.node.addDependency(this.clientCert);
        }
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.pattern.service.node.addDependency(this.asg);
        this.loadBalancer = this.pattern.loadBalancer;
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        this.loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        if (props.accessLogs) {
            const accessLogsBucket = props.accessLogs.destinationBucket;
            // Policies are applied according to
            // https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:PutObject'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [`${accessLogsBucket.bucketArn}/*`],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                    },
                },
            }));
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetBucketAcl'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [accessLogsBucket.bucketArn],
            }));
            this.loadBalancer.logAccessLogs(accessLogsBucket, props.accessLogs.prefix);
        }
        // Ensure tasks are run on separate container instances
        this.pattern.service.addPlacementConstraints(aws_ecs_1.PlacementConstraint.distinctInstances());
        /**
         * Uses an escape-hatch to set the target group protocol to HTTPS. We cannot configure server certificate
         * validation, but at least traffic is encrypted and terminated at the application layer.
         */
        const listener = this.loadBalancer.node.findChild('PublicListener');
        this.listener = listener;
        const targetGroup = listener.node.findChild('ECSGroup');
        const targetGroupResource = targetGroup.node.defaultChild;
        targetGroupResource.protocol = aws_elasticloadbalancingv2_1.ApplicationProtocol[internalProtocol];
        targetGroupResource.port = internalPortNumber;
        this.grantPrincipal = taskDefinition.taskRole;
        this.connections = new aws_ec2_1.Connections({
            defaultPort: aws_ec2_1.Port.tcp(externalPortNumber),
            securityGroups: this.pattern.loadBalancer.connections.securityGroups,
        });
        this.endpoint = new core_2.ConnectableApplicationEndpoint({
            address: loadBalancerFQDN !== null && loadBalancerFQDN !== void 0 ? loadBalancerFQDN : this.pattern.loadBalancer.loadBalancerDnsName,
            port: externalPortNumber,
            connections: this.connections,
            protocol: externalProtocol,
        });
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttp({
                endpoint: this.endpoint,
            });
        }
        else {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttps({
                endpoint: this.endpoint,
                caCert: this.certChain,
            });
        }
        this.ecsServiceStabilized = new wait_for_stable_service_1.WaitForStableService(this, 'WaitForStableService', {
            service: this.pattern.service,
        });
        this.node.defaultChild = taskDefinition;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        const validationErrors = [];
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof _1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                validationErrors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return validationErrors;
    }
    /**
     * Configures an ECS cluster to be able to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientECS(param) {
        param.hosts.forEach(host => this.addChildDependency(host));
        return this.rqConnection.configureClientECS(param);
    }
    /**
     * Configure an Instance/Autoscaling group to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientInstance(param) {
        this.addChildDependency(param.host);
        this.rqConnection.configureClientInstance(param);
    }
    /**
     * Adds AWS Managed Policies to the Render Queue so it is able to control Deadline's Spot Event Plugin.
     *
     * See: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html for additonal information.
     *
     * @param includeResourceTracker Whether or not the Resource tracker admin policy should also be added (Default: True).
     * @stability stable
     */
    addSEPPolicies(includeResourceTracker = true) {
        if (!this.haveAddedSEPPolicies) {
            const sepPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy');
            this.taskDefinition.taskRole.addManagedPolicy(sepPolicy);
            this.haveAddedSEPPolicies = true;
        }
        if (!this.haveAddedResourceTrackerPolicies) {
            if (includeResourceTracker) {
                const rtPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy');
                this.taskDefinition.taskRole.addManagedPolicy(rtPolicy);
                this.haveAddedResourceTrackerPolicies = true;
            }
        }
    }
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the RenderQueue has been deployed and is ready to recieve traffic.
     *
     * This can be used to ensure that the RenderQueue is fully up and serving queries before a client attempts to connect to it.
     *
     * @param child The child to make dependent upon this RenderQueue.
     * @stability stable
     */
    addChildDependency(child) {
        // Narrowly define the dependencies to reduce the probability of cycles
        // ex: cycles that involve the security group of the RenderQueue & child.
        child.node.addDependency(this.listener);
        child.node.addDependency(this.taskDefinition);
        child.node.addDependency(this.pattern.service);
        child.node.addDependency(this.ecsServiceStabilized);
    }
    /**
     * Adds security groups to the frontend of the Render Queue, which is its load balancer.
     *
     * @param securityGroups The security groups to add.
     * @stability stable
     */
    addFrontendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.loadBalancer.addSecurityGroup(securityGroup));
    }
    /**
     * Adds security groups to the backend components of the Render Queue, which consists of the AutoScalingGroup for the Deadline RCS.
     *
     * @param securityGroups The security groups to add.
     * @stability stable
     */
    addBackendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.asg.addSecurityGroup(securityGroup));
    }
    createTaskDefinition(props) {
        const { image, portNumber, protocol, repository } = props;
        const taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'RCSTask');
        // Mount the repo filesystem to RenderQueue.HOST_REPO_FS_MOUNT_PATH
        const connection = repository.configureClientECS({
            containerInstances: {
                hosts: [this.asg],
            },
            containers: {
                taskDefinition,
            },
        });
        const environment = connection.containerEnvironment;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            // Generate a self-signed X509 certificate, private key and passphrase for use by the RCS containers.
            // Note: the Application Load Balancer does not validate the certificate in any way.
            const rcsCertPem = new core_2.X509CertificatePem(this, 'TlsCaCertPem', {
                subject: {
                    cn: 'renderfarm.local',
                },
            });
            const rcsCertPkcs = new core_2.X509CertificatePkcs12(this, 'TlsRcsCertBundle', {
                sourceCertificate: rcsCertPem,
            });
            [rcsCertPem.cert, rcsCertPkcs.cert, rcsCertPkcs.passphrase].forEach(secret => {
                secret.grantRead(taskDefinition.taskRole);
            });
            environment.RCS_TLS_CA_CERT_URI = rcsCertPem.cert.secretArn;
            environment.RCS_TLS_CERT_URI = rcsCertPkcs.cert.secretArn;
            environment.RCS_TLS_CERT_PASSPHRASE_URI = rcsCertPkcs.passphrase.secretArn;
            environment.RCS_TLS_REQUIRE_CLIENT_CERT = 'no';
        }
        const containerDefinition = taskDefinition.addContainer('ContainerDefinition', {
            image,
            memoryReservationMiB: 2048,
            environment,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup: this.logGroup,
                streamPrefix: 'RCS',
            }),
        });
        containerDefinition.addMountPoints(connection.readWriteMountPoint);
        // Increase ulimits
        containerDefinition.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        containerDefinition.addPortMappings({
            containerPort: portNumber,
            hostPort: portNumber,
        });
        return taskDefinition;
    }
}
exports.RenderQueue = RenderQueue;
_a = JSII_RTTI_SYMBOL_1;
RenderQueue[_a] = { fqn: "aws-rfdk.deadline.RenderQueue", version: "0.28.0" };
/**
 * Container listening ports for each protocol.
 */
RenderQueue.RCS_PROTO_PORTS = {
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP]: 8080,
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS]: 4433,
};
/**
 * Regular expression that validates a hostname (portion in front of the subdomain).
 */
RenderQueue.RE_VALID_HOSTNAME = /^[a-z](?:[a-z0-9-]{0,61}[a-z0-9])?$/i;
//# sourceMappingURL=data:application/json;base64,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