"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigureSpotEventPlugin = exports.SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginPreJobTaskMode = exports.SpotEventPluginLoggingLevel = exports.SpotEventPluginState = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const render_queue_1 = require("./render-queue");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const version_1 = require("./version");
/**
 * How the event plug-in should respond to events.
 *
 * @stability stable
 */
var SpotEventPluginState;
(function (SpotEventPluginState) {
    SpotEventPluginState["GLOBAL_ENABLED"] = "Global Enabled";
    SpotEventPluginState["DISABLED"] = "Disabled";
})(SpotEventPluginState = exports.SpotEventPluginState || (exports.SpotEventPluginState = {}));
/**
 * Logging verbosity levels for the Spot Event Plugin.
 *
 * @stability stable
 */
var SpotEventPluginLoggingLevel;
(function (SpotEventPluginLoggingLevel) {
    SpotEventPluginLoggingLevel["STANDARD"] = "Standard";
    SpotEventPluginLoggingLevel["VERBOSE"] = "Verbose";
    SpotEventPluginLoggingLevel["DEBUG"] = "Debug";
    SpotEventPluginLoggingLevel["OFF"] = "Off";
})(SpotEventPluginLoggingLevel = exports.SpotEventPluginLoggingLevel || (exports.SpotEventPluginLoggingLevel = {}));
/**
 * How the Spot Event Plugin should handle Pre Job Tasks.
 *
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-scripts.html
 *
 * @stability stable
 */
var SpotEventPluginPreJobTaskMode;
(function (SpotEventPluginPreJobTaskMode) {
    SpotEventPluginPreJobTaskMode["CONSERVATIVE"] = "Conservative";
    SpotEventPluginPreJobTaskMode["IGNORE"] = "Ignore";
    SpotEventPluginPreJobTaskMode["NORMAL"] = "Normal";
})(SpotEventPluginPreJobTaskMode = exports.SpotEventPluginPreJobTaskMode || (exports.SpotEventPluginPreJobTaskMode = {}));
/**
 * The Worker Extra Info column to be used to display AWS Instance Status if the instance has been marked to be stopped or terminated by EC2 or Spot Event Plugin.
 *
 * See "AWS Instance Status" option at https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#event-plugin-configuration-options
 * and https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/worker-config.html#extra-info
 *
 * @stability stable
 */
var SpotEventPluginDisplayInstanceStatus;
(function (SpotEventPluginDisplayInstanceStatus) {
    SpotEventPluginDisplayInstanceStatus["DISABLED"] = "Disabled";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_0"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_1"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_2"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_3"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_4"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_5"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_6"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_7"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_8"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_9"] = "ExtraInfo0";
})(SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginDisplayInstanceStatus || (exports.SpotEventPluginDisplayInstanceStatus = {}));
/**
 * This construct configures the Deadline Spot Event Plugin to deploy and auto-scale one or more spot fleets.
 *
 * For example, to configure the Spot Event Plugin with one spot fleet:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { InstanceClass, InstanceSize, InstanceType } from '@aws-cdk/aws-ec2';
 * import { AwsThinkboxEulaAcceptance, ConfigureSpotEventPlugin, RenderQueue, Repository, SpotEventPluginFleet, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *    version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *    version,
 *    // Change this to AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA to accept the terms
 *    // of the AWS Thinkbox End User License Agreement
 *    userAwsThinkboxEulaAcceptance: AwsThinkboxEulaAcceptance.USER_REJECTS_AWS_THINKBOX_EULA,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *    vpc,
 *    version,
 * });
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *    vpc,
 *    images: images.forRenderQueue(),
 *    repository: repository,
 * });
 *
 * const fleet = new SpotEventPluginFleet(this, 'SpotEventPluginFleet', {
 *    vpc,
 *    renderQueue,
 *    deadlineGroups: ['group_name'],
 *    instanceTypes: [InstanceType.of(InstanceClass.T3, InstanceSize.LARGE)],
 *    workerMachineImage: new GenericLinuxImage({'us-west-2': 'ami-039f0c1faba28b015'}),
 *    naxCapacity: 1,
 * });
 *
 * const spotEventPluginConfig = new ConfigureSpotEventPlugin(this, 'ConfigureSpotEventPlugin', {
 *    vpc,
 *    renderQueue: renderQueue,
 *    spotFleets: [fleet],
 *    configuration: {
 *      enableResourceTracker: true,
 *    },
 * });
 * ```
 *
 * To provide this functionality, this construct will create an AWS Lambda function that is granted the ability
 * to connect to the render queue. This lambda is run automatically when you deploy or update the stack containing this construct.
 * Logs for all AWS Lambdas are automatically recorded in Amazon CloudWatch.
 *
 * This construct will configure the Spot Event Plugin, but the Spot Fleet Requests will not be created unless you:
 * - Create the Deadline Group associated with the Spot Fleet Request Configuration. See [Deadline Documentation](https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/pools-and-groups.html).
 * - Create the Deadline Pools to which the fleet Workers are added. See [Deadline Documentation](https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/pools-and-groups.html).
 * - Submit the job with the assigned Deadline Group and Deadline Pool. See [Deadline Documentation](https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-submitting.html#submitting-jobs).
 *
 * Important: Disable 'Allow Workers to Perform House Cleaning If Pulse is not Running' in the 'Configure Repository Options'
 * when using Spot Event Plugin.
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#prerequisites
 *
 * Important: Any resources created by the Spot Event Plugin will not be deleted with 'cdk destroy'.
 * Make sure that all such resources (e.g. Spot Fleet Request or Fleet Instances) are cleaned up, before destroying the stacks.
 * Disable the Spot Event Plugin by setting 'state' property to 'SpotEventPluginState.DISABLED' or via Deadline Monitor,
 * ensure you shutdown all Pulse instances and then terminate any Spot Fleet Requests in the AWS EC2 Instance Console.
 *
 * Note that this construct adds additional policies to the Render Queue's role
 * required to run the Spot Event Plugin and launch a Resource Tracker:
 *   - AWSThinkboxDeadlineSpotEventPluginAdminPolicy
 *   - AWSThinkboxDeadlineResourceTrackerAdminPolicy
 *   - A policy to pass a fleet and instance role
 *   - A policy to create tags for spot fleet requests
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the render queue, and save Spot Event Plugin configurations.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 * - An IAM Policy attached to Render Queue's Role.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda function that is created by this resource has access to both the certificates used to connect to the render queue,
 *    and the render queue port. An attacker that can find a way to modify and execute this lambda could use it to
 *    execute any requets against the render queue. You should not grant any additional actors/principals the ability to modify
 *    or execute this Lambda.
 *
 * @stability stable
 */
class ConfigureSpotEventPlugin extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2;
        super(scope, id);
        if (ConfigureSpotEventPlugin.uniqueRenderQueues.has(props.renderQueue)) {
            throw new Error('Only one ConfigureSpotEventPlugin construct is allowed per render queue.');
        }
        else {
            ConfigureSpotEventPlugin.uniqueRenderQueues.add(props.renderQueue);
        }
        if (props.renderQueue instanceof render_queue_1.RenderQueue) {
            // We do not check the patch version, so it's set to 0.
            const minimumVersion = new version_1.Version([10, 1, 12, 0]);
            if (props.renderQueue.version.isLessThan(minimumVersion)) {
                throw new Error(`Minimum supported Deadline version for ${this.constructor.name} is ` +
                    `${minimumVersion.versionString}. ` +
                    `Received: ${props.renderQueue.version.versionString}.`);
            }
            if (props.spotFleets && props.spotFleets.length !== 0) {
                // Always add Resource Tracker admin policy, even if props.configuration?.enableResourceTracker is false.
                // This improves usability, as customers won't need to add this policy manually, if they
                // enable Resource Tracker later in the Spot Event Plugin configuration (e.g., using Deadline Monitor and not RFDK).
                props.renderQueue.addSEPPolicies(true);
                const fleetRoles = props.spotFleets.map(sf => sf.fleetRole.roleArn);
                const fleetInstanceRoles = props.spotFleets.map(sf => sf.fleetInstanceRole.roleArn);
                new aws_iam_1.Policy(this, 'SpotEventPluginPolicy', {
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'iam:PassRole',
                            ],
                            resources: [...fleetRoles, ...fleetInstanceRoles],
                            conditions: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                        }),
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'ec2:CreateTags',
                            ],
                            resources: ['arn:aws:ec2:*:*:spot-fleet-request/*'],
                        }),
                    ],
                    roles: [
                        props.renderQueue.grantPrincipal,
                    ],
                });
            }
        }
        else {
            throw new Error('The provided render queue is not an instance of RenderQueue class. Some functionality is not supported.');
        }
        const region = core_1.Construct.isConstruct(props.renderQueue) ? core_1.Stack.of(props.renderQueue).region : core_1.Stack.of(this).region;
        const configurator = new aws_lambda_1.Function(this, 'Configurator', {
            vpc: props.vpc,
            vpcSubnets: (_b = props.vpcSubnets) !== null && _b !== void 0 ? _b : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            description: `Used by a ConfigureSpotEventPlugin ${this.node.addr} to perform configuration of Deadline Spot Event Plugin`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {}),
            environment: {
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'configure-spot-event-plugin.configureSEP',
            timeout: core_1.Duration.minutes(2),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        configurator.connections.allowToDefaultPort(props.renderQueue);
        (_c = props.renderQueue.certChain) === null || _c === void 0 ? void 0 : _c.grantRead(configurator.grantPrincipal);
        const pluginConfig = {
            AWSInstanceStatus: (_e = (_d = props.configuration) === null || _d === void 0 ? void 0 : _d.awsInstanceStatus) !== null && _e !== void 0 ? _e : SpotEventPluginDisplayInstanceStatus.DISABLED,
            DeleteInterruptedSlaves: (_g = (_f = props.configuration) === null || _f === void 0 ? void 0 : _f.deleteEC2SpotInterruptedWorkers) !== null && _g !== void 0 ? _g : false,
            DeleteTerminatedSlaves: (_j = (_h = props.configuration) === null || _h === void 0 ? void 0 : _h.deleteSEPTerminatedWorkers) !== null && _j !== void 0 ? _j : false,
            IdleShutdown: (_m = (_l = (_k = props.configuration) === null || _k === void 0 ? void 0 : _k.idleShutdown) === null || _l === void 0 ? void 0 : _l.toMinutes({ integral: true })) !== null && _m !== void 0 ? _m : 10,
            Logging: (_p = (_o = props.configuration) === null || _o === void 0 ? void 0 : _o.loggingLevel) !== null && _p !== void 0 ? _p : SpotEventPluginLoggingLevel.STANDARD,
            PreJobTaskMode: (_r = (_q = props.configuration) === null || _q === void 0 ? void 0 : _q.preJobTaskMode) !== null && _r !== void 0 ? _r : SpotEventPluginPreJobTaskMode.CONSERVATIVE,
            Region: (_t = (_s = props.configuration) === null || _s === void 0 ? void 0 : _s.region) !== null && _t !== void 0 ? _t : region,
            ResourceTracker: (_v = (_u = props.configuration) === null || _u === void 0 ? void 0 : _u.enableResourceTracker) !== null && _v !== void 0 ? _v : true,
            StaggerInstances: (_x = (_w = props.configuration) === null || _w === void 0 ? void 0 : _w.maximumInstancesStartedPerCycle) !== null && _x !== void 0 ? _x : 50,
            State: (_z = (_y = props.configuration) === null || _y === void 0 ? void 0 : _y.state) !== null && _z !== void 0 ? _z : SpotEventPluginState.GLOBAL_ENABLED,
            StrictHardCap: (_1 = (_0 = props.configuration) === null || _0 === void 0 ? void 0 : _0.strictHardCap) !== null && _1 !== void 0 ? _1 : false,
        };
        const spotFleetRequestConfigs = this.mergeSpotFleetRequestConfigs(props.spotFleets);
        const properties = {
            connection: {
                hostname: props.renderQueue.endpoint.hostname,
                port: props.renderQueue.endpoint.portAsString(),
                protocol: props.renderQueue.endpoint.applicationProtocol,
                caCertificateArn: (_2 = props.renderQueue.certChain) === null || _2 === void 0 ? void 0 : _2.secretArn,
            },
            spotFleetRequestConfigurations: spotFleetRequestConfigs,
            spotPluginConfigurations: pluginConfig,
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: configurator.functionArn,
            resourceType: 'Custom::RFDK_ConfigureSpotEventPlugin',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(configurator.role);
        // We need to add this dependency to avoid failures while deleting a Custom Resource:
        // 'Custom Resource failed to stabilize in expected time. If you are using the Python cfn-response module,
        // you may need to update your Lambda function code so that CloudFormation can attach the updated version.'.
        // This happens, because Route Table Associations are deleted before the Custom Resource and we
        // don't get a response from 'doDelete()'.
        // Ideally, we would only want to add dependency on 'internetConnectivityEstablished' as shown below,
        // but it seems that CDK misses dependencies on Route Table Associations in that case:
        // const { internetConnectivityEstablished } = props.vpc.selectSubnets(props.vpcSubnets);
        // resource.node.addDependency(internetConnectivityEstablished);
        resource.node.addDependency(props.vpc);
        // /* istanbul ignore next */
        // Add a dependency on the render queue to ensure that
        // it is running before we try to send requests to it.
        resource.node.addDependency(props.renderQueue);
        this.node.defaultChild = resource;
    }
    tagSpecifications(fleet, resourceType) {
        return core_1.Lazy.any({
            produce: () => {
                if (fleet.tags.hasTags()) {
                    const tagSpecification = {
                        ResourceType: resourceType,
                        Tags: fleet.tags.renderTags(),
                    };
                    return [tagSpecification];
                }
                return undefined;
            },
        });
    }
    /**
     * Construct Spot Fleet Configurations from the provided fleet.
     * Each congiguration is a mapping between one Deadline Group and one Spot Fleet Request Configuration.
     */
    generateSpotFleetRequestConfig(fleet) {
        var _b;
        const securityGroupsToken = core_1.Lazy.any({ produce: () => {
                return fleet.securityGroups.map(sg => {
                    const securityGroupId = {
                        GroupId: sg.securityGroupId,
                    };
                    return securityGroupId;
                });
            } });
        const userDataToken = core_1.Lazy.string({ produce: () => core_1.Fn.base64(fleet.userData.render()) });
        const blockDeviceMappings = (fleet.blockDevices !== undefined ?
            this.synthesizeBlockDeviceMappings(fleet.blockDevices) : undefined);
        const { subnetIds } = fleet.subnets;
        const subnetId = subnetIds.join(',');
        const instanceTagsToken = this.tagSpecifications(fleet, spot_event_plugin_fleet_ref_1.SpotFleetResourceType.INSTANCE);
        const spotFleetRequestTagsToken = this.tagSpecifications(fleet, spot_event_plugin_fleet_ref_1.SpotFleetResourceType.SPOT_FLEET_REQUEST);
        const launchSpecifications = [];
        fleet.instanceTypes.map(instanceType => {
            const launchSpecification = {
                BlockDeviceMappings: blockDeviceMappings,
                IamInstanceProfile: {
                    Arn: fleet.instanceProfile.attrArn,
                },
                ImageId: fleet.imageId,
                KeyName: fleet.keyName,
                // Need to convert from IResolvable to bypass TypeScript
                SecurityGroups: securityGroupsToken,
                SubnetId: subnetId,
                // Need to convert from IResolvable to bypass TypeScript
                TagSpecifications: instanceTagsToken,
                UserData: userDataToken,
                InstanceType: instanceType.toString(),
            };
            launchSpecifications.push(launchSpecification);
        });
        const spotFleetRequestProps = {
            AllocationStrategy: fleet.allocationStrategy,
            IamFleetRole: fleet.fleetRole.roleArn,
            LaunchSpecifications: launchSpecifications,
            ReplaceUnhealthyInstances: true,
            // In order to work with Deadline, the 'Target Capacity' of the Spot fleet Request is
            // the maximum number of Workers that Deadline will start.
            TargetCapacity: fleet.maxCapacity,
            TerminateInstancesWithExpiration: true,
            // In order to work with Deadline, Spot Fleets Requests must be set to maintain.
            Type: spot_event_plugin_fleet_ref_1.SpotFleetRequestType.MAINTAIN,
            ValidUntil: (_b = fleet.validUntil) === null || _b === void 0 ? void 0 : _b.date.toISOString(),
            // Need to convert from IResolvable to bypass TypeScript
            TagSpecifications: spotFleetRequestTagsToken,
        };
        const spotFleetRequestConfigurations = fleet.deadlineGroups.map(group => {
            const spotFleetRequestConfiguration = {
                [group]: spotFleetRequestProps,
            };
            return spotFleetRequestConfiguration;
        });
        return spotFleetRequestConfigurations;
    }
    /**
     * Synthesize an array of block device mappings from a list of block devices
     *
     * @param blockDevices list of block devices
     */
    synthesizeBlockDeviceMappings(blockDevices) {
        return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
            const { virtualName, ebsDevice: ebs } = volume;
            if (volume === aws_autoscaling_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
                return {
                    DeviceName: deviceName,
                    // To omit the device from the block device mapping, specify an empty string.
                    // See https://docs.aws.amazon.com/cli/latest/reference/ec2/request-spot-fleet.html
                    NoDevice: '',
                };
            }
            let Ebs;
            if (ebs) {
                const { iops, volumeType, volumeSize, snapshotId, deleteOnTermination } = ebs;
                Ebs = {
                    DeleteOnTermination: deleteOnTermination,
                    Iops: iops,
                    SnapshotId: snapshotId,
                    VolumeSize: volumeSize,
                    VolumeType: volumeType,
                    // encrypted is not exposed as part of ebsDeviceProps so we need to access it via [].
                    // eslint-disable-next-line dot-notation
                    Encrypted: 'encrypted' in ebs ? ebs['encrypted'] : undefined,
                };
            }
            return {
                DeviceName: deviceName,
                Ebs,
                VirtualName: virtualName,
            };
        });
    }
    mergeSpotFleetRequestConfigs(spotFleets) {
        if (!spotFleets || spotFleets.length === 0) {
            return undefined;
        }
        const fullSpotFleetRequestConfiguration = {};
        spotFleets.map(fleet => {
            const spotFleetRequestConfigurations = this.generateSpotFleetRequestConfig(fleet);
            spotFleetRequestConfigurations.map(configuration => {
                for (const [key, value] of Object.entries(configuration)) {
                    if (key in fullSpotFleetRequestConfiguration) {
                        throw new Error(`Bad Group Name: ${key}. Group names in Spot Fleet Request Configurations should be unique.`);
                    }
                    fullSpotFleetRequestConfiguration[key] = value;
                }
            });
        });
        return fullSpotFleetRequestConfiguration;
    }
}
exports.ConfigureSpotEventPlugin = ConfigureSpotEventPlugin;
_a = JSII_RTTI_SYMBOL_1;
ConfigureSpotEventPlugin[_a] = { fqn: "aws-rfdk.deadline.ConfigureSpotEventPlugin", version: "0.28.0" };
/**
 * Only one Spot Event Plugin Configuration is allowed per render queue / repository.
 */
ConfigureSpotEventPlugin.uniqueRenderQueues = new Set();
//# sourceMappingURL=data:application/json;base64,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