"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MountableBlockVolume', () => {
    let stack;
    let vpc;
    let ebsVol;
    let instance;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        ebsVol = new aws_ec2_1.Volume(stack, 'EBS', {
            availabilityZone: vpc.availabilityZones[0],
            size: core_1.Size.gibibytes(5),
        });
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
    });
    test('defaults', () => {
        // GIVEN
        const mount = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fs',
        });
        const userData = instance.userData.render();
        // THEN
        // Make sure the instance role has the correct permissions to get & run the script
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: 'ec2:DescribeVolumes',
                        Resource: '*',
                    },
                    {
                        Effect: 'Allow',
                        Action: 'ec2:AttachVolume',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':volume/',
                                        {
                                            Ref: 'EBSB2DACE72',
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ec2:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':instance/*',
                                    ],
                                ],
                            },
                        ],
                        Condition: {
                            'ForAnyValue:StringEquals': {
                                'ec2:ResourceTag/VolumeGrantAttach-a47ec0afb147979ebdf4265080024b2c': '274f1fe7e3ac0d55b2991dd00242f5ea',
                            },
                        },
                    },
                    {
                        Effect: 'Allow',
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':s3:::',
                                        {
                                            Ref: asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Bucket,
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':s3:::',
                                        {
                                            Ref: asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Bucket,
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
            },
        }));
        // Make sure we download the mountEFS script asset bundle
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy)));
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('unzip /tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}')));
        // Make sure we execute the script with the correct args
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEbsBlockVolume.sh ${Token[TOKEN.\\d+]} xfs /mnt/fs rw')));
    });
    test('assert Linux-only', () => {
        // GIVEN
        const windowsInstance = new aws_ec2_1.Instance(stack, 'WindowsInstance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_SQL_2017_STANDARD),
        });
        const mount = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        // THEN
        expect(() => {
            mount.mountToLinuxInstance(windowsInstance, {
                location: '/mnt/fs',
                permissions: lib_1.MountPermissions.READONLY,
            });
        }).toThrowError('Target instance must be Linux.');
    });
    test('readonly mount', () => {
        // GIVEN
        const mount = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fs',
            permissions: lib_1.MountPermissions.READONLY,
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEbsBlockVolume.sh ${Token[TOKEN.\\d+]} xfs /mnt/fs r')));
    });
    test('non-default filesystem type', () => {
        // GIVEN
        const mount = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
            volumeFormat: lib_1.BlockVolumeFormat.EXT4,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fs',
            permissions: lib_1.MountPermissions.READONLY,
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEbsBlockVolume.sh ${Token[TOKEN.\\d+]} ext4 /mnt/fs r')));
    });
    test('extra mount options', () => {
        // GIVEN
        const mount = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
            extraMountOptions: [
                'option1',
                'option2',
            ],
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fs',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEbsBlockVolume.sh ${Token[TOKEN.\\d+]} xfs /mnt/fs rw,option1,option2')));
    });
    test('fails if non-construct target', () => {
        // GIVEN
        const mount = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        // WHEN
        class FakeTarget {
            constructor() {
                this.connections = instance.connections;
                this.osType = instance.osType;
                this.userData = instance.userData;
                this.grantPrincipal = instance.grantPrincipal;
                this.node = instance.node;
            }
        }
        const fakeTarget = new FakeTarget();
        // THEN
        expect(() => {
            mount.mountToLinuxInstance(fakeTarget, {
                location: '/mnt/fs',
                permissions: lib_1.MountPermissions.READONLY,
            });
        }).toThrowError(/Target instance must be a construct./);
    });
    test('asset is singleton', () => {
        var _a;
        // GIVEN
        const mount1 = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        const mount2 = new lib_1.MountableBlockVolume(ebsVol, {
            blockVolume: ebsVol,
        });
        // WHEN
        mount1.mountToLinuxInstance(instance, {
            location: '/mnt/fs',
        });
        mount2.mountToLinuxInstance(instance, {
            location: '/mnt/fs',
        });
        const userData = instance.userData.render();
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        const regex = new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy), 'g');
        const matches = (_a = userData.match(regex)) !== null && _a !== void 0 ? _a : [];
        // THEN
        // The source of the asset copy should be identical from mount1 & mount2
        expect(matches).toHaveLength(2);
        expect(matches[0]).toBe(matches[1]);
    });
});
//# sourceMappingURL=data:application/json;base64,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