"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableBlockVolume = exports.BlockVolumeFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto = require("crypto");
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * Block format options for formatting a blank/new BlockVolume.
 *
 * @stability stable
 */
var BlockVolumeFormat;
(function (BlockVolumeFormat) {
    BlockVolumeFormat["EXT3"] = "ext3";
    BlockVolumeFormat["EXT4"] = "ext4";
    BlockVolumeFormat["XFS"] = "xfs";
})(BlockVolumeFormat = exports.BlockVolumeFormat || (exports.BlockVolumeFormat = {}));
/**
 * This class encapsulates scripting that can be used by an instance to mount, format, and resize an Amazon Elastic Block Store (EBS) Volume to itself when it is launched.
 *
 * The scripting is added to
 * the instance's UserData to be run when the instance is first launched.
 *
 * The script that is employed by this class will:
 * 1) Attach the volume to this instance if it is not already attached;
 * 2) Format the block volume to the filesystem format that's passed as an argument to this script but,
 *    **ONLY IF** the filesystem has no current format;
 * 3) Mount the volume to the given mount point with the given mount options; and
 * 4) Resize the filesystem on the volume if the volume is larger than the formatted filesystem size.
 *
 * Note: This does **NOT** support multiple partitions on the EBS Volume; the script will exit with a failure code
 * when it detects multiple partitions on the device. It is expected that the whole block device is a single partition.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 * @remark If using this script with an instance within an AWS Auto Scaling Group (ASG) and you resize
 * the EBS volume, then you can terminate the instance to let the ASG replace the instance and benefit
 * from the larger volume size when this script resizes the filesystem on instance launch.
 */
class MountableBlockVolume {
    /**
     * @stability stable
     */
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
    }
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target, mount) {
        var _b;
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        this.grantRequiredPermissions(target);
        const mountScriptAsset = this.mountAssetSingleton();
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScriptZip = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        const volumeFormat = (_b = this.props.volumeFormat) !== null && _b !== void 0 ? _b : BlockVolumeFormat.XFS;
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScriptZip}`, `bash ./mountEbsBlockVolume.sh ${this.props.blockVolume.volumeId} ${volumeFormat} ${mountDir} ${mountOptionsStr} ""`, 'popd', `rm -f ${mountScriptZip}`);
    }
    /**
     * Grant required permissions to the target.
     *
     * The mounting script requires two permissions:
     * 1) Permission to describe the volume
     * 2) Permission to attach the volume
     *
     * @stability stable
     */
    grantRequiredPermissions(target) {
        // Volume.grantAttachVolumeByResourceTag() requires that the target be a construct; it adds a tag to the construct.
        // So, we fail if we're given something that is not compatible.
        if (!core_1.Construct.isConstruct(target)) {
            throw new Error('Target instance must be a construct. It cannot be constructed from attributes.');
        }
        // See: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
        // Accessed July 2020
        // ec2:DescribeVolumes does not support resource or condition constraints.
        target.grantPrincipal.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ec2:DescribeVolumes',
            ],
            resources: ['*'],
        }));
        // Work-around a bug in v1.49.1 of CDK.
        // To be able to mount the *same* volume to multiple instances we must provide a tag suffix to the permission grant
        // that is unique to this particular combination of volume + mount target.
        // We can remove this, if desired, once it is fixed in upstream CDK.
        function hashUniqueIds(resources) {
            const md5 = crypto.createHash('md5');
            resources.forEach(res => md5.update(res.node.uniqueId));
            return md5.digest('hex');
        }
        this.props.blockVolume.grantAttachVolumeByResourceTag(target.grantPrincipal, [target], hashUniqueIds([target, this.props.blockVolume]));
    }
    /**
     * Fetch the Asset singleton for the Volume mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    mountAssetSingleton() {
        var _b;
        const stack = core_1.Stack.of(this.scope);
        const uuid = '01ca4aa6-d440-4f83-84d8-80a5a21fd0e3';
        const uniqueId = 'MountableBlockVolumeAsset' + uuid.replace(/[-]/g, '');
        return (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEbsBlockVolume.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableBlockVolume = MountableBlockVolume;
_a = JSII_RTTI_SYMBOL_1;
MountableBlockVolume[_a] = { fqn: "aws-rfdk.MountableBlockVolume", version: "0.28.0" };
//# sourceMappingURL=data:application/json;base64,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