"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbPostInstallSetup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
/**
 * This construct performs post-installation setup on a MongoDB database by logging into the database, and executing commands against it.
 *
 * To provide this functionality, this construct will create an AWS Lambda function
 * that is granted the ability to connect to the given MongoDB using its administrator credentials. This lambda
 * is run automatically when you deploy or update the stack containing this construct. Logs for all AWS Lambdas are
 * automatically recorded in Amazon CloudWatch.
 *
 * Presently, the only post-installation action that this construct can perform is creating users. There are two types
 * of users that it can create:
 * 1. Password-authenticated users -- these users will be created within the 'admin' database.
 * 2. X.509-authenticated users -- these users will be created within the '$external' database.
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the MongoDB application, and perform post-installation tasks.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda function that is created by this resource has access to both the MongoDB administrator credentials,
 *    and the MongoDB application port. An attacker that can find a way to modify and execute this lambda could use it to
 *    modify or read any data in the database. You should not grant any additional actors/principals the ability to modify
 *    or execute this Lambda.
 *
 * @stability stable
 */
class MongoDbPostInstallSetup extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g;
        super(scope, id);
        (_b = props.users.x509AuthUsers) === null || _b === void 0 ? void 0 : _b.forEach(user => {
            try {
                JSON.parse(user.roles);
            }
            catch (e) {
                throw new Error(`MongoDbPostInstallSetup: Could not parse JSON role for x509 user: ${user.roles}`);
            }
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lamdbaFunc = new aws_lambda_1.Function(this, 'Lambda', {
            vpc: props.vpc,
            vpcSubnets: (_c = props.vpcSubnets) !== null && _c !== void 0 ? _c : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            description: `Used by a MongoDbPostInstallSetup ${this.node.uniqueId} to perform post-installation setup on a MongoDB`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {
            // Exclude commented out, for now, as a work-around for a CDK bug with at least CDK v1.49.1.
            // If we exclude files, then the asset hash is not calculated correctly and can result in updates to these
            // files not being picked up by the live system.
            // exclude: [
            //   '**/*',
            //   '!mongodb', '!mongodb/*',
            //   '!lib',
            //   '!lib/custom-resource', '!lib/custom-resource/*',
            //   '!lib/aws-lambda', '!lib/aws-lambda/*',
            //   '!lib/secrets-manager', '!lib/secrets-manager/*',
            //   '**/test',
            // ],
            }),
            environment: {
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'mongodb.configureMongo',
            layers: [openSslLayer],
            timeout: core_1.Duration.minutes(2),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        lamdbaFunc.connections.allowTo(props.mongoDb, aws_ec2_1.Port.tcp(props.mongoDb.port));
        props.mongoDb.certificateChain.grantRead(lamdbaFunc.grantPrincipal);
        props.mongoDb.adminUser.grantRead(lamdbaFunc.grantPrincipal);
        (_d = props.users.passwordAuthUsers) === null || _d === void 0 ? void 0 : _d.forEach(secret => secret.grantRead(lamdbaFunc));
        (_e = props.users.x509AuthUsers) === null || _e === void 0 ? void 0 : _e.forEach(user => user.certificate.grantRead(lamdbaFunc));
        const properties = {
            Connection: {
                Hostname: props.mongoDb.fullHostname,
                Port: props.mongoDb.port.toString(),
                CaCertificate: props.mongoDb.certificateChain.secretArn,
                Credentials: props.mongoDb.adminUser.secretArn,
            },
            PasswordAuthUsers: (_f = props.users.passwordAuthUsers) === null || _f === void 0 ? void 0 : _f.map(secret => secret.secretArn),
            X509AuthUsers: (_g = props.users.x509AuthUsers) === null || _g === void 0 ? void 0 : _g.map(user => ({ Certificate: user.certificate.secretArn, Roles: user.roles })),
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: lamdbaFunc.functionArn,
            resourceType: 'Custom::RFDK_MongoDbPostInstallSetup',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(lamdbaFunc.role);
        /* istanbul ignore next */
        if (props.mongoDb.node.defaultChild) {
            // Add a dependency on the ASG within the StaticPrivateIpServer to ensure that
            // mongo is running before we try to login to it.
            resource.node.addDependency(props.mongoDb.node.defaultChild.node.defaultChild);
        }
        this.node.defaultChild = resource;
    }
}
exports.MongoDbPostInstallSetup = MongoDbPostInstallSetup;
_a = JSII_RTTI_SYMBOL_1;
MongoDbPostInstallSetup[_a] = { fqn: "aws-rfdk.MongoDbPostInstallSetup", version: "0.28.0" };
//# sourceMappingURL=data:application/json;base64,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