"""
This module defines a number of classes and helper functions to define and work
with commands that operate hardware devices. The goal is to be able to define /
create commands transparently from a YAML file without having to write (too much)
code.

# Definitions

**command**

a string that is sent to a device over an interface like TCP/IP or USB. This
string is generated by the get_cmd_string() method of the Command class.

The string contains format like syntax that looks like an f-string, but is
interpreted differently. See further: How to format device command strings.

**Command**

the base class for commands. This class contains the definition of the command
and provides methods to parse and check arguments. The Command can be 'called'
or 'executed' in which case a number of actions are performed based on the
provided arguments.

**CommandExecution**

this class contains all the information needed to execute a command, without
actually executing it. A CommandExecution contains the command definition and
the parameters for the execution. It is mainly served as a communication
mechanism to the control servers, i.e. the client side (Proxy) defines a
command execution and the server then executes the command.

**CommandError**

a catch-all exception for unrecoverable errors in this module

**InvalidArgumentsError**

a CommandError raised when the arguments provided are themselve invalid
or if the number of arguments is not matching expectations

The basic interface is:

```python
cmd = Command(name     = <command name>,
              cmd      = <command string>,
              response = <callable to retreive a response>,
              wait     = <callable to wait a specific time/delay>)
```

where:

* name: a name for the command, this is just needed for reporting, not used in commanding
* cmd: the command string to send or execute, see further for details
* response: send a second command to read or get a response on the 'cmd' sent
* wait: a function object that will wait for a specific duration,
  e.g. `partial(time.sleep, 10)`.

# Formatting device command strings

The `cmd` argument is a string that contains placeholders (replacement fields)
for future arguments that will be passed when calling the Command. The replacement
fields are marked with curly braces and are mandatory. When a name is provided
in the curly braces, the argument shall be provided as a keyword argument, otherwise
a positional argument is expected. In the current implementation the `cmd`
can only contain either positional arguments or keyword argument, not a mix of both.

The replacement fields may also have a format specifier to specify a precise format
for that field.

**Examples**

```python
moveAbsolute = Command(
    name = "moveAbsolute",
    cmd  = "&2 Q70=0 Q71={tx:.6f} Q72={ty:.6f} Q73={tz:.6f} "
           "Q74={rx:.6f} Q75={ry:.6f} Q76={rz:.6f} Q20=11"
)

response = moveAbsolute(1, 1, 1, 0, 0, 20)
response = moveAbsolute(tx=1, ty=1, tz=1, rx=0, ry=0, rz=20)
```

# Questions

Do we need additional hooks into this commanding?

* add a meaning to the check, what is it and what is it used for?
* add a output processor possibility. A callback function that will process the
  output value before returning it by the __call__.
* provide an execute method for the CommandExecution that executes the command
  with the saved parameters
"""
from __future__ import annotations

import functools
import inspect
import logging
import re
import typing
from collections import namedtuple
from typing import Any
from typing import Callable

from egse.decorators import deprecate
from egse.exceptions import Error
from egse.response import Success

logger = logging.getLogger(__name__)

if typing.TYPE_CHECKING:
    from egse.protocol import CommandProtocol


def stringify_function_call(function_info: dict) -> str:
    def quote(value):
        return f'"{value}"' if isinstance(value, str) else value

    description = function_info.get("description")
    if description:
        return description

    result = ""

    name = function_info.get("func_name")
    args = function_info.get("args")
    kwargs = function_info.get("kwargs")

    if name:
        result += name
    else:
        result += "unknown_function"

    result += "("

    if args:
        result += f"{args}"[1:-1]

    if kwargs:
        result += ", " if args else ""
        result += ", ".join([f"{k}={quote(v)}" for k, v in kwargs.items()])

    result += ")"

    return result


def dry_run(func: Callable) -> Callable:
    """This decorator prepares the function to handle a dry run.

    A dry run is used to check the logic of an instrument commanding script without
    actually executing the instrument commands. The commands are instead added to the
    command sequence in the global state.

    Args:
        func: the function that needs to be executed

    Returns:
        A wrapper around the given function.
    """

    @functools.wraps(func)
    def func_wrapper(self, *args, **kwargs):
        from egse.state import GlobalState  # prevent circular import

        if GlobalState.dry_run:
            if callable(func) and func.__name__ == "client_call":
                # This client_call function takes an additional argument which is the Proxy.
                # the Proxy is not part of the CommandExecution signature and shall be removed
                # FIXME: do we introduce a memory leak here by adding 'self' to this GlobalState?
                args = args[1:]
                try:
                    self.validate_arguments(*args, **kwargs)
                except CommandError as e_ce:
                    GlobalState.add_command(InvalidCommandExecution(e_ce, self, *args, **kwargs))
                else:
                    GlobalState.add_command(CommandExecution(self, *args, **kwargs))
            else:
                FunctionExecution = namedtuple("FunctionExecution", ["name", "args", "kwargs"])
                GlobalState.add_command(FunctionExecution(func.__name__, args, kwargs))
            return Success("Command execution appended to command sequence, function not executed in dry_run.")
        else:
            return func(self, *args, **kwargs)

    return func_wrapper


def parse_format_string(fstring):
    """
    Parse and decode the format string.
    """
    # Remove occurrences of {{ }} from the fstring as they are not replacement fields
    # and will occur after formatting as single braces.

    fstring = re.sub(r"{{.*}}", lambda x: "_", fstring)

    # logger.debug(f"fstring = '{fstring}', replaced {{{{.*}}}} with _")

    parts = re.findall(r"\{(.*?)\}", fstring)

    # logger.debug(f"Parts: {parts!r}, n={len(parts)}")

    tot_n_args = len(parts)
    n_args = 0
    n_kwargs = 0
    keys = []

    for part in parts:
        result = re.split(r"(:)", part)
        if result[0] == "":
            n_args += 1
        else:
            n_kwargs += 1
            keys.append(result[0])

    # If this assertion fails, there is a flaw in the algorithm above
    assert tot_n_args == n_args + n_kwargs, (
        f"Total number of arguments ({tot_n_args}) doesn't match # args ({n_args}) + # kwargs ({n_kwargs})."
    )

    if n_args > 0 and n_kwargs > 0:
        raise InvalidArgumentsError("Mixing of positional and keyword arguments is not available.")

    return tot_n_args, n_args, n_kwargs, keys


class CommandError(Error):
    """A Command Exception as a base class for this module."""


class InvalidArgumentsError(CommandError):
    """The arguments provided are invalid"""


class CommandExecution:
    """
    This class contains all the information that is needed to execute a command
    with a set of parameters/arguments. The command is however not executed
    automatically. That is the responsibility of the caller to actually execute
    the command with the given parameters.

    Developer info:
        you can see this as a partial (functools) which defines the command and
        its arguments, but doesn't execute until explicitly called. You can execute
        the command by calling the `cmd` with the given arguments:

        ```python
        ce = CommandExecution(cmd, 20.0)
        ...
        response = ce.run()
        ```
    """

    def __init__(self, cmd, *args, **kwargs):
        self._cmd = cmd
        self._args = args
        self._kwargs = kwargs

    def get_name(self):
        return self._cmd.get_name()

    def get_cmd(self):
        return self._cmd

    def get_args(self):
        return self._args

    def get_kwargs(self):
        return self._kwargs

    def run(self):
        return self._cmd(*self._args, **self._kwargs)

    def __str__(self):
        msg = f"[{self.get_cmd().__class__.__name__}] {self.get_name()}("
        for arg in self.get_args():
            msg += f"{arg}, "
        for k, v in self.get_kwargs().items():
            msg += f"{k}={v}, "
        if msg.endswith(", "):
            msg = msg[:-2]
        msg += ")"
        return msg


class InvalidCommandExecution(CommandExecution):
    """
    A invalid command execution.

    Args:
        exc: the Exception that was raised and describes the problem
        cmd: the Command object
        *args: the positional arguments that were given
        **kwargs: the keyword arguments that were given
    """

    def __init__(self, exc: Exception, cmd: type, *args: tuple, **kwargs: dict):
        super().__init__(cmd, *args, **kwargs)
        self._exc = exc

    def run(self):
        raise InvalidArgumentsError(f"The command {self.get_name()} can not be executed. Reason: {self._exc}")

    def __str__(self):
        msg = super().__str__()
        msg += f" [ERROR: {self._exc}]"
        return msg


class WaitCommand:
    def __init__(self, command, condition):
        self._command = command
        self._condition = condition

    def __call__(self):
        # .. todo:: do we need a timeout possibility here?

        while True:
            return_code = self._command()
            if self._condition(return_code):
                break

        return 0


class Command:
    """
    A Command is basically a string that is send to a device and for which the
    device returns a response.

    The command string can contain placeholders that will be filled when the
    command is 'called'.

    The arguments that are given will be filled into the formatted string.
    Arguments can be positional or keyword arguments, not both.
    """

    def __init__(self, name, cmd, response=None, wait=None, check=None, description=None, device_method=None):
        self._name = name
        self._cmd = cmd
        self._response = response
        self._wait = wait
        self._check = check
        self._description = description
        self._device_method = device_method

        tot_n_args, n_args, n_kwargs, keys = parse_format_string(cmd)

        self._tot_n_args = tot_n_args
        self._n_args = n_args
        self._n_kwargs = n_kwargs
        self._keys = keys

        self.__doc__ = self.doc_string()

    def doc_string(self):
        msg = f"usage: {self._name}(nargs={self._tot_n_args}, keys={self._keys})\n"
        msg += "       args & kwargs can be mixed (limited)"

        if self._description is not None:
            msg += "\n"
            msg += f"{self._description}\n"

        return msg

    def validate_arguments(self, *args, **kwargs):
        # Special case for commands with *args or **kwargs, we don't validate

        if self._cmd in ("*", "**"):
            return

        nargs = len(args)
        nkwargs = len(kwargs)

        if self._tot_n_args != nargs + nkwargs:
            raise InvalidArgumentsError(
                f"Expected {self._tot_n_args} arguments for command {self._name}, got {nargs + nkwargs} arguments."
            )

        if self._tot_n_args == 0:
            pass
        elif nargs and nargs == self._n_args:
            pass
        elif nkwargs and nkwargs == self._n_kwargs:
            pass
        elif nargs == self._n_kwargs and nkwargs == 0:
            pass
        else:
            raise InvalidArgumentsError(
                f"Expected {self._n_args} positional arguments and {self._n_kwargs} "
                f"keyword arguments for command {self._name}, got {nargs} positional "
                f"and {nkwargs} keyword arguments instead."
            )

    def execute(self, cmd):
        print(f'{self.__class__.__name__}({self._name}) executing "{cmd}"')
        return 0

    def get_name(self):
        return self._name

    def needs_argument(self, name):
        if name in self._keys:
            return True
        return False

    def __repr__(self):
        name = self._name if hasattr(self, "_name") else None
        return f"<{self.__class__.__name__}({name})>"

    def get_device_method(self):
        return self._device_method

    def get_device_method_name(self):
        return self._device_method.__name__

    def get_command_execution(self, *args, **kwargs):
        return CommandExecution(self, *args, **kwargs)

    def __call__(self, *args, **kwargs):
        cmd_string = self.get_cmd_string(*args, **kwargs)

        # Now execute the cmd_string
        logger.debug(f"{cmd_string = }")

        response = self.execute(cmd_string)

        if self._wait is not None:
            self._wait()

        if self._response is not None:
            response = self._response()

        if self._check is not None:
            response = self._check(response)

        return response

    def get_raw_cmd_string(self):
        return self._cmd

    def get_cmd_string(self, *args, **kwargs):
        nargs = len(args)
        nkwargs = len(kwargs)

        if self._tot_n_args != nargs + nkwargs:
            raise CommandError(
                f"Expected {self._tot_n_args} arguments for command {self._name}, got {nargs + nkwargs} arguments."
            )

        if self._tot_n_args == 0:
            cmd_string = self._cmd or self._name
        elif nargs and nargs == self._n_args:
            cmd_string = self._create_command_string_from_args(*args)
        elif nkwargs and nkwargs == self._n_kwargs:
            cmd_string = self._create_command_string_from_kwargs(**kwargs)
        elif nargs == self._n_kwargs and nkwargs == 0:
            cmd_string = self._create_command_string_from_args_with_kw(*args)
        else:
            raise CommandError(
                f"Expected {self._n_args} positional arguments and {self._n_kwargs} "
                f"keyword arguments for command {self._name}, got {nargs} positional "
                f"and {nkwargs} keyword arguments instead."
            )

        return cmd_string

    def _create_command_string_from_args(self, *args):
        full_command = self._cmd.format(*args)
        return full_command

    def _create_command_string_from_args_with_kw(self, *args):
        full_command = self._cmd.format(**{k: v for k, v in zip(self._keys, args)})
        return full_command

    def _create_command_string_from_kwargs(self, **kwargs):
        full_command = self._cmd.format(**kwargs)
        return full_command


class ClientServerCommand(Command):
    @dry_run
    def client_call(self, other: type, *args: tuple, **kwargs: dict) -> Any:
        """
        This method is called at the client side. It is used by the Proxy
        as a generic command to send a command execution to the server.

        Args:
            other: a subclass of the Proxy class
            args: arguments that will be passed on to this command when executed
            kwargs: keyword arguments that will be passed on to this command when executed

        Returns:
            the response that is returned by calling the command (at the server side).
        """

        try:
            self.validate_arguments(*args, **kwargs)
        except CommandError as e_ce:
            logger.error(str(e_ce))
            return None

        ce = CommandExecution(self, *args, **kwargs)
        rc = other.send(ce)

        # FIXME:
        #   not sure if I should do this, the Success/Failure returns from the CS should be
        #   re-designed. I have put this here for the following reason: when requesting an obsid
        #   from the CM_CS we don't want a Success message back, but we need the number!

        # if isinstance(rc, Success):
        #     logger.info(rc)
        #     rc = rc.return_code

        return rc

    def server_call(self, other: CommandProtocol, *args: tuple, **kwargs: dict) -> int:
        """
        This method is called at the server side. It is used by the CommandProtocol class in the
        ``execute`` method.

        Args:
            other: a subclass of the CommandProtocol
            args: arguments are passed on to the response method
            kwargs: keyword arguments are passed on to the response method

        Returns:
            0 on success and -1 on failure.
        """

        if self._response is None:
            logger.warning(f"No response defined for {other} command {self.get_name()}")
            return -1

        # Note that `_response` is of type 'function' because it was loaded from
        # a class object and therefore not bound to a class instance. The reason
        # that we could not use a bound method for `_response` is that we pass the
        # command objects back and forth from the control server to the proxy and
        # the class instances are not known at the other side.

        if self._response.__name__ == "handle_device_method":
            # call the handle_device_method of the Protocol sub-class

            logger.debug(f"Executing Command {self._response.__name__}({other!r}, {self!r}, {args}, {kwargs})")

            rc = self._response(other, self, *args, **kwargs)
        else:
            logger.debug(f"Executing Command {self._response.__name__}({other!r}, {args}, {kwargs})")

            rc = self._response(other, *args, **kwargs)

        logger.debug(f"Response is {rc}.")

        return 0


@deprecate(reason="it is a duplicate of get_method() from egse.protocol")
def get_method(parent_obj: type, method_name: str) -> Callable:
    """
    Returns a bound method from a given class instance.

    Args:
        parent_obj: the class instance that provides the method
        method_name: name of the method that is requested

    Returns:
        the method [type: method].

    Note:
        The method returned is an bound class instance method and therefore
        this method *does not* expects as its first argument the class
        instance, i.e. self, when you call this as a function.

    """
    if method_name is None or method_name == "None":
        return None

    if hasattr(parent_obj, method_name):
        method = getattr(parent_obj, method_name)
        if inspect.ismethod(method):
            return method
        logger.warning(f"{method_name} is not a method, type={type(method)}")
    else:
        logger.warning(f"{parent_obj!r} has no method called {method_name}")

    return None


@deprecate(reason="it is a duplicate of get_function() from egse.protocol")
def get_function(parent_class: type, method_name: str) -> Callable:
    """
    Returns a function (unbound method) from a given class.

    Args:
        parent_class: the class that provides the method
        method_name: name of the method that is requested

    Returns:
        the method [type: function].

    Note:
        The function returned is an unbound class instance method and
        therefore this function expects as its first argument the class
        instance, i.e. self, when you call it as a function.

    """
    if method_name is None or method_name == "None":
        return None

    if hasattr(parent_class, method_name):
        func = getattr(parent_class, method_name)
        if inspect.isfunction(func):
            return func
        logger.warning(f"{method_name} is not a function, type={type(func)}")
    else:
        logger.warning(f"{parent_class.__module__}.{parent_class.__name__} has no method called {method_name}")

    return None


def load_commands(
        protocol_class: type[CommandProtocol], command_settings: dict, command_class: type[Command],
        device_class: type
) -> dict[str, Command]:
    """
    Loads the command definitions from the given ``command_settings`` and builds an internal
    dictionary containing the command names as keys and the corresponding ``Command`` class objects
    as values.

    The purpose of this function is to gather information about the commands that will be
    transferred from the control server to the proxy. The actual response or device method
    will therefore be a function instead of an instance method (because the instance will
    not be known in the Proxy).

    The `command_settings` is usually loaded from a YAML configuration file containing the
    command definitions for the device.

    Args:
        protocol_class: the CommandProtocol or a subclass
        command_settings: a dictionary containing the command definitions for this device
        command_class: the type of command to create, a subclass of Command
        device_class: the type of the base device class from which the methods are loaded

    Returns:
        A dictionary with command names and their corresponding commands.
    """
    commands = {}

    for name in command_settings:
        command_settings_name = command_settings[name]
        if "cmd" in command_settings_name:
            cmd = command_settings_name["cmd"]
        else:
            cmd = ""

        if "description" in command_settings_name:
            description = command_settings_name["description"]
        else:
            description = None

        # The response field is the name of a function from the CommandProtocol class
        # or a subclass. This function shall send a response back to the client (Proxy). That's
        # why this field is called response.
        # By convention we like that this method name would start with `handle_` so that we can
        # make a distinction between response commands and normal methods in Protocol. Remember
        # that response methods should send a reply back to the client (which will be waiting for
        # it). If no response field is given, then the `handle_device_method` will be called.

        if "response" in command_settings_name:
            response_method = get_function(protocol_class, command_settings_name["response"])
        else:
            response_method = get_function(protocol_class, "handle_device_method")
            response_method = None

        # The device_method field is used in the `handle_device_method` to call the method on the
        # device class. That is the class that implements the DeviceInterface and is usually called
        # a Controller or a Simulator.
        #
        # If no device_name field is given, the name from the command_settings is used.

        if "device_method" in command_settings_name:
            device_method_name = command_settings_name["device_method"]
        else:
            device_method_name = name

        # check if the device_method exists in the device base class

        if device_method_name == "None":
            device_method = None
        else:
            device_method = get_function(device_class, device_method_name)

        logger.debug(
            f"Creating {command_class.__module__}.{command_class.__name__}(name='{name}', "
            f"cmd='{cmd}', response={response_method}, device_method={device_method})"
        )

        commands[name] = command_class(
            name=name,
            cmd=cmd,
            response=response_method,
            description=description,
            device_method=device_method,
        )

    return commands
