#!/bin/env python3
# -*- coding: UTF-8 -*-

# Copyright (c) 2020 growingio.com, Inc.  All Rights Reserved

import ftplib
import time

from importers.common.common_util import time_format
from importers.common.config_util import FTPConfig, BaseConfig

from collections import Counter

from importers.common.log_util import logger, my_logger
from importers.data_import.data_model import DataEvent


def check_sv_header_col_value(attr_header_list):
    """
       校验CSV/TSV格式数据头-固定列数的值
    """
    error_list = []

    required_cols = ['event', 'timestamp']

    for col in required_cols:
        if col not in attr_header_list:
            error_list.append(col)

    return error_list


def check_sv_header_col_count(attr_header_list, data_header_list):
    """
       校验CSV/TSV格式数据头-列数
    """
    if len(attr_header_list) != len(data_header_list):
        return False


def check_sv_header_col_order(attr_header_list, data_header_list):
    """
       校验CSV/TSV格式数据头-不为''的顺序
    """
    try:
        for i in range(len(attr_header_list)):
            if attr_header_list[i] != '' and attr_header_list[i] != data_header_list[i]:
                return False
    except Exception:
        return False


def check_sv_col_duplicate(data_list):
    """
       校验CSV/TSV格式数据列名是否重复
    """
    for item in Counter([i for i in data_list if i != '']).items():
        if item[1] > 1:
            return item[0]


def mkd_file():
    """
    FTP创建目录
    :target_directory: 目标目录
    :return:
    """
    ftp = ftplib.FTP()
    ftp.connect(host=FTPConfig.host, port=int(FTPConfig.port))
    ftp.login(user=FTPConfig.user, passwd=FTPConfig.password)
    target_directory = ftp.mkd('/jobs/importer/' + str(int(round(time.time() * 1000))))
    ftp.close()
    return target_directory


def put_file(file_list, target_directory):
    """
    FTP上传文件
    :param file_list: 待上传文件列表
    :param target_directory: 目标目录
    :return:
    """
    ftp_start_time = time.time()
    error_message = []
    ftp = ftplib.FTP()
    ftp.connect(host=FTPConfig.host, port=int(FTPConfig.port))
    ftp.login(user=FTPConfig.user, passwd=FTPConfig.password)
    try:
        for file in file_list:
            file_splits = file.split('/')
            simple_name = file_splits[len(file_splits) - 1]
            with open(file, 'rb') as fp:
                cmd = 'STOR %s/%s' % (target_directory, simple_name)
                ftp.storbinary(cmd, fp)
    except Exception as e:
        error_file_name = simple_name
        error_message.append((error_file_name, str(e)))
    finally:
        response = ftp.voidcmd('STAT')
        if response.startswith('500'):
            # 解析出错信息
            for error_file, msg in error_message:
                logger.error(f'上传FTP文件{error_file}出错:{msg}')
            exit(-1)
        ftp.close()
        ftp_end_time = time.time()
        ftp_cost_time = ftp_end_time - ftp_start_time
        my_logger.info("文件上传至FTP耗时:%.3f秒" % ftp_cost_time)


def delete_file(file_list, target_directory):
    ftp = ftplib.FTP()
    ftp.connect(host=FTPConfig.host, port=int(FTPConfig.port))
    ftp.login(user=FTPConfig.user, passwd=FTPConfig.password)
    for file in file_list:
        file_splits = file.split('/')
        simple_name = file_splits[len(file_splits) - 1]
        ftp.delete('%s/%s' % (target_directory, simple_name))
    ftp.close()


def extract_and_validate_data(json_data):
    error_message = ""
    # 检查 userKey 字段，如果存在且值为 $notuser，则不需要 userId
    if json_data.get('userKey') != '$notuser' and 'userId' not in json_data:
        error_message += f"缺少userId需指定\n若传主体事件,则数据需字段userKey, 且值为‘$notuser’\n"
        return None, error_message
    # 确保 event,timestamp 字段存在
    elif 'event' not in json_data or 'timestamp' not in json_data:
        error_message += "event或timestamp字段不存在\n"
        return None, error_message

        # 提取字段并创建 DataEvent 对象
    data_event = DataEvent(
        userId=json_data.get('userId', ''),
        event=json_data['event'],
        timestamp=json_data['timestamp'],
        attrs=json_data.get('attrs', {}),
        userKey=json_data.get('userKey', ''),
        eventId=json_data.get('eventId', None),
        dataSourceId=json_data.get('dataSourceId', None)
    )
    return data_event, error_message


def validate_data_event(data_event, eventStart, eventEnd, cstm_keys, cstm_attr_keys):
    error_message = ""
    normal = True

    event = data_event.event
    var_keys = cstm_keys.get(event)
    attr_all = BaseConfig.attr_all

    if event in ['$exit', '$bounce']:
        normal = False
        error_message += f"事件[{event}]为t+1离线计算生成，不支持导入\n"

    if var_keys is None:
        normal = False
        error_message += f"事件[{event}]在GIO平台未定义，请先在系统中定义\n"

    if not str(var_keys).startswith("$") and var_keys is not None:
        if hasattr(data_event, 'attrs'):
            attrs_customize_error = []
            attrs_bind_error = []
            for attr in data_event.attrs:
                if attr not in attr_all:
                    # 事件属性
                    if attr not in cstm_attr_keys:
                        attrs_customize_error.append(attr)
                    # 事件绑定属性
                    elif attr not in var_keys and attr is not None:
                        attrs_bind_error.append(attr)
            if len(attrs_customize_error) > 0 or len(attrs_bind_error) > 0:
                normal = False
                if len(attrs_customize_error) > 0:
                    error_message += f"事件属性[{attrs_customize_error}]在GIO平台未定义，请先在系统中定义\n"
                if len(attrs_bind_error) > 0:
                    error_message += f"不存在事件属性[{attrs_bind_error}]与事件[{event}]的绑定关系\n"

    elif str(var_keys).startswith("$") and var_keys is not None:
        if str(var_keys) not in ["$page", "$visit"]:
            normal = False
            error_message += f"预置事件只支持$page，$visit\n"
        else:
            if hasattr(data_event, 'attrs'):
                attrs_customize_error = []
                attrs_bind_error = []
                for attr in data_event.attrs:
                    if attr not in attr_all:
                        # 事件属性
                        if attr not in cstm_attr_keys:
                            attrs_customize_error.append(attr)
                    # 事件绑定属性
                    elif attr not in var_keys and attr is not None:
                        attrs_bind_error.append(attr)
                if len(attrs_customize_error) > 0 or len(attrs_bind_error) > 0:
                    normal = False
                    if len(attrs_customize_error) > 0:
                        error_message += f"事件属性[{attrs_customize_error}]在GIO平台未定义，请先在系统中定义\n"
                    if len(attrs_bind_error) > 0:
                        error_message += f"不存在事件属性[{attrs_bind_error}]与事件[{event}]的绑定关系\n"

    if not hasattr(data_event, 'timestamp'):
        normal = False
        error_message += "缺少timestamp需指定\n"
    else:
        timestamp = 0
        try:
            timestamp = time_format(str(data_event.timestamp), BaseConfig.timezone)
        except Exception:
            normal = False
            error_message += "timestamp格式错误，请参考数据导入帮助文档\n"

        if (timestamp < eventStart or timestamp > eventEnd) and timestamp != 0:
            normal = False
            error_message += "timestamp时间范围不合法\n"

    return normal, error_message
