#!/usr/bin/env python3

import argparse
import sys
import pyfiglet

from pentest_tls import port_scanner

help_text = """
PORT_SCANNER(1)                   Network Tools                   PORT_SCANNER(1)

DESCRIPTION
    Scans a target IP for open TCP ports. Supports single ports, ranges, and 
    comma-separated lists. Defaults to scanning top 1000 common ports if no 
    ports are specified.
"""

def main():
    banner = pyfiglet.figlet_format("Port Scanner")
    print(banner)

    parser = argparse.ArgumentParser(description="Lightweight TCP port scanner")
    parser.add_argument("ip", help="Target IP address")
    parser.add_argument("-p", "--ports", help="Port(s) to scan. E.g. 22,80 or 1-1000")
    args = parser.parse_args()

    if not port_scanner.is_valid_ipv4(args.ip):
        print("FormatError: Invalid IPv4 address.")
        sys.exit(1)

    try:
        if args.ports:
            ports = port_scanner.parse_ports(args.ports)
        else:
            ports = port_scanner.load_default_ports()
    except ValueError as e:
        print(f"FormatError: {e}")
        sys.exit(1)

    open_ports = []

    for port in sorted(ports):
        sys.stdout.flush()
        if port_scanner.scan_port(args.ip, port):
            open_ports.append(port)
            print(f"[+] {args.ip}:{port}     OPEN")
        else:
            print(f"[-] {args.ip}:{port}     CLOSED/FILTERED")

    if open_ports:
        print(f">>> Opened: {sorted(open_ports)}")
    else:
        print(">>> No open ports were detected.")

if __name__ == "__main__":
    main()
