# AUTOGENERATED! DO NOT EDIT! File to edit: 00_scrapers.ipynb (unless otherwise specified).

__all__ = ['make_session', 'make_browser', 's', 'browsers', 'cache_db', 'RS_URLS', 'ITEM_FORM', 'KW_OPTIONS', 'FORMATS',
           'LOCATIONS', 'ACCESS', 'AGENCY_FORM', 'AGENCY_LOCATIONS', 'AGENCY_STATUS', 'SERIES_FORM', 'RSBase',
           'RSEntity', 'RSSearch', 'RSItem', 'RSItemSearch', 'RSSeries', 'RSSeriesSearch', 'RSAgency', 'RSAgencySearch']

# Cell

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
import mechanicalsoup
from bs4 import BeautifulSoup
import re
import datetime
import math
import arrow
from sqlitedict import SqliteDict
import time

# We need separate browser/sessions so that we can nest searches -- eg series search -> series_details -> item search.
# But we also want to reuse them as much as possible or we get misleading ConnectionErrors (actually due to too many open files)

def make_session():
    s = requests.Session()
    retries = Retry(total=10, backoff_factor=1, status_forcelist=[ 502, 503, 504 ])
    s.mount('https://', HTTPAdapter(max_retries=retries))
    s.mount('http://', HTTPAdapter(max_retries=retries))
    return s

def make_browser():
    s = make_session()
    browser = mechanicalsoup.StatefulBrowser(soup_config={'features': 'lxml'}, session=s)
    return browser

s = make_session()

browsers = {}
browsers['item'] = make_browser()
browsers['series'] = make_browser()
browsers['agency'] = make_browser()
browsers['item_search'] = make_browser()
browsers['series_search'] = make_browser()
browsers['agency_search'] = make_browser()

cache_db = SqliteDict('./cache_db.sqlite', autocommit=True)

RS_URLS = {
    'item': 'https://recordsearch.naa.gov.au/scripts/AutoSearch.asp?O=I&Number=',
    'series': 'https://recordsearch.naa.gov.au/scripts/AutoSearch.asp?Number=',
    'agency': 'https://recordsearch.naa.gov.au/scripts/AutoSearch.asp?Number=',
    'search_results': 'https://recordsearch.naa.gov.au/SearchNRetrieve/Interface/ListingReports/ItemsListing.aspx',
    'ns_results': 'https://recordsearch.naa.gov.au/NameSearch/Interface/ItemsListing.aspx',
    'series_results': 'https://recordsearch.naa.gov.au/SearchNRetrieve/Interface/ListingReports/SeriesListing.aspx',
    'agency_results': 'https://recordsearch.naa.gov.au/SearchNRetrieve/Interface/ListingReports/AgencyListing.aspx'
}

ITEM_FORM = {
    'kw': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbKeywords',
        'type': 'input',
    },
    'kw_options': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlUsingKeywords',
        'type': 'select'
    },
    'kw_exclude': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbExKeywords',
        'type': 'input'
    },
    'kw_exclude_options': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlUsingExKwd',
        'type': 'select'
    },
    # Set to 'on' to search in item notes
    # It's a checkbox, but uses Javascript to set text value.
    # Pretend it's a select for validation purposes.
    'search_notes': {
        'id': 'ctl00$ContentPlaceHolderSNR$cbxKwdTitleNotes',
        'type': 'select'
    },
    'series': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbSerNo',
        'type': 'input'
    },
    'series_exclude': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbExSerNo',
        'type': 'input'
    },
    'control': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbIteControlSymb',
        'type': 'input'
    },
    'control_exclude': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbExIteControlSymb',
        'type': 'input'
    },
    'item_id': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbIteBarcode',
        'type': 'input'
    },
    'date_from': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbDateFrom',
        'type': 'input'
    },
    'date_to': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbDateTo',
        'type': 'input'
    },
    # Select lists (options below)
    'formats': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlPhysFormat',
        'type': 'select'
    },
    'formats_exclude': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlExPhysFormat',
        'type': 'select'
    },
    'locations': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlLocation',
        'type': 'select'
    },
    'locations_exclude': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlExLocation',
        'type': 'select'
    },
    'access': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlAccessStatus',
        'type': 'select'
    },
    'access_exclude': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlExAccessStatus',
        'type': 'select'
    },
    # Checkbox
    'digital': {
        'id': 'ctl00$ContentPlaceHolderSNR$cbxDigitalCopies',
        'type': 'checkbox'
    }
}

KW_OPTIONS = [
    'ALL',
    'ANY',
    'EXACT'
]

FORMATS = [
    'Paper files and documents',
    'Index cards',
    'Bound volumes',
    'Cartographic records',
    'Photographs',
    'Microforms',
    'Audio-visual records',
    'Audio records',
    'Electronic records',
    '3-dimensional records',
    'Scientific specimens',
    'Textiles'
]

LOCATIONS = [
    'NAT,ACT',
    'SA',
    'AWM',
    'QLD',
    'NT',
    'TAS',
    'VIC',
    'WA',
    'NSW'
]

ACCESS = [
    'OPEN',
    'OWE',
    'CLOSED',
    'NYE'
]

AGENCY_FORM = {
    'kw': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbKeywords',
        'type': 'input'
    },
    'kw_options': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlUsingKeywords',
        'type': 'select'
    },
    'kw_exclude': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbExcludeKeywords',
        'type': 'input'
    },
    'kw_exclude_options': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlExcludeUsingKeywords',
        'type': 'select'
    },
    'function': {
        'id': 'ctl00$ContentPlaceHolderSNR$txtFunctions',
        'type': 'input'
    },
    'date_from': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbDateFrom',
        'type': 'input'
    },
    'date_to': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbDateTo',
        'type': 'input'
    },
    'locations': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlLocation',
        'type': 'select'
    },
    'locations_exclude': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlExcludeLocation',
        'type': 'select'
    },
    'agency_status': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlAgencyStatus',
        'type': 'select'
    },
    'agency_status_exclude': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlExcludeAgencyStatus',
        'type': 'select'
    }
}

AGENCY_LOCATIONS = [
    'NAT,ACT',
    'COCOS OR CHRISTMAS ISLAND',
    'NSW',
    'NT',
    'OVERSEAS',
    'PNG', # Papua New Guinea
    'QLD',
    'SA',
    'TAS',
    'VIC',
    'WA'
]

AGENCY_STATUS = [
    'DOS', # Department of State
    'HO', # Head Office
    'RO', # Regional or State Office
    'INTGOV', # Intergovernmental agency
    'COURT', #Judicial Court or Tribunal
    'LO', #Local Office
    'NONEX' #Non-Executive government agency (Courts, Parliament)
]

SERIES_FORM = {
    'kw': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbKeywords',
        'type': 'input'
    },
    'kw_options': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlUsingKeywords',
        'type': 'select'
    },
    'kw_exclude': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbExKeywords',
        'type': 'input'
    },
    'kw_exclude_options': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlUsingExKwd',
        'type': 'select'
    },
    'search_notes': {
        'id': 'ctl00$ContentPlaceHolderSNR$cbxKwdTitleNotes',
        'type': 'select'
    },
    'series_id': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbSerNo',
        'type': 'input'
    },
    'agency_recording': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbAgencyRecording',
        'type': 'input'
    },
    'agency_controlling': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbAgencyControlling',
        'type': 'input'
    },
    'date_from': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbDateFrom',
        'type': 'input'
    },
    'date_to': {
        'id': 'ctl00$ContentPlaceHolderSNR$txbDateTo',
        'type': 'input'
    },
    'formats': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlPhysFormat',
        'type': 'select'
    },
    'formats_exclude': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlExPhysFormat',
        'type': 'select'
    },
    'locations': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlLocation',
        'type': 'select'
    },
    'locations_exclude': {
        'id': 'ctl00$ContentPlaceHolderSNR$ddlExLocation',
        'type': 'select'
    },

}

# Cell

class RSBase(object):
    '''
    Base class with utility methods.
    '''

    def parse_date(self, date_string):
        '''
        Tests a date string against a list of expected formats.
        If a known format is found, the date is refromatted as an ISO date.
        '''
        date_formats = [
            ['D MMMM YYYY', 'YYYY-MM-DD'],
            ['D MMM YYYY', 'YYYY-MM-DD'],
            ['MMMM YYYY', 'YYYY-MM'],
            ['MMM YYYY', 'YYYY-MM'],
            ['YYYY', 'YYYY']
        ]
        for df in date_formats:
            try:
                iso_date = arrow.get(date_string, df[0]).format(df[1])
                return iso_date
            # No matching formats
            except (arrow.parser.ParserMatchError, ValueError):
                pass
        return

    def process_date_string(self, date_string, prefix=''):
        '''
        Takes a date range in a string and returns ISO formatted start and end dates.
        '''
        if not date_string:
            start_date = None
            end_date = None
        else:
            dates = date_string.strip().replace(u'\u2013', '-').split('-')
            if dates:
                start_date = self.parse_date(dates[0].strip())
                try:
                    end_date = self.parse_date(dates[1].strip())
                except IndexError:
                    end_date = None
        return {f'{prefix}date_str': date_string, f'{prefix}start_date': start_date, f'{prefix}end_date': end_date}

    def get_url(self, url):
        '''
        Get a RecordSearch page, setting up a session and handling the weird redirects.
        '''
        self.browser.open(url, timeout=30)
        # Instead of going directly to the page, you get this weird intermediary page.
        # We have to select and submit the form to redirect to the page we want.
        self.browser.select_form('form[name="t"]')
        return self.browser.submit_selected()

class RSEntity(RSBase):
    '''
    Base class for individual RecordSearch entities – item, series, or agency.

    Not for direct use – use the appropriate subclass instead.
    '''

    def __init__(self, identifier=None, cache=True, **kwargs):
        self.identifier = identifier
        self.cache = cache
        self.browser = browsers[self.entity_type]

    def get_entity_page(self):
        item_url = f'{RS_URLS[self.entity_type]}{self.identifier}'
        response = self.get_url(item_url)
        return response

    def refresh_cache(self):
        '''
        Delete data for this entity from the cache, then extract a fresh version from RecordSearch.
        '''
        del cache_db[f'{self.entity_type}_{self.identifier}']
        self.__init__(self.identifier, self.cache)

    def get_cell(self, label):
        '''
        RecordSearch results for individual entities are formatted in tables, with labels and values.
        This retrieves the cell immediately after the cell containing the supplied label.
        '''
        try:
            cell = (
                self.details.find(['td', 'div'], string=re.compile(label)).parent.find_next_siblings('td')[0]
            )
        except (IndexError, AttributeError):
            # Sometimes the cell labels are inside an enclosing div,
            # but sometimes not. Try again assuming no div.
            try:
                cell = (
                    self.details.find(['td', 'div'], string=re.compile(label)).find_next_siblings('td')[0]
                )
            except (IndexError, AttributeError):
                cell = None
        return cell

    def get_value(self, label):
        '''
        Get the value associated with the supplied label from the RecordSearch entity description.
        '''
        # Get the table cell containing the value
        cell = self.get_cell(label)
        try:
            # Get the text from the cell
            value = ' '.join([string for string in cell.stripped_strings])
        except AttributeError:
            value = None
        return value

    def get_note(self, label):
        cell = self.get_cell(label)
        if cell:
            note = cell.find('div', style='display: none;')
            if note is None:
                note = cell.find('div', id='notes')
                note_text = note.get_text(' ', strip=True).replace('\n', ' ')
        else:
            note_text = ''
        return note_text

    def get_details(self):
        '''
        Extract the main results table from the HTML page.
        '''
        details = self.browser.page.find('div', 'mainDetailsTable')
        return details

    def get_formatted_dates(self, label, prefix=''):
        '''
        Extract dates from a date range string, try to parse them and format as ISO dates.
        Return the original date string, the start date and the end date.
        '''
        date_str = self.get_value(label)
        if date_str:
            formatted_dates = self.process_date_string(date_str, prefix)
        else:
            formatted_dates = {f'{prefix}date_str': '', f'{prefix}start_date': None, f'{prefix}end_date': None}
        return formatted_dates

    def get_formatted_date(self, label, prefix=''):
        date_str = self.get_value(label)
        if date_str:
            formatted_date = {f'{prefix}date_str': date_str, f'{prefix}date': self.parse_date(date_str)}
        else:
            formatted_date = {f'{prefix}date_str': '', f'{prefix}date': None}
        return formatted_date

    def get_relations(self, label):
        cell = self.get_cell(label)
        relations = []
        if cell is not None:
            for relation in cell.find_all('li'):
                try:
                    date_str = relation.find('div', 'dates').string.strip()
                except AttributeError:
                    formatted_dates = {'date_str': '', 'start_date': None, 'end_date': None}
                else:
                    formatted_dates = self.process_date_string(date_str)
                details = [string for string in relation.find('div', 'linkagesInfo').stripped_strings]
                try:
                    identifier = details[0]
                    title = details[1][2:]
                except IndexError:
                    identifier = details[0]
                    title = details[0]
                relations.append({
                    'date_str': formatted_dates['date_str'],
                    'start_date': formatted_dates['start_date'],
                    'end_date': formatted_dates['end_date'],
                    'identifier': identifier,
                    'title': title
                })
                relation.decompose()
            cell.decompose()
        return relations

class RSSearch(RSBase):
    '''
    Base class for an advanced search in RecordSearch. There are different search forms for the different
    RecordSearch entities, so don't use this directly. Use one of the subclasses.
    '''

    def __init__(self, results_per_page=20, sort=None, record_detail='brief', **kwargs):
        self.browser = browsers[f'{self.entity_type}_search']
        self.page = 0
        params = locals().copy()
        params.pop('self')
        params.pop('kwargs')
        self.params = params
        self.kwargs = kwargs
        self.initialise_search()

    def get_full_details(self, results):
        full_results = []
        for result in results:
            full_results.append(self.entity(result['identifier'], cache=False).data)
            time.sleep(0.5)
        return full_results

    def get_results(self, page=None):
        '''
        Return a list of results from a search results page.

        The `page` value is incremented with each request, so you can call this method
        in a loop to retrieve the complete results set. When you reach then of the results,
        this method will return an empty list.

        Optional parameter:

        * `page` – request a specific page from the results set
        '''
        if self.total_results == '20,000+':
            return {'total_results': self.total_results, 'page': None, 'number_of_results': 0, 'results': [], 'error': 'Your search returns too many results.'}
        if page:
            self.page = page
        else:
            self.page += 1
        if self.page <= self.total_pages:
            results = self.search(**self.params, **self.kwargs)
        else:
            # No more data, so return an empty list
            results = {'total_results': self.total_results, 'page': None, 'number_of_results': 0, 'results': []}
        return results

    def process_list(self, details):
        results = []
        retrieved = arrow.now(tz='Australia/Sydney').isoformat()
        for row in details.find_all('tr')[1:]:
            record = self.process_row(row)
            record['retrieved'] = retrieved
            results.append(record)
        return results

    def process_page(self, soup, record_detail):
        # Do something
        return []

    def get_total_results(self, soup):
        total = 0
        if soup.find(id='ContentPlaceHolderSNR_lblToManyRecordsError') is not None:
            total = '20,000+'
        elif soup.find('span', attrs={'id': re.compile('lblDisplaying$')}) is not None:
            total_text = soup.find('span', attrs={'id': re.compile('lblDisplaying$')}).text
            total = int(re.search(r'of (\d+)', total_text).group(1))
        elif soup.find('span', text='Displaying 1 of 1'):
            total = 1
        return total

    def refresh_cache(self):
        '''
        Delete data for this search from the cache, then retrieve a fresh version from RecordSearch.
        '''
        cache_key = self.generate_cache_key()
        cache_key = re.sub('_page_\d+$', '', cache_key)
        for key in cache_db.keys():
            if cache_key in key:
                del cache_db[key]
        self.page = 0
        self.initialise_search()

    def generate_cache_key(self):
        '''
        Use the search parameters to generate a key to use in storing the cached results.
        '''
        params = self.params.copy()
        params.update(self.kwargs)
        search_key = '_'.join(sorted([f'{k}_{v}' for k, v in params.items() if v is not None]))
        search_key = f'{self.entity_type}_{search_key}_page_{self.page}'
        return search_key

    def search(self, results_per_page=None, sort=None, record_detail='brief', **kwargs):
        # Generate key to use with cache
        cache_key = self.generate_cache_key()
        # Try to get results from cache first
        try:
            results = cache_db[cache_key]
        except KeyError:
            # Set the number of results per page
            if results_per_page != 20:
                search_form = self.browser.select_form('#formSNRMaster')
                search_form.set('ctl00$ContentPlaceHolderSNR$ddlResultsPerPage', results_per_page)
                submit_button = self.browser.page.find(id='ContentPlaceHolderSNR_btnSearch')
                self.browser.submit_selected()
            # Apply sort
            if sort:
                r = self.browser.open(f'{self.browser.url}?sort={sort}')
            # Retrieve a specific page in the results set
            if self.page > 1:
                url = self.browser.url.split('?')[0]
                self.browser.open(f'{url}?page={self.page-1}')
            # Get item details from list of search results
            data = self.process_page(self.browser.page, record_detail)
            results = {
                'total_results': self.total_results,
                'page': self.page,
                'number_of_results': len(data),
                'results': data,
                'retrieved': arrow.now(tz='Australia/Sydney').isoformat()
            }
            cache_db[cache_key] = results
        #self.page = results['page']
        return results

    def initialise_search(self):
        '''
        Populates the search form and retrieves the total number of results.
        '''
        # Start a session
        self.get_url('https://recordsearch.naa.gov.au/scripts/Logon.asp?N=guest')
        # Find the main advanced search link
        main_link = self.browser.find_link(url_regex='AdvSearchMain.aspx')
        self.browser.follow_link(main_link)
        # Find the advanced search link for this entity
        search_link = self.browser.find_link(url_regex=self.search_page)
        self.browser.follow_link(search_link)
        # Get the submit button for the search form
        submit_button = self.browser.page.find(id='ContentPlaceHolderSNR_btnSearch')
        # Get the search form
        search_form = self.browser.select_form('#formSNRMaster')
        # Populate the search form with the supplied params
        for key, value in self.kwargs.items():
            search_form.set(self.search_params[key]['id'], value)
        # Submit the form
        search_form.choose_submit(submit_button)
        self.browser.submit_selected()
        # There's a 'search is running' page that has a form that needs to be submitted.
        running_form = self.browser.select_form('#Form1')
        self.browser.submit_selected()
        # Save the total number of results
        self.total_results = self.get_total_results(self.browser.page)
        # Calcuate the number of pages in the results set
        try:
            self.total_pages = math.ceil(self.total_results / self.params['results_per_page'])
        except TypeError:
            # More than 20,000 results
            pass


# Cell

class RSItem(RSEntity):
    '''
    Class used for extracting data about an individual item (usually a file, but can be a volume, box, photograph etc) from RecordSearch.

    You need to supply one of the following parameters:

    * `identifier` – the Item ID (aka barcode)
    * `details` – the BeautifulSoup HTML element containing the item details

    You'd only use `details` if you already have a RecordSearch page and want to extract item data from it.
    (There's an example of this in the `RSItemSearch` class.)

    The item data is obtained by accessing the item's `.data` attribute.
    '''
    entity_type = 'item'

    def __init__(self, identifier=None, cache=True, details=None):
        super(RSItem, self).__init__(identifier, cache)
        self.details = details
        if details:
            self.identifier = self.get_value('Item ID')
        self.data = self.get_item()

    def get_series(self):
        cell = self.get_cell('Series number')
        return cell.find('a').string.strip()

    def get_access_reasons(self):
        '''
        Extract the list of reasons why material has been withheld after access examination.
        '''
        cell = self.get_cell('Reason for restriction')
        reasons = []
        if cell:
            for link in cell.find_all('a'):
                reasons.append(link.string.strip())
        return reasons

    def get_digitised_pages(self):
        '''
        Returns the number of pages (images) in a digitised file.
        This is scraped from the RecordSearch digitised file viewer.
        The file viewer is outside of RecordSearch's session system, so it can be requested directly.
        '''
        url = f'https://recordsearch.naa.gov.au/SearchNRetrieve/Interface/ViewImage.aspx?B={self.identifier}'
        response = s.get(url, timeout=30)
        soup = BeautifulSoup(response.text, features='lxml')
        try:
            # The last page number from the navigation will be the total number of pages
            pages = int(soup.find('span', attrs={'id': "lblEndPage"}).string)
        except AttributeError:
            # If there's no navigation it might be a single page
            if soup.find('span', attrs={'id': "lblCitation"}):
                pages = 1
            # Or something else...
            else:
                pages = 0
        return pages

    def check_if_digitised(self):
        '''
        Check to see if the file is digitised, by looking for a link to the digital copy.
        '''
        if self.details.find(text=re.compile("View digital copy")):
            return True
        else:
            return False

    def get_item(self, date_format='iso'):
        # Try to retrieve from cache first
        try:
            item = cache_db[f'item_{self.identifier}']
        # If not in the cache and the details are not supplied, get it from RS.
        except KeyError:
            if not self.details:
                self.get_entity_page()
                self.details = self.get_details()
            if self.details:
                item = {
                    'title': self.get_value('Title'),
                    'identifier': self.identifier,
                    'series': self.get_series(),
                    'control_symbol': self.get_value('Control symbol'),
                    'note': self.get_note('Item note'),
                    'digitised_status': self.check_if_digitised(),
                    'digitised_pages': self.get_digitised_pages(),
                    'access_status': self.get_value('Access status'),
                    'access_decision_reasons': self.get_access_reasons(),
                    'location': self.get_value('Location'),
                    'retrieved': arrow.now(tz='Australia/Sydney').isoformat()
                }
                item.update(self.get_formatted_dates('Contents date range', 'contents_'))
                item.update(self.get_formatted_date('Date of decision', 'access_decision_'))
                if self.cache:
                    # Add to the cache
                    cache_db[f'item_{self.identifier}'] = item
            else:
                item = {'identifier': self.identifier, 'error': 'Item not found'}
        return item

    def __repr__(self):
        return f'NAA: {self.data["series"]}, {self.data["control_symbol"]}'

# Cell

class RSItemSearch(RSSearch):
    '''
    Search for items in RecordSearch.

    Supply any of the item search parameters as kwargs to initialise the search.

    Optional parameters:

    * `results_per_page` (default: 20)
    * `sort` (default: 1 – order by id)
    * `page` – to retrieve a specific page of results
    * `record_detail` – amount of detail to include, options are:
        * 'brief' (default) – just the info in the search results
        * `digitised` – add the number of pages if the file is digitised (slower)
        * 'full' – get the full individual record for each result (slowest)

    To access a page of results, use the `.get_results()` method.
    This method increments the results page, so you can call it in a loop
    to retrieve the complete result set.

    Useful attributes:

    * `.total_results` – the total number of results in the results set
    * `.total_pages` – the total number of result pages
    * `.kwargs` – a dict containing the supplied search parameters
    * `.params` – a dict containing the values of the optional parameters
    '''

    entity_type = 'item'
    search_params = ITEM_FORM
    search_page = 'AdvSearchItems.aspx'
    entity = RSItem

    def __init__(self, results_per_page=20, sort=9, record_detail='brief', **kwargs):
        super(RSItemSearch, self).__init__(results_per_page=results_per_page, sort=sort, record_detail=record_detail, **kwargs)

    def process_row(self, row):
        cells = row.find_all('td')
        item = {
            'series': cells[1].string.strip(),
            'control_symbol': cells[2].string.strip(),
            'title': cells[3].contents[0].string.strip(),
            'identifier': cells[6].string.strip()
        }
        access_string = cells[3].find('div', 'CombinedTitleBottomLeft').string
        item['access_status'] = re.search(r'Access status: ([\w ]+)', access_string).group(1).strip()
        location_string = cells[3].find('div', 'CombinedTitleBottomRight').string
        item['location'] = re.search(r'Location: ([\w ]+)', location_string).group(1).strip()
        date_str = cells[4].string.strip()
        item.update(self.process_date_string(date_str, 'contents_'))
        if cells[5].find('a') is not None:
            item['digitised_status'] = True
        else:
            item['digitised_status'] = False
        return item

    def get_digitised_page_counts(self, items):
        for item in items:
            if item['digitised_status'] == True:
                item['digitised_pages'] = RSItem(item['identifier']).get_digitised_pages()
            else:
                item['digitised_pages'] = 0
        return items

    def process_page(self, soup, record_detail):
        '''
        Extract item data from a search results page.
        Level of item data can be varied using the `record_detail` parameter:
            * 'brief' - just the data in the search results
            * 'digitised' - if the file is digitised, get the number of pages
            * 'full': retrieve the individual item record to get extra fields
        '''
        # There's a list of items
        if details := soup.find(id=re.compile('tblItemDetails$')):
            items = self.process_list(details)
        # There's a single item
        elif soup.find(id=re.compile('ContentPlaceHolderSNR_ucItemDetails_phDetailsView')) is not None:
            details = soup.find('div', 'mainDetailsTable')
            items = [self.entity(details=details, cache=False).data]
        # No items?
        else:
            raise Exception('No results found on page!')
        # Add number of pages in digitised files
        if record_detail == 'digitised':
            items = self.get_digitised_page_counts(items)
        # Get full item information
        if record_detail == 'full':
            items = self.get_full_details(items)
        return items


# Cell

class RSSeries(RSEntity):
    '''
    Class used for extracting data about an individual series.
    You need to supply the following parameter:

    * `identifier` – the series number, eg 'A1', 'B2455'

    Optional parameters:

    * `include_number_digitised` (boolean, default: True) –
      include the number of items in this series that have been digitised.
    * `include_access_status` (boolean, default: True) –
      include the number of items in this series in each of the access status categories.

    The series data is obtained by accessing the series' `.data` attribute.
    '''
    entity_type = 'series'

    def __init__(self, identifier=None, cache=True, details=None, include_number_digitised=True, include_access_status=True):
        super(RSSeries, self).__init__(identifier, cache)
        self.digitised = include_number_digitised
        self.access_status = include_access_status
        self.details = details
        if details:
            self.identifier = self.get_value('Series number')
        self.data = self.get_series()

    def get_number_described(self):
        described = self.get_value('Items in this series on RecordSearch')
        try:
            described_number, described_note = re.search(r'(\d+)(.*)', described).groups()
            try:
                described_number = int(described_number)
            except ValueError:
                pass
        except AttributeError:
            described_number = 0
            described_note = described
        except TypeError:
            described_number = 0
            described_note = ''
        return {'items_described': described_number, 'items_described_note': described_note.strip()}

    def get_quantity_locations(self):
        cell = self.get_cell('Quantity and location')
        locations = []
        if cell:
            for location in cell.findAll('li'):
                try:
                    quantity, location = re.search(r'(\d+\.*\d*) metres held in ([A-Z,a-z]+)', location.string).groups()
                    quantity = float(quantity)
                except AttributeError:
                    quantity = None
                    location = None
                locations.append({
                    'quantity': quantity,
                    'location': location
                })
        return locations

    def get_access_status_totals(self):
        totals = {}
        for status in ACCESS:
            totals[status] = RSItemSearch(series=self.identifier, access=status).total_results
            time.sleep(0.5)
        return totals

    def generate_cache_key(self):
        cache_key = f'series_{self.identifier}'
        if self.digitised:
            cache_key = f'{cache_key}_digitised'
        if self.access_status:
            cache_key = f'{cache_key}_access'
        return cache_key

    def refresh_cache(self):
        '''
        Delete data for this item from the cache, then extract a fresh version from RecordSearch.
        '''
        cache_key = self.generate_cache_key()
        del cache_db[cache_key]
        self.__init__(self.identifier, self.cache, self.digitised, self.access_status)

    def get_series(self):
        # Try to retrieve from cache first
        cache_key = self.generate_cache_key()
        try:
            series = cache_db[cache_key]
        # If not in the cache and the details are not supplied, get it from RS.
        except KeyError:
            if not self.details:
                self.get_entity_page()
                self.details = self.get_details()
            if self.details:
                series = {
                    'identifier': self.identifier,
                    'title': self.get_value('Title'),
                    'physical_format': self.get_value('Predominant physical format'),
                    'arrangement': self.get_value('System of arrangement/ control'),
                    'control_symbols': self.get_value('Range of control symbols'),
                    'locations': self.get_quantity_locations(),
                    'note': self.get_note('Series note'),
                    'recording_agencies': self.get_relations('recording'),
                    'controlling_agencies': self.get_relations('controlling'),
                    'previous_series': self.get_relations('Previous'),
                    'subsequent_series': self.get_relations('Subsequent'),
                    'controlling_series': self.get_relations('Controlling'),
                    'related_series': self.get_relations('Related'),
                    'availability': self.get_value('Visibility & availability indicator'),
                    'retrieved': arrow.now(tz='Australia/Sydney').isoformat()
                }
                series.update(self.get_number_described())
                series.update(self.get_formatted_dates('Contents dates', 'contents_'))
                series.update(self.get_formatted_dates('Accumulation dates', 'accumulation_'))
                if self.digitised:
                    series['items_digitised'] = RSItemSearch(series=self.identifier, digital=True).total_results
                if self.access_status:
                    series['access_status_totals'] = self.get_access_status_totals()
                if self.cache:
                    # Add to the cache
                    cache_db[cache_key] = series
            else:
                series = {'identifier': self.identifier, 'error': 'Series not found'}
        return series

    def __repr__(self):
        return f'NAA: {self.identifier}'

# Cell

class RSSeriesSearch(RSSearch):
    '''
    Search for series in RecordSearch.

    Supply any of the series search parameters as kwargs to initialise the search.

    Optional parameters:

    * `results_per_page` (default: 20)
    * `sort` (default: 1 – order by id)
    * `page` – to retrieve a specific page of results
    * `record_detail` – amount of detail to include, options are:
        * 'brief' (default) – just the info in the search results
        * 'full' – get the full individual record for each result (slow)

    To access a page of results, use the `.get_results()` method.
    This method increments the results page, so you can call it in a loop
    to retrieve the complete result set.

    Useful attributes:

    * `.total_results` – the total number of results in the results set
    * `.total_pages` – the total number of result pages
    * `.kwargs` – a dict containing the supplied search parameters
    * `.params` – a dict containing the values of the optional parameters
    '''

    entity_type = 'series'
    search_params = SERIES_FORM
    search_page = 'AdvSearchSeries.aspx'
    entity = RSSeries

    def __init__(self, results_per_page=20, sort=1, record_detail='brief', **kwargs):
        super(RSSeriesSearch, self).__init__(results_per_page=results_per_page, sort=sort, record_detail=record_detail, **kwargs)

    def get_locations(self, cell):
        locations = []
        for row in cell.stripped_strings:
            location, quantity = row.split(': ')
            locations.append({'location': location, 'quantity': quantity})
        return locations

    def get_items_described(self, cell):
        if link := cell.find('a'):
            return int(link.string.strip())
        else:
            return int(cell.string.strip())

    def process_row(self, row):
        cells = row.find_all('td')
        series = {
            'identifier': cells[1].string.strip(),
            'title': cells[2].contents[0].string.strip(),
            'locations': self.get_locations(cells[5]),
            'items_described': self.get_items_described(cells[6])
        }
        accumulated_date_str = cells[3].string
        series.update(self.process_date_string(accumulated_date_str, 'accumulation_'))
        contents_date_str = cells[4].string
        series.update(self.process_date_string(contents_date_str, 'contents_'))
        return series

    def process_page(self, soup, record_detail):
        '''
        Extract item data from a search results page.
        Level of item data can be varied using the `record_detail` parameter:
            * 'brief' - just the data in the search results
            * 'full': retrieve the individual item record to get extra fields
        '''
        # There's a list of items
        if details := soup.find(id=re.compile('tblSeriesDetails$')):
            series = self.process_list(details)
        # There's a single item
        elif soup.find(id=re.compile('ContentPlaceHolderSNR_ucSeriesDetails_phDetailsView')) is not None:
            # If there's only one item & you have a 'sort' param, RS returns an empty details page
            # So we'll remove any sort parameter & resubmit the url
            details_url = re.sub(r'(\?|&)sort=\d', '', self.browser.url)
            self.browser.open(details_url)
            details = self.browser.page.find('div', 'mainDetailsTable')
            series = [self.entity(details=details, cache=False).data]
        # No items?
        else:
            raise Exception('No results found on page!')
        # Get full record information
        if record_detail == 'full':
            series = self.get_full_details(series)
        return series

# Cell

class RSAgency(RSEntity):
    '''
    Class used for extracting data about an individual agency.
    You need to supply the following parameter:

    * `identifier` – the series number, eg 'A1', 'B2455'

    The series data is obtained by accessing the series' `.data` attribute.
    '''
    entity_type = 'agency'

    def __init__(self, identifier=None, cache=True, details=None, include_series_count=True):
        super(RSAgency, self).__init__(identifier, cache)
        self.series = include_series_count
        self.details = details
        if details:
            self.identifier = self.get_value('Agency number')
        self.data = self.get_agency()

    def get_series_count(self):
        series_count = RSSeriesSearch(agency_recording=self.identifier).total_results
        return series_count

    def get_agency(self):
        # Try to retrieve from cache first
        cache_key = f'{self.entity_type}_{self.identifier}'
        try:
            agency = cache_db[cache_key]
        # If not in the cache and the details are not supplied, get it from RS.
        except KeyError:
            if not self.details:
                self.get_entity_page()
                self.details = self.get_details()
            if self.details:
                agency = {
                    'identifier': self.identifier,
                    'title': self.get_value('Title'),
                    'agency_status': self.get_value('Agency status'),
                    'location': self.get_value('Location'),
                    'functions': self.get_relations('Function'),
                    'note': self.get_note('Agency note'),
                    'controlling_organisation': self.get_relations('Organisation controlling'),
                    'previous_agencies': self.get_relations('Previous'),
                    'subsequent_agencies': self.get_relations('Subsequent'),
                    'superior_agencies': self.get_relations('Superior'),
                    'controlled_agencies': self.get_relations('Controlled'),
                    'associated_people': self.get_relations('Persons'),
                    'retrieved': arrow.now(tz='Australia/Sydney').isoformat()
                }
                # series.update(self.get_number_described())
                agency.update(self.get_formatted_dates('Date range'))
                if self.series:
                    agency['number_of_series'] = self.get_series_count()
                if self.cache:
                    # Add to the cache
                    cache_db[cache_key] = agency
            else:
                agency = {'identifier': self.identifier, 'error': 'Agency not found'}
        return agency

    def __repr__(self):
        return f'{self.identifier}, self.data["title"]'

# Cell

class RSAgencySearch(RSSearch):
    '''
    Search for agencies in RecordSearch.

    Supply any of the agency search parameters as kwargs to initialise the search.

    Optional parameters:

    * `results_per_page` (default: 20)
    * `sort` (default: 1 – order by id)
    * `page` – to retrieve a specific page of results
    * `record_detail` – amount of detail to include, options are:
        * 'brief' (default) – just the info in the search results
        * 'full' – get the full individual record for each result (slow)

    To access a page of results, use the `.get_results()` method.
    This method increments the results page, so you can call it in a loop
    to retrieve the complete result set.

    Useful attributes:

    * `.total_results` – the total number of results in the results set
    * `.total_pages` – the total number of result pages
    * `.kwargs` – a dict containing the supplied search parameters
    * `.params` – a dict containing the values of the optional parameters
    '''

    entity_type = 'agency'
    search_params = AGENCY_FORM
    search_page = 'AdvSearchAgencies.aspx'
    entity = RSAgency

    def __init__(self, results_per_page=20, sort=1, record_detail='brief', **kwargs):
        super(RSAgencySearch, self).__init__(results_per_page=results_per_page, sort=sort, record_detail=record_detail, **kwargs)

    def process_row(self, row):
        cells = row.find_all('td')
        agency = {
            'identifier': cells[1].string.strip(),
            'title': cells[2].contents[0].string.strip()
        }
        date_str = cells[3].string
        agency.update(self.process_date_string(date_str))
        return agency

    def process_page(self, soup, record_detail):
        '''
        Extract item data from a search results page.
        Level of record data can be varied using the `record_detail` parameter.
        '''
        # There's a list of items
        if results := soup.find(id=re.compile('tblProvDetails$')):
            agencies = self.process_list(results)
        # There's a single item
        elif soup.find(id=re.compile('ucAgencyDetails_phDetailsView')) is not None:
            # If there's only one item & you have a 'sort' param, RS returns an empty details page
            # So we'll remove any sort parameter & resubmit the url
            details_url = re.sub(r'(\?|&)sort=\d', '', self.browser.url)
            self.browser.open(details_url)
            details = self.browser.page.find('div', 'mainDetailsTable')
            agencies = [self.entity(details=details, cache=False).data]
        # No items?
        else:
            raise Exception('No results found on page!')
        # Get full item information
        if record_detail == 'full':
            agencies = self.get_full_details(agencies)
        return agencies