"""
database.py
====================================
The core module for communicating with the database.
"""

from sqlitedict import SqliteDict
from collections import OrderedDict

from .log import LOGGER

class DataBase:
    """ DataBase class (singleton pattern) """

    _instance = None

    def __new__(cls, dbname, cache_size):
        if not cls._instance:
            cls._instance = super(DataBase, cls).__new__(cls)
        return cls._instance

    def __init__(self, dbname, cache_size = 10000):
        
        self.name = f'{dbname}.sqlite'
        self.db = SqliteDict(self.name, autocommit=True)
        self.connected = True
        self.cache_size = cache_size
        self.cache = OrderedDict()

    def set_value(self, key, value):
        """ 
        Sets the key and given value in the database. If the key exists,
        it will override the value. In that case, it will remove the key from
        the in-memory dictionary. It will be loaded again with the get_value
        command if needed.

        Inputs:
            | key: hash value (generated by the package)
            | value: Seismic Record object

        """
        try:
            self.db[key] = value
            del self.cache[key]
        except KeyError:
            LOGGER.debug(f"Tried to delete non-existing {key} on the cache.")
        except Exception:
            LOGGER.warning(f"Tried to set {key} on the database."
             "Something went wrong.")
        finally:
            pass 

    def delete_value(self,key):
        """ Deletes the key, and its value from both in-memory dictionary and
        on-disk database. If the key is not found, simply ignores it.

        Inputs:
            | key: hash value (generated by the package)
        """
        try:
            del self.db[key]   
            try: 
                del self.cache[key]
            except:
                pass         
            LOGGER.debug(f"Value {key} is removed from database.")
        except KeyError:
            LOGGER.warning(f"Tried to delete {key} on the database."
             "Something went wrong.")
     
    def get_value(self, key):
        """ Returns the value in the following order:
        
        | 1) It will look for the value in the cache and return it, if not found
        | 2) will look for the value in the disk and return it, if not found
        | 3) will return None.

        Inputs:
            | key: hash value (generated by the package)

        Outputs:
            | If found, value, else returns None.         
        """
        value = None
        try:
            value = self.cache[key]
            LOGGER.debug(f"Key: {key}. Value is loaded from the cache.")
        except:
            LOGGER.debug(f"Key: {key}. Value is not found in the cache.")

        if not value:
            try:
                tmp = self.db[key]
                if len(self.cache) >  self.cache_size:
                    self.cache.popitem(last=False)
                    LOGGER.debug(f"cache size is more than limit"
                     f"{self.cache_size}. An item removed, and new item added.")
                self.cache[key] = tmp
                return tmp
            except Exception:
                LOGGER.debug(f"The requested key ({key}) is not in the"
                 " database. Returns None.")
                return None
        else:
            return value

    def update_nested_container(self, key1, key2, value, append=True):
        """ Updates nested container 
        
        Inputs:
            | key1 is the key for the contaier.
            | key2 is the key inside the container.
            | value is the value of key2
        """
        try:
            tracker_container = self.db[key1]
            if key2 in list(tracker_container.keys()):
                if append:
                    tracker_container[key2].extend(value)
                else: 
                    tracker_container[key2] = value
                self.db[key1] = tracker_container
                LOGGER.debug(f"Tracker for {key2} has been set. '{value}' added.")
            else:
                tracker_container[key2] = value
                self.db[key1] = tracker_container
                LOGGER.debug(f"Tracker for {key2} is set. '{value}' added.")

        except Exception:
            pass
    
    def get_nested_container(self, key):
        """ Returns the value of the nested container, directly from database.

        Inputs:
            | key: container key

        Outputs:
            | If found, value, else returns None.         
        """
        value = None
        try:
            value = self.db[key]
            LOGGER.debug(f"Key: {key}. Container is loaded from the database.")
        except:
            LOGGER.debug(f"Key: {key}. Value is not found in the cache.")

        return value

    def remove_nested_container(self, key1, key2):
        """ Removes key2 from tracking inside key1 container.
        Not implemented.
        """ 
        pass
        
    def close_db(self):
        """ Commits changes to the database, closes the database, clears the 
        cache.
        """

        self.db.commit()
        self.db.close()
        self.cache = None
        self.connected = False
        LOGGER.info(f"Database ({self.name}) is closed.")