"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepFunctionInvokeAction = exports.StateMachineInput = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * Represents the input for the StateMachine.
 *
 * @stability stable
 */
class StateMachineInput {
    constructor(input, inputArtifact, inputType) {
        this.input = input;
        this.inputArtifact = inputArtifact;
        this.inputType = inputType;
    }
    /**
     * When the input type is FilePath, input artifact and filepath must be specified.
     *
     * @stability stable
     */
    static filePath(inputFile) {
        return new StateMachineInput(inputFile.location, inputFile.artifact, 'FilePath');
    }
    /**
     * When the input type is Literal, input value is passed directly to the state machine input.
     *
     * @stability stable
     */
    static literal(object) {
        return new StateMachineInput(JSON.stringify(object), undefined, 'Literal');
    }
}
exports.StateMachineInput = StateMachineInput;
/**
 * StepFunctionInvokeAction that is provided by an AWS CodePipeline.
 *
 * @stability stable
 */
class StepFunctionInvokeAction extends action_1.Action {
    /**
     * @stability stable
     */
    constructor(props) {
        super({
            ...props,
            resource: props.stateMachine,
            category: codepipeline.ActionCategory.INVOKE,
            provider: 'StepFunctions',
            artifactBounds: {
                minInputs: 0,
                maxInputs: 1,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs: (props.stateMachineInput && props.stateMachineInput.inputArtifact) ? [props.stateMachineInput.inputArtifact] : [],
            outputs: (props.output) ? [props.output] : [],
        });
        this.props = props;
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @stability stable
     */
    bound(_scope, _stage, options) {
        var _a, _b, _c, _d, _e;
        // allow pipeline to invoke this step function
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:StartExecution', 'states:DescribeStateMachine'],
            resources: [this.props.stateMachine.stateMachineArn],
        }));
        // allow state machine executions to be inspected
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:DescribeExecution'],
            resources: [cdk.Stack.of(this.props.stateMachine).formatArn({
                    service: 'states',
                    resource: 'execution',
                    resourceName: `${cdk.Stack.of(this.props.stateMachine).parseArn(this.props.stateMachine.stateMachineArn, ':').resourceName}:${(_a = this.props.executionNamePrefix) !== null && _a !== void 0 ? _a : ''}*`,
                    sep: ':',
                })],
        }));
        // allow the Role access to the Bucket, if there are any inputs/outputs
        if (((_b = this.actionProperties.inputs) !== null && _b !== void 0 ? _b : []).length > 0) {
            options.bucket.grantRead(options.role);
        }
        if (((_c = this.actionProperties.outputs) !== null && _c !== void 0 ? _c : []).length > 0) {
            options.bucket.grantWrite(options.role);
        }
        return {
            configuration: {
                StateMachineArn: this.props.stateMachine.stateMachineArn,
                Input: (_d = this.props.stateMachineInput) === null || _d === void 0 ? void 0 : _d.input,
                InputType: (_e = this.props.stateMachineInput) === null || _e === void 0 ? void 0 : _e.inputType,
                ExecutionNamePrefix: this.props.executionNamePrefix,
            },
        };
    }
}
exports.StepFunctionInvokeAction = StepFunctionInvokeAction;
//# sourceMappingURL=data:application/json;base64,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