from pathlib import Path

from sample_finder.sources.source import Source
from sample_finder.validators import verify_sha256


class SourceMalwareBazaar(Source):
    """
    Implements MalwareBazaar Source.

    References
    ----------
        * https://mb-api.abuse.ch/api/

    """

    NAME = "malwarebazaar"
    URL_API = "https://mb-api.abuse.ch/api/v1/"

    PASSWORD_ZIP = b"infected"

    def __init__(self, config: dict) -> None:
        """
        Construct MalwareBazaar object.

        Add the api key to the session headers.
        """
        super().__init__(config)
        self._session.headers.update({"API-KEY": self._config["api_key"]})

    def _get_sha256(self, sample_hash: str) -> str | None:
        """
        Given a sample hash, use the get_info endpoint to try and get the corresponding SHA-256 hash.

        The get_file endpoint only support SHA-256 hashes.
        If the provided hash is not an SHA-256 hash,
        we first call the get_info endpoint to get the SHA-256 hash.
        """
        if not verify_sha256(sample_hash):
            response = self._session.post(
                self.URL_API,
                data={
                    "query": "get_info",
                    "hash": sample_hash,
                },
            )

            if not response or not response.ok:
                return None

            response_json = response.json()

            if response_json["query_status"] != "ok":
                return None

            return response_json["sha256_hash"]

        return sample_hash

    def download_file(self, sample_hash: str, output_path: Path) -> bool:
        """
        Download file from Malware Bazaar.

        The get_file endpoint only support SHA-256 hashes.
        If the provided hash is not an SHA-256 hash,
        we first call the get_info endpoint to get the SHA-256 hash.
        """
        sample_sha256 = self._get_sha256(sample_hash)
        if not sample_sha256:
            return False

        response = self._post(
            self.URL_API,
            data={
                "query": "get_file",
                "sha256_hash": sample_sha256,
            },
        )
        if not response or not response.ok:
            return False

        if response.headers["content-type"] != "application/zip":
            return False

        data = self._decrypt_zip(response.content)
        with output_path.open("wb") as h_file:
            h_file.write(data)

        return True
