"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationStackNotificationCheck = exports.CloudFormationStackDriftDetectionCheck = exports.AccessKeysRotated = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const rule_1 = require("./rule");
/**
 * Checks whether the active access keys are rotated within the number of days
 * specified in `maxAge`.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 *
 * @resource AWS::Config::ConfigRule
 */
class AccessKeysRotated extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'ACCESS_KEYS_ROTATED',
            inputParameters: {
                ...props.maxAge
                    ? {
                        maxAccessKeyAge: props.maxAge.toDays(),
                    }
                    : {},
            },
        });
    }
}
exports.AccessKeysRotated = AccessKeysRotated;
/**
 * Checks whether your CloudFormation stacks' actual configuration differs, or
 * has drifted, from its expected configuration.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-drift-detection-check.html
 *
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackDriftDetectionCheck extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK',
            inputParameters: {
                cloudformationRoleArn: core_1.Lazy.stringValue({ produce: () => this.role.roleArn }),
            },
        });
        this.scopeToResource('AWS::CloudFormation::Stack', props.ownStackOnly ? core_1.Stack.of(this).stackId : undefined);
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('config.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess'),
            ],
        });
    }
}
exports.CloudFormationStackDriftDetectionCheck = CloudFormationStackDriftDetectionCheck;
/**
 * Checks whether your CloudFormation stacks are sending event notifications to
 * a SNS topic. Optionally checks whether specified SNS topics are used.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-notification-check.html
 *
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackNotificationCheck extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        if (props.topics && props.topics.length > 5) {
            throw new Error('At most 5 topics can be specified.');
        }
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_NOTIFICATION_CHECK',
            inputParameters: props.topics && props.topics.reduce((params, topic, idx) => ({ ...params, [`snsTopic${idx + 1}`]: topic.topicArn }), {}),
        });
        this.scopeToResource('AWS::CloudFormation::Stack');
    }
}
exports.CloudFormationStackNotificationCheck = CloudFormationStackNotificationCheck;
//# sourceMappingURL=data:application/json;base64,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