from starkcore.utils.api import from_api_json
from starkcore.utils.checks import check_datetime, check_date
from starkcore.utils.resource import Resource
from ...utils import rest
from ..__issuingembossingrequest import _resource as _issuing_embossing_request_resource
from starkcore.error import Error


class Log(Resource):
    """# issuingembossingrequest.Log object
    Every time an IssuingEmbossingRequest entity is updated, a corresponding issuingembossingrequest.Log
    is generated for the entity. This log is never generated by the
    user, but it can be retrieved to check additional information
    on the IssuingEmbossingRequest.
    ## Attributes:
    - id [string]: unique id returned when the log is created. ex: "5656565656565656"
    - request [IssuingEmbossingRequest]: IssuingEmbossingRequest entity to which the log refers to.
    - errors [list of StarkCore.Error]: list of errors linked to this IssuingEmbossingRequest event.
    - type [string]: type of the IssuingEmbossingRequest event which triggered the log creation. ex: "created", "sending", "sent", "processing", "success", "failed"
    - created [datetime.datetime]: creation datetime for the log. ex: datetime.datetime(2020, 3, 10, 10, 30, 0, 0)
    """

    def __init__(self, id, request, errors, type, created):
        Resource.__init__(self, id=id)

        self.request = from_api_json(_issuing_embossing_request_resource, request)
        self.errors = _parse_errors(errors)
        self.type = type
        self.created = check_datetime(created)


_resource = {"class": Log, "name": "IssuingEmbossingRequestLog"}


def query(limit=None, ids=None, after=None, before=None, types=None, request_ids=None, user=None):
    """# Retrieve issuingembossingrequest.Log
    Receive a generator of issuingembossingrequest.Log objects previously created in the Stark Infra API
    ## Parameters (optional):
    - limit [integer, default None]: maximum number of objects to be retrieved. Unlimited if None. ex: 35
    - ids [list of strings, default None]: list of ids to filter retrieved objects. ex: ["5656565656565656", "4545454545454545"]
    - after [datetime.date or string, default None] date filter for objects created only after specified date. ex: datetime.date(2020, 3, 10)
    - before [datetime.date or string, default None] date filter for objects created only before specified date. ex: datetime.date(2020, 3, 10)
    - types [list of strings, default None]: filter for log event types. ex: ["created", "sending", "sent", "processing", "success", "failed"]
    - request_ids [list of strings, default None]: list of IssuingEmbossingRequest ids to filter logs. ex: ["5656565656565656", "4545454545454545"]
    - user [Organization/Project object, default None]: Organization or Project object. Not necessary if starkinfra.user was set before function call.
    ## Return:
    - generator of issuingembossingrequest.Log objects with updated attributes
    """
    return rest.get_stream(
        resource=_resource,
        limit=limit,
        ids=ids,
        after=check_date(after),
        before=check_date(before),
        types=types,
        request_ids=request_ids,
        user=user,
    )


def page(cursor=None, limit=None, ids=None, after=None, before=None, types=None, request_ids=None, user=None):
    """# Retrieve paged issuingembossingrequest.Log
    Receive a list of up to 100 issuingembossingrequest.Log objects previously created in the Stark Infra API and the cursor to the next page.
    Use this function instead of query if you want to manually page your requests.
    ## Parameters (optional):
    - cursor [string, default None]: cursor returned on the previous page function call
    - limit [integer, default 100]: maximum number of objects to be retrieved. Max = 100. ex: 50
    - ids [list of strings, default None]: list of ids to filter retrieved objects. ex: ["5656565656565656", "4545454545454545"]
    - after [datetime.date or string, default None] date filter for objects created only after specified date. ex: datetime.date(2020, 3, 10)
    - before [datetime.date or string, default None] date filter for objects created only before specified date. ex: datetime.date(2020, 3, 10)
    - types [list of strings, default None]: filter for log event types. ex: ["created", "sending", "sent", "processing", "success", "failed"]
    - request_ids [list of strings, default None]: list of IssuingEmbossingRequest ids to filter logs. ex: ["5656565656565656", "4545454545454545"]
    - user [Organization/Project object, default None]: Organization or Project object. Not necessary if starkinfra.user was set before function call.
    ## Return:
    - list of issuingembossingrequest.Log objects with updated attributes
    - cursor to retrieve the next page of issuingembossingrequest.Log objects
    """
    return rest.get_page(
        resource=_resource,
        cursor=cursor,
        limit=limit,
        ids=ids,
        after=check_date(after),
        before=check_date(before),
        types=types,
        request_ids=request_ids,
        user=user,
    )


def get(id, user=None):
    """# Retrieve a specific issuingembossingrequest.Log
    Receive a single issuingembossingrequest.Log object previously created by the Stark Infra API by its id
    ## Parameters (required):
    - id [string]: object unique id. ex: "5656565656565656"
    ## Parameters (optional):
    - user [Organization/Project object, default None]: Organization or Project object. Not necessary if starkinfra.user was set before function call.
    ## Return:
    - issuingembossingrequest.Log object with updated attributes
    """
    return rest.get_id(resource=_resource, id=id, user=user)


def _parse_errors(errors):
    parsed_errors = []
    for error in errors:
        if isinstance(error, Error):
            parsed_errors.append(error)
            continue
        parsed_errors.append(Error(code=error['code'], message=error['message']))
    return parsed_errors
