"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Key = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const alias_1 = require("./alias");
const kms_generated_1 = require("./kms.generated");
class KeyBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Collection of aliases added to the key
         *
         * Tracked to determine whether or not the aliasName should be added to the end of its ID
         */
        this.aliases = [];
    }
    /**
     * Defines a new alias for the key.
     */
    addAlias(aliasName) {
        const aliasId = this.aliases.length > 0 ? `Alias${aliasName}` : 'Alias';
        const alias = new alias_1.Alias(this, aliasId, { aliasName, targetKey: this });
        this.aliases.push(alias);
        return alias;
    }
    /**
     * Adds a statement to the KMS key resource policy.
     *
     * @param statement The policy statement to add.
     * @param allowNoOp If this is set to `false` and there is no policy defined (i.e. external key), the operation will fail. Otherwise, it will no-op.
     */
    addToResourcePolicy(statement, allowNoOp = true) {
        const stack = core_1.Stack.of(this);
        if (!this.policy) {
            if (allowNoOp) {
                return { statementAdded: false };
            }
            throw new Error(`Unable to add statement to IAM resource policy for KMS key: ${JSON.stringify(stack.resolve(this.keyArn))}`);
        }
        this.policy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.policy };
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     */
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Grant the indicated permissions on this key to the given principal.
     *
     * This modifies both the principal's policy as well as the resource policy,
     * since the default CloudFormation setup for KMS keys is that the policy
     * must not be empty and so default grants won't work.
     */
    grant(grantee, ...actions) {
        // KMS verifies whether the principals included in its key policy actually exist.
        // This is a problem if the stack the grantee is part of depends on the key stack
        // (as it won't exist before the key policy is attempted to be created).
        // In that case, make the account the resource policy principal
        const granteeStackDependsOnKeyStack = this.granteeStackDependsOnKeyStack(grantee);
        const principal = granteeStackDependsOnKeyStack
            ? new iam.AccountPrincipal(granteeStackDependsOnKeyStack)
            : grantee.grantPrincipal;
        const crossAccountAccess = this.isGranteeFromAnotherAccount(grantee);
        const crossRegionAccess = this.isGranteeFromAnotherRegion(grantee);
        const crossEnvironment = crossAccountAccess || crossRegionAccess;
        const grantOptions = {
            grantee,
            actions,
            resource: this,
            resourceArns: [this.keyArn],
            resourceSelfArns: crossEnvironment ? undefined : ['*'],
        };
        if (this.trustAccountIdentities) {
            return iam.Grant.addToPrincipalOrResource(grantOptions);
        }
        else {
            return iam.Grant.addToPrincipalAndResource({
                ...grantOptions,
                // if the key is used in a cross-environment matter,
                // we can't access the Key ARN (they don't have physical names),
                // so fall back to using '*'. ToDo we need to make this better... somehow
                resourceArns: crossEnvironment ? ['*'] : [this.keyArn],
                resourcePolicyPrincipal: principal,
            });
        }
    }
    /**
     * Grant decryption permisisons using this key to the given principal.
     */
    grantDecrypt(grantee) {
        return this.grant(grantee, 'kms:Decrypt');
    }
    /**
     * Grant encryption permisisons using this key to the given principal.
     */
    grantEncrypt(grantee) {
        return this.grant(grantee, 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*');
    }
    /**
     * Grant encryption and decryption permisisons using this key to the given principal.
     */
    grantEncryptDecrypt(grantee) {
        return this.grant(grantee, 'kms:Decrypt', 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*');
    }
    /**
     * Checks whether the grantee belongs to a stack that will be deployed
     * after the stack containing this key.
     *
     * @param grantee the grantee to give permissions to
     * @returns the account ID of the grantee stack if its stack does depend on this stack,
     *   undefined otherwise
     */
    granteeStackDependsOnKeyStack(grantee) {
        const grantPrincipal = grantee.grantPrincipal;
        if (!(grantPrincipal instanceof constructs_1.Construct)) {
            return undefined;
        }
        // this logic should only apply to newly created
        // (= not imported) resources
        if (!this.principalIsANewlyCreatedResource(grantPrincipal)) {
            return undefined;
        }
        // return undefined;
        const keyStack = core_1.Stack.of(this);
        const granteeStack = core_1.Stack.of(grantPrincipal);
        if (keyStack === granteeStack) {
            return undefined;
        }
        return granteeStack.dependencies.includes(keyStack)
            ? granteeStack.account
            : undefined;
    }
    principalIsANewlyCreatedResource(principal) {
        // yes, this sucks
        // this is just a temporary stopgap to stem the bleeding while we work on a proper fix
        return principal instanceof iam.Role ||
            principal instanceof iam.User ||
            principal instanceof iam.Group;
    }
    isGranteeFromAnotherRegion(grantee) {
        if (!(grantee instanceof constructs_1.Construct)) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.region !== identityStack.region;
    }
    isGranteeFromAnotherAccount(grantee) {
        if (!(grantee instanceof constructs_1.Construct)) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.account !== identityStack.account;
    }
}
/**
 * Defines a KMS key.
 *
 * @resource AWS::KMS::Key
 */
class Key extends KeyBase {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.policy = props.policy || new iam.PolicyDocument();
        this.trustAccountIdentities = props.trustAccountIdentities || false;
        if (this.trustAccountIdentities) {
            this.allowAccountIdentitiesToControl();
        }
        else {
            this.allowAccountToAdmin();
        }
        const resource = new kms_generated_1.CfnKey(this, 'Resource', {
            description: props.description,
            enableKeyRotation: props.enableKeyRotation,
            enabled: props.enabled,
            keyPolicy: this.policy,
        });
        this.keyArn = resource.attrArn;
        this.keyId = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy);
        if (props.alias !== undefined) {
            this.addAlias(props.alias);
        }
    }
    /**
     * Import an externally defined KMS Key using its ARN.
     *
     * @param scope the construct that will "own" the imported key.
     * @param id the id of the imported key in the construct tree.
     * @param keyArn the ARN of an existing KMS key.
     */
    static fromKeyArn(scope, id, keyArn) {
        class Import extends KeyBase {
            constructor(keyId) {
                super(scope, id);
                this.keyArn = keyArn;
                this.policy = undefined;
                // defaulting true: if we are importing the key the key policy is
                // undefined and impossible to change here; this means updating identity
                // policies is really the only option
                this.trustAccountIdentities = true;
                this.keyId = keyId;
            }
        }
        const keyResourceName = core_1.Stack.of(scope).parseArn(keyArn).resourceName;
        if (!keyResourceName) {
            throw new Error(`KMS key ARN must be in the format 'arn:aws:kms:<region>:<account>:key/<keyId>', got: '${keyArn}'`);
        }
        return new Import(keyResourceName);
    }
    allowAccountIdentitiesToControl() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['kms:*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Let users or IAM policies from this account admin this key.
     * @link https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default
     * @link https://aws.amazon.com/premiumsupport/knowledge-center/update-key-policy-future/
     */
    allowAccountToAdmin() {
        const actions = [
            'kms:Create*',
            'kms:Describe*',
            'kms:Enable*',
            'kms:List*',
            'kms:Put*',
            'kms:Update*',
            'kms:Revoke*',
            'kms:Disable*',
            'kms:Get*',
            'kms:Delete*',
            'kms:ScheduleKeyDeletion',
            'kms:CancelKeyDeletion',
            'kms:GenerateDataKey',
            'kms:TagResource',
            'kms:UntagResource',
        ];
        this.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions,
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
}
exports.Key = Key;
//# sourceMappingURL=data:application/json;base64,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