"""
author: Etienne Wallet

This module contains the classes used to check on-chain transactions
"""
from dataclasses import dataclass
import sys
from typing import Dict, List

from multiversx_sdk_network_providers.transactions import TransactionOnNetwork

from mxops import errors
from mxops.execution.network import raise_on_errors


@dataclass
class Check:
    """
    Represents a check to operate on the content of an on-chain transaction
    """

    def raise_on_failure(self, onchain_tx: TransactionOnNetwork):
        """
        Perform the check on the transaction and raise an error if it failed.

        :param onchain_tx: transaction to perform the check on
        :type onchain_tx: TransactionOnNetwork
        """
        try:
            if not self.get_check_status(onchain_tx):
                raise errors.CheckFailed(self, onchain_tx)
        except Exception as err:
            raise errors.CheckFailed(self, onchain_tx) from err

    def get_check_status(self, onchain_tx: TransactionOnNetwork) -> bool:
        """
        Interface for the method to execute the check described by a Check instance.
        Each child class must overrid this method

        :param onchain_tx: transaction to perform the check on
        :type onchain_tx: TransactionOnNetwork
        :return: True if the check pass
        :rtype: bool
        """
        raise NotImplementedError


@dataclass
class SuccessCheck(Check):
    """
    Check that verify that an on-chain transaction is successful
    """

    def get_check_status(self, onchain_tx: TransactionOnNetwork) -> bool:
        """
        Check that verify that an on-chain transaction is successful

        :param onchain_tx: transaction to perform the check on
        :type onchain_tx: TransactionOnNetwork
        :return: True if the check pass
        :rtype: bool
        """
        raise_on_errors(onchain_tx)
        return True


def instanciate_checks(raw_checks: List[Dict]) -> List[Check]:
    """
    Take checks as dictionaries and convert them to their corresponding check classes.

    :param raw_checks: checks to instantiate
    :type raw_checks: List[Dict]
    :return: checks instances
    :rtype: List[Check]
    """
    checks_list = []
    for raw_check in raw_checks:
        check_class_name = raw_check.pop('type') + 'Check'
        try:
            check_class_object = getattr(sys.modules[__name__], check_class_name)
        except AttributeError as err:
            raise ValueError(f'Unkown check type: {check_class_name}') from err
        checks_list.append(check_class_object(**raw_check))
    return checks_list
