# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['charidfield']

package_data = \
{'': ['*']}

install_requires = \
['django>=3.1,<4.0']

setup_kwargs = {
    'name': 'django-charid-field',
    'version': '0.1.0',
    'description': 'Provides a char-based, prefixable ID field for your Django models. Supports cuid, ksuid, ulid, et al.',
    'long_description': '# django-charid-field\n\nProvides a char-based, prefixable CharIDField for your Django models.\n\nIt can utilise [cuid], [ksuid], [ulid] or any other string-based UID generation systems.\n\nIt can be used as the primary key, or simple another key on your models.\n\n[cuid]: https://github.com/ericelliott/cuid\n[ksuid]: https://github.com/segmentio/ksuid\n[ulid]: https://github.com/ulid/spec\n\n## ⛲ Feature set\n\n-   Ability to work with the UID generation spec of your choice.\n-   Support for prefixing the ID on a per-model basis à la Stripe. e.g `cus_` => `cus_cjld2cjxh0000qzrmn831i7rn`\n-   Support for all database backends that support the `CharField`.\n-   Support for Python 3.9 & above only.\n## 🤷 Why?\n\nTo get us a global namespace of collision-resistant IDs that:\n\n* are URL-safe\n* can be represented in a visual-space-efficient manor\n* are collision-resistant to allow for client side generation\n* UUID v6, v7, v8 are in RFC draft and not ready.\n\n[cuid], [ksuid], [ulid] & many others offer this now, and prefixing gets us the global namespace.\n\n**Why not use integers?**\n\n* Auto-incrementing integers are easily enumerable and give away collection count.\n\n* You can solve that with HashID but then you either have to store the HashID as another column or deal with constant conversion when looking up values in your UI VS raw in your database.\n\n* Most importantly: relying on your database to generate IDs means sequential writes. Your clients are not free to generate their own IDs without a round trip to the database.\n\n**Why not use UUIDs?**\n\nThey solve the collision problem so why not?\n\n* The text formats use hex, which is not visually space-efficient.\n* UUIDv4 (the one usually recommended) is completely random and thus impossible to sort. This has the known on effect of making databases work harder when looking up/indexing as binary search goes out the window.\n* Optional hyphenation when representing the hex. This nuance results in more code.\n\n**Why prefix?**\n\nBecause global flat namespaces are powerful. An ID now represents the instance _and it\'s type_, which means you can have powerful lookup abilities with just the identifier alone. No more guessing whether `802302` is a `Dog` or a `Cat`.\n\n## 📗 Install\n\nInstall using your favourite Python dependency manager, or straight with pip:\n\n```\npip install django-charid-field\n```\n\nYou\'ll also need to install your ID-generation library of choice (or bring your own).\n\nFor example:\n\n|UID Spec|Python Library|What could it look like? (with a prefix `dev_`)|\n|--------|--------------|----------------------------------------|\n|[cuid]|cuid.py: [GH](https://github.com/necaris/cuid.py) / [PyPi](https://pypi.org/project/cuid/)|`dev_ckpffbliw000001mi3fw42vsn`|\n|[ksuid]|cyksuid: [GH](https://github.com/timonwong/cyksuid) / [PyPi](https://pypi.org/project/cyksuid/)|`dev_1tOMP4onidzvnUFuTww2UeamY39`|\n|[ulid]|python-ulid: [GH](https://github.com/mdomke/python-ulid) / [PyPi](https://pypi.org/project/python-ulid/)|`dev_01F769XGM83VR75H86ZPHKK595`|\n\n\n\n## ✨ Usage\n\n```\nfrom charidfield import CharIDField\n```\n\nWe recommend using `functool.partial` to create your own field for your codebase; this will allow you to specify your chosen ID generation and set the `max_length` parameter and then have an importable field you can use across all your models.\n\nHere\'s an example using the cuid spec and cuid.py:\n\n```python\n# Locate this somewhere importable\nfrom cuid import cuid\nfrom charidfield import CharIDField\n\nCuidField = partial(\n    CharIDField,\n    default=cuid,\n    max_length=30,\n    help_text="cuid-format identifier for this entity."\n)\n\n# models.py\nfrom wherever_you_put_it import CuidField\n\nclass Dog(models.Model):\n    id = CuidField(primary_key=True, prefix="dog_")\n    name = models.CharField()\n\n# shell\n>>> dog = Dog(name="Ronnie")\n>>> dog.id\n"dog_ckpffbliw000001mi3fw42vsn"\n\n```\n\n### Parameters\n\n|Param|Type|Required|Default|Note|\n|-----|----|--------|-------|----|\n|**default**|`Callable`|❌|-|This should be a callable which generates a UID in whatever system you chose. Your callable does not have to handle prefixing, the prefix will be applied onto the front of whatever string your default callable generates. Technically not required, but without it you must handle ID generation yourself.|\n|**max_length**|`int`|✅|Set it|Controls the maximum length of the stored strings. Provide your own to match whatever ID system you pick, remembering to take into account the length of any prefixes you have configured. Also note that there is no perf/storage impact for modern Postgres so for that backend it is effectively an arbitary char limit.|\n|**primary_key**|`boolean`|❌|`False`|Set to `True` to replace Django\'s default `Autofield` that gets used as the primary key, else the field will be additional ID field available to the model.|\n|**prefix**|`str or Callable` |❌|`""`|If provided, the ID strings generated as the field\'s default value will be prefixed. This provides a way to have a per-model prefix which can be helpful in providing a global namespace for your ID system. The prefix can be provided as a string literal (e.g `cus_`), or as a `Callable` which is run when the field is attached to the model instance and can allow for more dynamic prefixing needs. For more, see below.|\n|**unique**|`boolean`|❌|`True`|Whether the field should be treated as unique across the dataset; the field provides a sane default of `True` so that a database index is setup to protext you against collisions (whether due to chance or, more likely, a bug/human error). To turn the index off, simply pass `False`.|\n\nAll other `django.db.models.fields.CharField` keyword arguments should work as expected. See the [Django docs](https://docs.djangoproject.com/en/dev/ref/models/fields/#django.db.models.CharField).\n\n### Usage as the Primary Key\n\n\nThis will replace Django\'s `AutoField` and the cuid will become the main primary key\nfor the entity, thus removing the default database-genererated incremental integer ID.\n\n```python\n# models/some_model.py or models.py\n\nclass SomeModel(models.Model):\n    id = CharIDField(primary_key=True, default=your_id_generator)\n\n>>> some_model = SomeModel.objects.create()\n>>> some_model.id\n"ckp9jm3qn001001mrg5hw3sk4"\n>>> some_model.pk\n"ckp9jm3qn001001mrg5hw3sk4"\n""\n```\n### Setting up prefixing\n\n#### What?\n\nPrefixing allows per-entity ID namespacing, e.g:\n\n```\ncus_ckp9mdxpd000i01ld6gzjgyl4 (reference a specific customer)\nusr_ckp9me8zy000p01lda5579o3q (reference a specific user)\norg_ckp9mek2d000s01ld8ffhhvd3 (reference a specific organisation)\n```\n\n#### Why?\n\nBy prefixing your entities IDs you can create a global namespace for your ID system which has numerous advantages:\n\n* when displaying an ID you can immediately derive what type of object it represents from reading the prefix alone; most identifiers only showcase what instance is represented, but without information about the type it is machine-impossile to tell if ID `123` is from the `Dog` or `Cat` models. Whereas `cat_123` and `dog_123` make that clear.\n\n* by having a global system of prefixing, you can speed up internal processes as (think: support) by having features in your backoffice such as "quick find" which allows you to dump the ID in question and be taken straight to the page which represents the specific instance of that type of object.\n\nThis may sound familiar, as it\'s how [Stripe](http://stripe.com/) handle their public IDs - everything is referenceable.\n\n#### How?\n\nTwo options.\n\nFirst is to set a string literal during field instantiation. E.g:\n\n```python\n# models.py\n\nclass User(models.Model):\n    public_id = CharIDField(prefix="usr_", ...)\n\n>>> user = User.objects.create()\n>>> user.public_id\n"usr_ckp9me8zy000p01lda5579o3q"\n```\n\nSecond is to pass a callable which is executed after field initialisation and during its addition to the model itself (`Field.contribute_to_class`). This allows for dynamic generation of the prefix at runtime, which is especially helpful if you\'ve defined the field on an Abstract Django model class, as the prefix generator will be called once for every concrete model that inherits the abstract.\n\nThe callable should accept `model_class: models.Model` (the model the field is being added to), `field_instance: django.db.models.Field` (the field instance being added) & `field_name: str` (the name of the field on the model). E.g:\n\n```python\n# models.py\n\ndef get_prefix_from_class_name(\n    *,\n    model_class: models.Model,\n    field_instance: Field,\n    field_name: str,\n) -> str:\n    """Return the Model\'s name in snake_case for use as a cuid prefix."""\n    name = model_class.__name__\n    # CamelCase to snake_case\n    name = re.sub(\'(.)([A-Z][a-z]+)\', r\'\\1_\\2\', name)\n    return re.sub(\'([a-z0-9])([A-Z])\', r\'\\1_\\2\', name).lower() + "_"\n\n\nclass UserProfile(models.Model):\n    public_id = CharIDField(prefix=get_prefix_from_class_name, ...)\n\n>>> user_profile = UserProfile.objects.create()\n>>> user_profile.public_id\n"user_profile_ckp9me8zy000p01lda5579o3q"\n```\n\nSee the tests for more common usage patterns.\n## 👩\u200d💻 Development\n\n### 🏗️ Local environment\n\nThe local environment is handled with `poetry`, so install that first then:\n\n```\n$ poetry install\n```\n\n### 🧪 Running tests\n\nThe tests themselves use `pytest` as the test runner.\n\nAfter setting up the environment, run them using:\n\n```\n$ poetry run pytest\n```\n\nThe full CI suite is controlled by `tox`, which contains a set of environments that will format\n(`fmt`), lint, and test against all support Python + Django version combinations.\n\n```\n$ tox\n```\n\n#### ⚙️ CI\n\nUses GitHub Actions, see `./github/workflows`.\n\n[cuid]: https://github.com/ericelliott/cuid\n[ksuid]: https://github.com/segmentio/ksuid\n[ulid]: https://github.com/ulid/spec\n',
    'author': 'YunoJuno',
    'author_email': 'code@yunojuno.com',
    'maintainer': 'YunoJuno',
    'maintainer_email': 'code@yunojuno.com',
    'url': 'https://github.com/yunojuno/django-charid-field',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
