"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionProvider = exports.Product = exports.Platform = void 0;
const fs = require("fs");
const https = require("https");
const url = require("url");
const version_1 = require("./version");
var Platform;
(function (Platform) {
    Platform["linux"] = "linux";
    Platform["mac"] = "mac";
    Platform["windows"] = "windows";
})(Platform = exports.Platform || (exports.Platform = {}));
var Product;
(function (Product) {
    Product["deadline"] = "Deadline";
    Product["deadlineDocker"] = "DeadlineDocker";
})(Product = exports.Product || (exports.Product = {}));
/**
 * The version provider parses a JSON file containing version information for the Deadline and DockerDeadline products.
 * It  can be downloaded or loaded from local file and returns URIs for the specific products.
 * By default returns the last version of URIs or specified full or partial version.
 * If platform is not defined returns URIs for each platform.
 */
class VersionProvider {
    constructor(indexFilePath) {
        this.indexFilePath = indexFilePath;
    }
    /**
     * Returns URIs for specified product
     */
    async getVersionUris(resourceProperties) {
        const indexJson = this.indexFilePath ? this.readInstallersIndex() : await this.downloadInstallerIndex();
        const productSection = indexJson[resourceProperties.product];
        if (!productSection) {
            throw new Error(`Information about product ${resourceProperties.product} can't be found`);
        }
        let installers = new Map();
        if (resourceProperties.platform) {
            const versionedUris = this.getUrisForPlatform(resourceProperties.product, productSection, resourceProperties.platform, resourceProperties.versionString);
            if (versionedUris) {
                installers.set(resourceProperties.platform, versionedUris);
            }
        }
        else {
            Object.values(Platform).forEach(async (p) => {
                const versionedUris = this.getUrisForPlatform(resourceProperties.product, productSection, p, resourceProperties.versionString);
                if (versionedUris) {
                    installers.set(p, versionedUris);
                }
            });
        }
        return installers;
    }
    async downloadInstallerIndex() {
        const parsedUrl = url.parse(VersionProvider.VERSION_INDEX_URL);
        const options = {
            host: parsedUrl.hostname,
            path: parsedUrl.path,
        };
        return new Promise((resolve, reject) => {
            https.get(options, (res) => {
                let json = '';
                res.on('data', (chunk) => {
                    // keep appending the response chunks until we get 'end' event.
                    json += chunk;
                });
                res.on('end', () => {
                    // complete response is available here:
                    if (res.statusCode === 200) {
                        try {
                            // convert the response to a json object and return.
                            const data = JSON.parse(json);
                            resolve(data);
                        }
                        catch (e) {
                            reject(e);
                        }
                    }
                    else {
                        reject(new Error(`Expected status code 200, but got ${res.statusCode}`));
                    }
                });
            }).on('error', (err) => {
                reject(err);
            });
        });
    }
    /**
     * This method reads index file and return parsed JSON object from this file content.
     */
    readInstallersIndex() {
        if (!this.indexFilePath) {
            throw new Error('File path should be defined.');
        }
        if (!fs.existsSync(this.indexFilePath)) {
            throw new Error(`File ${this.indexFilePath} was not found`);
        }
        const data = fs.readFileSync(this.indexFilePath, 'utf8');
        // convert the response to a json object and return.
        const json = JSON.parse(data);
        return json;
    }
    /**
     * This method returns IVersionedUris (the patch version plus installer URI's) for a specific platform.
     */
    getUrisForPlatform(product, productSection, platform, version) {
        const versionString = version ? version : this.getLatestVersion(platform, productSection);
        const requestedVersion = version_1.Version.parseFromVersionString(versionString);
        if (!requestedVersion) {
            throw new Error(`Couldn't parse version from ${versionString}`);
        }
        return this.getRequestedUriVersion(requestedVersion, productSection.versions, platform, product);
    }
    /**
     * This method returns the latest version for specified platform.
     */
    getLatestVersion(platform, indexedVersionInfo) {
        const latestSection = indexedVersionInfo.latest;
        if (!latestSection) {
            throw new Error('Information about latest version can not be found');
        }
        const latestVersion = latestSection[platform];
        if (!latestVersion) {
            throw new Error(`Information about latest version for platform ${platform} can not be found`);
        }
        return latestVersion;
    }
    /**
     * This method looks for the requested version (partial or complete) in the
     * indexed version information. Based on the input, it iterates through all
     * four numbers in the version string and compares the requested version
     * with the indexed info.
     * If any of the requested version number is missing, it fetches the latest
     * (highest) available version for it.
     */
    getRequestedUriVersion(requestedVersion, indexedVersionInfo, platform, product) {
        let versionMap = indexedVersionInfo;
        const versionArray = [];
        // iterate through all 4 major, minor, release and patch numbers,
        // and get the matching version from the indexed version map.
        for (let versionIndex = 0; versionIndex < 4; versionIndex++) {
            let version;
            if (requestedVersion[versionIndex + 1] == null) {
                // version is not provided, get the max version.
                const numberValues = (Object.keys(versionMap)).map((val) => {
                    return parseInt(val, 10);
                });
                version = (Math.max(...numberValues)).toString();
            }
            else {
                version = requestedVersion[versionIndex + 1];
            }
            if (versionMap[version] === undefined) {
                throw new Error(`${product} version ${requestedVersion[0]} is not available on ${platform}`);
            }
            versionArray[versionIndex] = version;
            versionMap = versionMap[version];
        }
        let uriIndex;
        if ((platform in versionMap)) {
            const platformVersionMap = versionMap[platform];
            if (product == Product.deadline) {
                uriIndex = {
                    bundle: platformVersionMap.bundle,
                    clientInstaller: platformVersionMap.clientInstaller,
                    repositoryInstaller: platformVersionMap.repositoryInstaller,
                    certificateInstaller: platformVersionMap.certificateInstaller,
                };
            }
            else { // Product.deadlineDocker
                uriIndex = {
                    bundle: platformVersionMap,
                };
            }
        }
        if (uriIndex) {
            return {
                MajorVersion: versionArray[0],
                MinorVersion: versionArray[1],
                ReleaseVersion: versionArray[2],
                PatchVersion: versionArray[3],
                Uris: uriIndex,
            };
        }
        else {
            return undefined;
        }
    }
}
exports.VersionProvider = VersionProvider;
VersionProvider.VERSION_INDEX_URL = 'https://downloads.thinkboxsoftware.com/version_info.json';
//# sourceMappingURL=data:application/json;base64,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