"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const events_1 = require("events");
const fs = require("fs");
const ecr_provider_1 = require("../../lib/ecr-provider");
jest.mock('fs');
jest.mock('https');
describe('ThinkboxEcrProvider', () => {
    /**
     * Suite of parametrized tests for testing the ECR index schema validation.
     *
     * The suite is an array of tests, where each test should fail validation. Each test is represented as an array of two
     * elements: [name, indexObject]
     *
     * - The first element is the name describing what is contained in the value
     * - The second element is the value that should be JSON encoded and supplied to the ThinkboxEcrProvider
     */
    const INDEX_SCHEMA_VALIDATION_SUITE = [
        [
            'array',
            [],
        ],
        [
            'number',
            1,
        ],
        [
            'string',
            'abc',
        ],
        [
            'object missing registry',
            {
                products: {
                    deadline: {
                        namespace: 'a',
                    },
                },
            },
        ],
        [
            'object with registry with wrong type',
            {
                registry: 1,
                products: {
                    deadline: {
                        namespace: 'a',
                    },
                },
            },
        ],
        [
            'object missing products',
            {
                registry: {
                    uri: 'a',
                },
            },
        ],
        [
            'object with products with wrong type',
            {
                registry: {
                    uri: 'a',
                },
                products: 1,
            },
        ],
        [
            'object with registry missing uri',
            {
                registry: {},
                products: {
                    deadline: {
                        namespace: 'a',
                    },
                },
            },
        ],
        [
            'object with registry uri with wrong type',
            {
                registry: {
                    uri: 1,
                },
                products: {
                    deadline: {
                        namespace: 'a',
                    },
                },
            },
        ],
        [
            'object with missing products.deadline',
            {
                registry: {
                    uri: 1,
                },
                products: {},
            },
        ],
        [
            'object with products.deadline with wrong type',
            {
                registry: {
                    uri: 1,
                },
                products: {
                    deadline: 1,
                },
            },
        ],
        [
            'object with missing products.deadline.namespace',
            {
                registry: {
                    uri: 1,
                },
                products: {
                    deadline: {},
                },
            },
        ],
        [
            'object with products.deadline.namespace with wrong type',
            {
                registry: {
                    uri: 1,
                },
                products: {
                    deadline: {
                        namespace: 1,
                    },
                },
            },
        ],
    ];
    let ecrProvider;
    describe('without indexPath', () => {
        class MockResponse extends events_1.EventEmitter {
            constructor() {
                super(...arguments);
                this.statusCode = 200;
            }
        }
        let request;
        let response;
        /**
         * Mock implementation of a successful HTTPS GET request
         *
         * @param _url The URL of the HTTPS request
         * @param callback The callback to call when a response is available
         */
        function httpGetMockSuccess(_url, callback) {
            if (callback) {
                callback(response);
            }
            return request;
        }
        /**
         * Mock implementation of a failed HTTPS GET request
         *
         * @param _url The URL of the HTTPS request
         * @param _callback The callback to call when a response is available
         */
        function httpGetMockError(_url, _callback) {
            return request;
        }
        beforeEach(() => {
            request = new events_1.EventEmitter();
            response = new MockResponse();
            jest.requireMock('https').get.mockImplementation(httpGetMockSuccess);
            // GIVEN
            ecrProvider = new ecr_provider_1.ThinkboxEcrProvider();
        });
        const EXPECTED_URL = 'https://downloads.thinkboxsoftware.com/thinkbox_ecr.json';
        test(`gets ${EXPECTED_URL} for global lookup`, async () => {
            // GIVEN
            const registryUri = 'registryUri';
            const deadlineNamespace = 'namespace';
            const mockData = {
                registry: {
                    uri: registryUri,
                },
                products: {
                    deadline: {
                        namespace: deadlineNamespace,
                    },
                },
            };
            // WHEN
            const promise = ecrProvider.getGlobalEcrBaseURI();
            response.emit('data', JSON.stringify(mockData));
            response.emit('end');
            await promise;
            // THEN
            // should make an HTTPS request for the ECR index
            expect(jest.requireMock('https').get)
                .toBeCalledWith(EXPECTED_URL, expect.any(Function));
        });
        describe('.getGlobalEcrBaseArn()', () => {
            test('obtains global prefix from index', async () => {
                // GIVEN
                const registryUri = 'registryUri';
                const deadlineNamespace = 'namespace';
                const mockData = {
                    registry: {
                        uri: registryUri,
                    },
                    products: {
                        deadline: {
                            namespace: deadlineNamespace,
                        },
                    },
                };
                const expectedBaseArn = `${registryUri}/${deadlineNamespace}`;
                // WHEN
                const promise = ecrProvider.getGlobalEcrBaseURI();
                response.emit('data', JSON.stringify(mockData));
                response.emit('end');
                // THEN
                await expect(promise)
                    .resolves
                    .toEqual(expectedBaseArn);
            });
            test('handles request errors', async () => {
                // GIVEN
                const error = new Error('test');
                jest.requireMock('https').get.mockImplementation(httpGetMockError);
                function simulateRequestError() {
                    request.emit('error', error);
                }
                ;
                // WHEN
                const promise = ecrProvider.getGlobalEcrBaseURI();
                simulateRequestError();
                // THEN
                await expect(promise)
                    .rejects
                    .toThrowError(error);
            });
            test.each([
                [404],
                [401],
                [500],
            ])('handles %d response errors', async (statusCode) => {
                // GIVEN
                response.statusCode = statusCode;
                // WHEN
                const promise = ecrProvider.getGlobalEcrBaseURI();
                response.emit('data', '');
                response.emit('end');
                // THEN
                await expect(promise)
                    .rejects
                    .toThrowError(`Expected status code 200, but got ${statusCode}`);
            });
            test('fails on bad JSON', async () => {
                // GIVEN
                const responseBody = 'this is not json';
                // WHEN
                const promise = ecrProvider.getGlobalEcrBaseURI();
                response.emit('data', responseBody);
                response.emit('end');
                // THEN
                await expect(promise)
                    .rejects
                    .toThrow(/^ECR index file contains invalid JSON: ".*"$/);
            });
            describe('index schema validation', () => {
                test.each(INDEX_SCHEMA_VALIDATION_SUITE)('fails when fetching %s', async (_name, value) => {
                    // WHEN
                    const promise = ecrProvider.getGlobalEcrBaseURI();
                    response.emit('data', JSON.stringify(value));
                    response.emit('end');
                    // THEN
                    await expect(promise)
                        .rejects
                        .toThrowError(/^expected .+ to be an? .+ but got .+$/);
                });
            });
        });
    });
    describe('with indexPath', () => {
        // GIVEN
        const registryUri = 'registryUri';
        const deadlineNamespace = 'deadlineNamespace';
        const indexPath = 'somefile';
        const mockData = {
            registry: {
                uri: registryUri,
            },
            products: {
                deadline: {
                    namespace: deadlineNamespace,
                },
            },
        };
        const globalURIPrefix = `${registryUri}/${deadlineNamespace}`;
        beforeEach(() => {
            // WHEN
            const existsSync = jest.requireMock('fs').existsSync;
            const readFileSync = jest.requireMock('fs').readFileSync;
            // reset tracked calls to mocks
            existsSync.mockReset();
            readFileSync.mockReset();
            // set the default mock implementations
            existsSync.mockReturnValue(true);
            readFileSync.mockReturnValue(JSON.stringify(mockData));
            ecrProvider = new ecr_provider_1.ThinkboxEcrProvider(indexPath);
        });
        describe('.getGlobalEcrBaseURI', () => {
            let baseURIPromise;
            beforeEach(() => {
                // WHEN
                baseURIPromise = ecrProvider.getGlobalEcrBaseURI();
            });
            test('reads file', async () => {
                // THEN
                await expect(baseURIPromise)
                    .resolves.toEqual(globalURIPrefix);
                expect(fs.existsSync).toBeCalledTimes(1);
                expect(fs.readFileSync).toBeCalledWith(indexPath, 'utf8');
            });
            test('returns correct prefix', async () => {
                await expect(baseURIPromise)
                    .resolves
                    .toEqual(globalURIPrefix);
            });
            test.each([
                ['existsSync'],
                ['readFileSync'],
            ])('fails on fs.%s exception', async (methodName) => {
                // GIVEN
                const error = new Error('message');
                jest.requireMock('fs')[methodName].mockImplementation(() => {
                    throw error;
                });
                ecrProvider = new ecr_provider_1.ThinkboxEcrProvider(indexPath);
                // WHEN
                baseURIPromise = ecrProvider.getGlobalEcrBaseURI();
                // THEN
                await expect(baseURIPromise)
                    .rejects
                    .toThrowError(error);
            });
            describe('index schema validation', () => {
                test.each(INDEX_SCHEMA_VALIDATION_SUITE)('fails when fetching %s', async (_name, value) => {
                    // GIVEN
                    jest.requireMock('fs').readFileSync.mockReturnValue(JSON.stringify(value));
                    ecrProvider = new ecr_provider_1.ThinkboxEcrProvider(indexPath);
                    // WHEN
                    baseURIPromise = ecrProvider.getGlobalEcrBaseURI();
                    // THEN
                    await expect(baseURIPromise)
                        .rejects
                        .toThrowError(/^expected .+ to be an? .+ but got .+$/);
                });
            });
            test('fails on non-existent file', async () => {
                // GIVEN
                jest.requireMock('fs').existsSync.mockReturnValue(false);
                ecrProvider = new ecr_provider_1.ThinkboxEcrProvider(indexPath);
                // WHEN
                baseURIPromise = ecrProvider.getGlobalEcrBaseURI();
                // THEN
                await expect(baseURIPromise)
                    .rejects
                    .toThrowError(`File "${indexPath}" was not found`);
            });
            test('fails on bad JSON', async () => {
                // GIVEN
                jest.requireMock('fs').readFileSync.mockReturnValue('bad json');
                ecrProvider = new ecr_provider_1.ThinkboxEcrProvider(indexPath);
                // WHEN
                baseURIPromise = ecrProvider.getGlobalEcrBaseURI();
                // THEN
                await expect(baseURIPromise)
                    .rejects
                    .toThrow(/^ECR index file contains invalid JSON: ".*"$/);
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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