"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
test('VersionQuery constructor full version', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Stack');
    new lib_1.VersionQuery(stack, 'VersionQuery', { version: '10.1.9.2' });
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
        forceRun: assert_1.ABSENT,
        versionString: '10.1.9.2',
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                    Principal: {
                        Service: 'lambda.amazonaws.com',
                    },
                },
            ],
        },
        ManagedPolicyArns: [
            {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole',
                    ],
                ],
            },
        ],
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
        Handler: 'version-provider.handler',
        Role: {
            'Fn::GetAtt': [
                assert_1.stringLike('SingletonLambda*ServiceRole*'),
                'Arn',
            ],
        },
        Runtime: 'nodejs12.x',
    }));
});
test('VersionQuery constructor no versionString', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Stack');
    new lib_1.VersionQuery(stack, 'VersionQuery');
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
        forceRun: assert_1.stringLike('*'),
        versionString: assert_1.ABSENT,
    }));
});
test.each([
    ['10.1.9'],
    ['10.1'],
    ['10'],
])('VersionQuery constructor partial version: %s', (version) => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Stack');
    new lib_1.VersionQuery(stack, 'VersionQuery', { version });
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
        versionString: version,
        forceRun: assert_1.stringLike('*'),
    }));
});
//# sourceMappingURL=data:application/json;base64,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