"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('ThinkboxDockerRecipes', () => {
    let app;
    let depStack;
    let stack;
    let images;
    let userAwsThinkboxEulaAcceptance;
    describe('defaults', () => {
        beforeEach(() => {
            // GIVEN
            app = new core_1.App();
            stack = new core_1.Stack(app, 'Stack');
            userAwsThinkboxEulaAcceptance = lib_1.AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA;
            // WHEN
            images = new lib_1.ThinkboxDockerImages(stack, 'Images', {
                userAwsThinkboxEulaAcceptance,
            });
        });
        test('fails validation when EULA is not accepted', () => {
            // GIVEN
            const newStack = new core_1.Stack(app, 'NewStack');
            const expectedError = `
The ThinkboxDockerImages will install Deadline onto one or more EC2 instances.

Deadline is provided by AWS Thinkbox under the AWS Thinkbox End User License
Agreement (EULA). By installing Deadline, you are agreeing to the terms of this
license. Follow the link below to read the terms of the AWS Thinkbox EULA.

https://www.awsthinkbox.com/end-user-license-agreement

By using the ThinkboxDockerImages to install Deadline you agree to the terms of
the AWS Thinkbox EULA.

Please set the userAwsThinkboxEulaAcceptance property to
USER_ACCEPTS_AWS_THINKBOX_EULA to signify your acceptance of the terms of the
AWS Thinkbox EULA.
`;
            userAwsThinkboxEulaAcceptance = lib_1.AwsThinkboxEulaAcceptance.USER_REJECTS_AWS_THINKBOX_EULA;
            new lib_1.ThinkboxDockerImages(newStack, 'Images', {
                userAwsThinkboxEulaAcceptance,
            });
            // WHEN
            function synth() {
                assert_1.SynthUtils.synthesize(newStack);
            }
            // THEN
            expect(synth).toThrow(expectedError);
        });
        test('creates Custom::RFDK_ECR_PROVIDER', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_EcrProvider', {
                ForceRun: assert_1.stringLike('*'),
            }));
        });
        describe('provides container images for', () => {
            test.each([
                [
                    'RCS',
                    () => {
                        return images.remoteConnectionServer;
                    },
                    lib_1.ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER,
                ],
                [
                    'License Forwarder',
                    () => {
                        return images.licenseForwarder;
                    },
                    lib_1.ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER,
                ],
            ])('%s', (_label, imageGetter, recipe) => {
                // GIVEN
                const taskDefStack = new core_1.Stack(app, 'TaskDefStack');
                const image = imageGetter();
                const taskDefinition = new aws_ecs_1.TaskDefinition(taskDefStack, 'TaskDef', {
                    compatibility: aws_ecs_1.Compatibility.EC2,
                });
                const ecrProvider = images.node.defaultChild;
                const expectedImage = `${ecrProvider.getAtt('EcrURIPrefix')}${recipe}`;
                // WHEN
                new aws_ecs_1.ContainerDefinition(taskDefStack, 'ContainerDef', {
                    image,
                    taskDefinition,
                    memoryReservationMiB: 1024,
                });
                // THEN
                assert_1.expect(taskDefStack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                    ContainerDefinitions: assert_1.arrayWith(assert_1.objectLike({
                        Image: taskDefStack.resolve(expectedImage),
                    })),
                }));
            });
        });
        describe('.forRenderQueue()', () => {
            let rcsImage;
            let rqImages;
            beforeEach(() => {
                // GIVEN
                rcsImage = images.remoteConnectionServer;
                // WHEN
                rqImages = images.forRenderQueue();
            });
            test('returns correct RCS image', () => {
                // THEN
                expect(rqImages.remoteConnectionServer).toBe(rcsImage);
            });
        });
        describe('.forUsageBasedLicensing()', () => {
            let lfImage;
            let ublImages;
            beforeEach(() => {
                // GIVEN
                lfImage = images.licenseForwarder;
                // WHEN
                ublImages = images.forUsageBasedLicensing();
            });
            test('returns correct RCS image', () => {
                // THEN
                expect(ublImages.licenseForwarder).toBe(lfImage);
            });
        });
    });
    describe('with version', () => {
        let version;
        beforeEach(() => {
            // GIVEN
            app = new core_1.App();
            depStack = new core_1.Stack(app, 'DepStack');
            version = new lib_1.VersionQuery(depStack, 'Version');
            stack = new core_1.Stack(app, 'Stack');
            // WHEN
            images = new lib_1.ThinkboxDockerImages(stack, 'Images', {
                version,
                userAwsThinkboxEulaAcceptance,
            });
        });
        describe('provides container images for', () => {
            test.each([
                [
                    'RCS',
                    () => {
                        return images.remoteConnectionServer;
                    },
                    lib_1.ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER,
                ],
                [
                    'License Forwarder',
                    () => {
                        return images.licenseForwarder;
                    },
                    lib_1.ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER,
                ],
            ])('%s', (_label, imageGetter, recipe) => {
                // GIVEN
                const taskDefStack = new core_1.Stack(app, 'TaskDefStack');
                const image = imageGetter();
                const taskDefinition = new aws_ecs_1.TaskDefinition(taskDefStack, 'TaskDef', {
                    compatibility: aws_ecs_1.Compatibility.EC2,
                });
                const ecrProvider = images.node.defaultChild;
                const expectedImage = `${ecrProvider.getAtt('EcrURIPrefix')}${recipe}:${core_1.Token.asString(version.majorVersion)}.${core_1.Token.asString(version.minorVersion)}.${core_1.Token.asString(version.releaseVersion)}`;
                // WHEN
                new aws_ecs_1.ContainerDefinition(taskDefStack, 'ContainerDef', {
                    image,
                    taskDefinition,
                    memoryReservationMiB: 1024,
                });
                // THEN
                assert_1.expect(taskDefStack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                    ContainerDefinitions: assert_1.arrayWith(assert_1.objectLike({
                        Image: taskDefStack.resolve(expectedImage),
                    })),
                }));
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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