"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const __1 = require("../..");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
describe('RenderQueue', () => {
    let app;
    let dependencyStack;
    let stack;
    let vpc;
    let rcsImage;
    let images;
    let repository;
    let version;
    let renderQueueCommon;
    // GIVEN
    beforeEach(() => {
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DepStack');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'Vpc');
        version = new lib_1.VersionQuery(dependencyStack, 'Version');
        repository = new lib_1.Repository(dependencyStack, 'Repo', {
            version,
            vpc,
        });
        stack = new core_1.Stack(app, 'Stack');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        images = {
            remoteConnectionServer: rcsImage,
        };
        renderQueueCommon = new lib_1.RenderQueue(stack, 'RenderQueueCommon', {
            images,
            repository,
            version,
            vpc,
        });
    });
    afterEach(() => {
        jest.resetAllMocks();
    });
    test('creates cluster', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Cluster'));
    });
    test('creates service', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service'));
    });
    test('creates task definition', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition'));
    });
    test('closed ingress by default', () => {
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            // The openListener=true option would create an ingress rule in the listener's SG.
            // make sure that we don't have that.
            SecurityGroupIngress: assert_1.arrayWith(assert_1.objectLike({})),
        }));
    });
    test('creates load balancer with default values', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 1, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
        }));
    });
    test('creates a log group with default prefix of "/renderfarm/"', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: '/renderfarm/RenderQueueCommon',
            RetentionInDays: 3,
        }));
    });
    test('configure the container log driver', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                assert_1.objectLike({
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': {
                                'Fn::GetAtt': [
                                    'RenderQueueCommonLogGroupWrapperA0EF7057',
                                    'LogGroupName',
                                ],
                            },
                            'awslogs-stream-prefix': 'RCS',
                            'awslogs-region': { Ref: 'AWS::Region' },
                        },
                    },
                }),
            ],
        }));
    });
    test('child dependencies added', () => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Host', {
            vpc,
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        renderQueueCommon.addChildDependency(host);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: assert_1.arrayWith('RenderQueueCommonLBPublicListener935F5635', 'RenderQueueCommonRCSTask2A4D5EA5'),
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    describe('renderQueueSize.min', () => {
        describe('defaults to 1', () => {
            function assertSpecifiesMinSize(stackToAssert) {
                assert_1.expect(stackToAssert).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    MinSize: '1',
                }));
            }
            test('renderQueueSize unspecified', () => {
                // THEN
                assertSpecifiesMinSize(stack);
            });
            test('renderQueueSize.min unspecified', () => {
                // GIVEN
                const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', {
                    images,
                    repository,
                    version,
                    vpc,
                    renderQueueSize: {},
                });
                // THEN
                assertSpecifiesMinSize(isolatedStack);
            });
        });
        // Asserts that only one RCS container and ASG instance can be created.
        // Deadline currently requires that successive API requests are serviced by a single RCS.
        test.each([
            [0],
            [2],
        ])('clamps to 1 - using %d', (min) => {
            // GIVEN
            const props = {
                images,
                repository,
                version,
                vpc,
                renderQueueSize: {
                    min,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            })
                // THEN
                .toThrow('renderQueueSize.min');
        });
        // Asserts that when the renderQueueSize.min prop is specified, the underlying ASG's min property is set accordingly.
        test('configures minimum number of ASG instances', () => {
            // GIVEN
            const min = 1;
            const props = {
                images,
                repository,
                version,
                vpc,
                renderQueueSize: {
                    min,
                },
            };
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: min.toString(),
            }));
        });
    });
    describe('renderQueueSize.desired', () => {
        describe('defaults', () => {
            test('unset ASG desired', () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    DesiredCapacity: assert_1.ABSENT,
                }));
            });
        });
        test('caps at 1', () => {
            // GIVEN
            const props = {
                images,
                repository,
                version,
                vpc,
                renderQueueSize: {
                    desired: 2,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            })
                // THEN
                .toThrow('renderQueueSize.desired cannot be greater than 1');
        });
        describe('is specified', () => {
            const desired = 1;
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    renderQueueSize: {
                        desired,
                    },
                };
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('configures desired number of ASG instances', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    DesiredCapacity: desired.toString(),
                }));
            });
            // THEN
            test('configures desired number of ECS tasks in the service', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: desired,
                }));
            });
        });
    });
    describe('trafficEncryption', () => {
        describe('defaults', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    trafficEncryption: {},
                };
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('to HTTPS internally between ALB and RCS', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                }));
            });
            test('to HTTP externally between clients and ALB', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTP',
                    Port: 8080,
                }));
            });
        });
        describe('when interalProtocol is HTTPS', () => {
            let isolatedStack;
            let renderQueue;
            let caCertPemLogicalId;
            let caCertPkcsLogicalId;
            let caCertPkcsPassphraseLogicalId;
            beforeEach(() => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    trafficEncryption: {
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
                    },
                };
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                renderQueue = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
                caCertPemLogicalId = isolatedStack.getLogicalId(renderQueue.node.findChild('TlsCaCertPem').node.defaultChild);
                const caCertPkcs = renderQueue.node.findChild('TlsRcsCertBundle');
                const caCertPkcsPassphrase = caCertPkcs.node.findChild('Passphrase');
                caCertPkcsLogicalId = isolatedStack.getLogicalId(caCertPkcs.node.defaultChild);
                caCertPkcsPassphraseLogicalId = isolatedStack.getLogicalId(caCertPkcsPassphrase.node.defaultChild);
            });
            // THEN
            test('ALB connects with HTTPS to port 4433', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                }));
            });
            test('creates RCS cert', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
                    ServiceToken: {
                        'Fn::GetAtt': assert_1.arrayWith('Arn'),
                    },
                    DistinguishedName: { CN: 'renderfarm.local' },
                    Secret: {
                        NamePrefix: 'IsolatedStack/RenderQueue/TlsCaCertPem',
                    },
                }));
            });
            test('grants read access to secrets containing the certs and passphrase', () => {
                const taskDef = renderQueue.node.findChild('RCSTask');
                const taskRoleLogicalId = isolatedStack.getLogicalId(taskDef.taskRole.node.defaultChild);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    caCertPemLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    caCertPkcsLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: { Ref: caCertPkcsPassphraseLogicalId },
                        }),
                        Version: '2012-10-17',
                    },
                    Roles: assert_1.arrayWith({ Ref: taskRoleLogicalId }),
                }));
            });
            test('configures environment variables for cert secret URIs', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                    ContainerDefinitions: assert_1.arrayWith(assert_1.deepObjectLike({
                        Environment: assert_1.arrayWith({
                            Name: 'RCS_TLS_CA_CERT_URI',
                            Value: {
                                'Fn::GetAtt': [
                                    caCertPemLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Name: 'RCS_TLS_CERT_URI',
                            Value: {
                                'Fn::GetAtt': [
                                    caCertPkcsLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Name: 'RCS_TLS_CERT_PASSPHRASE_URI',
                            Value: { Ref: caCertPkcsPassphraseLogicalId },
                        }),
                    })),
                }));
            });
        });
        describe('when internal protocol is HTTP', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    trafficEncryption: {
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                    },
                };
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('no certs are created', () => {
                assert_1.expect(isolatedStack).notTo(assert_1.haveResource('Custom::RFDK_X509Generator'));
            });
            test('ALB connects with HTTP to port 8080', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTP',
                    Port: 8080,
                }));
            });
        });
        describe('externalProtocol is HTTPS', () => {
            let isolatedStack;
            const CERT_ARN = 'certarn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            const ZONE_NAME = 'renderfarm.local';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert', CA_ARN),
                        },
                    },
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('sets the listener port to 4433', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Port: 4433,
                }));
            });
            test('sets the listener protocol to HTTPS', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                }));
            });
            test('configures the ALB listener to use the specified ACM certificate', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                    Certificates: assert_1.arrayWith({
                        CertificateArn: CERT_ARN,
                    }),
                }));
            });
            test('raises an error when a cert is specified without a hostname', () => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Cert', 'certArn'),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert2', CA_ARN),
                        },
                    },
                };
                // WHEN
                expect(() => {
                    new lib_1.RenderQueue(stack, 'RenderQueue', props);
                })
                    // THEN
                    .toThrow(/A hostname must be provided when the external protocol is HTTPS/);
            });
        });
        describe('externalProtocol is HTTPS importing cert', () => {
            let isolatedStack;
            let zone;
            const ZONE_NAME = 'renderfarm.local';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const caCert = new __1.X509CertificatePem(isolatedStack, 'CaCert', {
                    subject: {
                        cn: `ca.${ZONE_NAME}`,
                    },
                });
                const serverCert = new __1.X509CertificatePem(isolatedStack, 'ServerCert', {
                    subject: {
                        cn: `server.${ZONE_NAME}`,
                    },
                    signingCertificate: caCert,
                });
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            rfdkCertificate: serverCert,
                        },
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                    },
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('sets the listener port to 4433', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Port: 4433,
                }));
            });
            test('sets the listener protocol to HTTPS', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                }));
            });
            test('Imports Cert to ACM', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
                    X509CertificatePem: {
                        Cert: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'Cert',
                            ],
                        },
                        Key: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'Key',
                            ],
                        },
                        Passphrase: {
                            Ref: 'ServerCertPassphraseE4C3CB38',
                        },
                        CertChain: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                }));
            });
        });
        test('Throws if given ACM cert and RFDK Cert', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certArn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const caCert = new __1.X509CertificatePem(isolatedStack, 'CaCert', {
                subject: {
                    cn: `ca.${ZONE_NAME}`,
                },
            });
            const serverCert = new __1.X509CertificatePem(isolatedStack, 'ServerCert', {
                subject: {
                    cn: `server.${ZONE_NAME}`,
                },
                signingCertificate: caCert,
            });
            const props = {
                images,
                repository,
                version,
                vpc,
                trafficEncryption: {
                    externalTLS: {
                        acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(isolatedStack, 'Certificate', CERT_ARN),
                        acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(isolatedStack, 'CA_Cert', CA_ARN),
                        rfdkCertificate: serverCert,
                    },
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS/);
        });
        test('Throws if no Cert given', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const props = {
                images,
                repository,
                version,
                vpc,
                trafficEncryption: {
                    externalTLS: {},
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS/);
        });
        test('Throws if ACM Cert is given without a cert chain', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certArn';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const props = {
                images,
                repository,
                version,
                vpc,
                trafficEncryption: {
                    externalTLS: {
                        acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(isolatedStack, 'Certificate', CERT_ARN),
                    },
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate./);
        });
    });
    describe('Client Connection', () => {
        describe('externalProtocol is http', () => {
            let isolatedStack;
            let zone;
            const ZONE_NAME = 'renderfarm.local';
            let rq;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                rq = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('ECS can connect', () => {
                // WHEN
                const hosts = [new aws_ec2_1.Instance(isolatedStack, 'Host', {
                        vpc,
                        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                    })];
                const role = new aws_iam_1.Role(isolatedStack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
                const env = rq.configureClientECS({
                    hosts,
                    grantee: role,
                });
                // THEN
                expect(env).toHaveProperty('RENDER_QUEUE_URI');
                expect(env.RENDER_QUEUE_URI).toMatch(/http:\/\/.*:8080$/);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(hosts[0].connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('Linux Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            "#!/bin/bash\nmkdir -p $(dirname '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\')\naws s3 cp \'s3://',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' \'/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\'\n' +
                                'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                                '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                                'fi\n' +
                                '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '" --render-queue "http://',
                            {
                                'Fn::GetAtt': [
                                    'RenderQueueLB235D35F4',
                                    'DNSName',
                                ],
                            },
                            ':8080" \n' +
                                'rm -f "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                                '  service deadline10launcher restart\n' +
                                'fi',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('Windows Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_CORE_BASE),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' ) -ea 0\n' +
                                'Read-S3Object -BucketName \'',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '\' -key \'',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -file \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -ErrorAction Stop\n' +
                                '$ErrorActionPreference = "Stop"\n' +
                                '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                                '& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '" --render-queue "http://',
                            {
                                'Fn::GetAtt': [
                                    'RenderQueueLB235D35F4',
                                    'DNSName',
                                ],
                            },
                            ':8080"  2>&1\n' +
                                'Remove-Item -Path "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {\n' +
                                '  Restart-Service "deadline10launcherservice"\n' +
                                '} Else {\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" 2>&1\n' +
                                '}</powershell>',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
        });
        describe('externalProtocol is https', () => {
            let isolatedStack;
            let zone;
            let rq;
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'arn:a:b:c:dcertarn';
            const CA_ARN = 'arn:aws:secretsmanager:123456789012:secret:ca/arn';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    hostname: {
                        zone,
                    },
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert', CA_ARN),
                        },
                    },
                };
                // WHEN
                rq = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('ECS can connect', () => {
                // WHEN
                const hosts = [new aws_ec2_1.Instance(isolatedStack, 'Host', {
                        vpc,
                        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                    })];
                const role = new aws_iam_1.Role(isolatedStack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
                const env = rq.configureClientECS({
                    hosts,
                    grantee: role,
                });
                // THEN
                expect(env).toHaveProperty('RENDER_QUEUE_URI');
                expect(env.RENDER_QUEUE_URI).toMatch(/https:\/\/.*:4433$/);
                expect(env).toHaveProperty('RENDER_QUEUE_TLS_CA_CERT_URI', CA_ARN);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(hosts[0].connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
            test('Linux Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            "#!/bin/bash\nmkdir -p $(dirname '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\')\naws s3 cp \'s3://',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' \'/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\'\n' +
                                'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                                '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                                'fi\n' +
                                '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '" --render-queue "https://',
                            {
                                'Fn::GetAtt': [
                                    'RenderQueueLB235D35F4',
                                    'DNSName',
                                ],
                            },
                            `:4433" --tls-ca "${CA_ARN}"\n` +
                                'rm -f "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                                '  service deadline10launcher restart\n' +
                                'fi',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
            test('Windows Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_CORE_BASE),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' ) -ea 0\n' +
                                'Read-S3Object -BucketName \'',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '\' -key \'',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -file \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -ErrorAction Stop\n' +
                                '$ErrorActionPreference = "Stop"\n' +
                                '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                                '& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '" --render-queue "https://',
                            {
                                'Fn::GetAtt': [
                                    'RenderQueueLB235D35F4',
                                    'DNSName',
                                ],
                            },
                            `:4433" --tls-ca \"${CA_ARN}\" 2>&1\n` +
                                'Remove-Item -Path "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {\n' +
                                '  Restart-Service "deadline10launcherservice"\n' +
                                '} Else {\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" 2>&1\n' +
                                '}</powershell>',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
        });
    });
    test('can specify subnets', () => {
        // GIVEN
        const subnets = [
            aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
                subnetId: 'SubnetID1',
                availabilityZone: 'us-west-2a',
            }),
            aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
                subnetId: 'SubnetID2',
                availabilityZone: 'us-west-2b',
            }),
        ];
        const props = {
            images,
            repository,
            version,
            vpc,
            vpcSubnets: {
                subnets,
            },
            vpcSubnetsAlb: {
                subnets,
            },
        };
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: assert_1.arrayWith('SubnetID1', 'SubnetID2'),
        }));
        assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                'SubnetID1',
                'SubnetID2',
            ],
        }));
    });
    test('can specify instance type', () => {
        // GIVEN
        const props = {
            images,
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            repository,
            version,
            vpc,
        };
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'c5.large',
        }));
    });
    test('no deletion protection', () => {
        // GIVEN
        const props = {
            images,
            repository,
            version,
            vpc,
            deletionProtection: false,
        };
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.not(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'deletion_protection.enabled',
                Value: 'true',
            }),
            Scheme: assert_1.ABSENT,
            Type: assert_1.ABSENT,
        })));
    });
    test('drop invalid http header fields enabled', () => {
        // GIVEN
        const props = {
            images,
            repository,
            version,
            vpc,
        };
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'routing.http.drop_invalid_header_fields.enabled',
                Value: 'true',
            }),
        }));
    });
    describe('hostname', () => {
        // GIVEN
        const zoneName = 'mydomain.local';
        describe('not specified', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('does not create a record set', () => {
                assert_1.expect(isolatedStack).notTo(assert_1.haveResource('AWS::Route53::RecordSet'));
            });
        });
        describe('specified with zone but no hostname', () => {
            let zone;
            let isolatedStack;
            let renderQueue;
            beforeEach(() => {
                // GIVEN
                zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                    vpc,
                    zoneName,
                });
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                renderQueue = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('creates a record set using default hostname', () => {
                const loadBalancerLogicalId = dependencyStack.getLogicalId(renderQueue.loadBalancer.node.defaultChild);
                assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
                    Name: `renderqueue.${zoneName}.`,
                    Type: 'A',
                    AliasTarget: assert_1.objectLike({
                        HostedZoneId: {
                            'Fn::GetAtt': [
                                loadBalancerLogicalId,
                                'CanonicalHostedZoneID',
                            ],
                        },
                    }),
                }));
            });
        });
        test.each([
            ['rq.somedomain.local'],
            ['1startswithnumber'],
            ['-startswithhyphen'],
            ['endswith-'],
            ['contains$symbol'],
            ['thisstringisexactlysixtyfourcharacterswhichisonelargerthanthemax'],
        ])('.hostname validation - %s', (hostname) => {
            // GIVEN
            const zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                zoneName: 'somedomain.local',
                vpc,
            });
            const props = {
                images,
                repository,
                version,
                vpc,
                hostname: {
                    hostname,
                    zone,
                },
            };
            // WHEN
            function when() {
                new lib_1.RenderQueue(stack, 'NewRenderQueue', props);
            }
            // THEN
            expect(when).toThrow(/Invalid RenderQueue hostname/);
        });
    });
    describe('Access Logs', () => {
        let isolatedStack;
        let isolatedVpc;
        let isolatedRepository;
        let isolatedVersion;
        let isolatedimages;
        let accessBucket;
        beforeEach(() => {
            // GIVEN
            isolatedStack = new core_1.Stack(app, 'IsolatedStack', {
                env: {
                    region: 'us-east-1',
                },
            });
            isolatedVpc = new aws_ec2_1.Vpc(isolatedStack, 'Vpc');
            isolatedVersion = new lib_1.VersionQuery(isolatedStack, 'Version');
            isolatedRepository = new lib_1.Repository(isolatedStack, 'Repo', {
                version: isolatedVersion,
                vpc: isolatedVpc,
            });
            isolatedimages = {
                remoteConnectionServer: rcsImage,
            };
            accessBucket = new aws_s3_1.Bucket(isolatedStack, 'AccessBucket');
        });
        test('enabling access logs sets attributes and policies', () => {
            // GIVEN
            const props = {
                images: isolatedimages,
                repository: isolatedRepository,
                version: isolatedVersion,
                vpc: isolatedVpc,
                accessLogs: {
                    destinationBucket: accessBucket,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assert_1.arrayWith({
                    Key: 'access_logs.s3.enabled',
                    Value: 'true',
                }, {
                    Key: 'access_logs.s3.bucket',
                    Value: {
                        Ref: 'AccessBucketE2803D76',
                    },
                }),
            }));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                Bucket: {
                    Ref: 'AccessBucketE2803D76',
                },
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 's3:PutObject',
                        Condition: {
                            StringEquals: {
                                's3:x-amz-acl': 'bucket-owner-full-control',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }, {
                        Action: 's3:GetBucketAcl',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::GetAtt': [
                                'AccessBucketE2803D76',
                                'Arn',
                            ],
                        },
                    }, {
                        Action: [
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::127311923021:root',
                                    ],
                                ],
                            },
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/AWSLogs/',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }),
                },
            }));
        });
        test('enabling access logs works with prefix', () => {
            // GIVEN
            const props = {
                images: isolatedimages,
                repository: isolatedRepository,
                version: isolatedVersion,
                vpc: isolatedVpc,
                accessLogs: {
                    destinationBucket: accessBucket,
                    prefix: 'PREFIX_STRING',
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assert_1.arrayWith({
                    Key: 'access_logs.s3.enabled',
                    Value: 'true',
                }, {
                    Key: 'access_logs.s3.bucket',
                    Value: {
                        Ref: 'AccessBucketE2803D76',
                    },
                }, {
                    Key: 'access_logs.s3.prefix',
                    Value: 'PREFIX_STRING',
                }),
            }));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                Bucket: {
                    Ref: 'AccessBucketE2803D76',
                },
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 's3:PutObject',
                        Condition: {
                            StringEquals: {
                                's3:x-amz-acl': 'bucket-owner-full-control',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }, {
                        Action: 's3:GetBucketAcl',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::GetAtt': [
                                'AccessBucketE2803D76',
                                'Arn',
                            ],
                        },
                    }, {
                        Action: [
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::127311923021:root',
                                    ],
                                ],
                            },
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/PREFIX_STRING/AWSLogs/',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }),
                },
            }));
        });
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'RenderQueue',
            createConstruct: () => {
                return stack;
            },
            resourceTypeCounts: {
                'AWS::ECS::Cluster': 1,
                'AWS::EC2::SecurityGroup': 2,
                'AWS::IAM::Role': 7,
                'AWS::AutoScaling::AutoScalingGroup': 1,
                'AWS::Lambda::Function': 3,
                'AWS::SNS::Topic': 1,
                'AWS::ECS::TaskDefinition': 1,
                'AWS::DynamoDB::Table': 2,
                'AWS::SecretsManager::Secret': 2,
                'AWS::ElasticLoadBalancingV2::LoadBalancer': 1,
                'AWS::ElasticLoadBalancingV2::TargetGroup': 1,
                'AWS::ECS::Service': 1,
            },
        });
    });
    describe('SEP Policies', () => {
        test('with resource tracker', () => {
            renderQueueCommon.addSEPPolicies();
            assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
                ManagedPolicyArns: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineSpotEventPluginAdminPolicy',
                        ],
                    ],
                }, {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineResourceTrackerAdminPolicy',
                        ],
                    ],
                }),
            }));
        });
        test('no resource tracker', () => {
            renderQueueCommon.addSEPPolicies(false);
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineSpotEventPluginAdminPolicy',
                        ],
                    ],
                }),
            }));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/AWSThinkboxDeadlineResourceTrackerAdminPolicy',
                        ],
                    ],
                }),
            }));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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