"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Version = void 0;
/**
 * This class is reposonsible to do basic operations on version format.
 *
 * @stability stable
 */
class Version {
    /**
     * @stability stable
     */
    constructor(components) {
        if (components.length != 4) {
            throw new Error('Invalid version format. Version should contain exactly 4 components.');
        }
        components.forEach((component) => {
            if (component < 0) {
                throw new RangeError('Invalid version format. None of the version components can be negative.');
            }
            if (!Number.isInteger(component)) {
                throw new RangeError('Invalid version format. None of the version components can contain decimal values.');
            }
        });
        this.components = components;
    }
    /**
     * This method parses the input string and returns the version object.
     *
     * @param version version string to parse.
     * @stability stable
     */
    static parse(version) {
        if (!Version.validateVersionFormat(version)) {
            throw new TypeError(`Invalid version format. Expected format 'a.b.c.d', found '${version}'`);
        }
        return new Version(version.split('.').map(x => parseInt(x)));
    }
    /**
     * This method validates the given string for a sequence '.' separated numbers.
     *
     * @param version the string to be validated.
     *
     * @returns true if the format is correct, else false.
     */
    static validateVersionFormat(version) {
        /**
         * Regex: ^\d+(?:\.\d+){3}$
         * Matches a sequence of '.' separated numbers with exactly 4 digits.
         * - ^ asserts position at start of a line.
         * - \d+ Matches one or more digits.
         * - (?:\.\d+) Matches a dot and the following one or more digits.
         * - {3} Matches previous pattern exactly 3 times.
         * - $ asserts position at the end of a line
         */
        if (version.match(/^\d+(?:\.\d+){3}$/)) {
            return true;
        }
        return false;
    }
    /**
     * The major version number.
     *
     * @stability stable
     * @inheritdoc true
     */
    get majorVersion() {
        return this.components[0];
    }
    /**
     * The minor version number.
     *
     * @stability stable
     * @inheritdoc true
     */
    get minorVersion() {
        return this.components[1];
    }
    /**
     * The release version number.
     *
     * @stability stable
     * @inheritdoc true
     */
    get releaseVersion() {
        return this.components[2];
    }
    /**
     * The patch version number.
     *
     * @stability stable
     * @inheritdoc true
     */
    get patchVersion() {
        return this.components[3];
    }
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is greater than the provided version;
     * false if this version is less than or equal to the provided verison.
     * @stability stable
     */
    isGreaterThan(version) {
        return this.compare(version) > 0;
    }
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is less than the provided version;
     * false if this version is greater than or equal to the provided verison.
     * @stability stable
     */
    isLessThan(version) {
        return this.compare(version) < 0;
    }
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is equal to the provided version;
     * false otherwise.
     * @stability stable
     */
    isEqual(version) {
        return this.compare(version) == 0;
    }
    /**
     * The method returns the version components in dot separated string format.
     *
     * @stability stable
     */
    toString() {
        return this.components.join('.');
    }
    /**
     * This method compares 2 versions.
     *
     * @param version version to compare
     *
     * @returns negative value if this version is less than the provided version;
     * 0 if both the versions are equal;
     * positive value if this version is greater than the provided verison.
     */
    compare(version) {
        if (this.components.length != version.components.length) {
            throw new TypeError('Component count in both the versions should be same.');
        }
        for (let i = 0; i < version.components.length; i++) {
            const diff = this.components[i] - version.components[i];
            if (diff != 0) {
                return diff;
            }
        }
        return 0;
    }
}
exports.Version = Version;
/**
 * This variable holds the value for minimum supported deadline version.
 *
 * @stability stable
 */
Version.MINIMUM_SUPPORTED_DEADLINE_VERSION = new Version([10, 1, 9, 2]);
//# sourceMappingURL=data:application/json;base64,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