"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ThinkboxDockerRecipes = exports.ThinkboxManagedDeadlineDockerRecipes = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
/**
 * An enum that is associated with AWS Thinkbox managed recipes that are available in the stage manifest.
 *
 * @stability stable
 */
var ThinkboxManagedDeadlineDockerRecipes;
(function (ThinkboxManagedDeadlineDockerRecipes) {
    ThinkboxManagedDeadlineDockerRecipes["REMOTE_CONNECTION_SERVER"] = "rcs";
    ThinkboxManagedDeadlineDockerRecipes["LICENSE_FORWARDER"] = "license-forwarder";
})(ThinkboxManagedDeadlineDockerRecipes = exports.ThinkboxManagedDeadlineDockerRecipes || (exports.ThinkboxManagedDeadlineDockerRecipes = {}));
/**
 * An API for interacting with staged Deadline Docker recipes provided by AWS Thinkbox.
 *
 * This provides container images as required by RFDK's Deadline constructs such as
 *
 * * {@link @aws-rfdk/deadline#RenderQueue}
 * * {@link @aws-rfdk/deadline#UsageBasedLicensing}
 *
 * @stability stable
 * @example
 *
 * Construct a RenderQueue
 *
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { RenderQueue, Repository, ThinkboxDockerRecipes } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(app, stack);
 * const recipes = new ThinkboxDockerRecipes(stack, 'Recipes', {
 * path: '/path/to/staged/recipes'
 * });
 * const repository = new Repository(stack, 'Repository', {
 * vpc,
 * version: recipes.version
 * });
 *
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 * images: recipes.renderQueueImages,
 * // ...
 * });
 */
class ThinkboxDockerRecipes extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.version = props.stage.getVersion(this, 'Version');
        for (const recipe of [ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER, ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER]) {
            if (!props.stage.manifest.recipes[recipe]) {
                throw new Error(`Could not find ${recipe} recipe`);
            }
        }
        this.remoteConnectionServer = props.stage.imageFromRecipe(this, 'RemoteConnectionServer', ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER.toString());
        this.licenseForwarder = props.stage.imageFromRecipe(this, 'LicenseForwarder', ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER.toString());
        this.renderQueueImages = {
            remoteConnectionServer: aws_ecs_1.ContainerImage.fromDockerImageAsset(this.remoteConnectionServer),
        };
        this.ublImages = {
            licenseForwarder: aws_ecs_1.ContainerImage.fromDockerImageAsset(this.licenseForwarder),
        };
    }
}
exports.ThinkboxDockerRecipes = ThinkboxDockerRecipes;
//# sourceMappingURL=data:application/json;base64,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