"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ThinkboxDockerImages = exports.AwsThinkboxEulaAcceptance = void 0;
const crypto_1 = require("crypto");
const path = require("path");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
/**
 * Choices for signifying the user's stance on the terms of the AWS Thinkbox End-User License Agreement (EULA).
 *
 * See: https://www.awsthinkbox.com/end-user-license-agreement
 *
 * @stability stable
 */
var AwsThinkboxEulaAcceptance;
(function (AwsThinkboxEulaAcceptance) {
    AwsThinkboxEulaAcceptance[AwsThinkboxEulaAcceptance["USER_REJECTS_AWS_THINKBOX_EULA"] = 0] = "USER_REJECTS_AWS_THINKBOX_EULA";
    AwsThinkboxEulaAcceptance[AwsThinkboxEulaAcceptance["USER_ACCEPTS_AWS_THINKBOX_EULA"] = 1] = "USER_ACCEPTS_AWS_THINKBOX_EULA";
})(AwsThinkboxEulaAcceptance = exports.AwsThinkboxEulaAcceptance || (exports.AwsThinkboxEulaAcceptance = {}));
/**
 * An API for interacting with publicly available Deadline container images published by AWS Thinkbox.
 *
 * This provides container images as required by RFDK's Deadline constructs such as
 *
 * * {@link @aws-rfdk/deadline#RenderQueue}
 * * {@link @aws-rfdk/deadline#UsageBasedLicensing}
 *
 * Successful usage of the published Deadline container images with this class requires:
 *
 * 1) Explicit acceptance of the terms of the AWS Thinkbox End User License Agreement, under which Deadline is
 *     distributed; and
 * 2) The lambda on which the custom resource looks up the Thinkbox container images is able to make HTTPS
 *     requests to the official AWS Thinbox download site: https://downloads.thinkboxsoftware.com
 *
 * Resources Deployed
 * ------------------------
 * - A Lambda function containing a script to look up the AWS Thinkbox container image registry
 *
 * Security Considerations
 * ------------------------
 * - CDK deploys the code for this lambda as an S3 object in the CDK bootstrap bucket. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly
 *    recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS
 *    CloudTrail on your account to assist in post-incident analysis of compromised production environments.
 *
 * For example, to construct a RenderQueue using the images:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { AwsThinkboxEulaAcceptance, RenderQueue, Repository, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *    version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *    version,
 *    // Change this to AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA to accept the terms
 *    // of the AWS Thinkbox End User License Agreement
 *    userAwsThinkboxEulaAcceptance: AwsThinkboxEulaAcceptance.USER_REJECTS_AWS_THINKBOX_EULA,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *    vpc,
 *    version,
 * });
 *
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *    images: images.forRenderQueue(),
 *    // ...
 * });
 * ```
 *
 * @stability stable
 */
class ThinkboxDockerImages extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.userAwsThinkboxEulaAcceptance = props.userAwsThinkboxEulaAcceptance;
        this.version = props === null || props === void 0 ? void 0 : props.version;
        const lambdaCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '08553416-1fc9-4be9-a818-609a31ae1b5b',
            description: 'Used by the ThinkboxDockerImages construct to look up the ECR repositories where AWS Thinkbox publishes Deadline container images.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'ecr-provider.handler',
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const ecrProvider = new core_1.CustomResource(this, 'ThinkboxEcrProvider', {
            serviceToken: lambdaFunc.functionArn,
            properties: {
                // create a random string that will force the Lambda to "update" on each deployment. Changes to its output will
                // be propagated to any CloudFormation resource providers that reference the output ARN
                ForceRun: this.forceRun(),
            },
            resourceType: 'Custom::RFDK_EcrProvider',
        });
        this.node.defaultChild = ecrProvider;
        this.ecrBaseURI = ecrProvider.getAtt('EcrURIPrefix').toString();
        this.remoteConnectionServer = this.ecrImageForRecipe(_1.ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER);
        this.licenseForwarder = this.ecrImageForRecipe(_1.ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        const errors = [];
        // Users must accept the AWS Thinkbox EULA to use the container images
        if (this.userAwsThinkboxEulaAcceptance !== AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA) {
            errors.push(ThinkboxDockerImages.AWS_THINKBOX_EULA_MESSAGE);
        }
        return errors;
    }
    ecrImageForRecipe(recipe) {
        let registryName = `${this.ecrBaseURI}${recipe}`;
        if (this.versionString) {
            registryName += `:${this.versionString}`;
        }
        return aws_ecs_1.ContainerImage.fromRegistry(registryName);
    }
    /**
     * Returns container images for use with the {@link RenderQueue} construct.
     *
     * @stability stable
     */
    forRenderQueue() {
        return this;
    }
    /**
     * Returns container images for use with the {@link UsageBasedLicensing} construct.
     *
     * @stability stable
     */
    forUsageBasedLicensing() {
        return this;
    }
    /**
     * A string representation of the Deadline version to retrieve images for.
     *
     * This can be undefined - in which case the latest available version of Deadline is used.
     */
    get versionString() {
        function numAsString(num) {
            return core_1.Token.isUnresolved(num) ? core_1.Token.asString(num) : num.toString();
        }
        const version = this.version;
        if (version) {
            const major = numAsString(version.majorVersion);
            const minor = numAsString(version.minorVersion);
            const release = numAsString(version.releaseVersion);
            return `${major}.${minor}.${release}`;
        }
        return undefined;
    }
    forceRun() {
        return crypto_1.randomBytes(32).toString('base64').slice(0, 32);
    }
}
exports.ThinkboxDockerImages = ThinkboxDockerImages;
/**
 * The AWS Thinkbox licensing message that is presented to the user if they create an instance of
 * this class without explicitly accepting the AWS Thinkbox EULA.
 *
 * Note to developers: The text of this string is a legal requirement, and must not be altered
 * witout approval.
 */
ThinkboxDockerImages.AWS_THINKBOX_EULA_MESSAGE = `
The ThinkboxDockerImages will install Deadline onto one or more EC2 instances.

Deadline is provided by AWS Thinkbox under the AWS Thinkbox End User License
Agreement (EULA). By installing Deadline, you are agreeing to the terms of this
license. Follow the link below to read the terms of the AWS Thinkbox EULA.

https://www.awsthinkbox.com/end-user-license-agreement

By using the ThinkboxDockerImages to install Deadline you agree to the terms of
the AWS Thinkbox EULA.

Please set the userAwsThinkboxEulaAcceptance property to
USER_ACCEPTS_AWS_THINKBOX_EULA to signify your acceptance of the terms of the
AWS Thinkbox EULA.
`;
//# sourceMappingURL=data:application/json;base64,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