"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueue = void 0;
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_ecs_patterns_1 = require("@aws-cdk/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const rq_connection_1 = require("./rq-connection");
/**
 * Base class for Render Queue providers
 */
class RenderQueueBase extends core_1.Construct {
}
/**
 * The RenderQueue construct deploys an Elastic Container Service (ECS) service that serves Deadline's REST HTTP API to Deadline Clients.
 *
 * Most Deadline clients will connect to a Deadline render farm via the the RenderQueue. The API provides Deadline
 * clients access to Deadline's database and repository file-system in a way that is secure, performant, and scalable.
 *
 * Resources Deployed
 * ------------------------
 * - An Amazon Elastic Container Service (ECS) cluster.
 * - An AWS EC2 auto-scaling group that provides the instances that host the ECS service.
 * - An ECS service with a task definition that deploys the Deadline Remote Connetion Server (RCS) in a container.
 * - A Amazon CloudWatch log group for streaming logs from the Deadline RCS.
 * - An application load balancer, listener and target group that balance incoming traffic among the RCS containers.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Care must be taken to secure what can connect to the RenderQueue. The RenderQueue does not authenticate API
 *    requests made against it. You must limit access to the RenderQueue endpoint to only trusted hosts. Those hosts
 *    should be governed carefully, as malicious software could use the API to remotely execute code across the entire render farm.
 * - The RenderQueue can be deployed with network encryption through Transport Layer Security (TLS) or without it. Unencrypted
 *    network communications can be eavesdropped upon or modified in transit. We strongly recommend deploying the RenderQueue
 *    with TLS enabled in production environments.
 *
 * @stability stable
 */
class RenderQueue extends RenderQueueBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        super(scope, id);
        // The RCS does not currently support horizontal scaling behind a load-balancer, so we limit to at most one instance
        if (props.renderQueueSize && props.renderQueueSize.min !== undefined && props.renderQueueSize.min > 1) {
            throw new Error(`renderQueueSize.min cannot be greater than 1 - got ${props.renderQueueSize.min}`);
        }
        if (props.renderQueueSize && props.renderQueueSize.desired !== undefined && props.renderQueueSize.desired > 1) {
            throw new Error(`renderQueueSize.desired cannot be greater than 1 - got ${props.renderQueueSize.desired}`);
        }
        let externalProtocol;
        if ((_a = props.trafficEncryption) === null || _a === void 0 ? void 0 : _a.externalTLS) {
            externalProtocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
            if ((props.trafficEncryption.externalTLS.acmCertificate === undefined) ===
                (props.trafficEncryption.externalTLS.rfdkCertificate === undefined)) {
                throw new Error('Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS.');
            }
            else if (props.trafficEncryption.externalTLS.rfdkCertificate) {
                if (props.trafficEncryption.externalTLS.rfdkCertificate.certChain === undefined) {
                    throw new Error('Provided rfdkCertificate does not contain a certificate chain.');
                }
                this.clientCert = new core_2.ImportedAcmCertificate(this, 'AcmCert', props.trafficEncryption.externalTLS.rfdkCertificate);
                this.certChain = props.trafficEncryption.externalTLS.rfdkCertificate.certChain;
            }
            else {
                if (props.trafficEncryption.externalTLS.acmCertificateChain === undefined) {
                    throw new Error('externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate.');
                }
                this.clientCert = props.trafficEncryption.externalTLS.acmCertificate;
                this.certChain = props.trafficEncryption.externalTLS.acmCertificateChain;
            }
        }
        else {
            externalProtocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP;
        }
        const internalProtocol = (_c = (_b = props.trafficEncryption) === null || _b === void 0 ? void 0 : _b.internalProtocol) !== null && _c !== void 0 ? _c : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS && !props.hostname) {
            throw new Error('A hostname must be provided when the external protocol is HTTPS');
        }
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            vpc: props.vpc,
        });
        const minCapacity = (_e = (_d = props.renderQueueSize) === null || _d === void 0 ? void 0 : _d.min) !== null && _e !== void 0 ? _e : 1;
        if (minCapacity < 1) {
            throw new Error(`renderQueueSize.min capacity must be at least 1: got ${minCapacity}`);
        }
        this.asg = this.cluster.addCapacity('RCS Capacity', {
            vpcSubnets: (_f = props.vpcSubnets) !== null && _f !== void 0 ? _f : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            instanceType: (_g = props.instanceType) !== null && _g !== void 0 ? _g : new aws_ec2_1.InstanceType('c5.large'),
            minCapacity,
            desiredCapacity: (_h = props.renderQueueSize) === null || _h === void 0 ? void 0 : _h.desired,
            maxCapacity: 1,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    // See: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-ami-storage-config.html
                    // We want the volume to be encrypted. The default AMI size is 30-GiB.
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            updateType: aws_autoscaling_1.UpdateType.ROLLING_UPDATE,
        });
        /**
         * The ECS-optimized AMI that is defaulted to when adding capacity to a cluster does not include the awscli or unzip
         * packages as is the case with the standard Amazon Linux AMI. These are required by RFDK scripts to configure the
         * direct connection on the host container instances.
         */
        this.asg.userData.addCommands('yum install -yq awscli unzip');
        const externalPortNumber = RenderQueue.RCS_PROTO_PORTS[externalProtocol];
        const internalPortNumber = RenderQueue.RCS_PROTO_PORTS[internalProtocol];
        this.logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, {
            logGroupPrefix: '/renderfarm/',
            ...props.logGroupProps,
        });
        this.logGroup.grantWrite(this.asg);
        const taskDefinition = this.createTaskDefinition({
            image: props.images.remoteConnectionServer,
            portNumber: internalPortNumber,
            protocol: internalProtocol,
            repository: props.repository,
        });
        this.taskDefinition = taskDefinition;
        // The fully-qualified domain name to use for the ALB
        let loadBalancerFQDN;
        if (props.hostname) {
            const label = (_j = props.hostname.hostname) !== null && _j !== void 0 ? _j : 'renderqueue';
            if (props.hostname.hostname && !RenderQueue.RE_VALID_HOSTNAME.test(label)) {
                throw new Error(`Invalid RenderQueue hostname: ${label}`);
            }
            loadBalancerFQDN = `${label}.${props.hostname.zone.zoneName}`;
        }
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', {
            vpc: this.cluster.vpc,
            vpcSubnets: (_k = props.vpcSubnetsAlb) !== null && _k !== void 0 ? _k : { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true },
            internetFacing: false,
            deletionProtection: (_l = props.deletionProtection) !== null && _l !== void 0 ? _l : true,
        });
        this.pattern = new aws_ecs_patterns_1.ApplicationLoadBalancedEc2Service(this, 'AlbEc2ServicePattern', {
            certificate: this.clientCert,
            cluster: this.cluster,
            desiredCount: (_m = props.renderQueueSize) === null || _m === void 0 ? void 0 : _m.desired,
            domainZone: (_o = props.hostname) === null || _o === void 0 ? void 0 : _o.zone,
            domainName: loadBalancerFQDN,
            listenerPort: externalPortNumber,
            loadBalancer,
            protocol: externalProtocol,
            taskDefinition,
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            // This is required to ensure that the ALB listener's security group does not allow any ingress by default.
            openListener: false,
        });
        // An explicit dependency is required from the Service to the Client certificate
        // Otherwise cloud formation will try to remove the cert before the ALB using it is disposed.
        if (this.clientCert) {
            this.pattern.node.addDependency(this.clientCert);
        }
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.pattern.service.node.addDependency(this.asg);
        this.loadBalancer = this.pattern.loadBalancer;
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        this.loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        if (props.accessLogs) {
            const accessLogsBucket = props.accessLogs.destinationBucket;
            // Policies are applied according to
            // https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:PutObject'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [`${accessLogsBucket.bucketArn}/*`],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                    },
                },
            }));
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetBucketAcl'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [accessLogsBucket.bucketArn],
            }));
            this.loadBalancer.logAccessLogs(accessLogsBucket, props.accessLogs.prefix);
        }
        // Ensure tasks are run on separate container instances
        this.pattern.service.addPlacementConstraints(aws_ecs_1.PlacementConstraint.distinctInstances());
        /**
         * Uses an escape-hatch to set the target group protocol to HTTPS. We cannot configure server certificate
         * validation, but at least traffic is encrypted and terminated at the application layer.
         */
        const listener = this.loadBalancer.node.findChild('PublicListener');
        this.listener = listener;
        const targetGroup = listener.node.findChild('ECSGroup');
        const targetGroupResource = targetGroup.node.defaultChild;
        targetGroupResource.protocol = aws_elasticloadbalancingv2_1.ApplicationProtocol[internalProtocol];
        targetGroupResource.port = internalPortNumber;
        this.grantPrincipal = taskDefinition.taskRole;
        this.connections = new aws_ec2_1.Connections({
            defaultPort: aws_ec2_1.Port.tcp(externalPortNumber),
            securityGroups: this.pattern.loadBalancer.connections.securityGroups,
        });
        this.endpoint = new core_2.ConnectableApplicationEndpoint({
            address: this.pattern.loadBalancer.loadBalancerDnsName,
            port: externalPortNumber,
            connections: this.connections,
            protocol: externalProtocol,
        });
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttp({
                endpoint: this.endpoint,
            });
        }
        else {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttps({
                endpoint: this.endpoint,
                caCert: this.certChain,
            });
        }
        this.node.defaultChild = taskDefinition;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Configures an ECS cluster to be able to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientECS(param) {
        param.hosts.forEach(host => this.addChildDependency(host));
        return this.rqConnection.configureClientECS(param);
    }
    /**
     * Configure an Instance/Autoscaling group to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientInstance(param) {
        this.addChildDependency(param.host);
        this.rqConnection.configureClientInstance(param);
    }
    /**
     * Adds AWS Managed Policies to the Render Queue so it is able to control Deadlines Spot Event Plugin.
     *
     * See: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html for additonal information.
     *
     * @param includeResourceTracker Whether or not the Resource tracker admin policy should also be addd (Default: True).
     * @stability stable
     */
    addSEPPolicies(includeResourceTracker = true) {
        const sepPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy');
        this.taskDefinition.taskRole.addManagedPolicy(sepPolicy);
        if (includeResourceTracker) {
            const rtPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy');
            this.taskDefinition.taskRole.addManagedPolicy(rtPolicy);
        }
    }
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the RenderQueue has been deployed and is ready to recieve traffic.
     *
     * This can be used to ensure that the RenderQueue is fully up and serving queries before a client attempts to connect to it.
     *
     * @param child The child to make dependent upon this RenderQueue.
     * @stability stable
     */
    addChildDependency(child) {
        // Narrowly define the dependencies to reduce the probability of cycles
        // ex: cycles that involve the security group of the RenderQueue & child.
        child.node.addDependency(this.listener);
        child.node.addDependency(this.taskDefinition);
    }
    createTaskDefinition(props) {
        const { image, portNumber, protocol, repository } = props;
        const taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'RCSTask');
        // Mount the repo filesystem to RenderQueue.HOST_REPO_FS_MOUNT_PATH
        const connection = repository.configureClientECS({
            containerInstances: {
                hosts: [this.asg],
            },
            containers: {
                taskDefinition,
            },
        });
        const environment = connection.containerEnvironment;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            // Generate a self-signed X509 certificate, private key and passphrase for use by the RCS containers.
            // Note: the Application Load Balancer does not validate the certificate in any way.
            const rcsCertPem = new core_2.X509CertificatePem(this, 'TlsCaCertPem', {
                subject: {
                    cn: 'renderfarm.local',
                },
            });
            const rcsCertPkcs = new core_2.X509CertificatePkcs12(this, 'TlsRcsCertBundle', {
                sourceCertificate: rcsCertPem,
            });
            [rcsCertPem.cert, rcsCertPkcs.cert, rcsCertPkcs.passphrase].forEach(secret => {
                secret.grantRead(taskDefinition.taskRole);
            });
            environment.RCS_TLS_CA_CERT_URI = rcsCertPem.cert.secretArn;
            environment.RCS_TLS_CERT_URI = rcsCertPkcs.cert.secretArn;
            environment.RCS_TLS_CERT_PASSPHRASE_URI = rcsCertPkcs.passphrase.secretArn;
            environment.RCS_TLS_REQUIRE_CLIENT_CERT = 'no';
        }
        const containerDefinition = taskDefinition.addContainer('ContainerDefinition', {
            image,
            memoryReservationMiB: 2048,
            environment,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup: this.logGroup,
                streamPrefix: 'RCS',
            }),
        });
        containerDefinition.addMountPoints(connection.readWriteMountPoint);
        // Increase ulimits
        containerDefinition.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        containerDefinition.addPortMappings({
            containerPort: portNumber,
            hostPort: portNumber,
        });
        return taskDefinition;
    }
}
exports.RenderQueue = RenderQueue;
/**
 * Container listening ports for each protocol.
 */
RenderQueue.RCS_PROTO_PORTS = {
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP]: 8080,
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS]: 4433,
};
/**
 * Regular expression that validates a hostname (portion in front of the subdomain).
 */
RenderQueue.RE_VALID_HOSTNAME = /^[a-z](?:[a-z0-9-]{0,61}[a-z0-9])?$/i;
//# sourceMappingURL=data:application/json;base64,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