"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('MongoDbPostInstall', () => {
    let stack;
    let vpc;
    let mongoDb;
    let pwUser1Arn;
    let pwUser2Arn;
    let pwUser1;
    let pwUser2;
    let x509User1Arn;
    let x509User2Arn;
    let x509User1;
    let x509User2;
    beforeEach(() => {
        const hostname = 'mongodb';
        const zoneName = 'testZone.internal';
        const version = lib_1.MongoDbVersion.COMMUNITY_3_6;
        const userSsplAcceptance = lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL;
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        const dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        const caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        const serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
        mongoDb = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        pwUser1Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User1';
        pwUser2Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User2';
        pwUser1 = aws_secretsmanager_1.Secret.fromSecretArn(stack, 'PwUser1', pwUser1Arn);
        pwUser2 = aws_secretsmanager_1.Secret.fromSecretArn(stack, 'PwUser2', pwUser2Arn);
        x509User1Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/X509User1';
        x509User2Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/X509User2';
        x509User1 = {
            certificate: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'x509User1', x509User1Arn),
            roles: JSON.stringify([{ role: 'readWrite', db: 'testdb1' }]),
        };
        x509User2 = {
            certificate: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'x509User2', x509User2Arn),
            roles: JSON.stringify([{ role: 'readWrite', db: 'testdb2' }]),
        };
    });
    test('created correctly: both user types', () => {
        // GIVEN
        const users = {
            passwordAuthUsers: [pwUser1, pwUser2],
            x509AuthUsers: [x509User1, x509User2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            mongoDb,
            users,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'mongodb.configureMongo',
            Environment: {
                Variables: {
                    DEBUG: 'false',
                },
            },
            Runtime: 'nodejs12.x',
            VpcConfig: {
                SecurityGroupIds: [
                    {
                        'Fn::GetAtt': [
                            'MongoPostInstallLambdaSecurityGroup62729E3A',
                            'GroupId',
                        ],
                    },
                ],
                SubnetIds: [
                    {
                        Ref: 'VpcPrivateSubnet1Subnet536B997A',
                    },
                    {
                        Ref: 'VpcPrivateSubnet2Subnet3788AAA1',
                    },
                ],
            },
        }));
        // Lambda role can get the required secrets.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'MongoDbInstanceAdminUser54147F2B',
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User1',
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User2',
                    },
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_MongoDbPostInstallSetup', {
            Properties: {
                Connection: {
                    Hostname: 'mongodb.testZone.internal',
                    Port: '27017',
                    CaCertificate: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'CertChain',
                        ],
                    },
                    Credentials: {
                        Ref: 'MongoDbInstanceAdminUser54147F2B',
                    },
                },
                PasswordAuthUsers: [
                    pwUser1Arn,
                    pwUser2Arn,
                ],
                X509AuthUsers: [
                    {
                        Certificate: x509User1Arn,
                        Roles: x509User1.roles,
                    },
                    {
                        Certificate: x509User2Arn,
                        Roles: x509User2.roles,
                    },
                ],
            },
            DependsOn: [
                'MongoDbInstanceServerAsgASG2643AD1D',
                'MongoPostInstallLambdaServiceRoleDefaultPolicy8B1C1CE8',
                'MongoPostInstallLambdaServiceRoleCD03B9B9',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('created correctly: only password users', () => {
        // GIVEN
        const users = {
            passwordAuthUsers: [pwUser1, pwUser2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            mongoDb,
            users,
        });
        // THEN
        // Lambda role can get the required secrets.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'MongoDbInstanceAdminUser54147F2B',
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User1',
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User2',
                    },
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_MongoDbPostInstallSetup', {
            Connection: {
                Hostname: 'mongodb.testZone.internal',
                Port: '27017',
                CaCertificate: {
                    'Fn::GetAtt': [
                        'ServerCert',
                        'CertChain',
                    ],
                },
                Credentials: {
                    Ref: 'MongoDbInstanceAdminUser54147F2B',
                },
            },
            PasswordAuthUsers: [
                pwUser1Arn,
                pwUser2Arn,
            ],
        }));
    });
    test('created correctly: only x509 users', () => {
        // GIVEN
        const users = {
            x509AuthUsers: [x509User1, x509User2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            mongoDb,
            users,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_MongoDbPostInstallSetup', {
            Connection: {
                Hostname: 'mongodb.testZone.internal',
                Port: '27017',
                CaCertificate: {
                    'Fn::GetAtt': [
                        'ServerCert',
                        'CertChain',
                    ],
                },
                Credentials: {
                    Ref: 'MongoDbInstanceAdminUser54147F2B',
                },
            },
            X509AuthUsers: [
                {
                    Certificate: x509User1Arn,
                    Roles: x509User1.roles,
                },
                {
                    Certificate: x509User2Arn,
                    Roles: x509User2.roles,
                },
            ],
        }));
    });
    test('use selected subnets', () => {
        // GIVEN
        const users = {
            passwordAuthUsers: [pwUser1, pwUser2],
            x509AuthUsers: [x509User1, x509User2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            vpcSubnets: { subnets: [vpc.privateSubnets[0]] },
            mongoDb,
            users,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'mongodb.configureMongo',
            VpcConfig: {
                SubnetIds: [
                    {
                        Ref: 'VpcPrivateSubnet1Subnet536B997A',
                    },
                ],
            },
        }));
    });
    test('assert bad x509 role', () => {
        // GIVEN
        const users = {
            x509AuthUsers: [
                {
                    certificate: x509User1.certificate,
                    roles: '}{',
                },
            ],
        };
        // THEN
        expect(() => {
            new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
                vpc,
                mongoDb,
                users,
            });
        }).toThrowError(/MongoDbPostInstallSetup: Could not parse JSON role for x509 user:/);
    });
});
//# sourceMappingURL=data:application/json;base64,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