"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const exporting_log_group_1 = require("../lib/exporting-log-group");
test('default exporting log group is created correctly', () => {
    const stack = new core_1.Stack();
    const bucket = new aws_s3_1.Bucket(stack, 'DestinationBucket', {
        bucketName: 'test-bucket',
    });
    // WHEN
    new exporting_log_group_1.ExportingLogGroup(stack, 'ExportingLogGroup', {
        bucketName: bucket.bucketName,
        logGroupName: 'logGroup',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        ServiceToken: {
            'Fn::GetAtt': [
                'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aFD4BFC8A',
                'Arn',
            ],
        },
        LogGroupName: 'logGroup',
        RetentionInDays: 3,
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'logs:DescribeExportTasks',
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'logs:CreateExportTask',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':logs:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':log-group:',
                                {
                                    'Fn::GetAtt': [
                                        'ExportingLogGroupLogRetention80FFADE8',
                                        'LogGroupName',
                                    ],
                                },
                                ':*',
                            ],
                        ],
                    },
                },
            ],
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
        ComparisonOperator: 'GreaterThanOrEqualToThreshold',
        EvaluationPeriods: 1,
        Dimensions: [
            {
                Name: 'FunctionName',
                Value: {
                    Ref: 'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eC741F553',
                },
            },
        ],
        MetricName: 'Errors',
        Namespace: 'AWS/Lambda',
        Period: 300,
        Statistic: 'Sum',
        Threshold: 1,
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
        ScheduleExpression: 'rate(1 hour)',
        State: 'ENABLED',
        Targets: [
            {
                Arn: {
                    'Fn::GetAtt': [
                        'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eC741F553',
                        'Arn',
                    ],
                },
                Id: 'Target0',
                Input: {
                    'Fn::Join': [
                        '',
                        [
                            '{\"BucketName\":\"',
                            {
                                Ref: 'DestinationBucket4BECDB47',
                            },
                            '\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"logGroup\",\"RetentionInHours\":72}',
                        ],
                    ],
                },
            },
        ],
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function'));
});
test('custom set retention is created correctly', () => {
    const stack = new core_1.Stack();
    const bucket = new aws_s3_1.Bucket(stack, 'DestinationBucket', {
        bucketName: 'test-bucket',
    });
    // WHEN
    new exporting_log_group_1.ExportingLogGroup(stack, 'ExportingLogGroup', {
        bucketName: bucket.bucketName,
        logGroupName: 'logGroup',
        retention: aws_logs_1.RetentionDays.ONE_WEEK,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        ServiceToken: {
            'Fn::GetAtt': [
                'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aFD4BFC8A',
                'Arn',
            ],
        },
        LogGroupName: 'logGroup',
        RetentionInDays: 7,
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function'));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm'));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule'));
});
//# sourceMappingURL=data:application/json;base64,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