"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticPrivateIpServer = void 0;
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
/**
 * This construct provides a single instance, provided by an Auto Scaling Group (ASG), that has an attached Elastic Network Interface (ENI) that is providing a private ip address.
 *
 * This ENI is automatically re-attached to the instance if the instance is replaced
 * by the ASG.
 *
 * The ENI provides an unchanging private IP address that can always be used to connect
 * to the instance regardless of how many times the instance has been replaced. Furthermore,
 * the ENI has a MAC address that remains unchanged unless the ENI is destroyed.
 *
 * Essentially, this provides an instance with an unchanging private IP address that will
 * automatically recover from termination. This instance is suitable for use as an application server,
 * such as a license server, that must always be reachable by the same IP address.
 *
 * Resources Deployed
 * ------------------------
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Elastic Network Interface (ENI).
 * - Security Group for the ASG.
 * - Instance Role and corresponding IAM Policy.
 * - SNS Topic & Role for instance-launch lifecycle events -- max one of each per stack.
 * - Lambda function, with role, to attach the ENI in response to instance-launch lifecycle events -- max one per stack.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda that is deployed through this construct has broad IAM permissions to attach any Elastic
 *    Network Interface (ENI) to any instance. You should not grant any additional actors/principals the ability
 *    to modify or execute this Lambda.
 * - The SNS Topic that is deployed through this construct controls the execution of the Lambda discussed above.
 *    Principals that can publish messages to this SNS Topic will be able to trigger the Lambda to run. You should
 *    not allow any additional principals to publish messages to this SNS Topic.
 *
 * @stability stable
 */
class StaticPrivateIpServer extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        if (props.resourceSignalTimeout && props.resourceSignalTimeout.toSeconds() > (12 * 60 * 60)) {
            throw new Error('Resource signal timeout cannot exceed 12 hours.');
        }
        this.autoscalingGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Asg', {
            minCapacity: 1,
            maxCapacity: 1,
            vpc: props.vpc,
            instanceType: props.instanceType,
            machineImage: props.machineImage,
            vpcSubnets: { subnets: [subnet] },
            blockDevices: props.blockDevices,
            keyName: props.keyName,
            resourceSignalCount: props.resourceSignalTimeout ? 1 : undefined,
            resourceSignalTimeout: props.resourceSignalTimeout,
            role: props.role,
            securityGroup: props.securityGroup,
            userData: props.userData,
        });
        this.connections = this.autoscalingGroup.connections;
        this.grantPrincipal = this.autoscalingGroup.grantPrincipal;
        this.osType = this.autoscalingGroup.osType;
        this.role = this.autoscalingGroup.role;
        this.userData = this.autoscalingGroup.userData;
        const scopePath = this.node.scopes.map(construct => construct.node.id).slice(1); // Slice to remove the unnamed <root> scope.
        const eni = new aws_ec2_1.CfnNetworkInterface(this, 'Eni', {
            subnetId: subnet.subnetId,
            description: `Static ENI for ${scopePath.join('/')}`,
            groupSet: core_1.Lazy.listValue({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
            privateIpAddress: props.privateIpAddress,
        });
        this.privateIpAddress = eni.attrPrimaryPrivateIpAddress;
        // We need to be sure that the ENI is created before the instance would be brought up; otherwise, we cannot attach it.
        this.autoscalingGroup.node.defaultChild.addDependsOn(eni);
        this.attachEniLifecyleTarget(eni);
        this.node.defaultChild = this.autoscalingGroup.node.defaultChild;
    }
    /**
     * Set up an instance launch lifecycle action that will attach the eni to the single instance in this construct's AutoScalingGroup when a new instance is launched.
     *
     * @stability stable
     */
    attachEniLifecyleTarget(eni) {
        // Note: The design of AutoScalingGroup life cycle notifications in CDK v1.49.1 is such that
        // using the provided AutoScalingGroup.addLifecycleHook() will result in a setup that misses
        // launch notifications for instances created when the ASG is created. This is because
        // it uses the separate CfnLifecycleHook resource to do it, and that resource references the
        // ASG ARN; i.e. it must be created after the ASG has an ARN... thus it can miss instance launches
        // when the ASG is first created.
        //
        // We work around this by using an escape-hatch to the L1 ASG to create our own notification from scratch.
        const eventHandler = this.setupLifecycleEventHandlerFunction();
        const { topic, role } = this.setupLifecycleNotificationTopic(eventHandler);
        // Ensure no race conditions that might prevent the lambda from being able to perform its required functions by making
        // the ASG depend on the creation of the SNS Subscription.
        // Note: The topic subscriptions are children of the lambda, and are given an id equal to the Topic's id.
        this.autoscalingGroup.node.defaultChild.node.addDependency(eventHandler.node.findChild(topic.node.id));
        this.autoscalingGroup.node.defaultChild.lifecycleHookSpecificationList = [
            {
                defaultResult: aws_autoscaling_1.DefaultResult.ABANDON,
                heartbeatTimeout: 120,
                lifecycleHookName: 'NewStaticPrivateIpServer',
                lifecycleTransition: aws_autoscaling_1.LifecycleTransition.INSTANCE_LAUNCHING,
                notificationTargetArn: topic.topicArn,
                roleArn: role.roleArn,
                notificationMetadata: JSON.stringify({ eniId: eni.ref }),
            },
        ];
    }
    /**
     * Create, or fetch, the lambda function that will process instance-start lifecycle events from this construct.
     *
     * @stability stable
     */
    setupLifecycleEventHandlerFunction() {
        const stack = core_1.Stack.of(this);
        // The SingletonFunction does not tell us when it's newly created vs. finding a pre-existing
        // one. So, we do our own singleton Function so that we know when it's the first creation, and, thus,
        // we must attach one-time permissions.
        const functionUniqueId = 'AttachEniToInstance' + this.removeHyphens('83a5dca5-db54-4aa4-85d2-8d419cdf85ce');
        let singletonPreExists = true;
        let eventHandler = stack.node.tryFindChild(functionUniqueId);
        if (!eventHandler) {
            const handlerCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'asg-attach-eni'), {
                exclude: ['**/*', '!index*'],
            });
            eventHandler = new aws_lambda_1.Function(stack, functionUniqueId, {
                code: handlerCode,
                handler: 'index.handler',
                runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                description: `Created by RFDK StaticPrivateIpServer to process instance launch lifecycle events in stack '${stack.stackName}'. This lambda attaches an ENI to newly launched instances.`,
                logRetention: aws_logs_1.RetentionDays.THREE_DAYS,
            });
            singletonPreExists = false;
        }
        // Note: We **cannot** reference the ASG's ARN in the lambda's policy. It would create a deadlock at deployment:
        //  Lambda policy waiting on ASG completion to get ARN
        //  -> lambda waiting on policy to be created
        //  -> ASG waiting on lambda to signal lifecycle continue for instance start
        //  -> back to the start of the cycle.
        // Instead we use resourcetags condition to limit the scope of the lambda.
        const tagKey = 'RfdkStaticPrivateIpServerGrantConditionKey';
        const tagValue = eventHandler.node.uniqueId;
        const grantCondition = {};
        grantCondition[`autoscaling:ResourceTag/${tagKey}`] = tagValue;
        core_1.Tags.of(this.autoscalingGroup).add(tagKey, tagValue);
        // Allow the lambda to complete the lifecycle action for only tagged ASGs.
        const iamCompleteLifecycle = new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'autoscaling:CompleteLifecycleAction',
            ],
            resources: [
                `arn:${stack.partition}:autoscaling:${stack.region}:${stack.account}:autoScalingGroup:*:autoScalingGroupName/*`,
            ],
            conditions: {
                'ForAnyValue:StringEquals': grantCondition,
            },
        });
        eventHandler.role.addToPolicy(iamCompleteLifecycle);
        if (!singletonPreExists) {
            // Allow the lambda to attach the ENI to the instance that was created.
            // Referencing: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
            // Last-Accessed: July 2020
            // The ec2:DescribeNetworkInterfaces, and ec2:AttachNetworkInterface operations
            // do not support conditions, and do not support resource restriction.
            // So, we only attach the policy to the lambda function once; when we first create it.
            const iamEniAttach = new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: [
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:AttachNetworkInterface',
                ],
                resources: ['*'],
            });
            eventHandler.role.addToPolicy(iamEniAttach);
        }
        return eventHandler;
    }
    /**
     * Create, or fetch, an SNS Topic to which we'll direct the ASG's instance-start lifecycle hook events.
     *
     * Also creates, or fetches,
     * the accompanying role that allows the lifecycle events to be published to the SNS Topic.
     *
     * @param lambdaHandler The lambda singleton that will be processing the lifecycle events.
     * @returns : Topic, role: Role }
     * @stability stable
     */
    setupLifecycleNotificationTopic(lambdaHandler) {
        const stack = core_1.Stack.of(this);
        // We only need to have a single SNS topic & subscription set up to handle lifecycle events for *all* instances of this class.
        // We have to be careful, however, to ensure that our initial setup only happens once when we first add the topic and such
        // to this stack; otherwise, we will not be able to deploy more than one of these constructs in a stack.
        const notificationRoleUniqueId = 'AttachEniNotificationRole' + this.removeHyphens('a0376ff8-248e-4534-bf42-58c6ffa4d5b4');
        const notificationTopicUniqueId = 'AttachEniNotificationTopic' + this.removeHyphens('c8b1e9a6-783c-4954-b191-204dd5e3b9e0');
        let notificationTopic = stack.node.tryFindChild(notificationTopicUniqueId);
        let notificationRole;
        if (!notificationTopic) {
            // First time creating the singleton Topic in this stack. Set it all up...
            notificationRole = new aws_iam_1.Role(stack, notificationRoleUniqueId, {
                assumedBy: new aws_iam_1.ServicePrincipal('autoscaling.amazonaws.com'),
            });
            notificationTopic = new aws_sns_1.Topic(stack, notificationTopicUniqueId, {
                displayName: `For RFDK instance-launch notifications for stack '${stack.stackName}'`,
            });
            notificationTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(lambdaHandler));
            notificationTopic.grantPublish(notificationRole);
        }
        else {
            notificationRole = stack.node.findChild(notificationRoleUniqueId);
        }
        return {
            topic: notificationTopic,
            role: notificationRole,
        };
    }
    /**
     * Convert a UUID into a string that's usable in a construct id.
     */
    removeHyphens(x) {
        return x.replace(/[-]/g, '');
    }
}
exports.StaticPrivateIpServer = StaticPrivateIpServer;
//# sourceMappingURL=data:application/json;base64,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