"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScriptAsset = void 0;
const path = require("path");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
var ScriptExtension;
(function (ScriptExtension) {
    ScriptExtension[ScriptExtension[".sh"] = 0] = ".sh";
    ScriptExtension[ScriptExtension[".ps1"] = 1] = ".ps1";
})(ScriptExtension || (ScriptExtension = {}));
var ScriptPathPrefix;
(function (ScriptPathPrefix) {
    ScriptPathPrefix[ScriptPathPrefix["bash"] = 0] = "bash";
    ScriptPathPrefix[ScriptPathPrefix["powershell"] = 1] = "powershell";
})(ScriptPathPrefix || (ScriptPathPrefix = {}));
/**
 * This method returns the path to a script based on RFDK conventional directory structure and the target
 * operating system of the script.
 *
 * The directory structure convention keeps linux scripts in `${scriptRoot}//bash/*.sh` and Windows scripts in
 * `${scriptRoot}/powershell/*.ps1`.
 *
 * @param osType
 * @param scriptName
 */
function getConventionalScriptPath(params) {
    const { rootDir: scriptDir, baseName: scriptName, osType } = params;
    // Make sure we have a known osType. The error message is pretty obtuse if we don't:
    //  The "path" argument must be of type string. Received undefined
    if (ScriptPathPrefix[osType] === undefined || ScriptExtension[osType] == undefined) {
        throw Error(`Unknown osType: ${osType}`);
    }
    return path.join(scriptDir, ScriptPathPrefix[osType], `${scriptName}${ScriptExtension[osType]}`);
}
/**
 * An S3 asset that contains a shell script intended to be executed through instance user data.
 *
 * This is used by other constructs to generalize the concept of a script
 * (bash or powershell) that executes on an instance.
 * It provides a wrapper around the CDK’s S3 Asset construct
 * ( https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-s3-assets.Asset.html )
 *
 * The script asset is placed into and fetched from the CDK bootstrap S3 bucket.
 *
 * Resources Deployed
 * ------------------------
 * - An Asset which is uploaded to the bootstrap S3 bucket.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 */
class ScriptAsset extends aws_s3_assets_1.Asset {
    /**
     * Returns a {@link ScriptAsset} instance by computing the path to the script using RFDK's script directory structure convention.
     *
     * By convention, scripts are kept in a `scripts` directory in each `aws-rfdk/*` package. The scripts are organized
     * based on target shell (and implicitly target operating system). The directory structure looks like:
     *
     * ```
     * scripts/
     *    bash/
     *      script-one.sh
     *      script-two.sh
     *    powershell
     *      script-one.ps1
     *      script-one.ps1
     * ```
     *
     * @param scope The scope for the created {@link ScriptAsset}.
     * @param id The construct id for the created {@link ScriptAsset}.
     * @param scriptParams The parameters that are used to compute the conventional path to the script file.
     * @stability stable
     */
    static fromPathConvention(scope, id, scriptParams) {
        const scriptPath = getConventionalScriptPath(scriptParams);
        return new ScriptAsset(scope, id, { path: scriptPath });
    }
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * Adds commands to the {@link IScriptHost} to download and execute the ScriptAsset.
     *
     * @param props The parameters for executing the script.
     * @stability stable
     */
    executeOn(props) {
        const { host, args } = props;
        // Grant permission to fetch the script asset
        this.grantRead(host);
        // Add a command to the user data to download the script asset
        const instanceScriptPath = host.userData.addS3DownloadCommand({
            bucket: this.bucket,
            bucketKey: this.s3ObjectKey,
        });
        // Add a command to the user data to execute the downloaded script
        host.userData.addExecuteFileCommand({
            filePath: instanceScriptPath,
            arguments: (args || []).join(' '),
        });
    }
}
exports.ScriptAsset = ScriptAsset;
//# sourceMappingURL=data:application/json;base64,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