"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbInstance = void 0;
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const _1 = require("./");
const runtime_info_1 = require("./runtime-info");
/**
 * This construct provides a {@link StaticPrivateIpServer} that is hosting MongoDB.
 *
 * The data for this MongoDB database
 * is stored in an Amazon Elastic Block Storage (EBS) Volume that is automatically attached to the instance when it is
 * launched, and is separate from the instance's root volume; it is recommended that you set up a backup schedule for
 * this volume.
 *
 * When this instance is first launched, or relaunched after an instance replacement, it will:
 * 1. Attach an EBS volume to /var/lib/mongo upon which the MongoDB data is stored;
 * 2. Automatically install the specified version of MongoDB, from the official Mongo Inc. sources;
 * 3. Create an admin user in that database if one has not yet been created -- the credentials for this user
 * can be provided by you, or randomly generated;
 * 4. Configure MongoDB to require authentication, and only allow encrypted connections over TLS.
 *
 * The instance's launch logs and MongoDB logs will be automatically stored in Amazon CloudWatch logs; the
 * default log group name is: /renderfarm/<this construct ID>
 *
 * Resources Deployed
 * ------------------------
 * - {@link StaticPrivateIpServer} that hosts MongoDB.
 * - An A-Record in the provided PrivateHostedZone to create a DNS entry for this server's static private IP.
 * - A Secret in AWS SecretsManager that contains the administrator credentials for MongoDB.
 * - An encrypted Amazon Elastic Block Store (EBS) Volume on which the MongoDB data is stored.
 * - Amazon CloudWatch log group that contains instance-launch and MongoDB application logs.
 *
 * Security Considerations
 * ------------------------
 * - The administrator credentials for MongoDB are stored in a Secret within AWS SecretsManager. You must strictly limit
 *    access to this secret to only entities that require it.
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The EBS Volume that is created by, or provided to, this construct is used to store the contents of your MongoDB data. To
 *    protect the sensitive data in your database, you should not grant access to this EBS Volume to any principal or instance
 *    other than the instance created by this construct. Furthermore, we recommend that you ensure that the volume that is
 *    used for this purpose is encrypted at rest.
 * - This construct uses this package's {@link StaticPrivateIpServer}, {@link MongoDbInstaller}, {@link CloudWatchAgent},
 *    {@link ExportingLogGroup}, and {@link MountableBlockVolume}. Security considerations that are outlined by the documentation
 *    for those constructs should also be taken into account.
 *
 * @stability stable
 */
class MongoDbInstance extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.version = props.mongoDb.version;
        // Select the subnet for this instance.
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        this.server = new _1.StaticPrivateIpServer(this, 'Server', {
            vpc: props.vpc,
            vpcSubnets: { subnets: [subnet] },
            instanceType: (_a = props.instanceType) !== null && _a !== void 0 ? _a : new aws_ec2_1.InstanceType('r5.large'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            blockDevices: [
                {
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(MongoDbInstance.ROOT_DEVICE_SIZE.toGibibytes(), { encrypted: true }),
                },
            ],
            keyName: props.keyName,
            resourceSignalTimeout: core_1.Duration.minutes(5),
            role: props.role,
            securityGroup: props.securityGroup,
        });
        new aws_route53_1.ARecord(this, 'ARecord', {
            target: aws_route53_1.RecordTarget.fromIpAddresses(this.server.privateIpAddress),
            zone: props.mongoDb.dnsZone,
            recordName: props.mongoDb.hostname,
        });
        this.adminUser = (_b = props.mongoDb.adminUser) !== null && _b !== void 0 ? _b : new aws_secretsmanager_1.Secret(this, 'AdminUser', {
            description: `Admin credentials for the MongoDB database ${this.node.uniqueId}`,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
                generateStringKey: 'password',
                secretStringTemplate: JSON.stringify({ username: 'admin' }),
            },
        });
        this.mongoDataVolume = (_d = (_c = props.mongoDb.mongoDataVolume) === null || _c === void 0 ? void 0 : _c.volume) !== null && _d !== void 0 ? _d : new aws_ec2_1.Volume(this, 'MongoDbData', {
            size: MongoDbInstance.DEFAULT_MONGO_DEVICE_SIZE,
            ...(_e = props.mongoDb.mongoDataVolume) === null || _e === void 0 ? void 0 : _e.volumeProps,
            availabilityZone: subnet.availabilityZone,
            encrypted: true,
        });
        const volumeMount = new _1.MountableBlockVolume(this, {
            blockVolume: this.mongoDataVolume,
            volumeFormat: _1.BlockVolumeFormat.XFS,
        });
        const mongoInstaller = new _1.MongoDbInstaller(this, {
            version: props.mongoDb.version,
            userSsplAcceptance: props.mongoDb.userSsplAcceptance,
        });
        // Set up the server's UserData.
        this.server.userData.addCommands('set -xefuo pipefail');
        this.server.userData.addSignalOnExitCommand(this.server.autoscalingGroup);
        this.configureCloudWatchLogStreams(this.server, id, props.logGroupProps); // MUST BE FIRST
        volumeMount.mountToLinuxInstance(this.server, {
            location: MongoDbInstance.MONGO_DEVICE_MOUNT_POINT,
        });
        mongoInstaller.installOnLinuxInstance(this.server);
        this.configureMongoDb(this.server, props.mongoDb);
        this.certificateChain = props.mongoDb.serverCertificate.certChain;
        this.connections = this.server.connections;
        this.grantPrincipal = this.server.grantPrincipal;
        this.port = 27017;
        this.role = this.server.role;
        this.userData = this.server.userData;
        this.fullHostname = `${props.mongoDb.hostname}.${props.mongoDb.dnsZone.zoneName}`;
        this.node.defaultChild = this.server;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Adds UserData commands to install & configure the CloudWatch Agent onto the instance.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *      - The cloud-init log
     *      - The MongoDB application log.
     *
     * @param host The instance/host to setup the CloudWatchAgent upon.
     * @param groupName Name to append to the log group prefix when forming the log group name.
     * @param logGroupProps Properties for the log group.
     * @stability stable
     */
    configureCloudWatchLogStreams(host, groupName, logGroupProps) {
        var _a;
        const prefix = (_a = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _a !== void 0 ? _a : MongoDbInstance.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = _1.LogGroupFactory.createOrFetch(this, 'MongoDbInstanceLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(host.grantPrincipal);
        const cloudWatchConfigurationBuilder = new _1.CloudWatchConfigBuilder(MongoDbInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'MongoDB', '/var/log/mongodb/mongod.log');
        new _1.CloudWatchAgent(this, 'MongoDbInstanceLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host,
        });
    }
    /**
     * Adds commands to the userData of the instance to install MongoDB, create an admin user if one does not exist, and to to start mongod running.
     *
     * @stability stable
     */
    configureMongoDb(instance, settings) {
        const scriptsAsset = new aws_s3_assets_1.Asset(this, 'MongoSetup', {
            path: path.join(__dirname, '..', 'scripts', 'mongodb', settings.version),
        });
        scriptsAsset.grantRead(instance.grantPrincipal);
        const scriptZipfile = instance.userData.addS3DownloadCommand({
            bucket: scriptsAsset.bucket,
            bucketKey: scriptsAsset.s3ObjectKey,
        });
        instance.userData.addCommands(
        // Ensure mongod is installed and stopped before we go any further
        'which mongod && test -f /etc/mongod.conf', 'sudo service mongod stop', 
        // We're going to make a temporary RAM filesystem for the mongo setup files.
        // This will let us write sensitive data to "disk" without worrying about it
        // being persisted in any physical disk, even temporarily.
        'MONGO_SETUP_DIR=$(mktemp -d)', 'mkdir -p "${MONGO_SETUP_DIR}"', 'sudo mount -t tmpfs -o size=50M tmpfs "${MONGO_SETUP_DIR}"', 'pushd "${MONGO_SETUP_DIR}"', `unzip ${scriptZipfile}`, 
        // Backup mongod.conf for now
        'cp /etc/mongod.conf .');
        const cert = settings.serverCertificate;
        instance.userData.addCommands(`bash serverCertFromSecrets.sh "${cert.cert.secretArn}" "${cert.certChain.secretArn}" "${cert.key.secretArn}" "${cert.passphrase.secretArn}"`);
        cert.cert.grantRead(instance.grantPrincipal);
        cert.certChain.grantRead(instance.grantPrincipal);
        cert.key.grantRead(instance.grantPrincipal);
        cert.passphrase.grantRead(instance.grantPrincipal);
        const certsDirectory = '/etc/mongod_certs';
        instance.userData.addCommands(
        // Move the certificates into place
        `sudo mkdir -p ${certsDirectory}`, `sudo mv ./ca.crt ./key.pem ${certsDirectory}`, 'sudo chown root.mongod -R /etc/mongod_certs/', // Something weird about shell interpretation. Can't use '*' on this or next line.
        'sudo chmod 640 -R /etc/mongod_certs/', 'sudo chmod 750 /etc/mongod_certs/', // Directory needs to be executable.
        // mongod user id might, potentially change on reboot. Make sure we own all mongo data
        `sudo chown mongod.mongod -R ${MongoDbInstance.MONGO_DEVICE_MOUNT_POINT}`, 
        // Configure mongod
        'bash ./setMongoLimits.sh', `bash ./setStoragePath.sh "${MongoDbInstance.MONGO_DEVICE_MOUNT_POINT}"`, 'bash ./setMongoNoAuth.sh', 'sudo service mongod start', `bash ./setAdminCredentials.sh "${this.adminUser.secretArn}"`);
        this.adminUser.grantRead(instance.grantPrincipal);
        instance.userData.addCommands(
        // Setup for live deployment, and start mongod
        'sudo service mongod stop', 'bash ./setLiveConfiguration.sh', 'sudo systemctl enable mongod', // Enable restart on reboot
        'sudo service mongod start', 'popd');
        instance.userData.addOnExitCommands(
        // Clean up the temporary RAM filesystem
        'test "${MONGO_SETUP_DIR} != "" && sudo umount "${MONGO_SETUP_DIR}');
    }
}
exports.MongoDbInstance = MongoDbInstance;
// How often Cloudwatch logs will be flushed.
MongoDbInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
// Default prefix for a LogGroup if one isn't provided in the props.
MongoDbInstance.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
// Size of the EBS volume for MongoDB data, if we create one.
MongoDbInstance.DEFAULT_MONGO_DEVICE_SIZE = core_1.Size.gibibytes(20);
// Mount point for the MongoDB data volume.
MongoDbInstance.MONGO_DEVICE_MOUNT_POINT = '/var/lib/mongo';
// Size of the root device volume on the instance.
MongoDbInstance.ROOT_DEVICE_SIZE = core_1.Size.gibibytes(10);
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibW9uZ29kYi1pbnN0YW5jZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIm1vbmdvZGItaW5zdGFuY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7R0FHRzs7O0FBRUgsNkJBQTZCO0FBRTdCLDhEQUVrQztBQUNsQyw4Q0FZMEI7QUFTMUIsc0RBSThCO0FBQzlCLDBEQUVnQztBQUNoQyxvRUFHcUM7QUFDckMsd0NBS3VCO0FBRXZCLHlCQWFZO0FBQ1osaURBRXdCOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBeVF4QixNQUFhLGVBQWdCLFNBQVEsZ0JBQVM7Ozs7SUFzRTVDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMkI7O1FBQ25FLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQztRQUVyQyx1Q0FBdUM7UUFDdkMsTUFBTSxFQUFFLE9BQU8sRUFBRSxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUM5RCxJQUFJLE9BQU8sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMscUNBQXFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1NBQzdIO1FBQ0QsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRTFCLElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSx3QkFBcUIsQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFO1lBQ3RELEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRztZQUNkLFVBQVUsRUFBRSxFQUFFLE9BQU8sRUFBRSxDQUFFLE1BQU0sQ0FBRSxFQUFFO1lBQ25DLFlBQVksUUFBRSxLQUFLLENBQUMsWUFBWSxtQ0FBSSxJQUFJLHNCQUFZLENBQUMsVUFBVSxDQUFDO1lBQ2hFLFlBQVksRUFBRSxzQkFBWSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWMsRUFBRSxDQUFDO1lBQ2xHLFlBQVksRUFBRTtnQkFDWjtvQkFDRSxVQUFVLEVBQUUsV0FBVztvQkFDdkIsTUFBTSxFQUFFLG1DQUFpQixDQUFDLEdBQUcsQ0FBQyxlQUFlLENBQUMsZ0JBQWdCLENBQUMsV0FBVyxFQUFFLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUM7aUJBQ25HO2FBQ0Y7WUFDRCxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIscUJBQXFCLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDMUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQ2hCLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtTQUNuQyxDQUFDLENBQUM7UUFFSCxJQUFJLHFCQUFPLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUMzQixNQUFNLEVBQUUsMEJBQVksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQztZQUNsRSxJQUFJLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPO1lBQzNCLFVBQVUsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLFFBQVE7U0FDbkMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFNBQVMsU0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLFNBQVMsbUNBQUksSUFBSSwyQkFBTSxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUU7WUFDeEUsV0FBVyxFQUFFLDhDQUE4QyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUMvRSxvQkFBb0IsRUFBRTtnQkFDcEIsaUJBQWlCLEVBQUUsUUFBUTtnQkFDM0Isa0JBQWtCLEVBQUUsSUFBSTtnQkFDeEIsWUFBWSxFQUFFLEtBQUs7Z0JBQ25CLGNBQWMsRUFBRSxFQUFFO2dCQUNsQix1QkFBdUIsRUFBRSxJQUFJO2dCQUM3QixpQkFBaUIsRUFBRSxVQUFVO2dCQUM3QixvQkFBb0IsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxDQUFDO2FBQzVEO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGVBQWUsZUFBRyxLQUFLLENBQUMsT0FBTyxDQUFDLGVBQWUsMENBQUUsTUFBTSxtQ0FBSSxJQUFJLGdCQUFNLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtZQUM5RixJQUFJLEVBQUUsZUFBZSxDQUFDLHlCQUF5QjtZQUMvQyxTQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsZUFBZSwwQ0FBRSxXQUFXO1lBQzdDLGdCQUFnQixFQUFFLE1BQU0sQ0FBQyxnQkFBZ0I7WUFDekMsU0FBUyxFQUFFLElBQUk7U0FDaEIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxXQUFXLEdBQUcsSUFBSSx1QkFBb0IsQ0FBQyxJQUFJLEVBQUU7WUFDakQsV0FBVyxFQUFFLElBQUksQ0FBQyxlQUFlO1lBQ2pDLFlBQVksRUFBRSxvQkFBaUIsQ0FBQyxHQUFHO1NBQ3BDLENBQUMsQ0FBQztRQUVILE1BQU0sY0FBYyxHQUFHLElBQUksbUJBQWdCLENBQUMsSUFBSSxFQUFFO1lBQ2hELE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU87WUFDOUIsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxrQkFBa0I7U0FDckQsQ0FBQyxDQUFDO1FBRUgsZ0NBQWdDO1FBQ2hDLElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ3hELElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUMxRSxJQUFJLENBQUMsNkJBQTZCLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCO1FBQzFGLFdBQVcsQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQzVDLFFBQVEsRUFBRSxlQUFlLENBQUMsd0JBQXdCO1NBQ25ELENBQUMsQ0FBQztRQUNILGNBQWMsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDbkQsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRWxELElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLGlCQUFpQixDQUFDLFNBQVUsQ0FBQztRQUNuRSxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDO1FBQzNDLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUM7UUFDakQsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUM7UUFDbEIsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQztRQUM3QixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDO1FBQ3JDLElBQUksQ0FBQyxZQUFZLEdBQUcsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLFFBQVEsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUVsRixJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDO1FBRXJDLDZDQUE2QztRQUM3QywyQkFBWSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3JCLENBQUM7Ozs7Ozs7Ozs7Ozs7SUFhUyw2QkFBNkIsQ0FBQyxJQUFpQixFQUFFLFNBQWlCLEVBQUUsYUFBb0M7O1FBQ2hILE1BQU0sTUFBTSxTQUFHLGFBQWEsYUFBYixhQUFhLHVCQUFiLGFBQWEsQ0FBRSxjQUFjLG1DQUFJLGVBQWUsQ0FBQyx3QkFBd0IsQ0FBQztRQUN6RixNQUFNLHNCQUFzQixHQUFHO1lBQzdCLEdBQUcsYUFBYTtZQUNoQixjQUFjLEVBQUUsTUFBTTtTQUN2QixDQUFDO1FBQ0YsTUFBTSxRQUFRLEdBQUcsa0JBQWUsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLGdDQUFnQyxFQUFFLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1FBRTFILFFBQVEsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBRXpDLE1BQU0sOEJBQThCLEdBQUcsSUFBSSwwQkFBdUIsQ0FBQyxlQUFlLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUVsSCw4QkFBOEIsQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsWUFBWSxFQUNyRSxtQkFBbUIsRUFDbkIsZ0NBQWdDLENBQUMsQ0FBQztRQUNwQyw4QkFBOEIsQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsWUFBWSxFQUNyRSxTQUFTLEVBQ1QsNkJBQTZCLENBQUMsQ0FBQztRQUVqQyxJQUFJLGtCQUFlLENBQUMsSUFBSSxFQUFFLDJCQUEyQixFQUFFO1lBQ3JELGdCQUFnQixFQUFFLDhCQUE4QixDQUFDLCtCQUErQixFQUFFO1lBQ2xGLElBQUk7U0FDTCxDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7SUFNUyxnQkFBZ0IsQ0FBQyxRQUErQixFQUFFLFFBQWlDO1FBQzNGLE1BQU0sWUFBWSxHQUFHLElBQUkscUJBQUssQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ2pELElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLFNBQVMsRUFBRSxRQUFRLENBQUMsT0FBTyxDQUFDO1NBQ3pFLENBQUMsQ0FBQztRQUNILFlBQVksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBRWhELE1BQU0sYUFBYSxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsb0JBQW9CLENBQUM7WUFDM0QsTUFBTSxFQUFFLFlBQVksQ0FBQyxNQUFNO1lBQzNCLFNBQVMsRUFBRSxZQUFZLENBQUMsV0FBVztTQUNwQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsUUFBUSxDQUFDLFdBQVc7UUFDM0Isa0VBQWtFO1FBQ2xFLDBDQUEwQyxFQUMxQywwQkFBMEI7UUFDMUIsNEVBQTRFO1FBQzVFLDRFQUE0RTtRQUM1RSwwREFBMEQ7UUFDMUQsOEJBQThCLEVBQzlCLCtCQUErQixFQUMvQiw0REFBNEQsRUFDNUQsNEJBQTRCLEVBQzVCLFNBQVMsYUFBYSxFQUFFO1FBQ3hCLDZCQUE2QjtRQUM3Qix1QkFBdUIsQ0FDeEIsQ0FBQztRQUVGLE1BQU0sSUFBSSxHQUFHLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQztRQUN4QyxRQUFRLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FDM0Isa0NBQWtDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxNQUFNLElBQUksQ0FBQyxTQUFVLENBQUMsU0FBUyxNQUFNLElBQUksQ0FBQyxHQUFHLENBQUMsU0FBUyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUyxHQUFHLENBQy9JLENBQUM7UUFDRixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDN0MsSUFBSSxDQUFDLFNBQVUsQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ25ELElBQUksQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUM1QyxJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFbkQsTUFBTSxjQUFjLEdBQUcsbUJBQW1CLENBQUM7UUFDM0MsUUFBUSxDQUFDLFFBQVEsQ0FBQyxXQUFXO1FBQzNCLG1DQUFtQztRQUNuQyxpQkFBaUIsY0FBYyxFQUFFLEVBQ2pDLDhCQUE4QixjQUFjLEVBQUUsRUFDOUMsOENBQThDLEVBQUUsa0ZBQWtGO1FBQ2xJLHNDQUFzQyxFQUN0QyxtQ0FBbUMsRUFBRSxvQ0FBb0M7UUFDekUsc0ZBQXNGO1FBQ3RGLCtCQUErQixlQUFlLENBQUMsd0JBQXdCLEVBQUU7UUFDekUsbUJBQW1CO1FBQ25CLDBCQUEwQixFQUMxQiw2QkFBNkIsZUFBZSxDQUFDLHdCQUF3QixHQUFHLEVBQ3hFLDBCQUEwQixFQUMxQiwyQkFBMkIsRUFDM0Isa0NBQWtDLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxHQUFHLENBQzlELENBQUM7UUFDRixJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFbEQsUUFBUSxDQUFDLFFBQVEsQ0FBQyxXQUFXO1FBQzNCLDhDQUE4QztRQUM5QywwQkFBMEIsRUFDMUIsZ0NBQWdDLEVBQ2hDLDhCQUE4QixFQUFFLDJCQUEyQjtRQUMzRCwyQkFBMkIsRUFDM0IsTUFBTSxDQUNQLENBQUM7UUFFRixRQUFRLENBQUMsUUFBUSxDQUFDLGlCQUFpQjtRQUNqQyx3Q0FBd0M7UUFDeEMsbUVBQW1FLENBQ3BFLENBQUM7SUFDSixDQUFDOztBQTFRSCwwQ0E0UUM7QUEzUUMsNkNBQTZDO0FBQzlCLDZDQUE2QixHQUFhLGVBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7QUFDOUUsb0VBQW9FO0FBQ3JELHdDQUF3QixHQUFXLGNBQWMsQ0FBQztBQUNqRSw2REFBNkQ7QUFDOUMseUNBQXlCLEdBQUcsV0FBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsQ0FBQztBQUM5RCwyQ0FBMkM7QUFDNUIsd0NBQXdCLEdBQUcsZ0JBQWdCLENBQUM7QUFDM0Qsa0RBQWtEO0FBQ25DLGdDQUFnQixHQUFHLFdBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuXG5pbXBvcnQge1xuICBCbG9ja0RldmljZVZvbHVtZSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWF1dG9zY2FsaW5nJztcbmltcG9ydCB7XG4gIEFtYXpvbkxpbnV4R2VuZXJhdGlvbixcbiAgQ29ubmVjdGlvbnMsXG4gIElDb25uZWN0YWJsZSxcbiAgSW5zdGFuY2VUeXBlLFxuICBJU2VjdXJpdHlHcm91cCxcbiAgSVZvbHVtZSxcbiAgSVZwYyxcbiAgTWFjaGluZUltYWdlLFxuICBTdWJuZXRTZWxlY3Rpb24sXG4gIFVzZXJEYXRhLFxuICBWb2x1bWUsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0IHtcbiAgSUdyYW50YWJsZSxcbiAgSVByaW5jaXBhbCxcbiAgSVJvbGUsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHtcbiAgSUtleSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWttcyc7XG5pbXBvcnQge1xuICBBUmVjb3JkLFxuICBJUHJpdmF0ZUhvc3RlZFpvbmUsXG4gIFJlY29yZFRhcmdldCxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLXJvdXRlNTMnO1xuaW1wb3J0IHtcbiAgQXNzZXQsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1zMy1hc3NldHMnO1xuaW1wb3J0IHtcbiAgSVNlY3JldCxcbiAgU2VjcmV0LFxufSBmcm9tICdAYXdzLWNkay9hd3Mtc2VjcmV0c21hbmFnZXInO1xuaW1wb3J0IHtcbiAgQ29uc3RydWN0LFxuICBEdXJhdGlvbixcbiAgSUNvbnN0cnVjdCxcbiAgU2l6ZSxcbn0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbmltcG9ydCB7XG4gIEJsb2NrVm9sdW1lRm9ybWF0LFxuICBDbG91ZFdhdGNoQWdlbnQsXG4gIENsb3VkV2F0Y2hDb25maWdCdWlsZGVyLFxuICBJU2NyaXB0SG9zdCxcbiAgSVg1MDlDZXJ0aWZpY2F0ZVBlbSxcbiAgTG9nR3JvdXBGYWN0b3J5LFxuICBMb2dHcm91cEZhY3RvcnlQcm9wcyxcbiAgTW9uZ29EYkluc3RhbGxlcixcbiAgTW9uZ29EYlNzcGxMaWNlbnNlQWNjZXB0YW5jZSxcbiAgTW9uZ29EYlZlcnNpb24sXG4gIE1vdW50YWJsZUJsb2NrVm9sdW1lLFxuICBTdGF0aWNQcml2YXRlSXBTZXJ2ZXIsXG59IGZyb20gJy4vJztcbmltcG9ydCB7XG4gIHRhZ0NvbnN0cnVjdCxcbn0gZnJvbSAnLi9ydW50aW1lLWluZm8nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgTW9uZ29EYkluc3RhbmNlTmV3Vm9sdW1lUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzaXplPzogU2l6ZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVuY3J5cHRpb25LZXk/OiBJS2V5O1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBNb25nb0RiSW5zdGFuY2VWb2x1bWVQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdm9sdW1lPzogSVZvbHVtZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdm9sdW1lUHJvcHM/OiBNb25nb0RiSW5zdGFuY2VOZXdWb2x1bWVQcm9wcztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgTW9uZ29EYkFwcGxpY2F0aW9uUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB1c2VyU3NwbEFjY2VwdGFuY2U/OiBNb25nb0RiU3NwbExpY2Vuc2VBY2NlcHRhbmNlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2ZXJzaW9uOiBNb25nb0RiVmVyc2lvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkbnNab25lOiBJUHJpdmF0ZUhvc3RlZFpvbmU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGhvc3RuYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzZXJ2ZXJDZXJ0aWZpY2F0ZTogSVg1MDlDZXJ0aWZpY2F0ZVBlbTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFkbWluVXNlcj86IElTZWNyZXQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbW9uZ29EYXRhVm9sdW1lPzogTW9uZ29EYkluc3RhbmNlVm9sdW1lUHJvcHM7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBNb25nb0RiSW5zdGFuY2VQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbW9uZ29EYjogTW9uZ29EYkFwcGxpY2F0aW9uUHJvcHM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdnBjOiBJVnBjO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZwY1N1Ym5ldHM/OiBTdWJuZXRTZWxlY3Rpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlPzogSW5zdGFuY2VUeXBlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBrZXlOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxvZ0dyb3VwUHJvcHM/OiBMb2dHcm91cEZhY3RvcnlQcm9wcztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByb2xlPzogSVJvbGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cD86IElTZWN1cml0eUdyb3VwO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSU1vbmdvRGIgZXh0ZW5kcyBJQ29ubmVjdGFibGUsIElDb25zdHJ1Y3Qge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhZG1pblVzZXI6IElTZWNyZXQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjZXJ0aWZpY2F0ZUNoYWluOiBJU2VjcmV0O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGZ1bGxIb3N0bmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHBvcnQ6IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2ZXJzaW9uOiBNb25nb0RiVmVyc2lvbjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBNb25nb0RiSW5zdGFuY2UgZXh0ZW5kcyBDb25zdHJ1Y3QgaW1wbGVtZW50cyBJTW9uZ29EYiwgSUdyYW50YWJsZSB7XG4gIC8vIEhvdyBvZnRlbiBDbG91ZHdhdGNoIGxvZ3Mgd2lsbCBiZSBmbHVzaGVkLlxuICBwcml2YXRlIHN0YXRpYyBDTE9VRFdBVENIX0xPR19GTFVTSF9JTlRFUlZBTDogRHVyYXRpb24gPSBEdXJhdGlvbi5zZWNvbmRzKDE1KTtcbiAgLy8gRGVmYXVsdCBwcmVmaXggZm9yIGEgTG9nR3JvdXAgaWYgb25lIGlzbid0IHByb3ZpZGVkIGluIHRoZSBwcm9wcy5cbiAgcHJpdmF0ZSBzdGF0aWMgREVGQVVMVF9MT0dfR1JPVVBfUFJFRklYOiBzdHJpbmcgPSAnL3JlbmRlcmZhcm0vJztcbiAgLy8gU2l6ZSBvZiB0aGUgRUJTIHZvbHVtZSBmb3IgTW9uZ29EQiBkYXRhLCBpZiB3ZSBjcmVhdGUgb25lLlxuICBwcml2YXRlIHN0YXRpYyBERUZBVUxUX01PTkdPX0RFVklDRV9TSVpFID0gU2l6ZS5naWJpYnl0ZXMoMjApO1xuICAvLyBNb3VudCBwb2ludCBmb3IgdGhlIE1vbmdvREIgZGF0YSB2b2x1bWUuXG4gIHByaXZhdGUgc3RhdGljIE1PTkdPX0RFVklDRV9NT1VOVF9QT0lOVCA9ICcvdmFyL2xpYi9tb25nbyc7XG4gIC8vIFNpemUgb2YgdGhlIHJvb3QgZGV2aWNlIHZvbHVtZSBvbiB0aGUgaW5zdGFuY2UuXG4gIHByaXZhdGUgc3RhdGljIFJPT1RfREVWSUNFX1NJWkUgPSBTaXplLmdpYmlieXRlcygxMCk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGFkbWluVXNlcjogSVNlY3JldDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgY2VydGlmaWNhdGVDaGFpbjogSVNlY3JldDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uczogQ29ubmVjdGlvbnM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbDogSVByaW5jaXBhbDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZnVsbEhvc3RuYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgc2VydmVyOiBTdGF0aWNQcml2YXRlSXBTZXJ2ZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IG1vbmdvRGF0YVZvbHVtZTogSVZvbHVtZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgcG9ydDogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgcm9sZTogSVJvbGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgdXNlckRhdGE6IFVzZXJEYXRhO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB2ZXJzaW9uOiBNb25nb0RiVmVyc2lvbjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogTW9uZ29EYkluc3RhbmNlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy52ZXJzaW9uID0gcHJvcHMubW9uZ29EYi52ZXJzaW9uO1xuXG4gICAgLy8gU2VsZWN0IHRoZSBzdWJuZXQgZm9yIHRoaXMgaW5zdGFuY2UuXG4gICAgY29uc3QgeyBzdWJuZXRzIH0gPSBwcm9wcy52cGMuc2VsZWN0U3VibmV0cyhwcm9wcy52cGNTdWJuZXRzKTtcbiAgICBpZiAoc3VibmV0cy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgRGlkIG5vdCBmaW5kIGFueSBzdWJuZXRzIG1hdGNoaW5nICR7SlNPTi5zdHJpbmdpZnkocHJvcHMudnBjU3VibmV0cyl9LiBQbGVhc2UgdXNlIGEgZGlmZmVyZW50IHNlbGVjdGlvbi5gKTtcbiAgICB9XG4gICAgY29uc3Qgc3VibmV0ID0gc3VibmV0c1swXTtcblxuICAgIHRoaXMuc2VydmVyID0gbmV3IFN0YXRpY1ByaXZhdGVJcFNlcnZlcih0aGlzLCAnU2VydmVyJywge1xuICAgICAgdnBjOiBwcm9wcy52cGMsXG4gICAgICB2cGNTdWJuZXRzOiB7IHN1Ym5ldHM6IFsgc3VibmV0IF0gfSxcbiAgICAgIGluc3RhbmNlVHlwZTogcHJvcHMuaW5zdGFuY2VUeXBlID8/IG5ldyBJbnN0YW5jZVR5cGUoJ3I1LmxhcmdlJyksXG4gICAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RBbWF6b25MaW51eCh7IGdlbmVyYXRpb246IEFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMiB9KSxcbiAgICAgIGJsb2NrRGV2aWNlczogW1xuICAgICAgICB7XG4gICAgICAgICAgZGV2aWNlTmFtZTogJy9kZXYveHZkYScsIC8vIFJvb3Qgdm9sdW1lXG4gICAgICAgICAgdm9sdW1lOiBCbG9ja0RldmljZVZvbHVtZS5lYnMoTW9uZ29EYkluc3RhbmNlLlJPT1RfREVWSUNFX1NJWkUudG9HaWJpYnl0ZXMoKSwgeyBlbmNyeXB0ZWQ6IHRydWUgfSksXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAga2V5TmFtZTogcHJvcHMua2V5TmFtZSxcbiAgICAgIHJlc291cmNlU2lnbmFsVGltZW91dDogRHVyYXRpb24ubWludXRlcyg1KSxcbiAgICAgIHJvbGU6IHByb3BzLnJvbGUsXG4gICAgICBzZWN1cml0eUdyb3VwOiBwcm9wcy5zZWN1cml0eUdyb3VwLFxuICAgIH0pO1xuXG4gICAgbmV3IEFSZWNvcmQodGhpcywgJ0FSZWNvcmQnLCB7XG4gICAgICB0YXJnZXQ6IFJlY29yZFRhcmdldC5mcm9tSXBBZGRyZXNzZXModGhpcy5zZXJ2ZXIucHJpdmF0ZUlwQWRkcmVzcyksXG4gICAgICB6b25lOiBwcm9wcy5tb25nb0RiLmRuc1pvbmUsXG4gICAgICByZWNvcmROYW1lOiBwcm9wcy5tb25nb0RiLmhvc3RuYW1lLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hZG1pblVzZXIgPSBwcm9wcy5tb25nb0RiLmFkbWluVXNlciA/PyBuZXcgU2VjcmV0KHRoaXMsICdBZG1pblVzZXInLCB7XG4gICAgICBkZXNjcmlwdGlvbjogYEFkbWluIGNyZWRlbnRpYWxzIGZvciB0aGUgTW9uZ29EQiBkYXRhYmFzZSAke3RoaXMubm9kZS51bmlxdWVJZH1gLFxuICAgICAgZ2VuZXJhdGVTZWNyZXRTdHJpbmc6IHtcbiAgICAgICAgZXhjbHVkZUNoYXJhY3RlcnM6ICdcIigpJFxcJycsIC8vIEV4Y2x1ZGUgY2hhcmFjdGVycyB0aGF0IG1pZ2h0IGludGVyYWN0IHdpdGggY29tbWFuZCBzaGVsbHMuXG4gICAgICAgIGV4Y2x1ZGVQdW5jdHVhdGlvbjogdHJ1ZSxcbiAgICAgICAgaW5jbHVkZVNwYWNlOiBmYWxzZSxcbiAgICAgICAgcGFzc3dvcmRMZW5ndGg6IDI0LFxuICAgICAgICByZXF1aXJlRWFjaEluY2x1ZGVkVHlwZTogdHJ1ZSxcbiAgICAgICAgZ2VuZXJhdGVTdHJpbmdLZXk6ICdwYXNzd29yZCcsXG4gICAgICAgIHNlY3JldFN0cmluZ1RlbXBsYXRlOiBKU09OLnN0cmluZ2lmeSh7IHVzZXJuYW1lOiAnYWRtaW4nIH0pLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRoaXMubW9uZ29EYXRhVm9sdW1lID0gcHJvcHMubW9uZ29EYi5tb25nb0RhdGFWb2x1bWU/LnZvbHVtZSA/PyBuZXcgVm9sdW1lKHRoaXMsICdNb25nb0RiRGF0YScsIHtcbiAgICAgIHNpemU6IE1vbmdvRGJJbnN0YW5jZS5ERUZBVUxUX01PTkdPX0RFVklDRV9TSVpFLCAvLyBGaXJzdCBzbyBpdCBjYW4gYmUgb3ZlcnJpZGVuIGJ5IHRoZSBuZXh0IGVudHJ5XG4gICAgICAuLi5wcm9wcy5tb25nb0RiLm1vbmdvRGF0YVZvbHVtZT8udm9sdW1lUHJvcHMsXG4gICAgICBhdmFpbGFiaWxpdHlab25lOiBzdWJuZXQuYXZhaWxhYmlsaXR5Wm9uZSxcbiAgICAgIGVuY3J5cHRlZDogdHJ1ZSxcbiAgICB9KTtcbiAgICBjb25zdCB2b2x1bWVNb3VudCA9IG5ldyBNb3VudGFibGVCbG9ja1ZvbHVtZSh0aGlzLCB7XG4gICAgICBibG9ja1ZvbHVtZTogdGhpcy5tb25nb0RhdGFWb2x1bWUsXG4gICAgICB2b2x1bWVGb3JtYXQ6IEJsb2NrVm9sdW1lRm9ybWF0LlhGUyxcbiAgICB9KTtcblxuICAgIGNvbnN0IG1vbmdvSW5zdGFsbGVyID0gbmV3IE1vbmdvRGJJbnN0YWxsZXIodGhpcywge1xuICAgICAgdmVyc2lvbjogcHJvcHMubW9uZ29EYi52ZXJzaW9uLFxuICAgICAgdXNlclNzcGxBY2NlcHRhbmNlOiBwcm9wcy5tb25nb0RiLnVzZXJTc3BsQWNjZXB0YW5jZSxcbiAgICB9KTtcblxuICAgIC8vIFNldCB1cCB0aGUgc2VydmVyJ3MgVXNlckRhdGEuXG4gICAgdGhpcy5zZXJ2ZXIudXNlckRhdGEuYWRkQ29tbWFuZHMoJ3NldCAteGVmdW8gcGlwZWZhaWwnKTtcbiAgICB0aGlzLnNlcnZlci51c2VyRGF0YS5hZGRTaWduYWxPbkV4aXRDb21tYW5kKHRoaXMuc2VydmVyLmF1dG9zY2FsaW5nR3JvdXApO1xuICAgIHRoaXMuY29uZmlndXJlQ2xvdWRXYXRjaExvZ1N0cmVhbXModGhpcy5zZXJ2ZXIsIGlkLCBwcm9wcy5sb2dHcm91cFByb3BzKTsgLy8gTVVTVCBCRSBGSVJTVFxuICAgIHZvbHVtZU1vdW50Lm1vdW50VG9MaW51eEluc3RhbmNlKHRoaXMuc2VydmVyLCB7XG4gICAgICBsb2NhdGlvbjogTW9uZ29EYkluc3RhbmNlLk1PTkdPX0RFVklDRV9NT1VOVF9QT0lOVCxcbiAgICB9KTtcbiAgICBtb25nb0luc3RhbGxlci5pbnN0YWxsT25MaW51eEluc3RhbmNlKHRoaXMuc2VydmVyKTtcbiAgICB0aGlzLmNvbmZpZ3VyZU1vbmdvRGIodGhpcy5zZXJ2ZXIsIHByb3BzLm1vbmdvRGIpO1xuXG4gICAgdGhpcy5jZXJ0aWZpY2F0ZUNoYWluID0gcHJvcHMubW9uZ29EYi5zZXJ2ZXJDZXJ0aWZpY2F0ZS5jZXJ0Q2hhaW4hO1xuICAgIHRoaXMuY29ubmVjdGlvbnMgPSB0aGlzLnNlcnZlci5jb25uZWN0aW9ucztcbiAgICB0aGlzLmdyYW50UHJpbmNpcGFsID0gdGhpcy5zZXJ2ZXIuZ3JhbnRQcmluY2lwYWw7XG4gICAgdGhpcy5wb3J0ID0gMjcwMTc7XG4gICAgdGhpcy5yb2xlID0gdGhpcy5zZXJ2ZXIucm9sZTtcbiAgICB0aGlzLnVzZXJEYXRhID0gdGhpcy5zZXJ2ZXIudXNlckRhdGE7XG4gICAgdGhpcy5mdWxsSG9zdG5hbWUgPSBgJHtwcm9wcy5tb25nb0RiLmhvc3RuYW1lfS4ke3Byb3BzLm1vbmdvRGIuZG5zWm9uZS56b25lTmFtZX1gO1xuXG4gICAgdGhpcy5ub2RlLmRlZmF1bHRDaGlsZCA9IHRoaXMuc2VydmVyO1xuXG4gICAgLy8gVGFnIGRlcGxveWVkIHJlc291cmNlcyB3aXRoIFJGREsgbWV0YS1kYXRhXG4gICAgdGFnQ29uc3RydWN0KHRoaXMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHJvdGVjdGVkIGNvbmZpZ3VyZUNsb3VkV2F0Y2hMb2dTdHJlYW1zKGhvc3Q6IElTY3JpcHRIb3N0LCBncm91cE5hbWU6IHN0cmluZywgbG9nR3JvdXBQcm9wcz86IExvZ0dyb3VwRmFjdG9yeVByb3BzKSB7XG4gICAgY29uc3QgcHJlZml4ID0gbG9nR3JvdXBQcm9wcz8ubG9nR3JvdXBQcmVmaXggPz8gTW9uZ29EYkluc3RhbmNlLkRFRkFVTFRfTE9HX0dST1VQX1BSRUZJWDtcbiAgICBjb25zdCBkZWZhdWx0ZWRMb2dHcm91cFByb3BzID0ge1xuICAgICAgLi4ubG9nR3JvdXBQcm9wcyxcbiAgICAgIGxvZ0dyb3VwUHJlZml4OiBwcmVmaXgsXG4gICAgfTtcbiAgICBjb25zdCBsb2dHcm91cCA9IExvZ0dyb3VwRmFjdG9yeS5jcmVhdGVPckZldGNoKHRoaXMsICdNb25nb0RiSW5zdGFuY2VMb2dHcm91cFdyYXBwZXInLCBncm91cE5hbWUsIGRlZmF1bHRlZExvZ0dyb3VwUHJvcHMpO1xuXG4gICAgbG9nR3JvdXAuZ3JhbnRXcml0ZShob3N0LmdyYW50UHJpbmNpcGFsKTtcblxuICAgIGNvbnN0IGNsb3VkV2F0Y2hDb25maWd1cmF0aW9uQnVpbGRlciA9IG5ldyBDbG91ZFdhdGNoQ29uZmlnQnVpbGRlcihNb25nb0RiSW5zdGFuY2UuQ0xPVURXQVRDSF9MT0dfRkxVU0hfSU5URVJWQUwpO1xuXG4gICAgY2xvdWRXYXRjaENvbmZpZ3VyYXRpb25CdWlsZGVyLmFkZExvZ3NDb2xsZWN0TGlzdChsb2dHcm91cC5sb2dHcm91cE5hbWUsXG4gICAgICAnY2xvdWQtaW5pdC1vdXRwdXQnLFxuICAgICAgJy92YXIvbG9nL2Nsb3VkLWluaXQtb3V0cHV0LmxvZycpO1xuICAgIGNsb3VkV2F0Y2hDb25maWd1cmF0aW9uQnVpbGRlci5hZGRMb2dzQ29sbGVjdExpc3QobG9nR3JvdXAubG9nR3JvdXBOYW1lLFxuICAgICAgJ01vbmdvREInLFxuICAgICAgJy92YXIvbG9nL21vbmdvZGIvbW9uZ29kLmxvZycpO1xuXG4gICAgbmV3IENsb3VkV2F0Y2hBZ2VudCh0aGlzLCAnTW9uZ29EYkluc3RhbmNlTG9nc0NvbmZpZycsIHtcbiAgICAgIGNsb3VkV2F0Y2hDb25maWc6IGNsb3VkV2F0Y2hDb25maWd1cmF0aW9uQnVpbGRlci5nZW5lcmF0ZUNsb3VkV2F0Y2hDb25maWd1cmF0aW9uKCksXG4gICAgICBob3N0LFxuICAgIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHJvdGVjdGVkIGNvbmZpZ3VyZU1vbmdvRGIoaW5zdGFuY2U6IFN0YXRpY1ByaXZhdGVJcFNlcnZlciwgc2V0dGluZ3M6IE1vbmdvRGJBcHBsaWNhdGlvblByb3BzKSB7XG4gICAgY29uc3Qgc2NyaXB0c0Fzc2V0ID0gbmV3IEFzc2V0KHRoaXMsICdNb25nb1NldHVwJywge1xuICAgICAgcGF0aDogcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJywgJ3NjcmlwdHMnLCAnbW9uZ29kYicsIHNldHRpbmdzLnZlcnNpb24pLFxuICAgIH0pO1xuICAgIHNjcmlwdHNBc3NldC5ncmFudFJlYWQoaW5zdGFuY2UuZ3JhbnRQcmluY2lwYWwpO1xuXG4gICAgY29uc3Qgc2NyaXB0WmlwZmlsZSA9IGluc3RhbmNlLnVzZXJEYXRhLmFkZFMzRG93bmxvYWRDb21tYW5kKHtcbiAgICAgIGJ1Y2tldDogc2NyaXB0c0Fzc2V0LmJ1Y2tldCxcbiAgICAgIGJ1Y2tldEtleTogc2NyaXB0c0Fzc2V0LnMzT2JqZWN0S2V5LFxuICAgIH0pO1xuXG4gICAgaW5zdGFuY2UudXNlckRhdGEuYWRkQ29tbWFuZHMoXG4gICAgICAvLyBFbnN1cmUgbW9uZ29kIGlzIGluc3RhbGxlZCBhbmQgc3RvcHBlZCBiZWZvcmUgd2UgZ28gYW55IGZ1cnRoZXJcbiAgICAgICd3aGljaCBtb25nb2QgJiYgdGVzdCAtZiAvZXRjL21vbmdvZC5jb25mJyxcbiAgICAgICdzdWRvIHNlcnZpY2UgbW9uZ29kIHN0b3AnLFxuICAgICAgLy8gV2UncmUgZ29pbmcgdG8gbWFrZSBhIHRlbXBvcmFyeSBSQU0gZmlsZXN5c3RlbSBmb3IgdGhlIG1vbmdvIHNldHVwIGZpbGVzLlxuICAgICAgLy8gVGhpcyB3aWxsIGxldCB1cyB3cml0ZSBzZW5zaXRpdmUgZGF0YSB0byBcImRpc2tcIiB3aXRob3V0IHdvcnJ5aW5nIGFib3V0IGl0XG4gICAgICAvLyBiZWluZyBwZXJzaXN0ZWQgaW4gYW55IHBoeXNpY2FsIGRpc2ssIGV2ZW4gdGVtcG9yYXJpbHkuXG4gICAgICAnTU9OR09fU0VUVVBfRElSPSQobWt0ZW1wIC1kKScsXG4gICAgICAnbWtkaXIgLXAgXCIke01PTkdPX1NFVFVQX0RJUn1cIicsXG4gICAgICAnc3VkbyBtb3VudCAtdCB0bXBmcyAtbyBzaXplPTUwTSB0bXBmcyBcIiR7TU9OR09fU0VUVVBfRElSfVwiJyxcbiAgICAgICdwdXNoZCBcIiR7TU9OR09fU0VUVVBfRElSfVwiJyxcbiAgICAgIGB1bnppcCAke3NjcmlwdFppcGZpbGV9YCxcbiAgICAgIC8vIEJhY2t1cCBtb25nb2QuY29uZiBmb3Igbm93XG4gICAgICAnY3AgL2V0Yy9tb25nb2QuY29uZiAuJyxcbiAgICApO1xuXG4gICAgY29uc3QgY2VydCA9IHNldHRpbmdzLnNlcnZlckNlcnRpZmljYXRlO1xuICAgIGluc3RhbmNlLnVzZXJEYXRhLmFkZENvbW1hbmRzKFxuICAgICAgYGJhc2ggc2VydmVyQ2VydEZyb21TZWNyZXRzLnNoIFwiJHtjZXJ0LmNlcnQuc2VjcmV0QXJufVwiIFwiJHtjZXJ0LmNlcnRDaGFpbiEuc2VjcmV0QXJufVwiIFwiJHtjZXJ0LmtleS5zZWNyZXRBcm59XCIgXCIke2NlcnQucGFzc3BocmFzZS5zZWNyZXRBcm59XCJgLFxuICAgICk7XG4gICAgY2VydC5jZXJ0LmdyYW50UmVhZChpbnN0YW5jZS5ncmFudFByaW5jaXBhbCk7XG4gICAgY2VydC5jZXJ0Q2hhaW4hLmdyYW50UmVhZChpbnN0YW5jZS5ncmFudFByaW5jaXBhbCk7XG4gICAgY2VydC5rZXkuZ3JhbnRSZWFkKGluc3RhbmNlLmdyYW50UHJpbmNpcGFsKTtcbiAgICBjZXJ0LnBhc3NwaHJhc2UuZ3JhbnRSZWFkKGluc3RhbmNlLmdyYW50UHJpbmNpcGFsKTtcblxuICAgIGNvbnN0IGNlcnRzRGlyZWN0b3J5ID0gJy9ldGMvbW9uZ29kX2NlcnRzJztcbiAgICBpbnN0YW5jZS51c2VyRGF0YS5hZGRDb21tYW5kcyhcbiAgICAgIC8vIE1vdmUgdGhlIGNlcnRpZmljYXRlcyBpbnRvIHBsYWNlXG4gICAgICBgc3VkbyBta2RpciAtcCAke2NlcnRzRGlyZWN0b3J5fWAsXG4gICAgICBgc3VkbyBtdiAuL2NhLmNydCAuL2tleS5wZW0gJHtjZXJ0c0RpcmVjdG9yeX1gLFxuICAgICAgJ3N1ZG8gY2hvd24gcm9vdC5tb25nb2QgLVIgL2V0Yy9tb25nb2RfY2VydHMvJywgLy8gU29tZXRoaW5nIHdlaXJkIGFib3V0IHNoZWxsIGludGVycHJldGF0aW9uLiBDYW4ndCB1c2UgJyonIG9uIHRoaXMgb3IgbmV4dCBsaW5lLlxuICAgICAgJ3N1ZG8gY2htb2QgNjQwIC1SIC9ldGMvbW9uZ29kX2NlcnRzLycsXG4gICAgICAnc3VkbyBjaG1vZCA3NTAgL2V0Yy9tb25nb2RfY2VydHMvJywgLy8gRGlyZWN0b3J5IG5lZWRzIHRvIGJlIGV4ZWN1dGFibGUuXG4gICAgICAvLyBtb25nb2QgdXNlciBpZCBtaWdodCwgcG90ZW50aWFsbHkgY2hhbmdlIG9uIHJlYm9vdC4gTWFrZSBzdXJlIHdlIG93biBhbGwgbW9uZ28gZGF0YVxuICAgICAgYHN1ZG8gY2hvd24gbW9uZ29kLm1vbmdvZCAtUiAke01vbmdvRGJJbnN0YW5jZS5NT05HT19ERVZJQ0VfTU9VTlRfUE9JTlR9YCxcbiAgICAgIC8vIENvbmZpZ3VyZSBtb25nb2RcbiAgICAgICdiYXNoIC4vc2V0TW9uZ29MaW1pdHMuc2gnLFxuICAgICAgYGJhc2ggLi9zZXRTdG9yYWdlUGF0aC5zaCBcIiR7TW9uZ29EYkluc3RhbmNlLk1PTkdPX0RFVklDRV9NT1VOVF9QT0lOVH1cImAsXG4gICAgICAnYmFzaCAuL3NldE1vbmdvTm9BdXRoLnNoJyxcbiAgICAgICdzdWRvIHNlcnZpY2UgbW9uZ29kIHN0YXJ0JyxcbiAgICAgIGBiYXNoIC4vc2V0QWRtaW5DcmVkZW50aWFscy5zaCBcIiR7dGhpcy5hZG1pblVzZXIuc2VjcmV0QXJufVwiYCxcbiAgICApO1xuICAgIHRoaXMuYWRtaW5Vc2VyLmdyYW50UmVhZChpbnN0YW5jZS5ncmFudFByaW5jaXBhbCk7XG5cbiAgICBpbnN0YW5jZS51c2VyRGF0YS5hZGRDb21tYW5kcyhcbiAgICAgIC8vIFNldHVwIGZvciBsaXZlIGRlcGxveW1lbnQsIGFuZCBzdGFydCBtb25nb2RcbiAgICAgICdzdWRvIHNlcnZpY2UgbW9uZ29kIHN0b3AnLFxuICAgICAgJ2Jhc2ggLi9zZXRMaXZlQ29uZmlndXJhdGlvbi5zaCcsXG4gICAgICAnc3VkbyBzeXN0ZW1jdGwgZW5hYmxlIG1vbmdvZCcsIC8vIEVuYWJsZSByZXN0YXJ0IG9uIHJlYm9vdFxuICAgICAgJ3N1ZG8gc2VydmljZSBtb25nb2Qgc3RhcnQnLFxuICAgICAgJ3BvcGQnLFxuICAgICk7XG5cbiAgICBpbnN0YW5jZS51c2VyRGF0YS5hZGRPbkV4aXRDb21tYW5kcyhcbiAgICAgIC8vIENsZWFuIHVwIHRoZSB0ZW1wb3JhcnkgUkFNIGZpbGVzeXN0ZW1cbiAgICAgICd0ZXN0IFwiJHtNT05HT19TRVRVUF9ESVJ9ICE9IFwiXCIgJiYgc3VkbyB1bW91bnQgXCIke01PTkdPX1NFVFVQX0RJUn0nLFxuICAgICk7XG4gIH1cblxufVxuIl19