"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudWatchAgent = void 0;
const path = require("path");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_ssm_1 = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const script_assets_1 = require("./script-assets");
/**
 * This construct is a thin wrapper that provides the ability to install and configure the CloudWatchAgent ( https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html ) on one or more EC2 instances during instance startup.
 *
 * It accomplishes this by downloading and executing the configuration script on the instance.
 * The script will download the CloudWatch Agent installer,
 * optionally verify the installer, and finally install the CloudWatch Agent.
 * The installer is downloaded via the Amazon S3 API, thus, this construct can be used
 * on instances that have no access to the internet as long as the VPC contains
 * an VPC Gateway Endpoint for S3 ( https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-s3.html ).
 *
 * {@link CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR} - Context variable to skip validation
 * of the downloaded CloudWatch Agent installer if set to 'TRUE'.
 * WARNING: Only use this if your deployments are failing due to a validation failure,
 * but you have verified that the failure is benign.
 *
 * Resources Deployed
 * ------------------------
 * - String SSM Parameter in Systems Manager Parameter Store to store the cloudwatch agent configuration;
 * - A script Asset which is uploaded to S3 bucket.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 */
class CloudWatchAgent extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Create the asset for the configuration script
        this.configurationScript = script_assets_1.ScriptAsset.fromPathConvention(scope, 'CloudWatchConfigurationScriptAsset', {
            osType: props.host.osType,
            baseName: 'configureCloudWatchAgent',
            rootDir: path.join(__dirname, '../scripts/'),
        });
        // Create a new SSM Parameter holding the json configuration
        this.ssmParameterForConfig = new aws_ssm_1.StringParameter(scope, 'StringParameter', {
            description: 'config file for Repository logs config',
            stringValue: props.cloudWatchConfig,
        });
        this.grantRead(props.host);
        this.configure(props.host, this.node.tryGetContext(CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR) === 'TRUE');
    }
    /**
     * Grants read permissions to the principal on the assets bucket and parameter store.
     */
    grantRead(grantee) {
        this.configurationScript.grantRead(grantee);
        this.ssmParameterForConfig.grantRead(grantee);
    }
    /**
     * Configures the CloudWatch Agent on the target host.
     *
     * This is done by adding UserData commands to the target host.
     *
     * @param host The host to configure the CloudWatch agent on
     * @param skipValidation Skips the validation of the CloudWatch agent installer if set to true.
     */
    configure(host, skipValidation) {
        // Grant access to the required CloudWatch Agent installer files
        const cloudWatchAgentBucket = aws_s3_1.Bucket.fromBucketArn(this, 'CloudWatchAgentBucket', 'arn:aws:s3:::amazoncloudwatch-agent');
        cloudWatchAgentBucket.grantRead(host);
        const scriptArgs = [];
        if (skipValidation) {
            // Flags must be set before positional arguments for some scripts
            scriptArgs.push(CloudWatchAgent.SKIP_CWAGENT_VALIDATION_FLAG);
        }
        scriptArgs.push(this.ssmParameterForConfig.parameterName);
        this.configurationScript.executeOn({
            host,
            args: scriptArgs,
        });
    }
}
exports.CloudWatchAgent = CloudWatchAgent;
/**
 * The context variable to indicate that CloudWatch agent installer validation should be skipped.
 *
 * @stability stable
 */
CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR = 'RFDK_SKIP_CWAGENT_VALIDATION';
/**
 * The flag for configureCloudWatchAgent script to skip CloudWatch agent installer validation.
 */
CloudWatchAgent.SKIP_CWAGENT_VALIDATION_FLAG = '-s';
//# sourceMappingURL=data:application/json;base64,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