# Copyright 2016-2018 Euratom
# Copyright 2016-2018 United Kingdom Atomic Energy Authority
# Copyright 2016-2018 Centro de Investigaciones Energéticas, Medioambientales y Tecnológicas
#
# Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the
# European Commission - subsequent versions of the EUPL (the "Licence");
# You may not use this work except in compliance with the Licence.
# You may obtain a copy of the Licence at:
#
# https://joinup.ec.europa.eu/software/page/eupl5
#
# Unless required by applicable law or agreed to in writing, software distributed
# under the Licence is distributed on an "AS IS" basis, WITHOUT WARRANTIES OR
# CONDITIONS OF ANY KIND, either express or implied.
#
# See the Licence for the specific language governing permissions and limitations
# under the Licence.

import unittest
import numpy as np
from cherab.core.math.interpolators import interpolators2d

X_LOWER = 0.
X_UPPER = 1.
NB_X = 10
NB_XSAMPLES = 30

Y_LOWER = 0.
Y_UPPER = 2.
NB_Y = 10
NB_YSAMPLES = 30

ABS_DELTA = 1e-8  # delta for values which must be rigorously equal
LIN_DELTA = 1e-8  # delta for values form linear interpolation
CUB_DELTA = 1e-8  # delta for values from cubic interpolation

# todo: replace numerical derivative (as it is noisy for higher orders) and re-enable higher order derivative tests
# highest derivative order to test (high orders can face numerical issues so test not always reliable for high orders at present)
MAX_DERIVATIVE_ORDER = 1


class TestInterpolators2D(unittest.TestCase):
    """
    2D interpolators tests.

    NB: every standard interpolator in this package must allow coordinates as
    1D numpy arrays and values as nD numpy arrays with C indexing (meaning
    that the first index is for x coordinate, the second for y coordinate, the
    third for z). This indexing is opposed to the Fortran indexing (1st index
    is y, 2nd is x, 3rd is z) which is the default one in numpy.
    """

    def setUp(self):
        """
        Initialise domains and sampled data.
        """

        self.x = np.linspace(X_LOWER, X_UPPER, NB_X)
        self.y = np.linspace(Y_LOWER, Y_UPPER, NB_Y)
        self.xsamples = np.linspace(X_LOWER, X_UPPER, NB_XSAMPLES)
        self.ysamples = np.linspace(Y_LOWER, Y_UPPER, NB_YSAMPLES)
        self.xsamples_ex = np.array([X_LOWER - 0.08, X_LOWER - 0.05, X_LOWER - 0.02] + list(self.xsamples) + [X_UPPER + 0.02, X_UPPER + 0.05, X_UPPER + 0.08], dtype=np.float64)
        self.ysamples_ex = np.array([Y_LOWER - 0.16, Y_LOWER - 0.10, Y_LOWER - 0.04] + list(self.ysamples) + [Y_UPPER + 0.04, Y_UPPER + 0.10, Y_UPPER + 0.16], dtype=np.float64)
        self.extrapol_xdomains = [(0, 3), (3, NB_XSAMPLES + 3), (NB_XSAMPLES + 3, NB_XSAMPLES + 6)]
        self.extrapol_ydomains = [(0, 3), (3, NB_YSAMPLES + 3), (NB_YSAMPLES + 3, NB_YSAMPLES + 6)]
        self.xsamples_out = np.array([X_LOWER - 1, X_LOWER - 0.05, 0.5, X_UPPER + 0.05, X_UPPER + 1], dtype=np.float64)
        self.ysamples_out = np.array([Y_LOWER - 1, Y_LOWER - 0.10, 1.0, Y_UPPER + 0.10, Y_UPPER + 1], dtype=np.float64)
        # data from function x, y -> (x-y)*y*cos(10*x) sampled on self.x and self.y:
        self.data = np.array([[ 0.000000000000e+00, -4.938271604938e-02, -1.975308641975e-01, -4.444444444444e-01, -7.901234567901e-01, -1.234567901235e+00, -1.777777777778e+00, -2.419753086420e+00, -3.160493827160e+00, -4.000000000000e+00],
                              [ 0.000000000000e+00, -1.095471658524e-02, -6.572829951144e-02, -1.643207487786e-01, -3.067320643867e-01, -4.929622463358e-01, -7.230112946259e-01, -9.968792092569e-01, -1.314565990229e+00, -1.676071637542e+00],
                              [-0.000000000000e+00, -0.000000000000e+00,  5.988354788877e-02,  1.796506436663e-01,  3.593012873326e-01,  5.988354788877e-01,  8.982532183316e-01,  1.257554505664e+00,  1.676739340886e+00,  2.155807723996e+00],
                              [-0.000000000000e+00, -2.423886431385e-02,  4.847772862771e-02,  2.181497788247e-01,  4.847772862771e-01,  8.483602509849e-01,  1.308898672948e+00,  1.866392552167e+00,  2.520841888641e+00,  3.272246682370e+00],
                              [-0.000000000000e+00, -1.307406805844e-02, -0.000000000000e+00,  3.922220417533e-02,  1.045925444675e-01,  1.961110208767e-01,  3.137776334026e-01,  4.575923820455e-01,  6.275552668053e-01,  8.236662876819e-01],
                              [ 0.000000000000e+00,  5.531503365270e-02,  3.687668910180e-02, -5.531503365270e-02, -2.212601346108e-01, -4.609586137725e-01, -7.744104711378e-01, -1.161615706707e+00, -1.622574320479e+00, -2.157286312455e+00],
                              [ 0.000000000000e+00,  9.159187190627e-02,  9.159187190627e-02,  0.000000000000e+00, -1.831837438125e-01, -4.579593595313e-01, -8.243268471564e-01, -1.282286206688e+00, -1.831837438125e+00, -2.472980541469e+00],
                              [ 0.000000000000e+00,  9.398780817787e-03,  1.127853698134e-02,  5.639268490672e-03, -7.519024654229e-03, -2.819634245336e-02, -5.639268490672e-02, -9.210805201431e-02, -1.353424437761e-01, -1.860958601922e-01],
                              [-0.000000000000e+00, -1.273800005931e-01, -1.698400007908e-01, -1.273800005931e-01, -0.000000000000e+00,  2.123000009886e-01,  5.095200023725e-01,  8.916600041519e-01,  1.358720006327e+00,  1.910700008897e+00],
                              [-0.000000000000e+00, -1.450247087293e-01, -2.071781553275e-01, -1.864603397948e-01, -8.287126213101e-02,  1.035890776638e-01,  3.729206795895e-01,  7.251235436463e-01,  1.160197669834e+00,  1.678143058153e+00]],
                             dtype=np.float64)

    def tearDown(self):

        try:
            del self.interp_func
        except AttributeError:
            pass

    def init_2dlinear(self, x=None, y=None, data=None, extrapolate=False, extrapolation_range=float('inf'),
                      extrapolation_type='nearest', tolerate_single_value=False):
        """Create the interpolating function and reference function."""

        if x is None:
            x = self.x

        if y is None:
            y = self.y

        if data is None:
            data = self.data

        # reference interpolated data:

        # linearly interpolated data sampled on self.xsamples and self.ysamples
        # calculated from linear interpolator scipy.interpolate.interp2d:
        self.interp_data = np.array([
       [ 0.000000000000e+00, -1.532567049808e-02, -3.065134099617e-02, -4.597701149425e-02, -8.514261387825e-02, -1.311196253725e-01, -1.770966368668e-01, -2.401021711367e-01, -3.167305236271e-01, -3.933588761175e-01,
        -4.802043422733e-01, -5.874840357599e-01, -6.947637292465e-01, -8.054491272882e-01, -9.433801617710e-01, -1.081311196254e+00, -1.219242230736e+00, -1.384418901660e+00, -1.553001277139e+00, -1.721583652618e+00,
        -1.910600255428e+00, -2.109833971903e+00, -2.309067688378e+00, -2.521924223074e+00, -2.751809280545e+00, -2.981694338016e+00, -3.218390804598e+00, -3.478927203065e+00, -3.739463601533e+00, -4.000000000000e+00],
       [ 0.000000000000e+00, -1.162451953899e-02, -2.324903907799e-02, -3.487355861698e-02, -6.622191771995e-02, -1.032056595555e-01, -1.401894013910e-01, -1.912616000580e-01, -2.536045641901e-01, -3.159475283222e-01,
        -3.867435665531e-01, -4.744457529817e-01, -5.621479394104e-01, -6.526678172053e-01, -7.657292259305e-01, -8.787906346556e-01, -9.918520433808e-01, -1.127454983036e+00, -1.265875614058e+00, -1.404296245080e+00,
        -1.559623024299e+00, -1.723402877617e+00, -1.887182730936e+00, -2.062233349719e+00, -2.251372425334e+00, -2.440511500949e+00, -2.635285959296e+00, -2.849784257207e+00, -3.064282555119e+00, -3.278780853030e+00],
       [ 0.000000000000e+00, -7.923368579902e-03, -1.584673715980e-02, -2.377010573971e-02, -4.730122156166e-02, -7.529169373849e-02, -1.032821659153e-01, -1.424210289793e-01, -1.904786047531e-01, -2.385361805268e-01,
        -2.932827908329e-01, -3.614074702036e-01, -4.295321495742e-01, -4.998865071223e-01, -5.880782900899e-01, -6.762700730575e-01, -7.644618560251e-01, -8.704910644122e-01, -9.787499509768e-01, -1.087008837541e+00,
        -1.208645793170e+00, -1.336971783332e+00, -1.465297773493e+00, -1.602542476365e+00, -1.750935570123e+00, -1.899328663881e+00, -2.052181113995e+00, -2.220641311350e+00, -2.389101508705e+00, -2.557561706060e+00],
       [ 0.000000000000e+00, -4.222217620811e-03, -8.444435241622e-03, -1.266665286243e-02, -2.838052540337e-02, -4.737772792148e-02, -6.637493043960e-02, -9.358045790066e-02, -1.273526453161e-01, -1.611248327315e-01,
        -1.998220151127e-01, -2.483691874254e-01, -2.969163597381e-01, -3.471051970394e-01, -4.104273542494e-01, -4.737495114595e-01, -5.370716686695e-01, -6.135271457882e-01, -6.916242878955e-01, -7.697214300029e-01,
        -8.576685620417e-01, -9.505406890463e-01, -1.043412816051e+00, -1.142851603010e+00, -1.250498714912e+00, -1.358145826814e+00, -1.469076268693e+00, -1.591498365492e+00, -1.713920462291e+00, -1.836342559091e+00],
       [ 0.000000000000e+00, -2.579112822684e-03, -5.158225645369e-03, -7.737338468053e-03, -1.485130452493e-02, -2.326094293444e-02, -3.167058134395e-02, -4.331940063502e-02, -5.755956463135e-02, -7.179972862768e-02,
        -8.798340115295e-02, -1.080540907361e-01, -1.281247803192e-01, -1.488433060787e-01, -1.747445212487e-01, -2.006457364187e-01, -2.265469515886e-01, -2.576308561691e-01, -2.893625969259e-01, -3.210943376827e-01,
        -3.567130954974e-01, -3.942753618410e-01, -4.318376281846e-01, -4.719912392335e-01, -5.153840311639e-01, -5.587768230944e-01, -6.034652873774e-01, -6.526886048947e-01, -7.019119224120e-01, -7.511352399292e-01],
       [ 0.000000000000e+00, -1.524021213404e-03, -3.048042426809e-03, -4.572063640213e-03, -2.862505155026e-03, -2.290667559563e-04,  2.404371643114e-03,  7.347509826892e-03,  1.413840783844e-02,  2.092930584998e-02,
         2.910602373235e-02,  4.005438135637e-02,  5.100273898039e-02,  6.241303656135e-02,  7.751885379784e-02,  9.262467103434e-02,  1.077304882708e-01,  1.265318251629e-01,  1.457951020118e-01,  1.650583788608e-01,
         1.870932954514e-01,  2.105140319129e-01,  2.339347683743e-01,  2.592032646635e-01,  2.867814607374e-01,  3.143596568113e-01,  3.428617327991e-01,  3.745973884855e-01,  4.063330441719e-01,  4.380686998583e-01],
       [ 0.000000000000e+00, -4.689296041244e-04, -9.378592082489e-04, -1.406788812373e-03,  9.126294214880e-03,  2.280280942253e-02,  3.647932463017e-02,  5.801442028880e-02,  8.583638030822e-02,  1.136583403276e-01,
         1.461954486176e-01,  1.881628534488e-01,  2.301302582800e-01,  2.736693792014e-01,  3.297822288444e-01,  3.858950784873e-01,  4.420079281303e-01,  5.106945064948e-01,  5.809528009496e-01,  6.512110954043e-01,
         7.308996864002e-01,  8.153034256667e-01,  8.997071649332e-01,  9.903977685605e-01,  1.088946952639e+00,  1.187496136717e+00,  1.289188752976e+00,  1.401883381866e+00,  1.514578010756e+00,  1.627272639646e+00],
       [ 0.000000000000e+00, -1.296966580408e-03, -2.593933160817e-03, -3.890899741225e-03,  1.080960936119e-02,  3.008082580155e-02,  4.935204224192e-02,  8.005002702715e-02,  1.198894264883e-01,  1.597288259494e-01,
         2.064242864175e-01,  2.668318688994e-01,  3.272394513813e-01,  3.899323875322e-01,  4.709081530349e-01,  5.518839185375e-01,  6.328596840402e-01,  7.321182788947e-01,  8.336622274181e-01,  9.352061759416e-01,
         1.050462246479e+00,  1.172574378023e+00,  1.294686509567e+00,  1.425940055787e+00,  1.568620370352e+00,  1.711300684917e+00,  1.858551706820e+00,  2.021800204406e+00,  2.185048701992e+00,  2.348297199578e+00],
       [ 0.000000000000e+00, -3.631506425143e-03, -7.263012850287e-03, -1.089451927543e-02,  4.248537128620e-03,  2.475575434101e-02,  4.526297155340e-02,  7.918059078665e-02,  1.238265316366e-01,  1.684724724865e-01,
         2.211646545489e-01,  2.899493190364e-01,  3.587339835239e-01,  4.302007284155e-01,  5.231241165405e-01,  6.160475046655e-01,  7.089708927905e-01,  8.233509241488e-01,  9.404130359114e-01,  1.057475147674e+00,
         1.190629741861e+00,  1.331830577262e+00,  1.473031412662e+00,  1.624960569678e+00,  1.790300128716e+00,  1.955639687754e+00,  2.126343407599e+00,  2.315821690275e+00,  2.505299972950e+00,  2.694778255625e+00],
       [ 0.000000000000e+00, -5.966046269878e-03, -1.193209253976e-02, -1.789813880964e-02, -2.312535103948e-03,  1.943068288047e-02,  4.117390086489e-02,  7.831115454614e-02,  1.277636367849e-01,  1.772161190236e-01,
         2.359050226804e-01,  3.130667691734e-01,  3.902285156664e-01,  4.704690692988e-01,  5.753400800461e-01,  6.802110907934e-01,  7.850821015408e-01,  9.145835694030e-01,  1.047163844405e+00,  1.179744119406e+00,
         1.330797237244e+00,  1.491086776500e+00,  1.651376315756e+00,  1.823981083569e+00,  2.011979887079e+00,  2.199978690590e+00,  2.394135108379e+00,  2.609843176143e+00,  2.825551243908e+00,  3.041259311672e+00],
       [ 0.000000000000e+00, -7.163964431653e-03, -1.432792886331e-02, -2.149189329496e-02, -7.020894493625e-03,  1.363152237427e-02,  3.428393924217e-02,  7.038990127649e-02,  1.188586994439e-01,  1.673274976114e-01,
         2.250684228787e-01,  3.013536023457e-01,  3.776387818127e-01,  4.570146703130e-01,  5.611162310795e-01,  6.652177918461e-01,  7.693193526127e-01,  8.981465856455e-01,  1.030064527712e+00,  1.161982469778e+00,
         1.312444666043e+00,  1.472178989409e+00,  1.631913312775e+00,  1.804010472274e+00,  1.991561176939e+00,  2.179111881604e+00,  2.372844004336e+00,  2.588211090301e+00,  2.803578176265e+00,  3.018945262230e+00],
       [ 0.000000000000e+00, -6.088639227505e-03, -1.217727845501e-02, -1.826591768251e-02, -8.023828197519e-03,  6.884183776763e-03,  2.179219575104e-02,  4.836501394854e-02,  8.426967712461e-02,  1.201743403007e-01,
         1.630778872107e-01,  2.199792015885e-01,  2.768805159664e-01,  3.361147915889e-01,  4.140127571685e-01,  4.919107227482e-01,  5.698086883278e-01,  6.663703438646e-01,  7.652649606460e-01,  8.641595774274e-01,
         9.770519616767e-01,  1.096943229660e+00,  1.216834497643e+00,  1.346057610605e+00,  1.486945529790e+00,  1.627833448975e+00,  1.773387290649e+00,  1.935271861036e+00,  2.097156431423e+00,  2.259041001810e+00],
       [ 0.000000000000e+00, -5.013314023357e-03, -1.002662804671e-02, -1.503994207007e-02, -9.026761901412e-03,  1.368451792517e-04,  9.300452259915e-03,  2.634012662059e-02,  4.968065480528e-02,  7.302118298996e-02,
         1.010873515427e-01,  1.386048008314e-01,  1.761222501201e-01,  2.152149128648e-01,  2.669092832575e-01,  3.186036536502e-01,  3.702980240430e-01,  4.345941020837e-01,  5.004653935804e-01,  5.663366850772e-01,
         6.416592573100e-01,  7.217074699107e-01,  8.017556825115e-01,  8.881047489363e-01,  9.823298826411e-01,  1.076555016346e+00,  1.173930576963e+00,  1.282332631771e+00,  1.390734686580e+00,  1.499136741389e+00],
       [ 0.000000000000e+00, -3.325600294713e-03, -6.651200589426e-03, -9.976800884138e-03, -7.822304827502e-03, -4.102066956194e-03, -3.818290848869e-04,  7.252763323099e-03,  1.801883936043e-02,  2.878491539775e-02,
         4.189960415709e-02,  5.971151836044e-02,  7.752343256379e-02,  9.611821767447e-02,  1.209759700438e-01,  1.458337224132e-01,  1.706914747826e-01,  2.018121944106e-01,  2.337157849460e-01,  2.656193754814e-01,
         3.022201914608e-01,  3.411696201622e-01,  3.801190488636e-01,  4.221999611944e-01,  4.681952280618e-01,  5.141904949292e-01,  5.617515036113e-01,  6.147926086448e-01,  6.678337136782e-01,  7.208748187117e-01],
       [ 0.000000000000e+00,  3.261221629904e-03,  6.522443259808e-03,  9.783664889712e-03,  1.104127846884e-02,  1.172643260488e-02,  1.241158674092e-02,  1.166559226927e-02,  9.774678911451e-03,  7.883765553634e-03,
         5.134163031198e-03,  6.671821795221e-04, -3.799798672154e-03, -8.553009245370e-03, -1.559605759091e-02, -2.263910593644e-02, -2.968215428198e-02, -3.901504039983e-02, -4.863415623923e-02, -5.825327207863e-02,
        -6.958976624726e-02, -8.178494958052e-02, -9.398013291378e-02, -1.073202351332e-01, -1.220914859603e-01, -1.368627367874e-01, -1.522064470576e-01, -1.695537653786e-01, -1.869010836996e-01, -2.042484020205e-01],
       [ 0.000000000000e+00,  9.848043554521e-03,  1.969608710904e-02,  2.954413066356e-02,  2.990486176517e-02,  2.755493216595e-02,  2.520500256673e-02,  1.607842121543e-02,  1.530518462475e-03, -1.301738429049e-02,
        -3.163127809469e-02, -5.837715400139e-02, -8.512302990809e-02, -1.132242361652e-01, -1.521680852257e-01, -1.911119342861e-01, -2.300557833465e-01, -2.798422752103e-01, -3.309840974245e-01, -3.821259196387e-01,
        -4.413997239553e-01, -5.047395193233e-01, -5.680793146912e-01, -6.368404314608e-01, -7.123781999824e-01, -7.879159685041e-01, -8.661643977266e-01, -9.539001394020e-01, -1.041635881077e+00, -1.129371622753e+00],
       [ 0.000000000000e+00,  1.643486547914e-02,  3.286973095827e-02,  4.930459643741e-02,  4.876844506151e-02,  4.338343172703e-02,  3.799841839254e-02,  2.049125016160e-02, -6.713641986501e-03, -3.391853413461e-02,
        -6.839671922058e-02, -1.174214901823e-01, -1.664462611440e-01, -2.178954630851e-01, -2.887401128604e-01, -3.595847626357e-01, -4.304294124111e-01, -5.206695100208e-01, -6.133340386097e-01, -7.059985671987e-01,
        -8.132096816634e-01, -9.276940890660e-01, -1.042178496469e+00, -1.166360627788e+00, -1.302664914005e+00, -1.438969200221e+00, -1.580122348396e+00, -1.738246513425e+00, -1.896370678455e+00, -2.054494843485e+00],
       [ 0.000000000000e+00,  2.027248054413e-02,  4.054496108826e-02,  6.081744163238e-02,  6.209956376667e-02,  5.795586921242e-02,  5.381217465818e-02,  3.610393838261e-02,  7.544068729995e-03, -2.101580092262e-02,
        -5.771439560803e-02, -1.106904403590e-01, -1.636664851100e-01, -2.193554382053e-01, -2.967476580546e-01, -3.741398779040e-01, -4.515320977534e-01, -5.506275843568e-01, -6.524359793046e-01, -7.542443742523e-01,
        -8.723302192656e-01, -9.985547893117e-01, -1.124779359358e+00, -1.261855562781e+00, -1.412496307925e+00, -1.563137053070e+00, -1.719203614903e+00, -1.894260535146e+00, -2.069317455389e+00, -2.244374375632e+00],
       [ 0.000000000000e+00,  2.376644475167e-02,  4.753288950333e-02,  7.129933425500e-02,  7.473912439792e-02,  7.237129893836e-02,  7.000347347879e-02,  5.311660901298e-02,  2.461451334217e-02, -3.887582328628e-03,
        -4.110110140318e-02, -9.573746728521e-02, -1.503738331672e-01, -2.079140068505e-01, -2.886846429438e-01, -3.694552790371e-01, -4.502259151303e-01, -5.542270136336e-01, -6.611319199381e-01, -7.680368262426e-01,
        -8.923645793546e-01, -1.025403755870e+00, -1.158442932386e+00, -1.303097340107e+00, -1.462270786834e+00, -1.621444233561e+00, -1.786425295890e+00, -1.971733012828e+00, -2.157040729767e+00, -2.342348446705e+00],
       [ 0.000000000000e+00,  2.726040895920e-02,  5.452081791841e-02,  8.178122687761e-02,  8.737868502918e-02,  8.678672866429e-02,  8.619477229940e-02,  7.012927964334e-02,  4.168495795436e-02,  1.324063626537e-02,
        -2.448780719832e-02, -8.078449421140e-02, -1.370811812245e-01, -1.964725754958e-01, -2.806216278330e-01, -3.647706801702e-01, -4.489197325073e-01, -5.578264429104e-01, -6.698278605716e-01, -7.818292782329e-01,
        -9.123989394436e-01, -1.052252722429e+00, -1.192106505414e+00, -1.344339117433e+00, -1.512045265742e+00, -1.679751414052e+00, -1.853646976877e+00, -2.049205490511e+00, -2.244764004145e+00, -2.440322517778e+00],
       [ 0.000000000000e+00,  2.314750493313e-02,  4.629500986625e-02,  6.944251479938e-02,  7.468028083549e-02,  7.480097861531e-02,  7.492167639514e-02,  6.224970353424e-02,  3.934359416076e-02,  1.643748478728e-02,
        -1.414422697063e-02, -6.007714349740e-02, -1.060100600242e-01, -1.545015106791e-01, -2.234612343592e-01, -2.924209580393e-01, -3.613806817193e-01, -4.508086784246e-01, -5.427952092580e-01, -6.347817400913e-01,
        -7.421194756936e-01, -8.571328136803e-01, -9.721461516669e-01, -1.097393626166e+00, -1.235433771306e+00, -1.373473916446e+00, -1.516631129842e+00, -1.677698082136e+00, -1.838765034429e+00, -1.999831986722e+00],
       [ 0.000000000000e+00,  1.523116679024e-02,  3.046233358048e-02,  4.569350037072e-02,  4.931289422932e-02,  4.961463867888e-02,  4.991638312844e-02,  4.192400405540e-02,  2.729632616428e-02,  1.266864827316e-02,
        -6.935503731516e-03, -3.649260396331e-02, -6.604970419511e-02, -9.726562913143e-02, -1.417521517039e-01, -1.862386742764e-01, -2.307251968489e-01, -2.884823170575e-01, -3.478982619706e-01, -4.073142068838e-01,
        -4.766831000241e-01, -5.510284672779e-01, -6.253738345317e-01, -7.063545006037e-01, -7.956292901982e-01, -8.849040797927e-01, -9.774965187962e-01, -1.081700730731e+00, -1.185904942667e+00, -1.290109154602e+00],
       [ 0.000000000000e+00,  7.314828647353e-03,  1.462965729471e-02,  2.194448594206e-02,  2.394550762315e-02,  2.442829874244e-02,  2.491108986174e-02,  2.159830457656e-02,  1.524905816780e-02,  8.899811759040e-03,
         2.732195075943e-04, -1.290806442922e-02, -2.608934836604e-02, -4.002974758375e-02, -6.004306904862e-02, -8.005639051350e-02, -1.000697119784e-01, -1.261559556904e-01, -1.530013146833e-01, -1.798466736763e-01,
        -2.112467243546e-01, -2.449241208756e-01, -2.786015173965e-01, -3.153153750411e-01, -3.558248090902e-01, -3.963342431392e-01, -4.383619077500e-01, -4.857033793271e-01, -5.330448509042e-01, -5.803863224813e-01],
       [ 0.000000000000e+00, -2.938114550952e-03, -5.876229101905e-03, -8.814343652857e-03, -1.048975820787e-02, -1.180440133548e-02, -1.311904446309e-02, -1.353175902217e-02, -1.322293072643e-02, -1.291410243069e-02,
        -1.206411699384e-02, -1.013181727475e-02, -8.199517555668e-03, -6.086832122878e-03, -2.531060980448e-03,  1.024710161982e-03,  4.580481304412e-03,  9.579338156483e-03,  1.475858072226e-02,  1.993782328803e-02,
         2.619938013604e-02,  3.300209412516e-02,  3.980480811428e-02,  4.732906495822e-02,  5.575525037069e-02,  6.418143578316e-02,  7.296839262304e-02,  8.301804945885e-02,  9.306770629466e-02,  1.031173631305e-01],
       [ 0.000000000000e+00, -1.611181406853e-02, -3.222362813706e-02, -4.833544220560e-02, -5.625987306996e-02, -6.184505159026e-02, -6.743023011057e-02, -6.716727777076e-02, -6.222582074285e-02, -5.728436371493e-02,
        -4.883402817094e-02, -3.336593559481e-02, -1.789784301867e-02, -1.260124270508e-03,  2.473460385385e-02,  5.072933197821e-02,  7.672406010257e-02,  1.120757976031e-01,  1.485971612757e-01,  1.851185249483e-01,
         2.286576456530e-01,  2.757056448738e-01,  3.227536440946e-01,  3.744801480035e-01,  4.320547827726e-01,  4.896294175416e-01,  5.495433046547e-01,  6.176445749719e-01,  6.857458452892e-01,  7.538471156064e-01],
       [ 0.000000000000e+00, -2.928551358611e-02, -5.857102717222e-02, -8.785654075834e-02, -1.020299879320e-01, -1.118857018450e-01, -1.217414157581e-01, -1.208027965193e-01, -1.112287107593e-01, -1.016546249992e-01,
        -8.560393934805e-02, -5.660005391486e-02, -2.759616848167e-02,  3.566583581862e-03,  5.200026868815e-02,  1.004339537944e-01,  1.488676389007e-01,  2.145722570498e-01,  2.824357418292e-01,  3.502992266086e-01,
         4.311159111700e-01,  5.184091956225e-01,  6.057024800750e-01,  7.016312310489e-01,  8.083543151744e-01,  9.150773993000e-01,  1.026118216686e+00,  1.152271100485e+00,  1.278423984284e+00,  1.404576868082e+00],
       [ 0.000000000000e+00, -3.990937562575e-02, -7.981875125151e-02, -1.197281268773e-01, -1.391736764096e-01, -1.527724184867e-01, -1.663711605637e-01, -1.653528840026e-01, -1.526409925310e-01, -1.399291010594e-01,
        -1.184469984049e-01, -7.942447338456e-02, -4.040194836425e-02,  1.543980383689e-03,  6.687713895268e-02,  1.322102975217e-01,  1.975434560907e-01,  2.862638444807e-01,  3.779076365984e-01,  4.695514287161e-01,
         5.787356431996e-01,  6.966900688659e-01,  8.146444945323e-01,  9.442925351092e-01,  1.088557594324e+00,  1.232822653539e+00,  1.382934520210e+00,  1.553510212973e+00,  1.724085905737e+00,  1.894661598501e+00],
       [ 0.000000000000e+00, -4.160880649261e-02, -8.321761298522e-02, -1.248264194778e-01, -1.461248725413e-01, -1.616203674821e-01, -1.771158624230e-01, -1.781039620573e-01, -1.674861454464e-01, -1.568683288355e-01,
        -1.375460750407e-01, -1.008149468780e-01, -6.408381871536e-02, -2.445121149139e-02,  3.839322822304e-02,  1.012376679375e-01,  1.640821076519e-01,  2.501383798568e-01,  3.390961311229e-01,  4.280538823891e-01,
         5.344205080231e-01,  6.494915708411e-01,  7.645626336590e-01,  8.912396127222e-01,  1.032423987092e+00,  1.173608361462e+00,  1.320595693954e+00,  1.487893379875e+00,  1.655191065797e+00,  1.822488751718e+00],
       [ 0.000000000000e+00, -4.330823735947e-02, -8.661647471893e-02, -1.299247120784e-01, -1.530760686730e-01, -1.704683164776e-01, -1.878605642823e-01, -1.908550401120e-01, -1.823312983618e-01, -1.738075566116e-01,
        -1.566451516765e-01, -1.222054203715e-01, -8.776568906646e-02, -5.044640336646e-02,  9.909317493388e-03,  7.026503835324e-02,  1.306207592131e-01,  2.140129152328e-01,  3.002846256475e-01,  3.865563360621e-01,
         4.901053728467e-01,  6.022930728162e-01,  7.144807727857e-01,  8.381866903351e-01,  9.762903798595e-01,  1.114394069384e+00,  1.258256867698e+00,  1.422276546777e+00,  1.586296225856e+00,  1.750315904936e+00],
       [ 0.000000000000e+00, -4.500766822632e-02, -9.001533645265e-02, -1.350230046790e-01, -1.600272648047e-01, -1.793162654731e-01, -1.986052661416e-01, -2.036061181667e-01, -1.971764512772e-01, -1.907467843878e-01,
        -1.757442283123e-01, -1.435958938649e-01, -1.114475594176e-01, -7.644159524153e-02, -1.857459323626e-02,  3.929240876901e-02,  9.715941077428e-02,  1.778874506088e-01,  2.614731201720e-01,  3.450587897351e-01,
         4.457902376702e-01,  5.550945747913e-01,  6.643989119124e-01,  7.851337679481e-01,  9.201567726271e-01,  1.055179777306e+00,  1.195918041442e+00,  1.356659713679e+00,  1.517401385916e+00,  1.678143058153e+00]
                                    ],
                                    dtype=np.float64)
    #################

        # Nearestly extrapolated data sampled on self.xsamples_ex and
        # self.ysamples_ex.
        # Calculated from nearest edge value.
        self.extrap_data_nea = [
        [  # x inf
            np.array([  # y inf
                [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00],
                [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00],
                [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00]
            ], dtype=np.float64),
            np.array([  # y mid
                [ 0.000000000000e+00, -1.532567049808e-02, -3.065134099617e-02, -4.597701149425e-02, -8.514261387825e-02, -1.311196253725e-01, -1.770966368668e-01, -2.401021711367e-01, -3.167305236271e-01, -3.933588761175e-01,
                 -4.802043422733e-01, -5.874840357599e-01, -6.947637292465e-01, -8.054491272882e-01, -9.433801617710e-01, -1.081311196254e+00, -1.219242230736e+00, -1.384418901660e+00, -1.553001277139e+00, -1.721583652618e+00,
                 -1.910600255428e+00, -2.109833971903e+00, -2.309067688378e+00, -2.521924223074e+00, -2.751809280545e+00, -2.981694338016e+00, -3.218390804598e+00, -3.478927203065e+00, -3.739463601533e+00, -4.000000000000e+00],
                [ 0.000000000000e+00, -1.532567049808e-02, -3.065134099617e-02, -4.597701149425e-02, -8.514261387825e-02, -1.311196253725e-01, -1.770966368668e-01, -2.401021711367e-01, -3.167305236271e-01, -3.933588761175e-01,
                 -4.802043422733e-01, -5.874840357599e-01, -6.947637292465e-01, -8.054491272882e-01, -9.433801617710e-01, -1.081311196254e+00, -1.219242230736e+00, -1.384418901660e+00, -1.553001277139e+00, -1.721583652618e+00,
                 -1.910600255428e+00, -2.109833971903e+00, -2.309067688378e+00, -2.521924223074e+00, -2.751809280545e+00, -2.981694338016e+00, -3.218390804598e+00, -3.478927203065e+00, -3.739463601533e+00, -4.000000000000e+00],
                [ 0.000000000000e+00, -1.532567049808e-02, -3.065134099617e-02, -4.597701149425e-02, -8.514261387825e-02, -1.311196253725e-01, -1.770966368668e-01, -2.401021711367e-01, -3.167305236271e-01, -3.933588761175e-01,
                 -4.802043422733e-01, -5.874840357599e-01, -6.947637292465e-01, -8.054491272882e-01, -9.433801617710e-01, -1.081311196254e+00, -1.219242230736e+00, -1.384418901660e+00, -1.553001277139e+00, -1.721583652618e+00,
                 -1.910600255428e+00, -2.109833971903e+00, -2.309067688378e+00, -2.521924223074e+00, -2.751809280545e+00, -2.981694338016e+00, -3.218390804598e+00, -3.478927203065e+00, -3.739463601533e+00, -4.000000000000e+00]
            ], dtype=np.float64),
            np.array([  # y sup
                [-4.000000000000e+00, -4.000000000000e+00, -4.000000000000e+00],
                [-4.000000000000e+00, -4.000000000000e+00, -4.000000000000e+00],
                [-4.000000000000e+00, -4.000000000000e+00, -4.000000000000e+00]
            ], dtype=np.float64)
        ],
        [  # x mid
            np.array([  # y inf
                [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,
                  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,
                  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00],
                [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,
                  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,
                  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00],
                [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,
                  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,
                  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00]
            ], dtype=np.float64).T,

            None,  # y mid (interpolation area)

            np.array([  # y sup
                [-4.000000000000e+00, -3.278780853030e+00, -2.557561706060e+00, -1.836342559091e+00, -7.511352399292e-01,  4.380686998583e-01,  1.627272639646e+00,  2.348297199578e+00,  2.694778255625e+00,  3.041259311672e+00,
                  3.018945262230e+00,  2.259041001810e+00,  1.499136741389e+00,  7.208748187117e-01, -2.042484020205e-01, -1.129371622753e+00, -2.054494843485e+00, -2.244374375632e+00, -2.342348446705e+00, -2.440322517778e+00,
                 -1.999831986722e+00, -1.290109154602e+00, -5.803863224813e-01,  1.031173631305e-01,  7.538471156064e-01,  1.404576868082e+00,  1.894661598501e+00,  1.822488751718e+00,  1.750315904936e+00,  1.678143058153e+00],
                [-4.000000000000e+00, -3.278780853030e+00, -2.557561706060e+00, -1.836342559091e+00, -7.511352399292e-01,  4.380686998583e-01,  1.627272639646e+00,  2.348297199578e+00,  2.694778255625e+00,  3.041259311672e+00,
                  3.018945262230e+00,  2.259041001810e+00,  1.499136741389e+00,  7.208748187117e-01, -2.042484020205e-01, -1.129371622753e+00, -2.054494843485e+00, -2.244374375632e+00, -2.342348446705e+00, -2.440322517778e+00,
                 -1.999831986722e+00, -1.290109154602e+00, -5.803863224813e-01,  1.031173631305e-01,  7.538471156064e-01,  1.404576868082e+00,  1.894661598501e+00,  1.822488751718e+00,  1.750315904936e+00,  1.678143058153e+00],
                [-4.000000000000e+00, -3.278780853030e+00, -2.557561706060e+00, -1.836342559091e+00, -7.511352399292e-01,  4.380686998583e-01,  1.627272639646e+00,  2.348297199578e+00,  2.694778255625e+00,  3.041259311672e+00,
                  3.018945262230e+00,  2.259041001810e+00,  1.499136741389e+00,  7.208748187117e-01, -2.042484020205e-01, -1.129371622753e+00, -2.054494843485e+00, -2.244374375632e+00, -2.342348446705e+00, -2.440322517778e+00,
                 -1.999831986722e+00, -1.290109154602e+00, -5.803863224813e-01,  1.031173631305e-01,  7.538471156064e-01,  1.404576868082e+00,  1.894661598501e+00,  1.822488751718e+00,  1.750315904936e+00,  1.678143058153e+00]
            ], dtype=np.float64).T
        ],
        [  # x sup
            np.array([  # y inf
                [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00],
                [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00],
                [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00]
            ], dtype=np.float64),
            np.array([  # y mid
                [ 0.000000000000e+00, -4.500766822632e-02, -9.001533645265e-02, -1.350230046790e-01, -1.600272648047e-01, -1.793162654731e-01, -1.986052661416e-01, -2.036061181667e-01, -1.971764512772e-01, -1.907467843878e-01,
                 -1.757442283123e-01, -1.435958938649e-01, -1.114475594176e-01, -7.644159524153e-02, -1.857459323626e-02,  3.929240876901e-02,  9.715941077428e-02,  1.778874506088e-01,  2.614731201720e-01,  3.450587897351e-01,
                  4.457902376702e-01,  5.550945747913e-01,  6.643989119124e-01,  7.851337679481e-01,  9.201567726271e-01,  1.055179777306e+00,  1.195918041442e+00,  1.356659713679e+00,  1.517401385916e+00,  1.678143058153e+00],
                [ 0.000000000000e+00, -4.500766822632e-02, -9.001533645265e-02, -1.350230046790e-01, -1.600272648047e-01, -1.793162654731e-01, -1.986052661416e-01, -2.036061181667e-01, -1.971764512772e-01, -1.907467843878e-01,
                 -1.757442283123e-01, -1.435958938649e-01, -1.114475594176e-01, -7.644159524153e-02, -1.857459323626e-02,  3.929240876901e-02,  9.715941077428e-02,  1.778874506088e-01,  2.614731201720e-01,  3.450587897351e-01,
                  4.457902376702e-01,  5.550945747913e-01,  6.643989119124e-01,  7.851337679481e-01,  9.201567726271e-01,  1.055179777306e+00,  1.195918041442e+00,  1.356659713679e+00,  1.517401385916e+00,  1.678143058153e+00],
                [ 0.000000000000e+00, -4.500766822632e-02, -9.001533645265e-02, -1.350230046790e-01, -1.600272648047e-01, -1.793162654731e-01, -1.986052661416e-01, -2.036061181667e-01, -1.971764512772e-01, -1.907467843878e-01,
                 -1.757442283123e-01, -1.435958938649e-01, -1.114475594176e-01, -7.644159524153e-02, -1.857459323626e-02,  3.929240876901e-02,  9.715941077428e-02,  1.778874506088e-01,  2.614731201720e-01,  3.450587897351e-01,
                  4.457902376702e-01,  5.550945747913e-01,  6.643989119124e-01,  7.851337679481e-01,  9.201567726271e-01,  1.055179777306e+00,  1.195918041442e+00,  1.356659713679e+00,  1.517401385916e+00,  1.678143058153e+00]
            ], dtype=np.float64),
            np.array([  # y sup
                [ 1.678143058153e+00,  1.678143058153e+00,  1.678143058153e+00],
                [ 1.678143058153e+00,  1.678143058153e+00,  1.678143058153e+00],
                [ 1.678143058153e+00,  1.678143058153e+00,  1.678143058153e+00]
            ], dtype=np.float64)
        ]
                                ]

        # Linearly extrapolated data sampled on self.xsamples_ex and
        # self.ysamples_ex.
        # Calculated from the nearest bilinear part of the interpolation f:
        # if f is known at four points (xn, yn), (xr, yr), (xn, yr) and (xr, yn)
        # (inside the interpolation area), one can calculate the value of f on
        # every point (x, y) aligned with (xn, yn) and (xr, yr) (in the
        # extrapolation area in particular) via this formula:
        # f(x, y) = f(xn, yn) + (x-xn)*(f(xn, yn)-f(xr, yn))/(xn-xr)
        #                     + (y-yn)*(f(xn, yn)-f(xn, yr))/(yn-yr)
        #                     + (x-xn)*(y-yn)*(f(xn, yn)-f(xr, yn)-f(xn, yr)+f(xr, yr))/(xn-xr)/(yn-yr)
        self.extrap_data_lin = [
        [  # x inf
            np.array([  # y inf
                [ 5.547663047777e-02,  3.467289404860e-02,  1.386915761944e-02],
                [ 4.800622738194e-02,  3.000389211371e-02,  1.200155684548e-02],
                [ 4.053582428611e-02,  2.533489017882e-02,  1.013395607153e-02]
            ], dtype=np.float64),
            np.array([  # y mid
                [ 0.000000000000e+00, -2.391234072318e-02, -4.782468144635e-02, -7.173702216953e-02, -1.290386289655e-01, -1.958800260680e-01, -2.627214231704e-01, -3.534122960392e-01, -4.631827495209e-01, -5.729532030027e-01,
                 -6.970333419442e-01, -8.497328518052e-01, -1.002432361666e+00, -1.159901766681e+00, -1.355530332921e+00, -1.551158899161e+00, -1.746787465402e+00, -1.980575192868e+00, -2.219132815488e+00, -2.457690438107e+00,
                 -2.724867431646e+00, -3.006354110645e+00, -3.287840789644e+00, -3.588407049256e+00, -3.912822784635e+00, -4.237238520013e+00, -4.571194045697e+00, -4.938538837455e+00, -5.305883629212e+00, -5.673228420970e+00],
                [ 0.000000000000e+00, -2.069233938877e-02, -4.138467877753e-02, -6.207701816630e-02, -1.125776233078e-01, -1.715948758072e-01, -2.306121283065e-01, -3.109209992007e-01, -4.082631648107e-01, -5.056053304207e-01,
                 -6.157224670676e-01, -7.513895457882e-01, -8.870566245089e-01, -1.026982026908e+00, -1.200974018740e+00, -1.374966010571e+00, -1.548958002402e+00, -1.757016583665e+00, -1.969333488607e+00, -2.181650393549e+00,
                 -2.419517240564e+00, -2.670159058617e+00, -2.920800876669e+00, -3.188475989438e+00, -3.477442720601e+00, -3.766409451764e+00, -4.063892830285e+00, -4.391184474559e+00, -4.718476118832e+00, -5.045767763106e+00],
                [ 0.000000000000e+00, -1.747233805436e-02, -3.494467610871e-02, -5.241701416307e-02, -9.611661765006e-02, -1.473097255464e-01, -1.985028334427e-01, -2.684297023623e-01, -3.533435801005e-01, -4.382574578388e-01,
                 -5.344115921910e-01, -6.530462397712e-01, -7.716808873514e-01, -8.940622871363e-01, -1.046417704558e+00, -1.198773121981e+00, -1.351128539403e+00, -1.533457974462e+00, -1.719534161726e+00, -1.905610348990e+00,
                 -2.114167049482e+00, -2.333964006588e+00, -2.553760963695e+00, -2.788544929619e+00, -3.042062656567e+00, -3.295580383515e+00, -3.556591614873e+00, -3.843830111663e+00, -4.131068608453e+00, -4.418307105242e+00]
            ], dtype=np.float64),
            np.array([  # y sup
                [-5.886288400189e+00, -6.205878369018e+00, -6.525468337847e+00],
                [-5.235596916785e+00, -5.520340647303e+00, -5.805084377815e+00],
                [-4.584905433381e+00, -4.834802925588e+00, -5.084700417795e+00]
            ], dtype=np.float64)
        ],
        [  # x mid
            np.array([  # y inf
                [ 3.555555555556e-02,  2.696888533046e-02,  1.838221510537e-02,  9.795544880282e-03,  5.983541748628e-03,  3.535729215098e-03,  1.087916681569e-03,  3.008962466547e-03,  8.425094906333e-03,  1.384122734612e-02,
                  1.662039748143e-02,  1.412564300781e-02,  1.163088853419e-02,  7.715392683734e-03, -7.566034181377e-03, -2.284746104649e-02, -3.812888791160e-02, -4.703215486238e-02, -5.513815182386e-02, -6.324414878535e-02,
                 -5.370221144485e-02, -3.533630695335e-02, -1.697040246186e-02,  6.816425758209e-03,  3.737940863899e-02,  6.794239151978e-02,  9.258975145175e-02,  9.653243106286e-02,  1.004751106740e-01,  1.044177902851e-01],
                [ 2.222222222222e-02,  1.685555333154e-02,  1.148888444086e-02,  6.122215550176e-03,  3.739713592892e-03,  2.209830759436e-03,  6.799479259804e-04,  1.880601541592e-03,  5.265684316458e-03,  8.650767091324e-03,
                  1.038774842590e-02,  8.828526879882e-03,  7.269305333868e-03,  4.822120427334e-03, -4.728771363361e-03, -1.427966315405e-02, -2.383055494475e-02, -2.939509678899e-02, -3.446134488992e-02, -3.952759299084e-02,
                 -3.356388215303e-02, -2.208519184585e-02, -1.060650153866e-02,  4.260266098881e-03,  2.336213039937e-02,  4.246399469986e-02,  5.786859465734e-02,  6.033276941428e-02,  6.279694417123e-02,  6.526111892817e-02],
                [ 8.888888888889e-03,  6.742221332616e-03,  4.595553776343e-03,  2.448886220071e-03,  1.495885437157e-03,  8.839323037746e-04,  2.719791703922e-04,  7.522406166368e-04,  2.106273726583e-03,  3.460306836529e-03,
                  4.155099370358e-03,  3.531410751953e-03,  2.907722133547e-03,  1.928848170933e-03, -1.891508545344e-03, -5.711865261622e-03, -9.532221977900e-03, -1.175803871559e-02, -1.378453795597e-02, -1.581103719634e-02,
                 -1.342555286121e-02, -8.834076738339e-03, -4.242600615465e-03,  1.704106439552e-03,  9.344852159749e-03,  1.698559787994e-02,  2.314743786294e-02,  2.413310776571e-02,  2.511877766849e-02,  2.610444757127e-02]
            ], dtype=np.float64).T,

            None,  # y mid (interpolation area)

            np.array([  # y sup
                [-4.151111111111e+00, -3.403189865819e+00, -2.655268620526e+00, -1.907347375234e+00, -7.796847640892e-01,  4.564753801564e-01,  1.692635524402e+00,  2.442981328177e+00,  2.804675659576e+00,  3.166369990976e+00,
                  3.143858172090e+00,  2.352934052634e+00,  1.562009933178e+00,  7.516386596311e-01, -2.143098466467e-01, -1.180258352925e+00, -2.146206859202e+00, -2.345907389372e+00, -2.449826922529e+00, -2.553746455686e+00,
                 -2.093250819052e+00, -1.350547597524e+00, -6.078443759961e-01,  1.089461640952e-01,  7.933458523905e-01,  1.477745540686e+00,  1.993595500304e+00,  1.919521409553e+00,  1.845447318801e+00,  1.771373228050e+00],
                [-4.377777777778e+00, -3.589803385002e+00, -2.801828992226e+00, -2.013854599449e+00, -8.225090503292e-01,  4.840854006036e-01,  1.790679851536e+00,  2.585007521077e+00,  2.969521765504e+00,  3.354036009931e+00,
                  3.331227536879e+00,  2.493773628870e+00,  1.656319720862e+00,  7.977844210103e-01, -2.294020135860e-01, -1.256588448182e+00, -2.283774882778e+00, -2.498206909984e+00, -2.611044636265e+00, -2.723882362547e+00,
                 -2.233379067548e+00, -1.441205261908e+00, -6.490314562681e-01,  1.176893655424e-01,  8.525939575665e-01,  1.587498549591e+00,  2.141996353008e+00,  2.065070396304e+00,  1.988144439600e+00,  1.911218482896e+00],
                [-4.604444444444e+00, -3.776416904184e+00, -2.948389363925e+00, -2.120361823665e+00, -8.653333365693e-01,  5.116954210508e-01,  1.888724178670e+00,  2.727033713976e+00,  3.134367871431e+00,  3.541702028886e+00,
                  3.518596901668e+00,  2.634613205107e+00,  1.750629508546e+00,  8.439301823894e-01, -2.444941805253e-01, -1.332918543440e+00, -2.421342906354e+00, -2.650506430595e+00, -2.772262350002e+00, -2.894018269408e+00,
                 -2.373507316043e+00, -1.531862926292e+00, -6.902185365402e-01,  1.264325669896e-01,  9.118420627426e-01,  1.697251558495e+00,  2.290397205713e+00,  2.210619383056e+00,  2.130841560399e+00,  2.051063737743e+00]
            ], dtype=np.float64).T
        ],
        [  # x sup
            np.array([  # y inf
                [ 1.067045444595e-01,  6.669034028720e-02,  2.667613611488e-02],
                [ 1.101346757212e-01,  6.883417232574e-02,  2.753366893029e-02],
                [ 1.135648069828e-01,  7.097800436427e-02,  2.839120174571e-02]
            ], dtype=np.float64),
            np.array([  # y mid
                [ 0.000000000000e+00, -4.599333812910e-02, -9.198667625820e-02, -1.379800143873e-01, -1.640589585611e-01, -1.844480758905e-01, -2.048371932199e-01, -2.110017434384e-01, -2.057866399682e-01, -2.005715364979e-01,
                 -1.868216927611e-01, -1.560023684911e-01, -1.251830442212e-01, -9.151880652908e-02, -3.509526145946e-02,  2.132828361016e-02,  7.775182867978e-02,  1.569346811269e-01,  2.389624469962e-01,  3.209902128655e-01,
                  4.200874592679e-01,  5.277194459369e-01,  6.353514326059e-01,  7.543630729636e-01,  8.875992804323e-01,  1.020835487901e+00,  1.159761522214e+00,  1.318601950482e+00,  1.477442378751e+00,  1.636282807019e+00],
                [ 0.000000000000e+00, -4.747184298327e-02, -9.494368596653e-02, -1.424155289498e-01, -1.701064991957e-01, -1.921457915166e-01, -2.141850838375e-01, -2.220951813460e-01, -2.187019230046e-01, -2.153086646631e-01,
                 -2.034378894342e-01, -1.746120804304e-01, -1.457862714267e-01, -1.141346234604e-01, -5.987626379425e-02, -5.617904128113e-03,  4.864045553803e-02,  1.255055269040e-01,  2.051964372325e-01,  2.848873475610e-01,
                  3.815332916644e-01,  4.866567526553e-01,  5.917802136461e-01,  7.082070304869e-01,  8.387630421401e-01,  9.693190537932e-01,  1.105526743371e+00,  1.261515305687e+00,  1.417503868003e+00,  1.573492430318e+00],
                [ 0.000000000000e+00, -4.895034783743e-02, -9.790069567486e-02, -1.468510435123e-01, -1.761540398303e-01, -1.998435071427e-01, -2.235329744551e-01, -2.331886192536e-01, -2.316172060410e-01, -2.300457928284e-01,
                 -2.200540861074e-01, -1.932217923698e-01, -1.663894986321e-01, -1.367504403917e-01, -8.465726612904e-02, -3.256409186639e-02,  1.952908239628e-02,  9.407637268118e-02,  1.714304274689e-01,  2.487844822565e-01,
                  3.429791240609e-01,  4.455940593736e-01,  5.482089946863e-01,  6.620509880102e-01,  7.899268038479e-01,  9.178026196855e-01,  1.051291964529e+00,  1.204428660892e+00,  1.357565357254e+00,  1.510702053617e+00]
            ], dtype=np.float64),
            np.array([  # y sup
                [ 1.728410255414e+00,  1.866601428008e+00,  2.004792600599e+00],
                [ 1.663965796461e+00,  1.799675845674e+00,  1.935385894887e+00],
                [ 1.599521337508e+00,  1.732750263344e+00,  1.865979189174e+00]
            ], dtype=np.float64)
        ]
                                ]

        self.interp_func = interpolators2d.Interpolate2DLinear(x, y, data,
                                                               extrapolate=extrapolate,
                                                               extrapolation_range=extrapolation_range,
                                                               extrapolation_type=extrapolation_type,
                                                               tolerate_single_value=tolerate_single_value)

    def init_2dcubic(self, x=None, y=None, data=None, extrapolate=False, extrapolation_range=float('inf'),
                      extrapolation_type='nearest', tolerate_single_value=False):
        """Create the interpolating function and reference function."""

        if x is None:
            x = self.x

        if y is None:
            y = self.y

        if data is None:
            data = self.data

        # reference interpolated data:

        # C2 cubic interpolated data sampled on self.xsamples and self.ysamples
        # calculated from solving manually the linear system given by the
        # following constraints on the cubic splines:
        #
        # if Pij is the cubic polynomial defining the spline on area
        # [x[i], x[i+1]]*[y[j], y[j+1]], constraints are:
        #   Pij(x[i  ], y[j  ]) = data[i  , j  ]
        #   Pij(x[i+1], y[j  ]) = data[i+1, j  ]
        #   Pij(x[i  ], y[j+1]) = data[i  , j+1]
        #   Pij(x[i+1], y[j+1]) = data[i+1, j+1]
        #   dPij/dx(x[i  ], y[j  ]) = (data[i+1, j  ]-data[i-1, j  ])/(x[i+1]-x[i-1])
        #   dPij/dx(x[i+1], y[j  ]) = (data[i+2, j  ]-data[i  , j  ])/(x[i+2]-x[i  ])
        #   dPij/dx(x[i  ], y[j+1]) = (data[i+1, j+1]-data[i-1, j+1])/(x[i+1]-x[i-1])
        #   dPij/dx(x[i+1], y[j+1]) = (data[i+2, j+1]-data[i  , j+1])/(x[i+2]-x[i  ])
        #   dPij/dy(x[i  ], y[j  ]) = (data[i  , j+1]-data[i  , j-1])/(y[j+1]-y[j-1])
        #   dPij/dy(x[i+1], y[j  ]) = (data[i+1, j+1]-data[i+1, j-1])/(y[i+1]-y[i-1])
        #   dPij/dy(x[i  ], y[j+1]) = (data[i  , j+2]-data[i  , j  ])/(y[j+2]-y[j  ])
        #   dPij/dy(x[i+1], y[j+1]) = (data[i+1, j+2]-data[i+1, j  ])/(y[j+2]-y[j  ])
        #   d2Pij/dxdy(x[i  ], y[j  ]) = (data[i+1, j+1]-data[i+1, j-1]-data[i-1, j+1]+data[i-1, j-1])/(x[i+1]-x[i-1])/(y[j+1]-y[j-1])
        #   d2Pij/dxdy(x[i+1], y[j  ]) = (data[i+2, j+1]-data[i+2, j-1]-data[i  , j+1]+data[i  , j-1])/(x[i+2]-x[i  ])/(y[j+2]-y[j  ])
        #   d2Pij/dxdy(x[i  ], y[j+1]) = (data[i+1, j+2]-data[i+1, j  ]-data[i-1, j+2]+data[i-1, j  ])/(x[i+1]-x[i-1])/(y[j+2]-y[j  ])
        #   d2Pij/dxdy(x[i+1], y[j+1]) = (data[i+2, j+2]-data[i+2, j  ]-data[i  , j+2]+data[i  , j  ])/(x[i+2]-x[i  ])/(y[j+2]-y[j  ])
        # for the edges of the interpolation area, where x, y or data are
        # requested with indices out of their range, they return the nearest
        # valid value, eg: x[-1] = x[0], y[10] = y[9], data[-1, 5] = data[0, 5], etc
        self.interp_data = np.array([
       [ 0.000000000000e+00, -1.204550320955e-02, -2.343497296140e-02, -4.302486093457e-02, -7.609988109394e-02, -1.189060642093e-01, -1.712247324614e-01, -2.330558858502e-01, -3.043995243757e-01, -3.852556480381e-01,
        -4.756242568371e-01, -5.755053507729e-01, -6.848989298454e-01, -8.038049940547e-01, -9.322235434007e-01, -1.070154577883e+00, -1.217598097503e+00, -1.374554102259e+00, -1.541022592152e+00, -1.717003567182e+00,
        -1.902497027348e+00, -2.097502972652e+00, -2.302021403092e+00, -2.516052318668e+00, -2.739595719382e+00, -2.972651605232e+00, -3.215438654038e+00, -3.471710835030e+00, -3.736183434244e+00, -4.000000000000e+00],
       [ 0.000000000000e+00, -8.573742992670e-03, -1.659396452661e-02, -3.153320429471e-02, -5.785038515901e-02, -9.237813432786e-02, -1.349319446483e-01, -1.855118161205e-01, -2.441177487442e-01, -3.107497425196e-01,
        -3.854077974467e-01, -4.680919135253e-01, -5.588020907557e-01, -6.575383291376e-01, -7.643006286712e-01, -8.790889893564e-01, -1.001903411193e+00, -1.132743894182e+00, -1.271610438322e+00, -1.418503043614e+00,
        -1.573421710057e+00, -1.736366437652e+00, -1.907337226399e+00, -2.086334076297e+00, -2.273356987347e+00, -2.468405959549e+00, -2.671665500055e+00, -2.886302981657e+00, -3.107859481494e+00, -3.328862459872e+00],
       [ 0.000000000000e+00, -5.034593223167e-03, -9.616025846155e-03, -1.986196597138e-02, -3.942287398917e-02, -6.570566424294e-02, -9.855928319931e-02, -1.379837308583e-01, -1.839790072198e-01, -2.365451122840e-01,
        -2.956820460507e-01, -3.613898085200e-01, -4.336683996920e-01, -5.125178195665e-01, -5.979380681436e-01, -6.899291454233e-01, -7.884910514056e-01, -8.936237860905e-01, -1.005327349478e+00, -1.123601741568e+00,
        -1.248446962361e+00, -1.379863011856e+00, -1.517849890054e+00, -1.662407596954e+00, -1.813536132557e+00, -1.971235496863e+00, -2.135656743405e+00, -2.309392957839e+00, -2.488793679775e+00, -2.667741241111e+00],
       [ 0.000000000000e+00, -2.337812861419e-03, -4.349715236437e-03, -1.043549869146e-02, -2.322055325583e-02, -4.083994670181e-02, -6.318504220060e-02, -9.025583975220e-02, -1.220523393566e-01, -1.585745410138e-01,
        -1.998224447238e-01, -2.457960504867e-01, -2.964953583023e-01, -3.519203681707e-01, -4.120710800920e-01, -4.769474940661e-01, -5.465496100929e-01, -6.208774281726e-01, -6.999309483051e-01, -7.837101704904e-01,
        -8.722150947285e-01, -9.654457210194e-01, -1.063402049363e+00, -1.166084079760e+00, -1.273491812209e+00, -1.385625246711e+00, -1.502593020095e+00, -1.626260064588e+00, -1.754000990161e+00, -1.881416005248e+00],
       [ 0.000000000000e+00, -1.058002874044e-03, -1.978501578211e-03, -4.617802405832e-03, -1.008357783646e-02, -1.758899845359e-02, -2.708822953393e-02, -3.858127107747e-02, -5.206812308422e-02, -6.754878555417e-02,
        -8.502325848733e-02, -1.044915418837e-01, -1.259536357433e-01, -1.494095400660e-01, -1.748592548520e-01, -2.023027801012e-01, -2.317401158136e-01, -2.631712619892e-01, -2.965962186280e-01, -3.320149857300e-01,
        -3.694275632952e-01, -4.088339513236e-01, -4.502341498153e-01, -4.936281587701e-01, -5.390159781881e-01, -5.863976080694e-01, -6.358188831371e-01, -6.880666328079e-01, -7.420331846021e-01, -7.958622322264e-01],
       [ 0.000000000000e+00, -8.123130058188e-04, -1.725852053577e-03, -1.374065577089e-03,  1.058478428161e-03,  4.992542055517e-03,  1.039438329100e-02,  1.726400213461e-02,  2.560139858634e-02,  3.540657264621e-02,
         4.667952431419e-02,  5.942025359031e-02,  7.362876047455e-02,  8.930504496692e-02,  1.064491070674e-01,  1.250609467760e-01,  1.451405640928e-01,  1.666879590176e-01,  1.897031315506e-01,  2.141860816918e-01,
         2.401368094410e-01,  2.675553147984e-01,  2.964415977639e-01,  3.267956583375e-01,  3.586174965193e-01,  3.919071123092e-01,  4.266982477213e-01,  4.635701406620e-01,  5.017073591270e-01,  5.397433515500e-01],
       [ 0.000000000000e+00, -1.453142876909e-03, -3.229069779428e-03, -9.701963618287e-04,  7.923682026631e-03,  2.160552346163e-02,  3.996773326797e-02,  6.301031144564e-02,  9.073325799465e-02,  1.231365729150e-01,
         1.602202562067e-01,  2.019843078697e-01,  2.484287279040e-01,  2.995535163097e-01,  3.553586730868e-01,  4.158441982351e-01,  4.810100917548e-01,  5.508563536459e-01,  6.253829839082e-01,  7.045899825420e-01,
         7.884773495470e-01,  8.770450849234e-01,  9.702931886711e-01,  1.068221660790e+00,  1.170830501281e+00,  1.278119710142e+00,  1.390196882051e+00,  1.508909058930e+00,  1.631656036129e+00,  1.754080229303e+00],
       [ 0.000000000000e+00, -3.163083246993e-03, -6.801334860442e-03, -4.499981893200e-03,  7.568720828956e-03,  2.668575432020e-02,  5.269272912504e-02,  8.558964524349e-02,  1.253765026755e-01,  1.720533014212e-01,
         2.256200414804e-01,  2.860767228533e-01,  3.534233455397e-01,  4.276599095398e-01,  5.087864148534e-01,  5.968028614807e-01,  6.917092494216e-01,  7.935055786760e-01,  9.021918492441e-01,  1.017768061126e+00,
         1.140234214321e+00,  1.269590308830e+00,  1.405836344652e+00,  1.548972321788e+00,  1.698998240238e+00,  1.855914100001e+00,  2.019878290534e+00,  2.193609830821e+00,  2.373280975689e+00,  2.552476952190e+00],
       [ 0.000000000000e+00, -6.530485959289e-03, -1.364938634617e-02, -1.341310638817e-02, -1.081640974135e-03,  1.997797178249e-02,  4.956959373918e-02,  8.769322489593e-02,  1.343488652527e-01,  1.895365148096e-01,
         2.532561735666e-01,  3.255078415236e-01,  4.062915186806e-01,  4.956072050378e-01,  5.934549005950e-01,  6.998346053522e-01,  8.147463193096e-01,  9.381900424669e-01,  1.070165774824e+00,  1.210673516382e+00,
         1.359713267139e+00,  1.517285027097e+00,  1.683388796255e+00,  1.858024574613e+00,  2.041192362170e+00,  2.232892158928e+00,  2.433320103029e+00,  2.645843232585e+00,  2.865721542487e+00,  3.085011437961e+00],
       [ 0.000000000000e+00, -9.731615017943e-03, -2.011043419610e-02, -2.239920137153e-02, -1.138432747579e-02,  9.230741463202e-03,  3.923027072537e-02,  7.861426031071e-02,  1.273827102192e-01,  1.855356204509e-01,
         2.530729910058e-01,  3.299948218838e-01,  4.163011130850e-01,  5.119918646094e-01,  6.170670764570e-01,  7.315267486277e-01,  8.553708811216e-01,  9.885994739387e-01,  1.131212527079e+00,  1.283210040542e+00,
         1.444592014329e+00,  1.615358448439e+00,  1.795509342872e+00,  1.985044697628e+00,  2.183964512707e+00,  2.392268788110e+00,  2.610173258555e+00,  2.841381370072e+00,  3.080679533592e+00,  3.319330492951e+00],
       [ 0.000000000000e+00, -1.083635035891e-02, -2.229877657480e-02, -2.593525457840e-02, -1.670239552178e-02,  1.817255413431e-03,  2.941500627490e-02,  6.609085706264e-02,  1.118448077766e-01,  1.666768584169e-01,
         2.305870089834e-01,  3.035752594762e-01,  3.856416098952e-01,  4.767860602405e-01,  5.770086105121e-01,  6.863092607100e-01,  8.046880108341e-01,  9.321448608844e-01,  1.068679810861e+00,  1.214292860764e+00,
         1.368984010593e+00,  1.532753260348e+00,  1.705600610030e+00,  1.887526059638e+00,  2.078529609172e+00,  2.278611258633e+00,  2.487979699972e+00,  2.710217478372e+00,  2.940281204983e+00,  3.169718855738e+00],
       [ 0.000000000000e+00, -1.031128189467e-02, -2.113260221892e-02, -2.557844217350e-02, -1.954015885354e-02, -5.936305494911e-03,  1.506310509253e-02,  4.345807290876e-02,  7.924859795381e-02,  1.224346802277e-01,
         1.730163197303e-01,  2.309935164618e-01,  2.963662704220e-01,  3.691345816111e-01,  4.492984500290e-01,  5.368578756757e-01,  6.318128585512e-01,  7.341633986555e-01,  8.439094959886e-01,  9.610511505505e-01,
         1.085588362341e+00,  1.217521131361e+00,  1.356849457609e+00,  1.503573341086e+00,  1.657692781792e+00,  1.819207779727e+00,  1.988288347701e+00,  2.167853038950e+00,  2.353793210567e+00,  2.539223343756e+00],
       [ 0.000000000000e+00, -8.286296850577e-03, -1.690974517404e-02, -2.131880008640e-02, -1.879751916716e-02, -1.127515806674e-02,  1.135899390570e-03,  1.843565320477e-02,  4.062410337584e-02,  6.770124990381e-02,
         9.966709278866e-02,  1.365216320304e-01,  1.782648676290e-01,  2.248967995845e-01,  2.764174278969e-01,  3.328267525662e-01,  3.941247735923e-01,  4.603114909754e-01,  5.313869047153e-01,  6.073510148121e-01,
         6.882038212658e-01,  7.739453240764e-01,  8.645755232439e-01,  9.600944187682e-01,  1.060502010649e+00,  1.165798298888e+00,  1.276095667306e+00,  1.393323371556e+00,  1.514765469219e+00,  1.635870415409e+00],
       [ 0.000000000000e+00, -4.373300840840e-03, -8.900197631666e-03, -1.150714504766e-02, -1.095684238059e-02, -8.128199788714e-03, -3.072415922034e-03,  4.210509219455e-03,  1.372057563576e-02,  2.545778332687e-02,
         3.942213229279e-02,  5.561362253349e-02,  7.403225404902e-02,  9.467802683936e-02,  1.175509409045e-01,  1.426509962445e-01,  1.699781928592e-01,  1.995325307488e-01,  2.313140099131e-01,  2.653226303524e-01,
         3.015583920665e-01,  3.400212950552e-01,  3.807113393188e-01,  4.236285248572e-01,  4.687728516705e-01,  5.161443197586e-01,  5.657941277700e-01,  6.186011858640e-01,  6.733281933343e-01,  7.279016048551e-01],
       [ 0.000000000000e+00,  2.645399192067e-03,  5.352864092081e-03,  7.284507642764e-03,  7.940356085648e-03,  7.675563194001e-03,  6.510817537129e-03,  4.446119115027e-03,  1.481467927710e-03, -2.383136024828e-03,
        -7.147692742592e-03, -1.281220222562e-02, -1.937666447386e-02, -2.684107948727e-02, -3.520544726594e-02, -4.446976780981e-02, -5.463404111894e-02, -6.569826719339e-02, -7.766244603299e-02, -9.052657763780e-02,
        -1.042906620077e-01, -1.189546991430e-01, -1.345186890434e-01, -1.509826317092e-01, -1.683465271402e-01, -1.866103753363e-01, -2.057948648667e-01, -2.262551495078e-01, -2.474912554931e-01, -2.686652957755e-01],
       [ 0.000000000000e+00,  1.136328800300e-02,  2.302046420617e-02,  3.100403790715e-02,  3.294657638231e-02,  3.052977322154e-02,  2.385159115824e-02,  1.291203019238e-02, -2.288909676002e-03, -2.175122844694e-02,
        -4.547492612041e-02, -7.346000269641e-02, -1.057064581750e-01, -1.422142925560e-01, -1.829835058396e-01, -2.280140980258e-01, -2.773060691145e-01, -3.308594191057e-01, -3.886741479995e-01, -4.507502557958e-01,
        -5.170877424947e-01, -5.876866080961e-01, -6.625468526000e-01, -7.416684760065e-01, -8.250514783156e-01, -9.126958595271e-01, -1.004699582376e+00, -1.102744340452e+00, -1.204462701026e+00, -1.305887173403e+00],
       [ 0.000000000000e+00,  1.896316620142e-02,  3.840199547316e-02,  5.189503636599e-02,  5.561055859129e-02,  5.227041194028e-02,  4.203315076974e-02,  2.489877507967e-02,  8.672848700670e-04, -3.006131985907e-02,
        -6.788703910772e-02, -1.126098728759e-01, -1.642298211636e-01, -2.227468839709e-01, -2.881610612977e-01, -3.604723531440e-01, -4.396807595099e-01, -5.257862803953e-01, -6.187889158002e-01, -7.186886657246e-01,
        -8.254855301687e-01, -9.391795091322e-01, -1.059770602615e+00, -1.187258810618e+00, -1.321644133140e+00, -1.462926570182e+00, -1.611264676099e+00, -1.769380300688e+00, -1.933441713650e+00, -2.097027463548e+00],
       [ 0.000000000000e+00,  2.490962471124e-02,  5.039990789806e-02,  6.863196014009e-02,  7.492825482847e-02,  7.261144879570e-02,  6.187509486697e-02,  4.271919304229e-02,  1.514374332165e-02, -2.085125429494e-02,
        -6.526579980746e-02, -1.180998932160e-01, -1.793535345204e-01, -2.490267237209e-01, -3.271194608172e-01, -4.136317458095e-01, -5.085635786978e-01, -6.119149594820e-01, -7.236858881622e-01, -8.438763647384e-01,
        -9.724863892105e-01, -1.109515961579e+00, -1.254965081843e+00, -1.408833750003e+00, -1.571121966058e+00, -1.741829730010e+00, -1.921150594683e+00, -2.112407216909e+00, -2.310922070081e+00, -2.508856264776e+00],
       [ 0.000000000000e+00,  3.005480310832e-02,  6.074444469676e-02,  8.349860528369e-02,  9.320330822369e-02,  9.349124037523e-02,  8.457401456502e-02,  6.645163079306e-02,  3.912408905934e-02,  2.591389363877e-03,
        -4.314646829332e-02, -9.808948391228e-02, -1.622376574930e-01, -2.355909890355e-01, -3.181494785397e-01, -4.099131260057e-01, -5.108819314334e-01, -6.210558948229e-01, -7.404350161741e-01, -8.690192954871e-01,
        -1.006808732762e+00, -1.153803327998e+00, -1.310003081197e+00, -1.475407992357e+00, -1.650018061478e+00, -1.833833288562e+00, -2.027065286433e+00, -2.233346741952e+00, -2.447563678473e+00, -2.661145776513e+00],
       [ 0.000000000000e+00,  3.197125619639e-02,  6.456939815845e-02,  8.933146804956e-02,  1.012075527573e-01,  1.037848319409e-01,  9.727226752219e-02,  8.166985950117e-02,  5.697760787786e-02,  2.319551265223e-02,
        -1.967642617569e-02, -7.163820860590e-02, -1.326898346384e-01, -2.028313042733e-01, -2.820626175104e-01, -3.703837743498e-01, -4.677947747916e-01, -5.742956188356e-01, -6.898863064819e-01, -8.145668377305e-01,
        -9.483372125815e-01, -1.091197431035e+00, -1.243147493090e+00, -1.404187398748e+00, -1.574317148008e+00, -1.753536740871e+00, -1.942055139257e+00, -2.143459522827e+00, -2.352699978468e+00, -2.561313548344e+00],
       [ 0.000000000000e+00,  2.851665652795e-02,  5.757114925571e-02,  7.990268948580e-02,  9.117870783082e-02,  9.447682254525e-02,  8.997631236240e-02,  7.767717728223e-02,  5.757941730475e-02,  2.968303243002e-02,
        -6.011977342065e-03, -4.950561201149e-02, -1.007978715782e-01, -1.598887560422e-01, -2.267782654033e-01, -3.014663996620e-01, -3.839531588179e-01, -4.742385428710e-01, -5.723225518215e-01, -6.782051856694e-01,
        -7.918864444148e-01, -9.133663280572e-01, -1.042644836597e+00, -1.179721970034e+00, -1.324597728368e+00, -1.477272111600e+00, -1.637924398462e+00, -1.809632207203e+00, -1.988062968461e+00, -2.165955893500e+00],
       [ 0.000000000000e+00,  2.036080533780e-02,  4.108553577920e-02,  5.726120242547e-02,  6.595618638637e-02,  6.925294797702e-02,  6.727279556531e-02,  6.001572915122e-02,  4.748174873476e-02,  2.967085431595e-02,
         6.583045894712e-03, -2.178167652881e-02, -5.542331295477e-02, -9.434186338310e-02, -1.385373278138e-01, -1.880097062469e-01, -2.427589986824e-01, -3.027852051201e-01, -3.680883255603e-01, -4.386683600030e-01,
        -5.145253084479e-01, -5.956591708953e-01, -6.820699473448e-01, -7.737576377968e-01, -8.707222422512e-01, -9.729637607078e-01, -1.080603501528e+00, -1.195723925044e+00, -1.315393412358e+00, -1.434698974559e+00],
       [ 0.000000000000e+00,  9.556463779888e-03,  1.927076202228e-02,  2.701212948337e-02,  3.150912188188e-02,  3.366490308658e-02,  3.353208458497e-02,  3.111066637706e-02,  2.640064846281e-02,  1.940203084226e-02,
         1.011481351542e-02, -1.461003517679e-03, -1.532542025705e-02, -3.147843670293e-02, -4.992005285505e-02, -7.065026871344e-02, -9.366908427825e-02, -1.189764995492e-01, -1.465725145266e-01, -1.764571292101e-01,
        -2.086303436001e-01, -2.430921576961e-01, -2.798425714987e-01, -3.188815850075e-01, -3.602091982222e-01, -4.038254111442e-01, -4.497828352428e-01, -4.989846344229e-01, -5.501593643871e-01, -6.011762598841e-01],
       [ 0.000000000000e+00, -2.558296061050e-03, -5.153638015560e-03, -7.285906301767e-03, -8.656675666766e-03, -9.477930945418e-03, -9.762020768799e-03, -9.508945137044e-03, -8.718704050013e-03, -7.391297507782e-03,
        -5.526725510405e-03, -3.124988057719e-03, -1.860851498332e-04,  3.289983213273e-03,  7.303217031643e-03,  1.185361630487e-02,  1.694118103353e-02,  2.256591121772e-02,  2.872780685672e-02,  3.542686795143e-02,
         4.266309450105e-02,  5.043648650604e-02,  5.874704396615e-02,  6.759476688116e-02,  7.697965525218e-02,  8.690170907791e-02,  9.737327700077e-02,  1.086063438279e-01,  1.203024843389e-01,  1.319615789444e-01],
       [ 0.000000000000e+00, -1.828710161987e-02, -3.683050630719e-02, -5.217012265134e-02, -6.224128705340e-02, -6.851062262157e-02, -7.106356371174e-02, -6.990011032380e-02, -6.502026245777e-02, -5.642402011370e-02,
        -4.411138329166e-02, -2.808235199134e-02, -8.336926213017e-03,  1.512489404322e-02,  4.230310877781e-02,  7.319771799026e-02,  1.078087216809e-01,  1.461361198496e-01,  1.881799124961e-01,  2.339400996209e-01,
         2.834166812235e-01,  3.366096573047e-01,  3.935190278637e-01,  4.541447929011e-01,  5.184869524157e-01,  5.865455064097e-01,  6.584058892486e-01,  7.355347240091e-01,  8.158673471307e-01,  8.959436671767e-01],
       [ 0.000000000000e+00, -3.377027073222e-02, -6.799876404405e-02, -9.649947511070e-02, -1.155811664852e-01, -1.278658895955e-01, -1.335063853013e-01, -1.325026536026e-01, -1.248546944996e-01, -1.105625079920e-01,
        -8.962609408010e-02, -6.204545276378e-02, -2.782058404281e-02,  1.304851208223e-02,  6.056183561213e-02,  1.147193865461e-01,  1.755211648846e-01,  2.429671706276e-01,  3.170574037748e-01,  3.977918643272e-01,
         4.851705522829e-01,  5.791934676438e-01,  6.798606104091e-01,  7.871719805789e-01,  9.011275781533e-01,  1.021727403132e+00,  1.149124196386e+00,  1.285940009305e+00,  1.428483604498e+00,  1.570568977092e+00],
       [ 0.000000000000e+00, -4.392395123557e-02, -8.842979125286e-02, -1.256620214986e-01, -1.509332045645e-01, -1.675730373682e-01, -1.757754828369e-01, -1.755405409706e-01, -1.668682117693e-01, -1.497584952331e-01,
        -1.242113913618e-01, -9.022690015573e-02, -4.780502161458e-02,  3.054244261338e-03,  6.235089747276e-02,  1.300849380188e-01,  2.062563658998e-01,  2.908651811162e-01,  3.839113836675e-01,  4.853949735534e-01,
         5.953159507742e-01,  7.136743153298e-01,  8.404700672213e-01,  9.757032064476e-01,  1.119373733008e+00,  1.271481646904e+00,  1.432220911076e+00,  1.604921222406e+00,  1.784895143508e+00,  1.964287175837e+00],
       [ 0.000000000000e+00, -4.733405498822e-02, -9.527287306609e-02, -1.356521525234e-01, -1.636001906935e-01, -1.825775763672e-01, -1.927858972409e-01, -1.942251533148e-01, -1.868953445888e-01, -1.707964710630e-01,
        -1.459285327371e-01, -1.122915296115e-01, -6.988546168603e-02, -1.871032896076e-02,  4.123386856466e-02,  1.099471308899e-01,  1.874294580151e-01,  2.736808499398e-01,  3.687013066646e-01,  4.724908281896e-01,
         5.850494145137e-01,  7.063770656380e-01,  8.364737815623e-01,  9.753395622870e-01,  1.122974407811e+00,  1.279378318135e+00,  1.444752880965e+00,  1.622558685089e+00,  1.807924027813e+00,  1.992684607445e+00],
       [ 0.000000000000e+00, -4.741379453005e-02, -9.540311567173e-02, -1.361983541681e-01, -1.651633234254e-01, -1.855913146103e-01, -1.976741699265e-01, -2.014118893741e-01, -1.968044729533e-01, -1.838519206637e-01,
        -1.625542325056e-01, -1.329114084789e-01, -9.492344858359e-02, -4.859035281976e-02,  6.087878812766e-03,  6.911124631401e-02,  1.404797496838e-01,  2.201933889222e-01,  3.082521640289e-01,  4.046560750043e-01,
         5.094051218482e-01,  6.224993045609e-01,  7.439386231414e-01,  8.737230775919e-01,  1.011852667910e+00,  1.158327394098e+00,  1.313339098415e+00,  1.480181071867e+00,  1.654217128031e+00,  1.827677657555e+00],
       [ 0.000000000000e+00, -4.775996320873e-02, -9.607038541395e-02, -1.375000701631e-01, -1.676147644291e-01, -1.895643169138e-01, -2.035322139496e-01, -2.095184555363e-01, -2.075230416741e-01, -1.975459723629e-01,
        -1.795872476025e-01, -1.536468673932e-01, -1.197248317351e-01, -7.782114062794e-02, -2.793579407142e-02,  2.993120793377e-02,  9.577986538790e-02,  1.696101782914e-01,  2.514221466436e-01,  3.412157704449e-01,
         4.389910496951e-01,  5.447479843945e-01,  6.584865745429e-01,  7.802068201398e-01,  9.099087211863e-01,  1.047592277682e+00,  1.193440975974e+00,  1.350604664727e+00,  1.514649090965e+00,  1.678143058154e+00]
                                    ],
                                    dtype=np.float64)

        # Nearestly extrapolated data sampled on self.xsamples_ex and
        # self.ysamples_ex.
        # Calculated from nearest edge value
        self.extrap_data_nea = [
        [  # x inf
            np.array([  # y inf
        [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00],
        [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00],
        [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00]
            ], dtype=np.float64),
            np.array([  # y mid
        [ 0.000000000000e+00, -1.204550320955e-02, -2.343497296140e-02, -4.302486093457e-02, -7.609988109394e-02, -1.189060642093e-01, -1.712247324614e-01, -2.330558858502e-01, -3.043995243757e-01, -3.852556480381e-01,
         -4.756242568371e-01, -5.755053507729e-01, -6.848989298454e-01, -8.038049940547e-01, -9.322235434007e-01, -1.070154577883e+00, -1.217598097503e+00, -1.374554102259e+00, -1.541022592152e+00, -1.717003567182e+00,
         -1.902497027348e+00, -2.097502972652e+00, -2.302021403092e+00, -2.516052318668e+00, -2.739595719382e+00, -2.972651605232e+00, -3.215438654038e+00, -3.471710835030e+00, -3.736183434244e+00, -4.000000000000e+00],
        [ 0.000000000000e+00, -1.204550320955e-02, -2.343497296140e-02, -4.302486093457e-02, -7.609988109394e-02, -1.189060642093e-01, -1.712247324614e-01, -2.330558858502e-01, -3.043995243757e-01, -3.852556480381e-01,
         -4.756242568371e-01, -5.755053507729e-01, -6.848989298454e-01, -8.038049940547e-01, -9.322235434007e-01, -1.070154577883e+00, -1.217598097503e+00, -1.374554102259e+00, -1.541022592152e+00, -1.717003567182e+00,
         -1.902497027348e+00, -2.097502972652e+00, -2.302021403092e+00, -2.516052318668e+00, -2.739595719382e+00, -2.972651605232e+00, -3.215438654038e+00, -3.471710835030e+00, -3.736183434244e+00, -4.000000000000e+00],
        [ 0.000000000000e+00, -1.204550320955e-02, -2.343497296140e-02, -4.302486093457e-02, -7.609988109394e-02, -1.189060642093e-01, -1.712247324614e-01, -2.330558858502e-01, -3.043995243757e-01, -3.852556480381e-01,
         -4.756242568371e-01, -5.755053507729e-01, -6.848989298454e-01, -8.038049940547e-01, -9.322235434007e-01, -1.070154577883e+00, -1.217598097503e+00, -1.374554102259e+00, -1.541022592152e+00, -1.717003567182e+00,
         -1.902497027348e+00, -2.097502972652e+00, -2.302021403092e+00, -2.516052318668e+00, -2.739595719382e+00, -2.972651605232e+00, -3.215438654038e+00, -3.471710835030e+00, -3.736183434244e+00, -4.000000000000e+00]
            ], dtype=np.float64),
            np.array([  # y sup
        [-4.000000000000e+00, -4.000000000000e+00, -4.000000000000e+00],
        [-4.000000000000e+00, -4.000000000000e+00, -4.000000000000e+00],
        [-4.000000000000e+00, -4.000000000000e+00, -4.000000000000e+00]
            ], dtype=np.float64)
        ],
        [  # x mid
            np.array([  # y inf
        [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,
          0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,
          0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00],
        [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,
          0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,
          0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00],
        [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,
          0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,
          0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00]
            ], dtype=np.float64).T,

            None,  # y mid (interpolation area)

            np.array([  # y sup
        [-4.000000000000e+00, -3.328862459872e+00, -2.667741241111e+00, -1.881416005248e+00, -7.958622322264e-01,  5.397433515500e-01,  1.754080229303e+00,  2.552476952190e+00,  3.085011437961e+00,  3.319330492951e+00,
          3.169718855738e+00,  2.539223343756e+00,  1.635870415409e+00,  7.279016048551e-01, -2.686652957755e-01, -1.305887173403e+00, -2.097027463548e+00, -2.508856264776e+00, -2.661145776513e+00, -2.561313548344e+00,
         -2.165955893500e+00, -1.434698974559e+00, -6.011762598841e-01,  1.319615789444e-01,  8.959436671767e-01,  1.570568977092e+00,  1.964287175837e+00,  1.992684607445e+00,  1.827677657555e+00,  1.678143058154e+00],
        [-4.000000000000e+00, -3.328862459872e+00, -2.667741241111e+00, -1.881416005248e+00, -7.958622322264e-01,  5.397433515500e-01,  1.754080229303e+00,  2.552476952190e+00,  3.085011437961e+00,  3.319330492951e+00,
          3.169718855738e+00,  2.539223343756e+00,  1.635870415409e+00,  7.279016048551e-01, -2.686652957755e-01, -1.305887173403e+00, -2.097027463548e+00, -2.508856264776e+00, -2.661145776513e+00, -2.561313548344e+00,
         -2.165955893500e+00, -1.434698974559e+00, -6.011762598841e-01,  1.319615789444e-01,  8.959436671767e-01,  1.570568977092e+00,  1.964287175837e+00,  1.992684607445e+00,  1.827677657555e+00,  1.678143058154e+00],
        [-4.000000000000e+00, -3.328862459872e+00, -2.667741241111e+00, -1.881416005248e+00, -7.958622322264e-01,  5.397433515500e-01,  1.754080229303e+00,  2.552476952190e+00,  3.085011437961e+00,  3.319330492951e+00,
          3.169718855738e+00,  2.539223343756e+00,  1.635870415409e+00,  7.279016048551e-01, -2.686652957755e-01, -1.305887173403e+00, -2.097027463548e+00, -2.508856264776e+00, -2.661145776513e+00, -2.561313548344e+00,
         -2.165955893500e+00, -1.434698974559e+00, -6.011762598841e-01,  1.319615789444e-01,  8.959436671767e-01,  1.570568977092e+00,  1.964287175837e+00,  1.992684607445e+00,  1.827677657555e+00,  1.678143058154e+00]
            ], dtype=np.float64).T
        ],
        [  # x sup
            np.array([  # y inf
        [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00],
        [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00],
        [ 0.000000000000e+00,  0.000000000000e+00,  0.000000000000e+00]
            ], dtype=np.float64),
            np.array([  # y mid
        [ 0.000000000000e+00, -4.775996320873e-02, -9.607038541395e-02, -1.375000701631e-01, -1.676147644291e-01, -1.895643169138e-01, -2.035322139496e-01, -2.095184555363e-01, -2.075230416741e-01, -1.975459723629e-01,
         -1.795872476025e-01, -1.536468673932e-01, -1.197248317351e-01, -7.782114062794e-02, -2.793579407142e-02,  2.993120793377e-02,  9.577986538790e-02,  1.696101782914e-01,  2.514221466436e-01,  3.412157704449e-01,
          4.389910496951e-01,  5.447479843945e-01,  6.584865745429e-01,  7.802068201398e-01,  9.099087211863e-01,  1.047592277682e+00,  1.193440975974e+00,  1.350604664727e+00,  1.514649090965e+00,  1.678143058154e+00],
        [ 0.000000000000e+00, -4.775996320873e-02, -9.607038541395e-02, -1.375000701631e-01, -1.676147644291e-01, -1.895643169138e-01, -2.035322139496e-01, -2.095184555363e-01, -2.075230416741e-01, -1.975459723629e-01,
         -1.795872476025e-01, -1.536468673932e-01, -1.197248317351e-01, -7.782114062794e-02, -2.793579407142e-02,  2.993120793377e-02,  9.577986538790e-02,  1.696101782914e-01,  2.514221466436e-01,  3.412157704449e-01,
          4.389910496951e-01,  5.447479843945e-01,  6.584865745429e-01,  7.802068201398e-01,  9.099087211863e-01,  1.047592277682e+00,  1.193440975974e+00,  1.350604664727e+00,  1.514649090965e+00,  1.678143058154e+00],
        [ 0.000000000000e+00, -4.775996320873e-02, -9.607038541395e-02, -1.375000701631e-01, -1.676147644291e-01, -1.895643169138e-01, -2.035322139496e-01, -2.095184555363e-01, -2.075230416741e-01, -1.975459723629e-01,
         -1.795872476025e-01, -1.536468673932e-01, -1.197248317351e-01, -7.782114062794e-02, -2.793579407142e-02,  2.993120793377e-02,  9.577986538790e-02,  1.696101782914e-01,  2.514221466436e-01,  3.412157704449e-01,
          4.389910496951e-01,  5.447479843945e-01,  6.584865745429e-01,  7.802068201398e-01,  9.099087211863e-01,  1.047592277682e+00,  1.193440975974e+00,  1.350604664727e+00,  1.514649090965e+00,  1.678143058154e+00]
            ], dtype=np.float64),
            np.array([  # y sup
        [ 1.678143058154e+00,  1.678143058154e+00,  1.678143058154e+00],
        [ 1.678143058154e+00,  1.678143058154e+00,  1.678143058154e+00],
        [ 1.678143058154e+00,  1.678143058154e+00,  1.678143058154e+00]
            ], dtype=np.float64)
        ]
                                ]

        # Linearly extrapolated data sampled on self.xsamples_ex and
        # self.ysamples_ex.
        # Calculated at (x, y) from the formula:
        # Pij(x_n, y_n) + (x-x_n)*dPij/dx(x_n, y_n) + (y-y_n)*dPij/dy(x_n, y_n)
        # where (x_n, y_n) is the nearest coordinates in the
        # interpolated range form (x, y), and Pij the nearest interpolation
        # polynomial.
        # NB: this extrapolation is not entirely bilinear as the d2Pij/dxdy
        # term is not taken into account.
        self.extrap_data_lin = [
        [  # x inf
            np.array([  # y inf
        [ 3.555555555556e-02,  2.222222222222e-02,  8.888888888889e-03],
        [ 3.555555555556e-02,  2.222222222222e-02,  8.888888888889e-03],
        [ 3.555555555556e-02,  2.222222222222e-02,  8.888888888889e-03]
            ], dtype=np.float64),
            np.array([  # y mid
        [ 0.000000000000e+00, -1.931826810232e-02, -3.771772168048e-02, -6.760235681076e-02, -1.163737349379e-01, -1.787741951735e-01, -2.544974660096e-01, -3.435435474462e-01, -4.459124394833e-01, -5.616041421208e-01,
         -6.906186553588e-01, -8.329559791973e-01, -9.886161136362e-01, -1.157599058676e+00, -1.339904814315e+00, -1.535533380556e+00, -1.744484757397e+00, -1.966758944838e+00, -2.202355942880e+00, -2.451275751522e+00,
         -2.713518370765e+00, -2.989083800608e+00, -3.277972041051e+00, -3.580183092095e+00, -3.895716953740e+00, -4.224573625985e+00, -4.567059380339e+00, -4.928431877689e+00, -5.301289556592e+00, -5.673228420970e+00],
        [ 0.000000000000e+00, -1.659098126753e-02, -3.236169091082e-02, -5.838579585719e-02, -1.012710397464e-01, -1.563236460619e-01, -2.232701909290e-01, -3.021106743477e-01, -3.928450963179e-01, -4.954734568398e-01,
         -6.099957559132e-01, -7.364119935381e-01, -8.747221697146e-01, -1.024926284443e+00, -1.187024337722e+00, -1.361016329554e+00, -1.546902259936e+00, -1.744682128871e+00, -1.954355936357e+00, -2.175923682394e+00,
         -2.409385366984e+00, -2.654740990124e+00, -2.911990551816e+00, -3.181134052060e+00, -3.462171490856e+00, -3.755102868203e+00, -4.060201607976e+00, -4.382161486692e+00, -4.714374760711e+00, -5.045767763106e+00],
        [ 0.000000000000e+00, -1.386369443275e-02, -2.700566014117e-02, -4.916923490362e-02, -8.616834455492e-02, -1.338730969503e-01, -1.920429158484e-01, -2.606778012492e-01, -3.397777531526e-01, -4.293427715587e-01,
         -5.293728564675e-01, -6.398680078790e-01, -7.608282257931e-01, -8.922535102099e-01, -1.034143861129e+00, -1.186499278552e+00, -1.349319762476e+00, -1.522605312904e+00, -1.706355929834e+00, -1.900571613267e+00,
         -2.105252363202e+00, -2.320398179641e+00, -2.546009062582e+00, -2.782085012025e+00, -3.028626027971e+00, -3.285632110420e+00, -3.553343835613e+00, -3.835891095695e+00, -4.127459964831e+00, -4.418307105242e+00]
            ], dtype=np.float64),
            np.array([  # y sup
        [-5.824339532081e+00, -6.051006198748e+00, -6.277672865414e+00],
        [-5.196878874217e+00, -5.423545540884e+00, -5.650212207551e+00],
        [-4.569418216354e+00, -4.796084883020e+00, -5.022751549687e+00]
            ], dtype=np.float64)
        ],
        [  # x mid
            np.array([  # y inf
        [ 3.555555555556e-02,  2.631193266430e-02,  1.693691923982e-02,  9.204287480738e-03,  4.049615038346e-03,  7.103440869985e-04, -3.730032226400e-04,  1.826400082144e-03,  8.325120065500e-03,  1.506977858309e-02,
          1.787785289166e-02,  1.800572821260e-02,  1.531325160783e-02,  8.364344930902e-03, -5.417363913415e-03, -2.295372504492e-02, -3.847685397168e-02, -5.105469101331e-02, -6.236301684190e-02, -6.690143949377e-02,
         -5.991974322700e-02, -4.301553718179e-02, -2.034011620429e-02,  5.505514497454e-03,  3.945296017563e-02,  7.303164617520e-02,  9.515365699852e-02,  1.027997557328e-01,  1.033238946148e-01,  1.044177902851e-01],
        [ 2.222222222222e-02,  1.644495791519e-02,  1.058557452489e-02,  5.752679675461e-03,  2.531009398966e-03,  4.439650543740e-04, -2.331270141500e-04,  1.141500051340e-03,  5.203200040938e-03,  9.418611614430e-03,
          1.117365805729e-02,  1.125358013287e-02,  9.570782254894e-03,  5.227715581814e-03, -3.385852445885e-03, -1.434607815307e-02, -2.404803373230e-02, -3.190918188332e-02, -3.897688552619e-02, -4.181339968361e-02,
         -3.744983951687e-02, -2.688471073862e-02, -1.271257262768e-02,  3.440946560909e-03,  2.465810010977e-02,  4.564477885950e-02,  5.947103562408e-02,  6.424984733298e-02,  6.457743413427e-02,  6.526111892817e-02],
        [ 8.888888888889e-03,  6.577983166076e-03,  4.234229809955e-03,  2.301071870185e-03,  1.012403759587e-03,  1.775860217496e-04, -9.325080566001e-05,  4.566000205359e-04,  2.081280016375e-03,  3.767444645772e-03,
          4.469463222915e-03,  4.501432053149e-03,  3.828312901958e-03,  2.091086232726e-03, -1.354340978354e-03, -5.738431261230e-03, -9.619213492920e-03, -1.276367275333e-02, -1.559075421047e-02, -1.672535987344e-02,
         -1.497993580675e-02, -1.075388429545e-02, -5.085029051073e-03,  1.376378624363e-03,  9.863240043908e-03,  1.825791154380e-02,  2.378841424963e-02,  2.569993893319e-02,  2.583097365371e-02,  2.610444757127e-02]
            ], dtype=np.float64).T,

            None,  # y mid (interpolation area)

            np.array([  # y sup
        [-4.151111111111e+00, -3.455438975100e+00, -2.770216660946e+00, -1.954371573588e+00, -8.266843177559e-01,  5.615106716337e-01,  1.824150187669e+00,  2.655032630299e+00,  3.210493175495e+00,  3.455871157315e+00,
          3.300977073190e+00,  2.645293709559e+00,  1.705133544927e+00,  7.591087552821e-01, -2.807662485848e-01, -1.363861091600e+00, -2.190527775583e+00, -2.621974188121e+00, -2.783182361784e+00, -2.680491450151e+00,
         -2.267574065047e+00, -1.502840817836e+00, -6.303083393307e-01,  1.386164207256e-01,  9.416446539072e-01,  1.651649647719e+00,  2.066647077121e+00,  2.098091930676e+00,  1.926615737510e+00,  1.771373228051e+00],
        [-4.377777777778e+00, -3.645303747944e+00, -2.923929790698e+00, -2.063804926097e+00, -8.729174460501e-01,  5.941616517594e-01,  1.929255125219e+00,  2.808866147461e+00,  3.398715781798e+00,  3.660682153861e+00,
          3.497864399369e+00,  2.804399258264e+00,  1.809028239204e+00,  8.059194809226e-01, -2.989176777988e-01, -1.450821968896e+00, -2.330778243635e+00, -2.791651073137e+00, -2.966237239690e+00, -2.859258302862e+00,
         -2.420001322367e+00, -1.605053582753e+00, -6.740064585006e-01,  1.485986833975e-01,  1.010196134003e+00,  1.773270653660e+00,  2.220186929046e+00,  2.256202915522e+00,  2.075022857444e+00,  1.911218482897e+00],
        [-4.604444444444e+00, -3.835168520787e+00, -3.077642920449e+00, -2.173238278607e+00, -9.191505743443e-01,  6.268126318850e-01,  2.034360062768e+00,  2.962699664623e+00,  3.586938388100e+00,  3.865493150407e+00,
          3.694751725547e+00,  2.963504806969e+00,  1.912922933482e+00,  8.527302065632e-01, -3.170691070128e-01, -1.537782846192e+00, -2.471028711687e+00, -2.961327958154e+00, -3.149292117597e+00, -3.038025155573e+00,
         -2.572428579688e+00, -1.707266347669e+00, -7.177045776704e-01,  1.585809460695e-01,  1.078747614099e+00,  1.894891659600e+00,  2.373726780972e+00,  2.414313900368e+00,  2.223429977377e+00,  2.051063737743e+00]
            ], dtype=np.float64).T
        ],
        [  # x sup
            np.array([  # y inf
        [ 1.044177902851e-01,  6.526111892817e-02,  2.610444757127e-02],
        [ 1.044177902851e-01,  6.526111892817e-02,  2.610444757127e-02],
        [ 1.044177902851e-01,  6.526111892817e-02,  2.610444757127e-02]
            ], dtype=np.float64),
            np.array([  # y mid
        [ 0.000000000000e+00, -4.873338553929e-02, -9.801478056061e-02, -1.404460570565e-01, -1.716126941993e-01, -1.946505240252e-01, -2.097422163617e-01, -2.168877712084e-01, -2.160871885657e-01, -2.073404684335e-01,
         -1.906476108115e-01, -1.660086157001e-01, -1.334234830993e-01, -9.289221300892e-02, -4.441480542860e-02,  1.200873964094e-02,  7.637842219996e-02,  1.486942422490e-01,  2.289561997871e-01,  3.171642948150e-01,
          4.133185273323e-01,  5.174188973393e-01,  6.294654048359e-01,  7.494580498217e-01,  8.773968322975e-01,  1.013281752263e+00,  1.157295479561e+00,  1.312573846189e+00,  1.474702331372e+00,  1.636282807020e+00],
        [ 0.000000000000e+00, -5.019351903512e-02, -1.009313732806e-01, -1.448650373965e-01, -1.776095888547e-01, -2.022798346924e-01, -2.190572199797e-01, -2.279417447166e-01, -2.289334089032e-01, -2.220322125394e-01,
         -2.072381556251e-01, -1.845512381605e-01, -1.539714601456e-01, -1.154988215804e-01, -6.913332246438e-02, -1.487496279831e-02,  4.727625741805e-02,  1.173203381853e-01,  1.952572795024e-01,  2.810870813701e-01,
          3.748097437881e-01,  4.764252667566e-01,  5.859336502755e-01,  7.033348943444e-01,  8.286289989642e-01,  9.618159641349e-01,  1.103077234941e+00,  1.255527618382e+00,  1.414782191983e+00,  1.573492430319e+00],
        [ 0.000000000000e+00, -5.165365253095e-02, -1.038479660006e-01, -1.492840177365e-01, -1.836064835101e-01, -2.099091453595e-01, -2.283722235978e-01, -2.389957182248e-01, -2.417796292407e-01, -2.367239566454e-01,
         -2.238287004386e-01, -2.030938606208e-01, -1.745194371919e-01, -1.381054301518e-01, -9.385183950015e-02, -4.175866523757e-02,  1.817409263614e-02,  8.594643412154e-02,  1.615583592177e-01,  2.450098679253e-01,
          3.363009602439e-01,  4.354316361739e-01,  5.424018957151e-01,  6.572117388672e-01,  7.798611656309e-01,  9.103501760065e-01,  1.048858990322e+00,  1.198481390576e+00,  1.354862052594e+00,  1.510702053618e+00]
            ], dtype=np.float64),
            np.array([  # y sup
        [ 1.729512976917e+00,  1.869358231763e+00,  2.009203486609e+00],
        [ 1.666722600216e+00,  1.806567855062e+00,  1.946413109908e+00],
        [ 1.603932223515e+00,  1.743777478361e+00,  1.883622733206e+00]
            ], dtype=np.float64)
        ]
                                ]

        # Quadraticaly extrapolated data sampled on self.xsamples_ex and
        # self.ysamples_ex.
        # Calculated at (x, y) from the formula:
        # Pij(x_n, y_n) + (x-x_n)*dPij/dx(x_n, y_n) + (y-y_n)*dPij/dy(x_n, y_n)
        #   + (x-x_n)*(y-y_n)*d2Pij/dxdy(x_n, y_n)
        #   + 0.5*(x-x_n)**2*d2Pij/dx2(x_n, y_n)
        #   + 0.5*(y-y_n)**2*d2Pij/dy2(x_n, y_n)
        # where (x_n, y_n) is the nearest coordinates in the
        # interpolated range form (x, y), and Pij the nearest interpolation
        # polynomial.
        # NB: this extrapolation is not entirely biquadratic as some higher
        # level derivatives are not taken into account.
        self.extrap_data_qua = [
        [  # x inf
            np.array([  # y inf
        [ 8.107663047777e-02,  4.467289404860e-02,  1.546915761944e-02],
        [ 7.360622738194e-02,  4.000389211371e-02,  1.360155684548e-02],
        [ 6.613582428611e-02,  3.533489017882e-02,  1.173395607153e-02]
            ], dtype=np.float64),
            np.array([  # y mid
        [ 0.000000000000e+00, -1.668856602415e-02, -3.237437486229e-02, -6.059464852558e-02, -1.094271532347e-01, -1.731338895363e-01, -2.513806050352e-01, -3.441672997312e-01, -4.514939736244e-01, -5.733606267147e-01,
         -7.097672590023e-01, -8.607138704871e-01, -1.026200461169e+00, -1.206227031048e+00, -1.400793580124e+00, -1.609900108398e+00, -1.833546615869e+00, -2.071733102536e+00, -2.324459568402e+00, -2.591726013464e+00,
         -2.873532437723e+00, -3.169878841180e+00, -3.480765223834e+00, -3.806191585685e+00, -4.146157926733e+00, -4.500664246978e+00, -4.870044798816e+00, -5.260037581700e+00, -5.662564829408e+00, -6.064089319931e+00],
        [ 0.000000000000e+00, -1.556375389325e-02, -3.027444605997e-02, -5.564840980829e-02, -9.855753126859e-02, -1.541204016724e-01, -2.220526671109e-01, -3.023543275840e-01, -3.950253830918e-01, -5.000658336343e-01,
         -6.174756792114e-01, -7.472549198232e-01, -8.894035554697e-01, -1.043921586151e+00, -1.210809011867e+00, -1.390065832617e+00, -1.581692048402e+00, -1.785687659222e+00, -2.002052665076e+00, -2.230787065965e+00,
         -2.471890861889e+00, -2.725364052848e+00, -2.991206638841e+00, -3.269418619868e+00, -3.559999995931e+00, -3.862950767028e+00, -4.178555287069e+00, -4.511694964821e+00, -4.855497914155e+00, -5.198447801763e+00],
        [ 0.000000000000e+00, -1.369933805286e-02, -2.667170096503e-02, -4.873125313580e-02, -8.573418319847e-02, -1.335205778480e-01, -1.918481120375e-01, -2.607167857670e-01, -3.401265990364e-01, -4.300775518459e-01,
         -5.305696441952e-01, -6.416028760846e-01, -7.631772475139e-01, -8.952927584832e-01, -1.037949408992e+00, -1.191147199042e+00, -1.354886128631e+00, -1.529166197760e+00, -1.713987406429e+00, -1.909349754638e+00,
         -2.115253242387e+00, -2.331697869676e+00, -2.558683636505e+00, -2.796210542874e+00, -3.044278588783e+00, -3.302887774232e+00, -3.572280424268e+00, -3.856616452196e+00, -4.150039669382e+00, -4.442735911428e+00]
            ], dtype=np.float64),
            np.array([  # y sup
        [-6.275549299151e+00, -6.586739267980e+00, -6.890729236809e+00],
        [-5.386676955442e+00, -5.663020685960e+00, -5.932164416478e+00],
        [-4.607734239566e+00, -4.849231731773e+00, -5.083529223980e+00]
            ], dtype=np.float64)
        ],
        [  # x mid
            np.array([  # y inf
        [ 6.115555555556e-02,  4.791166843559e-02,  3.462033344424e-02,  2.192209684526e-02,  9.415357756925e-03, -3.239739011991e-03, -1.296881058358e-02, -1.671580998102e-02, -1.463622309371e-02, -1.018568103864e-02,
         -6.553129629897e-03, -1.897195401563e-03,  2.156791972183e-03,  2.370662332141e-03, -2.995411031640e-03, -1.148550214241e-02, -1.991533937671e-02, -2.839600371571e-02, -3.759009573523e-02, -4.243885239126e-02,
         -3.893208390338e-02, -2.881430621862e-02, -1.418103667469e-02,  4.059894824363e-03,  2.945139935625e-02,  5.515070138439e-02,  7.244696020261e-02,  7.920044854331e-02,  8.086546496527e-02,  8.293755914070e-02],
        [ 3.222222222222e-02,  2.488235470085e-02,  1.749315819849e-02,  1.072057395848e-02,  4.627002648411e-03, -1.099036156169e-03, -5.153364264518e-03, -6.101550754584e-03, -3.766074630629e-03, -4.468023003073e-04,
          1.630305509804e-03,  3.479000596092e-03,  4.431540209719e-03,  2.886433316673e-03, -2.439777101441e-03, -9.866303581780e-03, -1.679744209364e-02, -2.305813215769e-02, -2.929996321889e-02, -3.225770159669e-02,
         -2.925153509358e-02, -2.133735489363e-02, -1.030668218643e-02,  2.876251376108e-03,  2.075124041470e-02,  3.866003480059e-02,  5.060123218817e-02,  5.503136796210e-02,  5.580461005241e-02,  5.687040363740e-02],
        [ 1.048888888889e-02,  7.927966651782e-03,  5.339443197731e-03,  3.095934955467e-03,  1.347762679498e-03, -6.929417193725e-05, -8.804887657188e-04, -7.022881084118e-04,  6.461960689244e-04,  2.188978419414e-03,
          2.942526815318e-03,  3.257499327264e-03,  3.006034174730e-03,  1.716481070303e-03, -1.202968923243e-03, -5.021667329823e-03, -8.459118830734e-03, -1.134750479723e-02, -1.404244664131e-02, -1.519644817954e-02,
         -1.366820709902e-02, -9.866307360249e-03, -4.700086580473e-03,  1.286027394795e-03,  9.238142492697e-03,  1.714035249438e-02,  2.236924569989e-02,  2.422498223385e-02,  2.442732180061e-02,  2.476193312474e-02]
            ], dtype=np.float64).T,

            None,  # y mid (interpolation area)

            np.array([  # y sup
        [-4.149511111111e+00, -3.454088991615e+00, -2.769111447558e+00, -1.953576710502e+00, -8.263489588360e-01,  5.612637914401e-01,  1.823362949709e+00,  2.653873742170e+00,  3.209058091548e+00,  3.454292691089e+00,
          3.299450136783e+00,  2.644049776833e+00,  1.704311266200e+00,  7.587341501197e-01, -2.806148765297e-01, -1.363144327669e+00, -2.189367680921e+00, -2.620558020165e+00, -2.781634054215e+00, -2.678962538457e+00,
         -2.266262336339e+00, -1.501953240901e+00, -6.299233968593e-01,  1.385260694958e-01,  9.410195563560e-01,  1.650532088670e+00,  2.065227908572e+00,  2.096616973977e+00,  1.925212085657e+00,  1.770030713605e+00],
        [-4.367777777778e+00, -3.636866351158e+00, -2.917022207024e+00, -2.058837031814e+00, -8.708214528006e-01,  5.926186505489e-01,  1.924334887968e+00,  2.801623096655e+00,  3.389746507126e+00,  3.650816739946e+00,
          3.488321046821e+00,  2.796624678727e+00,  1.803888997159e+00,  8.035781986573e-01, -2.979716024544e-01, -1.446342194324e+00, -2.323527651996e+00, -2.782800023412e+00, -2.956560317383e+00, -2.849702604775e+00,
         -2.411803017944e+00, -1.599506226905e+00, -6.716005680545e-01,  1.480339882109e-01,  1.006289274308e+00,  1.766285909604e+00,  2.211317125614e+00,  2.246984436154e+00,  2.066250033363e+00,  1.902827767607e+00],
        [-4.578844444444e+00, -3.813568785015e+00, -3.059959506245e+00, -2.160520469242e+00, -9.137848316257e-01,  6.228625487861e-01,  2.021764255408e+00,  2.944157454560e+00,  3.563977044941e+00,  3.840237690785e+00,
          3.670320743026e+00,  2.943601883354e+00,  1.899766473845e+00,  8.467365239639e-01, -3.146471541311e-01, -1.526314623289e+00, -2.452467197091e+00, -2.938669270856e+00, -3.124519196490e+00, -3.013562568470e+00,
         -2.551440920365e+00, -1.693065116700e+00, -7.115454981284e-01,  1.571353263917e-01,  1.068746053281e+00,  1.877010714818e+00,  2.351020084186e+00,  2.390714593186e+00,  2.200971547731e+00,  2.029583506600e+00]
            ], dtype=np.float64).T
        ],
        [  # x sup
            np.array([  # y inf
        [ 8.522431331514e-02,  5.829962499642e-02,  2.533362166835e-02],
        [ 8.865444457680e-02,  6.044345703496e-02,  2.619115448377e-02],
        [ 9.208457583847e-02,  6.258728907350e-02,  2.704868729919e-02]
            ], dtype=np.float64),
            np.array([  # y mid
        [ 0.000000000000e+00, -4.938317956119e-02, -9.932453604409e-02, -1.422886810057e-01, -1.737791892433e-01, -1.969900731423e-01, -2.121073916766e-01, -2.191311448461e-01, -2.180613326509e-01, -2.088979550911e-01,
         -1.916410121663e-01, -1.662905038770e-01, -1.328464302230e-01, -9.130879120430e-02, -4.167758682049e-02,  1.604718292772e-02,  8.186551804058e-02,  1.557774185186e-01,  2.377828843609e-01,  3.278819155682e-01,
          4.260745121401e-01,  5.323606740768e-01,  6.467404013782e-01,  7.692136940440e-01,  8.997805520750e-01,  1.038440975471e+00,  1.185381023221e+00,  1.343794707672e+00,  1.509185603673e+00,  1.674018322701e+00],
        [ 0.000000000000e+00, -5.425473167201e-02, -1.091173450523e-01, -1.563814370790e-01, -1.911501828797e-01, -2.169020166741e-01, -2.338395656983e-01, -2.419628299521e-01, -2.412718094359e-01, -2.317665041495e-01,
         -2.134469140926e-01, -1.863130392657e-01, -1.503648796687e-01, -1.056024353015e-01, -5.202570616365e-02,  1.036530774409e-02,  8.157060642196e-02,  1.615901898706e-01,  2.504240580888e-01,  3.480722110775e-01,
          4.545346488364e-01,  5.698113713655e-01,  6.939023786649e-01,  8.268076707340e-01,  9.685272475739e-01,  1.119061109185e+00,  1.278611882816e+00,  1.450658002650e+00,  1.630302643867e+00,  1.809339403323e+00],
        [ 0.000000000000e+00, -6.205035688140e-02, -1.248040537362e-01, -1.787660009238e-01, -2.182704042141e-01, -2.473419312326e-01, -2.662150286372e-01, -2.748896964278e-01, -2.733659346044e-01, -2.616437431671e-01,
         -2.397231221155e-01, -2.076040714502e-01, -1.652865911710e-01, -1.127706812779e-01, -5.005634177029e-02,  2.285642735100e-02,  1.059676260862e-01,  1.992772544359e-01,  3.027853123990e-01,  4.164917999762e-01,
          5.403967171675e-01,  6.745000639727e-01,  8.188018403920e-01,  9.733020464245e-01,  1.138000682072e+00,  1.312897747334e+00,  1.498227688882e+00,  1.698015174301e+00,  1.906594409418e+00,  2.114470304509e+00]
            ], dtype=np.float64),
            np.array([  # y sup
        [ 1.764803256650e+00,  1.895946228399e+00,  2.021047885140e+00],
        [ 1.898470255020e+00,  2.027132103390e+00,  2.149752636752e+00],
        [ 2.201947073953e+00,  2.328127798945e+00,  2.448267208927e+00]
            ], dtype=np.float64)
        ]
                                ]

        self.interp_func = interpolators2d.Interpolate2DCubic(x, y, data,
                                                              extrapolate=extrapolate,
                                                              extrapolation_range=extrapolation_range,
                                                              extrapolation_type=extrapolation_type,
                                                              tolerate_single_value=tolerate_single_value)

    def derivative(self, f, x, y, h, x_order, y_order):
        """
        Calculates a numerical derivative at point x, y.

        Obtains samples with spacing h and computes the central difference to
        obtain the first derivative. Method calls itself recursively to
         calculate higher deriviative orders.

        :param f: 1D function object.
        :param x: Sample point x.
        :param y: Sample point y.
        :param h: Sample distance.
        :param x_order: Derivative order along x.
        :param y_order: Derivative order along y.
        :return: Derivative value.
        """

        if x_order < 1 and y_order < 1:
            raise ValueError('At least one derivative order must be > 0.')

        if x_order < 0:
            raise ValueError('The y derivative order cannot be less than zero.')

        if y_order < 0:
            raise ValueError('The y derivative order cannot be less than zero.')

        d = 0.5 * h
        if x_order == 1 and y_order == 0:
            return (f(x+d, y) - f(x-d, y)) / h

        if x_order == 0 and y_order == 1:
            return (f(x, y+d) - f(x, y-d)) / h

        if y_order > 0:
            f0 = self.derivative(f, x, y-d, h, x_order, y_order - 1)
            f1 = self.derivative(f, x, y+d, h, x_order, y_order - 1)
            return (f1 - f0) / h

        if x_order > 0:
            f0 = self.derivative(f, x-d, y, h, x_order - 1, 0)
            f1 = self.derivative(f, x+d, y, h, x_order - 1, 0)
            return (f1 - f0) / h

    def interpolate_2d_extrapolate_assert(self, i_block, j_block, ref_data, delta):
        mini, maxi = self.extrapol_xdomains[i_block]
        minj, maxj = self.extrapol_ydomains[j_block]
        for iex in range(mini, maxi):
            for jex in range(minj, maxj):
                x = self.xsamples_ex[iex]
                y = self.ysamples_ex[jex]

                # test f(x,y)
                self.assertAlmostEqual(self.interp_func(x, y), ref_data[iex - mini, jex - minj], delta=delta)

                # skip derivatives on boundary as the numerical sampling routine will produce odd results with nearest neighbour extrapolation
                if x == X_LOWER or x == X_UPPER:
                    continue

                if y == Y_LOWER or y == Y_UPPER:
                    continue

                # test derivatives
                for x_order in range(0, MAX_DERIVATIVE_ORDER + 1):
                    for y_order in range(0, MAX_DERIVATIVE_ORDER + 1):

                        # skip invalid combination
                        if x_order == 0 and y_order == 0:
                            continue

                        v = self.derivative(self.interp_func, x, y, 1e-3, x_order, y_order)
                        r = self.interp_func.derivative(x, y, x_order, y_order)

                        # skip small values that suffer from numerical sampling accuracy issues
                        if abs(v) < 1e-6 * 10**max(x_order, y_order):
                            continue

                        if abs(r) < 1e-9:
                            r = 0.0

                        self.assertAlmostEqual(r, v, delta=1e-2 * abs(v))

    def interpolate_2d_xboundaries_assert(self, inf, sup, epsilon, y):
        with self.assertRaises(ValueError):
            self.interp_func(inf - epsilon, y)
        self.assertIsInstance(self.interp_func(inf + epsilon, y), float)
        with self.assertRaises(ValueError):
            self.interp_func(sup + epsilon, y)
        self.assertIsInstance(self.interp_func(sup - epsilon, y), float)

    def interpolate_2d_yboundaries_assert(self, x, inf, sup, epsilon):
        with self.assertRaises(ValueError):
            self.interp_func(x, inf - epsilon)
        self.assertIsInstance(self.interp_func(x, inf + epsilon), float)
        with self.assertRaises(ValueError):
            self.interp_func(x, sup + epsilon)
        self.assertIsInstance(self.interp_func(x, sup - epsilon), float)

    # General behaviour

    def test_interpolate_2d_invalid_coordinatesx(self):
        """2D interpolation. An error must be raises if coordinates are not an array-like object.
        """
        self.assertRaises(ValueError, interpolators2d._Interpolate2DBase, "blah", [1, 2, 3, 4], np.ones((4, 4)))

    def test_interpolate_2d_invalid_coordinatesy(self):
        """2D interpolation. An error must be raises if coordinates are not an array-like object.
        """
        self.assertRaises(ValueError, interpolators2d._Interpolate2DBase, [1, 2, 3, 4], "blah", np.ones((4, 4)))

    def test_interpolate_2d_invalid_data(self):
        """2D interpolation. An error must be raises if data is not an array-like object.
        """
        self.assertRaises(ValueError, interpolators2d._Interpolate2DBase, [1, 2, 3, 4], [2, 3, 4, 5], "blah")

    def test_interpolate_2d_invalid_data_size(self):
        """2D interpolation. An error must be raises if data has not the same size as the product of the coordinates.
        """
        self.assertRaises(ValueError, interpolators2d._Interpolate2DBase, [1, 2, 3, 4], [2, 3, 4], np.ones((4, 4)))

    def test_interpolate_2d_invalid_data_dimension(self):
        """2D interpolation. An error must be raises if data is not 2D.
        """
        self.assertRaises(ValueError, interpolators2d._Interpolate2DBase, [1, 2, 3, 4], [2, 3, 4, 5], np.ones((4,)))

    def test_interpolate_2d_double_coord(self):
        """2D interpolation. An error must be raises if there is a double coordinate.
        """
        self.assertRaises(ValueError, interpolators2d._Interpolate2DBase, [1, 2, 3, 4], [3, 3, 4, 5], np.ones((4, 4)))

    def test_interpolate_2d_single_value_invalid_x(self):
        """2D interpolation. By default, a single input value must raise a ValueError.
        """
        self.assertRaises(ValueError, interpolators2d._Interpolate2DBase, [2.], [2, 3, 4, 5], np.ones((1, 4)))

    def test_interpolate_2d_single_value_invalid_y(self):
        """2D interpolation. By default, a single input value must raise a ValueError.
        """
        self.assertRaises(ValueError, interpolators2d._Interpolate2DBase, [1, 2, 3, 4], [4.], np.ones((4, 1)))

    def test_interpolate_2d_single_value_invalid_xy(self):
        """2D interpolation. By default, a single input value must raise a ValueError.
        """
        self.assertRaises(ValueError, interpolators2d._Interpolate2DBase, [2.], [4.], np.ones((1, 1)))

    # Linear behaviour

    def test_interpolate_2d_linear(self):
        """2D linear interpolation. Test values inside the boundaries"""
        self.init_2dlinear()
        for i in range(len(self.xsamples)):
            for j in range(len(self.ysamples)):
                self.assertAlmostEqual(self.interp_func(self.xsamples[i], self.ysamples[j]), self.interp_data[i, j], delta=LIN_DELTA)

        # avoid end points for derivatives
        for i in range(1, len(self.xsamples) - 1):
            for j in range(1, len(self.ysamples) - 1):
                for x_order in range(MAX_DERIVATIVE_ORDER + 1):
                    for y_order in range(MAX_DERIVATIVE_ORDER + 1):

                        if x_order == 0 and y_order == 0:
                            continue

                        x = self.xsamples[i]
                        y = self.ysamples[j]

                        # higher order derivatives are zero
                        if x_order > 1 or y_order > 1:
                            v = 0
                        else:
                            v = self.derivative(self.interp_func, x, y, 1e-3, x_order, y_order)
                        r = self.interp_func.derivative(x, y, x_order, y_order)

                        self.assertAlmostEqual(r, v, delta=1e-5 * abs(v))

    def test_interpolate_2d_linear_bigvalues(self):
        """2D linear interpolation. Test with big values (1e20) inside the boundaries"""
        factor = 1.e20
        self.init_2dlinear(data=factor * self.data)
        for i in range(len(self.xsamples)):
            for j in range(len(self.ysamples)):
                self.assertAlmostEqual(self.interp_func(self.xsamples[i], self.ysamples[j]), factor * self.interp_data[i, j], delta=factor * LIN_DELTA)

        # avoid end points for derivatives
        for i in range(1, len(self.xsamples) - 1):
            for j in range(1, len(self.ysamples) - 1):
                for x_order in range(MAX_DERIVATIVE_ORDER + 1):
                    for y_order in range(MAX_DERIVATIVE_ORDER + 1):

                        if x_order == 0 and y_order == 0:
                            continue

                        x = self.xsamples[i]
                        y = self.ysamples[j]

                        # higher order derivatives are zero
                        if x_order > 1 or y_order > 1:
                            v = 0
                        else:
                            v = self.derivative(self.interp_func, x, y, 1e-3, x_order, y_order)
                        r = self.interp_func.derivative(x, y, x_order, y_order)

                        self.assertAlmostEqual(r, v, delta=1e-5 * abs(v))

    def test_interpolate_2d_linear_lowvalues(self):
        """2D linear interpolation. Test with low values (1e-20) inside the boundaries"""
        factor = 1.e-20
        self.init_2dlinear(data=factor * self.data)
        for i in range(len(self.xsamples)):
            for j in range(len(self.ysamples)):
                self.assertAlmostEqual(self.interp_func(self.xsamples[i], self.ysamples[j]), factor * self.interp_data[i, j], delta=factor * LIN_DELTA)

        # avoid end points for derivatives
        for i in range(1, len(self.xsamples) - 1):
            for j in range(1, len(self.ysamples) - 1):
                for x_order in range(MAX_DERIVATIVE_ORDER + 1):
                    for y_order in range(MAX_DERIVATIVE_ORDER + 1):

                        if x_order == 0 and y_order == 0:
                            continue

                        x = self.xsamples[i]
                        y = self.ysamples[j]

                        # higher order derivatives are zero
                        if x_order > 1 or y_order > 1:
                            v = 0
                        else:
                            v = self.derivative(self.interp_func, x, y, 1e-3, x_order, y_order)
                        r = self.interp_func.derivative(x, y, x_order, y_order)

                        self.assertAlmostEqual(r, v, delta=1e-5 * abs(v))

    def test_interpolate_2d_linear_edge(self):
        """2D linear interpolation. Test edges values"""
        self.init_2dlinear()
        for i in range(len(self.xsamples)):
            self.assertAlmostEqual(self.interp_func(self.xsamples[i], self.ysamples[0]), self.interp_data[i, 0], delta=LIN_DELTA)
            self.assertAlmostEqual(self.interp_func(self.xsamples[i], self.ysamples[-1]), self.interp_data[i, -1], delta=LIN_DELTA)
        for j in range(len(self.ysamples)):
            self.assertAlmostEqual(self.interp_func(self.xsamples[0], self.ysamples[j]), self.interp_data[0, j], delta=LIN_DELTA)
            self.assertAlmostEqual(self.interp_func(self.xsamples[-1], self.ysamples[j]), self.interp_data[-1, j], delta=LIN_DELTA)

    def test_interpolate_2d_linear_knot(self):
        """2D linear interpolation. Test knot values"""
        self.init_2dlinear()
        for i in range(len(self.x)):
            for j in range(len(self.y)):
                self.assertAlmostEqual(self.interp_func(self.x[i], self.y[j]), self.data[i, j], delta=ABS_DELTA)

    def test_interpolate_2d_linear_out(self):
        """2D linear interpolation. Test values outside the boundaries"""
        self.init_2dlinear(extrapolate=False)
        for i in range(len(self.xsamples_out)):
            self.assertRaises(ValueError, self.interp_func, self.xsamples_out[i], self.ysamples_out[0])
            self.assertRaises(ValueError, self.interp_func, self.xsamples_out[i], self.ysamples_out[-1])
        self.interpolate_2d_xboundaries_assert(X_LOWER, X_UPPER, 1e-6, (Y_LOWER + Y_UPPER) / 2)
        for j in range(len(self.ysamples_out)):
            self.assertRaises(ValueError, self.interp_func, self.xsamples_out[0], self.ysamples_out[j])
            self.assertRaises(ValueError, self.interp_func, self.xsamples_out[-1], self.ysamples_out[j])
        self.interpolate_2d_yboundaries_assert((X_LOWER + X_UPPER) / 2, Y_LOWER, Y_UPPER, 1e-6)

    def test_interpolate_2d_linear_extrapolate_nearest_range(self):
        """2D linear interpolation. Tests the size of the extrapolation range."""
        self.init_2dlinear(extrapolate=True, extrapolation_type='nearest', extrapolation_range=1.)
        self.interpolate_2d_xboundaries_assert(X_LOWER - 1, X_UPPER + 1, 1e-6, (Y_LOWER + Y_UPPER) / 2)
        self.interpolate_2d_yboundaries_assert((X_LOWER + X_UPPER) / 2, Y_LOWER - 1, Y_UPPER + 1, 1e-6)

    def test_interpolate_2d_linear_extrapolate_nearest_xinfyinf(self):
        """2D linear interpolation. Test values in the extrapolation area with x and y below the interpolation area.
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(0, 0, self.extrap_data_nea[0][0], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_nearest_xinfymid(self):
        """2D linear interpolation. Test values in the extrapolation area with x below and y inside the interpolation area.
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(0, 1, self.extrap_data_nea[0][1], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_nearest_xinfysup(self):
        """2D linear interpolation. Test values in the extrapolation area with x below and y above the interpolation area.
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(0, 2, self.extrap_data_nea[0][2], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_nearest_xmidyinf(self):
        """2D linear interpolation. Test values in the extrapolation area with x inside and y below the interpolation area.
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(1, 0, self.extrap_data_nea[1][0], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_nearest_xmidysup(self):
        """2D linear interpolation. Test values in the extrapolation area with x inside and y above the interpolation area.
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(1, 2, self.extrap_data_nea[1][2], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_nearest_xsupyinf(self):
        """2D linear interpolation. Test values in the extrapolation area with x above and y below the interpolation area.
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(2, 0, self.extrap_data_nea[2][0], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_nearest_xsupymid(self):
        """2D linear interpolation. Test values in the extrapolation area with x above and y inside the interpolation area.
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(2, 1, self.extrap_data_nea[2][1], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_nearest_xsupysup(self):
        """2D linear interpolation. Test values in the extrapolation area with x above and y above the interpolation area.
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(2, 2, self.extrap_data_nea[2][2], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_linear_range(self):
        """2D linear interpolation. Tests the size of the extrapolation range."""
        self.init_2dlinear(extrapolate=True, extrapolation_type='linear', extrapolation_range=1.)
        self.interpolate_2d_xboundaries_assert(X_LOWER - 1, X_UPPER + 1, 1e-6, (Y_LOWER + Y_UPPER) / 2)
        self.interpolate_2d_yboundaries_assert((X_LOWER + X_UPPER) / 2, Y_LOWER - 1, Y_UPPER + 1, 1e-6)

    def test_interpolate_2d_linear_extrapolate_linear_xinfyinf(self):
        """2D linear interpolation. Test values in the extrapolation area with x below and y below the interpolation area
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(0, 0, self.extrap_data_lin[0][0], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_linear_xinfymid(self):
        """2D linear interpolation. Test values in the extrapolation area with x below and y inside the interpolation area
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(0, 1, self.extrap_data_lin[0][1], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_linear_xinfysup(self):
        """2D linear interpolation. Test values in the extrapolation area with x below and y above the interpolation area
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(0, 2, self.extrap_data_lin[0][2], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_linear_xmidyinf(self):
        """2D linear interpolation. Test values in the extrapolation area with x inside and y below the interpolation area
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(1, 0, self.extrap_data_lin[1][0], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_linear_xmidysup(self):
        """2D linear interpolation. Test values in the extrapolation area with x inside and y above the interpolation area
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(1, 2, self.extrap_data_lin[1][2], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_linear_xsupyinf(self):
        """2D linear interpolation. Test values in the extrapolation area with x above and y below the interpolation area
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(2, 0, self.extrap_data_lin[2][0], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_linear_xsupymid(self):
        """2D linear interpolation. Test values in the extrapolation area with x above and y inside the interpolation area
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(2, 1, self.extrap_data_lin[2][1], LIN_DELTA)

    def test_interpolate_2d_linear_extrapolate_linear_xsupysup(self):
        """2D linear interpolation. Test values in the extrapolation area with x above and y above the interpolation area
        """
        self.init_2dlinear(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(2, 2, self.extrap_data_lin[2][2], LIN_DELTA)

    def test_interpolate_2d_linear_type_conversion(self):
        """2D linear interpolation. Whatever the type of input data, the interpolating function must provide float numbers.
        """
        self.init_2dlinear([1, 2, 3, 4], [2, 3, 4, 5], np.ones((4, 4), dtype=int))
        self.assertIsInstance(self.interp_func(2.5, 4.5), float)

    def test_interpolate_2d_linear_single_value_tolerated_x(self):
        """2D linear interpolation. If tolerated, a single value input must be extrapolated to every real value on its axis.
        """
        self.init_2dlinear([2.], [2, 3, 4, 5], np.ones((1, 4)), tolerate_single_value=True)
        self.assertAlmostEqual(self.interp_func(-31946139.346, 2.5), 1., delta=ABS_DELTA)
        self.assertAlmostEqual(self.interp_func(31946139.346, 2.5), 1., delta=ABS_DELTA)
        self.assertAlmostEqual(self.interp_func(2., 2.5), 1., delta=ABS_DELTA)

    def test_interpolate_2d_linear_single_value_tolerated_y(self):
        """2D linear interpolation. If tolerated, a single value input must be extrapolated to every real value on its axis.
        """
        self.init_2dlinear([1, 2, 3, 4], [4.], np.ones((4, 1)), tolerate_single_value=True)
        self.assertAlmostEqual(self.interp_func(1.5, -31946139.346), 1., delta=ABS_DELTA)
        self.assertAlmostEqual(self.interp_func(1.5, 31946139.346), 1., delta=ABS_DELTA)
        self.assertAlmostEqual(self.interp_func(1.5, 2.), 1., delta=ABS_DELTA)

    def test_interpolate_2d_linear_single_value_tolerated_xy(self):
        """2D linear interpolation. If tolerated, a single value input must be extrapolated to every real value on its axis.
        """
        self.init_2dlinear([2.], [4.], np.ones((1, 1)), tolerate_single_value=True)
        self.assertAlmostEqual(self.interp_func(-31946139.346, 7856.45), 1., delta=ABS_DELTA)
        self.assertAlmostEqual(self.interp_func(31946139.346, -7856.45), 1., delta=ABS_DELTA)
        self.assertAlmostEqual(self.interp_func(2., 4.), 1., delta=ABS_DELTA)

    # Cubic behaviour

    def test_interpolate_2d_cubic(self):
        """2D cubic interpolation. Test values inside the boundaries"""
        self.init_2dcubic()
        for i in range(len(self.xsamples)):
            for j in range(len(self.ysamples)):
                self.assertAlmostEqual(self.interp_func(self.xsamples[i], self.ysamples[j]), self.interp_data[i, j], delta=CUB_DELTA)

        # avoid end points for derivatives
        for i in range(1, len(self.xsamples) - 1):
            for j in range(1, len(self.ysamples) - 1):

                # only test up to d2f/dxdy as numerical differentiation starts to fail
                for x_order in range(MAX_DERIVATIVE_ORDER + 1):
                    for y_order in range(MAX_DERIVATIVE_ORDER + 1):
                        if x_order == 0 and y_order == 0:
                            continue
                        x = self.xsamples[i]
                        y = self.ysamples[j]
                        v = self.derivative(self.interp_func, x, y, 1e-3, x_order, y_order)

                        # skip small values that suffer from numerical sampling accuracy issues
                        if v < 1e-2:
                            continue

                        r = self.interp_func.derivative(x, y, x_order, y_order)

                        self.assertAlmostEqual(r, v, delta=1e-3 * abs(v))

    def test_interpolate_2d_cubic_bigvalues(self):
        """2D cubic interpolation. Test with big values (1e20) inside the boundaries"""
        factor = 1.e20
        self.init_2dcubic(data=factor * self.data)
        for i in range(len(self.xsamples)):
            for j in range(len(self.ysamples)):
                self.assertAlmostEqual(self.interp_func(self.xsamples[i], self.ysamples[j]), factor * self.interp_data[i, j], delta=factor * CUB_DELTA)

    def test_interpolate_2d_cubic_lowvalues(self):
        """2D cubic interpolation. Test with low values (1e-20) inside the boundaries"""
        factor = 1.e-20
        self.init_2dcubic(data=factor * self.data)
        for i in range(len(self.xsamples)):
            for j in range(len(self.ysamples)):
                self.assertAlmostEqual(self.interp_func(self.xsamples[i], self.ysamples[j]), factor * self.interp_data[i, j], delta=factor * CUB_DELTA)

    def test_interpolate_2d_cubic_edge(self):
        """2D cubic interpolation. Test edges values"""
        self.init_2dcubic()
        for i in range(len(self.xsamples)):
            self.assertAlmostEqual(self.interp_func(self.xsamples[i], self.ysamples[0]), self.interp_data[i, 0], delta=CUB_DELTA)
            self.assertAlmostEqual(self.interp_func(self.xsamples[i], self.ysamples[-1]), self.interp_data[i, -1], delta=CUB_DELTA)
        for j in range(len(self.ysamples)):
            self.assertAlmostEqual(self.interp_func(self.xsamples[0], self.ysamples[j]), self.interp_data[0, j], delta=CUB_DELTA)
            self.assertAlmostEqual(self.interp_func(self.xsamples[-1], self.ysamples[j]), self.interp_data[-1, j], delta=CUB_DELTA)

    def test_interpolate_2d_cubic_knot(self):
        """2D cubic interpolation. Test knot values"""
        self.init_2dcubic()
        for i in range(len(self.x)):
            for j in range(len(self.y)):
                self.assertAlmostEqual(self.interp_func(self.x[i], self.y[j]), self.data[i, j], delta=ABS_DELTA)

    def test_interpolate_2d_cubic_out(self):
        """2D cubic interpolation. Test values outside the boundaries"""
        self.init_2dcubic(extrapolate=False)
        for i in range(len(self.xsamples_out)):
            self.assertRaises(ValueError, self.interp_func, self.xsamples_out[i], self.ysamples_out[0])
            self.assertRaises(ValueError, self.interp_func, self.xsamples_out[i], self.ysamples_out[-1])
        self.interpolate_2d_xboundaries_assert(X_LOWER, X_UPPER, 1e-6, (Y_LOWER + Y_UPPER) / 2)
        for j in range(len(self.ysamples_out)):
            self.assertRaises(ValueError, self.interp_func, self.xsamples_out[0], self.ysamples_out[j])
            self.assertRaises(ValueError, self.interp_func, self.xsamples_out[-1], self.ysamples_out[j])
        self.interpolate_2d_yboundaries_assert((X_LOWER + X_UPPER) / 2, Y_LOWER, Y_UPPER, 1e-6)

    def test_interpolate_2d_cubic_extrapolate_nearest_range(self):
        """2D cubic interpolation. Tests the size of the extrapolation range."""
        self.init_2dcubic(extrapolate=True, extrapolation_type='nearest', extrapolation_range=1.)
        self.interpolate_2d_xboundaries_assert(X_LOWER - 1, X_UPPER + 1, 1e-6, (Y_LOWER + Y_UPPER) / 2)
        self.interpolate_2d_yboundaries_assert((X_LOWER + X_UPPER) / 2, Y_LOWER - 1, Y_UPPER + 1, 1e-6)

    def test_interpolate_2d_cubic_extrapolate_nearest_xinfyinf(self):
        """2D cubic interpolation. Test values in the extrapolation area with x below and y below the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(0, 0, self.extrap_data_nea[0][0], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_nearest_xinfymid(self):
        """2D cubic interpolation. Test values in the extrapolation area with x below and y inside the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(0, 1, self.extrap_data_nea[0][1], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_nearest_xinfysup(self):
        """2D cubic interpolation. Test values in the extrapolation area with x below and y above the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(0, 2, self.extrap_data_nea[0][2], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_nearest_xmidyinf(self):
        """2D cubic interpolation. Test values in the extrapolation area with x inside and y below the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(1, 0, self.extrap_data_nea[1][0], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_nearest_xmidysup(self):
        """2D cubic interpolation. Test values in the extrapolation area with x inside and y above the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(1, 2, self.extrap_data_nea[1][2], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_nearest_xsupyinf(self):
        """2D cubic interpolation. Test values in the extrapolation area with x above and y below the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(2, 0, self.extrap_data_nea[2][0], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_nearest_xsupymid(self):
        """2D cubic interpolation. Test values in the extrapolation area with x above and y inside the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(2, 1, self.extrap_data_nea[2][1], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_nearest_xsupysup(self):
        """2D cubic interpolation. Test values in the extrapolation area with x above and y above the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='nearest')
        self.interpolate_2d_extrapolate_assert(2, 2, self.extrap_data_nea[2][2], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_linear_range(self):
        """2D cubic interpolation. Tests the size of the extrapolation range."""
        self.init_2dcubic(extrapolate=True, extrapolation_type='linear', extrapolation_range=1.)
        self.interpolate_2d_xboundaries_assert(X_LOWER - 1, X_UPPER + 1, 1e-6, (Y_LOWER + Y_UPPER) / 2)
        self.interpolate_2d_yboundaries_assert((X_LOWER + X_UPPER) / 2, Y_LOWER - 1, Y_UPPER + 1, 1e-6)

    def test_interpolate_2d_cubic_extrapolate_linear_xinfyinf(self):
        """2D cubic interpolation. Test values in the extrapolation area with x below and y below the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(0, 0, self.extrap_data_lin[0][0], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_linear_xinfymid(self):
        """2D cubic interpolation. Test values in the extrapolation area with x below and y inside the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(0, 1, self.extrap_data_lin[0][1], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_linear_xinfysup(self):
        """2D cubic interpolation. Test values in the extrapolation area with x below and y above the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(0, 2, self.extrap_data_lin[0][2], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_linear_xmidyinf(self):
        """2D cubic interpolation. Test values in the extrapolation area with x inside and y below the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(1, 0, self.extrap_data_lin[1][0], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_linear_xmidysup(self):
        """2D cubic interpolation. Test values in the extrapolation area with x inside and y above the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(1, 2, self.extrap_data_lin[1][2], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_linear_xsupyinf(self):
        """2D cubic interpolation. Test values in the extrapolation area with x above and y below the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(2, 0, self.extrap_data_lin[2][0], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_linear_xsupymid(self):
        """2D cubic interpolation. Test values in the extrapolation area with x above and y inside the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(2, 1, self.extrap_data_lin[2][1], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_linear_xsupysup(self):
        """2D cubic interpolation. Test values in the extrapolation area with x above and y above the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='linear')
        self.interpolate_2d_extrapolate_assert(2, 2, self.extrap_data_lin[2][2], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_quadratic_range(self):
        """2D cubic interpolation. Tests the size of the extrapolation range."""
        self.init_2dcubic(extrapolate=True, extrapolation_type='quadratic', extrapolation_range=1.)
        self.interpolate_2d_xboundaries_assert(X_LOWER - 1, X_UPPER + 1, 1e-6, (Y_LOWER + Y_UPPER) / 2)
        self.interpolate_2d_yboundaries_assert((X_LOWER + X_UPPER) / 2, Y_LOWER - 1, Y_UPPER + 1, 1e-6)

    def test_interpolate_2d_cubic_extrapolate_quadratic_xinfyinf(self):
        """2D cubic interpolation. Test values in the extrapolation area with x below and y below the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='quadratic')
        self.interpolate_2d_extrapolate_assert(0, 0, self.extrap_data_qua[0][0], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_quadratic_xinfymid(self):
        """2D cubic interpolation. Test values in the extrapolation area with x below and y inside the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='quadratic')
        self.interpolate_2d_extrapolate_assert(0, 1, self.extrap_data_qua[0][1], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_quadratic_xinfysup(self):
        """2D cubic interpolation. Test values in the extrapolation area with x below and y above the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='quadratic')
        self.interpolate_2d_extrapolate_assert(0, 2, self.extrap_data_qua[0][2], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_quadratic_xmidyinf(self):
        """2D cubic interpolation. Test values in the extrapolation area with x inside and y below the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='quadratic')
        self.interpolate_2d_extrapolate_assert(1, 0, self.extrap_data_qua[1][0], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_quadratic_xmidysup(self):
        """2D cubic interpolation. Test values in the extrapolation area with x inside and y above the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='quadratic')
        self.interpolate_2d_extrapolate_assert(1, 2, self.extrap_data_qua[1][2], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_quadratic_xsupyinf(self):
        """2D cubic interpolation. Test values in the extrapolation area with x above and y below the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='quadratic')
        self.interpolate_2d_extrapolate_assert(2, 0, self.extrap_data_qua[2][0], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_quadratic_xsupymid(self):
        """2D cubic interpolation. Test values in the extrapolation area with x above and y inside the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='quadratic')
        self.interpolate_2d_extrapolate_assert(2, 1, self.extrap_data_qua[2][1], CUB_DELTA)

    def test_interpolate_2d_cubic_extrapolate_quadratic_xsupysup(self):
        """2D cubic interpolation. Test values in the extrapolation area with x above and y above the interpolation area
        """
        self.init_2dcubic(extrapolate=True, extrapolation_range=10, extrapolation_type='quadratic')
        self.interpolate_2d_extrapolate_assert(2, 2, self.extrap_data_qua[2][2], CUB_DELTA)

    def test_interpolate_2d_cubic_type_conversion(self):
        """2D cubic interpolation. Whatever the type of input data, the interpolating function must provide float numbers.
        """
        self.init_2dcubic([1, 2, 3, 4], [2, 3, 4, 5], np.ones((4, 4), dtype=int))
        self.assertIsInstance(self.interp_func(2.5, 4.5), float)

    def test_interpolate_2d_cubic_single_value_tolerated_x(self):
        """2D cubic interpolation. If tolerated, a single value input must be extrapolated to every real value on its axis.
        """
        self.init_2dcubic([2.], [2, 3, 4, 5], np.ones((1, 4)), tolerate_single_value=True)
        self.assertAlmostEqual(self.interp_func(-31946139.346, 2.5), 1., delta=ABS_DELTA)
        self.assertAlmostEqual(self.interp_func(31946139.346, 2.5), 1., delta=ABS_DELTA)
        self.assertAlmostEqual(self.interp_func(2., 2.5), 1., delta=ABS_DELTA)

    def test_interpolate_2d_cubic_single_value_tolerated_y(self):
        """2D cubic interpolation. If tolerated, a single value input must be extrapolated to every real value on its axis.
        """
        self.init_2dcubic([1, 2, 3, 4], [4.], np.ones((4, 1)), tolerate_single_value=True)
        self.assertAlmostEqual(self.interp_func(1.5, -31946139.346), 1., delta=ABS_DELTA)
        self.assertAlmostEqual(self.interp_func(1.5, 31946139.346), 1., delta=ABS_DELTA)
        self.assertAlmostEqual(self.interp_func(1.5, 2.), 1., delta=ABS_DELTA)

    def test_interpolate_2d_cubic_single_value_tolerated_xy(self):
        """2D cubic interpolation. If tolerated, a single value input must be extrapolated to every real value on its axis.
        """
        self.init_2dcubic([2.], [4.], np.ones((1, 1)), tolerate_single_value=True)
        self.assertAlmostEqual(self.interp_func(-31946139.346, 7856.45), 1., delta=ABS_DELTA)
        self.assertAlmostEqual(self.interp_func(31946139.346, -7856.45), 1., delta=ABS_DELTA)
        self.assertAlmostEqual(self.interp_func(2., 4.), 1., delta=ABS_DELTA)


if __name__ == '__main__':
    unittest.main()
