# apteco-api
A wrapper package for the Apteco API to allow access to Apteco Marketing Suite™ resources

This Python package is automatically generated by the [OpenAPI Generator](https://openapi-generator.tech) project:

- API version: v2
- OrbitAPI spec version: 1.8.13.2139 (with edits)
- Package version: 0.2.1
- Build package: `org.openapitools.codegen.languages.PythonClientCodegen`


## Requirements

- Python 2.7 and 3.4+
- Access to an installation of the Apteco API

The Apteco API, which also goes under the name **Orbit API**, is part of the Apteco Orbit™ installation.
If you are unsure whether you have access to this, please contact Apteco support (support@apteco.com).

*Note: the examples in this guide use Python 3.6+*


## Installation & Usage

You can install the package the usual way from PyPI using `pip`:

```
pip install apteco-api
```

Import the package in Python using:

```python
import apteco_api
```

*Note: the package follows the Python convention of a hyphen `-` in the package name
and underscore `_` in the import name.*


## Getting Started

The examples below shows how to connect to the API, log in to create an access token,
retrieve some data from one of the API's endpoints, and then log out.

*Note: your login credentials (`my_username` and `my_password` below)
are the same credentials you would use to log in to your FastStats system or Apteco Orbit™.*

### Logging in

```python
import apteco_api as aa

# configure the API client to use your instance of the API
config = aa.Configuration(host='https://example.com/OrbitAPI')
api_client = aa.ApiClient(configuration=config)

# send a simple login request
sessions_controller = aa.SessionsApi(api_client)
login_response = sessions_controller.sessions_create_session_simple(
    'my_data_view', 'my_username', 'my_password'
)

# keep a copy of the session ID (needed to log out)
session_id = login_response.session_id
# update the configuration with your credentials
config.api_key={'Authorization': login_response.access_token}
config.api_key_prefix={'Authorization': 'Bearer'}
```

Since the `config` object updated here is the configuration object for `api_client`,
the latter will now be authorised to access the API.

### Retrieving data from an endpoint

This code retrieves a list of tables in the `Holidays` system on the given DataView:

```python
tables = aa.FastStatsSystemsApi(api_client).fast_stats_systems_get_fast_stats_tables(
    'my_data_view',
    'holidays'
)
print(f"There are {tables.count} tables in the Holidays system:")
print('\n'.join(t.name for t in tables.list))
```

**Output:**

```commandline
There are 9 tables in the Holidays system:
Households
People
Bookings
Communication Responsible
Insurance
Communications
Content
Responses Attributed
Journey History
```

### Logging out

You can end your session using the `logout_session` endpoint,
and passing in the session ID returned from the logging in process above:

```python
aa.SessionsApi(api_client).sessions_logout_session('my_data_view', session_id)

try:
    tables = aa.FastStatsSystemsApi(api_client).fast_stats_systems_get_fast_stats_tables(
        'my_data_view',
        'holidays'
    )
except aa.ApiException as e:
    print(f"{e.status}: {e.reason}")
```

**Output:**

```commandline
401: Unauthorized
```

The first line successfully ended the session and so, as expected,
the attempt to retrieve the tables data failed and raised an exception.

Note that logging out like this ends the session on the server-side,
but it doesn't remove the copy of the (now invalid) access token kept in the
`api_client.configuration` object.


## General use of the API client

Every section of the API (`Audiences`, `FastStatsSystems`, `Queries`, `Sessions`, etc)
has a corresponding controller class in the `apteco-api` package, named `<section>Api`
e.g. `SessionsApi`, as seen above.

To use endpoints from a given section,
create an instance of the controller by passing an \[authorised\] `ApiClient` object
as the single argument to its constructor:

```python
queries_controller = aa.QueriesApi(api_client)
```

This object then has a method corresponding to each endpoint of that section of the API,
which can be called as a normal Python function.

```python
query_result = queries_controller.queries_perform_query_file_count_synchronously(
    'my_data_view',
    'holidays',
    query_file=aa.QueryFile('Private/Bookings to France or Germany.xml')
)
```

Some of the parameters for this function may need to be specific object types;
in the example here, the first two parameters are strings specifying the DataView and FastStats system to use for the query,
while the path of the file for the query is given via a `QueryFile` object supplied as a keyword-only argument.
The `QueryFile` object itself is initialised with a single argument, namely the filepath.

Similarly, the function call returns a Python object.
These API functions often return 'result' objects which bundle together various data and metadata as attributes,
and these attributes can then be accessed to obtain the information of interest.

```python
count = query_result.counts[0]
print(f"The query matched {count.count_value:,} {count.table_name.lower()}.")
```

**Output:**

```commandline
The query matched 985,734 bookings.
```

### Further details

All classes and methods have detailed docstrings providing further information about their parameters and return values.

You can also explore the API in a visual way by going to `/swagger/ui/index.html` from your API base URL,
e.g. `https://example.com/OrbitAPI/swagger/ui/index.html`

Here, each section of the API is listed, and clicking on one will expand the view
to show all the endpoints that belong to it.
Clicking on an endpoint will similarly expand that to detail its parameters
and show example values for using it.
You can also try out endpoints directly within the interface,
and view the returned data.

## Documentation for API Endpoints

All URIs are relative to *https://example.com/OrbitAPI*

Class | Method | HTTP request | Description
------------ | ------------- | ------------- | -------------
*AboutApi* | [**about_example_experimental_resource**](docs/AboutApi.md#about_example_experimental_resource) | **GET** /About/ExampleExperimentalResource | EXPERIMENTAL: Returns a sample string if experimental endpoints are enabled
*AboutApi* | [**about_force_process_garbage_collection**](docs/AboutApi.md#about_force_process_garbage_collection) | **POST** /About/Process/ForceGarbageCollection | Requires OrbitAdmin: Forces a garbage collection in the API&#39;s process and then returns details about the API&#39;s .Net process
*AboutApi* | [**about_get_data_view**](docs/AboutApi.md#about_get_data_view) | **GET** /About/DataViews/{dataViewName} | Get details for a particular DataView.
*AboutApi* | [**about_get_data_views**](docs/AboutApi.md#about_get_data_views) | **GET** /About/DataViews | Get the list of DataViews that are available.
*AboutApi* | [**about_get_data_views_for_domain**](docs/AboutApi.md#about_get_data_views_for_domain) | **GET** /About/DataViews/Domains/{domain} | Get the list of DataViews that are available to users with the specified email domain.
*AboutApi* | [**about_get_data_views_for_system_name**](docs/AboutApi.md#about_get_data_views_for_system_name) | **GET** /About/DataViews/Systems/{systemName} | Get the list of DataViews that are configured with the given FastStats system.
*AboutApi* | [**about_get_endpoints**](docs/AboutApi.md#about_get_endpoints) | **GET** /About/Endpoints | Returns details of all the endpoints in the API
*AboutApi* | [**about_get_language_details**](docs/AboutApi.md#about_get_language_details) | **GET** /About/Language | Returns information about the current language the API is operating in (based on details in the request)
*AboutApi* | [**about_get_orbit_settings**](docs/AboutApi.md#about_get_orbit_settings) | **GET** /About/Orbit/Settings/{settingsPath} | Gets Orbit settings at the given path
*AboutApi* | [**about_get_orbit_settings_root**](docs/AboutApi.md#about_get_orbit_settings_root) | **GET** /About/Orbit/Settings | Gets the complete Orbit settings object
*AboutApi* | [**about_get_process_details**](docs/AboutApi.md#about_get_process_details) | **GET** /About/Process | Requires OrbitAdmin: Returns details about the API&#39;s .Net process
*AboutApi* | [**about_get_version**](docs/AboutApi.md#about_get_version) | **GET** /About/Version | Returns version information about the API
*AudienceCompositionsApi* | [**audience_compositions_create_audience_composition**](docs/AudienceCompositionsApi.md#audience_compositions_create_audience_composition) | **POST** /{dataViewName}/AudienceCompositions/{systemName} | Requires OrbitAdmin: Create an audience composition for the given FastStats system.
*AudienceCompositionsApi* | [**audience_compositions_delete_audience_composition**](docs/AudienceCompositionsApi.md#audience_compositions_delete_audience_composition) | **DELETE** /{dataViewName}/AudienceCompositions/{systemName}/{audienceCompositionId} | Requires OrbitAdmin: Deletes a given audience composition for the given FastStats system.
*AudienceCompositionsApi* | [**audience_compositions_get_audience_composition**](docs/AudienceCompositionsApi.md#audience_compositions_get_audience_composition) | **GET** /{dataViewName}/AudienceCompositions/{systemName}/{audienceCompositionId} | Returns details for a given audience composition
*AudienceCompositionsApi* | [**audience_compositions_get_audience_compositions**](docs/AudienceCompositionsApi.md#audience_compositions_get_audience_compositions) | **GET** /{dataViewName}/AudienceCompositions | Returns all the avaiable audience compositions.
*AudienceCompositionsApi* | [**audience_compositions_get_audience_compositions_for_system**](docs/AudienceCompositionsApi.md#audience_compositions_get_audience_compositions_for_system) | **GET** /{dataViewName}/AudienceCompositions/{systemName} | Returns all the avaiable audience compositions for the given FastStats system
*AudienceCompositionsApi* | [**audience_compositions_update_audience_composition**](docs/AudienceCompositionsApi.md#audience_compositions_update_audience_composition) | **PUT** /{dataViewName}/AudienceCompositions/{systemName}/{audienceCompositionId} | Requires OrbitAdmin: Update a given audience composition for the given FastStats system.
*AudienceHitsApi* | [**audience_hits_get_audience_hit**](docs/AudienceHitsApi.md#audience_hits_get_audience_hit) | **GET** /{dataViewName}/AudienceHits/{audienceHitId} | Requires OrbitAdmin: Returns details for a given audience hit
*AudienceHitsApi* | [**audience_hits_get_audience_hits**](docs/AudienceHitsApi.md#audience_hits_get_audience_hits) | **GET** /{dataViewName}/AudienceHits | Requires OrbitAdmin: Returns all the hit information for all audiences.
*AudiencesApi* | [**audiences_calculate_audience_data_licensing_sync**](docs/AudiencesApi.md#audiences_calculate_audience_data_licensing_sync) | **POST** /{dataViewName}/Audiences/{audienceId}/DataLicensingSync | Get data licensing information for the latest version of this audience
*AudiencesApi* | [**audiences_calculate_audience_latest_update_sync**](docs/AudiencesApi.md#audiences_calculate_audience_latest_update_sync) | **POST** /{dataViewName}/Audiences/{audienceId}/CalculateSync | Calculate counts against the FastStats system for the latest version of this audience.  The different queries associated with the latest  version of this audience will be combined to produce the end result
*AudiencesApi* | [**audiences_cancel_calculate_audience_data_licensing_job**](docs/AudiencesApi.md#audiences_cancel_calculate_audience_data_licensing_job) | **DELETE** /{dataViewName}/Audiences/{audienceId}/DataLicensingJobs/{jobId} | Cancel a running data licensing job
*AudiencesApi* | [**audiences_cancel_calculate_audience_job**](docs/AudiencesApi.md#audiences_cancel_calculate_audience_job) | **DELETE** /{dataViewName}/Audiences/{audienceId}/CalculateJobs/{jobId} | Cancel a running calculate job
*AudiencesApi* | [**audiences_cancel_check_audience_job**](docs/AudiencesApi.md#audiences_cancel_check_audience_job) | **DELETE** /{dataViewName}/Audiences/{audienceId}/CheckJobs/{jobId} | Cancel a running check job
*AudiencesApi* | [**audiences_cancel_export_audience_job**](docs/AudiencesApi.md#audiences_cancel_export_audience_job) | **DELETE** /{dataViewName}/Audiences/{audienceId}/ExportJobs/{jobId} | Cancel a running export job
*AudiencesApi* | [**audiences_check_audience_latest_update_sync**](docs/AudiencesApi.md#audiences_check_audience_latest_update_sync) | **POST** /{dataViewName}/Audiences/{audienceId}/CheckSync | Calculate check statistics against the FastStats system for the latest version of this audience.  The different queries associated with the latest  version of this audience will be combined to identify the data to analyse and the specified dimensions will be used to perform the analysis.
*AudiencesApi* | [**audiences_create_audience**](docs/AudiencesApi.md#audiences_create_audience) | **POST** /{dataViewName}/Audiences | Creates a new audience from the given details for the logged in user.
*AudiencesApi* | [**audiences_create_audience_hit_for_audience**](docs/AudiencesApi.md#audiences_create_audience_hit_for_audience) | **POST** /{dataViewName}/Audiences/{audienceId}/Hits | Register a hit (view) for the given audience
*AudiencesApi* | [**audiences_create_audience_update**](docs/AudiencesApi.md#audiences_create_audience_update) | **POST** /{dataViewName}/Audiences/{audienceId}/Updates | Updates the details of a particular audience.  If you don&#39;t have an id for the  audience then POST to the /Audiences URL to create a new audience.
*AudiencesApi* | [**audiences_create_calculate_audience_data_licensing_job**](docs/AudiencesApi.md#audiences_create_calculate_audience_data_licensing_job) | **POST** /{dataViewName}/Audiences/{audienceId}/DataLicensingJobs | Create a new job to get data licensing information for the latest version of this audience
*AudiencesApi* | [**audiences_create_calculate_audience_job_for_latest_update**](docs/AudiencesApi.md#audiences_create_calculate_audience_job_for_latest_update) | **POST** /{dataViewName}/Audiences/{audienceId}/CalculateJobs | Create a new job to calculate counts against the FastStats system for the latest version of this audience.  The different queries associated with the latest  version of this audience will be combined to produce the end result
*AudiencesApi* | [**audiences_create_check_audience_job_for_latest_update**](docs/AudiencesApi.md#audiences_create_check_audience_job_for_latest_update) | **POST** /{dataViewName}/Audiences/{audienceId}/CheckJobs | Create a new job to calculate check statistics against the FastStats system for the latest version of this audience.  The different queries associated with the latest  version of this audience will be combined to identify the data to analyse and the specified dimensions will be used to perform the analysis.
*AudiencesApi* | [**audiences_create_export_audience_job_for_latest_update**](docs/AudiencesApi.md#audiences_create_export_audience_job_for_latest_update) | **POST** /{dataViewName}/Audiences/{audienceId}/ExportJobs | Create a new job to export data from the FastStats system for the latest version of this audience.  The different queries associated with the latest  version of this audience will be combined to identify the data to export and the specified columns will be used to export the data, to a file  and/or as a sample within the body of the result
*AudiencesApi* | [**audiences_delete_audience**](docs/AudiencesApi.md#audiences_delete_audience) | **DELETE** /{dataViewName}/Audiences/{audienceId} | Deletes the specified audience
*AudiencesApi* | [**audiences_export_audience_latest_update_sync**](docs/AudiencesApi.md#audiences_export_audience_latest_update_sync) | **POST** /{dataViewName}/Audiences/{audienceId}/ExportSync | Create a new job to export data from the FastStats system for the latest version of this audience.  The different queries associated with the latest  version of this audience will be combined to identify the data to export and the specified columns will be used to export the data, to a file  and/or as a sample within the body of the result
*AudiencesApi* | [**audiences_get_audience**](docs/AudiencesApi.md#audiences_get_audience) | **GET** /{dataViewName}/Audiences/{audienceId} | Returns the details of a particular audience
*AudiencesApi* | [**audiences_get_audience_hit_for_audience**](docs/AudiencesApi.md#audiences_get_audience_hit_for_audience) | **GET** /{dataViewName}/Audiences/{audienceId}/Hits/{audienceHitId} | Returns details for a given audience hit for this audience
*AudiencesApi* | [**audiences_get_audience_hits_for_audience**](docs/AudiencesApi.md#audiences_get_audience_hits_for_audience) | **GET** /{dataViewName}/Audiences/{audienceId}/Hits | Returns a summary of the hits for this audience - i.e. information about when users have viewed the audience.
*AudiencesApi* | [**audiences_get_audience_latest_native_for_nett_query**](docs/AudiencesApi.md#audiences_get_audience_latest_native_for_nett_query) | **GET** /{dataViewName}/Audiences/{audienceId}/Native/Nett | Returns native XML (i.e. for use with other FastStats applications) for the Nett query of the latest update for a particular audience
*AudiencesApi* | [**audiences_get_audience_result**](docs/AudiencesApi.md#audiences_get_audience_result) | **GET** /{dataViewName}/Audiences/{audienceId}/Results/{audienceResultId} | Returns details of a particular result for a particular audience
*AudiencesApi* | [**audiences_get_audience_results**](docs/AudiencesApi.md#audiences_get_audience_results) | **GET** /{dataViewName}/Audiences/{audienceId}/Results | Returns a summary of the results for a particular audience
*AudiencesApi* | [**audiences_get_audience_update**](docs/AudiencesApi.md#audiences_get_audience_update) | **GET** /{dataViewName}/Audiences/{audienceId}/Updates/{audienceUpdateId} | Returns details of an update for a particular audience
*AudiencesApi* | [**audiences_get_audience_updates**](docs/AudiencesApi.md#audiences_get_audience_updates) | **GET** /{dataViewName}/Audiences/{audienceId}/Updates | Returns a summary of the updates to a particular audience
*AudiencesApi* | [**audiences_get_audiences**](docs/AudiencesApi.md#audiences_get_audiences) | **GET** /{dataViewName}/Audiences | Requires OrbitAdmin: Gets summary information about each audience in the DataView.
*AudiencesApi* | [**audiences_get_calculate_audience_data_licensing_job**](docs/AudiencesApi.md#audiences_get_calculate_audience_data_licensing_job) | **POST** /{dataViewName}/Audiences/{audienceId}/DataLicensingJobs/{jobId} | Get the status of a running calculate job
*AudiencesApi* | [**audiences_get_calculate_audience_job**](docs/AudiencesApi.md#audiences_get_calculate_audience_job) | **GET** /{dataViewName}/Audiences/{audienceId}/CalculateJobs/{jobId} | Get the status of a running calculate job
*AudiencesApi* | [**audiences_get_check_audience_job**](docs/AudiencesApi.md#audiences_get_check_audience_job) | **GET** /{dataViewName}/Audiences/{audienceId}/CheckJobs/{jobId} | Get the status of a running check job
*AudiencesApi* | [**audiences_get_export_audience_job**](docs/AudiencesApi.md#audiences_get_export_audience_job) | **GET** /{dataViewName}/Audiences/{audienceId}/ExportJobs/{jobId} | Get the status of a running export job
*AudiencesApi* | [**audiences_transfer_audience_ownership**](docs/AudiencesApi.md#audiences_transfer_audience_ownership) | **POST** /{dataViewName}/Audiences/{audienceId}/TransferOwnership | Transfer ownership of an audience from the current user to a new owner
*CollectionHitsApi* | [**collection_hits_get_collection_hit**](docs/CollectionHitsApi.md#collection_hits_get_collection_hit) | **GET** /{dataViewName}/CollectionHits/{collectionHitId} | Requires OrbitAdmin: Returns details for a given collection hit.
*CollectionHitsApi* | [**collection_hits_get_collection_hits**](docs/CollectionHitsApi.md#collection_hits_get_collection_hits) | **GET** /{dataViewName}/CollectionHits | Requires OrbitAdmin: Returns all the hit information for all collections.
*CollectionsApi* | [**collections_create_collection**](docs/CollectionsApi.md#collections_create_collection) | **POST** /{dataViewName}/Collections | Creates a new collection from the given details for the logged in user.
*CollectionsApi* | [**collections_create_collection_hit_for_collection**](docs/CollectionsApi.md#collections_create_collection_hit_for_collection) | **POST** /{dataViewName}/Collections/{collectionId}/Hits | Register a hit (view) for the given collection
*CollectionsApi* | [**collections_delete_collection**](docs/CollectionsApi.md#collections_delete_collection) | **DELETE** /{dataViewName}/Collections/{collectionId} | Deletes the specified collection
*CollectionsApi* | [**collections_get_collection**](docs/CollectionsApi.md#collections_get_collection) | **GET** /{dataViewName}/Collections/{collectionId} | Returns the details of a particular collection
*CollectionsApi* | [**collections_get_collection_hit_for_collection**](docs/CollectionsApi.md#collections_get_collection_hit_for_collection) | **GET** /{dataViewName}/Collections/{collectionId}/Hits/{collectionHitId} | Returns details for a given collection hit for this collection
*CollectionsApi* | [**collections_get_collection_hits_for_collection**](docs/CollectionsApi.md#collections_get_collection_hits_for_collection) | **GET** /{dataViewName}/Collections/{collectionId}/Hits | Returns a summary of the hits for this collection - i.e. information about when users have viewed the collection.
*CollectionsApi* | [**collections_get_collection_part**](docs/CollectionsApi.md#collections_get_collection_part) | **GET** /{dataViewName}/Collections/{collectionId}/Parts/{partIndex} | Returns details of a part contained within a particular collection
*CollectionsApi* | [**collections_get_collection_parts**](docs/CollectionsApi.md#collections_get_collection_parts) | **GET** /{dataViewName}/Collections/{collectionId}/Parts | Returns a summary of the parts contained within a particular collection
*CollectionsApi* | [**collections_get_collections**](docs/CollectionsApi.md#collections_get_collections) | **GET** /{dataViewName}/Collections | Requires OrbitAdmin: Gets summary information about each collection in the DataView.
*CollectionsApi* | [**collections_transfer_collection_ownership**](docs/CollectionsApi.md#collections_transfer_collection_ownership) | **POST** /{dataViewName}/Collections/{collectionId}/TransferOwnership | Transfer ownership of a collection from the current user to a new owner
*CollectionsApi* | [**collections_upsert_collection**](docs/CollectionsApi.md#collections_upsert_collection) | **PUT** /{dataViewName}/Collections/{collectionId} | Updates the details of a particular collection.  If you don&#39;t have an id for the  collection then POST to the /Collections URL to create a new collection.
*CubesApi* | [**cubes_calculate_cube_synchronously**](docs/CubesApi.md#cubes_calculate_cube_synchronously) | **POST** /{dataViewName}/Cubes/{systemName}/CalculateSync | EXPERIMENTAL: Calcaultes a cube using the given definition and returns the results.  The data to build the cube from is defined by the base query provided.
*DashboardsApi* | [**dashboards_get_dashboard_item_data_sync**](docs/DashboardsApi.md#dashboards_get_dashboard_item_data_sync) | **POST** /{dataViewName}/Dashboards/{dashboardId}/Items/{dashboardItemId}/CalculateSync | EXPERIMENTAL: Return data needed to render visualisation for dashboard item
*DataLicensingApi* | [**data_licensing_cancel_data_purchase_job**](docs/DataLicensingApi.md#data_licensing_cancel_data_purchase_job) | **DELETE** /{dataViewName}/DataLicensing/{systemName}/DataPurchaseJobs/{jobId} | Cancel a running data purchasing job
*DataLicensingApi* | [**data_licensing_cancel_licensing_sets_job**](docs/DataLicensingApi.md#data_licensing_cancel_licensing_sets_job) | **DELETE** /{dataViewName}/DataLicensing/{systemName}/LicensingSetsJobs/{jobId} | Cancel a running licensing sets job
*DataLicensingApi* | [**data_licensing_create_get_licensing_sets_job**](docs/DataLicensingApi.md#data_licensing_create_get_licensing_sets_job) | **POST** /{dataViewName}/DataLicensing/{systemName}/LicensingSetsJobs | Create a new job to get the available licensing sets for a system
*DataLicensingApi* | [**data_licensing_create_purchase_data_licensing_job**](docs/DataLicensingApi.md#data_licensing_create_purchase_data_licensing_job) | **POST** /{dataViewName}/DataLicensing/{systemName}/DataPurchaseJobs | Create a new job to purchase data licensing information
*DataLicensingApi* | [**data_licensing_get_data_licensing_system**](docs/DataLicensingApi.md#data_licensing_get_data_licensing_system) | **GET** /{dataViewName}/DataLicensing/{systemName} | Returns some top-level details for the specified FastStats system to license data from
*DataLicensingApi* | [**data_licensing_get_data_licensing_systems**](docs/DataLicensingApi.md#data_licensing_get_data_licensing_systems) | **GET** /{dataViewName}/DataLicensing | Returns the list of FastStats systems available for licensing data from
*DataLicensingApi* | [**data_licensing_get_data_purchase_job**](docs/DataLicensingApi.md#data_licensing_get_data_purchase_job) | **GET** /{dataViewName}/DataLicensing/{systemName}/DataPurchaseJobs/{jobId} | Get the status of a running purchase job
*DataLicensingApi* | [**data_licensing_get_licensing_sets_job**](docs/DataLicensingApi.md#data_licensing_get_licensing_sets_job) | **GET** /{dataViewName}/DataLicensing/{systemName}/LicensingSetsJobs/{jobId} | Get the status of a running licensing sets job
*DataLicensingApi* | [**data_licensing_get_licensing_sets_sync**](docs/DataLicensingApi.md#data_licensing_get_licensing_sets_sync) | **GET** /{dataViewName}/DataLicensing/{systemName}/LicensingSets | Get the licensing sets for a system
*DataLicensingApi* | [**data_licensing_purchase_data_licensing_sync**](docs/DataLicensingApi.md#data_licensing_purchase_data_licensing_sync) | **POST** /{dataViewName}/DataLicensing/{systemName}/PurchaseSync | Purchase data licensing information
*DirectoriesApi* | [**directories_delete_directory**](docs/DirectoriesApi.md#directories_delete_directory) | **DELETE** /{dataViewName}/Directories/{systemName}/{directoryPath} | Deletes directory at location
*DirectoriesApi* | [**directories_get_file_entries**](docs/DirectoriesApi.md#directories_get_file_entries) | **GET** /{dataViewName}/Directories/{systemName}/{directoryPath} | Returns the list of files and subdirectories under the given directory
*DirectoriesApi* | [**directories_get_file_systems**](docs/DirectoriesApi.md#directories_get_file_systems) | **GET** /{dataViewName}/Directories | Returns the list of systems that have access to a filesystem
*DirectoriesApi* | [**directories_get_root_file_entries**](docs/DirectoriesApi.md#directories_get_root_file_entries) | **GET** /{dataViewName}/Directories/{systemName} | Returns the list of root directories configured in this FastStats system
*DirectoriesApi* | [**directories_upsert_directory**](docs/DirectoriesApi.md#directories_upsert_directory) | **PUT** /{dataViewName}/Directories/{systemName}/{directoryPath} | Ensure that a directory exists in a location
*ExportsApi* | [**exports_get_export_system**](docs/ExportsApi.md#exports_get_export_system) | **GET** /{dataViewName}/Exports/{systemName} | Returns some top-level details for the specified FastStats system to export from
*ExportsApi* | [**exports_get_export_systems**](docs/ExportsApi.md#exports_get_export_systems) | **GET** /{dataViewName}/Exports | Returns the list of FastStats systems available for exporting data from
*ExportsApi* | [**exports_perform_export_synchronously**](docs/ExportsApi.md#exports_perform_export_synchronously) | **POST** /{dataViewName}/Exports/{systemName}/ExportSync | EXPERIMENTAL: Exports data using the given export definition and returns the results.  The results might contain the actual data in the \&quot;rows\&quot; part of the result or this might be written to a file.  The data to be exported is defined by the base query provided, along with any limits defined in the export request.
*FastStatsBuildsApi* | [**fast_stats_builds_cancel_fast_stats_build_job**](docs/FastStatsBuildsApi.md#fast_stats_builds_cancel_fast_stats_build_job) | **DELETE** /{dataViewName}/FastStatsBuilds/{systemName}/BuildJobs/{jobId} | EXPERIMENTAL: Requires OrbitAdmin: Cancel a running data purchasing job
*FastStatsBuildsApi* | [**fast_stats_builds_create_fast_stats_build_job**](docs/FastStatsBuildsApi.md#fast_stats_builds_create_fast_stats_build_job) | **POST** /{dataViewName}/FastStatsBuilds/{systemName}/BuildJobs | EXPERIMENTAL: Requires OrbitAdmin: Create a new job to builds a FastStats system from the given definition
*FastStatsBuildsApi* | [**fast_stats_builds_fast_stats_build_sync**](docs/FastStatsBuildsApi.md#fast_stats_builds_fast_stats_build_sync) | **POST** /{dataViewName}/FastStatsBuilds/{systemName}/BuildSync | EXPERIMENTAL: Requires OrbitAdmin: Builds a FastStats system from the given definition
*FastStatsBuildsApi* | [**fast_stats_builds_get_fast_stats_build_job**](docs/FastStatsBuildsApi.md#fast_stats_builds_get_fast_stats_build_job) | **GET** /{dataViewName}/FastStatsBuilds/{systemName}/BuildJobs/{jobId} | EXPERIMENTAL: Requires OrbitAdmin: Get the status of a running FastStats build job
*FastStatsJobsApi* | [**fast_stats_jobs_calculate_processing_time_stats_for_system_sync**](docs/FastStatsJobsApi.md#fast_stats_jobs_calculate_processing_time_stats_for_system_sync) | **POST** /{dataViewName}/FastStatsJobs/{systemName}/CalculateProcessingTimeStatsSync | EXPERIMENTAL: Requires OrbitAdmin: Calculate processing statistics for the specified jobs in the FastStats job queue for the given system.
*FastStatsJobsApi* | [**fast_stats_jobs_calculate_processing_time_stats_sync**](docs/FastStatsJobsApi.md#fast_stats_jobs_calculate_processing_time_stats_sync) | **POST** /{dataViewName}/FastStatsJobs/CalculateProcessingTimeStatsSync | EXPERIMENTAL: Requires OrbitAdmin: Calculate processing statistics for the specified jobs in the FastStats job queue for all systems in a particular DataView.
*FastStatsJobsApi* | [**fast_stats_jobs_get_fast_stats_job**](docs/FastStatsJobsApi.md#fast_stats_jobs_get_fast_stats_job) | **GET** /{dataViewName}/FastStatsJobs/{systemName}/{jobId} | EXPERIMENTAL: Requires OrbitAdmin: Gets details for a particular job in the FastStats job queue for the given system.
*FastStatsJobsApi* | [**fast_stats_jobs_get_fast_stats_jobs**](docs/FastStatsJobsApi.md#fast_stats_jobs_get_fast_stats_jobs) | **GET** /{dataViewName}/FastStatsJobs | EXPERIMENTAL: Requires OrbitAdmin: Gets all the jobs in the FastStats job queue for all systems in a particular DataView.
*FastStatsJobsApi* | [**fast_stats_jobs_get_fast_stats_jobs_for_system**](docs/FastStatsJobsApi.md#fast_stats_jobs_get_fast_stats_jobs_for_system) | **GET** /{dataViewName}/FastStatsJobs/{systemName} | EXPERIMENTAL: Requires OrbitAdmin: Gets all the jobs in the FastStats job queue for the given system.
*FastStatsSystemsApi* | [**fast_stats_systems_get_all_fast_stats_system_items**](docs/FastStatsSystemsApi.md#fast_stats_systems_get_all_fast_stats_system_items) | **GET** /{dataViewName}/FastStatsSystems/{systemName}/All | Gets all FastStats systems items - variables, var codes, tables and folders
*FastStatsSystemsApi* | [**fast_stats_systems_get_fast_stats_folder**](docs/FastStatsSystemsApi.md#fast_stats_systems_get_fast_stats_folder) | **GET** /{dataViewName}/FastStatsSystems/{systemName}/Folders/{path} | Gets the folder structure for the FastStats system
*FastStatsSystemsApi* | [**fast_stats_systems_get_fast_stats_root_folder**](docs/FastStatsSystemsApi.md#fast_stats_systems_get_fast_stats_root_folder) | **GET** /{dataViewName}/FastStatsSystems/{systemName}/Folders | Gets the items in the root of the FastStats system folder structure
*FastStatsSystemsApi* | [**fast_stats_systems_get_fast_stats_system**](docs/FastStatsSystemsApi.md#fast_stats_systems_get_fast_stats_system) | **GET** /{dataViewName}/FastStatsSystems/{systemName} | Returns some top-level details for the specified FastStats system
*FastStatsSystemsApi* | [**fast_stats_systems_get_fast_stats_systems**](docs/FastStatsSystemsApi.md#fast_stats_systems_get_fast_stats_systems) | **GET** /{dataViewName}/FastStatsSystems | Returns the list of FastStats systems available
*FastStatsSystemsApi* | [**fast_stats_systems_get_fast_stats_table**](docs/FastStatsSystemsApi.md#fast_stats_systems_get_fast_stats_table) | **GET** /{dataViewName}/FastStatsSystems/{systemName}/Tables/{tableName} | Gets the details for a particular table in the FastStats system
*FastStatsSystemsApi* | [**fast_stats_systems_get_fast_stats_tables**](docs/FastStatsSystemsApi.md#fast_stats_systems_get_fast_stats_tables) | **GET** /{dataViewName}/FastStatsSystems/{systemName}/Tables | Gets all the tables present in the FastStats system
*FastStatsSystemsApi* | [**fast_stats_systems_get_fast_stats_variable**](docs/FastStatsSystemsApi.md#fast_stats_systems_get_fast_stats_variable) | **GET** /{dataViewName}/FastStatsSystems/{systemName}/Variables/{variableName} | Gets the details for a particular variable in the FastStats system
*FastStatsSystemsApi* | [**fast_stats_systems_get_fast_stats_variable_codes**](docs/FastStatsSystemsApi.md#fast_stats_systems_get_fast_stats_variable_codes) | **GET** /{dataViewName}/FastStatsSystems/{systemName}/Variables/{variableName}/Codes | Gets all the categories (var codes) for the specified variable in the FastStats system if it is a selector variable
*FastStatsSystemsApi* | [**fast_stats_systems_get_fast_stats_variables**](docs/FastStatsSystemsApi.md#fast_stats_systems_get_fast_stats_variables) | **GET** /{dataViewName}/FastStatsSystems/{systemName}/Variables | Gets all the variables present in the FastStats system
*FastStatsSystemsApi* | [**fast_stats_systems_refresh_system_information_sync**](docs/FastStatsSystemsApi.md#fast_stats_systems_refresh_system_information_sync) | **POST** /{dataViewName}/FastStatsSystems/{systemName}/RefreshInformationSync | Requires OrbitAdmin: An endpoint to request the API refresh any information it holds on the given FastStats system.  This endpoint will wait until the refresh has completed before returning.
*FilesApi* | [**files_delete_file**](docs/FilesApi.md#files_delete_file) | **DELETE** /{dataViewName}/Files/{systemName}/{filePath} | Deletes file at location
*FilesApi* | [**files_get_file**](docs/FilesApi.md#files_get_file) | **GET** /{dataViewName}/Files/{systemName}/{filePath} | Returns the contents for a file
*FilesApi* | [**files_upsert_file**](docs/FilesApi.md#files_upsert_file) | **PUT** /{dataViewName}/Files/{systemName}/{filePath} | Creates or updates a file at a location
*GroupsApi* | [**groups_get_group_details**](docs/GroupsApi.md#groups_get_group_details) | **GET** /{dataViewName}/Groups/{groupId} | Requires OrbitAdmin: Returns details for the given group
*GroupsApi* | [**groups_get_groups**](docs/GroupsApi.md#groups_get_groups) | **GET** /{dataViewName}/Groups | Requires OrbitAdmin: Returns all groups in the DataView.
*GroupsApi* | [**groups_get_user_details_for_group**](docs/GroupsApi.md#groups_get_user_details_for_group) | **GET** /{dataViewName}/Groups/{groupId}/Users | Requires OrbitAdmin: Returns all users in the given group.
*GroupsApi* | [**groups_get_user_details_for_unallocated_group**](docs/GroupsApi.md#groups_get_user_details_for_unallocated_group) | **GET** /{dataViewName}/Groups/Unallocated/Users | Requires OrbitAdmin: Returns all users that haven&#39;t been allocated to a group.
*PeopleStageApi* | [**people_stage_get_people_stage_channel**](docs/PeopleStageApi.md#people_stage_get_people_stage_channel) | **GET** /{dataViewName}/PeopleStage/{systemName}/Channels/{channelId} | Returns the details of a particular PeopleStage channel
*PeopleStageApi* | [**people_stage_get_people_stage_channels**](docs/PeopleStageApi.md#people_stage_get_people_stage_channels) | **GET** /{dataViewName}/PeopleStage/{systemName}/Channels | Returns the list of PeopleStage channels available in this FastStats system
*PeopleStageApi* | [**people_stage_get_people_stage_element**](docs/PeopleStageApi.md#people_stage_get_people_stage_element) | **GET** /{dataViewName}/PeopleStage/{systemName}/Elements/{elementId} | Returns the details of a particular PeopleStage element
*PeopleStageApi* | [**people_stage_get_people_stage_element_channel_statistics**](docs/PeopleStageApi.md#people_stage_get_people_stage_element_channel_statistics) | **GET** /{dataViewName}/PeopleStage/{systemName}/Elements/{elementId}/ChannelStats | Returns statistics for the total number of communications sent per channel for a particular PeopleStage element
*PeopleStageApi* | [**people_stage_get_people_stage_element_children**](docs/PeopleStageApi.md#people_stage_get_people_stage_element_children) | **GET** /{dataViewName}/PeopleStage/{systemName}/Elements/{elementId}/Children | Returns the list of children for a particular PeopleStage element
*PeopleStageApi* | [**people_stage_get_people_stage_element_communication_statistics**](docs/PeopleStageApi.md#people_stage_get_people_stage_element_communication_statistics) | **GET** /{dataViewName}/PeopleStage/{systemName}/Elements/{elementId}/CommunicationStats | Returns statistics for the number of communications sent over time for a particular PeopleStage element
*PeopleStageApi* | [**people_stage_get_people_stage_element_response_statistics**](docs/PeopleStageApi.md#people_stage_get_people_stage_element_response_statistics) | **GET** /{dataViewName}/PeopleStage/{systemName}/Elements/{elementId}/ResponseStats | Returns statistics for the total number of responses received per channel for a particular PeopleStage element
*PeopleStageApi* | [**people_stage_get_people_stage_element_status**](docs/PeopleStageApi.md#people_stage_get_people_stage_element_status) | **GET** /{dataViewName}/PeopleStage/{systemName}/Elements/{elementId}/Status | Returns the status for a PeopleStage element, where status information is available
*PeopleStageApi* | [**people_stage_get_people_stage_element_status_for_descendants**](docs/PeopleStageApi.md#people_stage_get_people_stage_element_status_for_descendants) | **GET** /{dataViewName}/PeopleStage/{systemName}/Elements/{elementId}/Status/Descendants | Returns the status for all the descendant elements of a PeopleStage element where status information is available
*PeopleStageApi* | [**people_stage_get_people_stage_elements**](docs/PeopleStageApi.md#people_stage_get_people_stage_elements) | **GET** /{dataViewName}/PeopleStage/{systemName}/Elements | Returns the list of PeopleStage elements available in this FastStats system
*PeopleStageApi* | [**people_stage_get_people_stage_range_statistics_for_descendants_sync**](docs/PeopleStageApi.md#people_stage_get_people_stage_range_statistics_for_descendants_sync) | **GET** /{dataViewName}/PeopleStage/{systemName}/Elements/{elementId}/RangeStats/Descendants/Sync | Returns some statistics for a given date range for a particular PeopleStage element.  This call may take a long time and will block until the information is available.
*PeopleStageApi* | [**people_stage_get_people_stage_range_statistics_sync**](docs/PeopleStageApi.md#people_stage_get_people_stage_range_statistics_sync) | **GET** /{dataViewName}/PeopleStage/{systemName}/Elements/{elementId}/RangeStats/Sync | Returns some statistics for a given date range for a particular PeopleStage element.  This call may take a long time and will block until the information is available.
*PeopleStageApi* | [**people_stage_get_people_stage_system**](docs/PeopleStageApi.md#people_stage_get_people_stage_system) | **GET** /{dataViewName}/PeopleStage/{systemName} | Returns details of the PeopleStage system for the given system name
*PeopleStageApi* | [**people_stage_get_people_stage_systems**](docs/PeopleStageApi.md#people_stage_get_people_stage_systems) | **GET** /{dataViewName}/PeopleStage | Returns the list of systems configured to support PeopleStage
*QueriesApi* | [**queries_perform_get_query_file_definition_synchronously**](docs/QueriesApi.md#queries_perform_get_query_file_definition_synchronously) | **POST** /{dataViewName}/Queries/{systemName}/GetFileSync | EXPERIMENTAL: Get the query definition in the specified file
*QueriesApi* | [**queries_perform_query_count_synchronously**](docs/QueriesApi.md#queries_perform_query_count_synchronously) | **POST** /{dataViewName}/Queries/{systemName}/CountSync | EXPERIMENTAL: Counts the given query and returns the results
*QueriesApi* | [**queries_perform_query_file_count_synchronously**](docs/QueriesApi.md#queries_perform_query_file_count_synchronously) | **POST** /{dataViewName}/Queries/{systemName}/CountFileSync | EXPERIMENTAL: Counts the query in the specified file and returns the results
*QueriesApi* | [**queries_perform_save_query_file_definition_synchronously**](docs/QueriesApi.md#queries_perform_save_query_file_definition_synchronously) | **POST** /{dataViewName}/Queries/{systemName}/SaveFileSync | EXPERIMENTAL: Get the query definition in the specified file
*SessionsApi* | [**sessions_convert_session_to_access_token**](docs/SessionsApi.md#sessions_convert_session_to_access_token) | **POST** /{dataViewName}/Sessions/ConvertSession | Creates an API session token from a traditional FastStats session id
*SessionsApi* | [**sessions_create_login_parameters**](docs/SessionsApi.md#sessions_create_login_parameters) | **POST** /{dataViewName}/Sessions/LoginParameters | Creates a new set of parameters to use when creating a new session via the salted login method.
*SessionsApi* | [**sessions_create_session_salted**](docs/SessionsApi.md#sessions_create_session_salted) | **POST** /{dataViewName}/Sessions/SaltedLogin | Creates a session to use for other API requests
*SessionsApi* | [**sessions_create_session_simple**](docs/SessionsApi.md#sessions_create_session_simple) | **POST** /{dataViewName}/Sessions/SimpleLogin | Creates a session to use for other API requests
*SessionsApi* | [**sessions_get_session_details**](docs/SessionsApi.md#sessions_get_session_details) | **GET** /{dataViewName}/Sessions/{sessionId} | Gets some simple user details for the given session id
*SessionsApi* | [**sessions_get_session_details_list**](docs/SessionsApi.md#sessions_get_session_details_list) | **GET** /{dataViewName}/Sessions | Requires OrbitAdmin: Gets some simple user details for all currently valid sessions.
*SessionsApi* | [**sessions_logout_session**](docs/SessionsApi.md#sessions_logout_session) | **DELETE** /{dataViewName}/Sessions/{sessionId} | Logs out the specified session
*SettingsApi* | [**settings_delete_data_view_settings**](docs/SettingsApi.md#settings_delete_data_view_settings) | **DELETE** /{dataViewName}/Settings/DataView/{settingsPath} | Deletes DataView settings at the given path
*SettingsApi* | [**settings_delete_data_view_settings_root**](docs/SettingsApi.md#settings_delete_data_view_settings_root) | **DELETE** /{dataViewName}/Settings/DataView | Deletes the complete DataView settings
*SettingsApi* | [**settings_get_data_view_settings**](docs/SettingsApi.md#settings_get_data_view_settings) | **GET** /{dataViewName}/Settings/DataView/{settingsPath} | Gets DataView settings at the given path
*SettingsApi* | [**settings_get_data_view_settings_root**](docs/SettingsApi.md#settings_get_data_view_settings_root) | **GET** /{dataViewName}/Settings/DataView | Gets the complete DataView settings object
*SettingsApi* | [**settings_update_data_view_settings**](docs/SettingsApi.md#settings_update_data_view_settings) | **PUT** /{dataViewName}/Settings/DataView/{settingsPath} | Updates DataView settings at the given path
*SettingsApi* | [**settings_update_data_view_settings_root**](docs/SettingsApi.md#settings_update_data_view_settings_root) | **PUT** /{dataViewName}/Settings/DataView | Updates the complete DataView settings
*SharesApi* | [**shares_create_share**](docs/SharesApi.md#shares_create_share) | **POST** /{dataViewName}/Shares | Creates a new share from the given details, sharing from the logged in user.
*SharesApi* | [**shares_create_share_update**](docs/SharesApi.md#shares_create_share_update) | **POST** /{dataViewName}/Shares/{shareId}/Updates | Creates a new share update from the given details, sharing from the logged in user.
*SharesApi* | [**shares_delete_share**](docs/SharesApi.md#shares_delete_share) | **DELETE** /{dataViewName}/Shares/{shareId} | Deletes the specified share
*SharesApi* | [**shares_get_share**](docs/SharesApi.md#shares_get_share) | **GET** /{dataViewName}/Shares/{shareId} | Returns the details of a particular share
*SharesApi* | [**shares_get_share_update**](docs/SharesApi.md#shares_get_share_update) | **GET** /{dataViewName}/Shares/{shareId}/Updates/{shareUpdateId} | Returns a specific update that is associated with a particular share
*SharesApi* | [**shares_get_share_update_added_users**](docs/SharesApi.md#shares_get_share_update_added_users) | **GET** /{dataViewName}/Shares/{shareId}/Updates/{shareUpdateId}/AddedUsers | Returns the list of the users added to a share as part of a specific update
*SharesApi* | [**shares_get_share_update_removed_users**](docs/SharesApi.md#shares_get_share_update_removed_users) | **GET** /{dataViewName}/Shares/{shareId}/Updates/{shareUpdateId}/RemovedUsers | Returns the list of the users removed from a share as part of a specific update
*SharesApi* | [**shares_get_share_updates**](docs/SharesApi.md#shares_get_share_updates) | **GET** /{dataViewName}/Shares/{shareId}/Updates | Returns the updates that are associated with a particular share
*SharesApi* | [**shares_get_share_users**](docs/SharesApi.md#shares_get_share_users) | **GET** /{dataViewName}/Shares/{shareId}/Users | Returns the list of users that are associated with a particular share
*SharesApi* | [**shares_get_shares**](docs/SharesApi.md#shares_get_shares) | **GET** /{dataViewName}/Shares | Requires OrbitAdmin: Gets summary information about each share in the DataView.
*StaticResourcesApi* | [**static_resources_get_static_resource_categories**](docs/StaticResourcesApi.md#static_resources_get_static_resource_categories) | **GET** /{dataViewName}/StaticResources | Requires OrbitAdmin: Returns a list of categories of resource files for the given system
*StaticResourcesApi* | [**static_resources_get_static_resource_category**](docs/StaticResourcesApi.md#static_resources_get_static_resource_category) | **GET** /{dataViewName}/StaticResources/{resourceCategory} | Requires OrbitAdmin: Returns details of a resource category for a given category name and system
*StaticResourcesApi* | [**static_resources_get_static_resource_file**](docs/StaticResourcesApi.md#static_resources_get_static_resource_file) | **GET** /{dataViewName}/StaticResources/{resourceCategory}/Resources/{resourceName} | Returns a resource file (such as an image file) for the given category and system
*StaticResourcesApi* | [**static_resources_get_static_resource_file_details**](docs/StaticResourcesApi.md#static_resources_get_static_resource_file_details) | **GET** /{dataViewName}/StaticResources/{resourceCategory}/Resources/{resourceName}/Details | Requires OrbitAdmin: Returns the details of a resource file (such as an image file) for the given category and system
*StaticResourcesApi* | [**static_resources_get_static_resources_for_category**](docs/StaticResourcesApi.md#static_resources_get_static_resources_for_category) | **GET** /{dataViewName}/StaticResources/{resourceCategory}/Resources | Requires OrbitAdmin: Returns a list of details for the resource files (such as image files) in the given resource category and system
*TelemetryApi* | [**telemetry_create_telemetry_session**](docs/TelemetryApi.md#telemetry_create_telemetry_session) | **POST** /{dataViewName}/Telemetry/States/{stateId}/Sessions/{sessionId} | Creates a new telemetry session from the given details
*TelemetryApi* | [**telemetry_create_telemetry_state**](docs/TelemetryApi.md#telemetry_create_telemetry_state) | **POST** /{dataViewName}/Telemetry/States | Creates a new telemetry state from the given details
*TelemetryApi* | [**telemetry_get_telemetry_session**](docs/TelemetryApi.md#telemetry_get_telemetry_session) | **GET** /{dataViewName}/Telemetry/States/{stateId}/Sessions/{sessionId} | Returns the details of a particular telemetry session
*TelemetryApi* | [**telemetry_get_telemetry_state**](docs/TelemetryApi.md#telemetry_get_telemetry_state) | **GET** /{dataViewName}/Telemetry/States/{stateId} | Returns the details of a particular telemetry state
*TelemetryApi* | [**telemetry_get_telemetry_state_for_user**](docs/TelemetryApi.md#telemetry_get_telemetry_state_for_user) | **GET** /{dataViewName}/Telemetry/States/ForUser/{username} | Returns the details of a given user&#39;s telemetry state
*TelemetryApi* | [**telemetry_update_telemetry_session**](docs/TelemetryApi.md#telemetry_update_telemetry_session) | **POST** /{dataViewName}/Telemetry/States/{stateId}/Sessions/{sessionId}/Update | Update a particular telemetry session from the given details
*TelemetryApi* | [**telemetry_update_telemetry_state**](docs/TelemetryApi.md#telemetry_update_telemetry_state) | **POST** /{dataViewName}/Telemetry/States/{stateId}/Update | Updates a particular telemetry state from the given details
*TemporaryFilesApi* | [**temporary_files_get_temporary_file**](docs/TemporaryFilesApi.md#temporary_files_get_temporary_file) | **GET** /{dataViewName}/TemporaryFiles/{id} | Returns the contents of a temporary file with the given id
*TemporaryFilesApi* | [**temporary_files_get_temporary_file_part**](docs/TemporaryFilesApi.md#temporary_files_get_temporary_file_part) | **GET** /{dataViewName}/TemporaryFiles/{id}/{partNumber} | Returns the contents of a temporary file part with the given id and part number
*TemporaryFilesApi* | [**temporary_files_upsert_temporary_file**](docs/TemporaryFilesApi.md#temporary_files_upsert_temporary_file) | **PUT** /{dataViewName}/TemporaryFiles/{id} | Creates or updates a temporary file with the given id
*TemporaryFilesApi* | [**temporary_files_upsert_temporary_file_part**](docs/TemporaryFilesApi.md#temporary_files_upsert_temporary_file_part) | **PUT** /{dataViewName}/TemporaryFiles/{id}/{partNumber} | Creates or updates part of a temporary file with the given id and part number
*UserRegistrationRequestsApi* | [**user_registration_requests_confirm_registration_request**](docs/UserRegistrationRequestsApi.md#user_registration_requests_confirm_registration_request) | **POST** /{dataViewName}/UserRegistrationRequests/{token} | Confirms a given user registration request and creates the user
*UserRegistrationRequestsApi* | [**user_registration_requests_create_registration_request**](docs/UserRegistrationRequestsApi.md#user_registration_requests_create_registration_request) | **POST** /{dataViewName}/UserRegistrationRequests | Creates a new user registration requests, which will check the sign-up details and then issue a confirmation notification
*UserRegistrationRequestsApi* | [**user_registration_requests_get_registration_request**](docs/UserRegistrationRequestsApi.md#user_registration_requests_get_registration_request) | **GET** /{dataViewName}/UserRegistrationRequests/{token} | Requires OrbitAdmin: Returns details for a given user registration request
*UserRegistrationRequestsApi* | [**user_registration_requests_get_registration_requests**](docs/UserRegistrationRequestsApi.md#user_registration_requests_get_registration_requests) | **GET** /{dataViewName}/UserRegistrationRequests | Requires OrbitAdmin: Returns all the current user regisration requests in the system.
*UserResetPasswordRequestsApi* | [**user_reset_password_requests_confirm_reset_password_request**](docs/UserResetPasswordRequestsApi.md#user_reset_password_requests_confirm_reset_password_request) | **POST** /{dataViewName}/UserResetPasswordRequests/{token} | Confirms a given reset password request and changes the password
*UserResetPasswordRequestsApi* | [**user_reset_password_requests_create_reset_password_request**](docs/UserResetPasswordRequestsApi.md#user_reset_password_requests_create_reset_password_request) | **POST** /{dataViewName}/UserResetPasswordRequests | Creates a new reset password requests, which will check that the provided email address exists and then issue a confirmation notification
*UserResetPasswordRequestsApi* | [**user_reset_password_requests_get_reset_password_request**](docs/UserResetPasswordRequestsApi.md#user_reset_password_requests_get_reset_password_request) | **GET** /{dataViewName}/UserResetPasswordRequests/{token} | Requires OrbitAdmin: Returns details for a given reset password request
*UserResetPasswordRequestsApi* | [**user_reset_password_requests_get_reset_password_requests**](docs/UserResetPasswordRequestsApi.md#user_reset_password_requests_get_reset_password_requests) | **GET** /{dataViewName}/UserResetPasswordRequests | Requires OrbitAdmin: Returns all the current reset password requests in the system.
*UsersApi* | [**users_change_user_password**](docs/UsersApi.md#users_change_user_password) | **POST** /{dataViewName}/Users/{username}/ChangePassword | Change the password for the user with the given username
*UsersApi* | [**users_get_all_user_dashboards**](docs/UsersApi.md#users_get_all_user_dashboards) | **GET** /{dataViewName}/Users/{username}/Dashboards | EXPERIMENTAL: Gets a dashboard in the DataView.
*UsersApi* | [**users_get_previous_login_history**](docs/UsersApi.md#users_get_previous_login_history) | **GET** /{dataViewName}/Users/{username}/LoginHistory | Gets a list of users last login history
*UsersApi* | [**users_get_user_audience**](docs/UsersApi.md#users_get_user_audience) | **GET** /{dataViewName}/Users/{username}/Audiences/{audienceId} | Returns the details of a particular audience
*UsersApi* | [**users_get_user_audiences**](docs/UsersApi.md#users_get_user_audiences) | **GET** /{dataViewName}/Users/{username}/Audiences | Returns the list of audiences associated with the given user
*UsersApi* | [**users_get_user_collection**](docs/UsersApi.md#users_get_user_collection) | **GET** /{dataViewName}/Users/{username}/Collections/{collectionId} | Returns the details of a particular collection
*UsersApi* | [**users_get_user_collections**](docs/UsersApi.md#users_get_user_collections) | **GET** /{dataViewName}/Users/{username}/Collections | Returns the list of collections associated with the given user
*UsersApi* | [**users_get_user_configuration**](docs/UsersApi.md#users_get_user_configuration) | **GET** /{dataViewName}/Users/Configuration | Gets the user configuration
*UsersApi* | [**users_get_user_dashboard**](docs/UsersApi.md#users_get_user_dashboard) | **GET** /{dataViewName}/Users/{username}/Dashboards/{dashboardId} | EXPERIMENTAL: Gets a dashboard in the DataView.
*UsersApi* | [**users_get_user_details**](docs/UsersApi.md#users_get_user_details) | **GET** /{dataViewName}/Users/{username} | Returns details for the given username
*UsersApi* | [**users_get_user_details_list**](docs/UsersApi.md#users_get_user_details_list) | **GET** /{dataViewName}/Users | Returns all users in the system.
*UsersApi* | [**users_modify_user_audiences**](docs/UsersApi.md#users_modify_user_audiences) | **POST** /{dataViewName}/Users/{username}/Audiences/Modify | Updates one or more audiences
*UsersApi* | [**users_modify_user_collections**](docs/UsersApi.md#users_modify_user_collections) | **POST** /{dataViewName}/Users/{username}/Collections/Modify | Updates one or more collections
*UsersApi* | [**users_patch_user_collection**](docs/UsersApi.md#users_patch_user_collection) | **PATCH** /{dataViewName}/Users/{username}/Collections/{collectionId} | EXPERIMENTAL: Update some properties of a collection
*UsersApi* | [**users_update_user_collection**](docs/UsersApi.md#users_update_user_collection) | **PUT** /{dataViewName}/Users/{username}/Collections/{collectionId} | EXPERIMENTAL: Updates a collection
*UsersApi* | [**users_update_user_details**](docs/UsersApi.md#users_update_user_details) | **POST** /{dataViewName}/Users/{username} | Updates user details for the given username
*VisualisationsApi* | [**visualisations_cancel_visualisation_render_data_refresh_job**](docs/VisualisationsApi.md#visualisations_cancel_visualisation_render_data_refresh_job) | **DELETE** /{dataViewName}/Visualisations/{visualisationId}/RenderData/RefreshJobs/{refreshRequestId} | Cancel a job refreshing the render data for a particular visualisation
*VisualisationsApi* | [**visualisations_create_visualisation_render_data_refresh_job**](docs/VisualisationsApi.md#visualisations_create_visualisation_render_data_refresh_job) | **POST** /{dataViewName}/Visualisations/{visualisationId}/RenderData/RefreshJobs | Creates a job to refresh the render data for a particular visualisation
*VisualisationsApi* | [**visualisations_get_visualisation**](docs/VisualisationsApi.md#visualisations_get_visualisation) | **GET** /{dataViewName}/Visualisations/{visualisationId} | Returns the details of a particular visualisation
*VisualisationsApi* | [**visualisations_get_visualisation_render_data**](docs/VisualisationsApi.md#visualisations_get_visualisation_render_data) | **GET** /{dataViewName}/Visualisations/{visualisationId}/RenderData | Returns the render data for a particular visualisation
*VisualisationsApi* | [**visualisations_get_visualisation_render_data_refresh_job**](docs/VisualisationsApi.md#visualisations_get_visualisation_render_data_refresh_job) | **GET** /{dataViewName}/Visualisations/{visualisationId}/RenderData/RefreshJobs/{refreshRequestId} | Returns the details of a job to refresh the render data for a particular visualisation
*VisualisationsApi* | [**visualisations_perform_visualisation_render_data_refresh_synchronously**](docs/VisualisationsApi.md#visualisations_perform_visualisation_render_data_refresh_synchronously) | **POST** /{dataViewName}/Visualisations/{visualisationId}/RenderData/RefreshSync | Performs a synchronous refresh of the render data for a particular visualisation


## Documentation For Models

 - [AbstractRenderSpec](docs/AbstractRenderSpec.md)
 - [AgeRule](docs/AgeRule.md)
 - [AudienceCalculateJobDetail](docs/AudienceCalculateJobDetail.md)
 - [AudienceCheckDetail](docs/AudienceCheckDetail.md)
 - [AudienceCheckJobDetail](docs/AudienceCheckJobDetail.md)
 - [AudienceDataLicensingInfoJobDetail](docs/AudienceDataLicensingInfoJobDetail.md)
 - [AudienceDetail](docs/AudienceDetail.md)
 - [AudienceExportDetail](docs/AudienceExportDetail.md)
 - [AudienceExportJobDetail](docs/AudienceExportJobDetail.md)
 - [AudienceHitDetail](docs/AudienceHitDetail.md)
 - [AudienceHitSummary](docs/AudienceHitSummary.md)
 - [AudienceQueryResult](docs/AudienceQueryResult.md)
 - [AudienceResultDetail](docs/AudienceResultDetail.md)
 - [AudienceResultSummary](docs/AudienceResultSummary.md)
 - [AudienceSummary](docs/AudienceSummary.md)
 - [AudienceUpdateDetail](docs/AudienceUpdateDetail.md)
 - [AudienceUpdateSummary](docs/AudienceUpdateSummary.md)
 - [Breakpoint](docs/Breakpoint.md)
 - [BuildDefinition](docs/BuildDefinition.md)
 - [BuildJobDetail](docs/BuildJobDetail.md)
 - [BuildResult](docs/BuildResult.md)
 - [CalculateAudienceDetails](docs/CalculateAudienceDetails.md)
 - [Capabilities](docs/Capabilities.md)
 - [ChangePasswordDetails](docs/ChangePasswordDetails.md)
 - [ChannelDetail](docs/ChannelDetail.md)
 - [ChannelStatistics](docs/ChannelStatistics.md)
 - [ChannelSummary](docs/ChannelSummary.md)
 - [CheckAudienceDetails](docs/CheckAudienceDetails.md)
 - [CheckCompositionDefinition](docs/CheckCompositionDefinition.md)
 - [CheckDimensionResult](docs/CheckDimensionResult.md)
 - [Clause](docs/Clause.md)
 - [CollectionDetail](docs/CollectionDetail.md)
 - [CollectionHitDetail](docs/CollectionHitDetail.md)
 - [CollectionHitSummary](docs/CollectionHitSummary.md)
 - [CollectionPartDetail](docs/CollectionPartDetail.md)
 - [CollectionPartSummary](docs/CollectionPartSummary.md)
 - [CollectionSummary](docs/CollectionSummary.md)
 - [Column](docs/Column.md)
 - [CommunicationStatistics](docs/CommunicationStatistics.md)
 - [CompositionDetail](docs/CompositionDetail.md)
 - [CompositionSummary](docs/CompositionSummary.md)
 - [ConfirmResetPasswordRequest](docs/ConfirmResetPasswordRequest.md)
 - [Count](docs/Count.md)
 - [CreateAudienceCompositionDetail](docs/CreateAudienceCompositionDetail.md)
 - [CreateAudienceDetail](docs/CreateAudienceDetail.md)
 - [CreateAudienceHitDetails](docs/CreateAudienceHitDetails.md)
 - [CreateAudienceUpdate](docs/CreateAudienceUpdate.md)
 - [CreateCollectionHitDetails](docs/CreateCollectionHitDetails.md)
 - [CreateResetPasswordRequest](docs/CreateResetPasswordRequest.md)
 - [CreateSessionParameters](docs/CreateSessionParameters.md)
 - [CreateShareDetail](docs/CreateShareDetail.md)
 - [CreateShareUpdate](docs/CreateShareUpdate.md)
 - [CreateTelemetrySessionDetails](docs/CreateTelemetrySessionDetails.md)
 - [CreateTelemetryStateDetails](docs/CreateTelemetryStateDetails.md)
 - [CreateUserRegistrationRequest](docs/CreateUserRegistrationRequest.md)
 - [CreatedShareUpdateDetail](docs/CreatedShareUpdateDetail.md)
 - [Criteria](docs/Criteria.md)
 - [Cube](docs/Cube.md)
 - [CubeResult](docs/CubeResult.md)
 - [DashboardContentItem](docs/DashboardContentItem.md)
 - [DashboardContentItemDetail](docs/DashboardContentItemDetail.md)
 - [DashboardDetail](docs/DashboardDetail.md)
 - [DashboardItem](docs/DashboardItem.md)
 - [DashboardItemData](docs/DashboardItemData.md)
 - [DashboardItemDataResult](docs/DashboardItemDataResult.md)
 - [DashboardSummary](docs/DashboardSummary.md)
 - [DataLicensingDetail](docs/DataLicensingDetail.md)
 - [DataLicensingSystemDetail](docs/DataLicensingSystemDetail.md)
 - [DataLicensingSystemSummary](docs/DataLicensingSystemSummary.md)
 - [DataPurchaseDetail](docs/DataPurchaseDetail.md)
 - [DataPurchaseJobDetail](docs/DataPurchaseJobDetail.md)
 - [DataViewDetails](docs/DataViewDetails.md)
 - [DataViewSummary](docs/DataViewSummary.md)
 - [DateRule](docs/DateRule.md)
 - [DiagramMetadata](docs/DiagramMetadata.md)
 - [Dimension](docs/Dimension.md)
 - [DimensionBanding](docs/DimensionBanding.md)
 - [DimensionResult](docs/DimensionResult.md)
 - [ElementDetail](docs/ElementDetail.md)
 - [ElementKey](docs/ElementKey.md)
 - [ElementStatus](docs/ElementStatus.md)
 - [ElementSummary](docs/ElementSummary.md)
 - [EmailRequirements](docs/EmailRequirements.md)
 - [EndpointDetails](docs/EndpointDetails.md)
 - [ErrorMessage](docs/ErrorMessage.md)
 - [ErrorMessageParameter](docs/ErrorMessageParameter.md)
 - [Export](docs/Export.md)
 - [ExportAudienceDetails](docs/ExportAudienceDetails.md)
 - [ExportCompositionDefinition](docs/ExportCompositionDefinition.md)
 - [ExportResult](docs/ExportResult.md)
 - [ExportSystemDetail](docs/ExportSystemDetail.md)
 - [ExportSystemSummary](docs/ExportSystemSummary.md)
 - [Expression](docs/Expression.md)
 - [FastStatsSystemDetail](docs/FastStatsSystemDetail.md)
 - [FastStatsSystemItem](docs/FastStatsSystemItem.md)
 - [FastStatsSystemSummary](docs/FastStatsSystemSummary.md)
 - [FileEntry](docs/FileEntry.md)
 - [FilePermission](docs/FilePermission.md)
 - [FileStream](docs/FileStream.md)
 - [FileSystemSummary](docs/FileSystemSummary.md)
 - [Folder](docs/Folder.md)
 - [FolderStructureNode](docs/FolderStructureNode.md)
 - [Fraction](docs/Fraction.md)
 - [GridItem](docs/GridItem.md)
 - [GroupSummary](docs/GroupSummary.md)
 - [InvalidToShareUserDisplayDetails](docs/InvalidToShareUserDisplayDetails.md)
 - [JobDetail](docs/JobDetail.md)
 - [JobSummary](docs/JobSummary.md)
 - [LanguageDetails](docs/LanguageDetails.md)
 - [LastRunDetails](docs/LastRunDetails.md)
 - [Licence](docs/Licence.md)
 - [LicensingInfo](docs/LicensingInfo.md)
 - [LicensingSet](docs/LicensingSet.md)
 - [LicensingSetsJobDetail](docs/LicensingSetsJobDetail.md)
 - [Limits](docs/Limits.md)
 - [Link](docs/Link.md)
 - [ListRule](docs/ListRule.md)
 - [Logic](docs/Logic.md)
 - [Measure](docs/Measure.md)
 - [MeasureResult](docs/MeasureResult.md)
 - [Message](docs/Message.md)
 - [ModelProperty](docs/ModelProperty.md)
 - [ModifyItemsModifyUserAudience](docs/ModifyItemsModifyUserAudience.md)
 - [ModifyItemsModifyUserCollection](docs/ModifyItemsModifyUserCollection.md)
 - [ModifyUserAudience](docs/ModifyUserAudience.md)
 - [ModifyUserAudienceDetail](docs/ModifyUserAudienceDetail.md)
 - [ModifyUserAudienceDetailResults](docs/ModifyUserAudienceDetailResults.md)
 - [ModifyUserCollection](docs/ModifyUserCollection.md)
 - [ModifyUserCollectionDetail](docs/ModifyUserCollectionDetail.md)
 - [ModifyUserCollectionDetailResults](docs/ModifyUserCollectionDetailResults.md)
 - [NPer](docs/NPer.md)
 - [NumericVariableInfo](docs/NumericVariableInfo.md)
 - [Operation](docs/Operation.md)
 - [OrbitFeaturePermission](docs/OrbitFeaturePermission.md)
 - [Output](docs/Output.md)
 - [PagedResultsAudienceHitSummary](docs/PagedResultsAudienceHitSummary.md)
 - [PagedResultsAudienceResultSummary](docs/PagedResultsAudienceResultSummary.md)
 - [PagedResultsAudienceSummary](docs/PagedResultsAudienceSummary.md)
 - [PagedResultsAudienceUpdateSummary](docs/PagedResultsAudienceUpdateSummary.md)
 - [PagedResultsChannelSummary](docs/PagedResultsChannelSummary.md)
 - [PagedResultsCollectionHitSummary](docs/PagedResultsCollectionHitSummary.md)
 - [PagedResultsCollectionPartSummary](docs/PagedResultsCollectionPartSummary.md)
 - [PagedResultsCollectionSummary](docs/PagedResultsCollectionSummary.md)
 - [PagedResultsCompositionSummary](docs/PagedResultsCompositionSummary.md)
 - [PagedResultsDashboardSummary](docs/PagedResultsDashboardSummary.md)
 - [PagedResultsDataLicensingSystemSummary](docs/PagedResultsDataLicensingSystemSummary.md)
 - [PagedResultsDataViewSummary](docs/PagedResultsDataViewSummary.md)
 - [PagedResultsElementStatus](docs/PagedResultsElementStatus.md)
 - [PagedResultsElementSummary](docs/PagedResultsElementSummary.md)
 - [PagedResultsEndpointDetails](docs/PagedResultsEndpointDetails.md)
 - [PagedResultsExportSystemSummary](docs/PagedResultsExportSystemSummary.md)
 - [PagedResultsFastStatsSystemItem](docs/PagedResultsFastStatsSystemItem.md)
 - [PagedResultsFastStatsSystemSummary](docs/PagedResultsFastStatsSystemSummary.md)
 - [PagedResultsFileEntry](docs/PagedResultsFileEntry.md)
 - [PagedResultsFileSystemSummary](docs/PagedResultsFileSystemSummary.md)
 - [PagedResultsFolderStructureNode](docs/PagedResultsFolderStructureNode.md)
 - [PagedResultsGroupSummary](docs/PagedResultsGroupSummary.md)
 - [PagedResultsJobSummary](docs/PagedResultsJobSummary.md)
 - [PagedResultsLicensingSet](docs/PagedResultsLicensingSet.md)
 - [PagedResultsModifyUserAudienceDetailResults](docs/PagedResultsModifyUserAudienceDetailResults.md)
 - [PagedResultsModifyUserCollectionDetailResults](docs/PagedResultsModifyUserCollectionDetailResults.md)
 - [PagedResultsPeopleStageSystemSummary](docs/PagedResultsPeopleStageSystemSummary.md)
 - [PagedResultsPermission](docs/PagedResultsPermission.md)
 - [PagedResultsResourceCategorySummary](docs/PagedResultsResourceCategorySummary.md)
 - [PagedResultsResourceSummary](docs/PagedResultsResourceSummary.md)
 - [PagedResultsSessionAndUserDetails](docs/PagedResultsSessionAndUserDetails.md)
 - [PagedResultsShareSummary](docs/PagedResultsShareSummary.md)
 - [PagedResultsShareUpdate](docs/PagedResultsShareUpdate.md)
 - [PagedResultsTable](docs/PagedResultsTable.md)
 - [PagedResultsUserAudienceSummary](docs/PagedResultsUserAudienceSummary.md)
 - [PagedResultsUserCollectionSummary](docs/PagedResultsUserCollectionSummary.md)
 - [PagedResultsUserDisplayDetails](docs/PagedResultsUserDisplayDetails.md)
 - [PagedResultsUserLogin](docs/PagedResultsUserLogin.md)
 - [PagedResultsUserRegistrationRequestSummary](docs/PagedResultsUserRegistrationRequestSummary.md)
 - [PagedResultsUserSummary](docs/PagedResultsUserSummary.md)
 - [PagedResultsVarCode](docs/PagedResultsVarCode.md)
 - [PagedResultsVariable](docs/PagedResultsVariable.md)
 - [PasswordRequirements](docs/PasswordRequirements.md)
 - [PeopleStageSystemDetail](docs/PeopleStageSystemDetail.md)
 - [PeopleStageSystemSummary](docs/PeopleStageSystemSummary.md)
 - [PerChannelStatistics](docs/PerChannelStatistics.md)
 - [PerResponseTypePerChannelStatistics](docs/PerResponseTypePerChannelStatistics.md)
 - [PerResponseTypeStatistics](docs/PerResponseTypeStatistics.md)
 - [Permission](docs/Permission.md)
 - [ProcessDetails](docs/ProcessDetails.md)
 - [ProcessingTimeStatistics](docs/ProcessingTimeStatistics.md)
 - [ProcessingTimeStatisticsDetails](docs/ProcessingTimeStatisticsDetails.md)
 - [PurchaseInfo](docs/PurchaseInfo.md)
 - [Query](docs/Query.md)
 - [QueryDetailCount](docs/QueryDetailCount.md)
 - [QueryDetails](docs/QueryDetails.md)
 - [QueryFile](docs/QueryFile.md)
 - [QueryResult](docs/QueryResult.md)
 - [RFV](docs/RFV.md)
 - [RFVFrequency](docs/RFVFrequency.md)
 - [RFVRecency](docs/RFVRecency.md)
 - [RFVValue](docs/RFVValue.md)
 - [RangeStatistics](docs/RangeStatistics.md)
 - [RecordSet](docs/RecordSet.md)
 - [RenderDataRefreshJobDetail](docs/RenderDataRefreshJobDetail.md)
 - [ResetPasswordRequestDetail](docs/ResetPasswordRequestDetail.md)
 - [ResourceCategoryDetails](docs/ResourceCategoryDetails.md)
 - [ResourceCategorySummary](docs/ResourceCategorySummary.md)
 - [ResourceDetails](docs/ResourceDetails.md)
 - [ResourceSummary](docs/ResourceSummary.md)
 - [ResponseStatistics](docs/ResponseStatistics.md)
 - [ResponseStatisticsPerResponseTypeStatisticsMap](docs/ResponseStatisticsPerResponseTypeStatisticsMap.md)
 - [Row](docs/Row.md)
 - [Rule](docs/Rule.md)
 - [SafeFileHandle](docs/SafeFileHandle.md)
 - [SaveQueryFile](docs/SaveQueryFile.md)
 - [Selection](docs/Selection.md)
 - [SelectionModifiers](docs/SelectionModifiers.md)
 - [SelectorVariableInfo](docs/SelectorVariableInfo.md)
 - [ServerMessage](docs/ServerMessage.md)
 - [SessionAndUserDetails](docs/SessionAndUserDetails.md)
 - [SessionDetails](docs/SessionDetails.md)
 - [ShareDetail](docs/ShareDetail.md)
 - [ShareSummary](docs/ShareSummary.md)
 - [ShareUpdate](docs/ShareUpdate.md)
 - [Size](docs/Size.md)
 - [SubSelection](docs/SubSelection.md)
 - [SystemLookup](docs/SystemLookup.md)
 - [Table](docs/Table.md)
 - [TableItem](docs/TableItem.md)
 - [TelemetrySession](docs/TelemetrySession.md)
 - [TelemetryState](docs/TelemetryState.md)
 - [TemporaryFile](docs/TemporaryFile.md)
 - [TemporaryFilePart](docs/TemporaryFilePart.md)
 - [TextVariableInfo](docs/TextVariableInfo.md)
 - [TimeRule](docs/TimeRule.md)
 - [TokenLoginDetails](docs/TokenLoginDetails.md)
 - [TopN](docs/TopN.md)
 - [TransferAudienceOwnershipDetails](docs/TransferAudienceOwnershipDetails.md)
 - [TransferCollectionOwnershipDetails](docs/TransferCollectionOwnershipDetails.md)
 - [UpdateTelemetrySessionDetails](docs/UpdateTelemetrySessionDetails.md)
 - [UpdateTelemetryStateDetails](docs/UpdateTelemetryStateDetails.md)
 - [UpdateUserDetails](docs/UpdateUserDetails.md)
 - [UpsertCollectionDetail](docs/UpsertCollectionDetail.md)
 - [UpsertUserCollectionDetail](docs/UpsertUserCollectionDetail.md)
 - [UserAudienceDetail](docs/UserAudienceDetail.md)
 - [UserAudienceSummary](docs/UserAudienceSummary.md)
 - [UserCollectionDetail](docs/UserCollectionDetail.md)
 - [UserCollectionSummary](docs/UserCollectionSummary.md)
 - [UserConfigurationDetails](docs/UserConfigurationDetails.md)
 - [UserDetail](docs/UserDetail.md)
 - [UserDisplayDetails](docs/UserDisplayDetails.md)
 - [UserLogin](docs/UserLogin.md)
 - [UserRegistrationRequestDetail](docs/UserRegistrationRequestDetail.md)
 - [UserRegistrationRequestSummary](docs/UserRegistrationRequestSummary.md)
 - [UserSummary](docs/UserSummary.md)
 - [ValueRule](docs/ValueRule.md)
 - [VarCode](docs/VarCode.md)
 - [Variable](docs/Variable.md)
 - [VariableLookup](docs/VariableLookup.md)
 - [VersionDetails](docs/VersionDetails.md)
 - [VisualisationDetail](docs/VisualisationDetail.md)


## Documentation For Authorization


## faststats_auth

- **Type**: API key
- **API key parameter name**: Authorization
- **Location**: HTTP header


## Author

Apteco Ltd

support@apteco.com
