"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
const metric_util_1 = require("./private/metric-util");
const object_1 = require("./private/object");
const rendering_1 = require("./private/rendering");
const statistic_1 = require("./private/statistic");
/**
 * Comparison operator for evaluating alarms
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    /**
     * Specified statistic is greater than or equal to the threshold
     */
    ComparisonOperator["GREATER_THAN_OR_EQUAL_TO_THRESHOLD"] = "GreaterThanOrEqualToThreshold";
    /**
     * Specified statistic is strictly greater than the threshold
     */
    ComparisonOperator["GREATER_THAN_THRESHOLD"] = "GreaterThanThreshold";
    /**
     * Specified statistic is strictly less than the threshold
     */
    ComparisonOperator["LESS_THAN_THRESHOLD"] = "LessThanThreshold";
    /**
     * Specified statistic is less than or equal to the threshold.
     */
    ComparisonOperator["LESS_THAN_OR_EQUAL_TO_THRESHOLD"] = "LessThanOrEqualToThreshold";
    /**
     * Specified statistic is lower than or greater than the anomaly model band.
     * Used only for alarms based on anomaly detection models
     */
    ComparisonOperator["LESS_THAN_LOWER_OR_GREATER_THAN_UPPER_THRESHOLD"] = "LessThanLowerOrGreaterThanUpperThreshold";
})(ComparisonOperator = exports.ComparisonOperator || (exports.ComparisonOperator = {}));
const OPERATOR_SYMBOLS = {
    GreaterThanOrEqualToThreshold: '>=',
    GreaterThanThreshold: '>',
    LessThanThreshold: '<',
    LessThanOrEqualToThreshold: '>=',
};
/**
 * Specify how missing data points are treated during alarm evaluation
 */
var TreatMissingData;
(function (TreatMissingData) {
    /**
     * Missing data points are treated as breaching the threshold
     */
    TreatMissingData["BREACHING"] = "breaching";
    /**
     * Missing data points are treated as being within the threshold
     */
    TreatMissingData["NOT_BREACHING"] = "notBreaching";
    /**
     * The current alarm state is maintained
     */
    TreatMissingData["IGNORE"] = "ignore";
    /**
     * The alarm does not consider missing data points when evaluating whether to change state
     */
    TreatMissingData["MISSING"] = "missing";
})(TreatMissingData = exports.TreatMissingData || (exports.TreatMissingData = {}));
/**
 * An alarm on a CloudWatch metric
 */
class Alarm extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.alarmName,
        });
        const comparisonOperator = props.comparisonOperator || ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD;
        // Render metric, process potential overrides from the alarm
        // (It would be preferable if the statistic etc. was worked into the metric,
        // but hey we're allowing overrides...)
        const metricProps = this.renderMetric(props.metric);
        if (props.period) {
            metricProps.period = props.period.toSeconds();
        }
        if (props.statistic) {
            // Will overwrite both fields if present
            Object.assign(metricProps, {
                statistic: renderIfSimpleStatistic(props.statistic),
                extendedStatistic: renderIfExtendedStatistic(props.statistic),
            });
        }
        const alarm = new cloudwatch_generated_1.CfnAlarm(this, 'Resource', {
            // Meta
            alarmDescription: props.alarmDescription,
            alarmName: this.physicalName,
            // Evaluation
            comparisonOperator,
            threshold: props.threshold,
            datapointsToAlarm: props.datapointsToAlarm,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            evaluationPeriods: props.evaluationPeriods,
            treatMissingData: props.treatMissingData,
            // Actions
            actionsEnabled: props.actionsEnabled,
            alarmActions: core_1.Lazy.listValue({ produce: () => this.alarmActionArns }),
            insufficientDataActions: core_1.Lazy.listValue({ produce: (() => this.insufficientDataActionArns) }),
            okActions: core_1.Lazy.listValue({ produce: () => this.okActionArns }),
            // Metric
            ...metricProps,
        });
        this.alarmArn = this.getResourceArnAttribute(alarm.attrArn, {
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.alarmName = this.getResourceNameAttribute(alarm.ref);
        this.metric = props.metric;
        const datapoints = props.datapointsToAlarm || props.evaluationPeriods;
        this.annotation = {
            // tslint:disable-next-line:max-line-length
            label: `${this.metric} ${OPERATOR_SYMBOLS[comparisonOperator]} ${props.threshold} for ${datapoints} datapoints within ${describePeriod(props.evaluationPeriods * metric_util_1.metricPeriod(props.metric).toSeconds())}`,
            value: props.threshold,
        };
    }
    /**
     * Import an existing CloudWatch alarm provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name
     * @param alarmArn Alarm ARN (i.e. arn:aws:cloudwatch:<region>:<account-id>:alarm:Foo)
     */
    static fromAlarmArn(scope, id, alarmArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.alarmArn = alarmArn;
                this.alarmName = core_1.Stack.of(scope).parseArn(alarmArn, ':').resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Trigger this action if the alarm fires
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     */
    addAlarmAction(...actions) {
        if (this.alarmActionArns === undefined) {
            this.alarmActionArns = [];
        }
        this.alarmActionArns.push(...actions.map(a => a.bind(this, this).alarmActionArn));
    }
    /**
     * Trigger this action if there is insufficient data to evaluate the alarm
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     */
    addInsufficientDataAction(...actions) {
        if (this.insufficientDataActionArns === undefined) {
            this.insufficientDataActionArns = [];
        }
        this.insufficientDataActionArns.push(...actions.map(a => a.bind(this, this).alarmActionArn));
    }
    /**
     * Trigger this action if the alarm returns from breaching state into ok state
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     */
    addOkAction(...actions) {
        if (this.okActionArns === undefined) {
            this.okActionArns = [];
        }
        this.okActionArns.push(...actions.map(a => a.bind(this, this).alarmActionArn));
    }
    /**
     * Turn this alarm into a horizontal annotation
     *
     * This is useful if you want to represent an Alarm in a non-AlarmWidget.
     * An `AlarmWidget` can directly show an alarm, but it can only show a
     * single alarm and no other metrics. Instead, you can convert the alarm to
     * a HorizontalAnnotation and add it as an annotation to another graph.
     *
     * This might be useful if:
     *
     * - You want to show multiple alarms inside a single graph, for example if
     *   you have both a "small margin/long period" alarm as well as a
     *   "large margin/short period" alarm.
     *
     * - You want to show an Alarm line in a graph with multiple metrics in it.
     */
    toAnnotation() {
        return this.annotation;
    }
    renderMetric(metric) {
        const self = this;
        return metric_util_1.dispatchMetric(metric, {
            withStat(st) {
                var _a;
                self.validateMetricStat(st, metric);
                return object_1.dropUndefined({
                    dimensions: st.dimensions,
                    namespace: st.namespace,
                    metricName: st.metricName,
                    period: (_a = st.period) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                    statistic: renderIfSimpleStatistic(st.statistic),
                    extendedStatistic: renderIfExtendedStatistic(st.statistic),
                    unit: st.unitFilter,
                });
            },
            withExpression() {
                // Expand the math expression metric into a set
                const mset = new rendering_1.MetricSet();
                mset.addTopLevel(true, metric);
                let eid = 0;
                function uniqueMetricId() {
                    return `expr_${++eid}`;
                }
                return {
                    metrics: mset.entries.map(entry => metric_util_1.dispatchMetric(entry.metric, {
                        withStat(stat, conf) {
                            var _a;
                            self.validateMetricStat(stat, entry.metric);
                            return {
                                metricStat: {
                                    metric: {
                                        metricName: stat.metricName,
                                        namespace: stat.namespace,
                                        dimensions: stat.dimensions,
                                    },
                                    period: stat.period.toSeconds(),
                                    stat: stat.statistic,
                                    unit: stat.unitFilter,
                                },
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                        withExpression(expr, conf) {
                            var _a;
                            return {
                                expression: expr.expression,
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                period: mathExprHasSubmetrics(expr) ? undefined : expr.period,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                    })),
                };
            },
        });
    }
    /**
     * Validate that if a region and account are in the given stat config, they match the Alarm
     */
    validateMetricStat(stat, metric) {
        const stack = core_1.Stack.of(this);
        if (definitelyDifferent(stat.region, stack.region)) {
            throw new Error(`Cannot create an Alarm in region '${stack.region}' based on metric '${metric}' in '${stat.region}'`);
        }
        if (definitelyDifferent(stat.account, stack.account)) {
            throw new Error(`Cannot create an Alarm in account '${stack.account}' based on metric '${metric}' in '${stat.account}'`);
        }
    }
}
exports.Alarm = Alarm;
function definitelyDifferent(x, y) {
    return x && !core_1.Token.isUnresolved(y) && x !== y;
}
/**
 * Return a human readable string for this period
 *
 * We know the seconds are always one of a handful of allowed values.
 */
function describePeriod(seconds) {
    if (seconds === 60) {
        return '1 minute';
    }
    if (seconds === 1) {
        return '1 second';
    }
    if (seconds > 60) {
        return (seconds / 60) + ' minutes';
    }
    return seconds + ' seconds';
}
function renderIfSimpleStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    return undefined;
}
function renderIfExtendedStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'percentile') {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return statistic.toLowerCase();
    }
    return undefined;
}
function mathExprHasSubmetrics(expr) {
    return Object.keys(expr.usingMetrics).length > 0;
}
//# sourceMappingURL=data:application/json;base64,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