from copy import copy
from functools import wraps
from time import time
from typing import Optional, List, Any, Iterable, Dict, Sequence, Union
import logging

from .core.values.value_mapping import get_map_helper
from .core.kb_sync.kb_iterator_config import KBIteratorConfig
from .core.kb_sync.object_time_interval import ObjectTimeInterval
from .tdm_builder.tdm_builder import AbstractTdmBuilder
from .providers.gql_providers import AbstractGQLClient
from .schema.api_schema import Query, Mutation, StoryPagination, ConceptLinkPagination, ConceptFactPagination, \
    ConceptPropertyPagination, SortDirection, ConceptSorting, DocumentSorting, DocumentGrouping, TimestampInterval, \
    PerformSynchronously
from .schema.api_schema import \
    ValueInput, IntValueInput, StringValueInput, DateTimeValue, StringLocaleValue, StringValue, LinkValue, \
    DoubleValue, IntValue, ConceptPropertyFilterSettings, ConceptLinkFilterSettings, ExtraSettings, Story, Document,\
    DocumentFilterSettings, Concept, ConceptLink, ConceptPagination, ConceptFilterSettings, PropertyFilterSettings, \
    StringFilter, ConceptType, ConceptTypeFilterSettings, ConceptTypePagination, ConceptPropertyType, \
    ConceptPropertyTypePagination, ConceptPropertyTypeFilterSettings, ConceptLinkType, ConceptLinkTypePagination, \
    ConceptLinkTypeFilterSettings, ConceptPropertyValueType, ConceptPropertyValueTypePagination, \
    ConceptPropertyValueTypeFilterSettings, ConceptPropertyValueTypeUpdateInput, ConceptProperty, \
    ConceptMutationInput, ConceptUpdateInput, ConceptPropertyUpdateInput, ConceptPropertyCreateInput, \
    ConceptLinkPropertyInput, ConceptLinkCreationMutationInput, ComponentValueInput, CompositeConcept, \
    CompositeConceptWidgetRowPagination, DocumentLinkFilterSetting, ConceptFact, \
    ConceptLinkPropertyTypeCreationInput, ConceptPropertyTypeCreationInput, State, \
    CompositePropertyValueTemplate, CompositePropertyValueType, CompositeValue, DateTimeValueInput, \
    LinkValueInput, DoubleValueInput, StringLocaleValueInput, CompositePropertyTypeFilterSettings, \
    ConceptLinkPropertyTypeUpdateInput
from .schema.crawlers_api_schema import Query as CrQuery
from .schema.crawlers_api_schema import Crawler, CrawlerPagination
from .schema import utils_api_schema as uas
from .schema import tcontroller_api_schema as tc
from sgqlc.operation import Operation, Fragment


logger = logging.getLogger(__name__)


def check_utils_gql_client(f):
    @wraps(f)
    def wrapper(self: 'TalismanAPIAdapter', *args, **kwargs):
        if self._utils_gql_client is None:
            raise Exception('Utils methods cannot be used because the corresponding gql_client is not specified.')
        return f(self, *args, **kwargs)

    return wrapper


class TalismanAPIAdapter:
    def __init__(
        self, gql_client: Optional[AbstractGQLClient], types: Dict, tdm_builder: AbstractTdmBuilder = None,
        utils_gql_client: Optional[AbstractGQLClient] = None, kb_iterator_config: KBIteratorConfig = None,
        limit: int = 100, perform_synchronously: bool = True
    ) -> None:
        self._gql_client = gql_client
        self._utils_gql_client = utils_gql_client
        self._types = types
        self._cache = {}
        self._limit = limit
        self._perform_synchronously = perform_synchronously

        self.document_fields_truncated = ('id', 'external_url')
        self.document_fields = (
            'id', 'title', 'external_url', 'publication_author', 'publication_date', 'internal_url', 'markers',
            'system_registration_date', 'system_update_date', 'notes', 'access_level', 'trust_level'
        )
        self.document_fields_extended = (
            'id', 'title', 'publication_author', 'publication_date', 'external_url',
            'markers', 'notes', 'access_level', 'trust_level'
        )

        self.document_text_fields_truncated = ('text',)
        self.document_text_fields = ('node_id', 'text')
        self.document_text_metadata_fields = ('paragraph_type',)

        self.document_platform_fields = ('id', 'name')
        self.document_account_fields = ('id', 'name')

        self.user_fields = ('id', 'name')

        self.concept_fields = (
            'id', 'name', 'notes', 'metric', 'markers', 'system_registration_date', 'system_update_date'
        )
        self.concept_type_fields = ('id', 'name')

        self.concept_property_fields = ('is_main', 'id', 'system_registration_date')
        self.concept_property_type_fields_truncated = ('id',)
        self.concept_property_type_fields = ('id', 'name')
        self.cpvt_fields_truncated = ('id', 'name', 'value_type')
        self.cpvt_fields = ('id', 'name', 'value_type', 'value_restriction', 'pretrained_nercmodels')

        self.concept_link_fields = ('id', 'notes')
        self.concept_link_concept_from_fields = ('id',)
        self.concept_link_concept_to_fields = ('id',)
        self.concept_link_type_fields = ('id', 'name', 'is_directed', 'is_hierarchical')
        self.concept_link_type_fields_truncated = ('id', 'name', 'is_directed')

        self.concept_fact_fields = ('id',)

        self.composite_concept_widget_type = ('id', 'name')
        self.composite_concept_widget_type_columns_info = ('name',)

        self.date_time_value_date_fields = ('year', 'month', 'day')
        self.date_time_value_time_fields = ('hour', 'minute', 'second')

        self.pipeline_config_fields = ('id', 'description')

        self.pipeline_topic_fields = ('topic', 'description', 'stopped', 'metrics', 'pipeline')
        self.pipeline_metrics_fields = ('duplicate', 'failed', 'messages', 'ok')

        self.tdm_builder = tdm_builder

        if kb_iterator_config:
            self.kb_iterator_config = kb_iterator_config
        else:
            self.kb_iterator_config = KBIteratorConfig(1000, 1609448400)  # Fri Jan 01 2021 00:00:00 GMT+0300

    def get_tdm_builder(self) -> AbstractTdmBuilder:
        return self.tdm_builder

    @property
    def limit(self):
        return self._limit

    @limit.setter
    def limit(self, new_limit: int):
        self._limit = new_limit

    @property
    def perform_synchronously(self):
        return self._perform_synchronously

    @perform_synchronously.setter
    def perform_synchronously(self, new_perform_synchronously: bool):
        self._perform_synchronously = new_perform_synchronously

    def get_take_value(self, take: Optional[int]) -> int:
        return self.limit if take is None else take

    def get_perform_synchronously_value(self, perform_synchronously: Optional[bool]) -> bool:
        return self.perform_synchronously if perform_synchronously is None else perform_synchronously

    def _configure_property_value_type_fields(self, graphql_value, truncated: bool = True):
        conpvt_frag: Fragment = Fragment(ConceptPropertyValueType, 'ConceptPropertyValueType')
        for f in self.cpvt_fields_truncated if truncated else self.cpvt_fields:
            conpvt_frag.__getattr__(f)()

        compvt_frag = Fragment(CompositePropertyValueTemplate, 'CompositePropertyValueTemplate')
        compvt_frag.__fields__('id', 'name')
        compvt_frag.component_value_types()

        graphql_value.__fragment__(conpvt_frag)
        graphql_value.__fragment__(compvt_frag)

    def _configure_output_value_fields(self, graphql_value):
        dtv_frag = Fragment(DateTimeValue, 'DateTimeFull')
        dtv_frag.date().__fields__(*self.date_time_value_date_fields)
        dtv_frag.time().__fields__(*self.date_time_value_time_fields)

        # dtiv_frag = Fragment(DateTimeIntervalValue, 'DateTimeIntervalFull')
        # dtiv_frag.start.__fragment__(dtv_frag)
        # dtiv_frag.end.__fragment__(dtv_frag)

        slv_frag = Fragment(StringLocaleValue, 'StringLocaleFull')
        slv_frag.value()
        slv_frag.locale()

        sv_frag = Fragment(StringValue, 'StringFull')
        sv_frag.value()

        lv_frag = Fragment(LinkValue, 'LinkFull')
        lv_frag.link()

        dv_frag = Fragment(DoubleValue, 'DoubleFull')
        dv_frag.value(__alias__='double')

        iv_frag = Fragment(IntValue, 'IntFull')
        iv_frag.value(__alias__='number')

        cv_frag = Fragment(CompositeValue, 'CompFull')
        cv_frag.list_value().id()
        cv_frag.list_value().property_value_type()
        cv_frag.list_value().value().__fragment__(slv_frag)
        cv_frag.list_value().value().__fragment__(sv_frag)
        cv_frag.list_value().value().__fragment__(lv_frag)
        cv_frag.list_value().value().__fragment__(dv_frag)
        cv_frag.list_value().value().__fragment__(iv_frag)
        cv_frag.list_value().value().__fragment__(dtv_frag)

        graphql_value.__fragment__(slv_frag)
        graphql_value.__fragment__(sv_frag)
        graphql_value.__fragment__(lv_frag)
        graphql_value.__fragment__(dv_frag)
        graphql_value.__fragment__(iv_frag)
        graphql_value.__fragment__(dtv_frag)
        graphql_value.__fragment__(cv_frag)
        # graphql_value.__fragment__(dtiv_frag)

    def _configure_output_concept_fields(
        self, concept_object, with_aliases=False, with_properties=False, with_links=False,
        with_link_properties=False, with_facts=False
    ):
        concept_object.__fields__(*self.concept_fields)
        concept_object.concept_type.__fields__(*self.concept_type_fields)
        if with_aliases:
            sv_frag = Fragment(StringValue, 'StringFull')
            sv_frag.value()
            concept_object.list_alias.value.__fragment__(sv_frag)
        if with_properties:
            pcp: ConceptPropertyPagination = concept_object.pagination_concept_property(
                offset=0,
                limit=10000,
                filter_settings=ConceptPropertyFilterSettings()
            )
            lcp = pcp.list_concept_property()
            lcp.__fields__(*self.concept_property_fields)
            lcp.property_type().__fields__(*self.concept_property_type_fields)
            self._configure_output_value_fields(lcp.value)
        if with_links:
            pcl: ConceptLinkPagination = concept_object.pagination_concept_link(
                offset=0,
                limit=10000,
                filter_settings=ConceptLinkFilterSettings()
            )
            self._configure_output_link_fields(pcl.list_concept_link(), with_link_properties=with_link_properties)
        if with_facts:
            pcf: ConceptFactPagination = concept_object.pagination_concept_fact(
                offset=0,
                limit=10000,
                filter_settings=DocumentLinkFilterSetting()
            )
            lcf = pcf.list_concept_fact()
            lcf.__fields__(*self.concept_fact_fields)
            d = lcf.document()
            d.__fields__(*self.document_fields_extended)
            dm = d.metadata()
            dm.platform().__fields__(*self.document_platform_fields)
            dm.account().__fields__(*self.document_account_fields)

    def _configure_output_link_fields(
        self, link_object, with_link_properties=False
    ):
        link_object.__fields__(*self.concept_link_fields)
        link_object.concept_from().__fields__(*self.concept_link_concept_from_fields)
        link_object.concept_to().__fields__(*self.concept_link_concept_to_fields)
        link_object.concept_link_type().__fields__(*self.concept_link_type_fields_truncated)
        if with_link_properties:
            pcp: ConceptPropertyPagination = link_object.pagination_concept_link_property(
                offset=0,
                limit=10000,
                filter_settings=ConceptPropertyFilterSettings()
            )
            lcp = pcp.list_concept_property()
            lcp.__fields__(*self.concept_property_fields)
            lcp.property_type().__fields__(*self.concept_property_type_fields)
            self._configure_output_value_fields(lcp.value)

    def _create_concept_with_input(
        self, input: ConceptMutationInput, with_properties=False, with_links=False, with_link_properties=False,
        perform_synchronously: Optional[bool] = None
    ) -> Concept:
        perform_synchronously = self.get_perform_synchronously_value(perform_synchronously)
        op = Operation(Mutation)
        ac = op.add_concept(
            performance_control=PerformSynchronously(
                perform_synchronously=perform_synchronously
            ),
            form=input
        )
        self._configure_output_concept_fields(
            ac, with_properties=with_properties, with_links=with_links,
            with_link_properties=with_link_properties
        )
        res = self._gql_client.execute(op)
        res = op + res

        if self.tdm_builder is not None:
            self.tdm_builder.add_concept_fact(res.add_concept)

        return res.add_concept

    def _get_components_mapping(
        self, component_values: Dict[str, str],
        component_value_types: List[CompositePropertyValueType]
    ) -> Dict[str, CompositePropertyValueType]:
        components_type_mapping = {}
        for component_value in component_values:
            for component_value_type in component_value_types:
                if component_value_type.name != component_values[component_value]:
                    continue
                components_type_mapping[component_value] = component_value_type
        return components_type_mapping

    def _get_value_input(
        self, values: dict, components_type_mapping: Dict[str, CompositePropertyValueType]
    ) -> List[ComponentValueInput]:
        value_input = []
        for field in values:
            if field not in components_type_mapping:
                continue
            value_id = components_type_mapping[field].id
            value_input.append(ComponentValueInput(
                id=value_id,
                value=get_map_helper(
                    components_type_mapping[field].value_type.value_type).get_value_input(values[field])
            ))
        return value_input

    def _configure_pipeline_topic_fields(self, kafka_topic: tc.KafkaTopic):
        kafka_topic.__fields__(*self.pipeline_topic_fields)
        kafka_topic.metrics().__fields__(*self.pipeline_metrics_fields)
        kafka_topic.pipeline().pipeline_config().__fields__(*self.pipeline_config_fields)

    def get_all_documents(
        self, grouping: DocumentGrouping = 'none', filter_settings: DocumentFilterSettings = None,
        direction: SortDirection = 'descending', sort_field: DocumentSorting = 'score',
        extra_settings: ExtraSettings = None, with_extended_information: bool = False
    ) -> Iterable[Story]:
        if filter_settings is None:
            filter_settings = DocumentFilterSettings()
        if extra_settings is None:
            extra_settings = ExtraSettings()

        total = self.get_documents_count(filter_settings=filter_settings)

        if total > self.kb_iterator_config.max_total_count:
            had_creation_date = hasattr(filter_settings, 'registration_date')
            old_timestamp_interval = None
            if had_creation_date:
                old_timestamp_interval = copy(filter_settings.registration_date)
            start: int = getattr(old_timestamp_interval, 'start', self.kb_iterator_config.earliest_created_time)
            end: int = getattr(old_timestamp_interval, 'end', int(time()))
            middle: int = (end + start) // 2

            for start, end in (start, middle), (middle, end):
                filter_settings.registration_date = TimestampInterval(start=start, end=end)
                for c in self.get_all_documents(
                    grouping=grouping, filter_settings=filter_settings, direction=direction, sort_field=sort_field,
                    extra_settings=extra_settings, with_extended_information=with_extended_information
                ):
                    yield c

            if had_creation_date:
                filter_settings.registration_date = old_timestamp_interval
            else:
                delattr(filter_settings, 'registration_date')
            return
        elif not total:
            return

        documents: Iterable = [None]
        i: int = 0
        while documents:
            documents = self.get_documents(
                skip=i * self._limit, take=self._limit, grouping=grouping, filter_settings=filter_settings,
                direction=direction, sort_field=sort_field, extra_settings=extra_settings,
                with_extended_information=with_extended_information
            )
            for d in documents:
                yield d
            i += 1

    def get_documents(
        self, skip: int = 0, take: Optional[int] = None, grouping: DocumentGrouping = 'none',
        filter_settings: DocumentFilterSettings = None, direction: SortDirection = 'descending',
        sort_field: DocumentSorting = 'score', extra_settings: ExtraSettings = None,
        with_extended_information: bool = False
    ) -> Sequence[Story]:
        take = self.get_take_value(take)
        pagination_story_kwargs = dict()
        if filter_settings is None:
            filter_settings = DocumentFilterSettings()
        if extra_settings is None:
            extra_settings = ExtraSettings()

        op = Operation(Query)
        ps: StoryPagination = op.pagination_story(
            offset=skip,
            limit=take,
            grouping=grouping,
            filter_settings=filter_settings,
            direction=direction,
            sort_field=sort_field,
            extra_settings=extra_settings,
            **pagination_story_kwargs
        )
        ps.list_story().list_document().__fields__(*self.document_fields_truncated)
        m = ps.list_story().main()
        if with_extended_information:
            m.__fields__(*self.document_fields_extended)
            mdm = m.metadata()
            mdm.platform().__fields__(*self.document_platform_fields)
            mdm.account().__fields__(*self.document_account_fields)
        else:
            m.__fields__(*self.document_fields_truncated)
        m.text().__fields__(*self.document_text_fields_truncated)

        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_story.list_story

    def get_documents_count(self, filter_settings: DocumentFilterSettings = None) -> int:
        if filter_settings is None:
            filter_settings = DocumentFilterSettings()

        op = Operation(Query)
        ps: StoryPagination = op.pagination_story(
            limit=1,
            filter_settings=filter_settings,
            extra_settings=ExtraSettings()
        )
        ps.show_total()
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_story.show_total

    def get_documents_by_limit_offset_filter_extra_settings(
        self, skip: int = 0, take: Optional[int] = None, filter_settings: Optional[DocumentFilterSettings] = None,
        extra_settings: Optional[ExtraSettings] = None
    ) -> Sequence[Story]:
        take = self.get_take_value(take)
        op = Operation(Query)
        ps: StoryPagination = op.pagination_story(
            offset=skip,
            limit=take,
            extra_settings=extra_settings if extra_settings else ExtraSettings(),
            filter_settings=filter_settings if filter_settings else DocumentFilterSettings()
        )
        ps.list_story().list_document().__fields__(*self.document_fields_truncated)
        m = ps.list_story().main()
        m.__fields__(*self.document_fields_truncated)
        m.text().__fields__(*self.document_text_fields_truncated)

        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_story.list_story

    def get_document(self, document_id: str) -> Document:
        op = Operation(Query)
        d: Document = op.document(
            id=document_id
        )
        d.__fields__(*self.document_fields)
        d.last_updater().__fields__(*self.user_fields)
        d.metadata().platform().__fields__(*self.document_platform_fields)
        d.metadata().account().__fields__(*self.document_account_fields)
        d.creator().__fields__(*self.user_fields)
        dt = d.text(show_hidden=True)
        dt.__fields__(*self.document_text_fields)
        dt.metadata().__fields__(*self.document_text_metadata_fields)
        dt.metadata().text_translations().text()
        dt.metadata().text_translations().language().id()
        cd = d.list_child()
        cd.__fields__(*self.document_fields)
        cd.last_updater().__fields__(*self.user_fields)
        cd.metadata().platform().__fields__(*self.document_platform_fields)
        cd.metadata().account().__fields__(*self.document_account_fields)
        cd.creator().__fields__(*self.user_fields)
        cdt = cd.text(show_hidden=True)
        cdt.__fields__(*self.document_text_fields)
        cdt.metadata().__fields__(*self.document_text_metadata_fields)

        res = self._gql_client.execute(op)
        res = op + res
        return res.document

    def get_concept_count(self, filter_settings: ConceptFilterSettings = None) -> int:
        op = Operation(Query)
        pc: ConceptPagination = op.pagination_concept(
            filter_settings=filter_settings if filter_settings else ConceptFilterSettings()
        )
        pc.show_total()
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept.show_total

    def get_concept_link_count(self, filter_settings: ConceptLinkFilterSettings = None) -> int:
        op = Operation(Query)
        pcl: ConceptLinkPagination = op.pagination_concept_link(
            filter_settings=filter_settings if filter_settings else ConceptLinkFilterSettings()
        )
        pcl.total()
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept_link.total

    def get_concept(
        self, concept_id: str, with_aliases: bool = False,
        with_properties: bool = True, with_links: bool = True,
        with_link_properties: bool = True, with_facts: bool = False
    ) -> Concept:
        op = Operation(Query)
        c: Concept = op.concept(
            id=concept_id
        )
        self._configure_output_concept_fields(
            c, with_aliases=with_aliases, with_properties=with_properties,
            with_links=with_links, with_link_properties=with_link_properties, with_facts=with_facts
        )
        res = self._gql_client.execute(op)
        res = op + res

        if self.tdm_builder is not None:
            self.tdm_builder.add_concept_fact(res.concept)

        return res.concept

    def get_concept_facts(
        self, concept_id: str, filter_settings: DocumentLinkFilterSetting = None
    ) -> Sequence[ConceptFact]:

        op = Operation(Query)
        c: Concept = op.concept(
            id=concept_id
        )
        pcf: ConceptFactPagination = c.pagination_concept_fact(
            filter_settings=filter_settings if filter_settings else DocumentLinkFilterSetting()
        )
        lcf = pcf.list_concept_fact()
        lcf.__fields__(*self.concept_fact_fields)
        d = lcf.document()
        d.__fields__(*self.document_fields_extended)
        dm = d.metadata()
        dm.platform().__fields__(*self.document_platform_fields)
        dm.account().__fields__(*self.document_account_fields)

        res = self._gql_client.execute(op)
        res = op + res

        return res.concept.pagination_concept_fact.list_concept_fact

    def get_concept_link(self, link_id: str) -> ConceptLink:
        op = Operation(Query)
        cl: ConceptLink = op.concept_link(
            id=link_id
        )
        cl.__fields__(*self.concept_link_fields)
        self._configure_output_concept_fields(cl.concept_from)
        self._configure_output_concept_fields(cl.concept_to)
        cl.concept_link_type.__fields__(*self.concept_link_type_fields_truncated)
        res = self._gql_client.execute(op)
        res = op + res

        if self.tdm_builder is not None:
            self.tdm_builder.add_link_fact(res.concept_link)

        return res.concept_link

    def get_all_concepts(
        self, filter_settings: ConceptFilterSettings = None,
        direction: SortDirection = 'descending', sort_field: ConceptSorting = 'score',
        with_aliases: bool = False, with_properties: bool = False, with_links: bool = False,
        with_link_properties: bool = False, with_facts: bool = False
    ) -> Iterable[Concept]:
        if not filter_settings:
            filter_settings = ConceptFilterSettings()
        total = self.get_concept_count(filter_settings=filter_settings)

        if total > self.kb_iterator_config.max_total_count:
            had_creation_date = hasattr(filter_settings, 'creation_date')
            old_timestamp_interval = None
            if had_creation_date:
                old_timestamp_interval = copy(filter_settings.creation_date)
            start: int = getattr(old_timestamp_interval, 'start', self.kb_iterator_config.earliest_created_time)
            end: int = getattr(old_timestamp_interval, 'end', int(time()))
            middle: int = (end + start) // 2

            for start, end in (start, middle), (middle, end):
                filter_settings.creation_date = TimestampInterval(start=start, end=end)
                for c in self.get_all_concepts(filter_settings=filter_settings,
                                               direction=direction, sort_field=sort_field, with_aliases=with_aliases,
                                               with_properties=with_properties, with_links=with_links,
                                               with_link_properties=with_link_properties, with_facts=with_facts):
                    yield c

            if had_creation_date:
                filter_settings.creation_date = old_timestamp_interval
            else:
                delattr(filter_settings, 'creation_date')
            return
        elif not total:
            return

        concepts: Iterable = [None]
        i: int = 0
        while concepts:
            concepts = self.get_concepts(
                skip=i * self._limit, take=self._limit, filter_settings=filter_settings,
                direction=direction, sort_field=sort_field, with_aliases=with_aliases,
                with_properties=with_properties, with_links=with_links,
                with_link_properties=with_link_properties, with_facts=with_facts
            )
            for c in concepts:
                yield c
            i += 1

    def get_concepts(
        self, skip: int = 0, take: Optional[int] = None, filter_settings: ConceptFilterSettings = None,
        direction: SortDirection = 'descending', sort_field: ConceptSorting = 'score',
        with_aliases: bool = False, with_properties: bool = False, with_links: bool = False,
        with_link_properties: bool = False, with_facts: bool = False
    ) -> Sequence[Concept]:
        take = self.get_take_value(take)
        pagination_concept_kwargs = dict()
        if not filter_settings:
            filter_settings = ConceptFilterSettings()

        op = Operation(Query)
        cp: ConceptPagination = op.pagination_concept(
            limit=take,
            offset=skip,
            filter_settings=filter_settings,
            direction=direction,
            sort_field=sort_field,
            **pagination_concept_kwargs
        )
        self._configure_output_concept_fields(
            cp.list_concept(), with_aliases=with_aliases, with_properties=with_properties,
            with_links=with_links, with_link_properties=with_link_properties, with_facts=with_facts
        )
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept.list_concept

    def get_concepts_by_limit_offset_filter_settings(
        self, skip: int = 0, take: Optional[int] = None, filter_settings: Optional[ConceptFilterSettings] = None,
        with_aliases: bool = False, with_facts: bool = False
    ) -> Sequence[Concept]:
        take = self.get_take_value(take)
        op = Operation(Query)
        cp: ConceptPagination = op.pagination_concept(
            filter_settings=filter_settings if filter_settings else ConceptFilterSettings(),
            offset=skip,
            limit=take
        )
        self._configure_output_concept_fields(cp.list_concept(), with_aliases=with_aliases, with_facts=with_facts)
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept.list_concept

    def get_all_concept_links(
        self, filter_settings: ConceptLinkFilterSettings = None, with_link_properties: bool = False
    ) -> Iterable[ConceptLink]:
        if not filter_settings:
            filter_settings = ConceptLinkFilterSettings()

        total = self.get_concept_link_count(filter_settings=filter_settings)

        if total > self.kb_iterator_config.max_total_count:
            had_creation_date = hasattr(filter_settings, 'creation_date')
            old_timestamp_interval = None
            if had_creation_date:
                old_timestamp_interval = copy(filter_settings.creation_date)
            start: int = getattr(old_timestamp_interval, 'start', self.kb_iterator_config.earliest_created_time)
            end: int = getattr(old_timestamp_interval, 'end', int(time()))
            middle: int = (end + start) // 2

            for start, end in (start, middle), (middle, end):
                filter_settings.creation_date = TimestampInterval(start=start, end=end)
                for c in self.get_all_concept_links(filter_settings=filter_settings,
                                                    with_link_properties=with_link_properties):
                    yield c

            if had_creation_date:
                filter_settings.creation_date = old_timestamp_interval
            else:
                delattr(filter_settings, 'creation_date')
            return
        elif not total:
            return

        links: Iterable = [None]
        i: int = 0
        while links:
            links = self.get_concept_links_by_limit_offset_filter_settings(
                skip=i * self._limit, take=self._limit,
                filter_settings=filter_settings, with_link_properties=with_link_properties
            )
            for link in links:
                yield link
            i += 1

    def get_concept_links_by_limit_offset_filter_settings(
        self, skip: int = 0, take: Optional[int] = None, filter_settings: ConceptLinkFilterSettings = None,
        with_link_properties: bool = False
    ) -> Sequence[ConceptLink]:
        take = self.get_take_value(take)
        op = Operation(Query)
        pcl: ConceptLinkPagination = op.pagination_concept_link(
            filter_settings=filter_settings if filter_settings else ConceptLinkFilterSettings(),
            offset=skip,
            limit=take
        )
        self._configure_output_link_fields(pcl.list_concept_link(), with_link_properties=with_link_properties)
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept_link.list_concept_link

    def get_concepts_by_type_id_with_offset(
        self, type_id: str, skip: int, take: Optional[int] = None, direction='descending',
        sort_field='systemRegistrationDate', with_aliases: bool = False, with_properties: bool = False,
        with_links: bool = False, with_link_properties: bool = False, with_facts: bool = False
    ) -> ConceptPagination:
        take = self.get_take_value(take)
        op = Operation(Query)
        cp: ConceptPagination = op.pagination_concept(
            filter_settings=ConceptFilterSettings(
                concept_type_ids=[type_id]
            ),
            limit=take,
            offset=skip,
            direction=direction,
            sort_field=sort_field
        )
        cp.total()
        self._configure_output_concept_fields(
            cp.list_concept(), with_aliases=with_aliases, with_properties=with_properties, with_links=with_links,
            with_link_properties=with_link_properties, with_facts=with_facts
        )
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept

    def get_concepts_by_type_id_with_offset_with_markers(
        self, type_id: str, skip: int = 0, take: Optional[int] = None, markers: Optional[List[str]] = None,
        direction='descending', sort_field='systemRegistrationDate', with_aliases: bool = False,
        with_facts: bool = False
    ) -> ConceptPagination:
        take = self.get_take_value(take)
        op = Operation(Query)
        cp: ConceptPagination = op.pagination_concept(
            filter_settings=ConceptFilterSettings(
                concept_type_ids=[type_id],
                markers=markers,
            ),
            limit=take,
            offset=skip,
            direction=direction,
            sort_field=sort_field
        )
        cp.total()
        self._configure_output_concept_fields(cp.list_concept(), with_aliases=with_aliases, with_facts=with_facts)
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept

    def get_concepts_by_name(
        self, name: str, type_id: Optional[str] = None, with_aliases: bool = False, with_facts: bool = False
    ) -> Sequence[Concept]:

        op = Operation(Query)
        if type_id:
            concept_filter_settings: ConceptFilterSettings = ConceptFilterSettings(
                exact_name=name,
                concept_type_ids=[type_id]
            )
        else:
            concept_filter_settings: ConceptFilterSettings = ConceptFilterSettings(
                exact_name=name
            )
        cp: ConceptPagination = op.pagination_concept(
            filter_settings=concept_filter_settings
        )
        self._configure_output_concept_fields(cp.list_concept(), with_aliases=with_aliases, with_facts=with_facts)
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept.list_concept

    def get_concepts_by_near_name(
        self, name: str, type_id: Optional[str] = None, with_aliases: bool = False, with_facts: bool = False
    ) -> Sequence[Concept]:

        op = Operation(Query)
        if type_id:
            concept_filter_settings: ConceptFilterSettings = ConceptFilterSettings(
                name=name,
                concept_type_ids=[type_id]
            )
        else:
            concept_filter_settings: ConceptFilterSettings = ConceptFilterSettings(
                name=name
            )
        cp: ConceptPagination = op.pagination_concept(
            filter_settings=concept_filter_settings
        )
        self._configure_output_concept_fields(cp.list_concept(), with_aliases=with_aliases, with_facts=with_facts)
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept.list_concept

    def get_concepts_by_property_name(
        self, property_type_id: str, string_filter: str, property_type: str = 'concept', with_aliases: bool = False,
        with_facts: bool = False
    ) -> Sequence[Concept]:

        op = Operation(Query)
        cp: ConceptPagination = op.pagination_concept(
            filter_settings=ConceptFilterSettings(
                property_filter_settings=[PropertyFilterSettings(
                    property_type=property_type,
                    property_type_id=property_type_id,
                    string_filter=StringFilter(
                        str=string_filter
                    )
                )]
            )
        )
        self._configure_output_concept_fields(cp.list_concept(), with_aliases=with_aliases, with_facts=with_facts)
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept.list_concept

    def get_concept_properties(self, concept_id: str) -> Sequence[ConceptProperty]:
        op = Operation(Query)
        concept: Concept = op.concept(
            id=concept_id
        )
        pcp: ConceptPropertyPagination = concept.pagination_concept_property(
            offset=0,
            limit=10000,
            filter_settings=ConceptPropertyFilterSettings()
        )
        lcp = pcp.list_concept_property()
        lcp.__fields__(*self.concept_property_fields)
        lcp.property_type().__fields__(*self.concept_property_type_fields)
        self._configure_output_value_fields(lcp.value)

        res = self._gql_client.execute(op)
        res = op + res  # type: Query
        return res.concept.pagination_concept_property.list_concept_property

    def get_concept_links(self, concept_id: str, with_link_properties: bool = False) -> Sequence[ConceptLink]:
        op = Operation(Query)

        concept: Concept = op.concept(
            id=concept_id
        )
        pcl: ConceptLinkPagination = concept.pagination_concept_link(
            offset=0,
            limit=10000,
            filter_settings=ConceptLinkFilterSettings()
        )
        self._configure_output_link_fields(pcl.list_concept_link(), with_link_properties=with_link_properties)
        res = self._gql_client.execute(op)
        res = op + res  # type: Query
        return res.concept.pagination_concept_link.list_concept_link

    def get_concept_links_concept(
            self, concept_id: str, link_type_id: str, with_link_properties: bool = False
    ) -> Sequence[ConceptLink]:
        op = Operation(Query)

        concept = op.concept(
            id=concept_id
        )
        pcl = concept.pagination_concept_link(
            offset=0,
            limit=10000,
            filter_settings=ConceptLinkFilterSettings(
                concept_link_type=[link_type_id]
            )
        )
        self._configure_output_link_fields(pcl.list_concept_link(), with_link_properties=with_link_properties)
        res = self._gql_client.execute(op)
        res = op + res  # type: Query
        return res.concept.pagination_concept_link.list_concept_link

    def get_link_properties(self, link_id: str) -> Sequence[ConceptProperty]:
        op = Operation(Query)
        concept_link: ConceptLink = op.concept_link(
            id=link_id
        )
        pcp: ConceptPropertyPagination = concept_link.pagination_concept_link_property(
            offset=0,
            limit=10000,
            filter_settings=ConceptPropertyFilterSettings()
        )
        lcp = pcp.list_concept_property()
        lcp.__fields__(*self.concept_property_fields)
        lcp.property_type().__fields__(*self.concept_property_type_fields)
        self._configure_output_value_fields(lcp.value)

        res = self._gql_client.execute(op)
        res = op + res  # type: Query
        return res.concept_link.pagination_concept_link_property.list_concept_property

    def get_concept_time_intervals(
        self, filter_settings: ConceptFilterSettings = None, max_interval_size: Optional[int] = None
    ) -> Iterable[ObjectTimeInterval]:
        if not filter_settings:
            filter_settings = ConceptFilterSettings()
        for time_interval in self._get_object_time_intervals(filter_settings, max_interval_size):
            yield time_interval

    def get_concept_link_time_intervals(
        self, filter_settings: ConceptLinkFilterSettings = None, max_interval_size: Optional[int] = None
    ) -> Iterable[ObjectTimeInterval]:
        if not filter_settings:
            filter_settings = ConceptLinkFilterSettings()
        for time_interval in self._get_object_time_intervals(filter_settings, max_interval_size):
            yield time_interval

    def get_document_time_intervals(
        self, filter_settings: DocumentFilterSettings = None, max_interval_size: Optional[int] = None
    ) -> Iterable[ObjectTimeInterval]:
        if not filter_settings:
            filter_settings = DocumentFilterSettings()
        for time_interval in self._get_object_time_intervals(filter_settings, max_interval_size):
            yield time_interval

    def _get_object_time_intervals(
        self, filter_settings: Union[ConceptFilterSettings, ConceptLinkFilterSettings, DocumentFilterSettings],
        max_interval_size: Optional[int] = None
    ) -> Iterable[ObjectTimeInterval]:
        max_interval_size = max_interval_size if max_interval_size else self.kb_iterator_config.max_total_count

        creation_date_field_name = 'creation_date'
        if isinstance(filter_settings, ConceptFilterSettings):
            object_count = self.get_concept_count(filter_settings)
        elif isinstance(filter_settings, ConceptLinkFilterSettings):
            object_count = self.get_concept_link_count(filter_settings)
        elif isinstance(filter_settings, DocumentFilterSettings):
            object_count = self.get_documents_count(filter_settings)
            creation_date_field_name = 'registration_date'
        else:
            raise Exception('Time division is only available for concepts, links and documents')
        creation_date = getattr(filter_settings, creation_date_field_name, None)
        start: int = getattr(creation_date, 'start', self.kb_iterator_config.earliest_created_time)
        end: int = getattr(creation_date, 'end', int(time()))

        if (object_count > max_interval_size) and (start < end):
            middle = (end + start) // 2

            for start, end in (start, middle), (middle + 1, end):
                setattr(filter_settings, creation_date_field_name, TimestampInterval(start=start, end=end))
                for time_interval in self._get_object_time_intervals(filter_settings, max_interval_size):
                    yield time_interval
        elif object_count > 0:
            yield ObjectTimeInterval(
                start_time=start,
                end_time=end,
                object_count=object_count,
                max_interval_size=max_interval_size
            )

    def create_concept(
        self, name: str, type_id: str, notes: str = None, with_properties=False, with_links=False,
        with_link_properties=False, perform_synchronously: Optional[bool] = None
    ) -> Concept:

        cmi: ConceptMutationInput = ConceptMutationInput(
            name=name,
            concept_type_id=type_id,
            notes=notes
        )
        return self._create_concept_with_input(
            cmi, with_properties=with_properties, with_links=with_links,
            with_link_properties=with_link_properties, perform_synchronously=perform_synchronously
        )

    def update_concept(
        self, c: Concept, markers: List[str] = None, notes: str = None, perform_synchronously: Optional[bool] = None
    ) -> Concept:
        perform_synchronously = self.get_perform_synchronously_value(perform_synchronously)
        op = Operation(Mutation)
        uc: Concept = op.update_concept(
            performance_control=PerformSynchronously(
                perform_synchronously=perform_synchronously
            ),
            form=ConceptUpdateInput(
                concept_id=c.id,
                name=c.name,
                concept_type_id=c.concept_type.id,
                markers=markers if markers is not None else c.markers,
                notes=notes if notes is not None else c.notes
            )
        )
        self._configure_output_concept_fields(uc)
        res = self._gql_client.execute(op)
        res = op + res

        return res.update_concept

    def update_concept_property_value_types(self, cpvt: ConceptPropertyValueType) -> ConceptPropertyValueType:
        op = Operation(Mutation)
        ucpvt = op.update_concept_property_value_type(
            form=ConceptPropertyValueTypeUpdateInput(
                id=cpvt.id,
                name=cpvt.name,
                value_type=cpvt.value_type,
                pretrained_nercmodels=cpvt.pretrained_nercmodels,
                value_restriction=cpvt.value_restriction
            )
        )
        ucpvt.__fields__('id')
        res = self._gql_client.execute(op)
        res = op + res

        return res.update_concept_property_value_type

    def update_concept_string_property(self, cp: ConceptProperty) -> ConceptProperty:
        op = Operation(Mutation)
        ucp: ConceptProperty = op.update_concept_property(
            form=ConceptPropertyUpdateInput(
                property_id=cp.id,
                is_main=cp.is_main,
                value_input=[
                    ComponentValueInput(
                        value=ValueInput(
                            string_value_input=StringValueInput(
                                value=cp.value.value
                            )
                        )
                    )
                ]
            )
        )
        ucp.__fields__('id')
        res = self._gql_client.execute(op)
        res = op + res

        return res.update_concept_property

    def update_concept_int_property(self, cp: ConceptProperty) -> ConceptProperty:
        op = Operation(Mutation)
        ucp: ConceptProperty = op.update_concept_property(
            form=ConceptPropertyUpdateInput(
                property_id=cp.id,
                is_main=cp.is_main,
                value_input=[
                    ComponentValueInput(
                        value=ValueInput(
                            int_value_input=IntValueInput(
                                value=cp.value.number
                            )
                        )
                    )
                ]
            )
        )
        ucp.__fields__('id')
        res = self._gql_client.execute(op)
        res = op + res

        return res.update_concept_property

    def update_concept_composite_property(self, cp: ConceptProperty) -> ConceptProperty:
        value_input = []
        for value in cp.value.list_value:
            if type(value.value) is StringValue:
                value_input.append(ComponentValueInput(
                    id=value.id,
                    value=ValueInput(
                        string_value_input=StringValueInput(value=value.value.value)
                    ))
                )
            elif type(value.value) is IntValue:
                value_input.append(ComponentValueInput(
                    id=value.id,
                    value=ValueInput(
                        int_value_input=IntValueInput(value=value.value.number)
                    ))
                )
            elif type(value.value) is DateTimeValue:
                value_input.append(ComponentValueInput(
                    id=value.id,
                    value=ValueInput(
                        date_time_value_input=DateTimeValueInput(date=value.value.date, time=value.value.time)
                    ))
                )
            elif type(value.value) is StringLocaleValue:
                value_input.append(ComponentValueInput(
                    id=value.id,
                    value=ValueInput(
                        string_locale_value_input=StringLocaleValueInput(
                            value=value.value.value,
                            locale=value.value.locale
                        )
                    ))
                )
            elif type(value.value) is LinkValue:
                value_input.append(ComponentValueInput(
                    id=value.id,
                    value=ValueInput(
                        link_value_input=LinkValueInput(link=value.value.link)
                    ))
                )
            elif type(value.value) is DoubleValue:
                value_input.append(ComponentValueInput(
                    id=value.id,
                    value=ValueInput(
                        double_value_input=DoubleValueInput(double=value.value.double)
                    ))
                )
        op = Operation(Mutation)
        ucp: ConceptProperty = op.update_concept_property(
            form=ConceptPropertyUpdateInput(
                property_id=cp.id,
                is_main=cp.is_main,
                value_input=value_input
            )
        )
        ucp.__fields__('id')
        res = self._gql_client.execute(op)
        res = op + res

        return res.update_concept_property

    def delete_concept_property(self, cp_id: str) -> bool:
        op = Operation(Mutation)
        dcp = op.delete_concept_property(
            id=cp_id
        )
        dcp.__fields__('is_success')
        res = self._gql_client.execute(op)
        res = op + res

        return res.delete_concept_property.is_success

    def get_concept_types_by_name(self, name: str) -> Sequence[ConceptType]:
        op = Operation(Query)
        ctp: ConceptTypePagination = op.pagination_concept_type(
            filter_settings=ConceptTypeFilterSettings(
                name=name
            )
        )
        ctp.list_concept_type().__fields__(*self.concept_type_fields)
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept_type.list_concept_type

    def get_concept_type_info(self, concept_type_id: str) -> ConceptType:
        op = Operation(Query)
        ct = op.concept_type(
            id=concept_type_id
        )
        ct.__fields__(*self.concept_type_fields)
        lcpt = ct.list_concept_property_type()
        lcpt.__fields__(*self.concept_property_type_fields)
        lclt = ct.list_concept_link_type()
        lclt.__fields__(*self.concept_link_type_fields)
        lclt.list_concept_link_property_type().__fields__(*self.concept_property_type_fields)

        res = self._gql_client.execute(op)
        res = op + res
        return res.concept_type

    def get_concept_type(self, concept_type_code: str) -> Optional[ConceptType]:
        concept_type: Optional[ConceptType] = self._cache.get(f'concept_type_{concept_type_code}', None)
        if concept_type:
            return concept_type
        concept_type_name = self._types['concepts_types_mapping'][concept_type_code]['name']
        types = self.get_concept_types_by_name(concept_type_name)
        if not types:
            return None
        concept_type = None
        for t in types:
            if t.name == concept_type_name:
                concept_type = t
                break
        if not concept_type:
            return None
        self._cache[f'concept_type_{concept_type_code}'] = concept_type
        return concept_type

    def get_concept_properties_types_by_name(
            self, concept_type_id: str, prop_name: str
    ) -> Sequence[ConceptPropertyType]:

        op = Operation(Query)
        cptp: ConceptPropertyTypePagination = op.pagination_concept_property_type(
            filter_settings=ConceptPropertyTypeFilterSettings(
                name=prop_name,
                concept_type_id=concept_type_id
            )
        )
        lcpt = cptp.list_concept_property_type()
        lcpt.__fields__(*self.concept_property_type_fields)
        self._configure_property_value_type_fields(lcpt.value_type, True)
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept_property_type.list_concept_property_type

    def get_concept_property_type(
        self, concept_type_code: str, property_type_code: str
    ) -> Optional[ConceptPropertyType]:

        concept_type: ConceptType = self.get_concept_type(concept_type_code)
        if not concept_type:
            raise Exception('Cannot get concept property type: no concept type id')

        prop_type = self._cache.get(f'concept_property_type_{concept_type_code}_{property_type_code}', None)
        if prop_type:
            return prop_type
        prop_type_name = self._types['concepts_types_mapping'][concept_type_code]['properties'][property_type_code]
        types = self.get_concept_properties_types_by_name(concept_type.id, prop_type_name)
        for prop_type in types:
            if prop_type.name == prop_type_name:
                self._cache[f'concept_property_type_{concept_type_code}_{property_type_code}'] = prop_type
                return prop_type
        return None

    def get_concept_property_value_types_by_name(
        self, prop_value_type_name: str, limit: int = 20, offset: int = 0
    ) -> Sequence[ConceptPropertyValueType]:
        op = Operation(Query)
        cpvtp: ConceptPropertyValueTypePagination = op.pagination_concept_property_value_type(
            filter_settings=ConceptPropertyValueTypeFilterSettings(
                name=prop_value_type_name
            ),
            limit=limit,
            offset=offset
        )
        cpvtp.list_concept_property_value_type().__fields__(*self.cpvt_fields)
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept_property_value_type.list_concept_property_value_type

    def get_concept_property_value_type(
        self, concept_property_value_type_code: str
    ) -> Optional[ConceptPropertyValueType]:

        concept_property_value_type: Optional[ConceptPropertyValueType] = self._cache.get(
            f'concept_property_value_type_{concept_property_value_type_code}', None
        )
        if concept_property_value_type:
            return concept_property_value_type
        concept_property_value_type_name = self._types['value_types_mapping'][concept_property_value_type_code]
        types = self.get_concept_property_value_types_by_name(concept_property_value_type_name)
        if not types:
            return None
        concept_property_value_type = None
        for t in types:
            if t.name == concept_property_value_type_name:
                concept_property_value_type = t
                break
        if not concept_property_value_type:
            return None
        self._cache[f'concept_property_value_type_{concept_property_value_type_code}'] = concept_property_value_type
        return concept_property_value_type

    def get_link_properties_types_by_name(self, link_type_id: str, prop_name: str) -> Sequence[ConceptPropertyType]:
        op = Operation(Query)
        cptp: ConceptPropertyTypePagination = op.pagination_concept_link_property_type(
            filter_settings=ConceptPropertyTypeFilterSettings(
                name=prop_name,
                concept_link_type_id=link_type_id
            )
        )
        lcpt = cptp.list_concept_property_type()
        lcpt.__fields__(*self.concept_property_type_fields)
        self._configure_property_value_type_fields(lcpt.value_type, True)
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept_link_property_type.list_concept_property_type

    def get_composite_link_properties_types_by_name(
        self, link_type_id: str, prop_name: str
    ) -> Sequence[ConceptPropertyType]:
        op = Operation(Query)
        cptp: ConceptPropertyTypePagination = op.pagination_composite_link_property_type(
            filter_settings=CompositePropertyTypeFilterSettings(
                name=prop_name,
                link_type_id=link_type_id
            )
        )
        lcpt = cptp.list_concept_property_type()
        lcpt.__fields__(*self.concept_property_type_fields)
        self._configure_property_value_type_fields(lcpt.value_type, True)
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_composite_link_property_type.list_concept_property_type

    def get_link_property_type(
        self, link_type_code: str, link_type: ConceptLinkType,
        property_type_code: str, is_composite: Optional[bool] = False
    ) -> Optional[ConceptPropertyType]:
        # link_type = self.get_link_type(link_type_code)
        # if not link_type:
        #     raise Exception('Cannot get link property type: no link type id')

        prop_type: Optional[ConceptPropertyType] = self._cache.get(
            f'link_property_type_{link_type_code}_{property_type_code}', None
        )
        if prop_type:
            return prop_type
        link_type_mapping = None
        for r in self._types['relations_types_mapping']:
            if r['old_relation_type'] == link_type_code:
                link_type_mapping = r
                break
        if not link_type_mapping:
            raise Exception('Cannot get link mapping object')
        if is_composite:
            prop_type_name = link_type_mapping['composite_properties'][property_type_code]['name']
            types = self.get_composite_link_properties_types_by_name(link_type.id, prop_type_name)
        else:
            prop_type_name = link_type_mapping['properties'][property_type_code]
            types = self.get_link_properties_types_by_name(link_type.id, prop_type_name)
        for prop_type in types:
            if prop_type.name == prop_type_name:
                self._cache[f'link_property_type_{link_type_code}_{property_type_code}'] = prop_type
                return prop_type
        return None

    def add_property_by_id(
        self, id: str, type_id: str, value: Any, is_main: bool, value_type: str,
        perform_synchronously: Optional[bool] = None
    ) -> ConceptProperty:
        perform_synchronously = self.get_perform_synchronously_value(perform_synchronously)
        op = Operation(Mutation)
        acp = op.add_concept_property(
            performance_control=PerformSynchronously(
                perform_synchronously=perform_synchronously
            ),
            form=ConceptPropertyCreateInput(
                concept_id=id,
                property_type_id=type_id,
                is_main=is_main,
                value_input=[
                    ComponentValueInput(
                        value=get_map_helper(value_type).get_value_input(value)
                    )
                ]
            )
        )
        acp.__fields__('id')
        acp.property_type().__fields__(*self.concept_property_type_fields)
        res = self._gql_client.execute(op)
        res = op + res

        return res.add_concept_property

    def add_property(
        self, concept_id: str, concept_type_code: str, property_type_code: str, value: Any, is_main: bool = False,
        perform_synchronously: Optional[bool] = None
    ) -> ConceptProperty:

        property_type: ConceptPropertyType = self.get_concept_property_type(concept_type_code, property_type_code)
        if not property_type:
            raise Exception('Cannot add property: no property type id')
        if type(property_type.value_type) is CompositePropertyValueTemplate:
            prop = self.add_composite_property_by_id(
                concept_id, property_type.id, value, is_main, property_type.value_type.component_value_types,
                concept_type_code, property_type_code, perform_synchronously
            )
        else:
            prop = self.add_property_by_id(
                concept_id, property_type.id, value, is_main, property_type.value_type.value_type, perform_synchronously
            )
        if self.tdm_builder is not None:
            self.tdm_builder.add_concept_property_fact(prop, self.get_concept(concept_id), value, property_type)

        return prop

    def add_link_property_by_id(
        self, link_id: str, type_id: str, value: str, is_main: bool, value_type: str,
        perform_synchronously: Optional[bool] = None
    ) -> ConceptProperty:
        perform_synchronously = self.get_perform_synchronously_value(perform_synchronously)
        op = Operation(Mutation)

        aclp = op.add_concept_link_property(
            performance_control=PerformSynchronously(
                perform_synchronously=perform_synchronously
            ),
            form=ConceptLinkPropertyInput(
                property_type_id=type_id,
                link_id=link_id,
                is_main=is_main,
                value_input=[
                    ComponentValueInput(
                        value=get_map_helper(value_type).get_value_input(value)
                    )
                ]
            )
        )
        aclp.__fields__('id')
        aclp.property_type().__fields__(*self.concept_property_type_fields)
        res = self._gql_client.execute(op)
        res = op + res

        return res.add_concept_link_property

    def add_concept_link_property_type(
        self, link_type_id: str, name: str, value_type_id: str
    ) -> ConceptPropertyType:

        op = Operation(Mutation)

        aclpt = op.add_concept_link_property_type(
            form=ConceptLinkPropertyTypeCreationInput(
                link_type_id=link_type_id,
                name=name,
                value_type_id=value_type_id,
            )
        )
        aclpt.__fields__(*self.concept_property_type_fields)
        res = self._gql_client.execute(op)
        res = op + res

        return res.add_concept_link_property_type

    def update_concept_link_property_type(
        self, link_property_type_id: str, name: str, value_type_id: str
    ) -> ConceptPropertyType:

        op = Operation(Mutation)

        acpt = op.update_concept_link_property_type(
            form=ConceptLinkPropertyTypeUpdateInput(
                id=link_property_type_id,
                name=name,
                value_type_id=value_type_id,
            )
        )
        acpt.__fields__(*self.concept_property_type_fields)
        res = self._gql_client.execute(op)
        res = op + res

        return res.update_concept_link_property_type

    def delete_concept_link_property_type(
        self, property_type_id: str
    ) -> State:

        op = Operation(Mutation)

        op.delete_concept_link_property_type(
            id=property_type_id
        )
        res = self._gql_client.execute(op)
        res = op + res

        return res.delete_concept_link_property_type

    def add_concept_property_type(
        self, concept_type_id: str, name: str, value_type_id: str
    ) -> ConceptPropertyType:

        op = Operation(Mutation)

        acpt = op.add_concept_property_type(
            form=ConceptPropertyTypeCreationInput(
                concept_type_id=concept_type_id,
                name=name,
                value_type_id=value_type_id,
            )
        )
        acpt.__fields__(*self.concept_property_type_fields)
        res = self._gql_client.execute(op)
        res = op + res

        return res.add_concept_property_type

    def delete_concept_property_type(
        self, property_type_id: str
    ) -> State:

        op = Operation(Mutation)

        op.delete_concept_property_type(
            id=property_type_id
        )
        res = self._gql_client.execute(op)
        res = op + res

        return res.delete_concept_property_type

    def add_link_composite_property_by_id(
        self, link_id: str, property_type_id: str, values: dict,
        component_value_types: List[CompositePropertyValueType],
        link_composite_property_code: str, link_code: str, is_main: bool, perform_synchronously: Optional[bool] = None
    ) -> ConceptProperty:
        for relation in self._types['relations_types_mapping']:
            if relation['old_relation_type'] != link_code:
                continue
            component_values = relation['composite_properties'][link_composite_property_code]['component_values']
            components_type_mapping = self._get_components_mapping(component_values, component_value_types)
            break
        value_input = self._get_value_input(values, components_type_mapping)
        perform_synchronously = self.get_perform_synchronously_value(perform_synchronously)
        op = Operation(Mutation)

        aclp = op.add_concept_link_property(
            performance_control=PerformSynchronously(
                perform_synchronously=perform_synchronously
            ),
            form=ConceptLinkPropertyInput(
                property_type_id=property_type_id,
                link_id=link_id,
                is_main=is_main,
                value_input=value_input
            )
        )
        aclp.__fields__('id')
        aclp.property_type().__fields__(*self.concept_property_type_fields)
        res = self._gql_client.execute(op)
        res = op + res

        return res.add_concept_link_property

    def add_composite_property_by_id(
        self, id: str, type_id: str, values: dict, is_main: bool,
        component_value_types: List[CompositePropertyValueType], concept_type_code: str, property_type_code: str,
        perform_synchronously: Optional[bool] = None
    ) -> ConceptProperty:
        concept_composite_properties = self._types['concepts_types_mapping'][concept_type_code]['composite_properties']
        concept_component_values = concept_composite_properties[property_type_code]['component_values']
        components_type_mapping = self._get_components_mapping(concept_component_values, component_value_types)
        value_input = self._get_value_input(values, components_type_mapping)
        perform_synchronously = self.get_perform_synchronously_value(perform_synchronously)
        op = Operation(Mutation)
        acp = op.add_concept_property(
            performance_control=PerformSynchronously(
                perform_synchronously=perform_synchronously
            ),
            form=ConceptPropertyCreateInput(
                concept_id=id,
                property_type_id=type_id,
                is_main=is_main,
                value_input=value_input
            )
        )
        acp.__fields__('id')
        acp.property_type().__fields__(*self.concept_property_type_fields)
        res = self._gql_client.execute(op)
        res = op + res

        return res.add_concept_property

    def add_link_property(
        self, link_id: str, link_type_code: str, link_type: ConceptLinkType, property_type_code: str, value: Any,
        is_composite: Optional[bool] = False, is_main: bool = False, perform_synchronously: Optional[bool] = None
    ) -> ConceptProperty:

        property_type: ConceptPropertyType = self.get_link_property_type(
            link_type_code, link_type, property_type_code, is_composite
        )
        if not property_type:
            raise Exception('Cannot add property: no property type id')
        if is_composite:
            link_property = self.add_link_composite_property_by_id(
                link_id, property_type.id, value, property_type.value_type.component_value_types,
                property_type_code, link_type_code, is_main, perform_synchronously
            )
        else:
            link_property = self.add_link_property_by_id(
                link_id, property_type.id, value, is_main,
                property_type.value_type.value_type, perform_synchronously=perform_synchronously
            )

        if self.tdm_builder is not None:
            self.tdm_builder.add_link_property_fact(link_property, self.get_concept_link(link_id), value, property_type)

        return link_property

    def get_concept_link_type_by_name(
        self, link_name: str, from_type_id: str, to_type_id: str, limit: int = 20
    ) -> Sequence[ConceptLinkType]:

        op = Operation(Query)
        pclt: ConceptLinkTypePagination = op.pagination_concept_link_type(
            filter_settings=ConceptLinkTypeFilterSettings(
                name=link_name,
                concept_from_type_id=from_type_id,
                concept_to_type_id=to_type_id
            ),
            limit=limit
        )
        pclt.list_concept_link_type().__fields__(*self.concept_link_type_fields)
        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_concept_link_type.list_concept_link_type

    def get_link_type(
        self, concept_from_type_code: str, concept_to_type_code: str, link_type_code: str
    ) -> Optional[ConceptLinkType]:

        link_type = self._cache.get(f'concept_link_type_{concept_from_type_code}_{link_type_code}', None)
        if link_type:
            return link_type
        from_type = self.get_concept_type(concept_from_type_code)
        to_type = self.get_concept_type(concept_to_type_code)
        if not from_type or not to_type:
            raise Exception('Cannot get link type: no concept type id')
        link_type_name = None
        for r in self._types['relations_types_mapping']:
            if r['old_relation_type'] == link_type_code:
                link_type_name = r['new_relation_type']
        if not link_type_name:
            raise Exception('Cannot get link type name')
        types = self.get_concept_link_type_by_name(link_type_name, from_type.id, to_type.id)
        for link_type in types:
            if link_type.name == link_type_name:
                self._cache[f'concept_link_type_{concept_from_type_code}_{link_type_code}'] = link_type
                return link_type
        return None

    def add_relation_by_id(
        self, from_id: str, to_id: str, link_type_id: str, perform_synchronously: Optional[bool] = None
    ) -> ConceptLink:
        perform_synchronously = self.get_perform_synchronously_value(perform_synchronously)
        op = Operation(Mutation)
        acl = op.add_concept_link(
            performance_control=PerformSynchronously(
                perform_synchronously=perform_synchronously
            ),
            form=ConceptLinkCreationMutationInput(
                concept_from_id=from_id,
                concept_to_id=to_id,
                link_type_id=link_type_id
            )
        )
        acl.__fields__(*self.concept_link_fields)
        acl.concept_link_type.__fields__(*self.concept_property_type_fields)
        acl.concept_link_type.__fields__(*self.concept_link_type_fields_truncated)
        self._configure_output_concept_fields(acl.concept_from)
        self._configure_output_concept_fields(acl.concept_to)

        res = self._gql_client.execute(op)
        res = op + res

        return res.add_concept_link

    def add_relation(
        self, concept_from_id: str, concept_to_id: str, concept_from_type_code: str, concept_to_type_code: str,
        type_code: str, perform_synchronously: Optional[bool] = None
    ) -> ConceptLink:

        link_type = self.get_link_type(concept_from_type_code, concept_to_type_code, type_code)
        if not link_type:
            raise Exception('Cannot add relation: no link type')
        relation = self.add_relation_by_id(
            concept_from_id, concept_to_id, link_type.id, perform_synchronously=perform_synchronously
        )

        if self.tdm_builder is not None:
            self.tdm_builder.add_link_fact(relation)

        return relation

    def delete_concept(self, concept_id: str) -> bool:

        op = Operation(Mutation)
        dc: Concept = op.delete_concept(
            id=concept_id
        )
        dc.__fields__('is_success')
        res = self._gql_client.execute(op)
        res = op + res

        return res.delete_concept.is_success

    def delete_concept_link(self, link_id: str) -> bool:

        op = Operation(Mutation)
        dcl: ConceptLink = op.delete_concept_link(
            id=link_id
        )
        dcl.__fields__('is_success')
        res = self._gql_client.execute(op)
        res = op + res

        return res.delete_concept_link.is_success

    def delete_concept_link_property(self, link_property_id: str) -> bool:

        op = Operation(Mutation)
        clp: ConceptProperty = op.delete_concept_link_property(
            id=link_property_id
        )
        clp.__fields__('is_success')
        res = self._gql_client.execute(op)
        res = op + res

        return res.delete_concept_link_property.is_success

    def add_concept_markers(
        self, concept_id: str, markers: List[str], perform_synchronously: Optional[bool] = None
    ) -> Concept:
        c = self.get_concept(concept_id)
        c.markers.extend(markers)
        new_markers = list(set(c.markers))
        return self.update_concept(c, markers=new_markers, perform_synchronously=perform_synchronously)

    def set_concept_markers(
        self, concept_id: str, markers: List[str], perform_synchronously: Optional[bool] = None
    ) -> Concept:
        c = self.get_concept(concept_id)
        return self.update_concept(c, markers=markers, perform_synchronously=perform_synchronously)

    def get_composite_concept(self, root_concept_id: str, composite_concept_type_id: str) -> CompositeConcept:
        op = Operation(Query)
        cc: CompositeConcept = op.composite_concept(
            root_concept_id=root_concept_id,
            composite_concept_type_id=composite_concept_type_id
        )
        self._configure_output_concept_fields(
            cc.root_concept, with_aliases=False, with_link_properties=False,
            with_links=False, with_properties=False
        )
        lwt = cc.composite_concept_type().list_widget_type()
        lwt.__fields__(*self.composite_concept_widget_type)
        lwt.columns_info.__fields__(*self.composite_concept_widget_type_columns_info)

        res = self._gql_client.execute(op)
        res = op + res
        return res.composite_concept

    def get_single_widget(
        self, root_concept_id: str, composite_concept_type_id: str, widget_type_id: str,
        limit: int = 20, offset: int = 0
    ) -> CompositeConceptWidgetRowPagination:

        op = Operation(Query)
        cc: CompositeConcept = op.composite_concept(
            root_concept_id=root_concept_id,
            composite_concept_type_id=composite_concept_type_id
        )
        psw: CompositeConceptWidgetRowPagination = cc.paginate_single_widget(
            widget_type_id=widget_type_id,
            limit=limit,
            offset=offset
        )
        psw.__fields__('total')
        self._configure_output_value_fields(psw.rows)

        res = self._gql_client.execute(op)
        res = op + res
        return res.composite_concept.paginate_single_widget

    # region Crawlers methods

    def get_crawler_start_urls(self, take: Optional[int] = None) -> Sequence[Crawler]:
        take = self.get_take_value(take)
        op = Operation(CrQuery)
        pc: CrawlerPagination = op.pagination_crawler(
            limit=take
        )
        lc = pc.list_crawler()
        lc.__fields__('start_urls')

        res = self._gql_client.execute(op)
        res = op + res
        return res.pagination_crawler.list_crawler

    # endregion

    # region Utils methods

    @check_utils_gql_client
    def create_or_get_concept_by_name(
        self, name: str, type_id: str, notes: str = None, take_first_result: bool = False,
        with_properties=False, with_links=False, with_link_properties=False
    ) -> Concept:

        if type_id:
            concept_filter_settings: ConceptFilterSettings = uas.ConceptFilterSettings(
                exact_name=name,
                concept_type_ids=[type_id]
            )
        else:
            concept_filter_settings: ConceptFilterSettings = uas.ConceptFilterSettings(
                exact_name=name
            )

        op = Operation(uas.Mutation)
        goac = op.get_or_add_concept(
            filter_settings=concept_filter_settings,
            form=uas.ConceptMutationInput(
                name=name,
                concept_type_id=type_id,
                notes=notes
            ),
            take_first_result=take_first_result
        )
        self._configure_output_concept_fields(
            goac, with_properties=with_properties, with_links=with_links,
            with_link_properties=with_link_properties
        )

        res = self._utils_gql_client.execute(op)
        res = op + res  # type: uas.Mutation

        if self.tdm_builder is not None:
            self.tdm_builder.add_concept_fact(res.get_or_add_concept)

        return res.get_or_add_concept

    # endregion

    # region tcontroller methods

    def get_pipeline_configs(
        self, with_transforms: bool = True, filter_settings: tc.PipelineConfigFilter = None,
        limit: Optional[int] = None, offset: Optional[int] = None, sort_by: tc.PipelineConfigSort = 'id',
        sort_direction: tc.SortDirection = 'ascending'
    ) -> tc.PipelineConfigList:
        op = Operation(tc.Query)
        pcl: tc.PipelineConfigList = op.pipeline_configs(
            filter=filter_settings,
            limit=limit,
            offset=offset,
            sort_by=sort_by,
            sort_direction=sort_direction
        )
        pc = pcl.pipeline_configs()
        pc.__fields__(*self.pipeline_config_fields)
        if with_transforms:
            pc.transforms().__fields__('id')
            pc.transforms().__fields__('params')
        res = self._gql_client.execute(op)
        res = op + res
        return res.pipeline_configs

    def get_pipeline_topics(
        self, filter_settings: tc.KafkaTopicFilter = None,
        limit: Optional[int] = None, offset: Optional[int] = None, sort_by: tc.KafkaTopicSort = 'topic',
        sort_direction: tc.SortDirection = 'ascending'
    ) -> tc.KafkaTopicList:
        op = Operation(tc.Query)
        ktl: tc.KafkaTopicList = op.kafka_topics(
            filter=filter_settings,
            limit=limit,
            offset=offset,
            sort_by=sort_by,
            sort_direction=sort_direction
        )
        self._configure_pipeline_topic_fields(ktl.topics())
        res = self._gql_client.execute(op)
        res = op + res
        return res.kafka_topics

    def upsert_pipeline_topic(self, topic_id: str, config_id: str, stopped: bool) -> tc.KafkaTopic:
        op = Operation(tc.Mutation)
        kt: tc.KafkaTopic = op.put_kafka_topic(
            topic=topic_id,
            pipeline=tc.PipelineSetupInput(
                pipeline_config=config_id
            ),
            stopped=stopped
        )
        self._configure_pipeline_topic_fields(kt)
        res = self._gql_client.execute(op)
        res = op + res
        return res.put_kafka_topic

    def get_failed_messages_from_topic(
        self, topic_id: str, filter_settings: tc.MessageFilter = None, offset: Optional[int] = None,
        limit: Optional[int] = None, sort_by: tc.MessageSort = 'timestamp',
        sort_direction: tc.SortDirection = 'descending'
    ) -> tc.FailedMessageList:
        op = Operation(tc.Query)
        fm: tc.FailedMessageList = op.failed_messages(
            topic=topic_id,
            filter=filter_settings,
            offset=offset,
            limit=limit,
            sort_by=sort_by,
            sort_direction=sort_direction
        )
        fm.messages().id()
        fm.messages().info().error().description()
        fm.messages().info().message()
        res = self._gql_client.execute(op)
        res = op + res
        return res.failed_messages

    def get_ok_messages_from_topic(
        self, topic_id: str, filter_settings: tc.MessageFilter = None, offset: Optional[int] = None,
        limit: Optional[int] = None, sort_by: tc.MessageSort = 'timestamp',
        sort_direction: tc.SortDirection = 'descending'
    ) -> tc.CompletedOkMessageList:
        op = Operation(tc.Query)
        om: tc.CompletedOkMessageList = op.completed_ok_messages(
            topic=topic_id,
            filter=filter_settings,
            offset=offset,
            limit=limit,
            sort_by=sort_by,
            sort_direction=sort_direction
        )
        om.messages().id()
        om.messages().info().message()
        res = self._gql_client.execute(op)
        res = op + res
        return res.completed_ok_messages

    def get_active_messages_from_topic(
        self, topic_id: str, filter_settings: tc.MessageFilter = None, offset: Optional[int] = None,
        limit: Optional[int] = None, sort_by: tc.MessageSort = 'timestamp',
        sort_direction: tc.SortDirection = 'descending'
    ) -> tc.CompletedOkMessageList:
        op = Operation(tc.Query)
        am: tc.CompletedOkMessageList = op.active_messages(
            topic=topic_id,
            filter=filter_settings,
            offset=offset,
            limit=limit,
            sort_by=sort_by,
            sort_direction=sort_direction
        )
        am.messages().id()
        am.messages().info().message()
        res = self._gql_client.execute(op)
        res = op + res
        return res.active_messages

    def retry_failed_in_topic(self, topic_id: str) -> int:
        op = Operation(tc.Mutation)
        op.retry_failed_in_topic(
            topic=topic_id
        )
        res = self._gql_client.execute(op)
        res = op + res
        return res.retry_failed_in_topic

    # endregion
