import datetime
import os
from pathlib import Path
from unittest.mock import MagicMock

import pytest
from semver import VersionInfo

V1 = VersionInfo.parse("0.0.1")


def test_pipx_path(pipx_updater):
    assert pipx_updater.is_pipx() == True


def test_brew_path(brew_updater):
    assert brew_updater.is_brew() == True


def test_local_if_v0(updater):
    updater.dir = Path("/foo/bar")
    assert updater.is_local() == True
    updater.get_current_version = MagicMock(return_value=V1)
    assert updater.is_local() == False


def test_default_path(updater):
    assert (updater.dir / "sym" / "cli").exists() == True


def test_noop_if_local(updater):
    assert updater.is_local() == True
    assert updater.should_check() == False


def test_no_manual_if_local(updater):
    updater.update = MagicMock()
    updater.manual_update()
    updater.update.assert_not_called()


def test_checks_first_time(pipx_updater):
    assert pipx_updater.should_check() == True


def test_no_check_second_time(pipx_updater):
    assert pipx_updater.should_check() == True
    assert pipx_updater.should_check() == False


def test_check_after_a_day(pipx_updater, time_travel):
    assert pipx_updater.should_check() == True
    time_travel.tick(delta=datetime.timedelta(days=1))
    assert pipx_updater.should_check() == True


def test_get_version(updater):
    assert updater.get_latest_version() is not None


def test_needs_update(pipx_updater):
    pipx_updater.get_current_version = MagicMock(return_value=V1)
    assert pipx_updater.needs_update() is True


def test_needs_no_update(pipx_updater):
    pipx_updater.get_current_version = MagicMock(return_value=V1)
    pipx_updater.get_latest_version = MagicMock(return_value=V1)
    assert pipx_updater.needs_update() is False


def test_failed_fetch(pipx_updater):
    pipx_updater.get_latest_version = MagicMock(return_value=None)
    assert pipx_updater.needs_update() is False


def test_update_pipx(pipx_updater, capture_command):
    with capture_command():
        assert pipx_updater._update() == True
    capture_command.assert_command(["pipx", "upgrade", "sym-cli"])


def test_update_pipx_failed(pipx_updater, capture_command):
    capture_command.register_output("pipx", "", exit_code=1)
    with capture_command():
        assert pipx_updater._update() == False


def test_update_brew(brew_updater, capture_command):
    with capture_command():
        assert brew_updater._update() == True
    capture_command.assert_command(["brew", "upgrade", "sym-cli"])


def test_update_brew_failed(brew_updater, capture_command):
    capture_command.register_output("brew", "", exit_code=1)
    with capture_command():
        assert brew_updater._update() == False


@pytest.mark.skipif(not os.getenv("CI"), reason="skipping real requests locally")
def test_update_other(other_updater):
    version = other_updater.get_latest_version()
    assert other_updater._update() == True
    assert (other_updater.dir / "sym").exists() == True
    assert str(version) in (other_updater.dir / "sym" / "cli" / "version.py").read_text()
