"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const url_1 = require("url");
const acm = require("@aws-cdk/aws-certificatemanager");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const route53 = require("@aws-cdk/aws-route53");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const opensearch_access_policy_1 = require("./opensearch-access-policy");
const opensearchservice_generated_1 = require("./opensearchservice.generated");
const perms = require("./perms");
/**
 * The minimum TLS version required for traffic to the domain.
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    /** Cipher suite TLS 1.0 */
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    /** Cipher suite TLS 1.2 */
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * Grant read permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read/write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read/write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Return the given named metric for this domain.
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensionsMap: {
                DomainName: this.domainName,
                ClientId: this.stack.account,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     */
    metricClusterIndexWritesBlocked(props) {
        return this.metric('ClusterIndexWritesBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatency', { statistic: 'p99', ...props });
    }
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatency', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * Provides an Amazon OpenSearch Service domain.
 */
class Domain extends DomainBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29;
        super(scope, id, {
            physicalName: props.domainName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_opensearchservice_DomainProps(props);
        const defaultInstanceType = 'r5.large.search';
        const warmDefaultInstanceType = 'ultrawarm1.medium.search';
        const dedicatedMasterType = initializeInstanceType(defaultInstanceType, (_b = props.capacity) === null || _b === void 0 ? void 0 : _b.masterNodeInstanceType);
        const dedicatedMasterCount = (_d = (_c = props.capacity) === null || _c === void 0 ? void 0 : _c.masterNodes) !== null && _d !== void 0 ? _d : 0;
        const dedicatedMasterEnabled = cdk.Token.isUnresolved(dedicatedMasterCount) ? true : dedicatedMasterCount > 0;
        const instanceType = initializeInstanceType(defaultInstanceType, (_e = props.capacity) === null || _e === void 0 ? void 0 : _e.dataNodeInstanceType);
        const instanceCount = (_g = (_f = props.capacity) === null || _f === void 0 ? void 0 : _f.dataNodes) !== null && _g !== void 0 ? _g : 1;
        const warmType = initializeInstanceType(warmDefaultInstanceType, (_h = props.capacity) === null || _h === void 0 ? void 0 : _h.warmInstanceType);
        const warmCount = (_k = (_j = props.capacity) === null || _j === void 0 ? void 0 : _j.warmNodes) !== null && _k !== void 0 ? _k : 0;
        const warmEnabled = cdk.Token.isUnresolved(warmCount) ? true : warmCount > 0;
        const availabilityZoneCount = (_m = (_l = props.zoneAwareness) === null || _l === void 0 ? void 0 : _l.availabilityZoneCount) !== null && _m !== void 0 ? _m : 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = (_p = (_o = props.zoneAwareness) === null || _o === void 0 ? void 0 : _o.enabled) !== null && _p !== void 0 ? _p : ((_q = props.zoneAwareness) === null || _q === void 0 ? void 0 : _q.availabilityZoneCount) != null;
        let securityGroups;
        let subnets;
        if (props.vpc) {
            subnets = selectSubnets(props.vpc, (_r = props.vpcSubnets) !== null && _r !== void 0 ? _r : [{ subnetType: ec2.SubnetType.PRIVATE }]);
            securityGroups = (_s = props.securityGroups) !== null && _s !== void 0 ? _s : [new ec2.SecurityGroup(this, 'SecurityGroup', {
                    vpc: props.vpc,
                    description: `Security group for domain ${this.node.id}`,
                })];
            this._connections = new ec2.Connections({ securityGroups });
        }
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (subnets && zoneAwarenessEnabled && new Set(subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        if ([dedicatedMasterType, instanceType, warmType].some(t => (!cdk.Token.isUnresolved(t) && !t.endsWith('.search')))) {
            throw new Error('Master, data and UltraWarm node instance types must end with ".search".');
        }
        if (!cdk.Token.isUnresolved(warmType) && !warmType.startsWith('ultrawarm')) {
            throw new Error('UltraWarm node instance type must start with "ultrawarm".');
        }
        const unsignedBasicAuthEnabled = (_t = props.useUnsignedBasicAuth) !== null && _t !== void 0 ? _t : false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (((_u = props.encryptionAtRest) === null || _u === void 0 ? void 0 : _u.enabled) == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.AnyPrincipal()],
            resources: [cdk.Lazy.string({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = (_v = props.fineGrainedAccessControl) === null || _v === void 0 ? void 0 : _v.masterUserArn;
        const masterUserNameProps = (_w = props.fineGrainedAccessControl) === null || _w === void 0 ? void 0 : _w.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps !== null && masterUserNameProps !== void 0 ? masterUserNameProps : 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn !== null && masterUserArn !== void 0 ? masterUserArn : masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = (_x = props.fineGrainedAccessControl) === null || _x === void 0 ? void 0 : _x.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp !== null && masterUserPasswordProp !== void 0 ? masterUserPasswordProp : createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = (_z = (_y = props.encryptionAtRest) === null || _y === void 0 ? void 0 : _y.enabled) !== null && _z !== void 0 ? _z : (((_0 = props.encryptionAtRest) === null || _0 === void 0 ? void 0 : _0.kmsKey) != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = (_1 = props.nodeToNodeEncryption) !== null && _1 !== void 0 ? _1 : unsignedBasicAuthEnabled;
        const volumeSize = (_3 = (_2 = props.ebs) === null || _2 === void 0 ? void 0 : _2.volumeSize) !== null && _3 !== void 0 ? _3 : 10;
        const volumeType = (_5 = (_4 = props.ebs) === null || _4 === void 0 ? void 0 : _4.volumeType) !== null && _5 !== void 0 ? _5 : ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = (_7 = (_6 = props.ebs) === null || _6 === void 0 ? void 0 : _6.enabled) !== null && _7 !== void 0 ? _7 : true;
        const enforceHttps = (_8 = props.enforceHttps) !== null && _8 !== void 0 ? _8 : unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryDatanodeInstanceType(...instanceTypes) {
            return instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch/OpenSearch version, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/features-by-version.html
        const { versionNum: versionNum, isElasticsearchVersion } = parseVersion(props.version);
        if (isElasticsearchVersion) {
            if (versionNum <= 7.7 &&
                ![
                    1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                    6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                    7.7,
                ].includes(versionNum)) {
                throw new Error(`Unknown Elasticsearch version: ${versionNum}`);
            }
            if (versionNum < 5.1) {
                if ((_9 = props.logging) === null || _9 === void 0 ? void 0 : _9.appLogEnabled) {
                    throw new Error('Error logs publishing requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if ((_10 = props.encryptionAtRest) === null || _10 === void 0 ? void 0 : _10.enabled) {
                    throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if (props.cognitoDashboardsAuth != null) {
                    throw new Error('Cognito authentication for OpenSearch Dashboards requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                    throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.0) {
                if (props.nodeToNodeEncryption) {
                    throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.7) {
                if (unsignedBasicAuthEnabled) {
                    throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later or OpenSearch version 1.0 or later.');
                }
                if (advancedSecurityEnabled) {
                    throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.8 && warmEnabled) {
                throw new Error('UltraWarm requires Elasticsearch version 6.8 or later or OpenSearch version 1.0 or later.');
            }
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/supported-instance-types.html
        if (isSomeInstanceType('i3', 'r6gd') && ebsEnabled) {
            throw new Error('I3 and R6GD instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && !(isElasticsearchVersion && versionNum <= 2.3)) {
            throw new Error('The t2.micro.search instance type supports only Elasticsearch versions 1.5 and 2.3.');
        }
        if (isSomeInstanceType('t2', 't3') && warmEnabled) {
            throw new Error('T2 and T3 instance types do not support UltraWarm storage.');
        }
        // Only R3, I3 and r6gd support instance storage, per
        // https://aws.amazon.com/opensearch-service/pricing/
        if (!ebsEnabled && !isEveryDatanodeInstanceType('r3', 'i3', 'r6gd')) {
            throw new Error('EBS volumes are required when using instance types other than r3, i3 or r6gd.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        // Validate fine grained access control enabled for audit logs, per
        // https://aws.amazon.com/about-aws/whats-new/2020/09/elasticsearch-audit-logs-now-available-on-amazon-elasticsearch-service/
        if (((_11 = props.logging) === null || _11 === void 0 ? void 0 : _11.auditLogEnabled) && !advancedSecurityEnabled) {
            throw new Error('Fine-grained access control is required when audit logs publishing is enabled.');
        }
        // Validate UltraWarm requirement for dedicated master nodes, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/ultrawarm.html
        if (warmEnabled && !dedicatedMasterEnabled) {
            throw new Error('Dedicated master node is required when UltraWarm storage is enabled.');
        }
        let cfnVpcOptions;
        if (securityGroups && subnets) {
            cfnVpcOptions = {
                securityGroupIds: securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if ((_12 = props.logging) === null || _12 === void 0 ? void 0 : _12.slowSearchLogEnabled) {
            this.slowSearchLogGroup = (_13 = props.logging.slowSearchLogGroup) !== null && _13 !== void 0 ? _13 : new logs.LogGroup(this, 'SlowSearchLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if ((_14 = props.logging) === null || _14 === void 0 ? void 0 : _14.slowIndexLogEnabled) {
            this.slowIndexLogGroup = (_15 = props.logging.slowIndexLogGroup) !== null && _15 !== void 0 ? _15 : new logs.LogGroup(this, 'SlowIndexLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if ((_16 = props.logging) === null || _16 === void 0 ? void 0 : _16.appLogEnabled) {
            this.appLogGroup = (_17 = props.logging.appLogGroup) !== null && _17 !== void 0 ? _17 : new logs.LogGroup(this, 'AppLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.appLogGroup);
        }
        ;
        if ((_18 = props.logging) === null || _18 === void 0 ? void 0 : _18.auditLogEnabled) {
            this.auditLogGroup = (_19 = props.logging.auditLogGroup) !== null && _19 !== void 0 ? _19 : new logs.LogGroup(this, 'AuditLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.auditLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, `ESLogGroupPolicy${this.node.addr}`, {
                // create a cloudwatch logs resource policy name that is unique to this domain instance
                policyName: `ESLogPolicy${this.node.addr}`,
                policyStatements: [logPolicyStatement],
            });
        }
        const logPublishing = {};
        if (this.appLogGroup) {
            logPublishing.ES_APPLICATION_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.appLogGroup.logGroupArn,
            };
        }
        if (this.slowSearchLogGroup) {
            logPublishing.SEARCH_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowSearchLogGroup.logGroupArn,
            };
        }
        if (this.slowIndexLogGroup) {
            logPublishing.INDEX_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowIndexLogGroup.logGroupArn,
            };
        }
        if (this.auditLogGroup) {
            logPublishing.AUDIT_LOGS = {
                enabled: this.auditLogGroup != null,
                cloudWatchLogsLogGroupArn: (_20 = this.auditLogGroup) === null || _20 === void 0 ? void 0 : _20.logGroupArn,
            };
        }
        let customEndpointCertificate;
        if (props.customEndpoint) {
            if (props.customEndpoint.certificate) {
                customEndpointCertificate = props.customEndpoint.certificate;
            }
            else {
                customEndpointCertificate = new acm.Certificate(this, 'CustomEndpointCertificate', {
                    domainName: props.customEndpoint.domainName,
                    validation: props.customEndpoint.hostedZone ? acm.CertificateValidation.fromDns(props.customEndpoint.hostedZone) : undefined,
                });
            }
        }
        // Create the domain
        this.domain = new opensearchservice_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            engineVersion: props.version.version,
            clusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                warmEnabled: warmEnabled
                    ? warmEnabled
                    : undefined,
                warmCount: warmEnabled
                    ? warmCount
                    : undefined,
                warmType: warmEnabled
                    ? warmType
                    : undefined,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? (_21 = props.ebs) === null || _21 === void 0 ? void 0 : _21.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? (_23 = (_22 = props.encryptionAtRest) === null || _22 === void 0 ? void 0 : _22.kmsKey) === null || _23 === void 0 ? void 0 : _23.keyId : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: logPublishing,
            cognitoOptions: {
                enabled: props.cognitoDashboardsAuth != null,
                identityPoolId: (_24 = props.cognitoDashboardsAuth) === null || _24 === void 0 ? void 0 : _24.identityPoolId,
                roleArn: (_25 = props.cognitoDashboardsAuth) === null || _25 === void 0 ? void 0 : _25.role.roleArn,
                userPoolId: (_26 = props.cognitoDashboardsAuth) === null || _26 === void 0 ? void 0 : _26.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: (_27 = props.tlsSecurityPolicy) !== null && _27 !== void 0 ? _27 : TLSSecurityPolicy.TLS_1_0,
                ...props.customEndpoint && {
                    customEndpointEnabled: true,
                    customEndpoint: props.customEndpoint.domainName,
                    customEndpointCertificateArn: customEndpointCertificate.certificateArn,
                },
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: (_28 = this.masterUserPassword) === null || _28 === void 0 ? void 0 : _28.toString(),
                    },
                }
                : undefined,
            advancedOptions: props.advancedOptions,
        });
        this.domain.applyRemovalPolicy(props.removalPolicy);
        if (props.enableVersionUpgrade) {
            this.domain.cfnOptions.updatePolicy = {
                ...this.domain.cfnOptions.updatePolicy,
                enableVersionUpgrade: props.enableVersionUpgrade,
            };
        }
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            if (!cdk.Token.isUnresolved(props.domainName)) {
                // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/configuration-api.html#configuration-api-datatypes-domainname
                if (!props.domainName.match(/^[a-z0-9\-]+$/)) {
                    throw new Error(`Invalid domainName '${props.domainName}'. Valid characters are a-z (lowercase only), 0-9, and – (hyphen).`);
                }
                if (props.domainName.length < 3 || props.domainName.length > 28) {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must be between 3 and 28 characters`);
                }
                if (props.domainName[0] < 'a' || props.domainName[0] > 'z') {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must start with a lowercase letter`);
                }
            }
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainId = this.domain.getAtt('Id').toString();
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        if ((_29 = props.customEndpoint) === null || _29 === void 0 ? void 0 : _29.hostedZone) {
            new route53.CnameRecord(this, 'CnameRecord', {
                recordName: props.customEndpoint.domainName,
                zone: props.customEndpoint.hostedZone,
                domainName: this.domainEndpoint,
            });
        }
        this.encryptionAtRestOptions = props.encryptionAtRest;
        if (props.accessPolicies) {
            this.addAccessPolicies(...props.accessPolicies);
        }
        if (unsignedBasicAuthEnabled) {
            this.addAccessPolicies(unsignedAccessPolicy);
        }
    }
    /**
     * Creates a domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * Creates a domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     */
    static fromDomainAttributes(scope, id, attrs) {
        var _b;
        jsiiDeprecationWarnings._aws_cdk_aws_opensearchservice_DomainAttributes(attrs);
        const { domainArn, domainEndpoint } = attrs;
        const domainName = (_b = cdk.Stack.of(scope).splitArn(domainArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName) !== null && _b !== void 0 ? _b : extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainId = domainName;
                this.domainEndpoint = domainEndpoint.replace(/^https?:\/\//, '');
            }
        };
    }
    /**
     * Manages network connections to the domain. This will throw an error in case the domain
     * is not placed inside a VPC.
     */
    get connections() {
        if (!this._connections) {
            throw new Error("Connections are only available on VPC enabled domains. Use the 'vpc' property to place a domain inside a VPC");
        }
        return this._connections;
    }
    /**
     * Add policy statements to the domain access policy
     */
    addAccessPolicies(...accessPolicyStatements) {
        var _b;
        if (accessPolicyStatements.length > 0) {
            if (!this.accessPolicy) {
                // Only create the custom resource after there are statements to set.
                this.accessPolicy = new opensearch_access_policy_1.OpenSearchAccessPolicy(this, 'AccessPolicy', {
                    domainName: this.domainName,
                    domainArn: this.domainArn,
                    accessPolicies: accessPolicyStatements,
                });
                if ((_b = this.encryptionAtRestOptions) === null || _b === void 0 ? void 0 : _b.kmsKey) {
                    // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/encryption-at-rest.html
                    // these permissions are documented as required during domain creation.
                    // while not strictly documented for updates as well, it stands to reason that an update
                    // operation might require these in case the cluster uses a kms key.
                    // empircal evidence shows this is indeed required: https://github.com/aws/aws-cdk/issues/11412
                    this.accessPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                        actions: ['kms:List*', 'kms:Describe*', 'kms:CreateGrant'],
                        resources: [this.encryptionAtRestOptions.kmsKey.keyArn],
                        effect: iam.Effect.ALLOW,
                    }));
                }
            }
            else {
                this.accessPolicy.addAccessPolicies(...accessPolicyStatements);
            }
        }
    }
}
exports.Domain = Domain;
_a = JSII_RTTI_SYMBOL_1;
Domain[_a] = { fqn: "@aws-cdk/aws-opensearchservice.Domain", version: "1.151.0" };
/**
 * Given an Amazon OpenSearch Service domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Amazon OpenSearch Service domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an engine version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The engine version object
 */
function parseVersion(version) {
    const elasticsearchPrefix = 'Elasticsearch_';
    const openSearchPrefix = 'OpenSearch_';
    const isElasticsearchVersion = version.version.startsWith(elasticsearchPrefix);
    const versionStr = isElasticsearchVersion
        ? version.version.substring(elasticsearchPrefix.length)
        : version.version.substring(openSearchPrefix.length);
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid engine version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return { versionNum: parseFloat(versionStr), isElasticsearchVersion };
        }
        else {
            return { versionNum: parseFloat(versionStr.substring(0, secondDot)), isElasticsearchVersion };
        }
    }
    catch (error) {
        throw new Error(`Invalid engine version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
function selectSubnets(vpc, vpcSubnets) {
    const selected = [];
    for (const selection of vpcSubnets) {
        selected.push(...vpc.selectSubnets(selection).subnets);
    }
    return selected;
}
/**
 * Initializes an instance type.
 *
 * @param defaultInstanceType Default instance type which is used if no instance type is provided
 * @param instanceType Instance type
 * @returns Instance type in lowercase (if provided) or default instance type
 */
function initializeInstanceType(defaultInstanceType, instanceType) {
    if (instanceType) {
        return cdk.Token.isUnresolved(instanceType) ? instanceType : instanceType.toLowerCase();
    }
    else {
        return defaultInstanceType;
    }
}
//# sourceMappingURL=data:application/json;base64,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