""" The functional module of the scan demo
Copyright Nanosurf AG 2021
License - MIT
"""
import numpy as np
from PySide2.QtCore import Signal

from app import module_base
from app import app_common
from app.app_base import ApplicationBase
from modules.scan_module import imaging_task, settings

class ScanModule(module_base.ModuleBase):

    sig_work_start_requested = Signal()
    sig_work_stop_requested = Signal()
    sig_work_active = Signal()
    sig_work_done = Signal()
    sig_new_data_available = Signal()
    sig_data_invalid = Signal()

    """ Initialization functions of the module """

    def __init__(self, app: ApplicationBase, gui):
        super().__init__(app, gui)
        """ Prepare here module settings which are stored and loaded from file by the app framework """
        self.settings = settings.ScanSettings()
        self.result = settings.ScanResults()

    def do_start(self):
        """ This function is called once at startup of application
            Initialize here all module specific values.
        """
        self.setup_imaging_task()
        self.connect_to_properties()

    def do_stop(self):
        """ This function is called at module shutdown"""
        if self.imaging_task.is_running():
            self.logger.info("Wait until worker thread has ended...")
            self.imaging_task.stop(wait=True)

    def connect_to_properties(self):
        """ Connect action functions to settings 
            The connected functions are called whenever a setting is changed (e.g. by GUI elements)
        """
        self.settings.image_size.sig_value_changed.connect(self.update_worker_parameter)
        self.settings.time_per_line.sig_value_changed.connect(self.update_worker_parameter)
        self.settings.points_per_line.sig_value_changed.connect(self.update_worker_parameter)
        self.settings.channel_id.sig_value_changed.connect(self.update_worker_parameter)
        self.settings.show_backward.sig_value_changed.connect(self.update_worker_parameter)

    def setup_imaging_task(self):
        """ Create the background worker task and connect to its event """
        self.imaging_task = imaging_task.ScanFrameWorker(self)
        self.imaging_task.sig_started.connect(self._on_sig_worker_started)
        self.imaging_task.sig_finished.connect(self._on_sig_worker_finished)
        self.imaging_task.sig_new_data.connect(self._on_sig_worker_new_data)
        self.imaging_task.sig_tick.connect(self._on_sig_worker_tick)

    """ Now the busines logic of the module """
        
    def start_worker(self):
        if not self.imaging_task.is_running():
            self.update_worker_parameter()
            self.sig_work_start_requested.emit()
            self.imaging_task.start()

    def stop_worker(self):
        if self.imaging_task.is_running():
            self.sig_work_stop_requested.emit()
            self.imaging_task.stop(wait=True)
    
    def is_worker_busy(self) -> bool:
        return self.imaging_task.is_running()

    def get_result(self) -> settings.ScanResults:
        return self.result

    def get_worker_result(self) -> imaging_task.ScanData:
        return self.imaging_task.get_result()

    def update_worker_parameter(self):
        self.imaging_task.par_image_size = self.settings.image_size.value
        self.imaging_task.par_points_per_line = self.settings.points_per_line.value
        self.imaging_task.par_time_per_line = self.settings.time_per_line.value
        self.imaging_task.par_channel_id = self.settings.channel_id.value

    def do_analysis(self):
        data = self.imaging_task.get_result()
        result_valid = True

        if result_valid:
            self.sig_new_data_available.emit()
        else:
            self.app.show_message(f"Could not analyse data", app_common.MsgType.Error) 
            self.sig_data_invalid.emit()

    """ worker thread state handling """

    def _on_sig_worker_started(self):
        self.app.show_message("Working ...") 
        self.logger.info("Thread started to work")
        self.sig_data_invalid.emit()
        self.sig_work_active.emit()

    def _on_sig_worker_finished(self):
        self.sig_work_done.emit()
        if not self.imaging_task.is_aborted():
            self.app.show_message("Work done") 
            self.logger.info("Thread done ")
            self.do_analysis()
        else:
            self.app.show_message("Work aborted") 
            self.logger.info("Thread aborted ")
 
    def _on_sig_worker_tick(self):
        self.app.show_message("Tick") 

    def _on_sig_worker_new_data(self):
        self.sig_new_data_available.emit()
        #self.do_analysis()
