"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const SECURITY_GROUP_SYMBOL = Symbol.for('@aws-cdk/iam.SecurityGroup');
const SECURITY_GROUP_DISABLE_INLINE_RULES_CONTEXT_KEY = '@aws-cdk/aws-ec2.securityGroupDisableInlineRules';
/**
 * A SecurityGroup that is not created in this template
 */
class SecurityGroupBase extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        /**
         * Whether the rule can be inlined into a SecurityGroup or not.
         *
         * @stability stable
         */
        this.canInlineRule = false;
        /**
         * @stability stable
         */
        this.connections = new connections_1.Connections({ securityGroups: [this] });
        Object.defineProperty(this, SECURITY_GROUP_SYMBOL, { value: true });
    }
    /**
     * Return whether the indicated object is a security group.
     *
     * @stability stable
     */
    static isSecurityGroup(x) {
        return SECURITY_GROUP_SYMBOL in x;
    }
    /**
     * A unique identifier for this connection peer.
     *
     * @stability stable
     */
    get uniqueId() {
        return core_1.Names.nodeUniqueId(this.node);
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'from', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupIngress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toIngressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `to ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'to', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupEgress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toEgressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    /**
     * Produce the ingress rule JSON for the given connection.
     *
     * @stability stable
     */
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    /**
     * Produce the egress rule JSON for the given connection.
     *
     * @stability stable
     */
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
}
/**
 * Determine where to parent a new ingress/egress rule
 *
 * A SecurityGroup rule is parented under the group it's related to, UNLESS
 * we're in a cross-stack scenario with another Security Group. In that case,
 * we respect the 'remoteRule' flag and will parent under the other security
 * group.
 *
 * This is necessary to avoid cyclic dependencies between stacks, since both
 * ingress and egress rules will reference both security groups, and a naive
 * parenting will lead to the following situation:
 *
 *   ╔════════════════════╗         ╔════════════════════╗
 *   ║  ┌───────────┐     ║         ║    ┌───────────┐   ║
 *   ║  │  GroupA   │◀────╬─┐   ┌───╬───▶│  GroupB   │   ║
 *   ║  └───────────┘     ║ │   │   ║    └───────────┘   ║
 *   ║        ▲           ║ │   │   ║          ▲         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║  ┌───────────┐     ║ └───┼───╬────┌───────────┐   ║
 *   ║  │  EgressA  │─────╬─────┘   ║    │ IngressB  │   ║
 *   ║  └───────────┘     ║         ║    └───────────┘   ║
 *   ║                    ║         ║                    ║
 *   ╚════════════════════╝         ╚════════════════════╝
 *
 * By having the ability to switch the parent, we avoid the cyclic reference by
 * keeping all rules in a single stack.
 *
 * If this happens, we also have to change the construct ID, because
 * otherwise we might have two objects with the same ID if we have
 * multiple reversed security group relationships.
 *
 *   ╔═══════════════════════════════════╗
 *   ║┌───────────┐                      ║
 *   ║│  GroupB   │                      ║
 *   ║└───────────┘                      ║
 *   ║      ▲                            ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├────"from A"──│ IngressB  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├─────"to B"───│  EgressA  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      └─────"to B"───│  EgressC  │ ║  <-- oops
 *   ║                     └───────────┘ ║
 *   ╚═══════════════════════════════════╝
 */
function determineRuleScope(group, peer, connection, fromTo, remoteRule) {
    if (remoteRule && SecurityGroupBase.isSecurityGroup(peer) && differentStacks(group, peer)) {
        // Reversed
        const reversedFromTo = fromTo === 'from' ? 'to' : 'from';
        return [peer, `${group.uniqueId}:${connection} ${reversedFromTo}`];
    }
    else {
        // Regular (do old ID escaping to in order to not disturb existing deployments)
        return [group, `${fromTo} ${renderPeer(peer)}:${connection}`.replace('/', '_')];
    }
}
function renderPeer(peer) {
    return core_1.Token.isUnresolved(peer.uniqueId) ? '{IndirectPeer}' : peer.uniqueId;
}
function differentStacks(group1, group2) {
    return core_1.Stack.of(group1) !== core_1.Stack.of(group2);
}
/**
 * Creates an Amazon EC2 security group within a VPC.
 *
 * Security Groups act like a firewall with a set of rules, and are associated
 * with any AWS resource that has or creates Elastic Network Interfaces (ENIs).
 * A typical example of a resource that has a security group is an Instance (or
 * Auto Scaling Group of instances)
 *
 * If you are defining new infrastructure in CDK, there is a good chance you
 * won't have to interact with this class at all. Like IAM Roles, Security
 * Groups need to exist to control access between AWS resources, but CDK will
 * automatically generate and populate them with least-privilege permissions
 * for you so you can concentrate on your business logic.
 *
 * All Constructs that require Security Groups will create one for you if you
 * don't specify one at construction. After construction, you can selectively
 * allow connections to and between constructs via--for example-- the `instance.connections`
 * object. Think of it as "allowing connections to your instance", rather than
 * "adding ingress rules a security group". See the [Allowing
 * Connections](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-ec2-readme.html#allowing-connections)
 * section in the library documentation for examples.
 *
 * Direct manipulation of the Security Group through `addIngressRule` and
 * `addEgressRule` is possible, but mutation through the `.connections` object
 * is recommended. If you peer two constructs with security groups this way,
 * appropriate rules will be created in both.
 *
 * If you have an existing security group you want to use in your CDK application,
 * you would import it like this:
 *
 * ```ts
 * const securityGroup = SecurityGroup.fromSecurityGroupId(this, 'SG', 'sg-12345', {
 *    mutable: false
 * });
 * ```
 *
 * @stability stable
 */
class SecurityGroup extends SecurityGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.securityGroupName,
        });
        this.directIngressRules = [];
        this.directEgressRules = [];
        const groupDescription = props.description || this.node.path;
        this.allowAllOutbound = props.allowAllOutbound !== false;
        this.disableInlineRules = props.disableInlineRules !== undefined ?
            !!props.disableInlineRules :
            !!this.node.tryGetContext(SECURITY_GROUP_DISABLE_INLINE_RULES_CONTEXT_KEY);
        this.securityGroup = new ec2_generated_1.CfnSecurityGroup(this, 'Resource', {
            groupName: this.physicalName,
            groupDescription,
            securityGroupIngress: core_1.Lazy.any({ produce: () => this.directIngressRules }, { omitEmptyArray: true }),
            securityGroupEgress: core_1.Lazy.any({ produce: () => this.directEgressRules }, { omitEmptyArray: true }),
            vpcId: props.vpc.vpcId,
        });
        this.securityGroupId = this.securityGroup.attrGroupId;
        this.securityGroupVpcId = this.securityGroup.attrVpcId;
        this.securityGroupName = this.securityGroup.ref;
        this.addDefaultEgressRule();
    }
    /**
     * Look up a security group by id.
     *
     * @stability stable
     */
    static fromLookup(scope, id, securityGroupId) {
        if (core_1.Token.isUnresolved(securityGroupId)) {
            throw new Error('All arguments to look up a security group must be concrete (no Tokens)');
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.SECURITY_GROUP_PROVIDER,
            props: { securityGroupId },
            dummyValue: {
                securityGroupId: 'sg-12345',
                allowAllOutbound: true,
            },
        }).value;
        return SecurityGroup.fromSecurityGroupId(scope, id, attributes.securityGroupId, {
            allowAllOutbound: attributes.allowAllOutbound,
            mutable: true,
        });
    }
    /**
     * Import an existing security group into this app.
     *
     * This method will assume that the Security Group has a rule in it which allows
     * all outbound traffic, and so will not add egress rules to the imported Security
     * Group (only ingress rules).
     *
     * If your existing Security Group needs to have egress rules added, pass the
     * `allowAllOutbound: false` option on import.
     *
     * @stability stable
     */
    static fromSecurityGroupId(scope, id, securityGroupId, options = {}) {
        class MutableImport extends SecurityGroupBase {
            constructor() {
                var _b;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_b = options.allowAllOutbound) !== null && _b !== void 0 ? _b : true;
            }
            addEgressRule(peer, connection, description, remoteRule) {
                // Only if allowAllOutbound has been disabled
                if (options.allowAllOutbound === false) {
                    super.addEgressRule(peer, connection, description, remoteRule);
                }
            }
        }
        class ImmutableImport extends SecurityGroupBase {
            constructor() {
                var _b;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_b = options.allowAllOutbound) !== null && _b !== void 0 ? _b : true;
            }
            addEgressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
            addIngressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
        }
        return options.mutable !== false
            ? new MutableImport(scope, id)
            : new ImmutableImport(scope, id);
    }
    /**
     * Add an ingress rule for the current security group.
     *
     * `remoteRule` controls where the Rule object is created if the peer is also a
     * securityGroup and they are in different stack. If false (default) the
     * rule object is created under the current SecurityGroup object. If true and the
     * peer is also a SecurityGroup, the rule object is created under the remote
     * SecurityGroup object.
     *
     * @stability stable
     */
    addIngressRule(peer, connection, description, remoteRule) {
        if (!peer.canInlineRule || !connection.canInlineRule || this.disableInlineRules) {
            super.addIngressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        this.addDirectIngressRule({
            ...peer.toIngressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        });
    }
    /**
     * Add an egress rule for the current security group.
     *
     * `remoteRule` controls where the Rule object is created if the peer is also a
     * securityGroup and they are in different stack. If false (default) the
     * rule object is created under the current SecurityGroup object. If true and the
     * peer is also a SecurityGroup, the rule object is created under the remote
     * SecurityGroup object.
     *
     * @stability stable
     */
    addEgressRule(peer, connection, description, remoteRule) {
        if (this.allowAllOutbound) {
            // In the case of "allowAllOutbound", we don't add any more rules. There
            // is only one rule which allows all traffic and that subsumes any other
            // rule.
            if (!remoteRule) { // Warn only if addEgressRule() was explicitely called
                core_1.Annotations.of(this).addWarning('Ignoring Egress rule since \'allowAllOutbound\' is set to true; To add customize rules, set allowAllOutbound=false on the SecurityGroup');
            }
            return;
        }
        else {
            // Otherwise, if the bogus rule exists we can now remove it because the
            // presence of any other rule will get rid of EC2's implicit "all
            // outbound" rule anyway.
            this.removeNoTrafficRule();
        }
        if (!peer.canInlineRule || !connection.canInlineRule || this.disableInlineRules) {
            super.addEgressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const rule = {
            ...peer.toEgressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        };
        if (isAllTrafficRule(rule)) {
            // We cannot allow this; if someone adds the rule in this way, it will be
            // removed again if they add other rules. We also can't automatically switch
            // to "allOutbound=true" mode, because we might have already emitted
            // EgressRule objects (which count as rules added later) and there's no way
            // to recall those. Better to prevent this for now.
            throw new Error('Cannot add an "all traffic" egress rule in this way; set allowAllOutbound=true on the SecurityGroup instead.');
        }
        this.addDirectEgressRule(rule);
    }
    /**
     * Add a direct ingress rule
     */
    addDirectIngressRule(rule) {
        if (!this.hasIngressRule(rule)) {
            this.directIngressRules.push(rule);
        }
    }
    /**
     * Return whether the given ingress rule exists on the group
     */
    hasIngressRule(rule) {
        return this.directIngressRules.findIndex(r => ingressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add a direct egress rule
     */
    addDirectEgressRule(rule) {
        if (!this.hasEgressRule(rule)) {
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Return whether the given egress rule exists on the group
     */
    hasEgressRule(rule) {
        return this.directEgressRules.findIndex(r => egressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add the default egress rule to the securityGroup
     *
     * This depends on allowAllOutbound:
     *
     * - If allowAllOutbound is true, we *TECHNICALLY* don't need to do anything, because
     *   EC2 is going to create this default rule anyway. But, for maximum readability
     *   of the template, we will add one anyway.
     * - If allowAllOutbound is false, we add a bogus rule that matches no traffic in
     *   order to get rid of the default "all outbound" rule that EC2 creates by default.
     *   If other rules happen to get added later, we remove the bogus rule again so
     *   that it doesn't clutter up the template too much (even though that's not
     *   strictly necessary).
     */
    addDefaultEgressRule() {
        if (this.disableInlineRules) {
            const peer = this.allowAllOutbound ? ALL_TRAFFIC_PEER : NO_TRAFFIC_PEER;
            const port = this.allowAllOutbound ? ALL_TRAFFIC_PORT : NO_TRAFFIC_PORT;
            const description = this.allowAllOutbound ? ALLOW_ALL_RULE.description : MATCH_NO_TRAFFIC.description;
            super.addEgressRule(peer, port, description, false);
        }
        else {
            const rule = this.allowAllOutbound ? ALLOW_ALL_RULE : MATCH_NO_TRAFFIC;
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Remove the bogus rule if it exists
     */
    removeNoTrafficRule() {
        if (this.disableInlineRules) {
            const [scope, id] = determineRuleScope(this, NO_TRAFFIC_PEER, NO_TRAFFIC_PORT, 'to', false);
            scope.node.tryRemoveChild(id);
        }
        else {
            const i = this.directEgressRules.findIndex(r => egressRulesEqual(r, MATCH_NO_TRAFFIC));
            if (i > -1) {
                this.directEgressRules.splice(i, 1);
            }
        }
    }
}
exports.SecurityGroup = SecurityGroup;
_a = JSII_RTTI_SYMBOL_1;
SecurityGroup[_a] = { fqn: "@aws-cdk/aws-ec2.SecurityGroup", version: "1.110.1" };
/**
 * Egress rule that purposely matches no traffic
 *
 * This is used in order to disable the "all traffic" default of Security Groups.
 *
 * No machine can ever actually have the 255.255.255.255 IP address, but
 * in order to lock it down even more we'll restrict to a nonexistent
 * ICMP traffic type.
 */
const MATCH_NO_TRAFFIC = {
    cidrIp: '255.255.255.255/32',
    description: 'Disallow all traffic',
    ipProtocol: 'icmp',
    fromPort: 252,
    toPort: 86,
};
const NO_TRAFFIC_PEER = peer_1.Peer.ipv4(MATCH_NO_TRAFFIC.cidrIp);
const NO_TRAFFIC_PORT = port_1.Port.icmpTypeAndCode(MATCH_NO_TRAFFIC.fromPort, MATCH_NO_TRAFFIC.toPort);
/**
 * Egress rule that matches all traffic
 */
const ALLOW_ALL_RULE = {
    cidrIp: '0.0.0.0/0',
    description: 'Allow all outbound traffic by default',
    ipProtocol: '-1',
};
const ALL_TRAFFIC_PEER = peer_1.Peer.anyIpv4();
const ALL_TRAFFIC_PORT = port_1.Port.allTraffic();
/**
 * Compare two ingress rules for equality the same way CloudFormation would (discarding description)
 */
function ingressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.sourceSecurityGroupId === b.sourceSecurityGroupId
        && a.sourceSecurityGroupName === b.sourceSecurityGroupName
        && a.sourceSecurityGroupOwnerId === b.sourceSecurityGroupOwnerId;
}
/**
 * Compare two egress rules for equality the same way CloudFormation would (discarding description)
 */
function egressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.destinationPrefixListId === b.destinationPrefixListId
        && a.destinationSecurityGroupId === b.destinationSecurityGroupId;
}
/**
 * Whether this rule refers to all traffic
 */
function isAllTrafficRule(rule) {
    return rule.cidrIp === '0.0.0.0/0' && rule.ipProtocol === '-1';
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2VjdXJpdHktZ3JvdXAuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzZWN1cml0eS1ncm91cC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDJEQUEyRDtBQUMzRCx3Q0FBNEg7QUFHNUgsK0NBQTRDO0FBQzVDLG1EQUFvRztBQUNwRyxpQ0FBcUM7QUFDckMsaUNBQThCO0FBRzlCLE1BQU0scUJBQXFCLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0FBRXZFLE1BQU0sK0NBQStDLEdBQUcsa0RBQWtELENBQUM7QUFpQjNHOztHQUVHO0FBQ0gsTUFBZSxpQkFBa0IsU0FBUSxlQUFRO0lBYS9DLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBcUI7UUFDN0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztRQUxWLGtCQUFhLEdBQUcsS0FBSyxDQUFDOzs7O1FBQ3RCLGdCQUFXLEdBQWdCLElBQUkseUJBQVcsQ0FBQyxFQUFFLGNBQWMsRUFBRSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztRQU1yRixNQUFNLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQ3RFLENBQUM7Ozs7OztJQWZNLE1BQU0sQ0FBQyxlQUFlLENBQUMsQ0FBTTtRQUNsQyxPQUFPLHFCQUFxQixJQUFJLENBQUMsQ0FBQztJQUNwQyxDQUFDOzs7Ozs7SUFlRCxJQUFXLFFBQVE7UUFDakIsT0FBTyxZQUFLLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRU0sY0FBYyxDQUFDLElBQVcsRUFBRSxVQUFnQixFQUFFLFdBQW9CLEVBQUUsVUFBb0I7UUFDN0YsSUFBSSxXQUFXLEtBQUssU0FBUyxFQUFFO1lBQzdCLFdBQVcsR0FBRyxRQUFRLElBQUksQ0FBQyxRQUFRLElBQUksVUFBVSxFQUFFLENBQUM7U0FDckQ7UUFFRCxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxHQUFHLGtCQUFrQixDQUFDLElBQUksRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLE1BQU0sRUFBRSxVQUFVLENBQUMsQ0FBQztRQUVuRixrQkFBa0I7UUFDbEIsSUFBSSxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsS0FBSyxTQUFTLEVBQUU7WUFDN0MsSUFBSSx1Q0FBdUIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO2dCQUNyQyxPQUFPLEVBQUUsSUFBSSxDQUFDLGVBQWU7Z0JBQzdCLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixFQUFFO2dCQUM3QixHQUFHLFVBQVUsQ0FBQyxVQUFVLEVBQUU7Z0JBQzFCLFdBQVc7YUFDWixDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFTSxhQUFhLENBQUMsSUFBVyxFQUFFLFVBQWdCLEVBQUUsV0FBb0IsRUFBRSxVQUFvQjtRQUM1RixJQUFJLFdBQVcsS0FBSyxTQUFTLEVBQUU7WUFDN0IsV0FBVyxHQUFHLE1BQU0sSUFBSSxDQUFDLFFBQVEsSUFBSSxVQUFVLEVBQUUsQ0FBQztTQUNuRDtRQUVELE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLEdBQUcsa0JBQWtCLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBRWpGLGtCQUFrQjtRQUNsQixJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQyxLQUFLLFNBQVMsRUFBRTtZQUM3QyxJQUFJLHNDQUFzQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7Z0JBQ3BDLE9BQU8sRUFBRSxJQUFJLENBQUMsZUFBZTtnQkFDN0IsR0FBRyxJQUFJLENBQUMsa0JBQWtCLEVBQUU7Z0JBQzVCLEdBQUcsVUFBVSxDQUFDLFVBQVUsRUFBRTtnQkFDMUIsV0FBVzthQUNaLENBQUMsQ0FBQztTQUNKO0lBQ0gsQ0FBQzs7Ozs7O0lBRU0sbUJBQW1CO1FBQ3hCLE9BQU8sRUFBRSxxQkFBcUIsRUFBRSxJQUFJLENBQUMsZUFBZSxFQUFFLENBQUM7SUFDekQsQ0FBQzs7Ozs7O0lBRU0sa0JBQWtCO1FBQ3ZCLE9BQU8sRUFBRSwwQkFBMEIsRUFBRSxJQUFJLENBQUMsZUFBZSxFQUFFLENBQUM7SUFDOUQsQ0FBQztDQUNGO0FBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBK0NHO0FBQ0gsU0FBUyxrQkFBa0IsQ0FDekIsS0FBd0IsRUFDeEIsSUFBVyxFQUNYLFVBQWdCLEVBQ2hCLE1BQXFCLEVBQ3JCLFVBQW9CO0lBRXBCLElBQUksVUFBVSxJQUFJLGlCQUFpQixDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsSUFBSSxlQUFlLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxFQUFFO1FBQ3pGLFdBQVc7UUFDWCxNQUFNLGNBQWMsR0FBRyxNQUFNLEtBQUssTUFBTSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQztRQUN6RCxPQUFPLENBQUMsSUFBSSxFQUFFLEdBQUcsS0FBSyxDQUFDLFFBQVEsSUFBSSxVQUFVLElBQUksY0FBYyxFQUFFLENBQUMsQ0FBQztLQUNwRTtTQUFNO1FBQ0wsK0VBQStFO1FBQy9FLE9BQU8sQ0FBQyxLQUFLLEVBQUUsR0FBRyxNQUFNLElBQUksVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLFVBQVUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztLQUNqRjtBQUNILENBQUM7QUFFRCxTQUFTLFVBQVUsQ0FBQyxJQUFXO0lBQzdCLE9BQU8sWUFBSyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO0FBQzlFLENBQUM7QUFFRCxTQUFTLGVBQWUsQ0FBQyxNQUF5QixFQUFFLE1BQXlCO0lBQzNFLE9BQU8sWUFBSyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxZQUFLLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0FBQy9DLENBQUM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztBQTZCRCxNQUFhLGFBQWMsU0FBUSxpQkFBaUI7Ozs7SUEyRWxELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBeUI7UUFDakUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjtTQUN0QyxDQUFDLENBQUM7UUFYWSx1QkFBa0IsR0FBdUMsRUFBRSxDQUFDO1FBQzVELHNCQUFpQixHQUFzQyxFQUFFLENBQUM7UUFZekUsTUFBTSxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsV0FBVyxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO1FBRTdELElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsZ0JBQWdCLEtBQUssS0FBSyxDQUFDO1FBRXpELElBQUksQ0FBQyxrQkFBa0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLEtBQUssU0FBUyxDQUFDLENBQUM7WUFDaEUsQ0FBQyxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1lBQzVCLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQywrQ0FBK0MsQ0FBQyxDQUFDO1FBRTdFLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxnQ0FBZ0IsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzFELFNBQVMsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUM1QixnQkFBZ0I7WUFDaEIsb0JBQW9CLEVBQUUsV0FBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsRUFBRSxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsQ0FBRTtZQUNyRyxtQkFBbUIsRUFBRSxXQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxFQUFFLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxDQUFFO1lBQ25HLEtBQUssRUFBRSxLQUFLLENBQUMsR0FBRyxDQUFDLEtBQUs7U0FDdkIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQztRQUN0RCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTLENBQUM7UUFDdkQsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDO1FBRWhELElBQUksQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO0lBQzlCLENBQUM7Ozs7OztJQW5HTSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGVBQXVCO1FBQzVFLElBQUksWUFBSyxDQUFDLFlBQVksQ0FBQyxlQUFlLENBQUMsRUFBRTtZQUN2QyxNQUFNLElBQUksS0FBSyxDQUFDLHdFQUF3RSxDQUFDLENBQUM7U0FDM0Y7UUFFRCxNQUFNLFVBQVUsR0FBdUMsc0JBQWUsQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFO1lBQ3JGLFFBQVEsRUFBRSxRQUFRLENBQUMsZUFBZSxDQUFDLHVCQUF1QjtZQUMxRCxLQUFLLEVBQUUsRUFBRSxlQUFlLEVBQUU7WUFDMUIsVUFBVSxFQUFFO2dCQUNWLGVBQWUsRUFBRSxVQUFVO2dCQUMzQixnQkFBZ0IsRUFBRSxJQUFJO2FBQ2U7U0FDeEMsQ0FBQyxDQUFDLEtBQUssQ0FBQztRQUVULE9BQU8sYUFBYSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsVUFBVSxDQUFDLGVBQWUsRUFBRTtZQUM5RSxnQkFBZ0IsRUFBRSxVQUFVLENBQUMsZ0JBQWdCO1lBQzdDLE9BQU8sRUFBRSxJQUFJO1NBQ2QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7Ozs7Ozs7OztJQUdNLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxlQUF1QixFQUFFLFVBQXNDLEVBQUU7UUFDL0gsTUFBTSxhQUFjLFNBQVEsaUJBQWlCO1lBQTdDOzs7Z0JBQ1Msb0JBQWUsR0FBRyxlQUFlLENBQUM7Z0JBQ2xDLHFCQUFnQixTQUFHLE9BQU8sQ0FBQyxnQkFBZ0IsbUNBQUksSUFBSSxDQUFDO1lBUTdELENBQUM7WUFOUSxhQUFhLENBQUMsSUFBVyxFQUFFLFVBQWdCLEVBQUUsV0FBb0IsRUFBRSxVQUFvQjtnQkFDNUYsNkNBQTZDO2dCQUM3QyxJQUFJLE9BQU8sQ0FBQyxnQkFBZ0IsS0FBSyxLQUFLLEVBQUU7b0JBQ3RDLEtBQUssQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRSxXQUFXLEVBQUUsVUFBVSxDQUFDLENBQUM7aUJBQ2hFO1lBQ0gsQ0FBQztTQUNGO1FBRUQsTUFBTSxlQUFnQixTQUFRLGlCQUFpQjtZQUEvQzs7O2dCQUNTLG9CQUFlLEdBQUcsZUFBZSxDQUFDO2dCQUNsQyxxQkFBZ0IsU0FBRyxPQUFPLENBQUMsZ0JBQWdCLG1DQUFJLElBQUksQ0FBQztZQVM3RCxDQUFDO1lBUFEsYUFBYSxDQUFDLEtBQVksRUFBRSxXQUFpQixFQUFFLFlBQXFCLEVBQUUsV0FBcUI7Z0JBQ2hHLGFBQWE7WUFDZixDQUFDO1lBRU0sY0FBYyxDQUFDLEtBQVksRUFBRSxXQUFpQixFQUFFLFlBQXFCLEVBQUUsV0FBcUI7Z0JBQ2pHLGFBQWE7WUFDZixDQUFDO1NBQ0Y7UUFFRCxPQUFPLE9BQU8sQ0FBQyxPQUFPLEtBQUssS0FBSztZQUM5QixDQUFDLENBQUMsSUFBSSxhQUFhLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQztZQUM5QixDQUFDLENBQUMsSUFBSSxlQUFlLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7Ozs7Ozs7Ozs7OztJQW1ETSxjQUFjLENBQUMsSUFBVyxFQUFFLFVBQWdCLEVBQUUsV0FBb0IsRUFBRSxVQUFvQjtRQUM3RixJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLGtCQUFrQixFQUFFO1lBQy9FLEtBQUssQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRSxXQUFXLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFDaEUsT0FBTztTQUNSO1FBRUQsSUFBSSxXQUFXLEtBQUssU0FBUyxFQUFFO1lBQzdCLFdBQVcsR0FBRyxRQUFRLElBQUksQ0FBQyxRQUFRLElBQUksVUFBVSxFQUFFLENBQUM7U0FDckQ7UUFFRCxJQUFJLENBQUMsb0JBQW9CLENBQUM7WUFDeEIsR0FBRyxJQUFJLENBQUMsbUJBQW1CLEVBQUU7WUFDN0IsR0FBRyxVQUFVLENBQUMsVUFBVSxFQUFFO1lBQzFCLFdBQVc7U0FDWixDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7Ozs7Ozs7SUFFTSxhQUFhLENBQUMsSUFBVyxFQUFFLFVBQWdCLEVBQUUsV0FBb0IsRUFBRSxVQUFvQjtRQUM1RixJQUFJLElBQUksQ0FBQyxnQkFBZ0IsRUFBRTtZQUN6Qix3RUFBd0U7WUFDeEUsd0VBQXdFO1lBQ3hFLFFBQVE7WUFDUixJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsc0RBQXNEO2dCQUN2RSxrQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQUMseUlBQXlJLENBQUMsQ0FBQzthQUM1SztZQUNELE9BQU87U0FDUjthQUFNO1lBQ0wsdUVBQXVFO1lBQ3ZFLGlFQUFpRTtZQUNqRSx5QkFBeUI7WUFDekIsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUM7U0FDNUI7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLGtCQUFrQixFQUFFO1lBQy9FLEtBQUssQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRSxXQUFXLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFDL0QsT0FBTztTQUNSO1FBRUQsSUFBSSxXQUFXLEtBQUssU0FBUyxFQUFFO1lBQzdCLFdBQVcsR0FBRyxRQUFRLElBQUksQ0FBQyxRQUFRLElBQUksVUFBVSxFQUFFLENBQUM7U0FDckQ7UUFFRCxNQUFNLElBQUksR0FBRztZQUNYLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixFQUFFO1lBQzVCLEdBQUcsVUFBVSxDQUFDLFVBQVUsRUFBRTtZQUMxQixXQUFXO1NBQ1osQ0FBQztRQUVGLElBQUksZ0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDMUIseUVBQXlFO1lBQ3pFLDRFQUE0RTtZQUM1RSxvRUFBb0U7WUFDcEUsMkVBQTJFO1lBQzNFLG1EQUFtRDtZQUNuRCxNQUFNLElBQUksS0FBSyxDQUFDLDhHQUE4RyxDQUFDLENBQUM7U0FDakk7UUFFRCxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOztPQUVHO0lBQ0ssb0JBQW9CLENBQUMsSUFBc0M7UUFDakUsSUFBSSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDOUIsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNwQztJQUNILENBQUM7SUFFRDs7T0FFRztJQUNLLGNBQWMsQ0FBQyxJQUFzQztRQUMzRCxPQUFPLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7O09BRUc7SUFDSyxtQkFBbUIsQ0FBQyxJQUFxQztRQUMvRCxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUM3QixJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ25DO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0ssYUFBYSxDQUFDLElBQXFDO1FBQ3pELE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBQy9FLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ssb0JBQW9CO1FBQzFCLElBQUksSUFBSSxDQUFDLGtCQUFrQixFQUFFO1lBQzNCLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FBQztZQUN4RSxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUM7WUFDeEUsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxXQUFXLENBQUM7WUFDdEcsS0FBSyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFLFdBQVcsRUFBRSxLQUFLLENBQUMsQ0FBQztTQUNyRDthQUFNO1lBQ0wsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFBLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDO1lBQ3RFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDbkM7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxtQkFBbUI7UUFDekIsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDM0IsTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsR0FBRyxrQkFBa0IsQ0FDcEMsSUFBSSxFQUNKLGVBQWUsRUFDZixlQUFlLEVBQ2YsSUFBSSxFQUNKLEtBQUssQ0FBQyxDQUFDO1lBRVQsS0FBSyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDL0I7YUFBTTtZQUNMLE1BQU0sQ0FBQyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO1lBQ3ZGLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFO2dCQUNWLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO2FBQ3JDO1NBQ0Y7SUFDSCxDQUFDOztBQWhQSCxzQ0FpUEM7OztBQUVEOzs7Ozs7OztHQVFHO0FBQ0gsTUFBTSxnQkFBZ0IsR0FBRztJQUN2QixNQUFNLEVBQUUsb0JBQW9CO0lBQzVCLFdBQVcsRUFBRSxzQkFBc0I7SUFDbkMsVUFBVSxFQUFFLE1BQU07SUFDbEIsUUFBUSxFQUFFLEdBQUc7SUFDYixNQUFNLEVBQUUsRUFBRTtDQUNYLENBQUM7QUFFRixNQUFNLGVBQWUsR0FBRyxXQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDO0FBQzNELE1BQU0sZUFBZSxHQUFHLFdBQUksQ0FBQyxlQUFlLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxFQUFFLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDO0FBRWpHOztHQUVHO0FBQ0gsTUFBTSxjQUFjLEdBQUc7SUFDckIsTUFBTSxFQUFFLFdBQVc7SUFDbkIsV0FBVyxFQUFFLHVDQUF1QztJQUNwRCxVQUFVLEVBQUUsSUFBSTtDQUNqQixDQUFDO0FBRUYsTUFBTSxnQkFBZ0IsR0FBRyxXQUFJLENBQUMsT0FBTyxFQUFFLENBQUM7QUFDeEMsTUFBTSxnQkFBZ0IsR0FBRyxXQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7QUFnQjNDOztHQUVHO0FBQ0gsU0FBUyxpQkFBaUIsQ0FBQyxDQUFtQyxFQUFFLENBQW1DO0lBQ2pHLE9BQU8sQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsTUFBTTtXQUN2QixDQUFDLENBQUMsUUFBUSxLQUFLLENBQUMsQ0FBQyxRQUFRO1dBQ3pCLENBQUMsQ0FBQyxRQUFRLEtBQUssQ0FBQyxDQUFDLFFBQVE7V0FDekIsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsTUFBTTtXQUNyQixDQUFDLENBQUMsVUFBVSxLQUFLLENBQUMsQ0FBQyxVQUFVO1dBQzdCLENBQUMsQ0FBQyxxQkFBcUIsS0FBSyxDQUFDLENBQUMscUJBQXFCO1dBQ25ELENBQUMsQ0FBQyx1QkFBdUIsS0FBSyxDQUFDLENBQUMsdUJBQXVCO1dBQ3ZELENBQUMsQ0FBQywwQkFBMEIsS0FBSyxDQUFDLENBQUMsMEJBQTBCLENBQUM7QUFDckUsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxnQkFBZ0IsQ0FBQyxDQUFrQyxFQUFFLENBQWtDO0lBQzlGLE9BQU8sQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsTUFBTTtXQUN2QixDQUFDLENBQUMsUUFBUSxLQUFLLENBQUMsQ0FBQyxRQUFRO1dBQ3pCLENBQUMsQ0FBQyxRQUFRLEtBQUssQ0FBQyxDQUFDLFFBQVE7V0FDekIsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsTUFBTTtXQUNyQixDQUFDLENBQUMsVUFBVSxLQUFLLENBQUMsQ0FBQyxVQUFVO1dBQzdCLENBQUMsQ0FBQyx1QkFBdUIsS0FBSyxDQUFDLENBQUMsdUJBQXVCO1dBQ3ZELENBQUMsQ0FBQywwQkFBMEIsS0FBSyxDQUFDLENBQUMsMEJBQTBCLENBQUM7QUFDckUsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxnQkFBZ0IsQ0FBQyxJQUFTO0lBQ2pDLE9BQU8sSUFBSSxDQUFDLE1BQU0sS0FBSyxXQUFXLElBQUksSUFBSSxDQUFDLFVBQVUsS0FBSyxJQUFJLENBQUM7QUFDakUsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGN4c2NoZW1hIGZyb20gJ0Bhd3MtY2RrL2Nsb3VkLWFzc2VtYmx5LXNjaGVtYSc7XG5pbXBvcnQgeyBBbm5vdGF0aW9ucywgQ29udGV4dFByb3ZpZGVyLCBJUmVzb3VyY2UsIExhenksIE5hbWVzLCBSZXNvdXJjZSwgUmVzb3VyY2VQcm9wcywgU3RhY2ssIFRva2VuIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBjeGFwaSBmcm9tICdAYXdzLWNkay9jeC1hcGknO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDb25uZWN0aW9ucyB9IGZyb20gJy4vY29ubmVjdGlvbnMnO1xuaW1wb3J0IHsgQ2ZuU2VjdXJpdHlHcm91cCwgQ2ZuU2VjdXJpdHlHcm91cEVncmVzcywgQ2ZuU2VjdXJpdHlHcm91cEluZ3Jlc3MgfSBmcm9tICcuL2VjMi5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSVBlZXIsIFBlZXIgfSBmcm9tICcuL3BlZXInO1xuaW1wb3J0IHsgUG9ydCB9IGZyb20gJy4vcG9ydCc7XG5pbXBvcnQgeyBJVnBjIH0gZnJvbSAnLi92cGMnO1xuXG5jb25zdCBTRUNVUklUWV9HUk9VUF9TWU1CT0wgPSBTeW1ib2wuZm9yKCdAYXdzLWNkay9pYW0uU2VjdXJpdHlHcm91cCcpO1xuXG5jb25zdCBTRUNVUklUWV9HUk9VUF9ESVNBQkxFX0lOTElORV9SVUxFU19DT05URVhUX0tFWSA9ICdAYXdzLWNkay9hd3MtZWMyLnNlY3VyaXR5R3JvdXBEaXNhYmxlSW5saW5lUnVsZXMnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIElTZWN1cml0eUdyb3VwIGV4dGVuZHMgSVJlc291cmNlLCBJUGVlciB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cElkOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFsbG93QWxsT3V0Ym91bmQ6IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGFkZEluZ3Jlc3NSdWxlKHBlZXI6IElQZWVyLCBjb25uZWN0aW9uOiBQb3J0LCBkZXNjcmlwdGlvbj86IHN0cmluZywgcmVtb3RlUnVsZT86IGJvb2xlYW4pOiB2b2lkO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGFkZEVncmVzc1J1bGUocGVlcjogSVBlZXIsIGNvbm5lY3Rpb246IFBvcnQsIGRlc2NyaXB0aW9uPzogc3RyaW5nLCByZW1vdGVSdWxlPzogYm9vbGVhbik6IHZvaWQ7XG59XG5cbi8qKlxuICogQSBTZWN1cml0eUdyb3VwIHRoYXQgaXMgbm90IGNyZWF0ZWQgaW4gdGhpcyB0ZW1wbGF0ZVxuICovXG5hYnN0cmFjdCBjbGFzcyBTZWN1cml0eUdyb3VwQmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVNlY3VyaXR5R3JvdXAge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgaXNTZWN1cml0eUdyb3VwKHg6IGFueSk6IHggaXMgU2VjdXJpdHlHcm91cEJhc2Uge1xuICAgIHJldHVybiBTRUNVUklUWV9HUk9VUF9TWU1CT0wgaW4geDtcbiAgfVxuXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBzZWN1cml0eUdyb3VwSWQ6IHN0cmluZztcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGFsbG93QWxsT3V0Ym91bmQ6IGJvb2xlYW47XG5cbiAgcHVibGljIHJlYWRvbmx5IGNhbklubGluZVJ1bGUgPSBmYWxzZTtcbiAgcHVibGljIHJlYWRvbmx5IGNvbm5lY3Rpb25zOiBDb25uZWN0aW9ucyA9IG5ldyBDb25uZWN0aW9ucyh7IHNlY3VyaXR5R3JvdXBzOiBbdGhpc10gfSk7XG4gIHB1YmxpYyByZWFkb25seSBkZWZhdWx0UG9ydD86IFBvcnQ7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM/OiBSZXNvdXJjZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICBPYmplY3QuZGVmaW5lUHJvcGVydHkodGhpcywgU0VDVVJJVFlfR1JPVVBfU1lNQk9MLCB7IHZhbHVlOiB0cnVlIH0pO1xuICB9XG5cbiAgcHVibGljIGdldCB1bmlxdWVJZCgpIHtcbiAgICByZXR1cm4gTmFtZXMubm9kZVVuaXF1ZUlkKHRoaXMubm9kZSk7XG4gIH1cblxuICBwdWJsaWMgYWRkSW5ncmVzc1J1bGUocGVlcjogSVBlZXIsIGNvbm5lY3Rpb246IFBvcnQsIGRlc2NyaXB0aW9uPzogc3RyaW5nLCByZW1vdGVSdWxlPzogYm9vbGVhbikge1xuICAgIGlmIChkZXNjcmlwdGlvbiA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICBkZXNjcmlwdGlvbiA9IGBmcm9tICR7cGVlci51bmlxdWVJZH06JHtjb25uZWN0aW9ufWA7XG4gICAgfVxuXG4gICAgY29uc3QgW3Njb3BlLCBpZF0gPSBkZXRlcm1pbmVSdWxlU2NvcGUodGhpcywgcGVlciwgY29ubmVjdGlvbiwgJ2Zyb20nLCByZW1vdGVSdWxlKTtcblxuICAgIC8vIFNraXAgZHVwbGljYXRlc1xuICAgIGlmIChzY29wZS5ub2RlLnRyeUZpbmRDaGlsZChpZCkgPT09IHVuZGVmaW5lZCkge1xuICAgICAgbmV3IENmblNlY3VyaXR5R3JvdXBJbmdyZXNzKHNjb3BlLCBpZCwge1xuICAgICAgICBncm91cElkOiB0aGlzLnNlY3VyaXR5R3JvdXBJZCxcbiAgICAgICAgLi4ucGVlci50b0luZ3Jlc3NSdWxlQ29uZmlnKCksXG4gICAgICAgIC4uLmNvbm5lY3Rpb24udG9SdWxlSnNvbigpLFxuICAgICAgICBkZXNjcmlwdGlvbixcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBhZGRFZ3Jlc3NSdWxlKHBlZXI6IElQZWVyLCBjb25uZWN0aW9uOiBQb3J0LCBkZXNjcmlwdGlvbj86IHN0cmluZywgcmVtb3RlUnVsZT86IGJvb2xlYW4pIHtcbiAgICBpZiAoZGVzY3JpcHRpb24gPT09IHVuZGVmaW5lZCkge1xuICAgICAgZGVzY3JpcHRpb24gPSBgdG8gJHtwZWVyLnVuaXF1ZUlkfToke2Nvbm5lY3Rpb259YDtcbiAgICB9XG5cbiAgICBjb25zdCBbc2NvcGUsIGlkXSA9IGRldGVybWluZVJ1bGVTY29wZSh0aGlzLCBwZWVyLCBjb25uZWN0aW9uLCAndG8nLCByZW1vdGVSdWxlKTtcblxuICAgIC8vIFNraXAgZHVwbGljYXRlc1xuICAgIGlmIChzY29wZS5ub2RlLnRyeUZpbmRDaGlsZChpZCkgPT09IHVuZGVmaW5lZCkge1xuICAgICAgbmV3IENmblNlY3VyaXR5R3JvdXBFZ3Jlc3Moc2NvcGUsIGlkLCB7XG4gICAgICAgIGdyb3VwSWQ6IHRoaXMuc2VjdXJpdHlHcm91cElkLFxuICAgICAgICAuLi5wZWVyLnRvRWdyZXNzUnVsZUNvbmZpZygpLFxuICAgICAgICAuLi5jb25uZWN0aW9uLnRvUnVsZUpzb24oKSxcbiAgICAgICAgZGVzY3JpcHRpb24sXG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgdG9JbmdyZXNzUnVsZUNvbmZpZygpOiBhbnkge1xuICAgIHJldHVybiB7IHNvdXJjZVNlY3VyaXR5R3JvdXBJZDogdGhpcy5zZWN1cml0eUdyb3VwSWQgfTtcbiAgfVxuXG4gIHB1YmxpYyB0b0VncmVzc1J1bGVDb25maWcoKTogYW55IHtcbiAgICByZXR1cm4geyBkZXN0aW5hdGlvblNlY3VyaXR5R3JvdXBJZDogdGhpcy5zZWN1cml0eUdyb3VwSWQgfTtcbiAgfVxufVxuXG4vKipcbiAqIERldGVybWluZSB3aGVyZSB0byBwYXJlbnQgYSBuZXcgaW5ncmVzcy9lZ3Jlc3MgcnVsZVxuICpcbiAqIEEgU2VjdXJpdHlHcm91cCBydWxlIGlzIHBhcmVudGVkIHVuZGVyIHRoZSBncm91cCBpdCdzIHJlbGF0ZWQgdG8sIFVOTEVTU1xuICogd2UncmUgaW4gYSBjcm9zcy1zdGFjayBzY2VuYXJpbyB3aXRoIGFub3RoZXIgU2VjdXJpdHkgR3JvdXAuIEluIHRoYXQgY2FzZSxcbiAqIHdlIHJlc3BlY3QgdGhlICdyZW1vdGVSdWxlJyBmbGFnIGFuZCB3aWxsIHBhcmVudCB1bmRlciB0aGUgb3RoZXIgc2VjdXJpdHlcbiAqIGdyb3VwLlxuICpcbiAqIFRoaXMgaXMgbmVjZXNzYXJ5IHRvIGF2b2lkIGN5Y2xpYyBkZXBlbmRlbmNpZXMgYmV0d2VlbiBzdGFja3MsIHNpbmNlIGJvdGhcbiAqIGluZ3Jlc3MgYW5kIGVncmVzcyBydWxlcyB3aWxsIHJlZmVyZW5jZSBib3RoIHNlY3VyaXR5IGdyb3VwcywgYW5kIGEgbmFpdmVcbiAqIHBhcmVudGluZyB3aWxsIGxlYWQgdG8gdGhlIGZvbGxvd2luZyBzaXR1YXRpb246XG4gKlxuICogICDilZTilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZcgICAgICAgICDilZTilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZDilZdcbiAqICAg4pWRICDilIzilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilJAgICAgIOKVkSAgICAgICAgIOKVkSAgICDilIzilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilJAgICDilZFcbiAqICAg4pWRICDilIIgIEdyb3VwQSAgIOKUguKXgOKUgOKUgOKUgOKUgOKVrOKUgOKUkCAgIOKUjOKUgOKUgOKUgOKVrOKUgOKUgOKUgOKWtuKUgiAgR3JvdXBCICAg4pSCICAg4pWRXG4gKiAgIOKVkSAg4pSU4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSYICAgICDilZEg4pSCICAg4pSCICAg4pWRICAgIOKUlOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUmCAgIOKVkVxuICogICDilZEgICAgICAgIOKWsiAgICAgICAgICAg4pWRIOKUgiAgIOKUgiAgIOKVkSAgICAgICAgICDilrIgICAgICAgICDilZFcbiAqICAg4pWRICAgICAgICDilIIgICAgICAgICAgIOKVkSDilIIgICDilIIgICDilZEgICAgICAgICAg4pSCICAgICAgICAg4pWRXG4gKiAgIOKVkSAgICAgICAg4pSCICAgICAgICAgICDilZEg4pSCICAg4pSCICAg4pWRICAgICAgICAgIOKUgiAgICAgICAgIOKVkVxuICogICDilZEgIOKUjOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUkCAgICAg4pWRIOKUlOKUgOKUgOKUgOKUvOKUgOKUgOKUgOKVrOKUgOKUgOKUgOKUgOKUjOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUkCAgIOKVkVxuICogICDilZEgIOKUgiAgRWdyZXNzQSAg4pSC4pSA4pSA4pSA4pSA4pSA4pWs4pSA4pSA4pSA4pSA4pSA4pSYICAg4pWRICAgIOKUgiBJbmdyZXNzQiAg4pSCICAg4pWRXG4gKiAgIOKVkSAg4pSU4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSYICAgICDilZEgICAgICAgICDilZEgICAg4pSU4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSYICAg4pWRXG4gKiAgIOKVkSAgICAgICAgICAgICAgICAgICAg4pWRICAgICAgICAg4pWRICAgICAgICAgICAgICAgICAgICDilZFcbiAqICAg4pWa4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWdICAgICAgICAg4pWa4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWQ4pWdXG4gKlxuICogQnkgaGF2aW5nIHRoZSBhYmlsaXR5IHRvIHN3aXRjaCB0aGUgcGFyZW50LCB3ZSBhdm9pZCB0aGUgY3ljbGljIHJlZmVyZW5jZSBieVxuICoga2VlcGluZyBhbGwgcnVsZXMgaW4gYSBzaW5nbGUgc3RhY2suXG4gKlxuICogSWYgdGhpcyBoYXBwZW5zLCB3ZSBhbHNvIGhhdmUgdG8gY2hhbmdlIHRoZSBjb25zdHJ1Y3QgSUQsIGJlY2F1c2VcbiAqIG90aGVyd2lzZSB3ZSBtaWdodCBoYXZlIHR3byBvYmplY3RzIHdpdGggdGhlIHNhbWUgSUQgaWYgd2UgaGF2ZVxuICogbXVsdGlwbGUgcmV2ZXJzZWQgc2VjdXJpdHkgZ3JvdXAgcmVsYXRpb25zaGlwcy5cbiAqXG4gKiAgIOKVlOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVl1xuICogICDilZHilIzilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilJAgICAgICAgICAgICAgICAgICAgICAg4pWRXG4gKiAgIOKVkeKUgiAgR3JvdXBCICAg4pSCICAgICAgICAgICAgICAgICAgICAgIOKVkVxuICogICDilZHilJTilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilJggICAgICAgICAgICAgICAgICAgICAg4pWRXG4gKiAgIOKVkSAgICAgIOKWsiAgICAgICAgICAgICAgICAgICAgICAgICAgICDilZFcbiAqICAg4pWRICAgICAg4pSCICAgICAgICAgICAgICDilIzilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilJAg4pWRXG4gKiAgIOKVkSAgICAgIOKUnOKUgOKUgOKUgOKUgFwiZnJvbSBBXCLilIDilIDilIIgSW5ncmVzc0IgIOKUgiDilZFcbiAqICAg4pWRICAgICAg4pSCICAgICAgICAgICAgICDilJTilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilJgg4pWRXG4gKiAgIOKVkSAgICAgIOKUgiAgICAgICAgICAgICAg4pSM4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSQIOKVkVxuICogICDilZEgICAgICDilJzilIDilIDilIDilIDilIBcInRvIEJcIuKUgOKUgOKUgOKUgiAgRWdyZXNzQSAg4pSCIOKVkVxuICogICDilZEgICAgICDilIIgICAgICAgICAgICAgIOKUlOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUgOKUmCDilZFcbiAqICAg4pWRICAgICAg4pSCICAgICAgICAgICAgICDilIzilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilJAg4pWRXG4gKiAgIOKVkSAgICAgIOKUlOKUgOKUgOKUgOKUgOKUgFwidG8gQlwi4pSA4pSA4pSA4pSCICBFZ3Jlc3NDICDilIIg4pWRICA8LS0gb29wc1xuICogICDilZEgICAgICAgICAgICAgICAgICAgICDilJTilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilJgg4pWRXG4gKiAgIOKVmuKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVkOKVnVxuICovXG5mdW5jdGlvbiBkZXRlcm1pbmVSdWxlU2NvcGUoXG4gIGdyb3VwOiBTZWN1cml0eUdyb3VwQmFzZSxcbiAgcGVlcjogSVBlZXIsXG4gIGNvbm5lY3Rpb246IFBvcnQsXG4gIGZyb21UbzogJ2Zyb20nIHwgJ3RvJyxcbiAgcmVtb3RlUnVsZT86IGJvb2xlYW4pOiBbU2VjdXJpdHlHcm91cEJhc2UsIHN0cmluZ10ge1xuXG4gIGlmIChyZW1vdGVSdWxlICYmIFNlY3VyaXR5R3JvdXBCYXNlLmlzU2VjdXJpdHlHcm91cChwZWVyKSAmJiBkaWZmZXJlbnRTdGFja3MoZ3JvdXAsIHBlZXIpKSB7XG4gICAgLy8gUmV2ZXJzZWRcbiAgICBjb25zdCByZXZlcnNlZEZyb21UbyA9IGZyb21UbyA9PT0gJ2Zyb20nID8gJ3RvJyA6ICdmcm9tJztcbiAgICByZXR1cm4gW3BlZXIsIGAke2dyb3VwLnVuaXF1ZUlkfToke2Nvbm5lY3Rpb259ICR7cmV2ZXJzZWRGcm9tVG99YF07XG4gIH0gZWxzZSB7XG4gICAgLy8gUmVndWxhciAoZG8gb2xkIElEIGVzY2FwaW5nIHRvIGluIG9yZGVyIHRvIG5vdCBkaXN0dXJiIGV4aXN0aW5nIGRlcGxveW1lbnRzKVxuICAgIHJldHVybiBbZ3JvdXAsIGAke2Zyb21Ub30gJHtyZW5kZXJQZWVyKHBlZXIpfToke2Nvbm5lY3Rpb259YC5yZXBsYWNlKCcvJywgJ18nKV07XG4gIH1cbn1cblxuZnVuY3Rpb24gcmVuZGVyUGVlcihwZWVyOiBJUGVlcikge1xuICByZXR1cm4gVG9rZW4uaXNVbnJlc29sdmVkKHBlZXIudW5pcXVlSWQpID8gJ3tJbmRpcmVjdFBlZXJ9JyA6IHBlZXIudW5pcXVlSWQ7XG59XG5cbmZ1bmN0aW9uIGRpZmZlcmVudFN0YWNrcyhncm91cDE6IFNlY3VyaXR5R3JvdXBCYXNlLCBncm91cDI6IFNlY3VyaXR5R3JvdXBCYXNlKSB7XG4gIHJldHVybiBTdGFjay5vZihncm91cDEpICE9PSBTdGFjay5vZihncm91cDIpO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFNlY3VyaXR5R3JvdXBQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGM6IElWcGM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFsbG93QWxsT3V0Ym91bmQ/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGlzYWJsZUlubGluZVJ1bGVzPzogYm9vbGVhbjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBTZWN1cml0eUdyb3VwSW1wb3J0T3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhbGxvd0FsbE91dGJvdW5kPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtdXRhYmxlPzogYm9vbGVhbjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBTZWN1cml0eUdyb3VwIGV4dGVuZHMgU2VjdXJpdHlHcm91cEJhc2Uge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUxvb2t1cChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBzZWN1cml0eUdyb3VwSWQ6IHN0cmluZykge1xuICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQoc2VjdXJpdHlHcm91cElkKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBbGwgYXJndW1lbnRzIHRvIGxvb2sgdXAgYSBzZWN1cml0eSBncm91cCBtdXN0IGJlIGNvbmNyZXRlIChubyBUb2tlbnMpJyk7XG4gICAgfVxuXG4gICAgY29uc3QgYXR0cmlidXRlczogY3hhcGkuU2VjdXJpdHlHcm91cENvbnRleHRSZXNwb25zZSA9IENvbnRleHRQcm92aWRlci5nZXRWYWx1ZShzY29wZSwge1xuICAgICAgcHJvdmlkZXI6IGN4c2NoZW1hLkNvbnRleHRQcm92aWRlci5TRUNVUklUWV9HUk9VUF9QUk9WSURFUixcbiAgICAgIHByb3BzOiB7IHNlY3VyaXR5R3JvdXBJZCB9LFxuICAgICAgZHVtbXlWYWx1ZToge1xuICAgICAgICBzZWN1cml0eUdyb3VwSWQ6ICdzZy0xMjM0NScsXG4gICAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IHRydWUsXG4gICAgICB9IGFzIGN4YXBpLlNlY3VyaXR5R3JvdXBDb250ZXh0UmVzcG9uc2UsXG4gICAgfSkudmFsdWU7XG5cbiAgICByZXR1cm4gU2VjdXJpdHlHcm91cC5mcm9tU2VjdXJpdHlHcm91cElkKHNjb3BlLCBpZCwgYXR0cmlidXRlcy5zZWN1cml0eUdyb3VwSWQsIHtcbiAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IGF0dHJpYnV0ZXMuYWxsb3dBbGxPdXRib3VuZCxcbiAgICAgIG11dGFibGU6IHRydWUsXG4gICAgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVNlY3VyaXR5R3JvdXBJZChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBzZWN1cml0eUdyb3VwSWQ6IHN0cmluZywgb3B0aW9uczogU2VjdXJpdHlHcm91cEltcG9ydE9wdGlvbnMgPSB7fSk6IElTZWN1cml0eUdyb3VwIHtcbiAgICBjbGFzcyBNdXRhYmxlSW1wb3J0IGV4dGVuZHMgU2VjdXJpdHlHcm91cEJhc2Uge1xuICAgICAgcHVibGljIHNlY3VyaXR5R3JvdXBJZCA9IHNlY3VyaXR5R3JvdXBJZDtcbiAgICAgIHB1YmxpYyBhbGxvd0FsbE91dGJvdW5kID0gb3B0aW9ucy5hbGxvd0FsbE91dGJvdW5kID8/IHRydWU7XG5cbiAgICAgIHB1YmxpYyBhZGRFZ3Jlc3NSdWxlKHBlZXI6IElQZWVyLCBjb25uZWN0aW9uOiBQb3J0LCBkZXNjcmlwdGlvbj86IHN0cmluZywgcmVtb3RlUnVsZT86IGJvb2xlYW4pIHtcbiAgICAgICAgLy8gT25seSBpZiBhbGxvd0FsbE91dGJvdW5kIGhhcyBiZWVuIGRpc2FibGVkXG4gICAgICAgIGlmIChvcHRpb25zLmFsbG93QWxsT3V0Ym91bmQgPT09IGZhbHNlKSB7XG4gICAgICAgICAgc3VwZXIuYWRkRWdyZXNzUnVsZShwZWVyLCBjb25uZWN0aW9uLCBkZXNjcmlwdGlvbiwgcmVtb3RlUnVsZSk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG5cbiAgICBjbGFzcyBJbW11dGFibGVJbXBvcnQgZXh0ZW5kcyBTZWN1cml0eUdyb3VwQmFzZSB7XG4gICAgICBwdWJsaWMgc2VjdXJpdHlHcm91cElkID0gc2VjdXJpdHlHcm91cElkO1xuICAgICAgcHVibGljIGFsbG93QWxsT3V0Ym91bmQgPSBvcHRpb25zLmFsbG93QWxsT3V0Ym91bmQgPz8gdHJ1ZTtcblxuICAgICAgcHVibGljIGFkZEVncmVzc1J1bGUoX3BlZXI6IElQZWVyLCBfY29ubmVjdGlvbjogUG9ydCwgX2Rlc2NyaXB0aW9uPzogc3RyaW5nLCBfcmVtb3RlUnVsZT86IGJvb2xlYW4pIHtcbiAgICAgICAgLy8gZG8gbm90aGluZ1xuICAgICAgfVxuXG4gICAgICBwdWJsaWMgYWRkSW5ncmVzc1J1bGUoX3BlZXI6IElQZWVyLCBfY29ubmVjdGlvbjogUG9ydCwgX2Rlc2NyaXB0aW9uPzogc3RyaW5nLCBfcmVtb3RlUnVsZT86IGJvb2xlYW4pIHtcbiAgICAgICAgLy8gZG8gbm90aGluZ1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiBvcHRpb25zLm11dGFibGUgIT09IGZhbHNlXG4gICAgICA/IG5ldyBNdXRhYmxlSW1wb3J0KHNjb3BlLCBpZClcbiAgICAgIDogbmV3IEltbXV0YWJsZUltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBOYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgc2VjdXJpdHlHcm91cElkOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBzZWN1cml0eUdyb3VwVnBjSWQ6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGFsbG93QWxsT3V0Ym91bmQ6IGJvb2xlYW47XG5cbiAgcHJpdmF0ZSByZWFkb25seSBzZWN1cml0eUdyb3VwOiBDZm5TZWN1cml0eUdyb3VwO1xuICBwcml2YXRlIHJlYWRvbmx5IGRpcmVjdEluZ3Jlc3NSdWxlczogQ2ZuU2VjdXJpdHlHcm91cC5JbmdyZXNzUHJvcGVydHlbXSA9IFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IGRpcmVjdEVncmVzc1J1bGVzOiBDZm5TZWN1cml0eUdyb3VwLkVncmVzc1Byb3BlcnR5W10gPSBbXTtcblxuICAvKipcbiAgICogV2hldGhlciB0byBkaXNhYmxlIG9wdGltaXphdGlvbiBmb3IgaW5saW5lIHNlY3VyaXR5IGdyb3VwIHJ1bGVzLlxuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSBkaXNhYmxlSW5saW5lUnVsZXM6IGJvb2xlYW47XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFNlY3VyaXR5R3JvdXBQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy5zZWN1cml0eUdyb3VwTmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGdyb3VwRGVzY3JpcHRpb24gPSBwcm9wcy5kZXNjcmlwdGlvbiB8fCB0aGlzLm5vZGUucGF0aDtcblxuICAgIHRoaXMuYWxsb3dBbGxPdXRib3VuZCA9IHByb3BzLmFsbG93QWxsT3V0Ym91bmQgIT09IGZhbHNlO1xuXG4gICAgdGhpcy5kaXNhYmxlSW5saW5lUnVsZXMgPSBwcm9wcy5kaXNhYmxlSW5saW5lUnVsZXMgIT09IHVuZGVmaW5lZCA/XG4gICAgICAhIXByb3BzLmRpc2FibGVJbmxpbmVSdWxlcyA6XG4gICAgICAhIXRoaXMubm9kZS50cnlHZXRDb250ZXh0KFNFQ1VSSVRZX0dST1VQX0RJU0FCTEVfSU5MSU5FX1JVTEVTX0NPTlRFWFRfS0VZKTtcblxuICAgIHRoaXMuc2VjdXJpdHlHcm91cCA9IG5ldyBDZm5TZWN1cml0eUdyb3VwKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGdyb3VwTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgICBncm91cERlc2NyaXB0aW9uLFxuICAgICAgc2VjdXJpdHlHcm91cEluZ3Jlc3M6IExhenkuYW55KHsgcHJvZHVjZTogKCkgPT4gdGhpcy5kaXJlY3RJbmdyZXNzUnVsZXMgfSwgeyBvbWl0RW1wdHlBcnJheTogdHJ1ZSB9ICksXG4gICAgICBzZWN1cml0eUdyb3VwRWdyZXNzOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuZGlyZWN0RWdyZXNzUnVsZXMgfSwgeyBvbWl0RW1wdHlBcnJheTogdHJ1ZSB9ICksXG4gICAgICB2cGNJZDogcHJvcHMudnBjLnZwY0lkLFxuICAgIH0pO1xuXG4gICAgdGhpcy5zZWN1cml0eUdyb3VwSWQgPSB0aGlzLnNlY3VyaXR5R3JvdXAuYXR0ckdyb3VwSWQ7XG4gICAgdGhpcy5zZWN1cml0eUdyb3VwVnBjSWQgPSB0aGlzLnNlY3VyaXR5R3JvdXAuYXR0clZwY0lkO1xuICAgIHRoaXMuc2VjdXJpdHlHcm91cE5hbWUgPSB0aGlzLnNlY3VyaXR5R3JvdXAucmVmO1xuXG4gICAgdGhpcy5hZGREZWZhdWx0RWdyZXNzUnVsZSgpO1xuICB9XG5cbiAgcHVibGljIGFkZEluZ3Jlc3NSdWxlKHBlZXI6IElQZWVyLCBjb25uZWN0aW9uOiBQb3J0LCBkZXNjcmlwdGlvbj86IHN0cmluZywgcmVtb3RlUnVsZT86IGJvb2xlYW4pIHtcbiAgICBpZiAoIXBlZXIuY2FuSW5saW5lUnVsZSB8fCAhY29ubmVjdGlvbi5jYW5JbmxpbmVSdWxlIHx8IHRoaXMuZGlzYWJsZUlubGluZVJ1bGVzKSB7XG4gICAgICBzdXBlci5hZGRJbmdyZXNzUnVsZShwZWVyLCBjb25uZWN0aW9uLCBkZXNjcmlwdGlvbiwgcmVtb3RlUnVsZSk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgaWYgKGRlc2NyaXB0aW9uID09PSB1bmRlZmluZWQpIHtcbiAgICAgIGRlc2NyaXB0aW9uID0gYGZyb20gJHtwZWVyLnVuaXF1ZUlkfToke2Nvbm5lY3Rpb259YDtcbiAgICB9XG5cbiAgICB0aGlzLmFkZERpcmVjdEluZ3Jlc3NSdWxlKHtcbiAgICAgIC4uLnBlZXIudG9JbmdyZXNzUnVsZUNvbmZpZygpLFxuICAgICAgLi4uY29ubmVjdGlvbi50b1J1bGVKc29uKCksXG4gICAgICBkZXNjcmlwdGlvbixcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRFZ3Jlc3NSdWxlKHBlZXI6IElQZWVyLCBjb25uZWN0aW9uOiBQb3J0LCBkZXNjcmlwdGlvbj86IHN0cmluZywgcmVtb3RlUnVsZT86IGJvb2xlYW4pIHtcbiAgICBpZiAodGhpcy5hbGxvd0FsbE91dGJvdW5kKSB7XG4gICAgICAvLyBJbiB0aGUgY2FzZSBvZiBcImFsbG93QWxsT3V0Ym91bmRcIiwgd2UgZG9uJ3QgYWRkIGFueSBtb3JlIHJ1bGVzLiBUaGVyZVxuICAgICAgLy8gaXMgb25seSBvbmUgcnVsZSB3aGljaCBhbGxvd3MgYWxsIHRyYWZmaWMgYW5kIHRoYXQgc3Vic3VtZXMgYW55IG90aGVyXG4gICAgICAvLyBydWxlLlxuICAgICAgaWYgKCFyZW1vdGVSdWxlKSB7IC8vIFdhcm4gb25seSBpZiBhZGRFZ3Jlc3NSdWxlKCkgd2FzIGV4cGxpY2l0ZWx5IGNhbGxlZFxuICAgICAgICBBbm5vdGF0aW9ucy5vZih0aGlzKS5hZGRXYXJuaW5nKCdJZ25vcmluZyBFZ3Jlc3MgcnVsZSBzaW5jZSBcXCdhbGxvd0FsbE91dGJvdW5kXFwnIGlzIHNldCB0byB0cnVlOyBUbyBhZGQgY3VzdG9taXplIHJ1bGVzLCBzZXQgYWxsb3dBbGxPdXRib3VuZD1mYWxzZSBvbiB0aGUgU2VjdXJpdHlHcm91cCcpO1xuICAgICAgfVxuICAgICAgcmV0dXJuO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBPdGhlcndpc2UsIGlmIHRoZSBib2d1cyBydWxlIGV4aXN0cyB3ZSBjYW4gbm93IHJlbW92ZSBpdCBiZWNhdXNlIHRoZVxuICAgICAgLy8gcHJlc2VuY2Ugb2YgYW55IG90aGVyIHJ1bGUgd2lsbCBnZXQgcmlkIG9mIEVDMidzIGltcGxpY2l0IFwiYWxsXG4gICAgICAvLyBvdXRib3VuZFwiIHJ1bGUgYW55d2F5LlxuICAgICAgdGhpcy5yZW1vdmVOb1RyYWZmaWNSdWxlKCk7XG4gICAgfVxuXG4gICAgaWYgKCFwZWVyLmNhbklubGluZVJ1bGUgfHwgIWNvbm5lY3Rpb24uY2FuSW5saW5lUnVsZSB8fCB0aGlzLmRpc2FibGVJbmxpbmVSdWxlcykge1xuICAgICAgc3VwZXIuYWRkRWdyZXNzUnVsZShwZWVyLCBjb25uZWN0aW9uLCBkZXNjcmlwdGlvbiwgcmVtb3RlUnVsZSk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgaWYgKGRlc2NyaXB0aW9uID09PSB1bmRlZmluZWQpIHtcbiAgICAgIGRlc2NyaXB0aW9uID0gYGZyb20gJHtwZWVyLnVuaXF1ZUlkfToke2Nvbm5lY3Rpb259YDtcbiAgICB9XG5cbiAgICBjb25zdCBydWxlID0ge1xuICAgICAgLi4ucGVlci50b0VncmVzc1J1bGVDb25maWcoKSxcbiAgICAgIC4uLmNvbm5lY3Rpb24udG9SdWxlSnNvbigpLFxuICAgICAgZGVzY3JpcHRpb24sXG4gICAgfTtcblxuICAgIGlmIChpc0FsbFRyYWZmaWNSdWxlKHJ1bGUpKSB7XG4gICAgICAvLyBXZSBjYW5ub3QgYWxsb3cgdGhpczsgaWYgc29tZW9uZSBhZGRzIHRoZSBydWxlIGluIHRoaXMgd2F5LCBpdCB3aWxsIGJlXG4gICAgICAvLyByZW1vdmVkIGFnYWluIGlmIHRoZXkgYWRkIG90aGVyIHJ1bGVzLiBXZSBhbHNvIGNhbid0IGF1dG9tYXRpY2FsbHkgc3dpdGNoXG4gICAgICAvLyB0byBcImFsbE91dGJvdW5kPXRydWVcIiBtb2RlLCBiZWNhdXNlIHdlIG1pZ2h0IGhhdmUgYWxyZWFkeSBlbWl0dGVkXG4gICAgICAvLyBFZ3Jlc3NSdWxlIG9iamVjdHMgKHdoaWNoIGNvdW50IGFzIHJ1bGVzIGFkZGVkIGxhdGVyKSBhbmQgdGhlcmUncyBubyB3YXlcbiAgICAgIC8vIHRvIHJlY2FsbCB0aG9zZS4gQmV0dGVyIHRvIHByZXZlbnQgdGhpcyBmb3Igbm93LlxuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgYWRkIGFuIFwiYWxsIHRyYWZmaWNcIiBlZ3Jlc3MgcnVsZSBpbiB0aGlzIHdheTsgc2V0IGFsbG93QWxsT3V0Ym91bmQ9dHJ1ZSBvbiB0aGUgU2VjdXJpdHlHcm91cCBpbnN0ZWFkLicpO1xuICAgIH1cblxuICAgIHRoaXMuYWRkRGlyZWN0RWdyZXNzUnVsZShydWxlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBkaXJlY3QgaW5ncmVzcyBydWxlXG4gICAqL1xuICBwcml2YXRlIGFkZERpcmVjdEluZ3Jlc3NSdWxlKHJ1bGU6IENmblNlY3VyaXR5R3JvdXAuSW5ncmVzc1Byb3BlcnR5KSB7XG4gICAgaWYgKCF0aGlzLmhhc0luZ3Jlc3NSdWxlKHJ1bGUpKSB7XG4gICAgICB0aGlzLmRpcmVjdEluZ3Jlc3NSdWxlcy5wdXNoKHJ1bGUpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gd2hldGhlciB0aGUgZ2l2ZW4gaW5ncmVzcyBydWxlIGV4aXN0cyBvbiB0aGUgZ3JvdXBcbiAgICovXG4gIHByaXZhdGUgaGFzSW5ncmVzc1J1bGUocnVsZTogQ2ZuU2VjdXJpdHlHcm91cC5JbmdyZXNzUHJvcGVydHkpOiBib29sZWFuIHtcbiAgICByZXR1cm4gdGhpcy5kaXJlY3RJbmdyZXNzUnVsZXMuZmluZEluZGV4KHIgPT4gaW5ncmVzc1J1bGVzRXF1YWwociwgcnVsZSkpID4gLTE7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgZGlyZWN0IGVncmVzcyBydWxlXG4gICAqL1xuICBwcml2YXRlIGFkZERpcmVjdEVncmVzc1J1bGUocnVsZTogQ2ZuU2VjdXJpdHlHcm91cC5FZ3Jlc3NQcm9wZXJ0eSkge1xuICAgIGlmICghdGhpcy5oYXNFZ3Jlc3NSdWxlKHJ1bGUpKSB7XG4gICAgICB0aGlzLmRpcmVjdEVncmVzc1J1bGVzLnB1c2gocnVsZSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiB3aGV0aGVyIHRoZSBnaXZlbiBlZ3Jlc3MgcnVsZSBleGlzdHMgb24gdGhlIGdyb3VwXG4gICAqL1xuICBwcml2YXRlIGhhc0VncmVzc1J1bGUocnVsZTogQ2ZuU2VjdXJpdHlHcm91cC5FZ3Jlc3NQcm9wZXJ0eSk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB0aGlzLmRpcmVjdEVncmVzc1J1bGVzLmZpbmRJbmRleChyID0+IGVncmVzc1J1bGVzRXF1YWwociwgcnVsZSkpID4gLTE7XG4gIH1cblxuICAvKipcbiAgICogQWRkIHRoZSBkZWZhdWx0IGVncmVzcyBydWxlIHRvIHRoZSBzZWN1cml0eUdyb3VwXG4gICAqXG4gICAqIFRoaXMgZGVwZW5kcyBvbiBhbGxvd0FsbE91dGJvdW5kOlxuICAgKlxuICAgKiAtIElmIGFsbG93QWxsT3V0Ym91bmQgaXMgdHJ1ZSwgd2UgKlRFQ0hOSUNBTExZKiBkb24ndCBuZWVkIHRvIGRvIGFueXRoaW5nLCBiZWNhdXNlXG4gICAqICAgRUMyIGlzIGdvaW5nIHRvIGNyZWF0ZSB0aGlzIGRlZmF1bHQgcnVsZSBhbnl3YXkuIEJ1dCwgZm9yIG1heGltdW0gcmVhZGFiaWxpdHlcbiAgICogICBvZiB0aGUgdGVtcGxhdGUsIHdlIHdpbGwgYWRkIG9uZSBhbnl3YXkuXG4gICAqIC0gSWYgYWxsb3dBbGxPdXRib3VuZCBpcyBmYWxzZSwgd2UgYWRkIGEgYm9ndXMgcnVsZSB0aGF0IG1hdGNoZXMgbm8gdHJhZmZpYyBpblxuICAgKiAgIG9yZGVyIHRvIGdldCByaWQgb2YgdGhlIGRlZmF1bHQgXCJhbGwgb3V0Ym91bmRcIiBydWxlIHRoYXQgRUMyIGNyZWF0ZXMgYnkgZGVmYXVsdC5cbiAgICogICBJZiBvdGhlciBydWxlcyBoYXBwZW4gdG8gZ2V0IGFkZGVkIGxhdGVyLCB3ZSByZW1vdmUgdGhlIGJvZ3VzIHJ1bGUgYWdhaW4gc29cbiAgICogICB0aGF0IGl0IGRvZXNuJ3QgY2x1dHRlciB1cCB0aGUgdGVtcGxhdGUgdG9vIG11Y2ggKGV2ZW4gdGhvdWdoIHRoYXQncyBub3RcbiAgICogICBzdHJpY3RseSBuZWNlc3NhcnkpLlxuICAgKi9cbiAgcHJpdmF0ZSBhZGREZWZhdWx0RWdyZXNzUnVsZSgpIHtcbiAgICBpZiAodGhpcy5kaXNhYmxlSW5saW5lUnVsZXMpIHtcbiAgICAgIGNvbnN0IHBlZXIgPSB0aGlzLmFsbG93QWxsT3V0Ym91bmQgPyBBTExfVFJBRkZJQ19QRUVSIDogTk9fVFJBRkZJQ19QRUVSO1xuICAgICAgY29uc3QgcG9ydCA9IHRoaXMuYWxsb3dBbGxPdXRib3VuZCA/IEFMTF9UUkFGRklDX1BPUlQgOiBOT19UUkFGRklDX1BPUlQ7XG4gICAgICBjb25zdCBkZXNjcmlwdGlvbiA9IHRoaXMuYWxsb3dBbGxPdXRib3VuZCA/IEFMTE9XX0FMTF9SVUxFLmRlc2NyaXB0aW9uIDogTUFUQ0hfTk9fVFJBRkZJQy5kZXNjcmlwdGlvbjtcbiAgICAgIHN1cGVyLmFkZEVncmVzc1J1bGUocGVlciwgcG9ydCwgZGVzY3JpcHRpb24sIGZhbHNlKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgcnVsZSA9IHRoaXMuYWxsb3dBbGxPdXRib3VuZD8gQUxMT1dfQUxMX1JVTEUgOiBNQVRDSF9OT19UUkFGRklDO1xuICAgICAgdGhpcy5kaXJlY3RFZ3Jlc3NSdWxlcy5wdXNoKHJ1bGUpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBSZW1vdmUgdGhlIGJvZ3VzIHJ1bGUgaWYgaXQgZXhpc3RzXG4gICAqL1xuICBwcml2YXRlIHJlbW92ZU5vVHJhZmZpY1J1bGUoKSB7XG4gICAgaWYgKHRoaXMuZGlzYWJsZUlubGluZVJ1bGVzKSB7XG4gICAgICBjb25zdCBbc2NvcGUsIGlkXSA9IGRldGVybWluZVJ1bGVTY29wZShcbiAgICAgICAgdGhpcyxcbiAgICAgICAgTk9fVFJBRkZJQ19QRUVSLFxuICAgICAgICBOT19UUkFGRklDX1BPUlQsXG4gICAgICAgICd0bycsXG4gICAgICAgIGZhbHNlKTtcblxuICAgICAgc2NvcGUubm9kZS50cnlSZW1vdmVDaGlsZChpZCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IGkgPSB0aGlzLmRpcmVjdEVncmVzc1J1bGVzLmZpbmRJbmRleChyID0+IGVncmVzc1J1bGVzRXF1YWwociwgTUFUQ0hfTk9fVFJBRkZJQykpO1xuICAgICAgaWYgKGkgPiAtMSkge1xuICAgICAgICB0aGlzLmRpcmVjdEVncmVzc1J1bGVzLnNwbGljZShpLCAxKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBFZ3Jlc3MgcnVsZSB0aGF0IHB1cnBvc2VseSBtYXRjaGVzIG5vIHRyYWZmaWNcbiAqXG4gKiBUaGlzIGlzIHVzZWQgaW4gb3JkZXIgdG8gZGlzYWJsZSB0aGUgXCJhbGwgdHJhZmZpY1wiIGRlZmF1bHQgb2YgU2VjdXJpdHkgR3JvdXBzLlxuICpcbiAqIE5vIG1hY2hpbmUgY2FuIGV2ZXIgYWN0dWFsbHkgaGF2ZSB0aGUgMjU1LjI1NS4yNTUuMjU1IElQIGFkZHJlc3MsIGJ1dFxuICogaW4gb3JkZXIgdG8gbG9jayBpdCBkb3duIGV2ZW4gbW9yZSB3ZSdsbCByZXN0cmljdCB0byBhIG5vbmV4aXN0ZW50XG4gKiBJQ01QIHRyYWZmaWMgdHlwZS5cbiAqL1xuY29uc3QgTUFUQ0hfTk9fVFJBRkZJQyA9IHtcbiAgY2lkcklwOiAnMjU1LjI1NS4yNTUuMjU1LzMyJyxcbiAgZGVzY3JpcHRpb246ICdEaXNhbGxvdyBhbGwgdHJhZmZpYycsXG4gIGlwUHJvdG9jb2w6ICdpY21wJyxcbiAgZnJvbVBvcnQ6IDI1MixcbiAgdG9Qb3J0OiA4Nixcbn07XG5cbmNvbnN0IE5PX1RSQUZGSUNfUEVFUiA9IFBlZXIuaXB2NChNQVRDSF9OT19UUkFGRklDLmNpZHJJcCk7XG5jb25zdCBOT19UUkFGRklDX1BPUlQgPSBQb3J0LmljbXBUeXBlQW5kQ29kZShNQVRDSF9OT19UUkFGRklDLmZyb21Qb3J0LCBNQVRDSF9OT19UUkFGRklDLnRvUG9ydCk7XG5cbi8qKlxuICogRWdyZXNzIHJ1bGUgdGhhdCBtYXRjaGVzIGFsbCB0cmFmZmljXG4gKi9cbmNvbnN0IEFMTE9XX0FMTF9SVUxFID0ge1xuICBjaWRySXA6ICcwLjAuMC4wLzAnLFxuICBkZXNjcmlwdGlvbjogJ0FsbG93IGFsbCBvdXRib3VuZCB0cmFmZmljIGJ5IGRlZmF1bHQnLFxuICBpcFByb3RvY29sOiAnLTEnLFxufTtcblxuY29uc3QgQUxMX1RSQUZGSUNfUEVFUiA9IFBlZXIuYW55SXB2NCgpO1xuY29uc3QgQUxMX1RSQUZGSUNfUE9SVCA9IFBvcnQuYWxsVHJhZmZpYygpO1xuXG5leHBvcnQgaW50ZXJmYWNlIENvbm5lY3Rpb25SdWxlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcm90b2NvbD86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZnJvbVBvcnQ6IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHRvUG9ydD86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIENvbXBhcmUgdHdvIGluZ3Jlc3MgcnVsZXMgZm9yIGVxdWFsaXR5IHRoZSBzYW1lIHdheSBDbG91ZEZvcm1hdGlvbiB3b3VsZCAoZGlzY2FyZGluZyBkZXNjcmlwdGlvbilcbiAqL1xuZnVuY3Rpb24gaW5ncmVzc1J1bGVzRXF1YWwoYTogQ2ZuU2VjdXJpdHlHcm91cC5JbmdyZXNzUHJvcGVydHksIGI6IENmblNlY3VyaXR5R3JvdXAuSW5ncmVzc1Byb3BlcnR5KSB7XG4gIHJldHVybiBhLmNpZHJJcCA9PT0gYi5jaWRySXBcbiAgICAmJiBhLmNpZHJJcHY2ID09PSBiLmNpZHJJcHY2XG4gICAgJiYgYS5mcm9tUG9ydCA9PT0gYi5mcm9tUG9ydFxuICAgICYmIGEudG9Qb3J0ID09PSBiLnRvUG9ydFxuICAgICYmIGEuaXBQcm90b2NvbCA9PT0gYi5pcFByb3RvY29sXG4gICAgJiYgYS5zb3VyY2VTZWN1cml0eUdyb3VwSWQgPT09IGIuc291cmNlU2VjdXJpdHlHcm91cElkXG4gICAgJiYgYS5zb3VyY2VTZWN1cml0eUdyb3VwTmFtZSA9PT0gYi5zb3VyY2VTZWN1cml0eUdyb3VwTmFtZVxuICAgICYmIGEuc291cmNlU2VjdXJpdHlHcm91cE93bmVySWQgPT09IGIuc291cmNlU2VjdXJpdHlHcm91cE93bmVySWQ7XG59XG5cbi8qKlxuICogQ29tcGFyZSB0d28gZWdyZXNzIHJ1bGVzIGZvciBlcXVhbGl0eSB0aGUgc2FtZSB3YXkgQ2xvdWRGb3JtYXRpb24gd291bGQgKGRpc2NhcmRpbmcgZGVzY3JpcHRpb24pXG4gKi9cbmZ1bmN0aW9uIGVncmVzc1J1bGVzRXF1YWwoYTogQ2ZuU2VjdXJpdHlHcm91cC5FZ3Jlc3NQcm9wZXJ0eSwgYjogQ2ZuU2VjdXJpdHlHcm91cC5FZ3Jlc3NQcm9wZXJ0eSkge1xuICByZXR1cm4gYS5jaWRySXAgPT09IGIuY2lkcklwXG4gICAgJiYgYS5jaWRySXB2NiA9PT0gYi5jaWRySXB2NlxuICAgICYmIGEuZnJvbVBvcnQgPT09IGIuZnJvbVBvcnRcbiAgICAmJiBhLnRvUG9ydCA9PT0gYi50b1BvcnRcbiAgICAmJiBhLmlwUHJvdG9jb2wgPT09IGIuaXBQcm90b2NvbFxuICAgICYmIGEuZGVzdGluYXRpb25QcmVmaXhMaXN0SWQgPT09IGIuZGVzdGluYXRpb25QcmVmaXhMaXN0SWRcbiAgICAmJiBhLmRlc3RpbmF0aW9uU2VjdXJpdHlHcm91cElkID09PSBiLmRlc3RpbmF0aW9uU2VjdXJpdHlHcm91cElkO1xufVxuXG4vKipcbiAqIFdoZXRoZXIgdGhpcyBydWxlIHJlZmVycyB0byBhbGwgdHJhZmZpY1xuICovXG5mdW5jdGlvbiBpc0FsbFRyYWZmaWNSdWxlKHJ1bGU6IGFueSkge1xuICByZXR1cm4gcnVsZS5jaWRySXAgPT09ICcwLjAuMC4wLzAnICYmIHJ1bGUuaXBQcm90b2NvbCA9PT0gJy0xJztcbn1cbiJdfQ==