"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// tslint:disable:object-literal-key-quotes
const ACTIONS = [
    'kms:Create*',
    'kms:Describe*',
    'kms:Enable*',
    'kms:List*',
    'kms:Put*',
    'kms:Update*',
    'kms:Revoke*',
    'kms:Disable*',
    'kms:Get*',
    'kms:Delete*',
    'kms:ScheduleKeyDeletion',
    'kms:CancelKeyDeletion',
    'kms:GenerateDataKey',
    'kms:TagResource',
    'kms:UntagResource',
];
module.exports = {
    'default key'(test) {
        const stack = new core_1.Stack();
        new lib_1.Key(stack, 'MyKey');
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: 'AWS::KMS::Key',
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: ACTIONS,
                                    Effect: 'Allow',
                                    Principal: {
                                        AWS: {
                                            'Fn::Join': [
                                                '',
                                                [
                                                    'arn:',
                                                    {
                                                        Ref: 'AWS::Partition',
                                                    },
                                                    ':iam::',
                                                    {
                                                        Ref: 'AWS::AccountId',
                                                    },
                                                    ':root',
                                                ],
                                            ],
                                        },
                                    },
                                    Resource: '*',
                                },
                            ],
                            Version: '2012-10-17',
                        },
                    },
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain',
                },
            },
        }));
        test.done();
    },
    'default with no retention'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'TestStack');
        new lib_1.Key(stack, 'MyKey', { removalPolicy: core_1.RemovalPolicy.DESTROY });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', { DeletionPolicy: 'Delete', UpdateReplacePolicy: 'Delete' }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default with some permission'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey');
        const p = new iam.PolicyStatement({ resources: ['*'], actions: ['kms:encrypt'] });
        p.addArnPrincipal('arn');
        key.addToResourcePolicy(p);
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: 'AWS::KMS::Key',
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: ACTIONS,
                                    Effect: 'Allow',
                                    Principal: {
                                        AWS: {
                                            'Fn::Join': [
                                                '',
                                                [
                                                    'arn:',
                                                    {
                                                        Ref: 'AWS::Partition',
                                                    },
                                                    ':iam::',
                                                    {
                                                        Ref: 'AWS::AccountId',
                                                    },
                                                    ':root',
                                                ],
                                            ],
                                        },
                                    },
                                    Resource: '*',
                                },
                                {
                                    Action: 'kms:encrypt',
                                    Effect: 'Allow',
                                    Principal: {
                                        AWS: 'arn',
                                    },
                                    Resource: '*',
                                },
                            ],
                            Version: '2012-10-17',
                        },
                    },
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain',
                },
            },
        }));
        test.done();
    },
    'key with some options'(test) {
        const stack = new core_1.Stack();
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false,
        });
        const p = new iam.PolicyStatement({ resources: ['*'], actions: ['kms:encrypt'] });
        p.addArnPrincipal('arn');
        key.addToResourcePolicy(p);
        key.node.applyAspect(new core_1.Tag('tag1', 'value1'));
        key.node.applyAspect(new core_1.Tag('tag2', 'value2'));
        key.node.applyAspect(new core_1.Tag('tag3', ''));
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: 'AWS::KMS::Key',
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: ACTIONS,
                                    Effect: 'Allow',
                                    Principal: {
                                        AWS: {
                                            'Fn::Join': [
                                                '',
                                                [
                                                    'arn:',
                                                    {
                                                        Ref: 'AWS::Partition',
                                                    },
                                                    ':iam::',
                                                    {
                                                        Ref: 'AWS::AccountId',
                                                    },
                                                    ':root',
                                                ],
                                            ],
                                        },
                                    },
                                    Resource: '*',
                                },
                                {
                                    Action: 'kms:encrypt',
                                    Effect: 'Allow',
                                    Principal: {
                                        AWS: 'arn',
                                    },
                                    Resource: '*',
                                },
                            ],
                            Version: '2012-10-17',
                        },
                        Enabled: false,
                        EnableKeyRotation: true,
                        Tags: [
                            {
                                Key: 'tag1',
                                Value: 'value1',
                            },
                            {
                                Key: 'tag2',
                                Value: 'value2',
                            },
                            {
                                Key: 'tag3',
                                Value: '',
                            },
                        ],
                    },
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain',
                },
            },
        }));
        test.done();
    },
    'addAlias creates an alias'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false,
        });
        const alias = key.addAlias('alias/xoo');
        test.ok(alias.aliasName);
        assert_1.expect(stack).to(assert_1.countResources('AWS::KMS::Alias', 1));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Alias', {
            AliasName: 'alias/xoo',
            TargetKeyId: {
                'Fn::GetAtt': [
                    'MyKey6AB29FA6',
                    'Arn',
                ],
            },
        }));
        test.done();
    },
    'can run multiple addAlias'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false,
        });
        const alias1 = key.addAlias('alias/alias1');
        const alias2 = key.addAlias('alias/alias2');
        test.ok(alias1.aliasName);
        test.ok(alias2.aliasName);
        assert_1.expect(stack).to(assert_1.countResources('AWS::KMS::Alias', 2));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Alias', {
            AliasName: 'alias/alias1',
            TargetKeyId: {
                'Fn::GetAtt': [
                    'MyKey6AB29FA6',
                    'Arn',
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Alias', {
            AliasName: 'alias/alias2',
            TargetKeyId: {
                'Fn::GetAtt': [
                    'MyKey6AB29FA6',
                    'Arn',
                ],
            },
        }));
        test.done();
    },
    'grant decrypt on a key'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const key = new lib_1.Key(stack, 'Key');
        const user = new iam.User(stack, 'User');
        // WHEN
        key.grantDecrypt(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    // This one is there by default
                    {
                        Action: ACTIONS,
                        Effect: 'Allow',
                        Principal: { AWS: { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::', { Ref: 'AWS::AccountId' }, ':root']] } },
                        Resource: '*',
                    },
                    // This is the interesting one
                    {
                        Action: 'kms:Decrypt',
                        Effect: 'Allow',
                        Principal: { AWS: { 'Fn::GetAtt': ['User00B015A1', 'Arn'] } },
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'kms:Decrypt',
                        Effect: 'Allow',
                        Resource: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'grant for a principal in a dependent stack works correctly'(test) {
        const app = new core_1.App();
        const principalStack = new core_1.Stack(app, 'PrincipalStack');
        const principal = new iam.Role(principalStack, 'Role', {
            assumedBy: new iam.AnyPrincipal(),
        });
        const keyStack = new core_1.Stack(app, 'KeyStack');
        const key = new lib_1.Key(keyStack, 'Key');
        principalStack.addDependency(keyStack);
        key.grantEncrypt(principal);
        assert_1.expect(keyStack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            'KeyPolicy': {
                'Statement': [
                    {
                    // owning account management permissions - we don't care about them in this test
                    },
                    {
                        'Action': [
                            'kms:Encrypt',
                            'kms:ReEncrypt*',
                            'kms:GenerateDataKey*',
                        ],
                        'Effect': 'Allow',
                        'Principal': {
                            'AWS': {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { 'Ref': 'AWS::Partition' },
                                        ':iam::',
                                        { 'Ref': 'AWS::AccountId' },
                                        ':root',
                                    ]],
                            },
                        },
                        'Resource': '*',
                    },
                ],
            },
        }));
        test.done();
    },
    'keyId resolves to a Ref'(test) {
        const stack = new core_1.Stack();
        const key = new lib_1.Key(stack, 'MyKey');
        new core_1.CfnOutput(stack, 'Out', {
            value: key.keyId,
        });
        const template = assert_1.SynthUtils.synthesize(stack).template.Outputs;
        test.deepEqual(template, {
            'Out': {
                'Value': {
                    'Ref': 'MyKey6AB29FA6',
                },
            },
        });
        test.done();
    },
    'enablePolicyControl changes key policy to allow IAM control'(test) {
        const stack = new core_1.Stack();
        new lib_1.Key(stack, 'MyKey', { trustAccountIdentities: true });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            'KeyPolicy': {
                'Statement': [
                    {
                        'Action': 'kms:*',
                        'Effect': 'Allow',
                        'Principal': {
                            'AWS': {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { 'Ref': 'AWS::Partition' },
                                        ':iam::',
                                        { 'Ref': 'AWS::AccountId' },
                                        ':root',
                                    ]],
                            },
                        },
                        'Resource': '*',
                    },
                ],
            },
        }));
        test.done();
    },
    'imported keys': {
        'throw an error when providing something that is not a valid key ARN'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                lib_1.Key.fromKeyArn(stack, 'Imported', 'arn:aws:kms:us-east-1:123456789012:key');
            }, /KMS key ARN must be in the format 'arn:aws:kms:<region>:<account>:key\/<keyId>', got: 'arn:aws:kms:us-east-1:123456789012:key'/);
            test.done();
        },
        'can have aliases added to them'(test) {
            const stack2 = new core_1.Stack();
            const myKeyImported = lib_1.Key.fromKeyArn(stack2, 'MyKeyImported', 'arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012');
            // addAlias can be called on imported keys.
            myKeyImported.addAlias('alias/hello');
            test.equal(myKeyImported.keyId, '12345678-1234-1234-1234-123456789012');
            assert_1.expect(stack2).toMatch({
                Resources: {
                    MyKeyImportedAliasB1C5269F: {
                        Type: 'AWS::KMS::Alias',
                        Properties: {
                            AliasName: 'alias/hello',
                            TargetKeyId: 'arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012',
                        },
                    },
                },
            });
            test.done();
        },
        'addToResourcePolicy allowNoOp and there is no policy': {
            'succeed if set to true (default)'(test) {
                const stack = new core_1.Stack();
                const key = lib_1.Key.fromKeyArn(stack, 'Imported', 'arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012');
                key.addToResourcePolicy(new iam.PolicyStatement({ resources: ['*'], actions: ['*'] }));
                test.done();
            },
            'fails if set to false'(test) {
                const stack = new core_1.Stack();
                const key = lib_1.Key.fromKeyArn(stack, 'Imported', 'arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012');
                test.throws(() => {
                    key.addToResourcePolicy(new iam.PolicyStatement({ resources: ['*'], actions: ['*'] }), /* allowNoOp */ false);
                }, 'Unable to add statement to IAM resource policy for KMS key: "foo/bar"');
                test.done();
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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