"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ToolkitCleaner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_assets_1 = require("aws-cdk-lib/aws-ecr-assets");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
const clean_images_function_1 = require("./clean-images-function");
const clean_objects_function_1 = require("./clean-objects-function");
const extract_template_hashes_function_1 = require("./extract-template-hashes-function");
const get_stack_names_function_1 = require("./get-stack-names-function");
/**
 * Clean unused S3 and ECR assets from your CDK Toolkit.
 */
class ToolkitCleaner extends constructs_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        // Dummy assets to reference S3 bucket and ECR repository
        const fileAsset = new aws_s3_assets_1.Asset(this, 'FileAsset', {
            path: path.join(__dirname, '..', '..', 'assets', 'toolkit-cleaner', 'docker', 'dummy.txt'),
        });
        const dockerImageAsset = new aws_ecr_assets_1.DockerImageAsset(this, 'DockerImageAsset', {
            directory: path.join(__dirname, '..', '..', 'assets', 'toolkit-cleaner', 'docker'),
        });
        const getStackNamesFunction = new get_stack_names_function_1.GetStackNamesFunction(this, 'GetStackNamesFunction', {
            timeout: aws_cdk_lib_1.Duration.seconds(30),
        });
        getStackNamesFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['cloudformation:DescribeStacks', 'cloudformation:ListStacks'],
            resources: ['*'],
        }));
        const getStackNames = new tasks.LambdaInvoke(this, 'GetStackNames', {
            lambdaFunction: getStackNamesFunction,
            payloadResponseOnly: true,
        });
        const stacksMap = new sfn.Map(this, 'StacksMap', {
            maxConcurrency: 1,
            resultSelector: {
                AssetHashes: sfn.JsonPath.stringAt('$'),
            },
        });
        const extractTemplateHashesFunction = new extract_template_hashes_function_1.ExtractTemplateHashesFunction(this, 'ExtractTemplateHashesFunction', {
            timeout: aws_cdk_lib_1.Duration.seconds(30),
        });
        extractTemplateHashesFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['cloudformation:GetTemplate'],
            resources: ['*'],
        }));
        const extractTemplateHashes = new tasks.LambdaInvoke(this, 'ExtractTemplateHashes', {
            lambdaFunction: extractTemplateHashesFunction,
            payloadResponseOnly: true,
        }).addRetry({
            errors: ['Throttling'],
        });
        const flattenHashes = new tasks.EvaluateExpression(this, 'FlattenHashes', {
            expression: '[...new Set(($.AssetHashes).flat())]',
        });
        const cleanObjectsFunction = new clean_objects_function_1.CleanObjectsFunction(this, 'CleanObjectsFunction', {
            timeout: aws_cdk_lib_1.Duration.minutes(5),
        });
        cleanObjectsFunction.addEnvironment('BUCKET_NAME', fileAsset.bucket.bucketName);
        fileAsset.bucket.grantRead(cleanObjectsFunction);
        fileAsset.bucket.grantDelete(cleanObjectsFunction);
        const cleanObjects = new tasks.LambdaInvoke(this, 'CleanObjects', {
            lambdaFunction: cleanObjectsFunction,
            payloadResponseOnly: true,
        });
        const cleanImagesFunction = new clean_images_function_1.CleanImagesFunction(this, 'CleanImagesFunction', {
            timeout: aws_cdk_lib_1.Duration.minutes(5),
        });
        cleanImagesFunction.addEnvironment('REPOSITORY_NAME', dockerImageAsset.repository.repositoryName);
        dockerImageAsset.repository.grant(cleanImagesFunction, 'ecr:DescribeImages', 'ecr:BatchDeleteImage');
        const cleanImages = new tasks.LambdaInvoke(this, 'CleanImages', {
            lambdaFunction: cleanImagesFunction,
            payloadResponseOnly: true,
        });
        if (!props.dryRun) {
            cleanObjectsFunction.addEnvironment('RUN', 'true');
            cleanImagesFunction.addEnvironment('RUN', 'true');
        }
        if (props.retainAssetsNewerThan) {
            const retainMilliseconds = props.retainAssetsNewerThan.toMilliseconds().toString();
            cleanObjectsFunction.addEnvironment('RETAIN_MILLISECONDS', retainMilliseconds);
            cleanImagesFunction.addEnvironment('RETAIN_MILLISECONDS', retainMilliseconds);
        }
        const sumReclaimed = new tasks.EvaluateExpression(this, 'SumReclaimed', {
            expression: '({ Deleted: $[0].Deleted + $[1].Deleted, Reclaimed: $[0].Reclaimed + $[1].Reclaimed })',
        });
        const stateMachine = new sfn.StateMachine(this, 'Resource', {
            definition: getStackNames
                .next(stacksMap.iterator(extractTemplateHashes))
                .next(flattenHashes)
                .next(new sfn.Parallel(this, 'Clean')
                .branch(cleanObjects)
                .branch(cleanImages))
                .next(sumReclaimed),
        });
        const rule = new aws_events_1.Rule(this, 'Rule', {
            enabled: props.scheduleEnabled ?? true,
            schedule: props.schedule ?? aws_events_1.Schedule.rate(aws_cdk_lib_1.Duration.days(1)),
        });
        rule.addTarget(new aws_events_targets_1.SfnStateMachine(stateMachine));
    }
}
exports.ToolkitCleaner = ToolkitCleaner;
_a = JSII_RTTI_SYMBOL_1;
ToolkitCleaner[_a] = { fqn: "cloudstructs.ToolkitCleaner", version: "0.6.25" };
//# sourceMappingURL=data:application/json;base64,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