import * as secretsmanager from 'aws-cdk-lib/aws-secretsmanager';
import { Construct, IConstruct } from 'constructs';
import { SlackAppManifestProps } from './manifest';
/**
 * Properties for a SlackApp
 */
export interface SlackAppProps {
    /**
     * The definition of the app manifest
     *
     * @see https://api.slack.com/reference/manifests
     */
    readonly manifest: SlackAppManifestDefinition;
    /**
     * An AWS Secrets Manager secret containing the app configuration token
     *
     * Must use the following JSON format:
     *
     * ```
     * {
     *   "refreshToken": "<token>"
     * }
     * ```
     */
    readonly configurationTokenSecret: secretsmanager.ISecret;
    /**
     * The AWS Secrets Manager secret where to store the app credentials
     *
     * @default - a new secret is created
     */
    readonly credentialsSecret?: secretsmanager.ISecret;
}
/**
 * A Slack app manifest definition
 */
export declare abstract class SlackAppManifestDefinition {
    /**
     * Create a Slack app manifest from a JSON app manifest encoded as a string
     */
    static fromString(manifest: string): SlackAppManifestDefinition;
    /**
     * Creates a Slack app manifest from a file containg a JSON app manifest
     */
    static fromFile(file: string): SlackAppManifestDefinition;
    /**
     * Creates a Slack app manifest by specifying properties
     */
    static fromManifest(props: SlackAppManifestProps): SlackAppManifestDefinition;
    /**
     * Renders the JSON app manifest encoded as a string
     */
    abstract render(construct: IConstruct): string;
}
/**
 * A Slack application deployed with a manifest
 *
 * @see https://api.slack.com/reference/manifests
 */
export declare class SlackApp extends Construct {
    /**
     * The ID of the application
     */
    readonly appId: string;
    /**
     * An AWS Secrets Manager secret containing the credentials of the application.
     *
     * ```
     * {
     *   "appId": "...",
     *   "clientId": "...",
     *   "clientSecret": "...",
     *   "verificationToken": "...",
     *   "signingSecret": "..."
     * }
     * ```
     */
    readonly credentials: secretsmanager.ISecret;
    /**
     * A dynamic reference to the client ID of the app
     */
    readonly clientId: string;
    /**
     * A dynamic reference to the client secret of the app
     */
    readonly clientSecret: string;
    /**
     * A dynamic reference to the verification token of the app
     */
    readonly verificationToken: string;
    /**
     * A dynamic reference to the signing secret of the app
     */
    readonly signingSecret: string;
    constructor(scope: Construct, id: string, props: SlackAppProps);
}
