# Overview

While components encapsulate functionality using life cycle events, it is up to the execution to invoke the components. The event paradigm of the components allows for the composition of arbitrary component execution schedules. To make this more concrete, consider the following simple example:

```python
from machinable import Experiment, execute

execute(Experiment().component("optimization").repeat(3))
```

The execution definition can be read as *Import component 'optimization' and repeat its execution in three independent trials*. Note that the experiment object does not trigger the execution but merely describes the execution plan and is then triggered using the ``execute`` method. 

Crucially, machinable can take care of the intricacies of the execution based on this high-level description, i.e. import and construction of the components and trigger of its event life cycle. It can also keep track of the used configuration, generate seeds for controlled randomness and prepare a unique storage path to keep results. Since the execution details are abstracted away, it does not matter whether you run on a local computer or a distributed remote cluster.

## Defining executions

All aspect of the execution can be controlled as arguments of the [execute()](../reference/execution.md#execute) method.

```python
from machinable import execute
execute(
    experiment,  # which components with what configuration
    storage,     # where to store results etc.
    engine,      # execution target, e.g. remote execution, multiprocessing etc.
    index,       # database that can be used to search for executions later
    project,     # the machinable project to use
    seed         # random seed
)
```

For even finer grained control, you can instantiate the [Execution](../reference/execution.md#execution) directly using the same arguments. In fact, `execute` is an alias for `Execution().summary().submit()`.

For every execution, machinable will generate a unique 6-digit experiment ID (e.g. `OY1p1o`) that will be printed at the beginning of the execution output. The ID encodes the global random seed and is used as a relative directory to write any data generated by the experiment.

### Experiment

The experiment is the only required argument of the execution and specifies what components are ought to be executed. In the simplest case, it can be the name of a single component that will be executed using its default configuration. We will discuss the experiment specification in detail in the [following section](./experiments.md).
The experiment is the only required argument of the execution and specifies what components are ought to be executed. In the simplest case, it can be the name of a single component that will be executed using its default configuration. We will discuss the experiment specification in detail in the [following section](./experiments.md).

### Storage

By default, the storage is the non-permanent system memory which is useful during development. To keep your results, make sure to pass in a [filesystem url](https://docs.pyfilesystem.org/en/latest/openers.html) to the `storage` parameter.

``` python
execute(..., storage='~/results')         # local file system
execute(..., storage='s3://bucket')       # s3 store
```

### Engines

While experiments are executed locally and sequential by default, machinable provides different [Engines](./engines.md) for parallel and remote execution. For example, to execute components in parallel processes you may specify the number of processes:

``` python
execute(..., engine='native:5')
```

To learn more about available engines and their options, refer to the [Engine](./engines.md) section.

### Seeding and reproducibility

machinable chooses and sets a global random seed automatically at execution time. You can also determine the seed with the `seed` parameter by passing in a number or an experiment ID:

``` python
execute(Experiment().component('controlled_randomness'), seed=42)
```

To re-use the seed of a given experiment and reproduce the execution results, you can pass the experiment ID as the seed:

```python 
execute(Experiment().component('controlled_randomness'), seed='OY1p1o')
```

If you need more control over randomness and how packages are being seeded, you can overwrite the `on_seeding` event in your component class.

## Code backups

machinable automatically backs up the code base at execution time in a zip file that can be used to reproduce the results. Note that the project directory needs to be under Git-version control to determine which files are included and ignored during the backup (``.gitignore`` file).
