import { Duration } from '@aws-cdk/core';
/**
 * Interface for metrics.
 */
export interface IMetric {
    /**
     * Inspect the details of the metric object.
     */
    toMetricConfig(): MetricConfig;
    /**
     * (deprecated) Turn this metric object into an alarm configuration.
     *
     * @deprecated Use `toMetricsConfig()` instead.
     */
    toAlarmConfig(): MetricAlarmConfig;
    /**
     * (deprecated) Turn this metric object into a graph configuration.
     *
     * @deprecated Use `toMetricsConfig()` instead.
     */
    toGraphConfig(): MetricGraphConfig;
}
/**
 * Metric dimension.
 */
export interface Dimension {
    /**
     * Name of the dimension.
     */
    readonly name: string;
    /**
     * Value of the dimension.
     */
    readonly value: any;
}
/**
 * Statistic to use over the aggregation period.
 */
export declare enum Statistic {
    /**
     * The count (number) of data points used for the statistical calculation.
     */
    SAMPLE_COUNT = "SampleCount",
    /**
     * The value of Sum / SampleCount during the specified period.
     */
    AVERAGE = "Average",
    /**
     * All values submitted for the matching metric added together.
     *
     * This statistic can be useful for determining the total volume of a metric.
     */
    SUM = "Sum",
    /**
     * The lowest value observed during the specified period.
     *
     * You can use this value to determine low volumes of activity for your application.
     */
    MINIMUM = "Minimum",
    /**
     * The highest value observed during the specified period.
     *
     * You can use this value to determine high volumes of activity for your application.
     */
    MAXIMUM = "Maximum"
}
/**
 * Unit for metric.
 */
export declare enum Unit {
    /**
     * Seconds.
     */
    SECONDS = "Seconds",
    /**
     * Microseconds.
     */
    MICROSECONDS = "Microseconds",
    /**
     * Milliseconds.
     */
    MILLISECONDS = "Milliseconds",
    /**
     * Bytes.
     */
    BYTES = "Bytes",
    /**
     * Kilobytes.
     */
    KILOBYTES = "Kilobytes",
    /**
     * Megabytes.
     */
    MEGABYTES = "Megabytes",
    /**
     * Gigabytes.
     */
    GIGABYTES = "Gigabytes",
    /**
     * Terabytes.
     */
    TERABYTES = "Terabytes",
    /**
     * Bits.
     */
    BITS = "Bits",
    /**
     * Kilobits.
     */
    KILOBITS = "Kilobits",
    /**
     * Megabits.
     */
    MEGABITS = "Megabits",
    /**
     * Gigabits.
     */
    GIGABITS = "Gigabits",
    /**
     * Terabits.
     */
    TERABITS = "Terabits",
    /**
     * Percent.
     */
    PERCENT = "Percent",
    /**
     * Count.
     */
    COUNT = "Count",
    /**
     * Bytes/second (B/s).
     */
    BYTES_PER_SECOND = "Bytes/Second",
    /**
     * Kilobytes/second (kB/s).
     */
    KILOBYTES_PER_SECOND = "Kilobytes/Second",
    /**
     * Megabytes/second (MB/s).
     */
    MEGABYTES_PER_SECOND = "Megabytes/Second",
    /**
     * Gigabytes/second (GB/s).
     */
    GIGABYTES_PER_SECOND = "Gigabytes/Second",
    /**
     * Terabytes/second (TB/s).
     */
    TERABYTES_PER_SECOND = "Terabytes/Second",
    /**
     * Bits/second (b/s).
     */
    BITS_PER_SECOND = "Bits/Second",
    /**
     * Kilobits/second (kb/s).
     */
    KILOBITS_PER_SECOND = "Kilobits/Second",
    /**
     * Megabits/second (Mb/s).
     */
    MEGABITS_PER_SECOND = "Megabits/Second",
    /**
     * Gigabits/second (Gb/s).
     */
    GIGABITS_PER_SECOND = "Gigabits/Second",
    /**
     * Terabits/second (Tb/s).
     */
    TERABITS_PER_SECOND = "Terabits/Second",
    /**
     * Count/second.
     */
    COUNT_PER_SECOND = "Count/Second",
    /**
     * No unit.
     */
    NONE = "None"
}
/**
 * Properties of a rendered metric.
 */
export interface MetricConfig {
    /**
     * In case the metric represents a query, the details of the query.
     *
     * @default - None
     */
    readonly metricStat?: MetricStatConfig;
    /**
     * In case the metric is a math expression, the details of the math expression.
     *
     * @default - None
     */
    readonly mathExpression?: MetricExpressionConfig;
    /**
     * Additional properties which will be rendered if the metric is used in a dashboard.
     *
     * Examples are 'label' and 'color', but any key in here will be
     * added to dashboard graphs.
     *
     * @default - None
     */
    readonly renderingProperties?: Record<string, unknown>;
}
/**
 * Properties for a concrete metric.
 *
 * NOTE: `unit` is no longer on this object since it is only used for `Alarms`, and doesn't mean what one
 * would expect it to mean there anyway. It is most likely to be misused.
 */
export interface MetricStatConfig {
    /**
     * The dimensions to apply to the alarm.
     *
     * @default []
     */
    readonly dimensions?: Dimension[];
    /**
     * Namespace of the metric.
     */
    readonly namespace: string;
    /**
     * Name of the metric.
     */
    readonly metricName: string;
    /**
     * How many seconds to aggregate over.
     */
    readonly period: Duration;
    /**
     * Aggregation function to use (can be either simple or a percentile).
     */
    readonly statistic: string;
    /**
     * Unit used to filter the metric stream.
     *
     * Only refer to datums emitted to the metric stream with the given unit and
     * ignore all others. Only useful when datums are being emitted to the same
     * metric stream under different units.
     *
     * This field has been renamed from plain `unit` to clearly communicate
     * its purpose.
     *
     * @default - Refer to all metric datums
     */
    readonly unitFilter?: Unit;
    /**
     * Region which this metric comes from.
     *
     * @default Deployment region.
     */
    readonly region?: string;
    /**
     * Account which this metric comes from.
     *
     * @default Deployment account.
     */
    readonly account?: string;
}
/**
 * Properties for a concrete metric.
 */
export interface MetricExpressionConfig {
    /**
     * Math expression for the metric.
     */
    readonly expression: string;
    /**
     * Metrics used in the math expression.
     */
    readonly usingMetrics: Record<string, IMetric>;
    /**
     * How many seconds to aggregate over.
     */
    readonly period: number;
}
/**
 * (deprecated) Properties used to construct the Metric identifying part of an Alarm.
 *
 * @deprecated Replaced by MetricConfig
 */
export interface MetricAlarmConfig {
    /**
     * (deprecated) The dimensions to apply to the alarm.
     *
     * @deprecated
     */
    readonly dimensions?: Dimension[];
    /**
     * (deprecated) Namespace of the metric.
     *
     * @deprecated
     */
    readonly namespace: string;
    /**
     * (deprecated) Name of the metric.
     *
     * @deprecated
     */
    readonly metricName: string;
    /**
     * (deprecated) How many seconds to aggregate over.
     *
     * @deprecated
     */
    readonly period: number;
    /**
     * (deprecated) Simple aggregation function to use.
     *
     * @deprecated
     */
    readonly statistic?: Statistic;
    /**
     * (deprecated) Percentile aggregation function to use.
     *
     * @deprecated
     */
    readonly extendedStatistic?: string;
    /**
     * (deprecated) The unit of the alarm.
     *
     * @deprecated
     */
    readonly unit?: Unit;
}
/**
 * (deprecated) Properties used to construct the Metric identifying part of a Graph.
 *
 * @deprecated Replaced by MetricConfig
 */
export interface MetricGraphConfig {
    /**
     * (deprecated) The dimensions to apply to the alarm.
     *
     * @deprecated
     */
    readonly dimensions?: Dimension[];
    /**
     * (deprecated) Namespace of the metric.
     *
     * @deprecated
     */
    readonly namespace: string;
    /**
     * (deprecated) Name of the metric.
     *
     * @deprecated
     */
    readonly metricName: string;
    /**
     * (deprecated) Rendering properties override yAxis parameter of the widget object.
     *
     * @deprecated
     */
    readonly renderingProperties: MetricRenderingProperties;
    /**
     * (deprecated) How many seconds to aggregate over.
     *
     * @deprecated Use `period` in `renderingProperties`
     */
    readonly period: number;
    /**
     * (deprecated) Label for the metric.
     *
     * @deprecated Use `label` in `renderingProperties`
     */
    readonly label?: string;
    /**
     * (deprecated) Color for the graph line.
     *
     * @deprecated Use `color` in `renderingProperties`
     */
    readonly color?: string;
    /**
     * (deprecated) Aggregation function to use (can be either simple or a percentile).
     *
     * @deprecated Use `stat` in `renderingProperties`
     */
    readonly statistic?: string;
    /**
     * (deprecated) The unit of the alarm.
     *
     * @deprecated not used in dashboard widgets
     */
    readonly unit?: Unit;
}
/**
 * (deprecated) Custom rendering properties that override the default rendering properties specified in the yAxis parameter of the widget object.
 *
 * @deprecated Replaced by MetricConfig.
 */
export interface MetricRenderingProperties {
    /**
     * (deprecated) How many seconds to aggregate over.
     *
     * @deprecated
     */
    readonly period: number;
    /**
     * (deprecated) Label for the metric.
     *
     * @deprecated
     */
    readonly label?: string;
    /**
     * (deprecated) The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The `Color` class has a set of standard colors that can be used here.
     *
     * @deprecated
     */
    readonly color?: string;
    /**
     * (deprecated) Aggregation function to use (can be either simple or a percentile).
     *
     * @deprecated
     */
    readonly stat?: string;
}
