from .context import gpytoolbox as gpy
from .context import numpy as np
from .context import unittest
import scipy as sp

class TestCotangentLaplacian(unittest.TestCase):

    def test_single_triangle_2d(self):
        v = np.array([[0.0,0.0],[1.0,0.0],[0.0,1.0]])
        f = np.array([[0,1,2]],dtype=int)
        L = gpy.cotangent_laplacian(v,f)
        L_gt_arr = np.array([[1., -0.5, -0.5], [-0.5, 0.5, 0.], [-0.5, 0., 0.5]])
        self.assertTrue(np.isclose(L.toarray(), L_gt_arr).all())
    
    def test_single_triangle_3d(self):
        v = np.array([[0.0,0.0,0.0],[1.0,0.0,0.0],[0.0,0.0,1.0]])
        f = np.array([[0,1,2]],dtype=int)
        L = gpy.cotangent_laplacian(v,f)
        L_gt_arr = np.array([[1., -0.5, -0.5], [-0.5, 0.5, 0.], [-0.5, 0., 0.5]])
        self.assertTrue(np.isclose(L.toarray(), L_gt_arr).all())

    def test_wrt_grad(self):
        meshes = ["armadillo.obj", "bunny.obj", "mountain.obj"]
        for mesh in meshes:
            v,f = gpy.read_mesh("test/unit_tests_data/" + mesh)

            L = gpy.cotangent_laplacian(v,f)

            A = 0.5 * gpy.doublearea(v,f)
            M = sp.sparse.diags(diagonals=np.concatenate((A,A,A)), format="csr")
            G = gpy.grad(v,f)
            L_grad = G.transpose() * M * G

            self.assertTrue(np.isclose(sp.sparse.linalg.norm(L-L_grad) / L.nnz, 0.))

    def test_bunny_oded(self):
        v,f = gpy.read_mesh("test/unit_tests_data/bunny_oded.obj")

        L = gpy.cotangent_laplacian(v,f)

        data = np.array([ 3.77657441e+00, -1.23687651e+00, -1.67329878e-01, -5.42416714e-01,
       -7.59533623e-01, -4.81226408e-01, -5.89191277e-01, -1.23687651e+00,
        3.99437134e+00, -7.92712920e-01, -4.66948122e-01, -1.11800155e+00,
       -3.79832244e-01, -1.67329878e-01, -7.92712920e-01,  3.60308223e+00,
       -3.62054882e-01, -5.46829746e-01, -4.69126744e-01, -7.37933623e-01,
       -5.27094442e-01, -5.42416714e-01, -4.66948122e-01,  3.49182225e+00,
       -6.33036703e-01, -6.84846441e-01, -5.53660369e-01, -6.10913899e-01,
       -1.11800155e+00, -3.62054882e-01,  3.69759821e+00, -7.08704653e-01,
       -5.55282405e-01, -4.30392029e-01, -5.23162696e-01, -7.59533623e-01,
       -5.46829746e-01,  3.56082163e+00, -4.75726012e-01, -5.37972402e-01,
       -5.41449917e-01, -6.99309933e-01, -4.81226408e-01, -6.33036703e-01,
        3.42105544e+00, -6.34249386e-01, -5.86069701e-01, -5.68038889e-01,
       -5.18434354e-01, -3.79832244e-01, -6.84846441e-01, -7.08704653e-01,
        3.62908161e+00, -4.99547993e-01, -7.86377745e-01, -5.69772534e-01,
       -4.69126744e-01, -5.55282405e-01,  3.45907812e+00, -6.85279093e-01,
       -6.39437478e-01, -4.86423811e-01, -6.23528587e-01, -5.89191277e-01,
       -4.75726012e-01, -6.34249386e-01,  3.37900822e+00, -6.32676318e-01,
       -5.25472142e-01, -5.21693087e-01, -7.37933623e-01, -5.37972402e-01,
        3.54662982e+00, -5.12518864e-01, -5.65467067e-01, -5.74016959e-01,
       -6.18720900e-01, -5.53660369e-01, -5.86069701e-01,  3.42250911e+00,
       -5.60759229e-01, -5.04128340e-01, -6.06329570e-01, -6.11561898e-01,
       -6.10913899e-01, -4.99547993e-01, -5.60759229e-01,  3.45394718e+00,
       -5.20905582e-01, -5.47747894e-01, -7.14072579e-01, -4.30392029e-01,
       -7.86377745e-01,  3.59931325e+00, -5.90761172e-01, -5.61684554e-01,
       -4.51968851e-01, -7.78128898e-01, -5.27094442e-01, -6.85279093e-01,
       -5.12518864e-01,  3.46284123e+00, -5.57989051e-01, -6.23888112e-01,
       -5.56071669e-01, -5.23162696e-01, -6.39437478e-01, -5.90761172e-01,
        3.46990541e+00, -4.61003402e-01, -5.22457066e-01, -7.33083593e-01,
       -5.41449917e-01, -6.32676318e-01,  3.35855934e+00, -5.09933697e-01,
       -5.79578615e-01, -5.27590217e-01, -5.67330573e-01, -6.99309933e-01,
       -5.65467067e-01, -5.09933697e-01,  3.48788469e+00, -5.38733325e-01,
       -6.01797237e-01, -5.72643432e-01, -5.68038889e-01, -5.04128340e-01,
        3.46481918e+00, -6.90011276e-01, -6.42130326e-01, -6.05371965e-01,
       -4.55138380e-01, -5.18434354e-01, -5.25472142e-01, -6.90011276e-01,
        3.46566423e+00, -6.95526551e-01, -5.56382455e-01, -4.79837456e-01,
       -5.69772534e-01, -5.20905582e-01, -5.61684554e-01,  3.57823521e+00,
       -4.13309818e-01, -8.25592824e-01, -6.86969898e-01, -4.86423811e-01,
       -5.57989051e-01,  3.47297310e+00, -6.58752018e-01, -6.54912410e-01,
       -6.46002013e-01, -4.68893801e-01, -6.23528587e-01, -4.61003402e-01,
       -6.58752018e-01,  3.50446307e+00, -5.36902263e-01, -3.30344635e-01,
       -8.93932162e-01, -5.21693087e-01, -5.79578615e-01, -6.95526551e-01,
        3.43700186e+00, -5.93956222e-01, -5.37741009e-01, -5.08506371e-01,
       -5.74016959e-01, -6.23888112e-01,  3.47572195e+00, -5.57608790e-01,
       -5.30397422e-01, -5.88947378e-01, -6.00863286e-01, -6.18720900e-01,
       -5.38733325e-01, -5.57608790e-01,  3.49104204e+00, -6.32574909e-01,
       -5.95907459e-01, -5.47496656e-01, -6.06329570e-01, -6.42130326e-01,
        3.43550261e+00, -4.83154618e-01, -4.54494834e-01, -6.24586581e-01,
       -6.24806683e-01, -6.11561898e-01, -5.47747894e-01, -4.83154618e-01,
        3.40831227e+00, -5.33924877e-01, -6.24643796e-01, -6.07279188e-01,
       -7.14072579e-01, -4.13309818e-01, -5.33924877e-01,  3.45645360e+00,
       -5.55526793e-01, -5.13410081e-01, -7.26209449e-01, -4.51968851e-01,
       -8.25592824e-01,  3.53282197e+00, -4.76451508e-01, -4.74592539e-01,
       -5.51292591e-01, -7.52923659e-01, -7.78128898e-01, -5.22457066e-01,
       -4.76451508e-01,  3.50883615e+00, -4.50890387e-01, -5.39334299e-01,
       -7.41573990e-01, -5.56071669e-01, -6.54912410e-01, -5.30397422e-01,
        3.46067191e+00, -5.44676438e-01, -6.29822518e-01, -5.44791458e-01,
       -7.33083593e-01, -5.36902263e-01, -4.50890387e-01,  3.50173760e+00,
       -4.45953408e-01, -5.45183155e-01, -7.89724789e-01, -5.27590217e-01,
       -5.93956222e-01,  3.44830253e+00, -6.90982582e-01, -5.89223711e-01,
       -4.12536721e-01, -6.34013074e-01, -5.67330573e-01, -6.01797237e-01,
       -6.90982582e-01,  3.52325092e+00, -5.98875881e-01, -6.47276820e-01,
       -4.16987832e-01, -5.72643432e-01, -6.32574909e-01, -5.98875881e-01,
        3.49747561e+00, -5.63886211e-01, -6.81711050e-01, -4.47784129e-01,
       -6.05371965e-01, -4.54494834e-01,  3.48774280e+00, -7.16435664e-01,
       -6.97156182e-01, -4.41033207e-01, -5.73250954e-01, -4.55138380e-01,
       -5.56382455e-01, -7.16435664e-01,  3.50106949e+00, -7.78472987e-01,
       -5.73853249e-01, -4.20786755e-01, -4.79837456e-01, -5.37741009e-01,
       -7.78472987e-01,  3.48626383e+00, -7.04806496e-01, -5.17206031e-01,
       -4.68199853e-01, -6.86969898e-01, -5.55526793e-01, -4.74592539e-01,
        3.63417071e+00, -3.66721269e-01, -7.47143646e-01, -8.03216568e-01,
       -6.46002013e-01, -3.30344635e-01,  3.77300213e+00, -7.59903482e-01,
       -6.30171407e-01, -1.64641231e-01, -1.24193937e+00, -4.68893801e-01,
       -5.44676438e-01, -7.59903482e-01,  3.48263391e+00, -6.40575719e-01,
       -5.31635813e-01, -5.36948660e-01, -8.93932162e-01, -4.45953408e-01,
       -6.30171407e-01,  3.57654606e+00, -4.90823865e-01, -2.69648018e-01,
       -8.46017197e-01, -5.08506371e-01, -5.89223711e-01, -7.04806496e-01,
        3.47034535e+00, -5.79897016e-01, -5.20477256e-01, -5.67434502e-01,
       -5.88947378e-01, -6.29822518e-01,  3.47366586e+00, -5.26420945e-01,
       -5.43474941e-01, -6.14213304e-01, -5.70786777e-01, -6.00863286e-01,
       -5.95907459e-01, -5.26420945e-01,  3.48808880e+00, -5.88090152e-01,
       -6.00191923e-01, -5.76615036e-01, -5.47496656e-01, -5.63886211e-01,
       -5.88090152e-01,  3.47412375e+00, -6.65832603e-01, -6.19896898e-01,
       -4.88921225e-01, -6.24586581e-01, -6.97156182e-01,  3.46485867e+00,
       -4.41574515e-01, -4.93409661e-01, -6.99576572e-01, -5.08555158e-01,
       -6.24806683e-01, -6.24643796e-01, -4.41574515e-01,  3.42517389e+00,
       -5.20330573e-01, -6.50270200e-01, -5.63548124e-01, -6.07279188e-01,
       -5.13410081e-01, -5.20330573e-01,  3.39652434e+00, -5.54056564e-01,
       -6.39955619e-01, -5.61492319e-01, -7.26209449e-01, -3.66721269e-01,
       -5.54056564e-01,  3.44231266e+00, -5.58896247e-01, -5.05212798e-01,
       -7.31216331e-01, -5.51292591e-01, -7.47143646e-01,  3.52465443e+00,
       -4.66170049e-01, -4.29271694e-01, -7.00643026e-01, -6.30133422e-01,
       -7.52923659e-01, -5.39334299e-01, -4.66170049e-01,  3.49430308e+00,
       -4.78989890e-01, -5.83157666e-01, -6.73727520e-01, -7.41573990e-01,
       -5.45183155e-01, -4.78989890e-01,  3.49593378e+00, -4.70211850e-01,
       -5.58298406e-01, -7.01676489e-01, -5.44791458e-01, -6.40575719e-01,
       -5.43474941e-01,  3.48468576e+00, -5.87652135e-01, -6.44917268e-01,
       -5.23274242e-01, -7.89724789e-01, -4.90823865e-01, -4.70211850e-01,
        3.52175845e+00, -5.16562205e-01, -5.39748550e-01, -7.14687188e-01,
       -4.12536721e-01, -6.47276820e-01,  3.68179703e+00, -4.79689860e-01,
       -8.41665945e-01, -1.01960105e+00, -2.52720856e-01, -2.83057814e-02,
       -6.34013074e-01, -5.79897016e-01, -4.79689860e-01,  3.48592570e+00,
       -5.81062002e-01, -6.94946817e-01, -5.16316930e-01, -4.16987832e-01,
       -6.81711050e-01, -8.41665945e-01,  3.63601817e+00, -6.86498908e-01,
       -6.39793221e-01, -3.69361218e-01, -4.47784129e-01, -6.65832603e-01,
       -6.86498908e-01,  3.52392987e+00, -6.36292218e-01, -6.23681990e-01,
       -4.63840022e-01, -4.41033207e-01, -5.73853249e-01,  3.58339042e+00,
       -6.43286514e-01, -9.26440155e-01, -2.67965177e-01, -7.30812118e-01,
       -5.73250954e-01, -4.93409661e-01, -6.43286514e-01,  3.42542889e+00,
       -6.30968125e-01, -5.41320519e-01, -5.43193120e-01, -4.20786755e-01,
       -5.17206031e-01, -9.26440155e-01,  3.76088945e+00, -1.04101570e+00,
       -8.55440805e-01, -4.68199853e-01, -5.20477256e-01, -1.04101570e+00,
        3.62446447e+00, -6.04846496e-01, -2.52948475e-01, -7.36976683e-01,
       -8.03216568e-01, -5.58896247e-01, -4.29271694e-01,  3.70785766e+00,
       -3.19977266e-01, -4.79556204e-01, -9.06161100e-01, -2.10778580e-01,
       -1.64641231e-01, -5.31635813e-01,  3.49307364e+00, -7.85208647e-01,
       -6.06714147e-01, -2.90887881e-01, -5.96320579e-01, -5.17665341e-01,
       -1.24193937e+00, -2.69648018e-01, -7.85208647e-01,  3.95623596e+00,
       -5.11834193e-01, -1.14760574e+00, -5.36948660e-01, -5.87652135e-01,
       -6.06714147e-01,  3.48828381e+00, -6.22266564e-01, -5.91221873e-01,
       -5.43480435e-01, -8.46017197e-01, -5.16562205e-01, -5.11834193e-01,
        3.53282142e+00, -5.06038519e-01, -4.88615200e-01, -6.63754104e-01,
       -5.67434502e-01, -5.81062002e-01, -6.04846496e-01,  3.45965397e+00,
       -5.73839596e-01, -5.64755022e-01, -5.67716347e-01, -6.14213304e-01,
       -6.00191923e-01,  3.48060905e+00, -5.19044089e-01, -5.26840508e-01,
       -6.44278149e-01, -5.76041075e-01, -5.70786777e-01, -6.44917268e-01,
       -5.19044089e-01,  3.49430620e+00, -5.79139690e-01, -6.53604379e-01,
       -5.26813995e-01, -5.76615036e-01, -6.19896898e-01, -5.26840508e-01,
        3.50915430e+00, -6.38634626e-01, -5.69270445e-01, -5.77896792e-01,
       -4.88921225e-01, -6.36292218e-01, -6.38634626e-01,  3.49021385e+00,
       -5.80231725e-01, -6.15055639e-01, -5.31078415e-01, -6.99576572e-01,
       -6.50270200e-01,  3.51468426e+00, -6.30299227e-01, -4.41227564e-01,
       -4.91621416e-01, -6.01689283e-01, -5.08555158e-01, -6.30968125e-01,
       -6.30299227e-01,  3.45089503e+00, -6.26510840e-01, -3.62441927e-01,
       -6.92119750e-01, -5.63548124e-01, -6.39955619e-01, -4.41227564e-01,
        3.41296088e+00, -5.31907775e-01, -6.76560680e-01, -5.59761122e-01,
       -5.61492319e-01, -5.05212798e-01, -5.31907775e-01,  3.37695860e+00,
       -5.77880254e-01, -6.55822847e-01, -5.44642606e-01, -7.31216331e-01,
       -3.19977266e-01, -5.77880254e-01,  3.48498738e+00, -6.52993721e-01,
       -5.71725364e-01, -6.31194448e-01, -7.00643026e-01, -4.79556204e-01,
        3.48602121e+00, -5.05453188e-01, -5.72614219e-01, -5.20303698e-01,
       -7.07450870e-01, -6.30133422e-01, -5.83157666e-01, -5.05453188e-01,
        3.47566094e+00, -5.09975798e-01, -6.40003046e-01, -6.06937820e-01,
       -6.73727520e-01, -5.58298406e-01, -5.09975798e-01,  3.47268953e+00,
       -5.05106244e-01, -5.85289591e-01, -6.40291971e-01, -7.01676489e-01,
       -5.39748550e-01, -5.05106244e-01,  3.47656057e+00, -5.02392440e-01,
       -5.69833251e-01, -6.57803599e-01, -5.23274242e-01, -6.22266564e-01,
       -5.79139690e-01,  3.46746246e+00, -5.90349483e-01, -6.11763921e-01,
       -5.40668562e-01, -7.14687188e-01, -5.06038519e-01, -5.02392440e-01,
        3.48949301e+00, -5.45747305e-01, -5.51688383e-01, -6.68939173e-01,
       -1.01960105e+00, -6.94946817e-01,  3.78118665e+00, -6.28815030e-01,
       -9.94611990e-01, -4.43211766e-01, -2.52720856e-01, -6.39793221e-01,
        3.71683105e+00, -8.25366231e-01, -1.08980060e+00, -4.27254270e-01,
       -4.81895874e-01, -5.16316930e-01, -5.73839596e-01, -6.28815030e-01,
        3.47016808e+00, -6.17500276e-01, -6.10200169e-01, -5.23496080e-01,
       -3.69361218e-01, -6.23681990e-01, -8.25366231e-01,  3.57173856e+00,
       -7.34031402e-01, -5.60672484e-01, -4.58625236e-01, -4.63840022e-01,
       -5.80231725e-01, -7.34031402e-01,  3.51790766e+00, -6.95365505e-01,
       -5.30098626e-01, -5.14340376e-01, -2.67965177e-01, -8.55440805e-01,
       -2.52948475e-01,  3.50572164e+00, -4.25982053e-01, -3.77351241e-01,
       -7.00292985e-01, -6.25740901e-01, -7.30812118e-01, -5.41320519e-01,
       -4.25982053e-01,  3.46571444e+00, -5.51385244e-01, -6.13922672e-01,
       -6.02291836e-01, -5.43193120e-01, -6.26510840e-01, -5.51385244e-01,
        3.44333663e+00, -4.96974403e-01, -6.15849361e-01, -6.09423663e-01,
       -7.36976683e-01, -5.64755022e-01, -3.77351241e-01,  3.50416963e+00,
       -5.41392084e-01, -6.88340702e-01, -5.95353896e-01, -9.06161100e-01,
       -6.52993721e-01,  4.63529745e+00, -1.99356803e+00, -6.04430978e-01,
       -4.78143619e-01, -2.10778580e-01, -5.72614219e-01, -1.99356803e+00,
        4.24514807e+00, -4.10445789e-01, -5.06940401e-01, -5.50801044e-01,
       -2.90887881e-01, -1.14760574e+00, -4.88615200e-01,  3.65612120e+00,
       -4.87712092e-01, -6.64757717e-01, -5.76542574e-01, -5.96320579e-01,
       -5.91221873e-01,  3.47597344e+00, -5.65942303e-01, -5.69496088e-01,
       -5.79086590e-01, -5.73906006e-01, -5.43480435e-01, -5.90349483e-01,
       -5.65942303e-01,  3.41405270e+00, -5.92882806e-01, -5.93229352e-01,
       -5.28168321e-01, -6.63754104e-01, -5.45747305e-01, -4.87712092e-01,
        3.46751540e+00, -5.47663040e-01, -5.96487092e-01, -6.26151767e-01,
       -2.83057814e-02, -9.94611990e-01, -1.08980060e+00,  3.81878863e+00,
       -6.91488804e-01, -5.30888958e-01, -4.83692493e-01, -5.67716347e-01,
       -6.17500276e-01, -5.41392084e-01,  3.46555933e+00, -5.63525558e-01,
       -6.26172050e-01, -5.49253015e-01, -6.44278149e-01, -5.69270445e-01,
        3.46665190e+00, -5.16590392e-01, -5.15455910e-01, -5.71950177e-01,
       -6.49106822e-01, -5.76041075e-01, -6.53604379e-01, -5.16590392e-01,
        3.45934042e+00, -5.25965522e-01, -6.33162431e-01, -5.53976620e-01,
       -5.26813995e-01, -6.11763921e-01, -5.25965522e-01,  3.37689952e+00,
       -6.03184068e-01, -6.11661033e-01, -4.97510978e-01, -5.77896792e-01,
       -6.15055639e-01, -5.15455910e-01,  3.44303848e+00, -5.48492215e-01,
       -5.77808960e-01, -6.08328959e-01, -5.31078415e-01, -6.95365505e-01,
       -5.48492215e-01,  3.48323832e+00, -4.68674492e-01, -6.59023933e-01,
       -5.80603760e-01, -4.91621416e-01, -3.62441927e-01,  4.40026740e+00,
       -1.78442196e+00, -5.59325285e-01, -1.31587587e+00,  1.13419060e-01,
       -6.01689283e-01, -6.76560680e-01, -1.78442196e+00,  4.10011173e+00,
       -4.25971110e-01, -6.11468688e-01, -6.92119750e-01, -4.96974403e-01,
       -5.59325285e-01,  3.47884456e+00, -5.71289132e-01, -4.54952890e-01,
       -7.04183104e-01, -5.59761122e-01, -6.55822847e-01, -4.25971110e-01,
        3.39110591e+00, -5.36035402e-01, -6.70547233e-01, -5.42968196e-01,
       -5.44642606e-01, -5.71725364e-01, -5.36035402e-01,  3.29524553e+00,
       -5.13793023e-01, -5.52977851e-01, -5.76071289e-01, -6.31194448e-01,
       -6.04430978e-01, -5.13793023e-01,  3.48138890e+00, -4.25672173e-01,
       -5.77545839e-01, -7.28752442e-01, -5.20303698e-01, -6.40003046e-01,
        3.47100888e+00, -5.68653313e-01, -5.25192520e-01, -6.51809929e-01,
       -5.65046378e-01, -7.07450870e-01, -4.10445789e-01, -5.68653313e-01,
        3.56086264e+00, -7.71854317e-01, -4.72723721e-01, -6.29734630e-01,
       -6.06937820e-01, -5.85289591e-01, -5.25192520e-01,  3.46952415e+00,
       -5.26643990e-01, -6.12873072e-01, -6.12587160e-01, -6.40291971e-01,
       -5.69833251e-01, -5.26643990e-01,  3.47138373e+00, -5.22429667e-01,
       -5.86447132e-01, -6.25737720e-01, -6.57803599e-01, -5.51688383e-01,
       -5.22429667e-01,  3.47428422e+00, -5.27213805e-01, -5.81781927e-01,
       -6.33366840e-01, -5.17665341e-01, -6.64757717e-01, -5.69496088e-01,
        3.47594204e+00, -5.36582085e-01, -6.05284149e-01, -5.82156654e-01,
       -5.40668562e-01, -5.92882806e-01, -6.03184068e-01,  3.43938814e+00,
       -5.99665898e-01, -5.90139657e-01, -5.12847149e-01, -6.68939173e-01,
       -5.47663040e-01, -5.27213805e-01,  3.47965770e+00, -5.43076764e-01,
       -5.68244634e-01, -6.24520289e-01, -4.43211766e-01, -6.10200169e-01,
       -6.91488804e-01,  3.47842924e+00, -6.42413884e-01, -5.64288811e-01,
       -5.26825806e-01, -4.27254270e-01, -5.60672484e-01,  3.52065547e+00,
       -7.34855525e-01, -7.33636697e-01, -6.13645228e-01, -4.50591269e-01,
       -4.81895874e-01, -5.30888958e-01, -7.34855525e-01,  3.49390339e+00,
       -6.97156695e-01, -4.77338899e-01, -5.71767437e-01, -5.23496080e-01,
       -5.63525558e-01, -6.42413884e-01,  3.46551760e+00, -6.32231919e-01,
       -5.56869446e-01, -5.46980711e-01, -4.58625236e-01, -5.30098626e-01,
       -7.33636697e-01,  3.51557810e+00, -7.32543230e-01, -5.62867732e-01,
       -4.97806580e-01, -5.14340376e-01, -4.68674492e-01, -7.32543230e-01,
        3.50188836e+00, -7.32466170e-01, -5.12271390e-01, -5.41592700e-01,
       -7.00292985e-01, -6.13922672e-01,  3.47249066e+00, -5.10008097e-01,
       -4.25818068e-01, -5.99235030e-01, -6.23213804e-01, -6.02291836e-01,
       -6.15849361e-01, -5.10008097e-01,  3.44778290e+00, -5.36002288e-01,
       -6.25472347e-01, -5.58158969e-01, -6.09423663e-01, -5.71289132e-01,
       -5.36002288e-01,  3.45408135e+00, -5.43537636e-01, -6.05865671e-01,
       -5.87962960e-01, -6.25740901e-01, -6.88340702e-01, -4.25818068e-01,
        3.48185841e+00, -4.89970508e-01, -6.84087331e-01, -5.67900900e-01,
       -5.95353896e-01, -6.26172050e-01, -4.89970508e-01,  3.47013373e+00,
       -5.50093629e-01, -6.65815847e-01, -5.42727800e-01, -4.78143619e-01,
       -5.06940401e-01, -4.25672173e-01,  3.55006104e+00, -4.71685585e-01,
       -5.42554558e-01, -7.78499528e-01, -3.46565175e-01, -5.50801044e-01,
       -7.71854317e-01, -4.71685585e-01,  3.55599882e+00, -4.43875142e-01,
       -8.51746325e-01, -4.66036404e-01, -5.76542574e-01, -5.96487092e-01,
       -5.36582085e-01,  3.44740556e+00, -5.41341882e-01, -5.91306858e-01,
       -6.05145068e-01, -5.79086590e-01, -6.05284149e-01,  3.47097310e+00,
       -5.76054385e-01, -5.60991763e-01, -5.53231640e-01, -5.96324576e-01,
       -5.73906006e-01, -5.93229352e-01, -5.76054385e-01,  3.48131978e+00,
       -5.94747134e-01, -6.11340315e-01, -5.32042585e-01, -5.28168321e-01,
       -5.99665898e-01, -5.94747134e-01,  3.48371522e+00, -6.06968180e-01,
       -6.29229138e-01, -5.24936549e-01, -6.26151767e-01, -5.43076764e-01,
       -5.41341882e-01,  3.44981339e+00, -5.74376556e-01, -5.52069076e-01,
       -6.12797342e-01, -4.83692493e-01, -5.64288811e-01, -6.97156695e-01,
        3.47276688e+00, -6.45633598e-01, -5.27694239e-01, -5.54301041e-01,
       -5.49253015e-01, -6.32231919e-01, -5.50093629e-01,  3.46815999e+00,
       -5.56446705e-01, -6.36816800e-01, -5.43317920e-01, -5.71950177e-01,
       -6.33162431e-01,  3.49952383e+00, -5.14055868e-01, -5.65859694e-01,
       -5.70638098e-01, -6.43857562e-01, -6.49106822e-01, -5.77808960e-01,
       -5.14055868e-01,  3.42952287e+00, -5.01340749e-01, -5.48974669e-01,
       -6.38235804e-01, -5.53976620e-01, -6.11661033e-01, -5.65859694e-01,
        3.52973991e+00, -6.49673053e-01, -6.07981621e-01, -5.40587885e-01,
       -4.97510978e-01, -5.90139657e-01, -6.49673053e-01,  3.49894205e+00,
       -6.74564438e-01, -6.03075609e-01, -4.83978313e-01, -6.08328959e-01,
       -6.59023933e-01, -5.01340749e-01,  3.42065881e+00, -4.42381096e-01,
       -5.87285619e-01, -6.22298459e-01, -5.80603760e-01, -7.32466170e-01,
       -4.42381096e-01,  3.49283461e+00, -4.60510048e-01, -7.35692722e-01,
       -5.41180815e-01, -1.31587587e+00, -4.54952890e-01,  3.86606370e+00,
       -8.24856926e-01, -4.48549150e-01, -8.21828867e-01,  1.13419060e-01,
       -6.11468688e-01, -6.70547233e-01, -8.24856926e-01,  3.51935444e+00,
       -6.18535448e-01, -3.20531621e-01, -5.86833581e-01, -7.04183104e-01,
       -5.43537636e-01, -4.48549150e-01,  3.46800927e+00, -5.17120640e-01,
       -6.04431914e-01, -6.50186824e-01, -5.42968196e-01, -5.52977851e-01,
       -6.18535448e-01,  3.36734289e+00, -5.95519071e-01, -4.84101380e-01,
       -5.73240948e-01, -5.76071289e-01, -5.77545839e-01, -5.95519071e-01,
        3.43580383e+00, -5.38693331e-01, -5.61574579e-01, -5.86399716e-01,
       -7.28752442e-01, -5.42554558e-01, -5.38693331e-01,  3.49786193e+00,
       -4.34961643e-01, -5.07688845e-01, -7.45211107e-01, -6.51809929e-01,
       -4.72723721e-01,  3.46614692e+00, -5.51295078e-01, -6.07624891e-01,
       -6.28198458e-01, -5.54494844e-01, -5.65046378e-01, -6.12873072e-01,
       -5.51295078e-01,  3.46514023e+00, -5.19655703e-01, -6.19242785e-01,
       -5.97027214e-01, -6.29734630e-01, -4.43875142e-01, -6.07624891e-01,
        3.47778489e+00, -7.25761674e-01, -4.96750108e-01, -5.74038440e-01,
       -6.12587160e-01, -5.86447132e-01, -5.19655703e-01,  3.46186907e+00,
       -5.27343862e-01, -6.04875284e-01, -6.10959933e-01, -6.25737720e-01,
       -5.81781927e-01, -5.27343862e-01,  3.46848907e+00, -5.16420897e-01,
       -5.97947033e-01, -6.19257628e-01, -6.33366840e-01, -5.68244634e-01,
       -5.16420897e-01,  3.45710922e+00, -5.40060223e-01, -5.90114018e-01,
       -6.08902604e-01, -5.82156654e-01, -5.91306858e-01, -5.60991763e-01,
        3.46937146e+00, -5.59111178e-01, -5.67206075e-01, -6.08598933e-01,
       -5.12847149e-01, -6.06968180e-01, -6.74564438e-01,  3.52207563e+00,
       -6.37883069e-01, -5.83341952e-01, -5.06470843e-01, -6.24520289e-01,
       -5.74376556e-01, -5.40060223e-01,  3.46825187e+00, -5.34507173e-01,
       -5.95132204e-01, -5.99655426e-01, -5.26825806e-01, -5.56869446e-01,
       -6.45633598e-01,  3.46538244e+00, -6.30723144e-01, -5.42588191e-01,
       -5.62742252e-01, -6.13645228e-01, -4.77338899e-01,  3.49662536e+00,
       -6.54662647e-01, -6.16216219e-01, -4.75266341e-01, -6.59496021e-01,
       -4.50591269e-01, -5.62867732e-01, -6.54662647e-01,  3.48751616e+00,
       -6.76982180e-01, -6.43241108e-01, -4.99171224e-01, -5.71767437e-01,
       -5.27694239e-01, -6.16216219e-01,  3.46592760e+00, -6.38601932e-01,
       -5.30175113e-01, -5.81472658e-01, -5.46980711e-01, -5.56446705e-01,
       -6.30723144e-01,  3.45786626e+00, -6.23571319e-01, -5.45528249e-01,
       -5.54616135e-01, -4.97806580e-01, -5.12271390e-01, -6.76982180e-01,
        3.49066240e+00, -7.00738174e-01, -5.81351609e-01, -5.21512462e-01,
       -5.41592700e-01, -4.60510048e-01, -7.00738174e-01,  3.50316201e+00,
       -7.88094558e-01, -5.28902456e-01, -4.83324071e-01, -5.99235030e-01,
       -6.84087331e-01,  3.46894445e+00, -4.88236974e-01, -5.02762878e-01,
       -5.51709646e-01, -6.42912593e-01, -6.23213804e-01, -6.25472347e-01,
       -4.88236974e-01,  3.45509765e+00, -5.62782999e-01, -6.16567265e-01,
       -5.38824264e-01, -5.58158969e-01, -6.05865671e-01, -5.62782999e-01,
        3.44830816e+00, -5.69047863e-01, -6.36624121e-01, -5.15828532e-01,
       -5.87962960e-01, -5.17120640e-01, -5.69047863e-01,  3.46148900e+00,
       -6.54867484e-01, -6.29927526e-01, -5.02562528e-01, -5.67900900e-01,
       -6.65815847e-01, -5.02762878e-01,  3.46780530e+00, -5.46969660e-01,
       -6.52931640e-01, -5.31424377e-01, -5.42727800e-01, -6.36816800e-01,
       -5.46969660e-01,  3.46879450e+00, -5.68579396e-01, -6.48088921e-01,
       -5.25611924e-01, -7.78499528e-01, -4.34961643e-01,  3.65148185e+00,
       -7.09837031e-01, -5.93710605e-01, -4.01112068e-01, -7.33360975e-01,
       -3.46565175e-01, -8.51746325e-01, -7.09837031e-01,  3.55019817e+00,
       -5.85358819e-01, -6.44387686e-01, -4.12303136e-01, -4.66036404e-01,
       -7.25761674e-01, -5.85358819e-01,  3.49317087e+00, -5.19091590e-01,
       -7.03701770e-01, -4.93220617e-01, -6.05145068e-01, -5.52069076e-01,
       -5.59111178e-01,  3.48421485e+00, -5.97727167e-01, -5.54939585e-01,
       -6.15222775e-01, -5.53231640e-01, -6.11340315e-01,  3.47163524e+00,
       -5.81939068e-01, -5.90463630e-01, -6.24597226e-01, -5.10063359e-01,
       -5.96324576e-01, -5.67206075e-01, -5.81939068e-01,  3.46798410e+00,
       -5.71577612e-01, -5.19668027e-01, -6.31268739e-01, -5.32042585e-01,
       -6.29229138e-01, -5.90463630e-01,  3.46895672e+00, -5.80392962e-01,
       -6.17592301e-01, -5.19236101e-01, -5.24936549e-01, -6.37883069e-01,
       -5.80392962e-01,  3.46905121e+00, -5.64247686e-01, -6.20258678e-01,
       -5.41332270e-01, -6.12797342e-01, -5.34507173e-01, -5.97727167e-01,
        3.47575826e+00, -6.07965575e-01, -5.29604506e-01, -5.93156500e-01,
       -5.54301041e-01, -5.42588191e-01, -6.38601932e-01,  3.45619350e+00,
       -6.16782520e-01, -5.21298662e-01, -5.82621155e-01, -5.43317920e-01,
       -6.23571319e-01, -5.68579396e-01,  3.45882978e+00, -5.64271299e-01,
       -6.18640308e-01, -5.40449538e-01, -5.70638098e-01, -6.07981621e-01,
        3.49318608e+00, -5.64062925e-01, -5.69785500e-01, -5.58966861e-01,
       -6.21751076e-01, -6.43857562e-01, -5.48974669e-01, -5.64062925e-01,
        3.46614235e+00, -5.88863230e-01, -5.06694251e-01, -6.13689716e-01,
       -6.38235804e-01, -5.87285619e-01, -5.88863230e-01,  3.38630592e+00,
       -4.59877146e-01, -4.83495644e-01, -6.28548476e-01, -5.40587885e-01,
       -6.03075609e-01, -5.69785500e-01,  3.48854384e+00, -6.22293628e-01,
       -6.23314110e-01, -5.29487106e-01, -4.83978313e-01, -5.83341952e-01,
       -6.22293628e-01,  3.45316962e+00, -6.65418213e-01, -5.88988803e-01,
       -5.09148712e-01, -6.22298459e-01, -7.35692722e-01, -4.59877146e-01,
        3.43023307e+00, -4.13984036e-01, -7.48364462e-01, -4.50016250e-01,
       -5.41180815e-01, -7.88094558e-01, -4.13984036e-01,  3.52719052e+00,
       -4.95089097e-01, -8.24072861e-01, -4.64769153e-01, -8.21828867e-01,
       -3.20531621e-01, -6.04431914e-01,  3.53378289e+00, -7.59378997e-01,
       -3.53492555e-01, -6.74118935e-01, -5.86833581e-01, -4.84101380e-01,
       -7.59378997e-01,  3.49050562e+00, -6.32765713e-01, -3.02914146e-01,
       -7.24511801e-01, -6.50186824e-01, -6.54867484e-01, -3.53492555e-01,
        3.48812377e+00, -5.37900462e-01, -8.61774369e-01, -4.29902077e-01,
       -5.73240948e-01, -5.61574579e-01, -6.32765713e-01,  3.44768886e+00,
       -6.57031903e-01, -5.43490638e-01, -4.79585081e-01, -5.86399716e-01,
       -5.07688845e-01, -6.57031903e-01,  3.36163188e+00, -5.13199969e-01,
       -5.20718265e-01, -5.76593179e-01, -7.45211107e-01, -5.93710605e-01,
       -5.13199969e-01,  3.58738116e+00, -4.30717066e-01, -6.06429281e-01,
       -6.98113136e-01, -6.28198458e-01, -4.96750108e-01,  3.46593648e+00,
       -5.65065972e-01, -6.26724432e-01, -5.45671484e-01, -6.03526023e-01,
       -5.54494844e-01, -6.19242785e-01, -5.65065972e-01,  3.45534291e+00,
       -5.25511252e-01, -6.16965253e-01, -5.74062807e-01, -5.97027214e-01,
       -6.04875284e-01, -5.25511252e-01,  3.46302601e+00, -5.18304540e-01,
       -6.22215651e-01, -5.95092071e-01, -5.74038440e-01, -5.19091590e-01,
       -6.26724432e-01,  3.47334927e+00, -6.66677055e-01, -5.34322869e-01,
       -5.52494882e-01, -6.10959933e-01, -5.97947033e-01, -5.18304540e-01,
        3.46734032e+00, -5.19047694e-01, -6.20879552e-01, -6.00201566e-01,
       -6.19257628e-01, -5.90114018e-01, -5.19047694e-01,  3.46862652e+00,
       -5.24122643e-01, -6.11627775e-01, -6.04456761e-01, -6.08902604e-01,
       -5.95132204e-01, -5.24122643e-01,  3.46484795e+00, -5.25606426e-01,
       -6.15393931e-01, -5.95690146e-01, -6.08598933e-01, -5.54939585e-01,
       -5.71577612e-01,  3.44971750e+00, -5.80985679e-01, -5.32218952e-01,
       -6.01396740e-01, -5.06470843e-01, -5.64247686e-01, -6.65418213e-01,
        3.40047365e+00, -6.20141099e-01, -5.46155329e-01, -4.98040483e-01,
       -5.99655426e-01, -6.07965575e-01, -5.25606426e-01,  3.46440281e+00,
       -5.27230334e-01, -6.30571939e-01, -5.73373108e-01, -5.62742252e-01,
       -5.45528249e-01, -6.16782520e-01,  3.45296596e+00, -6.19517519e-01,
       -5.39400489e-01, -5.68994934e-01, -4.75266341e-01, -6.43241108e-01,
        3.49125212e+00, -6.08664947e-01, -6.16882622e-01, -5.06339452e-01,
       -6.40857646e-01, -6.59496021e-01, -5.30175113e-01, -6.08664947e-01,
        3.49244984e+00, -6.27019175e-01, -4.87697513e-01, -5.79397069e-01,
       -4.99171224e-01, -5.81351609e-01, -6.16882622e-01,  3.47486619e+00,
       -6.42642787e-01, -6.10617325e-01, -5.24200627e-01, -5.81472658e-01,
       -5.21298662e-01, -6.27019175e-01,  3.45148988e+00, -6.07946349e-01,
       -5.07170482e-01, -6.06582557e-01, -5.54616135e-01, -5.64271299e-01,
       -6.19517519e-01,  3.45503978e+00, -6.14795378e-01, -5.54981825e-01,
       -5.46857623e-01, -5.21512462e-01, -5.28902456e-01, -6.42642787e-01,
        3.48543374e+00, -7.01247537e-01, -5.98605794e-01, -4.92522702e-01,
       -4.83324071e-01, -4.95089097e-01, -7.01247537e-01,  3.68563353e+00,
       -8.16738942e-01, -7.03655394e-01, -4.85578491e-01, -5.51709646e-01,
       -6.52931640e-01,  3.45804390e+00, -5.53197480e-01, -5.78974217e-01,
       -5.07976329e-01, -6.13254589e-01, -6.42912593e-01, -6.16567265e-01,
       -5.53197480e-01,  3.46597951e+00, -5.60744508e-01, -5.59806236e-01,
       -5.32751426e-01, -5.38824264e-01, -6.36624121e-01, -5.60744508e-01,
        3.47260949e+00, -5.76014269e-01, -6.87333664e-01, -4.73068660e-01,
       -5.15828532e-01, -6.29927526e-01, -5.76014269e-01,  3.44893795e+00,
       -6.18023040e-01, -6.71953016e-01, -4.37191569e-01, -5.02562528e-01,
       -5.37900462e-01, -6.18023040e-01,  3.45359629e+00, -7.26172662e-01,
       -6.65665865e-01, -4.03271729e-01, -5.31424377e-01, -6.48088921e-01,
       -5.78974217e-01,  3.47040047e+00, -5.68936642e-01, -6.11505344e-01,
       -5.31470972e-01, -5.25611924e-01, -6.18640308e-01, -5.68936642e-01,
        3.44985070e+00, -5.79907272e-01, -6.29457272e-01, -5.27297283e-01,
       -4.01112068e-01, -6.44387686e-01,  3.50041283e+00, -6.72789472e-01,
       -6.80525118e-01, -6.24830713e-01, -4.76767773e-01, -7.33360975e-01,
       -4.30717066e-01, -6.72789472e-01,  3.59418574e+00, -6.68886743e-01,
       -4.99815252e-01, -5.88616233e-01, -4.12303136e-01, -7.03701770e-01,
       -6.80525118e-01,  3.49417811e+00, -5.91876665e-01, -6.28300225e-01,
       -4.77471196e-01, -4.93220617e-01, -6.66677055e-01, -5.91876665e-01,
        3.47425915e+00, -5.63881868e-01, -6.36717421e-01, -5.21885527e-01,
       -6.15222775e-01, -5.29604506e-01, -5.80985679e-01,  3.45281799e+00,
       -5.98292124e-01, -5.35406833e-01, -5.93306073e-01, -6.24597226e-01,
       -5.19668027e-01,  3.48294890e+00, -6.23085660e-01, -5.84681414e-01,
       -6.71606638e-01, -4.59309931e-01, -5.10063359e-01, -6.17592301e-01,
       -6.23085660e-01,  3.46243641e+00, -5.95330840e-01, -6.02074777e-01,
       -5.14289476e-01, -6.31268739e-01, -5.32218952e-01, -5.84681414e-01,
        3.45483209e+00, -5.85327928e-01, -4.87860996e-01, -6.33474060e-01,
       -5.19236101e-01, -6.20258678e-01, -5.95330840e-01,  3.47320870e+00,
       -6.01500140e-01, -6.17529215e-01, -5.19353730e-01, -5.41332270e-01,
       -6.20141099e-01, -6.01500140e-01,  3.50499979e+00, -6.25452955e-01,
       -6.00137493e-01, -5.16435834e-01, -5.93156500e-01, -5.27230334e-01,
       -5.98292124e-01,  3.46217934e+00, -6.31478952e-01, -5.52735246e-01,
       -5.59286180e-01, -5.82621155e-01, -5.39400489e-01, -6.07946349e-01,
        3.46812801e+00, -6.12047029e-01, -5.33742193e-01, -5.92370794e-01,
       -5.40449538e-01, -6.14795378e-01, -5.79907272e-01,  3.46220130e+00,
       -5.92248538e-01, -6.26898386e-01, -5.07902188e-01, -5.58966861e-01,
       -6.23314110e-01,  3.50710593e+00, -6.25171771e-01, -5.62468733e-01,
       -4.98135862e-01, -6.39048592e-01, -6.21751076e-01, -5.06694251e-01,
       -6.25171771e-01,  3.49797013e+00, -6.10379015e-01, -5.06545597e-01,
       -6.27428420e-01, -6.13689716e-01, -4.83495644e-01, -6.10379015e-01,
        3.48790368e+00, -6.43396665e-01, -5.47481129e-01, -5.89461509e-01,
       -6.28548476e-01, -7.48364462e-01, -6.43396665e-01,  3.57358222e+00,
       -5.04886830e-01, -9.17294968e-01, -1.31090816e-01, -5.29487106e-01,
       -5.88988803e-01, -5.62468733e-01,  3.46240547e+00, -6.45022802e-01,
       -6.36710743e-01, -4.99727283e-01, -5.09148712e-01, -5.46155329e-01,
       -6.45022802e-01,  3.47549533e+00, -6.61681971e-01, -6.15482273e-01,
       -4.98004244e-01, -4.50016250e-01, -8.24072861e-01, -5.04886830e-01,
        3.54700952e+00, -3.09068376e-01, -8.14399825e-01, -5.92956785e-01,
       -5.16085968e-02, -4.64769153e-01, -8.16738942e-01, -3.09068376e-01,
        3.99749702e+00, -3.38454623e-01, -1.61388800e+00, -4.54577922e-01,
       -6.74118935e-01, -3.02914146e-01, -8.61774369e-01,  3.58146271e+00,
       -8.61946954e-01, -5.60325107e-01, -3.20383198e-01, -7.24511801e-01,
       -5.43490638e-01, -8.61946954e-01,  3.56665760e+00, -6.71990194e-01,
       -7.64718011e-01, -4.29902077e-01, -7.26172662e-01, -5.60325107e-01,
        3.47778205e+00, -6.68113016e-01, -6.32999318e-01, -4.60269869e-01,
       -4.79585081e-01, -5.20718265e-01, -6.71990194e-01,  3.33361383e+00,
       -6.17012204e-01, -4.74621411e-01, -5.69686677e-01, -5.76593179e-01,
       -6.06429281e-01, -6.17012204e-01,  3.41244842e+00, -4.22661571e-01,
       -5.08411045e-01, -6.81341145e-01, -6.98113136e-01, -6.68886743e-01,
       -4.22661571e-01,  3.77907861e+00, -6.49220370e-01, -5.91819248e-01,
       -7.48377545e-01, -5.45671484e-01, -6.16965253e-01,  3.46789967e+00,
       -5.73995943e-01, -5.47076706e-01, -6.11681986e-01, -5.72508294e-01,
       -6.03526023e-01, -5.34322869e-01, -5.73995943e-01,  3.45599559e+00,
       -6.01146726e-01, -5.52652698e-01, -5.90351331e-01, -5.74062807e-01,
       -6.22215651e-01, -5.47076706e-01,  3.46843859e+00, -5.23768488e-01,
       -6.15243360e-01, -5.86071579e-01, -5.95092071e-01, -6.20879552e-01,
       -5.23768488e-01,  3.47040644e+00, -5.12203074e-01, -6.29599083e-01,
       -5.88864174e-01, -5.52494882e-01, -5.63881868e-01, -6.01146726e-01,
        3.46506160e+00, -6.15067909e-01, -5.73080611e-01, -5.59389606e-01,
       -6.00201566e-01, -6.11627775e-01, -5.12203074e-01,  3.46447595e+00,
       -5.10331013e-01, -6.29526830e-01, -6.00585692e-01, -6.04456761e-01,
       -6.15393931e-01, -5.10331013e-01,  3.45034333e+00, -4.99757307e-01,
       -6.36545752e-01, -5.83858563e-01, -5.95690146e-01, -6.30571939e-01,
       -4.99757307e-01,  3.47362359e+00, -5.20742384e-01, -6.76470677e-01,
       -5.50391139e-01, -6.01396740e-01, -5.35406833e-01, -5.85327928e-01,
        3.44975247e+00, -6.05013303e-01, -5.26174878e-01, -5.96432788e-01,
       -4.98040483e-01, -6.25452955e-01, -6.61681971e-01,  3.49733941e+00,
       -6.02104465e-01, -5.58605532e-01, -5.51453999e-01, -5.73373108e-01,
       -6.31478952e-01, -5.20742384e-01,  3.48131028e+00, -5.60599391e-01,
       -6.89626369e-01, -5.05490072e-01, -5.68994934e-01, -5.54981825e-01,
       -6.12047029e-01,  3.45852792e+00, -6.18167691e-01, -5.37409357e-01,
       -5.66927086e-01, -5.06339452e-01, -6.10617325e-01,  3.47834448e+00,
       -6.14255073e-01, -6.16472500e-01, -6.03762476e-01, -5.26897653e-01,
       -6.40857646e-01, -4.87697513e-01, -6.14255073e-01,  3.44742989e+00,
       -6.21955964e-01, -4.76607839e-01, -6.06055850e-01, -5.79397069e-01,
       -5.07170482e-01, -6.21955964e-01,  3.43126987e+00, -6.18717036e-01,
       -5.09351728e-01, -5.94677589e-01, -5.24200627e-01, -5.98605794e-01,
       -6.16472500e-01,  3.47569542e+00, -5.25732134e-01, -6.48312085e-01,
       -5.62372283e-01, -6.06582557e-01, -5.33742193e-01, -6.18717036e-01,
        3.48231166e+00, -5.96235144e-01, -5.32749432e-01, -5.94285300e-01,
       -5.46857623e-01, -5.92248538e-01, -6.18167691e-01,  3.46121055e+00,
       -5.96306871e-01, -5.71698673e-01, -5.35931154e-01, -4.92522702e-01,
       -7.03655394e-01, -5.25732134e-01,  3.55302578e+00, -6.19788789e-01,
       -7.97227236e-01, -4.14099522e-01, -4.85578491e-01, -3.38454623e-01,
       -6.19788789e-01,  3.38575523e+00, -7.89734528e-01, -7.46173507e-01,
       -4.06025294e-01, -5.07976329e-01, -6.11505344e-01,  3.45506551e+00,
       -6.24119229e-01, -6.04036616e-01, -5.71855369e-01, -5.35572619e-01,
       -6.13254589e-01, -5.59806236e-01, -6.24119229e-01,  3.44849043e+00,
       -5.41942707e-01, -4.96204278e-01, -6.13163392e-01, -5.32751426e-01,
       -6.87333664e-01, -5.41942707e-01,  3.54375025e+00, -6.51568391e-01,
       -8.31762056e-01, -2.98392010e-01, -4.73068660e-01, -6.71953016e-01,
       -6.51568391e-01,  3.46973519e+00, -6.47319497e-01, -6.16124952e-01,
       -4.09700677e-01, -4.37191569e-01, -6.65665865e-01, -6.47319497e-01,
        3.48934044e+00, -6.47460835e-01, -6.72107324e-01, -4.19595349e-01,
       -4.03271729e-01, -6.68113016e-01, -6.47460835e-01,  3.47640016e+00,
       -6.12066604e-01, -6.43273204e-01, -5.02214769e-01, -5.31470972e-01,
       -6.29457272e-01, -6.04036616e-01,  3.47453529e+00, -5.92268150e-01,
       -5.77835787e-01, -5.39466491e-01, -5.27297283e-01, -6.26898386e-01,
       -5.92268150e-01,  3.44968622e+00, -5.92350048e-01, -5.93618232e-01,
       -5.17254116e-01, -6.24830713e-01, -4.99815252e-01,  3.48695106e+00,
       -6.00429313e-01, -5.55324753e-01, -5.32576031e-01, -6.73974999e-01,
       -4.76767773e-01, -6.28300225e-01, -6.00429313e-01,  3.47264481e+00,
       -6.14894419e-01, -6.42352035e-01, -5.09901042e-01, -5.88616233e-01,
       -6.49220370e-01, -5.55324753e-01,  3.62583639e+00, -5.47520015e-01,
       -6.52618571e-01, -6.32536449e-01, -4.77471196e-01, -6.36717421e-01,
       -6.14894419e-01,  3.47560090e+00, -6.03220609e-01, -6.32197363e-01,
       -5.11099887e-01, -5.21885527e-01, -6.15067909e-01, -6.03220609e-01,
        3.46760678e+00, -5.86556902e-01, -6.01909968e-01, -5.38965867e-01,
       -5.93306073e-01, -5.52735246e-01, -6.05013303e-01,  3.47459277e+00,
       -6.08231056e-01, -5.31560604e-01, -5.83746493e-01, -6.71606638e-01,
       -4.87860996e-01,  3.52193758e+00, -6.38767656e-01, -5.51010549e-01,
       -3.60421149e-01, -8.12270595e-01, -4.59309931e-01, -6.02074777e-01,
       -6.38767656e-01,  3.49834514e+00, -6.67509350e-01, -7.09310596e-01,
       -4.21372828e-01, -5.14289476e-01, -6.17529215e-01, -6.67509350e-01,
        3.48750475e+00, -5.98449970e-01, -5.32680053e-01, -5.57046683e-01,
       -6.33474060e-01, -5.26174878e-01, -5.51010549e-01,  3.45337019e+00,
       -5.94972086e-01, -5.37531333e-01, -6.10207288e-01, -5.19353730e-01,
       -6.00137493e-01, -5.98449970e-01,  3.43236915e+00, -6.21449931e-01,
       -5.82029775e-01, -5.10948250e-01, -5.16435834e-01, -6.02104465e-01,
       -6.21449931e-01,  3.46623028e+00, -6.25977561e-01, -6.09976249e-01,
       -4.90286238e-01, -5.59286180e-01, -5.60599391e-01, -6.08231056e-01,
        3.45730438e+00, -6.15927071e-01, -5.43986564e-01, -5.69274118e-01,
       -5.92370794e-01, -5.37409357e-01, -5.96235144e-01,  3.44039272e+00,
       -6.02853209e-01, -5.40103468e-01, -5.71420745e-01, -5.07902188e-01,
       -5.96306871e-01, -5.92350048e-01,  3.45119308e+00, -6.17420365e-01,
       -5.84935913e-01, -5.52277693e-01, -4.98135862e-01, -6.36710743e-01,
        3.42940526e+00, -5.84344563e-01, -5.92378687e-01, -5.59099031e-01,
       -5.58736373e-01, -6.39048592e-01, -5.06545597e-01, -5.84344563e-01,
        3.53539292e+00, -5.60006798e-01, -6.61811961e-01, -5.83635406e-01,
       -6.27428420e-01, -5.47481129e-01, -5.60006798e-01,  3.47570815e+00,
       -5.40941287e-01, -6.27161495e-01, -5.72689019e-01, -5.89461509e-01,
       -9.17294968e-01, -5.40941287e-01,  3.60590918e+00, -1.01256695e+00,
       -5.45644464e-01, -1.31090816e-01, -8.14399825e-01, -1.01256695e+00,
        3.71228858e+00, -8.05314455e-01, -5.74468500e-01, -3.74448033e-01,
       -4.99727283e-01, -6.15482273e-01, -5.92378687e-01,  3.44103621e+00,
       -6.11634102e-01, -6.09884646e-01, -5.11929218e-01, -4.98004244e-01,
       -5.58605532e-01, -6.11634102e-01,  3.43323122e+00, -5.97304377e-01,
       -6.60815174e-01, -5.06867791e-01, -5.92956785e-01, -1.61388800e+00,
        4.56351189e+00, -3.76462153e-01, -1.62026399e+00, -3.59940962e-01,
       -4.54577922e-01, -7.89734528e-01, -3.76462153e-01,  3.39115922e+00,
       -6.24842281e-01, -7.23995595e-01, -4.21546738e-01, -3.20383198e-01,
       -7.64718011e-01, -6.32999318e-01, -4.74621411e-01,  3.50039001e+00,
       -6.61661728e-01, -6.46006343e-01, -4.60269869e-01, -6.12066604e-01,
       -6.61661728e-01,  3.40087226e+00, -5.21693674e-01, -5.70696054e-01,
       -5.74484328e-01, -5.69686677e-01, -5.08411045e-01, -6.46006343e-01,
       -5.21693674e-01,  3.34198584e+00, -5.22065850e-01, -5.74122247e-01,
       -6.81341145e-01, -5.91819248e-01, -5.22065850e-01,  3.44908033e+00,
       -4.08633786e-01, -5.86905072e-01, -6.58315225e-01, -7.48377545e-01,
       -5.47520015e-01, -4.08633786e-01,  3.55542144e+00, -5.55297415e-01,
       -5.52593680e-01, -7.42998995e-01, -6.11681986e-01, -5.52652698e-01,
        3.44778830e+00, -5.43097053e-01, -5.62979919e-01, -6.05280117e-01,
       -5.72096529e-01, -5.72508294e-01, -6.15243360e-01, -5.43097053e-01,
        3.45819946e+00, -5.23571585e-01, -6.22245008e-01, -5.81534158e-01,
       -5.90351331e-01, -5.73080611e-01, -5.62979919e-01,  3.46131897e+00,
       -5.75596742e-01, -5.78574293e-01, -5.80736076e-01, -5.86071579e-01,
       -6.29599083e-01, -5.23571585e-01,  3.47020035e+00, -5.11682175e-01,
       -6.29184431e-01, -5.90091495e-01, -5.88864174e-01, -6.29526830e-01,
       -5.11682175e-01,  3.45004124e+00, -4.96085212e-01, -6.32628453e-01,
       -5.91254400e-01, -5.59389606e-01, -5.86556902e-01, -5.75596742e-01,
        3.46161579e+00, -5.87109306e-01, -5.96773136e-01, -5.56190101e-01,
       -6.00585692e-01, -6.36545752e-01, -4.96085212e-01,  3.45411484e+00,
       -4.85495054e-01, -6.50903551e-01, -5.84499575e-01, -5.83858563e-01,
       -6.76470677e-01, -4.85495054e-01,  3.47426895e+00, -4.62133630e-01,
       -6.91502525e-01, -5.74808507e-01, -5.50391139e-01, -6.89626369e-01,
       -4.62133630e-01,  3.54168400e+00, -6.69764200e-01, -8.39510376e-01,
       -3.30258288e-01, -5.96432788e-01, -5.31560604e-01, -5.94972086e-01,
        3.46085380e+00, -6.21774744e-01, -5.35957552e-01, -5.80156027e-01,
       -5.51453999e-01, -6.25977561e-01, -5.97304377e-01,  3.53339065e+00,
       -6.14025378e-01, -5.84692394e-01, -5.59936940e-01, -5.05490072e-01,
       -6.15927071e-01, -6.69764200e-01,  3.46839064e+00, -5.84436552e-01,
       -5.15209974e-01, -5.77562766e-01, -5.66927086e-01, -5.71698673e-01,
       -6.02853209e-01,  3.45828607e+00, -5.81539848e-01, -5.59013870e-01,
       -5.76253388e-01, -6.03762476e-01, -4.76607839e-01,  3.47360766e+00,
       -6.28813035e-01, -6.26540673e-01, -6.09666822e-01, -5.28216819e-01,
       -5.26897653e-01, -6.48312085e-01, -6.28813035e-01,  3.50801999e+00,
       -4.15291833e-01, -5.56137547e-01, -7.32567839e-01, -6.06055850e-01,
       -5.09351728e-01, -6.26540673e-01,  3.47708810e+00, -6.23618988e-01,
       -4.98464226e-01, -6.13056632e-01, -5.94677589e-01, -5.32749432e-01,
       -6.23618988e-01,  3.47365031e+00, -5.96408736e-01, -5.17278936e-01,
       -6.08916629e-01, -5.62372283e-01, -7.97227236e-01, -4.15291833e-01,
        3.66579128e+00, -4.18217593e-01, -2.74599129e-01, -1.50437096e-01,
       -1.04764611e+00, -5.94285300e-01, -5.40103468e-01, -5.96408736e-01,
        3.48197783e+00, -6.43956272e-01, -5.46637065e-01, -5.60586992e-01,
       -5.35931154e-01, -6.17420365e-01, -5.81539848e-01,  3.44894799e+00,
       -5.32837804e-01, -6.40978234e-01, -5.40240584e-01, -4.14099522e-01,
       -7.46173507e-01, -4.18217593e-01,  3.54486769e+00, -6.46971206e-01,
       -9.39840196e-01, -3.79565666e-01, -5.16085968e-02, -8.05314455e-01,
       -1.62026399e+00,  4.03977498e+00, -6.54185006e-01, -6.37458108e-01,
       -2.70944824e-01, -4.06025294e-01, -6.24842281e-01, -6.46971206e-01,
        3.47724575e+00, -6.68502961e-01, -7.20275903e-01, -4.10628109e-01,
       -5.71855369e-01, -4.96204278e-01,  3.47470829e+00, -6.22747238e-01,
       -5.83465388e-01, -6.44019523e-01, -5.56416494e-01, -5.35572619e-01,
       -5.77835787e-01, -6.22747238e-01,  3.46152750e+00, -6.16827992e-01,
       -5.46483526e-01, -5.62060334e-01, -6.13163392e-01, -8.31762056e-01,
       -5.83465388e-01,  3.70865497e+00, -1.07232003e+00, -6.07944105e-01,
       -2.98392010e-01, -6.16124952e-01, -1.07232003e+00,  3.61926441e+00,
       -6.66713823e-01, -4.13880015e-01, -5.51833577e-01, -4.09700677e-01,
       -6.72107324e-01, -6.66713823e-01,  3.47038757e+00, -6.00083349e-01,
       -6.00253489e-01, -5.21528912e-01, -4.19595349e-01, -6.43273204e-01,
       -6.00083349e-01,  3.40619522e+00, -6.29552005e-01, -6.77418376e-01,
       -4.36272933e-01, -5.02214769e-01, -5.70696054e-01, -6.29552005e-01,
        3.49209550e+00, -6.56308125e-01, -6.13336425e-01, -5.19988123e-01,
       -5.39466491e-01, -5.93618232e-01, -6.16827992e-01,  3.47569153e+00,
       -6.29823163e-01, -5.53263115e-01, -5.42692536e-01, -5.17254116e-01,
       -5.84935913e-01, -6.29823163e-01,  3.44239008e+00, -6.68329159e-01,
       -5.54901613e-01, -4.87146115e-01, -5.32576031e-01, -6.42352035e-01,
        3.47672527e+00, -4.89538982e-01, -5.76339097e-01, -7.13137950e-01,
       -5.22781179e-01, -6.73974999e-01, -6.52618571e-01, -4.89538982e-01,
        3.54781963e+00, -3.91715007e-01, -4.00018693e-01, -9.39006455e-02,
       -8.46052732e-01, -5.09901042e-01, -6.32197363e-01, -5.76339097e-01,
        3.47229146e+00, -5.97789273e-01, -6.22028200e-01, -5.34036482e-01,
       -6.32536449e-01, -5.55297415e-01, -3.91715007e-01,  3.53037097e+00,
       -4.84696631e-01, -8.08402699e-01, -6.57722773e-01, -5.11099887e-01,
       -6.01909968e-01, -5.97789273e-01,  3.45057164e+00, -6.04321599e-01,
       -5.90800742e-01, -5.44650173e-01, -5.38965867e-01, -5.87109306e-01,
       -6.04321599e-01,  3.46031856e+00, -5.97972466e-01, -5.82831892e-01,
       -5.49117426e-01, -5.83746493e-01, -5.43986564e-01, -6.21774744e-01,
        3.47113501e+00, -6.17585060e-01, -5.10327333e-01, -5.93714816e-01,
       -3.60421149e-01, -7.09310596e-01,  4.16613575e+00, -4.75542654e-01,
       -8.20239008e-01, -1.80062234e+00, -8.12270595e-01, -5.37531333e-01,
       -4.75542654e-01,  3.53734217e+00, -5.28917305e-01, -5.65417222e-01,
       -6.17663062e-01, -4.21372828e-01, -5.32680053e-01, -8.20239008e-01,
        3.50838031e+00, -6.90696822e-01, -1.01631326e-01, -6.00923607e-01,
       -3.40836667e-01, -5.57046683e-01, -5.82029775e-01, -6.90696822e-01,
        3.49270491e+00, -6.13947146e-01, -4.50381900e-01, -5.98602586e-01,
       -6.10207288e-01, -5.35957552e-01, -5.28917305e-01,  3.47768739e+00,
       -6.28923356e-01, -6.19612665e-01, -5.54069225e-01, -5.10948250e-01,
       -6.09976249e-01, -6.13947146e-01,  3.40527097e+00, -6.57775960e-01,
       -5.03247491e-01, -5.09375873e-01, -4.90286238e-01, -6.14025378e-01,
       -6.57775960e-01,  3.47779908e+00, -6.43833147e-01, -5.98057612e-01,
       -4.73820741e-01, -5.69274118e-01, -5.84436552e-01, -6.17585060e-01,
        3.47315907e+00, -5.64711288e-01, -5.00406024e-01, -6.36746030e-01,
       -5.71420745e-01, -5.59013870e-01, -6.43956272e-01,  3.48266779e+00,
       -5.61267415e-01, -5.87225880e-01, -5.59783611e-01, -5.52277693e-01,
       -5.32837804e-01, -6.68329159e-01,  3.51802767e+00, -7.31325321e-01,
       -4.34680439e-01, -5.98577252e-01, -5.59099031e-01, -6.09884646e-01,
        3.46710301e+00, -5.53193629e-01, -5.81826304e-01, -6.12377862e-01,
       -5.50721539e-01, -5.58736373e-01, -6.61811961e-01, -5.53193629e-01,
        3.47351349e+00, -4.86900314e-01, -6.43140084e-01, -5.69731126e-01,
       -5.83635406e-01, -6.27161495e-01, -4.86900314e-01,  3.47681673e+00,
       -5.18540822e-01, -6.67946754e-01, -5.92631943e-01, -5.72689019e-01,
       -5.45644464e-01, -5.74468500e-01, -5.18540822e-01,  3.45948582e+00,
       -7.29639969e-01, -5.18503042e-01, -3.74448033e-01, -6.54185006e-01,
       -7.29639969e-01,  3.52457384e+00, -7.48537745e-01, -6.55776285e-01,
       -3.61986802e-01, -5.11929218e-01, -6.60815174e-01, -5.81826304e-01,
        3.47130352e+00, -5.58601693e-01, -6.02892428e-01, -5.55238703e-01,
       -5.06867791e-01, -5.84692394e-01, -5.58601693e-01,  3.24241521e+00,
       -4.93015153e-01, -5.74186990e-01, -5.25051185e-01, -3.59940962e-01,
       -7.23995595e-01, -6.37458108e-01,  3.48948513e+00, -5.90858631e-01,
       -7.74397637e-01, -4.02834199e-01, -4.21546738e-01, -6.68502961e-01,
       -5.90858631e-01,  3.42990840e+00, -6.69153714e-01, -6.24246553e-01,
       -4.55599806e-01, -5.74484328e-01, -5.74122247e-01, -5.86905072e-01,
       -6.56308125e-01,  3.43644316e+00, -5.76146219e-01, -4.68477174e-01,
       -6.58315225e-01, -5.52593680e-01, -5.76146219e-01,  3.46566004e+00,
       -5.24430385e-01, -6.72167807e-01, -4.82006727e-01, -7.42998995e-01,
       -4.84696631e-01, -5.24430385e-01,  3.48034578e+00, -5.05934632e-01,
       -7.73680927e-01, -4.48604212e-01, -6.05280117e-01, -5.78574293e-01,
        3.45627590e+00, -5.47775185e-01, -5.56673904e-01, -5.77821098e-01,
       -5.90151307e-01, -5.72096529e-01, -6.22245008e-01, -5.47775185e-01,
        3.46193651e+00, -5.28222579e-01, -6.00151262e-01, -5.91445942e-01,
       -5.81534158e-01, -6.29184431e-01, -5.28222579e-01,  3.46067756e+00,
       -5.16894274e-01, -6.12098526e-01, -5.92743588e-01, -5.80736076e-01,
       -5.96773136e-01, -5.56673904e-01,  3.46198355e+00, -5.67323403e-01,
       -6.00417281e-01, -5.60059752e-01, -5.90091495e-01, -6.32628453e-01,
       -5.16894274e-01,  3.46735980e+00, -5.03313363e-01, -6.22051170e-01,
       -6.02381042e-01, -5.91254400e-01, -6.50903551e-01, -5.03313363e-01,
        3.47103957e+00, -4.81005045e-01, -6.46416276e-01, -5.98146934e-01,
       -5.56190101e-01, -5.97972466e-01, -5.67323403e-01,  3.45980987e+00,
       -5.77774170e-01, -6.07575244e-01, -5.52974483e-01, -5.84499575e-01,
       -6.91502525e-01, -4.81005045e-01,  3.48178170e+00, -4.15123554e-01,
       -6.81815141e-01, -6.27835858e-01, -5.74808507e-01, -8.39510376e-01,
       -4.15123554e-01,  3.56735194e+00, -6.51160352e-01, -7.74432050e-01,
       -3.12317102e-01, -3.30258288e-01, -5.15209974e-01, -6.51160352e-01,
        3.71052620e+00, -6.47720422e-01, -1.24530301e+00, -3.20874156e-01,
       -5.80156027e-01, -5.10327333e-01, -6.28923356e-01,  3.47766590e+00,
       -6.90601926e-01, -5.49107985e-01, -5.18549277e-01, -5.59936940e-01,
       -6.43833147e-01, -4.93015153e-01,  3.47692630e+00, -6.15781515e-01,
       -6.65352619e-01, -4.99006930e-01, -5.77562766e-01, -5.64711288e-01,
       -6.47720422e-01,  3.47414885e+00, -5.17865990e-01, -4.82135462e-01,
       -6.84152923e-01, -5.76253388e-01, -6.40978234e-01, -5.61267415e-01,
        3.49705466e+00, -5.13501781e-01, -6.77429678e-01, -5.27624160e-01,
       -6.09666822e-01, -4.98464226e-01,  3.47804573e+00, -5.96409221e-01,
       -6.10813508e-01, -5.41775691e-01, -6.20916264e-01, -5.28216819e-01,
       -5.56137547e-01, -5.96409221e-01,  3.47134620e+00, -6.43321628e-01,
       -5.98555562e-01, -5.48705424e-01, -7.32567839e-01, -2.74599129e-01,
       -6.43321628e-01,  3.57819408e+00, -8.71027161e-01, -5.62263102e-01,
       -4.94415226e-01, -6.13056632e-01, -5.17278936e-01, -6.10813508e-01,
        3.47693169e+00, -5.92217102e-01, -5.03599460e-01, -6.39966056e-01,
       -6.08916629e-01, -5.46637065e-01, -5.92217102e-01,  3.45938410e+00,
       -5.67778345e-01, -5.26495358e-01, -6.17339605e-01, -1.50437096e-01,
       -9.39840196e-01,  3.65124998e+00, -6.20050421e-01, -7.72985423e-01,
       -4.18646072e-01, -7.49290770e-01, -5.60586992e-01, -5.87225880e-01,
       -5.67778345e-01,  3.47954270e+00, -5.63748203e-01, -5.54894272e-01,
       -6.45309008e-01, -5.40240584e-01, -7.31325321e-01, -5.13501781e-01,
        3.51890309e+00, -4.01068910e-01, -7.43070599e-01, -5.89695898e-01,
       -3.79565666e-01, -7.20275903e-01, -6.20050421e-01,  3.50662575e+00,
       -6.48212612e-01, -7.04872814e-01, -4.33648329e-01, -2.70944824e-01,
       -7.48537745e-01, -7.74397637e-01,  3.59125790e+00, -7.30282152e-01,
       -7.42584006e-01, -3.24511532e-01, -4.10628109e-01, -6.69153714e-01,
       -6.48212612e-01,  3.49226864e+00, -6.27526932e-01, -6.58629860e-01,
       -4.78117416e-01, -6.44019523e-01, -6.07944105e-01, -4.13880015e-01,
        3.48418040e+00, -5.73717346e-01, -6.01831765e-01, -6.42787649e-01,
       -5.56416494e-01, -5.46483526e-01, -5.73717346e-01,  3.41613737e+00,
       -6.09711899e-01, -5.48597380e-01, -5.81210724e-01, -5.62060334e-01,
       -5.53263115e-01, -6.09711899e-01,  3.45245417e+00, -6.22581103e-01,
       -5.41402902e-01, -5.63434816e-01, -5.51833577e-01, -6.00253489e-01,
       -6.01831765e-01,  3.44291348e+00, -5.40206136e-01, -5.29834323e-01,
       -6.18954193e-01, -5.21528912e-01, -6.77418376e-01, -5.40206136e-01,
        3.47773086e+00, -4.94668389e-01, -6.29794430e-01, -6.14114618e-01,
       -4.36272933e-01, -6.13336425e-01, -4.94668389e-01,  3.94719321e+00,
       -8.83497105e-01, -1.58843445e+00,  6.90160844e-02, -5.19988123e-01,
       -4.68477174e-01, -6.72167807e-01, -8.83497105e-01,  3.55151194e+00,
       -5.99801774e-01, -4.07579958e-01, -5.42692536e-01, -5.54901613e-01,
       -6.22581103e-01,  3.47256853e+00, -7.32178946e-01, -5.92282383e-01,
       -4.27931953e-01, -4.87146115e-01, -4.34680439e-01, -7.32178946e-01,
        3.99552892e+00, -1.51657624e+00, -8.50503748e-01,  2.55565677e-02,
       -7.13137950e-01, -4.00018693e-01,  3.49177208e+00, -5.54349610e-01,
       -6.44802421e-01, -5.07060703e-01, -6.72402706e-01, -5.22781179e-01,
       -6.22028200e-01, -5.54349610e-01,  3.47592346e+00, -5.88218511e-01,
       -6.54141761e-01, -5.34404200e-01, -9.39006455e-02, -8.08402699e-01,
        4.13109688e+00, -4.26735353e-01, -1.76490248e+00, -4.48964561e-01,
       -5.88191149e-01, -5.34036482e-01, -5.90800742e-01, -5.88218511e-01,
        3.46157378e+00, -5.85096794e-01, -5.94256643e-01, -5.69164609e-01,
       -6.57722773e-01, -5.05934632e-01, -4.26735353e-01,  3.48074934e+00,
       -6.19944779e-01, -6.14453399e-01, -6.55958402e-01, -5.44650173e-01,
       -5.82831892e-01, -5.85096794e-01,  3.45685326e+00, -5.94732277e-01,
       -5.86526382e-01, -5.63015743e-01, -5.49117426e-01, -5.77774170e-01,
       -5.94732277e-01,  3.44902352e+00, -5.94356832e-01, -5.83410017e-01,
       -5.49632794e-01, -5.93714816e-01, -5.00406024e-01, -6.90601926e-01,
        3.50141576e+00, -6.54366915e-01, -4.60705909e-01, -6.01620169e-01,
       -1.80062234e+00, -5.65417222e-01, -1.01631326e-01,  4.17972111e+00,
       -4.13651498e-01, -8.61756208e-01, -4.36642511e-01, -6.17663062e-01,
       -6.19612665e-01, -4.13651498e-01,  3.49870543e+00, -6.00258214e-01,
       -7.80237866e-01, -4.67282126e-01, -6.00923607e-01, -4.50381900e-01,
        3.49475032e+00, -7.01930505e-01, -7.22785001e-01, -5.78161211e-01,
       -4.40568098e-01, -5.98602586e-01, -5.03247491e-01, -7.01930505e-01,
        3.52733741e+00, -6.62100171e-01, -4.84820987e-01, -5.76635669e-01,
       -5.54069225e-01, -5.49107985e-01, -6.00258214e-01,  3.62261163e+00,
       -8.27074467e-01, -8.64289110e-01, -2.27812631e-01, -5.09375873e-01,
       -5.98057612e-01, -6.62100171e-01,  3.45631645e+00, -6.68508284e-01,
       -5.32004020e-01, -4.86270491e-01, -4.73820741e-01, -6.15781515e-01,
       -6.68508284e-01,  3.49541482e+00, -6.27948402e-01, -6.19286129e-01,
       -4.90069753e-01, -6.36746030e-01, -5.17865990e-01, -6.54366915e-01,
        3.51451580e+00, -5.59778574e-01, -4.15867557e-01, -7.29890740e-01,
       -5.59783611e-01, -6.77429678e-01, -5.63748203e-01,  3.45833776e+00,
       -4.57053490e-01, -5.88563023e-01, -6.11759751e-01, -1.04764611e+00,
       -8.71027161e-01, -7.72985423e-01,  3.73253447e+00, -6.27874198e-01,
       -4.13001582e-01, -5.98577252e-01, -4.01068910e-01, -1.51657624e+00,
        4.00600455e+00, -6.75200306e-01, -8.14581845e-01, -6.12377862e-01,
       -6.43140084e-01,  3.48138440e+00, -5.63383075e-01, -4.90594877e-01,
       -6.24128215e-01, -5.47760291e-01, -5.50721539e-01, -6.02892428e-01,
       -5.63383075e-01,  3.47718235e+00, -6.04298720e-01, -6.68242694e-01,
       -4.87643891e-01, -5.69731126e-01, -6.67946754e-01, -4.90594877e-01,
        3.48459235e+00, -4.75615806e-01, -6.92103451e-01, -5.88600340e-01,
       -5.92631943e-01, -5.18503042e-01, -6.55776285e-01, -4.75615806e-01,
        3.50116988e+00, -6.87614980e-01, -5.71027826e-01, -3.61986802e-01,
       -7.30282152e-01, -6.87614980e-01,  3.53391267e+00, -7.35065851e-01,
       -6.30715411e-01, -3.88247471e-01, -5.55238703e-01, -5.74186990e-01,
       -6.04298720e-01,  3.47666624e+00, -6.19834472e-01, -6.18881850e-01,
       -5.04225506e-01, -5.25051185e-01, -6.65352619e-01, -6.19834472e-01,
        3.54964996e+00, -5.91807226e-01, -6.24205400e-01, -5.23399053e-01,
       -4.02834199e-01, -6.24246553e-01, -7.42584006e-01,  3.54961349e+00,
       -6.50312031e-01, -7.20275389e-01, -4.09361315e-01, -4.55599806e-01,
       -6.27526932e-01, -6.50312031e-01,  3.46193592e+00, -5.91274814e-01,
       -6.34724384e-01, -5.02497959e-01, -4.82006727e-01, -7.73680927e-01,
       -5.99801774e-01,  3.69585339e+00, -8.01965990e-01, -8.65261860e-01,
       -1.73136107e-01, -8.46052732e-01, -6.44802421e-01, -1.76490248e+00,
        4.13588691e+00, -4.32544549e-01, -4.47584731e-01, -4.48604212e-01,
       -6.19944779e-01, -8.01965990e-01,  3.98841399e+00, -4.77382439e-01,
       -1.39167922e+00, -2.48837354e-01, -5.77821098e-01, -6.00417281e-01,
        3.46475318e+00, -5.54604937e-01, -5.66730651e-01, -5.96021907e-01,
       -5.69157302e-01, -5.90151307e-01, -6.00151262e-01, -5.54604937e-01,
        3.46511427e+00, -5.39830438e-01, -5.89276738e-01, -5.91099591e-01,
       -5.91445942e-01, -6.12098526e-01, -5.39830438e-01,  3.46597389e+00,
       -5.30768517e-01, -6.03352591e-01, -5.88477876e-01, -5.92743588e-01,
       -6.22051170e-01, -5.30768517e-01,  3.46948127e+00, -5.09789463e-01,
       -6.13909838e-01, -6.00218694e-01, -5.60059752e-01, -6.07575244e-01,
       -5.66730651e-01,  3.46392836e+00, -5.71419615e-01, -5.99875225e-01,
       -5.58267873e-01, -6.02381042e-01, -6.46416276e-01, -5.09789463e-01,
        3.46870155e+00, -4.80103722e-01, -6.23257927e-01, -6.06753118e-01,
       -5.98146934e-01, -6.81815141e-01, -4.80103722e-01,  3.47943430e+00,
       -4.23041930e-01, -6.47016677e-01, -6.49309896e-01, -5.52974483e-01,
       -5.94356832e-01, -5.71419615e-01,  3.46248702e+00, -5.95227104e-01,
       -5.99545512e-01, -5.48963471e-01, -6.27835858e-01, -7.74432050e-01,
       -4.23041930e-01,  3.52254968e+00, -4.25354150e-01, -6.79658574e-01,
       -5.92227121e-01, -3.12317102e-01, -1.24530301e+00, -4.25354150e-01,
        4.02221541e+00, -7.54080047e-01, -1.28516110e+00, -3.20874156e-01,
       -4.82135462e-01, -7.54080047e-01,  3.46630153e+00, -5.22431991e-01,
       -1.70443636e-01, -6.76342292e-01, -5.39993948e-01, -5.18549277e-01,
       -4.60705909e-01, -8.27074467e-01,  3.74667040e+00, -9.23877278e-01,
       -1.01646347e+00, -3.40836667e-01, -8.61756208e-01, -7.22785001e-01,
        3.56486025e+00, -5.28162095e-01, -6.73158606e-01, -4.38161677e-01,
       -4.99006930e-01, -6.27948402e-01, -5.91807226e-01,  3.44173735e+00,
       -5.77131654e-01, -6.40992073e-01, -5.04851063e-01, -6.84152923e-01,
       -5.59778574e-01, -5.22431991e-01,  3.48364706e+00, -4.83906490e-01,
       -5.58096196e-01, -6.75280886e-01, -5.27624160e-01, -7.43070599e-01,
       -4.57053490e-01,  3.55424378e+00, -4.08816599e-01, -8.06167534e-01,
       -6.11511402e-01, -5.41775691e-01, -5.98555562e-01,  3.44640991e+00,
       -5.69279446e-01, -5.58194466e-01, -6.02982961e-01, -5.75621786e-01,
       -6.20916264e-01, -5.03599460e-01, -5.69279446e-01,  3.45040115e+00,
       -6.12425198e-01, -5.62167574e-01, -5.82013203e-01, -5.48705424e-01,
       -5.62263102e-01, -5.58194466e-01,  3.45748031e+00, -6.48724715e-01,
       -5.93869780e-01, -5.45722820e-01, -4.94415226e-01, -6.27874198e-01,
       -6.48724715e-01,  3.44846721e+00, -5.85911248e-01, -6.25782332e-01,
       -4.65759492e-01, -6.39966056e-01, -5.26495358e-01, -6.12425198e-01,
        3.47265570e+00, -5.68246922e-01, -5.06016522e-01, -6.19505640e-01,
       -6.17339605e-01, -5.54894272e-01, -5.68246922e-01,  3.47046519e+00,
       -5.42474102e-01, -5.35843730e-01, -6.51666563e-01, -4.18646072e-01,
       -7.04872814e-01,  3.50064967e+00, -5.94937334e-01, -6.32420490e-01,
       -5.03847633e-01, -6.45925325e-01, -7.49290770e-01, -4.13001582e-01,
       -5.85911248e-01, -5.94937334e-01,  3.54271037e+00, -7.15487721e-01,
       -4.84081712e-01, -6.45309008e-01, -5.88563023e-01, -5.42474102e-01,
        3.54869969e+00, -5.58039790e-01, -5.52180948e-01, -6.62132815e-01,
       -5.89695898e-01, -6.75200306e-01, -4.08816599e-01,  3.57235494e+00,
       -3.82646486e-01, -8.32151876e-01, -6.83843780e-01, -4.33648329e-01,
       -6.58629860e-01, -6.32420490e-01,  3.48097507e+00, -6.50468610e-01,
       -5.86221452e-01, -5.19586331e-01, -3.24511532e-01, -7.35065851e-01,
       -7.20275389e-01,  3.53468910e+00, -6.85222608e-01, -6.88467498e-01,
       -3.81146224e-01, -4.78117416e-01, -5.91274814e-01, -6.50468610e-01,
        3.47263264e+00, -6.19243158e-01, -5.72346754e-01, -5.61181884e-01,
       -6.42787649e-01, -5.48597380e-01, -5.29834323e-01,  3.44571772e+00,
       -5.18001030e-01, -5.36277018e-01, -6.70220321e-01, -5.81210724e-01,
       -5.41402902e-01, -5.18001030e-01,  3.40555856e+00, -5.48618420e-01,
       -5.40008816e-01, -6.76316669e-01, -5.63434816e-01, -5.92282383e-01,
       -5.48618420e-01,  3.45945573e+00, -5.62258759e-01, -5.52509713e-01,
       -6.40351639e-01, -6.18954193e-01, -6.29794430e-01, -5.36277018e-01,
        3.42910264e+00, -4.11031622e-01, -5.13388543e-01, -7.19656838e-01,
       -6.14114618e-01, -1.58843445e+00, -4.11031622e-01,  4.04758992e+00,
       -9.39951138e-01, -4.94058094e-01,  6.90160844e-02, -4.07579958e-01,
       -8.65261860e-01, -9.39951138e-01,  3.66271019e+00, -5.46960159e-01,
       -6.23723942e-01, -3.48249221e-01, -4.27931953e-01, -8.50503748e-01,
       -5.62258759e-01,  3.55279133e+00, -5.13804907e-01, -5.97838043e-01,
       -6.00453923e-01,  2.55565677e-02, -8.14581845e-01, -3.82646486e-01,
       -5.13804907e-01,  3.54174934e+00, -6.14410025e-01, -6.43035935e-01,
       -5.98826709e-01, -5.07060703e-01, -6.54141761e-01,  3.47779125e+00,
       -5.72164039e-01, -5.51403338e-01, -6.34899944e-01, -5.58121466e-01,
       -6.72402706e-01, -4.32544549e-01, -5.72164039e-01,  3.49048374e+00,
       -6.73860224e-01, -5.18868530e-01, -6.20643694e-01, -5.34404200e-01,
       -5.94256643e-01, -5.51403338e-01,  3.43506498e+00, -5.69069868e-01,
       -6.04370916e-01, -5.81560011e-01, -4.48964561e-01, -6.14453399e-01,
        3.47483836e+00, -7.10083739e-01, -5.64759552e-01, -6.15562905e-01,
       -5.21014205e-01, -5.88191149e-01, -4.47584731e-01, -6.73860224e-01,
       -7.10083739e-01,  3.53063700e+00, -5.28301762e-01, -5.82615398e-01,
       -5.69164609e-01, -5.86526382e-01, -5.69069868e-01,  3.46159894e+00,
       -5.75312803e-01, -5.88968933e-01, -5.72556351e-01, -6.55958402e-01,
       -4.77382439e-01, -5.64759552e-01,  3.48636380e+00, -6.48179464e-01,
       -5.06938716e-01, -6.33145230e-01, -5.63015743e-01, -5.83410017e-01,
       -5.75312803e-01,  3.45855168e+00, -5.83527224e-01, -5.90235002e-01,
       -5.63050887e-01, -5.49632794e-01, -5.95227104e-01, -5.83527224e-01,
        3.46412448e+00, -5.85346200e-01, -5.94230673e-01, -5.56160480e-01,
       -6.01620169e-01, -4.15867557e-01, -9.23877278e-01,  3.56095342e+00,
       -2.14466878e-01, -5.94642347e-01, -8.10479196e-01, -4.36642511e-01,
       -7.80237866e-01, -5.28162095e-01,  3.51353476e+00, -5.36296416e-01,
       -7.64080415e-01, -4.68115452e-01, -4.67282126e-01, -8.64289110e-01,
       -5.36296416e-01,  3.52812277e+00, -3.33129732e-01, -7.88602178e-01,
       -5.38523208e-01, -5.78161211e-01, -4.84820987e-01,  3.42819340e+00,
       -6.54241076e-01, -6.72768184e-01, -5.39110905e-01, -4.99091032e-01,
       -4.40568098e-01, -6.73158606e-01, -6.54241076e-01,  3.45408584e+00,
       -6.03789106e-01, -6.04021318e-01, -4.78307633e-01, -5.76635669e-01,
       -5.32004020e-01, -6.72768184e-01,  3.50712156e+00, -6.67217038e-01,
       -5.38356328e-01, -5.20140318e-01, -2.27812631e-01, -1.01646347e+00,
       -2.14466878e-01, -3.33129732e-01,  3.59546109e+00, -4.04981293e-01,
       -7.50775210e-01, -6.47831878e-01, -4.86270491e-01, -6.19286129e-01,
       -6.67217038e-01,  3.49135178e+00, -6.70385225e-01, -5.82209487e-01,
       -4.65983415e-01, -4.90069753e-01, -5.77131654e-01, -6.70385225e-01,
        3.46068953e+00, -7.16443290e-01, -5.55416256e-01, -4.51243354e-01,
       -7.29890740e-01, -4.83906490e-01, -5.94642347e-01,  3.52164375e+00,
       -5.57919965e-01, -4.52087196e-01, -7.03197011e-01, -6.11759751e-01,
       -8.06167534e-01, -5.58039790e-01,  3.71268647e+00, -2.62242583e-01,
       -5.81562674e-01, -8.92914132e-01, -6.24128215e-01, -6.68242694e-01,
        3.50031333e+00, -4.78750037e-01, -6.15098517e-01, -6.19547722e-01,
       -4.94546149e-01, -5.47760291e-01, -6.92103451e-01, -4.78750037e-01,
        3.50278831e+00, -4.78034381e-01, -7.58077389e-01, -5.48062761e-01,
       -4.87643891e-01, -6.18881850e-01, -6.15098517e-01,  3.50397700e+00,
       -6.23704495e-01, -7.15108866e-01, -4.43539381e-01, -5.88600340e-01,
       -5.71027826e-01, -6.30715411e-01, -4.78034381e-01,  3.55437866e+00,
       -6.87754326e-01, -5.98246376e-01, -3.88247471e-01, -6.85222608e-01,
       -6.87754326e-01,  3.50175779e+00, -6.93379593e-01, -5.76856723e-01,
       -4.70297066e-01, -5.04225506e-01, -6.24205400e-01, -6.23704495e-01,
        3.47913598e+00, -6.02314183e-01, -6.25696145e-01, -4.98990248e-01,
       -5.23399053e-01, -6.40992073e-01, -6.02314183e-01,  3.46983763e+00,
       -5.75943253e-01, -5.96894948e-01, -5.30294118e-01, -4.09361315e-01,
       -6.34724384e-01, -6.88467498e-01,  3.50954859e+00, -6.20651378e-01,
       -7.00357403e-01, -4.55986618e-01, -5.02497959e-01, -6.19243158e-01,
       -6.20651378e-01,  3.46846887e+00, -5.15471822e-01, -6.54941580e-01,
       -5.55662977e-01, -1.73136107e-01, -1.39167922e+00, -5.46960159e-01,
        4.22604027e+00, -7.24117373e-01, -1.39014741e+00, -2.48837354e-01,
       -6.48179464e-01, -7.24117373e-01,  3.49312783e+00, -4.44061174e-01,
       -2.57023608e-01, -6.20061820e-01, -5.50847039e-01, -5.96021907e-01,
       -5.89276738e-01,  3.46325441e+00, -5.61596692e-01, -5.45437477e-01,
       -5.64962172e-01, -6.05959419e-01, -5.69157302e-01, -5.99875225e-01,
       -5.61596692e-01,  3.46091653e+00, -5.74635338e-01, -6.04256409e-01,
       -5.51395565e-01, -5.91099591e-01, -6.03352591e-01, -5.45437477e-01,
        3.45569423e+00, -5.44931977e-01, -5.85432596e-01, -5.85440002e-01,
       -5.88477876e-01, -6.13909838e-01, -5.44931977e-01,  3.46168827e+00,
       -5.21056389e-01, -6.03440787e-01, -5.89871403e-01, -6.00218694e-01,
       -6.23257927e-01, -5.21056389e-01,  3.47244710e+00, -5.09561082e-01,
       -6.18715136e-01, -5.99637876e-01, -5.58267873e-01, -5.99545512e-01,
       -5.74635338e-01,  3.46178145e+00, -5.85301367e-01, -5.93223452e-01,
       -5.50807903e-01, -6.06753118e-01, -6.47016677e-01, -5.09561082e-01,
        3.48486547e+00, -4.62187868e-01, -6.21242040e-01, -6.38104683e-01,
       -6.49309896e-01, -6.79658574e-01, -4.62187868e-01,  3.50642945e+00,
       -3.74113145e-01, -6.15148542e-01, -7.26011422e-01, -5.48963471e-01,
       -5.85346200e-01, -5.85301367e-01,  3.45563343e+00, -5.92198307e-01,
       -5.80204760e-01, -5.63619327e-01, -5.92227121e-01, -1.28516110e+00,
       -1.70443636e-01, -3.74113145e-01,  3.77843763e+00, -7.57512570e-01,
       -5.98980059e-01, -6.76342292e-01, -5.58096196e-01,  3.47262664e+00,
       -5.30835257e-01, -5.33757515e-01, -5.34862947e-01, -6.38732436e-01,
       -4.38161677e-01, -7.64080415e-01, -6.03789106e-01,  3.51076047e+00,
       -5.24941345e-01, -6.84495161e-01, -4.95292769e-01, -5.04851063e-01,
       -7.16443290e-01, -5.75943253e-01,  3.47229292e+00, -5.25661684e-01,
       -6.14917074e-01, -5.34476561e-01, -6.75280886e-01, -5.57919965e-01,
       -5.30835257e-01,  3.46802784e+00, -5.08080247e-01, -5.37806303e-01,
       -6.58105184e-01, -6.11511402e-01, -8.32151876e-01, -2.62242583e-01,
        3.63939307e+00, -3.67336775e-01, -2.27076676e-01, -2.62814519e-01,
       -1.07625924e+00, -6.02982961e-01, -5.93869780e-01,  3.47040829e+00,
       -5.16609914e-01, -5.65974377e-01, -5.85363045e-01, -6.05608214e-01,
       -5.75621786e-01, -5.62167574e-01, -5.16609914e-01,  3.47773083e+00,
       -6.26025472e-01, -6.49091403e-01, -5.48214683e-01, -5.82013203e-01,
       -5.06016522e-01, -6.26025472e-01,  3.45583492e+00, -5.96530493e-01,
       -5.75420411e-01, -5.69828823e-01, -5.45722820e-01, -6.25782332e-01,
       -5.65974377e-01,  3.42563548e+00, -5.62271918e-01, -6.47561597e-01,
       -4.78322437e-01, -4.65759492e-01, -7.15487721e-01, -5.62271918e-01,
        3.53143446e+00, -5.72256344e-01, -6.99336542e-01, -5.16322447e-01,
       -6.19505640e-01, -5.35843730e-01, -5.96530493e-01,  3.46063367e+00,
       -5.68263278e-01, -5.40268615e-01, -6.00221913e-01, -6.51666563e-01,
       -5.52180948e-01, -5.68263278e-01,  3.50201501e+00, -4.71550549e-01,
       -4.94965101e-01, -7.63388575e-01, -5.03847633e-01, -5.86221452e-01,
        3.46916390e+00, -6.19261771e-01, -6.35991821e-01, -5.70003222e-01,
       -5.53838004e-01, -6.45925325e-01, -4.84081712e-01, -5.72256344e-01,
       -6.19261771e-01,  3.47610420e+00, -6.07562905e-01, -5.47016138e-01,
       -6.62132815e-01, -5.81562674e-01, -4.71550549e-01,  3.49986840e+00,
       -4.98138810e-01, -4.98877997e-01, -7.87605558e-01, -6.83843780e-01,
       -6.14410025e-01, -3.67336775e-01,  3.55063837e+00, -5.44770273e-01,
       -7.90971997e-01, -5.49305518e-01, -5.19586331e-01, -5.72346754e-01,
       -6.35991821e-01,  3.46991916e+00, -6.28503775e-01, -5.26715589e-01,
       -5.86774894e-01, -3.81146224e-01, -6.93379593e-01, -7.00357403e-01,
        3.51297305e+00, -6.64268108e-01, -6.27838494e-01, -4.45983224e-01,
       -5.61181884e-01, -5.15471822e-01, -6.28503775e-01,  3.47385589e+00,
       -6.46125576e-01, -5.13105928e-01, -6.09466905e-01, -6.70220321e-01,
       -5.40008816e-01, -5.13388543e-01,  3.44516419e+00, -5.74441729e-01,
       -5.61268024e-01, -5.85836755e-01, -6.76316669e-01, -5.52509713e-01,
       -5.74441729e-01,  3.49559388e+00, -5.63860974e-01, -4.68619939e-01,
       -6.59844859e-01, -6.40351639e-01, -5.97838043e-01, -5.63860974e-01,
        3.52722908e+00, -6.66536869e-01, -4.92004725e-01, -5.66636833e-01,
       -7.19656838e-01, -4.94058094e-01, -6.23723942e-01, -5.61268024e-01,
        3.47211707e+00, -4.25380171e-01, -6.48029996e-01, -6.00453923e-01,
       -6.43035935e-01, -6.66536869e-01,  3.50800162e+00, -5.56700530e-01,
       -4.86619974e-01, -5.54654388e-01, -6.34899944e-01, -5.18868530e-01,
        3.46468955e+00, -5.46870046e-01, -5.65308773e-01, -6.29574155e-01,
       -5.69168104e-01, -5.58121466e-01, -6.04370916e-01, -5.46870046e-01,
        3.46019985e+00, -5.48238434e-01, -6.09030298e-01, -5.93568689e-01,
       -6.20643694e-01, -5.28301762e-01, -5.65308773e-01,  3.48336887e+00,
       -5.97885357e-01, -5.26716941e-01, -6.44512339e-01, -5.81560011e-01,
       -5.88968933e-01, -5.48238434e-01,  3.45991359e+00, -5.65699011e-01,
       -6.09459987e-01, -5.65987211e-01, -6.15562905e-01, -5.06938716e-01,
        3.46871169e+00, -6.03648750e-01, -6.19574777e-01, -6.23317783e-01,
       -4.99668756e-01, -5.21014205e-01, -5.82615398e-01, -5.97885357e-01,
       -6.03648750e-01,  3.47525043e+00, -5.30117926e-01, -6.39968794e-01,
       -5.72556351e-01, -5.90235002e-01, -5.65699011e-01,  3.46828561e+00,
       -5.76513451e-01, -6.05841559e-01, -5.57440240e-01, -6.33145230e-01,
       -4.44061174e-01, -6.19574777e-01,  3.45007935e+00, -6.53561064e-01,
       -4.97057098e-01, -6.02680002e-01, -5.63050887e-01, -5.94230673e-01,
       -5.76513451e-01,  3.45861767e+00, -5.73575445e-01, -5.81106574e-01,
       -5.70140637e-01, -5.56160480e-01, -5.92198307e-01, -5.73575445e-01,
        3.42094803e+00, -5.69717607e-01, -5.83863839e-01, -5.45432351e-01,
       -5.39993948e-01, -7.57512570e-01, -5.33757515e-01,  3.46715681e+00,
       -4.55506510e-01, -6.55048932e-01, -5.25337337e-01, -8.10479196e-01,
       -4.04981293e-01, -4.52087196e-01,  3.53043923e+00, -6.02304605e-01,
       -5.86861360e-01, -6.73725584e-01, -4.68115452e-01, -7.88602178e-01,
       -5.24941345e-01,  3.49710432e+00, -4.78907603e-01, -7.11190800e-01,
       -5.25346942e-01, -5.38523208e-01, -7.50775210e-01, -4.78907603e-01,
        3.50454926e+00, -4.58395737e-01, -6.95463034e-01, -5.82484470e-01,
       -5.39110905e-01, -5.38356328e-01,  3.48232173e+00, -7.03643280e-01,
       -6.85766739e-01, -5.08083336e-01, -5.07361146e-01, -4.99091032e-01,
       -6.04021318e-01, -7.03643280e-01,  3.46284585e+00, -6.47474045e-01,
       -5.24745849e-01, -4.83870330e-01, -4.78307633e-01, -6.84495161e-01,
       -6.47474045e-01,  3.48472820e+00, -5.63028030e-01, -5.87059462e-01,
       -5.24363866e-01, -5.20140318e-01, -5.82209487e-01, -6.85766739e-01,
        3.50010977e+00, -6.91456157e-01, -5.88098268e-01, -4.32438797e-01,
       -4.65983415e-01, -5.55416256e-01, -6.91456157e-01,  3.86958314e+00,
       -1.24163819e+00, -9.15089125e-01, -4.51243354e-01, -5.25661684e-01,
       -1.24163819e+00,  3.85786151e+00, -7.22734284e-01, -1.11773749e-02,
       -9.05406622e-01, -7.03197011e-01, -5.08080247e-01, -6.02304605e-01,
        3.52195406e+00, -5.55802501e-01, -4.94414681e-01, -6.58155017e-01,
       -8.92914132e-01, -2.27076676e-01, -4.98138810e-01,  3.64131629e+00,
       -7.51243203e-01, -5.68704067e-01, -7.03239405e-01, -5.98826709e-01,
       -5.44770273e-01, -5.56700530e-01,  3.49083770e+00, -6.04126766e-01,
       -5.47599779e-01, -6.38813645e-01, -6.19547722e-01, -7.15108866e-01,
        3.49155092e+00, -4.74859200e-01, -6.56964601e-01, -3.87180822e-01,
       -6.37889705e-01, -4.94546149e-01, -7.58077389e-01, -4.74859200e-01,
        3.52729911e+00, -4.56167584e-01, -8.28925319e-01, -5.14723472e-01,
       -5.48062761e-01, -5.98246376e-01, -5.76856723e-01, -4.56167584e-01,
        3.45227774e+00, -6.09029340e-01, -6.63914960e-01, -4.43539381e-01,
       -6.25696145e-01, -6.56964601e-01,  3.56937839e+00, -6.80196413e-01,
       -5.69424142e-01, -5.93557705e-01, -4.70297066e-01, -6.64268108e-01,
       -6.09029340e-01,  3.47868628e+00, -6.57566507e-01, -5.42323591e-01,
       -5.35201665e-01, -4.98990248e-01, -5.96894948e-01, -6.80196413e-01,
        3.49038912e+00, -6.22537023e-01, -5.43560865e-01, -5.48209628e-01,
       -5.30294118e-01, -6.14917074e-01, -6.22537023e-01,  3.47637757e+00,
       -5.94643177e-01, -5.23020211e-01, -5.90965968e-01, -4.55986618e-01,
       -6.54941580e-01, -6.27838494e-01,  3.48485193e+00, -5.88727952e-01,
       -6.51397742e-01, -5.05959546e-01, -5.55662977e-01, -6.46125576e-01,
       -5.88727952e-01,  3.48375496e+00, -3.93501890e-01, -6.73816191e-01,
       -6.25920376e-01, -3.48249221e-01, -1.39014741e+00, -2.57023608e-01,
       -4.25380171e-01,  3.66885245e+00, -4.97292028e-01, -7.50760004e-01,
       -6.20061820e-01, -6.53561064e-01,  3.48379630e+00, -4.98786363e-01,
       -4.87026109e-01, -5.97836267e-01, -6.26524674e-01, -5.64962172e-01,
       -6.04256409e-01,  3.46440572e+00, -5.60978513e-01, -5.83739511e-01,
       -5.36853322e-01, -6.13615791e-01, -6.05959419e-01, -5.85432596e-01,
       -5.60978513e-01,  3.45379567e+00, -5.49434749e-01, -5.50488547e-01,
       -6.01501849e-01, -5.51395565e-01, -5.93223452e-01, -5.83739511e-01,
        3.46065396e+00, -6.01380796e-01, -5.94652675e-01, -5.36261958e-01,
       -5.85440002e-01, -6.03440787e-01, -5.49434749e-01,  3.43924634e+00,
       -5.37735962e-01, -5.80313963e-01, -5.82880872e-01, -5.89871403e-01,
       -6.18715136e-01, -5.37735962e-01,  3.46726528e+00, -5.17543335e-01,
       -6.03544370e-01, -5.99855077e-01, -5.99637876e-01, -6.21242040e-01,
       -5.17543335e-01,  3.46257196e+00, -5.12668465e-01, -6.02379035e-01,
       -6.09101212e-01, -5.50807903e-01, -5.80204760e-01, -6.01380796e-01,
        3.46125167e+00, -6.01711829e-01, -5.81126646e-01, -5.46019738e-01,
       -6.38104683e-01, -6.15148542e-01, -5.12668465e-01,  3.48621066e+00,
       -4.67738149e-01, -5.88678486e-01, -6.63872340e-01, -7.26011422e-01,
       -5.98980059e-01, -4.55506510e-01, -4.67738149e-01,  3.51594146e+00,
       -7.04221952e-01, -5.63483367e-01, -5.63619327e-01, -5.69717607e-01,
       -6.01711829e-01,  3.46989609e+00, -6.22548664e-01, -5.72892301e-01,
       -5.39406362e-01, -5.34862947e-01, -6.55048932e-01,  3.48422752e+00,
       -5.74861543e-01, -5.61535917e-01, -6.11621783e-01, -5.46296401e-01,
       -6.38732436e-01, -5.37806303e-01, -5.74861543e-01,  3.40320806e+00,
       -5.53202967e-01, -5.26908862e-01, -5.71695946e-01, -6.47831878e-01,
       -5.86861360e-01, -4.58395737e-01,  3.49090340e+00, -5.59718311e-01,
       -5.98014298e-01, -6.40081818e-01, -4.95292769e-01, -7.11190800e-01,
       -5.63028030e-01,  3.46156255e+00, -5.32199692e-01, -6.27469369e-01,
       -5.32381889e-01, -5.34476561e-01, -7.22734284e-01, -5.94643177e-01,
        3.45145915e+00, -4.49988755e-01, -5.42364330e-01, -6.07252043e-01,
       -6.58105184e-01, -5.55802501e-01, -5.53202967e-01,  3.46643232e+00,
       -5.09300517e-01, -5.91776469e-01, -5.98244678e-01, -2.62814519e-01,
       -7.90971997e-01,  3.53712496e+00, -7.34742665e-01, -5.76827689e-01,
       -5.26154897e-01, -6.45613191e-01, -1.07625924e+00, -7.51243203e-01,
       -7.34742665e-01,  3.74840545e+00, -7.43683698e-01, -4.42476649e-01,
       -5.85363045e-01, -6.47561597e-01,  3.46010069e+00, -5.29364531e-01,
       -5.59058538e-01, -6.12368810e-01, -5.26384173e-01, -6.05608214e-01,
       -6.49091403e-01, -5.29364531e-01,  3.44774280e+00, -5.94921913e-01,
       -5.27059895e-01, -5.41696839e-01, -5.48214683e-01, -5.75420411e-01,
       -5.94921913e-01,  3.41564232e+00, -6.10912490e-01, -5.46742327e-01,
       -5.39430496e-01, -5.69828823e-01, -5.40268615e-01, -6.10912490e-01,
        3.45367694e+00, -5.61688296e-01, -6.36143149e-01, -5.34835569e-01,
       -4.78322437e-01, -6.99336542e-01, -5.59058538e-01,  3.48816669e+00,
       -5.76090141e-01, -6.73645321e-01, -5.01713712e-01, -5.16322447e-01,
       -6.07562905e-01, -5.76090141e-01,  3.49000998e+00, -6.04315549e-01,
       -6.21664679e-01, -5.64054258e-01, -6.00221913e-01, -4.94965101e-01,
       -5.61688296e-01,  3.47840193e+00, -4.88684535e-01, -7.03062230e-01,
       -6.29779852e-01, -7.63388575e-01, -4.98877997e-01, -4.88684535e-01,
        3.62508985e+00, -4.15392497e-01, -6.11545935e-01, -8.47200310e-01,
       -5.70003222e-01, -5.26715589e-01,  3.47021621e+00, -6.26667821e-01,
       -6.18770515e-01, -6.18136933e-01, -5.09922124e-01, -5.53838004e-01,
       -5.47016138e-01, -6.04315549e-01, -6.26667821e-01,  3.46702181e+00,
       -5.42746203e-01, -5.92438091e-01, -7.87605558e-01, -5.68704067e-01,
       -4.15392497e-01,  3.54670740e+00, -3.32888196e-01, -2.37100129e-01,
       -2.43318540e-01, -9.61698412e-01, -5.49305518e-01, -6.04126766e-01,
       -5.76827689e-01,  3.45799063e+00, -5.21277937e-01, -5.78396113e-01,
       -6.28056603e-01, -5.86774894e-01, -5.13105928e-01, -6.18770515e-01,
        3.47200677e+00, -6.27585618e-01, -5.04866422e-01, -6.20903395e-01,
       -4.45983224e-01, -6.57566507e-01, -6.51397742e-01,  3.48131704e+00,
       -5.96235244e-01, -6.16113573e-01, -5.14020753e-01, -6.09466905e-01,
       -3.93501890e-01, -6.27585618e-01,  3.53718924e+00, -8.22491519e-01,
       -5.20572062e-01,  3.99490468e-02, -6.03520289e-01, -5.85836755e-01,
       -4.68619939e-01, -6.48029996e-01, -4.97292028e-01,  3.70541125e+00,
       -8.24597714e-01, -6.81034816e-01, -6.59844859e-01, -4.92004725e-01,
       -8.24597714e-01,  3.55751297e+00, -7.20052877e-01, -4.26216763e-01,
       -4.34796028e-01, -5.66636833e-01, -4.86619974e-01, -7.20052877e-01,
        3.54607063e+00, -7.00869737e-01, -5.21368335e-01, -5.50522869e-01,
       -5.54654388e-01, -5.47599779e-01, -7.00869737e-01,  3.52780298e+00,
       -5.67929772e-01, -3.82370754e-01, -7.74378550e-01, -5.50847039e-01,
       -7.50760004e-01, -4.98786363e-01, -6.81034816e-01, -4.26216763e-01,
        3.65062309e+00, -7.42978103e-01, -6.29574155e-01, -5.26716941e-01,
        3.46389015e+00, -5.40099383e-01, -5.72451048e-01, -6.27753683e-01,
       -5.67294943e-01, -5.69168104e-01, -6.09030298e-01, -5.40099383e-01,
        3.44904574e+00, -5.24508431e-01, -6.03777544e-01, -6.02461979e-01,
       -5.93568689e-01, -6.09459987e-01, -5.24508431e-01,  3.43375635e+00,
       -5.38808144e-01, -6.09396794e-01, -5.58014305e-01, -6.44512339e-01,
       -5.30117926e-01, -5.72451048e-01,  3.48671253e+00, -5.62785672e-01,
       -5.18960890e-01, -6.57884651e-01, -5.65987211e-01, -6.05841559e-01,
       -5.38808144e-01,  3.47945046e+00, -5.75462860e-01, -6.46960301e-01,
       -5.46390385e-01, -6.23317783e-01, -4.97057098e-01,  3.46197439e+00,
       -6.01297792e-01, -6.21748074e-01, -5.05939514e-01, -6.12614130e-01,
       -4.99668756e-01, -6.39968794e-01, -5.62785672e-01, -6.01297792e-01,
        3.48424992e+00, -5.42392271e-01, -6.38136637e-01, -5.57440240e-01,
       -5.81106574e-01, -5.75462860e-01,  3.45619980e+00, -6.00083577e-01,
       -6.04534058e-01, -5.37572492e-01, -6.02680002e-01, -4.87026109e-01,
       -6.21748074e-01,  3.46492963e+00, -6.39128025e-01, -5.21867333e-01,
       -5.92480087e-01, -5.70140637e-01, -5.83863839e-01, -6.00083577e-01,
        3.46829556e+00, -5.90340273e-01, -5.89791121e-01, -5.34076115e-01,
       -5.45432351e-01, -6.22548664e-01, -5.90340273e-01,  3.47600579e+00,
       -5.80508121e-01, -6.20062390e-01, -5.17113991e-01, -5.25337337e-01,
       -7.04221952e-01, -5.61535917e-01,  3.47617940e+00, -5.24207656e-01,
       -6.33654235e-01, -5.27222302e-01, -6.73725584e-01, -4.94414681e-01,
       -5.59718311e-01,  3.51872806e+00, -6.18647659e-01, -4.73842244e-01,
       -6.98379581e-01, -5.25346942e-01, -6.95463034e-01, -5.32199692e-01,
        3.47304404e+00, -5.01902740e-01, -6.29059115e-01, -5.89072513e-01,
       -5.82484470e-01, -5.98014298e-01, -5.01902740e-01,  3.41976834e+00,
       -5.64104324e-01, -5.96169391e-01, -5.77093115e-01, -5.08083336e-01,
       -5.24745849e-01,  3.53643129e+00, -7.71288232e-01, -8.09362646e-01,
       -4.97496123e-01, -4.25455105e-01, -5.07361146e-01, -5.88098268e-01,
       -7.71288232e-01,  3.49630634e+00, -6.14091300e-01, -4.53270385e-01,
       -5.62197007e-01, -4.83870330e-01, -5.87059462e-01, -8.09362646e-01,
        3.50583447e+00, -6.22489597e-01, -3.88893458e-01, -6.14158975e-01,
       -5.24363866e-01, -6.27469369e-01, -6.22489597e-01,  3.46772457e+00,
       -5.97958397e-01, -5.57240172e-01, -5.38203166e-01, -4.32438797e-01,
       -9.15089125e-01, -1.11773749e-02, -6.14091300e-01,  3.59775655e+00,
       -3.53236483e-01, -6.85221336e-01, -5.86502133e-01, -9.05406622e-01,
       -4.49988755e-01, -3.53236483e-01,  3.62345561e+00, -5.92435340e-01,
       -6.55163587e-01, -6.67224818e-01, -6.58155017e-01, -5.09300517e-01,
       -6.18647659e-01,  3.50508464e+00, -6.46330415e-01, -4.01894899e-01,
       -6.70756135e-01, -7.03239405e-01, -7.43683698e-01, -3.32888196e-01,
        3.73586291e+00, -1.96756407e-01, -6.77193790e-01, -1.08210142e+00,
       -6.38813645e-01, -5.21277937e-01, -5.67929772e-01,  3.45218463e+00,
       -5.28399190e-01, -4.69167536e-01, -7.26596551e-01, -3.87180822e-01,
       -8.28925319e-01,  3.54671140e+00, -5.90545560e-01, -4.66453383e-01,
       -7.97160017e-01, -4.76446303e-01, -6.37889705e-01, -5.69424142e-01,
       -5.90545560e-01,  3.56952460e+00, -7.01372162e-01, -3.61387397e-01,
       -7.08905630e-01, -5.14723472e-01, -6.63914960e-01, -5.42323591e-01,
       -4.66453383e-01,  3.51013714e+00, -6.07469137e-01, -7.15252597e-01,
       -5.93557705e-01, -5.43560865e-01, -7.01372162e-01,  3.55978139e+00,
       -5.13936207e-01, -2.98772636e-01, -9.08581813e-01, -5.35201665e-01,
       -5.96235244e-01, -6.07469137e-01,  3.48643815e+00, -6.66740915e-01,
       -4.77927880e-01, -6.02863309e-01, -5.48209628e-01, -5.23020211e-01,
       -5.13936207e-01,  3.64232754e+00, -6.32439608e-01, -5.48210116e-01,
       -6.02011014e-01, -2.74500755e-01, -5.90965968e-01, -5.42364330e-01,
       -6.32439608e-01,  3.45754838e+00, -6.28492873e-01, -4.92870239e-01,
       -5.70415366e-01, -5.05959546e-01, -6.73816191e-01, -6.16113573e-01,
        3.48146974e+00, -4.87815776e-01, -5.88002891e-01, -6.09761759e-01,
       -6.25920376e-01, -8.22491519e-01, -4.87815776e-01,  3.90164498e+00,
       -1.38182449e+00, -5.83592821e-01, -5.97836267e-01, -4.34796028e-01,
       -5.21368335e-01, -7.42978103e-01,  3.52913565e+00, -4.62912756e-01,
       -7.69244161e-01, -6.26524674e-01, -6.39128025e-01, -4.62912756e-01,
        3.49245469e+00, -5.96623461e-01, -4.99950953e-01, -6.67314820e-01,
       -5.36853322e-01, -5.94652675e-01,  3.46978891e+00, -5.99772716e-01,
       -6.27854460e-01, -4.93059990e-01, -6.17595747e-01, -6.13615791e-01,
       -5.50488547e-01, -5.99772716e-01,  3.46660128e+00, -5.74227257e-01,
       -5.01478160e-01, -6.27018814e-01, -6.01501849e-01, -5.80313963e-01,
       -5.74227257e-01,  3.46331305e+00, -5.67751692e-01, -5.31027689e-01,
       -6.08490604e-01, -5.36261958e-01, -5.81126646e-01, -6.27854460e-01,
        3.46983317e+00, -6.27005477e-01, -5.96874627e-01, -5.00709997e-01,
       -5.82880872e-01, -6.03544370e-01, -5.67751692e-01,  3.45924865e+00,
       -5.37676645e-01, -5.73594868e-01, -5.93800207e-01, -5.99855077e-01,
       -6.02379035e-01, -5.37676645e-01,  3.46059627e+00, -5.36576059e-01,
       -5.88515810e-01, -5.95593646e-01, -6.09101212e-01, -5.88678486e-01,
       -5.36576059e-01,  3.45262959e+00, -5.18521520e-01, -5.76274044e-01,
       -6.23478270e-01, -5.46019738e-01, -5.72892301e-01, -6.27005477e-01,
        3.46623232e+00, -6.29545296e-01, -5.75404578e-01, -5.15364932e-01,
       -6.63872340e-01, -5.63483367e-01, -5.24207656e-01, -5.18521520e-01,
        3.45722082e+00, -6.39781507e-01, -5.47354427e-01, -5.39406362e-01,
       -5.80508121e-01, -6.29545296e-01,  3.46156594e+00, -6.26821679e-01,
       -5.70233220e-01, -5.15051261e-01, -6.11621783e-01, -5.26908862e-01,
        3.47780291e+00, -5.97061768e-01, -6.03259097e-01, -5.85079813e-01,
       -5.53871587e-01, -5.46296401e-01, -6.33654235e-01, -5.97061768e-01,
        3.49436515e+00, -5.83707468e-01, -5.68758936e-01, -5.64886342e-01,
       -5.71695946e-01, -5.91776469e-01, -6.03259097e-01,  3.47667030e+00,
       -5.61335265e-01, -6.04017472e-01, -5.44586050e-01, -6.40081818e-01,
       -4.73842244e-01, -5.64104324e-01,  3.46666539e+00, -6.95084443e-01,
       -5.42593117e-01, -5.50959448e-01, -5.32381889e-01, -6.29059115e-01,
       -5.97958397e-01,  3.47995513e+00, -5.65880080e-01, -5.38938235e-01,
       -6.15737416e-01, -6.07252043e-01, -5.92435340e-01, -6.28492873e-01,
        3.49758829e+00, -5.82885961e-01, -5.06222167e-01, -5.80299902e-01,
       -5.98244678e-01, -6.46330415e-01, -5.61335265e-01,  3.47686781e+00,
       -4.63718792e-01, -5.91286033e-01, -6.15952633e-01, -5.26154897e-01,
       -5.78396113e-01,  3.44957113e+00, -5.93484860e-01, -5.58431001e-01,
       -5.59831462e-01, -6.33272798e-01, -6.45613191e-01, -4.42476649e-01,
       -1.96756407e-01, -5.93484860e-01,  3.56545200e+00, -8.72142839e-01,
       -5.40680956e-01, -2.74297100e-01, -6.12368810e-01, -5.27059895e-01,
        3.50981723e+00, -6.35083551e-01, -6.47462093e-01, -5.40687327e-01,
       -5.47155557e-01, -5.26384173e-01, -6.73645321e-01, -6.35083551e-01,
        3.48704293e+00, -5.94351014e-01, -5.68857460e-01, -4.88721410e-01,
       -5.41696839e-01, -5.46742327e-01, -6.47462093e-01,  3.49020164e+00,
       -6.14964192e-01, -5.68166285e-01, -5.71169904e-01, -5.39430496e-01,
       -6.36143149e-01, -6.14964192e-01,  3.48535568e+00, -4.70824368e-01,
       -6.62678879e-01, -5.61314596e-01, -5.34835569e-01, -7.03062230e-01,
       -4.70824368e-01,  3.51947661e+00, -4.99457187e-01, -8.20880748e-01,
       -4.90416508e-01, -5.01713712e-01, -6.21664679e-01, -5.94351014e-01,
        3.47489963e+00, -5.90864047e-01, -6.11630957e-01, -5.54675223e-01,
       -5.64054258e-01, -5.42746203e-01, -5.90864047e-01,  3.46661549e+00,
       -6.14227924e-01, -5.54416570e-01, -6.00306492e-01, -6.29779852e-01,
       -6.11545935e-01, -4.99457187e-01,  3.56033105e+00, -5.33261364e-01,
       -6.76699718e-01, -6.09586991e-01, -8.47200310e-01, -2.37100129e-01,
       -5.33261364e-01,  3.55090050e+00, -7.14188974e-01, -5.35822200e-01,
       -6.83327526e-01, -6.18136933e-01, -5.04866422e-01,  3.47494891e+00,
       -6.14159928e-01, -6.01374179e-01, -4.83060676e-01, -6.53350769e-01,
       -5.09922124e-01, -5.92438091e-01, -6.14227924e-01, -6.14159928e-01,
        3.47153335e+00, -5.01877991e-01, -6.38907294e-01, -2.43318540e-01,
       -6.77193790e-01,  3.67918739e+00, -4.90162669e-01, -1.10503742e+00,
       -7.48099078e-01, -4.15375895e-01, -6.28056603e-01, -5.28399190e-01,
       -5.58431001e-01,  3.43649325e+00, -5.57253513e-01, -5.28463263e-01,
       -6.35889679e-01, -6.20903395e-01, -5.20572062e-01, -6.01374179e-01,
        3.47641280e+00, -5.88191013e-01, -4.93302466e-01, -6.52069685e-01,
       -5.14020753e-01, -6.66740915e-01, -5.88002891e-01,  3.47934815e+00,
       -5.15061537e-01, -5.91434818e-01, -6.04087235e-01,  3.99490468e-02,
       -1.38182449e+00,  3.95111121e+00, -7.44497204e-01, -4.57444715e-01,
       -5.21192181e-01, -8.86101671e-01, -5.50522869e-01, -3.82370754e-01,
       -7.69244161e-01, -5.96623461e-01,  3.59447693e+00, -9.40474992e-01,
       -3.15356832e-01, -3.98838602e-02, -7.74378550e-01, -4.69167536e-01,
       -9.40474992e-01,  3.88769951e+00, -4.87526186e-01, -1.21615224e+00,
       -6.27753683e-01, -5.18960890e-01,  3.41046587e+00, -5.47414547e-01,
       -5.35186666e-01, -5.64136917e-01, -6.17013171e-01, -5.67294943e-01,
       -6.03777544e-01, -5.47414547e-01,  3.44399671e+00, -5.42968620e-01,
       -6.08739713e-01, -5.73801339e-01, -6.02461979e-01, -6.09396794e-01,
       -5.42968620e-01,  3.47691022e+00, -5.37742154e-01, -6.07732616e-01,
       -5.76608056e-01, -5.58014305e-01, -6.46960301e-01, -5.37742154e-01,
        3.48195466e+00, -5.49946026e-01, -6.47368032e-01, -5.41923844e-01,
       -6.57884651e-01, -5.42392271e-01, -5.35186666e-01,  3.48488956e+00,
       -5.65212966e-01, -5.65481035e-01, -6.18731971e-01, -5.46390385e-01,
       -6.04534058e-01, -5.49946026e-01,  3.47092556e+00, -6.03362238e-01,
       -6.54380298e-01, -5.12312553e-01, -5.05939514e-01, -6.38136637e-01,
       -5.65212966e-01,  3.43179618e+00, -5.95338680e-01, -5.15885123e-01,
       -6.11283255e-01, -6.12614130e-01, -5.21867333e-01, -5.95338680e-01,
        3.47759656e+00, -6.18426219e-01, -5.22247426e-01, -6.07102772e-01,
       -5.37572492e-01, -5.89791121e-01, -6.03362238e-01,  3.44805183e+00,
       -5.99549720e-01, -5.98406501e-01, -5.19369761e-01, -5.92480087e-01,
       -4.99950953e-01, -6.18426219e-01,  3.46574273e+00, -6.67958049e-01,
       -5.37672303e-01, -5.49255117e-01, -5.34076115e-01, -6.20062390e-01,
       -5.99549720e-01,  3.47383744e+00, -6.00423464e-01, -6.23876778e-01,
       -4.95848972e-01, -5.17113991e-01, -6.26821679e-01, -6.00423464e-01,
        3.46429305e+00, -5.86769184e-01, -6.29735989e-01, -5.03428747e-01,
       -5.27222302e-01, -6.39781507e-01, -5.83707468e-01,  3.46925194e+00,
       -5.74918867e-01, -5.84216140e-01, -5.59405655e-01, -6.98379581e-01,
       -4.01894899e-01, -6.95084443e-01,  3.55931327e+00, -7.06112957e-01,
       -3.38075405e-01, -7.19765980e-01, -5.89072513e-01, -5.96169391e-01,
       -5.65880080e-01,  3.48811749e+00, -5.45831641e-01, -5.44830829e-01,
       -6.46333033e-01, -5.77093115e-01, -5.42593117e-01, -5.45831641e-01,
        3.51361650e+00, -7.85762703e-01, -6.45307227e-01, -4.17028699e-01,
       -4.97496123e-01, -3.88893458e-01,  4.49151385e+00, -2.25260683e+00,
       -5.98144026e-01, -7.54373412e-01, -4.25455105e-01, -4.53270385e-01,
       -2.25260683e+00,  4.48680911e+00, -5.86769933e-01, -2.79655084e-02,
       -7.40741343e-01, -5.62197007e-01, -6.85221336e-01, -5.86769933e-01,
        3.47630810e+00, -4.88038955e-01, -5.91642610e-01, -5.62438255e-01,
       -6.14158975e-01, -5.57240172e-01, -5.98144026e-01,  3.49291589e+00,
       -6.94385540e-01, -6.30034686e-01, -3.98952491e-01, -5.38203166e-01,
       -5.38938235e-01, -6.94385540e-01,  3.50078088e+00, -6.53151624e-01,
       -4.27340956e-01, -6.48761357e-01, -5.86502133e-01, -6.55163587e-01,
       -4.88038955e-01,  3.46464476e+00, -5.04604840e-01, -6.24510764e-01,
       -6.05824482e-01, -6.67224818e-01, -5.82885961e-01, -5.04604840e-01,
        3.47912553e+00, -5.24558301e-01, -5.97295378e-01, -6.02556231e-01,
       -6.70756135e-01, -4.63718792e-01, -7.06112957e-01,  3.49550478e+00,
       -6.25726946e-01, -3.00136930e-01, -7.29053022e-01, -1.08210142e+00,
       -8.72142839e-01, -4.90162669e-01,  3.86199399e+00, -1.07575485e+00,
       -3.41832219e-01, -9.61698412e-01, -7.14188974e-01, -1.10503742e+00,
        3.78835871e+00, -4.78617791e-01, -5.28816118e-01, -7.26596551e-01,
       -5.57253513e-01, -4.87526186e-01,  3.50207778e+00, -4.90621497e-01,
       -5.18510213e-01, -7.21569823e-01, -6.03520289e-01, -5.88191013e-01,
       -7.44497204e-01,  3.52115925e+00, -5.17769308e-01, -3.96059686e-01,
       -6.71121755e-01, -7.97160017e-01, -3.61387397e-01,  3.52522014e+00,
       -4.85260218e-01, -6.94294482e-01, -4.70817354e-01, -7.16300671e-01,
       -4.76446303e-01, -7.15252597e-01, -4.77927880e-01, -4.85260218e-01,
        3.53174745e+00, -6.23730675e-01, -7.53129780e-01, -7.08905630e-01,
       -2.98772636e-01, -6.94294482e-01,  3.55367840e+00, -7.78566160e-01,
       -4.09375078e-01, -6.63764415e-01, -9.08581813e-01, -5.48210116e-01,
       -7.78566160e-01,  3.88914260e+00, -8.52693311e-01, -3.45861053e-01,
       -4.55230143e-01, -6.02863309e-01, -5.15061537e-01, -6.23730675e-01,
        3.49928078e+00, -6.59420378e-01, -3.93499597e-01, -7.04705279e-01,
       -6.02011014e-01, -4.92870239e-01,  3.53905639e+00, -6.55596835e-01,
       -6.72367506e-01, -6.23381406e-01, -4.92829393e-01, -5.70415366e-01,
       -5.06222167e-01, -6.55596835e-01,  3.48144193e+00, -6.62493791e-01,
       -5.10707392e-01, -5.76006382e-01, -6.09761759e-01, -5.83592821e-01,
       -5.91434818e-01, -4.57444715e-01,  3.47499984e+00, -5.51615322e-01,
       -6.81150401e-01, -6.67314820e-01, -3.15356832e-01, -6.67958049e-01,
        3.58801817e+00, -9.05907767e-01, -5.78127697e-01, -4.53353000e-01,
       -4.93059990e-01, -5.96874627e-01,  3.48351393e+00, -6.47881143e-01,
       -6.75432668e-01, -6.31203687e-01, -4.39061815e-01, -6.17595747e-01,
       -5.01478160e-01, -6.47881143e-01,  3.47460353e+00, -6.22720155e-01,
       -4.51697673e-01, -6.33230649e-01, -6.27018814e-01, -5.31027689e-01,
       -6.22720155e-01,  3.46322777e+00, -5.87539654e-01, -4.77721903e-01,
       -6.17199557e-01, -6.08490604e-01, -5.73594868e-01, -5.87539654e-01,
        3.45187710e+00, -5.59458667e-01, -5.27775527e-01, -5.95017776e-01,
       -5.00709997e-01, -5.75404578e-01, -6.75432668e-01,  3.47578976e+00,
       -6.69230422e-01, -5.92283478e-01, -4.62728619e-01, -5.93800207e-01,
       -5.88515810e-01, -5.59458667e-01,  3.45365449e+00, -5.53904269e-01,
       -5.72725524e-01, -5.85250011e-01, -5.95593646e-01, -5.76274044e-01,
       -5.53904269e-01,  3.43404809e+00, -5.66997152e-01, -5.73482369e-01,
       -5.67796613e-01, -6.23478270e-01, -5.47354427e-01, -5.74918867e-01,
       -5.66997152e-01,  3.45137461e+00, -5.78307589e-01, -5.60318302e-01,
       -5.15364932e-01, -5.70233220e-01, -6.69230422e-01,  3.47110234e+00,
       -6.58491977e-01, -5.63048157e-01, -4.94733631e-01, -5.15051261e-01,
       -5.86769184e-01, -6.58491977e-01,  3.46524449e+00, -6.40795841e-01,
       -5.58329547e-01, -5.05806683e-01, -5.85079813e-01, -5.68758936e-01,
        3.47215286e+00, -5.78746898e-01, -5.88416961e-01, -5.87452078e-01,
       -5.63698174e-01, -5.53871587e-01, -6.04017472e-01, -5.78746898e-01,
        3.48116397e+00, -5.59328501e-01, -6.07150969e-01, -5.78048542e-01,
       -5.64886342e-01, -5.84216140e-01, -5.88416961e-01,  3.47383761e+00,
       -5.98156134e-01, -5.68226806e-01, -5.69935222e-01, -5.44586050e-01,
       -5.91286033e-01, -5.59328501e-01,  3.49736423e+00, -5.24581744e-01,
       -6.26943913e-01, -6.50637988e-01, -5.50959448e-01, -3.38075405e-01,
       -7.85762703e-01,  3.80767159e+00, -1.28816274e+00, -8.44711294e-01,
       -6.15737416e-01, -5.44830829e-01, -6.53151624e-01,  3.46579822e+00,
       -4.98014956e-01, -3.74753056e-01, -7.79310336e-01, -5.80299902e-01,
       -5.24558301e-01, -6.62493791e-01,  3.47790200e+00, -6.26400442e-01,
       -5.08725522e-01, -5.75424040e-01, -6.15952633e-01, -6.25726946e-01,
       -5.24581744e-01,  3.35407086e+00, -4.64365843e-01, -5.70051051e-01,
       -5.53392647e-01, -5.59831462e-01, -5.40680956e-01,  3.55554675e+00,
       -5.66781088e-01, -6.55014996e-01, -5.31927008e-01, -7.01311242e-01,
       -6.33272798e-01, -5.28463263e-01, -5.66781088e-01,  3.47880521e+00,
       -6.03267770e-01, -5.45879625e-01, -6.01140663e-01, -2.74297100e-01,
       -1.07575485e+00, -6.55014996e-01,  3.62063144e+00, -6.76802316e-01,
       -5.25221237e-01, -4.13540938e-01, -5.40687327e-01, -5.68857460e-01,
        3.49150400e+00, -6.26475997e-01, -6.49992049e-01, -5.55371579e-01,
       -5.50119588e-01, -5.47155557e-01, -5.68166285e-01, -6.26475997e-01,
        3.47282427e+00, -5.89153572e-01, -5.61903450e-01, -5.79969407e-01,
       -4.88721410e-01, -6.11630957e-01, -6.49992049e-01,  3.47160509e+00,
       -6.05912050e-01, -5.56729775e-01, -5.58618852e-01, -5.71169904e-01,
       -6.62678879e-01, -5.89153572e-01,  3.51137330e+00, -3.82366056e-01,
       -5.52358186e-01, -7.53646708e-01, -5.61314596e-01, -8.20880748e-01,
       -3.82366056e-01,  3.61440694e+00, -3.30544050e-01, -3.48944833e-01,
       -1.28019110e-01, -1.04233755e+00, -4.90416508e-01, -6.76699718e-01,
       -3.30544050e-01,  3.40757671e+00, -5.61366319e-01, -9.34488397e-01,
       -4.14061715e-01, -5.54675223e-01, -5.54416570e-01, -6.05912050e-01,
        3.46614488e+00, -6.09268507e-01, -5.46394056e-01, -5.95478476e-01,
       -6.00306492e-01, -5.01877991e-01, -6.09268507e-01,  3.47109357e+00,
       -6.17054833e-01, -5.12694035e-01, -6.29891708e-01, -6.09586991e-01,
       -5.35822200e-01, -5.61366319e-01,  3.54885775e+00, -6.13088878e-01,
       -7.10310592e-01, -5.18682772e-01, -6.83327526e-01, -4.78617791e-01,
       -6.13088878e-01,  3.58265789e+00, -5.88816784e-01, -6.53578487e-01,
       -5.65228424e-01, -4.83060676e-01, -6.38907294e-01, -6.17054833e-01,
        3.47800638e+00, -6.04981244e-01, -4.78824193e-01, -6.55178141e-01,
       -6.53350769e-01, -4.93302466e-01, -6.04981244e-01,  3.47876585e+00,
       -6.01413727e-01, -4.65910921e-01, -6.59806720e-01, -7.48099078e-01,
       -3.41832219e-01, -6.76802316e-01,  3.50559356e+00, -6.88645309e-01,
       -6.00247032e-01, -4.49967605e-01, -4.15375895e-01, -5.28816118e-01,
       -5.88816784e-01, -6.88645309e-01,  3.31927369e+00, -5.21372655e-01,
       -5.76246931e-01, -6.35889679e-01, -4.90621497e-01, -6.03267770e-01,
        3.41543265e+00, -5.50730596e-01, -5.22641618e-01, -6.12281488e-01,
       -6.52069685e-01, -5.17769308e-01, -6.01413727e-01,  3.47823999e+00,
       -5.51061154e-01, -4.76022969e-01, -6.79903143e-01, -2.74500755e-01,
       -8.52693311e-01, -6.72367506e-01,  3.56810432e+00, -5.69496978e-01,
       -6.32754342e-01, -5.66291425e-01, -6.04087235e-01, -6.59420378e-01,
       -5.51615322e-01,  3.51967474e+00, -5.32046094e-01, -4.75058951e-01,
       -6.97446760e-01, -5.21192181e-01, -6.81150401e-01, -5.32046094e-01,
        3.48336695e+00, -4.96549397e-01, -5.54666972e-01, -6.97761906e-01,
       -8.86101671e-01, -3.96059686e-01, -4.96549397e-01,  3.53383623e+00,
       -5.78852512e-01, -5.25527531e-01, -6.50745433e-01, -3.98838602e-02,
       -1.21615224e+00, -5.18510213e-01, -9.05907767e-01,  3.85095217e+00,
       -4.66591700e-01, -7.03906385e-01, -5.64136917e-01, -6.08739713e-01,
        3.44287348e+00, -5.63175166e-01, -5.59918557e-01, -5.64821514e-01,
       -5.82081615e-01, -6.17013171e-01, -5.65481035e-01, -5.63175166e-01,
        3.48181273e+00, -5.70385926e-01, -5.80979825e-01, -5.84777611e-01,
       -5.73801339e-01, -6.07732616e-01, -5.59918557e-01,  3.40957111e+00,
       -5.31380374e-01, -5.99518976e-01, -5.37219245e-01, -5.76608056e-01,
       -6.47368032e-01, -5.31380374e-01,  3.48413229e+00, -5.38001505e-01,
       -6.50727357e-01, -5.40046965e-01, -5.41923844e-01, -6.54380298e-01,
       -5.38001505e-01,  3.47476010e+00, -5.70844800e-01, -6.67518746e-01,
       -5.02090909e-01, -6.18731971e-01, -5.15885123e-01, -5.70385926e-01,
        3.47995783e+00, -6.04536140e-01, -5.66308026e-01, -6.04110645e-01,
       -5.12312553e-01, -5.98406501e-01, -5.70844800e-01,  3.43357375e+00,
       -6.31695753e-01, -6.45534425e-01, -4.74779714e-01, -6.11283255e-01,
       -5.22247426e-01, -6.04536140e-01,  3.48688608e+00, -5.94727740e-01,
       -5.08830898e-01, -6.45260620e-01, -6.07102772e-01, -5.37672303e-01,
       -5.94727740e-01,  3.47341478e+00, -5.96042275e-01, -5.21170345e-01,
       -6.16699346e-01, -5.19369761e-01, -6.23876778e-01, -6.31695753e-01,
        3.47543484e+00, -6.15641606e-01, -5.97633444e-01, -4.87217499e-01,
       -5.49255117e-01, -5.78127697e-01, -5.96042275e-01,  3.46336474e+00,
       -6.12178461e-01, -5.73151777e-01, -5.54609409e-01, -4.95848972e-01,
       -6.29735989e-01, -6.15641606e-01,  3.44821547e+00, -5.94293954e-01,
       -6.16338653e-01, -4.96356300e-01, -5.03428747e-01, -6.40795841e-01,
       -5.94293954e-01,  3.45277258e+00, -5.81563916e-01, -6.29476740e-01,
       -5.03213382e-01, -5.59405655e-01, -5.78307589e-01, -5.98156134e-01,
        3.48352725e+00, -6.23865062e-01, -5.73165939e-01, -5.50626868e-01,
       -7.19765980e-01, -3.00136930e-01, -1.28816274e+00,  3.89839291e+00,
       -9.90863251e-02, -6.49668990e-01, -8.41571944e-01, -6.46333033e-01,
       -6.45307227e-01, -4.98014956e-01,  3.44498159e+00, -4.20665397e-01,
       -5.70756843e-01, -6.63904129e-01, -4.17028699e-01, -8.44711294e-01,
       -9.90863251e-02, -4.20665397e-01,  3.49957977e+00, -4.55123809e-01,
       -6.16008664e-01, -6.46955580e-01, -7.54373412e-01, -2.79655084e-02,
       -6.30034686e-01,  3.57962607e+00, -4.08101623e-01, -6.14656096e-01,
       -3.82652169e-01, -7.61842581e-01, -7.40741343e-01, -5.91642610e-01,
       -4.08101623e-01,  3.55705420e+00, -5.87879711e-01, -6.69553105e-01,
       -5.59135808e-01, -5.62438255e-01, -6.24510764e-01, -5.87879711e-01,
        3.51079277e+00, -5.62574648e-01, -5.23347313e-01, -6.50042078e-01,
       -3.98952491e-01, -4.27340956e-01, -6.14656096e-01,  3.51816861e+00,
       -8.33324217e-01, -6.74493769e-01, -1.60620566e-01, -4.08780511e-01,
       -6.48761357e-01, -3.74753056e-01, -8.33324217e-01,  4.00826758e+00,
       -5.42937218e-01, -1.60849173e+00, -6.05824482e-01, -5.97295378e-01,
       -5.62574648e-01,  3.48290169e+00, -5.34195798e-01, -5.37602608e-01,
       -6.45408779e-01, -6.02556231e-01, -6.26400442e-01, -5.34195798e-01,
        3.46867659e+00, -5.28454101e-01, -6.01221146e-01, -5.75848871e-01,
       -7.29053022e-01, -4.64365843e-01, -6.49668990e-01,  3.59157930e+00,
       -6.73762689e-01, -4.54202733e-01, -6.20526020e-01, -7.21569823e-01,
       -5.50730596e-01, -4.66591700e-01,  3.48321758e+00, -5.68575833e-01,
       -5.28276902e-01, -6.47472724e-01, -6.71121755e-01, -5.51061154e-01,
       -5.78852512e-01,  3.48440026e+00, -5.16766659e-01, -4.55987109e-01,
       -7.10611070e-01, -4.70817354e-01, -7.53129780e-01, -3.93499597e-01,
        3.46519448e+00, -5.12858039e-01, -5.91557419e-01, -7.43332294e-01,
       -7.16300671e-01, -4.09375078e-01, -5.12858039e-01,  3.46166782e+00,
       -6.48564362e-01, -4.85145373e-01, -6.89424296e-01, -6.63764415e-01,
       -3.45861053e-01, -6.48564362e-01,  3.46058855e+00, -7.34829960e-01,
       -4.89532404e-01, -5.78036360e-01, -4.55230143e-01, -5.69496978e-01,
       -7.34829960e-01,  3.45211156e+00, -5.84002935e-01, -5.44449428e-01,
       -5.64102115e-01, -7.04705279e-01, -4.75058951e-01, -5.91557419e-01,
        3.52875136e+00, -5.15581535e-01, -4.39261367e-01, -8.02586805e-01,
       -6.23381406e-01, -5.10707392e-01,  3.47002033e+00, -5.56366203e-01,
       -6.13140717e-01, -5.95359716e-01, -5.71064898e-01, -4.92829393e-01,
       -6.32754342e-01, -5.56366203e-01,  3.48546202e+00, -6.03284499e-01,
       -4.98530313e-01, -7.01697270e-01, -5.76006382e-01, -5.08725522e-01,
       -6.13140717e-01,  3.46172729e+00, -6.26164181e-01, -5.65643196e-01,
       -5.72047287e-01, -4.53353000e-01, -7.03906385e-01, -6.12178461e-01,
       -5.68575833e-01,  3.48016813e+00, -6.13935535e-01, -5.28218917e-01,
       -6.31203687e-01, -4.51697673e-01,  3.49986499e+00, -7.15755620e-01,
       -6.60997779e-01, -6.58935500e-01, -3.81274729e-01, -4.39061815e-01,
       -5.92283478e-01, -7.15755620e-01,  3.51253861e+00, -7.57451079e-01,
       -6.36374419e-01, -3.71612201e-01, -6.33230649e-01, -4.77721903e-01,
       -6.60997779e-01,  3.48415739e+00, -6.59331210e-01, -4.22775705e-01,
       -6.30100149e-01, -6.17199557e-01, -5.27775527e-01, -6.59331210e-01,
        3.45698987e+00, -6.04054548e-01, -4.73434055e-01, -5.75194968e-01,
       -5.95017776e-01, -5.72725524e-01, -6.04054548e-01,  3.45285241e+00,
       -5.81484802e-01, -5.56613400e-01, -5.42956357e-01, -4.62728619e-01,
       -5.63048157e-01, -7.57451079e-01,  3.50056738e+00, -7.15094541e-01,
       -5.76273875e-01, -4.25971111e-01, -5.85250011e-01, -5.73482369e-01,
       -5.81484802e-01,  3.46720139e+00, -6.03021857e-01, -5.94608035e-01,
       -5.29354310e-01, -5.67796613e-01, -5.60318302e-01, -6.23865062e-01,
       -6.03021857e-01,  3.48722656e+00, -5.36980931e-01, -5.95243799e-01,
       -4.94733631e-01, -5.58329547e-01, -7.15094541e-01,  3.46333440e+00,
       -6.64714357e-01, -5.40630709e-01, -4.89831614e-01, -5.05806683e-01,
       -5.81563916e-01, -6.64714357e-01,  3.46876996e+00, -6.39064955e-01,
       -5.71442304e-01, -5.06177748e-01, -5.87452078e-01, -5.68226806e-01,
        3.46790058e+00, -5.69218564e-01, -5.86760137e-01, -5.92306200e-01,
       -5.63936797e-01, -5.63698174e-01, -6.07150969e-01, -5.69218564e-01,
        3.47419250e+00, -5.45474981e-01, -5.92043160e-01, -5.96606651e-01,
       -5.78048542e-01, -6.26943913e-01, -5.45474981e-01,  3.47430881e+00,
       -5.21007882e-01, -5.92827950e-01, -6.10005546e-01, -5.69935222e-01,
       -5.73165939e-01, -5.86760137e-01,  3.45578892e+00, -5.91516141e-01,
       -5.35597650e-01, -5.98813834e-01, -6.50637988e-01, -5.70051051e-01,
       -5.21007882e-01,  3.47826758e+00, -5.52912531e-01, -5.66280326e-01,
       -6.17377801e-01, -7.79310336e-01, -5.70756843e-01, -5.42937218e-01,
        3.52814338e+00, -4.70796921e-01, -4.67948878e-01, -6.96393184e-01,
       -5.75424040e-01, -5.28454101e-01, -6.26164181e-01,  3.47757844e+00,
       -6.73102730e-01, -6.05305170e-01, -4.69128213e-01, -5.53392647e-01,
       -6.73762689e-01, -5.52912531e-01,  3.51388571e+00, -4.93660683e-01,
       -5.91058945e-01, -6.49098214e-01, -5.31927008e-01, -5.25221237e-01,
        3.33665453e+00, -5.45479315e-01, -6.75990635e-01, -5.61317806e-01,
       -4.96718530e-01, -7.01311242e-01, -5.45879625e-01, -5.45479315e-01,
        3.53647522e+00, -5.60219434e-01, -5.88162341e-01, -5.95423259e-01,
       -6.01140663e-01, -5.22641618e-01, -5.60219434e-01,  3.40993257e+00,
       -6.08533175e-01, -5.43107002e-01, -5.74290676e-01, -4.13540938e-01,
       -6.00247032e-01, -6.75990635e-01,  3.46198409e+00, -6.74248025e-01,
       -6.24177213e-01, -4.73780243e-01, -5.55371579e-01, -5.61903450e-01,
        3.46530975e+00, -6.16382803e-01, -5.93610971e-01, -5.73421255e-01,
       -5.64619691e-01, -5.50119588e-01, -5.56729775e-01, -6.16382803e-01,
        3.46653434e+00, -6.20940177e-01, -5.46399202e-01, -5.75962792e-01,
       -5.79969407e-01, -5.52358186e-01, -5.93610971e-01,  3.46489220e+00,
       -5.85503443e-01, -5.68701981e-01, -5.84748213e-01, -5.58618852e-01,
       -5.46394056e-01, -6.20940177e-01,  3.46702244e+00, -6.14467720e-01,
       -5.35381416e-01, -5.91220214e-01, -7.53646708e-01, -3.48944833e-01,
       -5.85503443e-01,  3.53277408e+00, -7.28503094e-01, -5.50224165e-01,
       -5.65951838e-01, -1.28019110e-01, -9.34488397e-01,  3.67560317e+00,
       -6.07646632e-01, -9.21172817e-01, -6.24813387e-01, -4.59462828e-01,
       -4.14061715e-01, -7.10310592e-01, -6.07646632e-01,  3.46628599e+00,
       -5.49106855e-01, -7.09930140e-01, -4.75230058e-01, -5.95478476e-01,
       -5.12694035e-01, -6.14467720e-01,  3.47002376e+00, -6.13191394e-01,
       -5.12684090e-01, -6.21508050e-01, -6.29891708e-01, -4.78824193e-01,
       -6.13191394e-01,  3.47702554e+00, -6.20338989e-01, -4.86226151e-01,
       -6.48553105e-01, -5.18682772e-01, -6.53578487e-01, -5.49106855e-01,
        3.48341239e+00, -5.10504902e-01, -7.33677675e-01, -5.17861695e-01,
       -5.65228424e-01, -5.21372655e-01, -5.10504902e-01,  3.44266236e+00,
       -6.78468136e-01, -6.11563775e-01, -5.55524466e-01, -6.55178141e-01,
       -4.65910921e-01, -6.20338989e-01,  3.48482998e+00, -6.14288469e-01,
       -4.56927748e-01, -6.72185717e-01, -6.59806720e-01, -4.76022969e-01,
       -6.14288469e-01,  3.48703720e+00, -5.85198915e-01, -4.53834344e-01,
       -6.97885787e-01, -4.49967605e-01, -5.76246931e-01, -6.74248025e-01,
       -6.78468136e-01,  3.49219439e+00, -6.43880884e-01, -4.69382806e-01,
       -6.12281488e-01, -5.28276902e-01, -6.08533175e-01,  3.45917435e+00,
       -5.75608559e-01, -5.37230572e-01, -5.97243655e-01, -6.79903143e-01,
       -5.16766659e-01, -5.85198915e-01,  3.48746418e+00, -5.13029407e-01,
       -4.77214554e-01, -7.15351497e-01, -5.66291425e-01, -5.84002935e-01,
       -6.03284499e-01,  3.46299014e+00, -5.89583667e-01, -4.67323710e-01,
       -6.52503907e-01, -6.97446760e-01, -5.54666972e-01, -5.15581535e-01,
        3.53846348e+00, -4.67124129e-01, -5.15440192e-01, -7.88203895e-01,
       -6.97761906e-01, -5.25527531e-01, -4.67124129e-01,  3.42548387e+00,
       -5.67730168e-01, -5.41005652e-01, -6.26334488e-01, -6.50745433e-01,
       -4.55987109e-01, -5.67730168e-01,  3.42081593e+00, -5.38261370e-01,
       -5.47659204e-01, -6.60432651e-01, -5.64821514e-01, -5.99518976e-01,
        3.45324071e+00, -5.80887090e-01, -5.89454043e-01, -5.66375030e-01,
       -5.52184061e-01, -5.82081615e-01, -5.80979825e-01, -5.80887090e-01,
        3.43247363e+00, -5.77779081e-01, -5.74673835e-01, -5.36072182e-01,
       -5.84777611e-01, -5.66308026e-01, -5.77779081e-01,  3.48209804e+00,
       -6.09087939e-01, -6.03484579e-01, -5.40660806e-01, -5.37219245e-01,
       -6.50727357e-01, -5.89454043e-01,  3.47773131e+00, -5.46219086e-01,
       -6.12270421e-01, -5.41841162e-01, -5.40046965e-01, -6.67518746e-01,
       -5.46219086e-01,  3.47990479e+00, -5.53383082e-01, -6.59945414e-01,
       -5.12791501e-01, -5.02090909e-01, -6.45534425e-01, -5.53383082e-01,
        3.59386323e+00, -8.03054868e-01, -8.57812701e-01, -2.31987247e-01,
       -6.04110645e-01, -5.08830898e-01, -6.09087939e-01,  3.46931020e+00,
       -6.26647324e-01, -5.26754027e-01, -5.93879363e-01, -4.74779714e-01,
       -5.97633444e-01, -8.03054868e-01,  3.50185288e+00, -7.00619194e-01,
       -3.89737215e-01, -5.36028445e-01, -6.45260620e-01, -5.21170345e-01,
       -6.26647324e-01,  3.48067653e+00, -5.45166467e-01, -4.18195620e-01,
       -7.24236157e-01, -6.16699346e-01, -5.73151777e-01, -5.45166467e-01,
        3.47571687e+00, -5.75032418e-01, -5.63692791e-01, -6.01974070e-01,
       -4.87217499e-01, -6.16338653e-01, -7.00619194e-01,  3.47841126e+00,
       -6.34872329e-01, -5.39000126e-01, -5.00363463e-01, -5.54609409e-01,
       -6.13935535e-01, -5.75032418e-01,  3.47282268e+00, -5.65109534e-01,
       -5.92142664e-01, -5.71993124e-01, -4.96356300e-01, -6.29476740e-01,
       -6.34872329e-01,  3.46362586e+00, -6.15001031e-01, -5.83885559e-01,
       -5.04033898e-01, -5.03213382e-01, -6.39064955e-01, -6.15001031e-01,
        3.46547135e+00, -6.02573036e-01, -5.95491287e-01, -5.10127661e-01,
       -5.50626868e-01, -5.36980931e-01, -5.91516141e-01,  3.39267056e+00,
       -5.93869612e-01, -5.38627944e-01, -5.81049065e-01, -8.41571944e-01,
       -4.55123809e-01, -4.54202733e-01,  3.53433247e+00, -6.63472212e-01,
       -4.97111109e-01, -6.22850667e-01, -6.63904129e-01, -6.16008664e-01,
       -4.70796921e-01,  3.46621754e+00, -5.33138135e-01, -6.36891058e-01,
       -5.45478637e-01, -3.82652169e-01, -6.74493769e-01,  3.49852141e+00,
       -7.23578634e-01, -6.26425066e-01, -5.97754414e-01, -4.93617354e-01,
       -7.61842581e-01, -6.69553105e-01, -7.23578634e-01,  3.41055682e+00,
       -5.92459437e-01, -6.63123066e-01, -5.59135808e-01, -5.23347313e-01,
       -5.92459437e-01,  3.42771626e+00, -5.71058527e-01, -4.85674977e-01,
       -6.96040199e-01, -6.50042078e-01, -5.37602608e-01, -5.71058527e-01,
        3.48718993e+00, -5.24859626e-01, -4.92938067e-01, -7.10689024e-01,
       -1.60620566e-01, -1.60849173e+00, -4.67948878e-01,  3.94263354e+00,
       -4.43581099e-01, -7.38790392e-01, -5.23200872e-01, -6.45408779e-01,
       -6.01221146e-01, -5.24859626e-01,  3.47583049e+00, -5.18088927e-01,
       -5.61243740e-01, -6.25008275e-01, -5.75848871e-01, -6.73102730e-01,
       -5.18088927e-01,  3.47375844e+00, -5.32972611e-01, -6.36342425e-01,
       -5.37402871e-01, -6.20526020e-01, -4.93660683e-01, -6.63472212e-01,
        3.51101997e+00, -5.74584008e-01, -4.51459927e-01, -7.07317122e-01,
       -1.04233755e+00, -7.28503094e-01, -9.21172817e-01,  3.75088530e+00,
       -5.59981325e-01, -4.98890516e-01, -6.47472724e-01, -5.28218917e-01,
       -5.75608559e-01, -5.65109534e-01,  3.45616194e+00, -5.37889752e-01,
       -6.01862453e-01, -7.10611070e-01, -5.13029407e-01, -5.38261370e-01,
        3.45920425e+00, -4.72978841e-01, -5.21482130e-01, -7.02841436e-01,
       -7.43332294e-01, -4.85145373e-01, -4.39261367e-01,  3.30127018e+00,
       -6.80576610e-01, -4.44184887e-01, -5.08769645e-01, -6.89424296e-01,
       -4.89532404e-01, -6.80576610e-01,  3.41645121e+00, -5.63029907e-01,
       -2.66872124e-01, -7.27015871e-01, -5.78036360e-01, -5.44449428e-01,
       -5.63029907e-01,  3.41659444e+00, -6.16350269e-01, -5.97204800e-01,
       -5.17523681e-01, -5.64102115e-01, -5.89583667e-01, -6.16350269e-01,
        3.43085522e+00, -5.34714949e-01, -6.03084588e-01, -5.23019630e-01,
       -8.02586805e-01, -5.15440192e-01, -4.44184887e-01,  3.53918982e+00,
       -3.98507388e-01, -6.61293989e-01, -7.17176558e-01, -5.95359716e-01,
       -4.98530313e-01,  3.42482791e+00, -5.61675235e-01, -6.34089271e-01,
       -3.15904151e-01, -8.19269221e-01, -5.71064898e-01, -5.65643196e-01,
       -5.61675235e-01,  3.46040796e+00, -5.40083288e-01, -6.08970518e-01,
       -6.12970821e-01, -7.01697270e-01, -4.67323710e-01, -6.34089271e-01,
        3.50857092e+00, -6.44904427e-01, -3.89223078e-01, -6.71333168e-01,
       -5.72047287e-01, -6.05305170e-01, -5.40083288e-01,  3.51658238e+00,
       -5.30419070e-01, -6.58621214e-01, -6.10106355e-01, -6.58935500e-01,
       -4.22775705e-01,  3.50343990e+00, -7.00385297e-01, -6.88509441e-01,
       -6.38352784e-01, -3.94481177e-01, -3.81274729e-01, -6.36374419e-01,
       -7.00385297e-01,  3.63741893e+00, -9.23632725e-01, -8.09362001e-01,
       -1.86389762e-01, -3.71612201e-01, -5.76273875e-01, -9.23632725e-01,
        3.77696026e+00, -1.01340906e+00, -8.92032403e-01, -6.30100149e-01,
       -4.73434055e-01, -6.88509441e-01,  3.48638510e+00, -6.66743390e-01,
       -4.29771262e-01, -5.97826807e-01, -5.75194968e-01, -5.56613400e-01,
       -6.66743390e-01,  3.46168652e+00, -6.48229440e-01, -4.83732275e-01,
       -5.31173048e-01, -5.42956357e-01, -5.94608035e-01, -6.48229440e-01,
        3.46484387e+00, -6.15354553e-01, -5.53092675e-01, -5.10602812e-01,
       -4.25971111e-01, -5.40630709e-01, -1.01340906e+00,  3.63896595e+00,
       -7.16937920e-01, -2.46360123e-01, -6.95657023e-01, -5.29354310e-01,
       -5.95243799e-01, -5.93869612e-01, -6.15354553e-01,  3.46468884e+00,
       -5.45238099e-01, -5.85628463e-01, -4.89831614e-01, -5.71442304e-01,
       -7.16937920e-01,  3.47670363e+00, -6.45401082e-01, -5.22565402e-01,
       -5.30525307e-01, -5.06177748e-01, -6.02573036e-01, -6.45401082e-01,
        3.46370463e+00, -6.07385062e-01, -5.78551541e-01, -5.23616162e-01,
       -5.92306200e-01, -5.35597650e-01,  3.47699750e+00, -5.97657192e-01,
       -6.24463394e-01, -5.34269478e-01, -5.92703583e-01, -5.63936797e-01,
       -5.92043160e-01, -5.97657192e-01,  3.47524431e+00, -5.46404015e-01,
       -5.74533172e-01, -6.00669973e-01, -5.96606651e-01, -5.92827950e-01,
       -5.46404015e-01,  3.46346177e+00, -5.42412747e-01, -5.91059838e-01,
       -5.94150567e-01, -6.10005546e-01, -5.66280326e-01, -5.42412747e-01,
        3.43616512e+00, -5.31090282e-01, -5.85010779e-01, -6.01365438e-01,
       -5.98813834e-01, -5.38627944e-01, -6.24463394e-01,  3.47537754e+00,
       -6.11927743e-01, -5.25147932e-01, -5.76396689e-01, -6.17377801e-01,
       -5.91058945e-01, -5.31090282e-01,  3.47365548e+00, -5.03036019e-01,
       -5.91436756e-01, -6.39655675e-01, -6.46955580e-01, -4.97111109e-01,
       -5.33138135e-01,  3.61507814e+00, -9.12915481e-01, -8.00392195e-01,
       -2.24565643e-01, -6.96393184e-01, -6.36891058e-01, -4.43581099e-01,
        3.47217660e+00, -4.72444281e-01, -6.87622617e-01, -5.35244359e-01,
       -4.08780511e-01, -6.26425066e-01, -7.38790392e-01,  3.47730458e+00,
       -5.71449760e-01, -5.42413807e-01, -5.89445041e-01, -4.69128213e-01,
       -5.32972611e-01, -5.30419070e-01,  3.41212173e+00, -5.81699360e-01,
       -7.42483945e-01, -5.55418534e-01, -6.49098214e-01, -5.74584008e-01,
       -5.03036019e-01,  3.47284726e+00, -4.71201504e-01, -5.81437913e-01,
       -6.93489604e-01, -5.61317806e-01, -5.88162341e-01,  3.48525372e+00,
       -5.82824245e-01, -5.99142228e-01, -5.82136820e-01, -5.71670281e-01,
       -4.96718530e-01, -6.24177213e-01, -5.82824245e-01,  3.47594371e+00,
       -6.07068255e-01, -6.27581740e-01, -5.37573731e-01, -5.95423259e-01,
       -5.43107002e-01, -5.99142228e-01,  3.47419009e+00, -6.15680630e-01,
       -5.47805920e-01, -5.73031054e-01, -5.74290676e-01, -5.37230572e-01,
       -6.15680630e-01,  3.42545116e+00, -5.95846808e-01, -5.39193538e-01,
       -5.63208938e-01, -4.73780243e-01, -6.43880884e-01, -6.07068255e-01,
        3.50944742e+00, -6.42854012e-01, -6.35345065e-01, -5.06518964e-01,
       -5.73421255e-01, -5.46399202e-01,  3.46541756e+00, -5.94106794e-01,
       -6.09787492e-01, -5.54212853e-01, -5.87489968e-01, -5.64619691e-01,
       -5.68701981e-01, -5.94106794e-01,  3.46449668e+00, -5.88401726e-01,
       -5.81357414e-01, -5.67309072e-01, -5.75962792e-01, -5.35381416e-01,
       -6.09787492e-01,  3.46653587e+00, -6.12736431e-01, -5.41733392e-01,
       -5.90934344e-01, -5.84748213e-01, -5.50224165e-01, -5.88401726e-01,
        3.46513902e+00, -6.05303411e-01, -5.62504139e-01, -5.73957370e-01,
       -5.91220214e-01, -5.12684090e-01, -6.12736431e-01,  3.46886737e+00,
       -6.14501052e-01, -5.24658616e-01, -6.13066965e-01, -5.65951838e-01,
       -5.59981325e-01, -6.05303411e-01,  3.46754913e+00, -6.21513413e-01,
       -5.35036067e-01, -5.79763075e-01, -6.24813387e-01, -4.98890516e-01,
       -6.21513413e-01,  3.44739367e+00, -6.04259453e-01, -5.51748424e-01,
       -5.46168474e-01, -4.59462828e-01, -7.09930140e-01, -6.04259453e-01,
        3.46167288e+00, -5.40683672e-01, -6.42886335e-01, -5.04450454e-01,
       -4.75230058e-01, -7.33677675e-01, -5.40683672e-01,  3.51364398e+00,
       -5.18665963e-01, -7.41588221e-01, -5.03798389e-01, -6.21508050e-01,
       -4.86226151e-01, -6.14501052e-01,  3.47432326e+00, -6.16907835e-01,
       -4.97485316e-01, -6.37694850e-01, -6.48553105e-01, -4.56927748e-01,
       -6.16907835e-01,  3.48530159e+00, -6.29229236e-01, -4.65597065e-01,
       -6.68086607e-01, -5.17861695e-01, -6.11563775e-01, -5.18665963e-01,
        3.45555589e+00, -6.12387993e-01, -6.57832397e-01, -5.37244069e-01,
       -5.55524466e-01, -4.69382806e-01, -6.42854012e-01, -6.12387993e-01,
        3.42315012e+00, -6.22596113e-01, -5.20404726e-01, -6.72185717e-01,
       -4.53834344e-01, -6.29229236e-01,  3.49950701e+00, -5.66281563e-01,
       -4.55651271e-01, -7.22324883e-01, -6.97885787e-01, -4.77214554e-01,
       -5.66281563e-01,  3.51313620e+00, -5.34382369e-01, -4.74290024e-01,
       -7.63081904e-01, -5.97243655e-01, -5.37889752e-01, -5.95846808e-01,
        3.45513110e+00, -5.83930386e-01, -5.62318724e-01, -5.77901771e-01,
       -7.15351497e-01, -4.72978841e-01, -5.34382369e-01,  3.47649702e+00,
       -5.98477342e-01, -4.96223653e-01, -6.59083317e-01, -6.52503907e-01,
       -5.34714949e-01, -6.44904427e-01,  3.48096201e+00, -5.82070013e-01,
       -4.56186404e-01, -6.10582311e-01, -7.88203895e-01, -5.41005652e-01,
       -3.98507388e-01,  3.52730227e+00, -4.22683719e-01, -7.37749408e-01,
       -6.39152212e-01, -6.26334488e-01, -5.47659204e-01, -4.22683719e-01,
        3.69200581e+00, -6.05055615e-01, -1.17169181e+00, -3.18580975e-01,
       -6.60432651e-01, -5.21482130e-01, -6.05055615e-01,  3.46874338e+00,
       -4.86584948e-01, -4.83955288e-01, -7.11232748e-01, -5.66375030e-01,
       -5.74673835e-01,  3.47175460e+00, -6.08518358e-01, -6.18790799e-01,
       -5.28048252e-01, -5.75348321e-01, -5.52184061e-01, -6.12270421e-01,
       -6.08518358e-01,  3.46224538e+00, -5.58836853e-01, -5.69259579e-01,
       -5.61176111e-01, -5.36072182e-01, -6.03484579e-01, -6.18790799e-01,
        3.48316085e+00, -6.23207210e-01, -6.05237226e-01, -4.96368858e-01,
       -5.40660806e-01, -5.26754027e-01, -6.23207210e-01,  3.52145246e+00,
       -7.88653471e-01, -6.80258338e-01, -3.61918604e-01, -5.41841162e-01,
       -6.59945414e-01, -5.58836853e-01,  3.47570943e+00, -5.15747327e-01,
       -6.38346218e-01, -5.60992457e-01, -5.12791501e-01, -8.57812701e-01,
       -5.15747327e-01,  3.74037028e+00, -9.88432946e-01, -8.65585803e-01,
       -2.31987247e-01, -3.89737215e-01, -9.88432946e-01,  3.59026705e+00,
       -7.34927624e-01, -2.14231687e-01, -6.80038666e-01, -3.50911665e-01,
       -5.93879363e-01, -4.18195620e-01, -7.88653471e-01,  3.85048601e+00,
       -5.28689735e-01, -1.37983480e-01, -1.38308434e+00, -5.36028445e-01,
       -5.39000126e-01, -7.34927624e-01,  3.48191552e+00, -6.76996188e-01,
       -4.31110368e-01, -5.63852765e-01, -7.24236157e-01, -5.63692791e-01,
       -5.28689735e-01,  3.50233959e+00, -5.06574018e-01, -5.09095948e-01,
       -6.70050939e-01, -6.01974070e-01, -5.92142664e-01, -5.06574018e-01,
        3.44333241e+00, -5.52871472e-01, -6.04377563e-01, -5.85392625e-01,
       -5.00363463e-01, -5.83885559e-01, -6.76996188e-01,  3.48236076e+00,
       -6.55619587e-01, -5.44842403e-01, -5.20653564e-01, -5.71993124e-01,
       -6.01862453e-01, -5.83930386e-01, -5.52871472e-01,  3.49356121e+00,
       -5.99365965e-01, -5.83537810e-01, -5.04033898e-01, -5.95491287e-01,
       -6.55619587e-01,  3.47238991e+00, -6.62007270e-01, -5.48904269e-01,
       -5.06333603e-01, -5.10127661e-01, -6.07385062e-01, -6.62007270e-01,
        3.46613592e+00, -6.06509271e-01, -5.41087712e-01, -5.39018948e-01,
       -5.81049065e-01, -5.45238099e-01, -6.11927743e-01,  3.46456408e+00,
       -6.25297960e-01, -5.34545622e-01, -5.66505592e-01, -6.22850667e-01,
       -4.51459927e-01, -9.12915481e-01,  3.66005213e+00, -6.91826638e-01,
       -9.80999414e-01, -5.45478637e-01, -8.00392195e-01, -4.72444281e-01,
        3.51582226e+00, -4.15060393e-01, -7.71129093e-01, -5.11317660e-01,
       -5.97754414e-01, -5.71449760e-01,  3.42332432e+00, -5.79582723e-01,
       -5.38031570e-01, -5.10614681e-01, -6.25891170e-01, -4.93617354e-01,
       -6.63123066e-01, -4.85674977e-01, -5.79582723e-01,  3.41540238e+00,
       -5.12344988e-01, -6.81059271e-01, -6.96040199e-01, -4.92938067e-01,
       -5.12344988e-01,  3.42242215e+00, -4.98455601e-01, -5.36756402e-01,
       -6.85886892e-01, -7.10689024e-01, -5.61243740e-01, -4.98455601e-01,
        3.48128404e+00, -4.85945544e-01, -5.48466614e-01, -6.76483520e-01,
       -5.23200872e-01, -6.87622617e-01, -5.42413807e-01,  3.40233549e+00,
       -5.21826726e-01, -6.03764338e-01, -5.23507133e-01, -6.25008275e-01,
       -6.36342425e-01, -4.85945544e-01,  3.46967109e+00, -5.41606146e-01,
       -6.08196997e-01, -5.72571706e-01, -5.37402871e-01, -5.81699360e-01,
       -5.41606146e-01,  3.45749682e+00, -6.54452634e-01, -5.75426396e-01,
       -5.66909418e-01, -7.07317122e-01, -4.71201504e-01, -6.91826638e-01,
        3.55519437e+00, -3.35433438e-01, -5.02195226e-01, -8.47220442e-01,
       -7.02841436e-01, -5.98477342e-01, -4.86584948e-01,  3.46224151e+00,
       -4.70381267e-01, -5.95088164e-01, -6.08868347e-01, -5.08769645e-01,
       -2.66872124e-01, -6.61293989e-01,  3.36848896e+00, -9.60769306e-01,
       -4.52051064e-01, -1.64043035e-01, -3.54689795e-01, -7.27015871e-01,
       -5.97204800e-01, -9.60769306e-01,  3.65005219e+00, -5.75667007e-01,
       -7.89395202e-01, -5.17523681e-01, -6.03084588e-01, -5.75667007e-01,
        3.56120883e+00, -7.92311310e-01, -8.58012819e-01, -2.14609419e-01,
       -5.23019630e-01, -5.82070013e-01, -7.92311310e-01,  3.49834259e+00,
       -5.74360890e-01, -4.40932253e-01, -5.85648490e-01, -7.17176558e-01,
       -7.37749408e-01, -4.52051064e-01,  3.54586563e+00, -2.91232508e-01,
       -6.63749243e-01, -6.83906846e-01, -3.15904151e-01, -6.08970518e-01,
        3.37823757e+00, -7.28581967e-01, -5.94208398e-01, -4.36553301e-01,
       -6.94019237e-01, -8.19269221e-01, -3.89223078e-01, -7.28581967e-01,
        3.55928871e+00, -7.29569263e-01, -8.92645183e-01, -6.12970821e-01,
       -6.58621214e-01, -5.94208398e-01,  3.35441706e+00, -5.40719449e-01,
       -4.87286774e-01, -4.60610406e-01, -6.71333168e-01, -4.56186404e-01,
       -7.29569263e-01,  3.53977442e+00, -6.87705291e-01, -3.78798985e-01,
       -6.16181307e-01, -6.10106355e-01, -7.42483945e-01, -5.40719449e-01,
        3.57548804e+00, -5.30238856e-01, -6.49187348e-01, -5.02752091e-01,
       -6.38352784e-01, -4.29771262e-01,  3.49912115e+00, -6.72587862e-01,
       -7.20925956e-01, -4.38240987e-01, -5.99242302e-01, -3.94481177e-01,
       -8.09362001e-01, -6.72587862e-01,  3.52739484e+00, -4.73812978e-01,
       -6.70126631e-01, -5.07024195e-01, -1.86389762e-01, -8.92032403e-01,
       -2.46360123e-01, -4.73812978e-01,  3.52664543e+00, -4.30118760e-01,
       -6.93820097e-01, -6.04111305e-01, -5.97826807e-01, -4.83732275e-01,
       -7.20925956e-01,  3.53240408e+00, -7.61588512e-01, -3.41577718e-01,
       -6.26752818e-01, -5.31173048e-01, -5.53092675e-01, -7.61588512e-01,
        3.49407968e+00, -6.84985468e-01, -4.80795570e-01, -4.82444404e-01,
       -5.10602812e-01, -5.85628463e-01, -6.25297960e-01, -6.84985468e-01,
        3.44342227e+00, -5.05465600e-01, -5.31441961e-01, -6.95657023e-01,
       -5.22565402e-01, -4.30118760e-01,  3.49569412e+00, -6.41119254e-01,
       -6.21536534e-01, -5.84697144e-01, -5.30525307e-01, -5.78551541e-01,
       -6.41119254e-01,  3.46442970e+00, -6.27103900e-01, -5.56401318e-01,
       -5.30728378e-01, -5.23616162e-01, -6.06509271e-01, -6.27103900e-01,
        3.46071978e+00, -5.74184288e-01, -5.80917515e-01, -5.48388647e-01,
       -5.34269478e-01, -5.74533172e-01,  3.41314113e+00, -6.08576566e-01,
       -5.62483656e-01, -5.65957207e-01, -5.67321052e-01, -5.92703583e-01,
       -5.25147932e-01, -6.08576566e-01,  3.46213587e+00, -6.29871872e-01,
       -5.47937359e-01, -5.57898558e-01, -6.00669973e-01, -5.91059838e-01,
       -5.62483656e-01,  3.48564593e+00, -5.50239522e-01, -5.94846265e-01,
       -5.86346677e-01, -5.94150567e-01, -5.85010779e-01, -5.50239522e-01,
        3.44556355e+00, -5.37981431e-01, -5.89752340e-01, -5.88428912e-01,
       -6.01365438e-01, -5.91436756e-01, -5.37981431e-01,  3.43612255e+00,
       -5.15898595e-01, -5.87174611e-01, -6.02265722e-01, -5.76396689e-01,
       -5.34545622e-01, -6.29871872e-01,  3.47237648e+00, -6.44415951e-01,
       -5.46131288e-01, -5.41015058e-01, -6.39655675e-01, -5.81437913e-01,
       -5.15898595e-01,  3.47099594e+00, -5.10234141e-01, -5.93899168e-01,
       -6.29870451e-01, -2.24565643e-01, -9.80999414e-01, -4.15060393e-01,
       -3.35433438e-01,  3.67190934e+00, -7.88682034e-01, -3.33125579e-01,
       -5.94042843e-01, -5.35244359e-01, -7.71129093e-01, -5.21826726e-01,
        3.50708670e+00, -4.19324415e-01, -6.90438676e-01, -5.69123432e-01,
       -5.89445041e-01, -5.38031570e-01, -6.03764338e-01,  3.43996358e+00,
       -5.92852013e-01, -5.72462771e-01, -5.43407846e-01, -5.55418534e-01,
       -6.54452634e-01, -5.30238856e-01,  3.43871099e+00, -5.12844653e-01,
       -6.50452190e-01, -5.35304118e-01, -6.93489604e-01, -5.02195226e-01,
       -5.10234141e-01,  3.45998325e+00, -5.56981044e-01, -5.58279910e-01,
       -6.38803325e-01, -5.82136820e-01, -6.27581740e-01,  3.48191278e+00,
       -5.83008906e-01, -5.38908241e-01, -5.68979914e-01, -5.81297165e-01,
       -5.71670281e-01, -5.47805920e-01, -5.83008906e-01,  3.47356705e+00,
       -6.55441030e-01, -6.11594753e-01, -5.04046157e-01, -5.37573731e-01,
       -6.35345065e-01, -5.38908241e-01,  3.46503301e+00, -5.77194063e-01,
       -6.37416289e-01, -5.38595621e-01, -5.73031054e-01, -5.39193538e-01,
       -6.55441030e-01,  3.47355088e+00, -6.34668548e-01, -5.39171802e-01,
       -5.32044904e-01, -5.63208938e-01, -5.62318724e-01, -6.34668548e-01,
        3.43544110e+00, -5.87376698e-01, -5.16624045e-01, -5.71244146e-01,
       -5.06518964e-01, -6.22596113e-01, -5.77194063e-01,  3.47563934e+00,
       -6.20726359e-01, -6.61031362e-01, -4.87572479e-01, -5.54212853e-01,
       -5.81357414e-01,  3.46470725e+00, -5.89184084e-01, -5.86707401e-01,
       -5.91977343e-01, -5.61268153e-01, -5.87489968e-01, -5.41733392e-01,
       -5.89184084e-01,  3.46517029e+00, -5.96363450e-01, -5.58833545e-01,
       -5.91565848e-01, -5.67309072e-01, -5.62504139e-01, -5.86707401e-01,
        3.46512875e+00, -6.11638452e-01, -5.81055422e-01, -5.55914261e-01,
       -5.90934344e-01, -5.24658616e-01, -5.96363450e-01,  3.46638218e+00,
       -6.03389705e-01, -5.52850544e-01, -5.98185525e-01, -5.73957370e-01,
       -5.35036067e-01, -6.11638452e-01,  3.46968424e+00, -6.39023957e-01,
       -5.35129266e-01, -5.74899130e-01, -6.13066965e-01, -4.97485316e-01,
       -6.03389705e-01,  3.47025758e+00, -6.02008790e-01, -5.31876854e-01,
       -6.22429948e-01, -5.79763075e-01, -5.51748424e-01, -6.39023957e-01,
        3.47519204e+00, -6.23660140e-01, -5.01148679e-01, -5.79847762e-01,
       -5.46168474e-01, -6.42886335e-01, -6.23660140e-01,  3.45874048e+00,
       -5.49489667e-01, -5.94184444e-01, -5.02351415e-01, -5.04450454e-01,
       -7.41588221e-01, -5.49489667e-01,  3.50537069e+00, -5.09552491e-01,
       -7.30168829e-01, -4.70121025e-01, -5.03798389e-01, -6.57832397e-01,
       -5.09552491e-01,  3.46967328e+00, -5.71944394e-01, -6.81614411e-01,
       -5.44931195e-01, -6.37694850e-01, -4.65597065e-01, -6.02008790e-01,
        3.50205494e+00, -6.23866008e-01, -5.65924574e-01, -6.06963658e-01,
       -6.68086607e-01, -4.55651271e-01, -6.23866008e-01,  3.51438980e+00,
       -6.63532352e-01, -5.53586890e-01, -5.49666668e-01, -5.37244069e-01,
       -5.20404726e-01, -6.20726359e-01, -5.71944394e-01,  3.41099613e+00,
       -6.21947708e-01, -5.38728873e-01, -7.22324883e-01, -4.74290024e-01,
       -6.63532352e-01,  3.50530038e+00, -5.50407720e-01, -3.80376988e-01,
       -7.14368417e-01, -7.63081904e-01, -4.96223653e-01, -5.50407720e-01,
        3.54707276e+00, -4.89304250e-01, -4.53044983e-01, -7.95010247e-01,
       -5.77901771e-01, -5.99365965e-01, -5.87376698e-01,  3.48786946e+00,
       -5.49200471e-01, -5.68858318e-01, -6.05166238e-01, -6.59083317e-01,
       -4.70381267e-01, -4.89304250e-01,  3.47771198e+00, -6.11392555e-01,
       -5.81922149e-01, -6.65628442e-01, -6.10582311e-01, -5.74360890e-01,
       -6.87705291e-01,  3.51410986e+00, -5.92129054e-01, -3.99157995e-01,
       -6.50174315e-01, -6.39152212e-01, -1.17169181e+00, -2.91232508e-01,
        3.91114619e+00, -7.57964483e-01, -1.05110518e+00, -3.18580975e-01,
       -4.83955288e-01, -7.57964483e-01,  3.55379558e+00, -5.30326183e-01,
       -1.90407985e-01, -6.65594357e-01, -6.06966314e-01, -7.11232748e-01,
       -5.95088164e-01, -5.30326183e-01,  3.51123202e+00, -5.06349047e-01,
       -6.47930337e-01, -5.20305540e-01, -5.28048252e-01, -6.05237226e-01,
        3.48531567e+00, -6.01203609e-01, -6.35620145e-01, -4.81162320e-01,
       -6.34044118e-01, -5.75348321e-01, -5.69259579e-01, -6.01203609e-01,
        3.42950557e+00, -5.66865351e-01, -5.27941287e-01, -5.88887422e-01,
       -5.61176111e-01, -6.38346218e-01, -5.66865351e-01,  3.47178051e+00,
       -5.15118716e-01, -5.95776325e-01, -5.94497790e-01, -4.96368858e-01,
       -6.80258338e-01, -6.35620145e-01,  3.48419876e+00, -6.16706610e-01,
       -6.44254343e-01, -4.10990463e-01, -3.61918604e-01, -1.37983480e-01,
       -6.16706610e-01,  3.56522174e+00, -9.20428622e-01, -7.66420896e-01,
       -3.57908848e-01, -4.03854685e-01, -5.60992457e-01, -8.65585803e-01,
       -2.14231687e-01, -5.15118716e-01,  3.61392168e+00, -8.41016308e-01,
       -6.16976709e-01, -6.80038666e-01, -4.31110368e-01,  3.49746490e+00,
       -6.82131423e-01, -6.45631640e-01, -6.07846071e-01, -4.50706734e-01,
       -1.38308434e+00, -5.09095948e-01, -9.20428622e-01,  3.91622658e+00,
       -4.37059796e-01, -6.66557865e-01, -5.63852765e-01, -5.44842403e-01,
       -6.82131423e-01,  3.48173161e+00, -6.41922606e-01, -4.91677917e-01,
       -5.57304494e-01, -6.70050939e-01, -6.04377563e-01, -4.37059796e-01,
        3.48773337e+00, -5.29508128e-01, -6.52619789e-01, -5.94117157e-01,
       -5.85392625e-01, -5.83537810e-01, -5.49200471e-01, -5.29508128e-01,
        3.42541471e+00, -5.66531395e-01, -6.11244280e-01, -5.20653564e-01,
       -5.48904269e-01, -6.41922606e-01,  3.59376461e+00, -8.89672424e-01,
       -7.57983941e-01, -2.34627802e-01, -5.06333603e-01, -5.41087712e-01,
       -8.89672424e-01,  3.72225899e+00, -8.13498536e-01, -9.71666714e-01,
       -5.39018948e-01, -5.74184288e-01, -8.13498536e-01,  3.56012062e+00,
       -5.83604794e-01, -2.76407190e-01, -7.73406867e-01, -5.66505592e-01,
       -5.05465600e-01, -6.44415951e-01,  3.49111114e+00, -6.88944015e-01,
       -5.57067563e-01, -5.28712422e-01, -5.11317660e-01, -7.88682034e-01,
       -4.19324415e-01,  3.42526064e+00, -6.57248551e-01, -4.72619926e-01,
       -5.76068054e-01, -5.10614681e-01, -6.81059271e-01, -5.36756402e-01,
        3.42581538e+00, -5.41143460e-01, -5.07071417e-01, -6.49170148e-01,
       -6.25891170e-01, -5.92852013e-01, -5.41143460e-01,  3.42373784e+00,
       -5.30243440e-01, -5.80337565e-01, -5.53270194e-01, -6.85886892e-01,
       -5.48466614e-01, -5.07071417e-01,  3.46213782e+00, -5.27826884e-01,
       -5.52049612e-01, -6.40836400e-01, -6.76483520e-01, -6.08196997e-01,
       -5.27826884e-01,  3.48096289e+00, -4.90296235e-01, -5.12625672e-01,
       -6.65533577e-01, -5.23507133e-01, -6.90438676e-01, -5.72462771e-01,
        3.47478055e+00, -5.24515085e-01, -6.75638493e-01, -4.88218388e-01,
       -5.72571706e-01, -5.75426396e-01, -4.90296235e-01,  3.49542177e+00,
       -6.72865937e-01, -5.97416237e-01, -5.86845255e-01, -5.66909418e-01,
       -5.12844653e-01, -6.72865937e-01,  3.50667577e+00, -6.96342342e-01,
       -4.93937681e-01, -5.63775737e-01, -8.47220442e-01, -3.33125579e-01,
       -5.56981044e-01,  3.55068565e+00, -7.42469902e-01, -5.10888992e-01,
       -5.59999689e-01, -6.08868347e-01, -6.11392555e-01, -5.06349047e-01,
        3.52462413e+00, -5.36157518e-01, -7.78805995e-01, -4.83050663e-01,
       -1.64043035e-01, -7.89395202e-01, -8.58012819e-01,  3.70376341e+00,
       -8.97261513e-01, -9.95050838e-01, -2.14609419e-01, -4.40932253e-01,
       -8.97261513e-01,  3.52881237e+00, -7.15595830e-01, -2.27329311e-01,
       -5.01065523e-01, -5.32018526e-01, -5.85648490e-01, -5.92129054e-01,
       -7.15595830e-01,  3.49974526e+00, -4.82509011e-01, -4.89653497e-01,
       -6.34209376e-01, -3.54689795e-01, -6.63749243e-01, -9.95050838e-01,
       -2.27329311e-01,  3.43024268e+00, -4.29366380e-01, -7.60057114e-01,
       -6.83906846e-01, -1.05110518e+00, -1.90407985e-01, -4.29366380e-01,
        3.65393954e+00, -6.92259789e-01, -6.06893363e-01, -4.36553301e-01,
       -8.92645183e-01, -3.78798985e-01,  3.35935705e+00, -4.35684043e-01,
       -6.72537819e-01, -5.43137718e-01, -6.94019237e-01, -4.87286774e-01,
       -4.35684043e-01,  3.49324844e+00, -6.47287692e-01, -6.53770056e-01,
       -5.75200641e-01, -4.60610406e-01, -6.49187348e-01, -6.47287692e-01,
        3.48874862e+00, -6.18331884e-01, -5.92146390e-01, -5.21184896e-01,
       -6.16181307e-01, -3.99157995e-01, -6.72537819e-01,  3.70130747e+00,
       -8.07540431e-01, -5.60710465e-01, -6.45179455e-01, -5.02752091e-01,
       -6.50452190e-01, -6.18331884e-01,  3.48841697e+00, -5.48727868e-01,
       -5.79767273e-01, -5.88385668e-01, -3.50911665e-01, -8.41016308e-01,
       -6.45631640e-01,  3.56021545e+00, -5.29307077e-01, -6.62821297e-01,
       -5.30527463e-01, -4.38240987e-01, -6.70126631e-01,  3.48929801e+00,
       -6.88670646e-01, -6.17867954e-01, -5.88491286e-01, -4.85900511e-01,
       -5.99242302e-01, -3.41577718e-01, -6.88670646e-01,  3.63358380e+00,
       -1.08253028e+00, -4.72836253e-01, -4.48726600e-01, -5.07024195e-01,
       -6.93820097e-01, -6.17867954e-01,  3.47640050e+00, -4.97062683e-01,
       -6.28023568e-01, -5.32602001e-01, -6.04111305e-01, -6.21536534e-01,
       -4.97062683e-01,  3.46709373e+00, -5.58332752e-01, -6.24078880e-01,
       -5.61971572e-01, -6.26752818e-01, -4.80795570e-01, -1.08253028e+00,
        3.72055675e+00, -7.69081535e-01, -7.61396541e-01, -4.82444404e-01,
       -5.31441961e-01, -6.88944015e-01, -7.69081535e-01,  3.51287964e+00,
       -5.37118643e-01, -5.03849086e-01, -5.84697144e-01, -5.56401318e-01,
       -5.58332752e-01,  3.45111593e+00, -6.29320205e-01, -5.80855097e-01,
       -5.41509418e-01, -5.30728378e-01, -5.80917515e-01, -6.29320205e-01,
        3.46441464e+00, -6.18564692e-01, -5.78996150e-01, -5.25887697e-01,
       -5.48388647e-01, -5.83604794e-01, -6.18564692e-01,  3.46398188e+00,
       -5.75881701e-01, -5.80194410e-01, -5.57347639e-01, -5.65957207e-01,
       -5.47937359e-01,  3.44525468e+00, -5.85782166e-01, -6.24145495e-01,
       -5.75730408e-01, -5.45702046e-01, -5.67321052e-01, -5.94846265e-01,
       -5.85782166e-01,  3.47264918e+00, -5.62839849e-01, -5.81104836e-01,
       -5.80755012e-01, -5.57898558e-01, -5.46131288e-01, -6.24145495e-01,
        3.44997588e+00, -6.32983299e-01, -5.75907240e-01, -5.12910001e-01,
       -5.86346677e-01, -5.89752340e-01, -5.62839849e-01,  3.46469671e+00,
       -5.63617818e-01, -5.83103238e-01, -5.79036788e-01, -5.88428912e-01,
       -5.87174611e-01, -5.63617818e-01,  3.46093000e+00, -5.44760501e-01,
       -5.86689168e-01, -5.90258988e-01, -6.02265722e-01, -5.93899168e-01,
       -5.44760501e-01,  3.46573116e+00, -5.35111518e-01, -5.92267297e-01,
       -5.97426953e-01, -5.41015058e-01, -5.57067563e-01, -6.32983299e-01,
        3.43122289e+00, -6.16988625e-01, -6.01275867e-01, -4.81892478e-01,
       -6.29870451e-01, -5.58279910e-01, -5.35111518e-01,  3.46296743e+00,
       -5.58948043e-01, -5.90234512e-01, -5.90522994e-01, -5.69123432e-01,
       -6.57248551e-01, -5.24515085e-01,  3.58604218e+00, -5.84078485e-01,
       -6.67731398e-01, -5.83345231e-01, -5.94042843e-01, -4.72619926e-01,
       -7.42469902e-01,  3.49894203e+00, -7.07373590e-01, -5.12024814e-01,
       -4.70410954e-01, -5.43407846e-01, -5.30243440e-01, -6.75638493e-01,
        3.49699630e+00, -6.89185183e-01, -5.75353546e-01, -4.83167790e-01,
       -5.35304118e-01, -6.96342342e-01, -5.48727868e-01,  3.47960162e+00,
       -5.14192547e-01, -6.36920159e-01, -5.48114581e-01, -6.38803325e-01,
       -5.10888992e-01, -5.58948043e-01,  3.47094321e+00, -6.50265309e-01,
       -6.12320548e-01, -4.99716991e-01, -5.68979914e-01, -6.37416289e-01,
        3.45977068e+00, -5.31346563e-01, -5.35944248e-01, -5.73802874e-01,
       -6.12280790e-01, -5.81297165e-01, -6.11594753e-01, -5.31346563e-01,
        3.47468949e+00, -6.21368261e-01, -6.38126073e-01, -4.90956672e-01,
       -5.04046157e-01, -5.39171802e-01, -6.21368261e-01,  3.49089171e+00,
       -7.63534904e-01, -7.16691919e-01, -3.46078670e-01, -5.38595621e-01,
       -6.61031362e-01, -5.35944248e-01,  3.50187393e+00, -6.01463256e-01,
       -6.43454906e-01, -5.21384537e-01, -5.32044904e-01, -5.16624045e-01,
       -7.63534904e-01,  3.50143581e+00, -6.58155801e-01, -4.33891823e-01,
       -5.97184331e-01, -5.71244146e-01, -5.68858318e-01, -6.58155801e-01,
        3.48434697e+00, -5.83367729e-01, -5.02862490e-01, -5.99858483e-01,
       -4.87572479e-01, -6.21947708e-01, -6.01463256e-01,  3.49282679e+00,
       -6.29260974e-01, -7.08842667e-01, -4.43739707e-01, -5.91977343e-01,
       -5.58833545e-01,  3.46445176e+00, -5.75741197e-01, -5.75462479e-01,
       -5.88267704e-01, -5.74169488e-01, -5.61268153e-01, -5.81055422e-01,
       -5.75741197e-01,  3.46512967e+00, -5.95524948e-01, -6.01665895e-01,
       -5.49874059e-01, -5.91565848e-01, -5.52850544e-01, -5.75462479e-01,
        3.46444272e+00, -5.75627610e-01, -5.86723084e-01, -5.82213156e-01,
       -5.55914261e-01, -5.35129266e-01, -5.95524948e-01,  3.47053670e+00,
       -6.74328896e-01, -5.88570315e-01, -5.21069014e-01, -5.98185525e-01,
       -5.31876854e-01, -5.75627610e-01,  3.46751642e+00, -5.80257743e-01,
       -6.01039832e-01, -5.80528856e-01, -5.74899130e-01, -5.01148679e-01,
       -6.74328896e-01,  3.48841572e+00, -6.83811731e-01, -4.93379181e-01,
       -5.60848102e-01, -6.22429948e-01, -5.65924574e-01, -5.80257743e-01,
        3.49779448e+00, -5.39913124e-01, -5.84121482e-01, -6.05147606e-01,
       -5.79847762e-01, -5.94184444e-01, -6.83811731e-01,  3.47548606e+00,
       -5.81992676e-01, -5.52989945e-01, -4.82659499e-01, -5.02351415e-01,
       -7.30168829e-01, -5.81992676e-01,  3.49096766e+00, -5.24404675e-01,
       -7.09631189e-01, -4.42418874e-01, -4.70121025e-01, -6.81614411e-01,
       -5.24404675e-01,  3.48571757e+00, -5.67412537e-01, -6.70600274e-01,
       -5.71564653e-01, -5.44931195e-01, -5.38728873e-01, -6.29260974e-01,
       -5.67412537e-01,  3.45170075e+00, -6.72241650e-01, -4.99125523e-01,
       -6.06963658e-01, -5.53586890e-01, -5.39913124e-01,  3.52780720e+00,
       -5.63784982e-01, -5.87912298e-01, -6.75646248e-01, -5.49666668e-01,
       -3.80376988e-01, -5.63784982e-01,  3.31254084e+00, -7.27264012e-01,
       -4.64624846e-01, -6.26823341e-01, -7.14368417e-01, -4.53044983e-01,
       -7.27264012e-01,  3.54670631e+00, -8.20151272e-01, -4.27016962e-01,
       -4.04860663e-01, -7.95010247e-01, -5.81922149e-01, -8.20151272e-01,
        3.76385392e+00, -5.01274400e-01, -6.68253025e-02, -9.98670548e-01,
       -6.05166238e-01, -5.66531395e-01, -5.83367729e-01,  3.46441605e+00,
       -5.59213638e-01, -5.38398977e-01, -6.11738077e-01, -6.65628442e-01,
       -5.36157518e-01, -5.01274400e-01,  3.50195929e+00, -6.92299399e-01,
       -5.66106456e-01, -5.40493074e-01, -6.50174315e-01, -4.82509011e-01,
       -8.07540431e-01,  3.63958327e+00, -3.85810093e-01, -2.43706229e-01,
       -9.76764279e-01, -9.30789100e-02, -6.65594357e-01, -6.47930337e-01,
        3.58546975e+00, -2.50833118e-01, -6.41433772e-01, -4.44101795e-01,
       -9.35576369e-01, -5.20305540e-01, -7.78805995e-01, -2.50833118e-01,
        3.56515820e+00, -5.98923481e-01, -8.88621315e-01, -2.49388382e-01,
       -2.78280365e-01, -4.81162320e-01, -6.44254343e-01,  3.48141522e+00,
       -6.11555273e-01, -6.50085012e-01, -4.40613107e-01, -6.53745162e-01,
       -6.34044118e-01, -5.27941287e-01, -6.11555273e-01,  3.48676249e+00,
       -5.84821106e-01, -4.85362373e-01, -6.43038331e-01, -5.88887422e-01,
       -5.95776325e-01, -5.84821106e-01,  3.46768669e+00, -5.43832085e-01,
       -5.47275322e-01, -6.07094435e-01, -5.94497790e-01, -6.16976709e-01,
       -5.29307077e-01, -5.43832085e-01,  3.48395942e+00, -5.96214172e-01,
       -6.03131589e-01, -4.10990463e-01, -7.66420896e-01, -6.50085012e-01,
        3.50387287e+00, -5.93206820e-01, -6.54181839e-01, -4.28987843e-01,
       -3.57908848e-01, -6.66557865e-01, -6.52619789e-01,  3.51848291e+00,
       -5.15767784e-01, -7.85066578e-01, -5.40562046e-01, -6.07846071e-01,
       -4.91677917e-01,  3.49804949e+00, -7.17751376e-01, -6.56997708e-01,
       -4.15835296e-01, -6.07941119e-01, -4.50706734e-01, -6.62821297e-01,
       -7.17751376e-01,  3.52561925e+00, -7.03719163e-01, -6.16544251e-01,
       -3.74076430e-01, -5.57304494e-01, -7.57983941e-01, -6.56997708e-01,
        3.53273950e+00, -3.38389496e-01, -5.27264497e-01, -6.94799367e-01,
       -5.94117157e-01, -6.11244280e-01, -5.59213638e-01, -5.15767784e-01,
        3.46542687e+00, -5.51159310e-01, -6.33924704e-01, -2.34627802e-01,
       -9.71666714e-01, -2.76407190e-01, -3.38389496e-01,  3.58331204e+00,
       -3.52399131e-01, -6.67940252e-01, -7.41881461e-01, -7.73406867e-01,
       -5.75881701e-01, -3.52399131e-01,  3.51895752e+00, -5.54476656e-01,
       -6.35792076e-01, -6.27001090e-01, -5.28712422e-01, -5.37118643e-01,
       -6.16988625e-01,  3.48084911e+00, -6.04161662e-01, -6.46000994e-01,
       -5.47866764e-01, -5.76068054e-01, -5.84078485e-01, -7.07373590e-01,
        3.52307509e+00, -6.14779417e-01, -5.66538751e-01, -4.74236794e-01,
       -6.49170148e-01, -5.80337565e-01, -5.52049612e-01,  3.48458029e+00,
       -5.19574929e-01, -5.86029335e-01, -5.97418702e-01, -5.53270194e-01,
       -6.89185183e-01, -5.19574929e-01,  3.53197566e+00, -6.80662696e-01,
       -6.93948064e-01, -3.95334591e-01, -6.40836400e-01, -5.12625672e-01,
       -5.86029335e-01,  3.48552517e+00, -6.30707070e-01, -5.03809628e-01,
       -6.11517062e-01, -6.65533577e-01, -5.97416237e-01, -6.30707070e-01,
        3.56504982e+00, -5.73429461e-01, -5.37157017e-01, -5.60806460e-01,
       -4.88218388e-01, -6.67731398e-01, -5.75353546e-01,  3.45968973e+00,
       -5.86446433e-01, -5.90156534e-01, -5.51783434e-01, -5.86845255e-01,
       -4.93937681e-01, -5.73429461e-01,  3.48207488e+00, -6.56362464e-01,
       -5.97186399e-01, -5.74313621e-01, -5.63775737e-01, -5.14192547e-01,
       -6.56362464e-01,  3.47520836e+00, -6.62792513e-01, -5.21031771e-01,
       -5.57053324e-01, -5.59999689e-01, -5.12024814e-01, -6.50265309e-01,
        3.49130099e+00, -7.21953229e-01, -6.10080000e-01, -4.36977945e-01,
       -4.83050663e-01, -6.92299399e-01, -5.98923481e-01,  3.53695163e+00,
       -6.34818635e-01, -7.80207075e-01, -3.47652379e-01, -5.01065523e-01,
       -4.89653497e-01,  3.55031169e+00, -7.53804113e-01, -6.84589034e-01,
       -6.18669010e-01, -5.02530511e-01, -6.34209376e-01, -3.85810093e-01,
       -7.53804113e-01,  3.49716927e+00, -7.55778464e-01, -4.92926178e-01,
       -4.74641042e-01, -5.32018526e-01, -7.60057114e-01, -6.92259789e-01,
       -6.84589034e-01,  4.01467551e+00, -6.50640512e-01, -6.95110536e-01,
       -6.06966314e-01, -6.06893363e-01, -6.41433772e-01, -6.50640512e-01,
        3.54032720e+00, -5.93270930e-01, -4.41122306e-01, -5.43137718e-01,
       -6.53770056e-01, -5.60710465e-01,  3.51505461e+00, -5.59223373e-01,
       -6.61081685e-01, -5.37131317e-01, -5.75200641e-01, -5.92146390e-01,
       -5.59223373e-01,  3.48862250e+00, -5.68227836e-01, -5.94303856e-01,
       -5.99520403e-01, -5.21184896e-01, -5.79767273e-01, -5.68227836e-01,
        3.36744292e+00, -5.62125393e-01, -5.85479495e-01, -5.50658025e-01,
       -6.45179455e-01, -2.43706229e-01, -6.61081685e-01,  3.65708013e+00,
       -9.65445069e-01, -6.02830009e-01, -5.38837686e-01, -5.88385668e-01,
       -6.36920159e-01, -5.62125393e-01,  3.49012692e+00, -5.44479988e-01,
       -6.09059647e-01, -5.49156069e-01, -4.03854685e-01, -5.93206820e-01,
       -7.85066578e-01,  3.50436203e+00, -6.87134044e-01, -5.35954397e-01,
       -4.99145503e-01, -5.30527463e-01, -5.96214172e-01, -7.03719163e-01,
        3.38325243e+00, -5.35018101e-01, -4.28920578e-01, -5.88852951e-01,
       -5.88491286e-01, -4.72836253e-01,  3.47375723e+00, -6.41049931e-01,
       -6.41243590e-01, -5.47711359e-01, -5.82424815e-01, -4.85900511e-01,
       -6.28023568e-01, -6.41049931e-01,  3.47063059e+00, -6.00611076e-01,
       -5.39483448e-01, -5.75562060e-01, -4.48726600e-01, -7.61396541e-01,
       -5.03849086e-01, -6.04161662e-01, -6.41243590e-01,  3.53771301e+00,
       -5.78335535e-01, -5.32602001e-01, -6.24078880e-01, -6.00611076e-01,
        3.46580214e+00, -5.77684204e-01, -5.67340500e-01, -5.63485483e-01,
       -5.61971572e-01, -5.80855097e-01, -5.77684204e-01,  3.46279789e+00,
       -6.16087486e-01, -5.79195234e-01, -5.47004294e-01, -5.41509418e-01,
       -5.78996150e-01, -6.16087486e-01,  3.46277301e+00, -6.19289588e-01,
       -5.86924879e-01, -5.19965492e-01, -5.25887697e-01, -5.80194410e-01,
       -6.19289588e-01,  3.46769904e+00, -6.16768374e-01, -6.26942044e-01,
       -4.98616926e-01, -5.57347639e-01, -5.54476656e-01, -6.16768374e-01,
        3.45970785e+00, -6.09178974e-01, -6.15408356e-01, -5.06527854e-01,
       -5.75730408e-01, -5.81104836e-01,  3.45857458e+00, -5.92785722e-01,
       -5.74862279e-01, -5.49939381e-01, -5.84151954e-01, -5.45702046e-01,
       -5.75907240e-01, -5.92785722e-01,  3.45895813e+00, -6.32794303e-01,
       -6.04561695e-01, -5.07207128e-01, -5.80755012e-01, -5.83103238e-01,
       -5.74862279e-01,  3.46930166e+00, -5.65150945e-01, -5.74740434e-01,
       -5.90689747e-01, -5.12910001e-01, -6.01275867e-01, -6.32794303e-01,
        3.46773927e+00, -6.22284829e-01, -6.21443142e-01, -4.77031133e-01,
       -5.79036788e-01, -5.86689168e-01, -5.65150945e-01,  3.42925941e+00,
       -5.53719182e-01, -5.75867335e-01, -5.68795987e-01, -5.90258988e-01,
       -5.92267297e-01, -5.53719182e-01,  3.46455652e+00, -5.45052100e-01,
       -6.08544831e-01, -5.74714120e-01, -5.97426953e-01, -5.90234512e-01,
       -5.45052100e-01,  3.45680062e+00, -5.33532400e-01, -6.18163531e-01,
       -5.72391123e-01, -4.81892478e-01, -6.46000994e-01, -6.22284829e-01,
        3.46927696e+00, -5.41156723e-01, -6.73070958e-01, -5.04870979e-01,
       -5.90522994e-01, -6.12320548e-01, -5.33532400e-01,  3.47097264e+00,
       -5.83605776e-01, -6.62426336e-01, -4.88564585e-01, -5.83345231e-01,
       -6.14779417e-01, -5.86446433e-01,  3.49133262e+00, -6.39296843e-01,
       -4.57592501e-01, -6.09872190e-01, -4.70410954e-01, -5.66538751e-01,
       -7.21953229e-01,  3.49216147e+00, -6.57013904e-01, -5.71244103e-01,
       -5.05000531e-01, -4.83167790e-01, -6.80662696e-01, -5.90156534e-01,
        4.01051340e+00, -9.94174772e-01, -1.22334579e+00, -3.90058113e-02,
       -5.48114581e-01, -6.62792513e-01, -5.44479988e-01,  3.47821759e+00,
       -5.13025374e-01, -6.26335470e-01, -5.83469667e-01, -4.99716991e-01,
       -6.10080000e-01, -5.83605776e-01,  3.56020713e+00, -7.85233570e-01,
       -8.09513537e-01, -2.72057260e-01, -5.73802874e-01, -6.43454906e-01,
        3.47691481e+00, -5.21707026e-01, -4.97458627e-01, -6.32323628e-01,
       -6.08167751e-01, -6.12280790e-01, -6.38126073e-01, -5.21707026e-01,
        3.49619310e+00, -5.33532911e-01, -6.07678003e-01, -5.82868294e-01,
       -4.90956672e-01, -7.16691919e-01, -5.33532911e-01,  3.78321660e+00,
       -1.03273401e+00, -1.00930109e+00, -3.46078670e-01, -4.33891823e-01,
       -1.03273401e+00,  3.63733873e+00, -7.09783333e-01, -1.03121809e-01,
       -7.33046335e-01, -2.78682752e-01, -5.21384537e-01, -7.08842667e-01,
       -4.97458627e-01,  3.46617124e+00, -6.08057805e-01, -5.68510756e-01,
       -5.61916846e-01, -5.97184331e-01, -5.02862490e-01, -7.09783333e-01,
        3.49413847e+00, -6.36982116e-01, -4.12364058e-01, -6.34962142e-01,
       -5.99858483e-01, -5.38398977e-01, -6.36982116e-01,  3.47198383e+00,
       -5.92170031e-01, -4.97974559e-01, -6.06599669e-01, -4.43739707e-01,
       -6.72241650e-01, -6.08057805e-01,  3.49024520e+00, -6.64759611e-01,
       -6.55704821e-01, -4.45741604e-01, -5.88267704e-01, -5.86723084e-01,
        3.46503082e+00, -5.58087191e-01, -5.45667778e-01, -5.96041433e-01,
       -5.90243627e-01, -5.74169488e-01, -6.01665895e-01, -5.58087191e-01,
        3.46594428e+00, -5.68914403e-01, -6.14164638e-01, -5.48942663e-01,
       -5.49874059e-01, -5.88570315e-01, -5.68914403e-01,  3.47853910e+00,
       -6.53968801e-01, -6.61128897e-01, -4.56082627e-01, -5.82213156e-01,
       -6.01039832e-01, -5.45667778e-01,  3.46945752e+00, -5.01042460e-01,
       -6.09603491e-01, -6.29890805e-01, -5.21069014e-01, -4.93379181e-01,
       -6.53968801e-01,  3.50548615e+00, -7.75292164e-01, -5.56047380e-01,
       -5.05729606e-01, -5.80528856e-01, -5.84121482e-01, -5.01042460e-01,
        3.44485021e+00, -5.40465294e-01, -5.77063808e-01, -6.61628307e-01,
       -5.60848102e-01, -5.52989945e-01, -7.75292164e-01,  3.48731906e+00,
       -6.41340069e-01, -4.98902278e-01, -4.57946505e-01, -6.05147606e-01,
       -5.87912298e-01, -5.40465294e-01,  3.45690123e+00, -5.03755650e-01,
       -5.82671332e-01, -6.36949047e-01, -4.82659499e-01, -7.09631189e-01,
       -6.41340069e-01,  3.49985361e+00, -5.51259642e-01, -6.93754461e-01,
       -4.21208748e-01, -4.42418874e-01, -6.70600274e-01, -5.51259642e-01,
        3.49823512e+00, -5.99954814e-01, -6.63120495e-01, -5.70881026e-01,
       -5.71564653e-01, -4.99125523e-01, -6.64759611e-01, -5.99954814e-01,
        3.49128812e+00, -6.47607813e-01, -5.08275704e-01, -6.75646248e-01,
       -4.64624846e-01, -5.03755650e-01,  3.54280765e+00, -6.88701179e-01,
       -5.54399644e-01, -6.55680080e-01, -6.26823341e-01, -4.27016962e-01,
       -6.88701179e-01,  3.56971285e+00, -8.65211581e-01, -5.40867576e-01,
       -4.21092208e-01, -4.04860663e-01, -6.68253025e-02, -8.65211581e-01,
        3.61795073e+00, -9.84988167e-01, -5.22741075e-01, -4.44360450e-01,
       -3.28963488e-01, -9.98670548e-01, -5.66106456e-01, -9.84988167e-01,
        3.74747070e+00, -5.64117560e-01, -6.33587971e-01, -6.11738077e-01,
       -5.51159310e-01, -5.92170031e-01,  3.46094089e+00, -5.75641785e-01,
       -5.32941096e-01, -5.97290595e-01, -5.40493074e-01, -6.34818635e-01,
       -5.64117560e-01,  3.49041297e+00, -6.46850250e-01, -6.12782032e-01,
       -4.91351415e-01, -9.76764279e-01, -7.55778464e-01,  3.80381527e+00,
       -1.01761104e+00, -5.44662795e-01, -5.08998687e-01, -9.30789100e-02,
       -9.65445069e-01, -1.01761104e+00,  3.81909709e+00, -6.04315862e-01,
       -5.68181323e-01, -5.70464881e-01, -4.44101795e-01, -5.93270930e-01,
        3.62391090e+00, -4.35252095e-01, -9.10488557e-01, -9.44658065e-01,
       -2.96139456e-01, -9.35576369e-01, -8.88621315e-01, -4.35252095e-01,
        3.75516507e+00, -9.55219838e-01, -5.40495452e-01, -2.49388382e-01,
       -7.80207075e-01,  3.78673278e+00, -8.48511646e-01, -8.31893972e-01,
       -9.59445858e-01, -1.17285845e-01, -4.40613107e-01, -6.54181839e-01,
        3.49582361e+00, -6.36232451e-01, -6.41516585e-01, -6.16494781e-01,
       -5.06784847e-01, -6.53745162e-01, -4.85362373e-01, -6.36232451e-01,
        3.50466709e+00, -5.80980050e-01, -4.92398581e-01, -6.55948472e-01,
       -6.43038331e-01, -5.47275322e-01, -5.80980050e-01,  3.47764784e+00,
       -5.51169824e-01, -5.09495734e-01, -6.45688583e-01, -6.07094435e-01,
       -6.03131589e-01, -5.35018101e-01, -5.51169824e-01,  3.47820749e+00,
       -6.15055372e-01, -5.66738166e-01, -4.28987843e-01, -6.87134044e-01,
       -6.41516585e-01,  3.48625489e+00, -6.00122639e-01, -6.08509838e-01,
       -5.19983939e-01, -5.40562046e-01, -6.33924704e-01, -5.35954397e-01,
       -5.75641785e-01,  3.46994556e+00, -6.43668686e-01, -5.40193943e-01,
       -4.15835296e-01, -6.16544251e-01,  3.57080503e+00, -6.96582620e-01,
       -8.30723858e-01, -3.35327396e-01, -6.75791606e-01, -6.07941119e-01,
       -5.27264497e-01, -6.96582620e-01,  3.48995198e+00, -5.62417330e-01,
       -4.16913403e-01, -6.78833015e-01, -3.74076430e-01, -4.28920578e-01,
       -8.30723858e-01,  3.74423065e+00, -1.13491938e+00, -7.42806106e-01,
       -7.82824691e-02, -1.54501831e-01, -6.94799367e-01, -6.67940252e-01,
       -5.62417330e-01,  3.49808692e+00, -3.63019116e-01, -5.12345083e-01,
       -6.97565773e-01, -7.41881461e-01, -6.35792076e-01, -3.63019116e-01,
        3.49653154e+00, -4.81960196e-01, -5.89137051e-01, -6.84741638e-01,
       -6.27001090e-01, -6.09178974e-01, -4.81960196e-01,  3.45879362e+00,
       -5.80752470e-01, -6.03367586e-01, -5.56533308e-01, -5.47866764e-01,
       -5.47711359e-01, -5.78335535e-01, -5.41156723e-01,  3.47326888e+00,
       -5.87330102e-01, -6.70868400e-01, -4.74236794e-01, -6.39296843e-01,
       -6.57013904e-01,  3.44188417e+00, -5.19137899e-01, -6.05323175e-01,
       -5.46875550e-01, -5.97418702e-01, -6.93948064e-01, -5.03809628e-01,
        3.62015565e+00, -5.17021737e-01, -6.51773048e-01, -6.56184473e-01,
       -3.95334591e-01, -9.94174772e-01, -5.17021737e-01,  3.61061644e+00,
       -3.85206117e-01, -8.01700203e-01, -5.17179024e-01, -6.11517062e-01,
       -5.37157017e-01, -6.51773048e-01,  3.40495079e+00, -6.01287922e-01,
       -4.81191539e-01, -5.22024206e-01, -5.60806460e-01, -5.97186399e-01,
       -6.01287922e-01,  3.45379005e+00, -5.78725111e-01, -6.09572489e-01,
       -5.06211667e-01, -5.51783434e-01, -4.57592501e-01, -1.22334579e+00,
        3.85055156e+00, -6.75409653e-01, -9.42420181e-01, -5.74313621e-01,
       -5.21031771e-01, -5.78725111e-01,  3.51308945e+00, -6.62442441e-01,
       -5.94611674e-01, -5.81964831e-01, -5.57053324e-01, -5.13025374e-01,
       -6.62442441e-01,  3.44710690e+00, -6.33702715e-01, -5.08482042e-01,
       -5.72401008e-01, -4.36977945e-01, -5.71244103e-01, -7.85233570e-01,
        3.49636878e+00, -6.61688709e-01, -5.24015921e-01, -5.17208528e-01,
       -3.47652379e-01, -6.46850250e-01, -8.48511646e-01,  3.51016779e+00,
       -6.90487565e-01, -5.60795963e-01, -4.15869984e-01, -6.18669010e-01,
       -4.92926178e-01,  3.53896520e+00, -7.05799781e-01, -6.84734650e-01,
       -5.13918672e-01, -5.22916906e-01, -5.02530511e-01, -6.95110536e-01,
       -4.41122306e-01, -9.10488557e-01, -7.05799781e-01,  3.84992106e+00,
       -5.94869373e-01, -4.74641042e-01, -5.44662795e-01, -6.84734650e-01,
        3.49713780e+00, -6.74421071e-01, -5.58759798e-01, -5.59918445e-01,
       -5.37131317e-01, -5.94303856e-01, -6.02830009e-01,  3.46972125e+00,
       -5.86139289e-01, -5.46182926e-01, -6.03133856e-01, -5.99520403e-01,
       -5.85479495e-01, -5.86139289e-01,  3.49182165e+00, -5.72662764e-01,
       -5.34672059e-01, -6.13347644e-01, -5.50658025e-01, -6.09059647e-01,
       -5.72662764e-01,  3.46421520e+00, -5.78907183e-01, -6.00282611e-01,
       -5.52644971e-01, -5.38837686e-01, -6.04315862e-01, -5.46182926e-01,
        3.56059098e+00, -5.55176585e-01, -5.97118838e-01, -7.18959085e-01,
       -5.49156069e-01, -6.26335470e-01, -5.78907183e-01,  3.47758610e+00,
       -5.51167778e-01, -6.11440023e-01, -5.60579577e-01, -2.78280365e-01,
       -9.55219838e-01, -8.31893972e-01,  3.67107039e+00, -3.59641742e-01,
       -5.04319427e-01, -7.41715046e-01, -4.99145503e-01, -6.00122639e-01,
       -6.43668686e-01,  3.47311660e+00, -5.96525431e-01, -6.09005826e-01,
       -5.24648520e-01, -5.88852951e-01, -6.15055372e-01, -1.13491938e+00,
        3.83653199e+00, -5.30743455e-01, -9.66960837e-01, -5.82424815e-01,
       -5.39483448e-01, -5.87330102e-01,  3.46610883e+00, -5.86552717e-01,
       -5.21719557e-01, -6.48598188e-01, -5.75562060e-01, -5.67340500e-01,
       -5.86552717e-01,  3.46323370e+00, -5.93647447e-01, -5.23647292e-01,
       -6.16483686e-01, -5.63485483e-01, -5.79195234e-01, -5.93647447e-01,
        3.45767103e+00, -5.85138266e-01, -5.52055198e-01, -5.84149402e-01,
       -5.47004294e-01, -5.86924879e-01, -5.85138266e-01,  3.44148532e+00,
       -5.83399186e-01, -5.85974796e-01, -5.53043900e-01, -5.19965492e-01,
       -6.26942044e-01, -5.83399186e-01,  3.46626449e+00, -6.11486357e-01,
       -6.43754368e-01, -4.80717040e-01, -4.98616926e-01, -6.15408356e-01,
       -6.11486357e-01,  3.49477265e+00, -6.92375896e-01, -6.89556239e-01,
       -3.87328872e-01, -5.06527854e-01, -5.80752470e-01, -6.92375896e-01,
        3.47091667e+00, -6.37394793e-01, -5.34115375e-01, -5.19750278e-01,
       -5.49939381e-01, -6.04561695e-01,  3.46576092e+00, -5.84143951e-01,
       -5.86086488e-01, -5.40008104e-01, -6.01021298e-01, -5.84151954e-01,
       -5.74740434e-01, -5.84143951e-01,  3.46290917e+00, -5.70525262e-01,
       -5.56475487e-01, -5.92872078e-01, -5.07207128e-01, -6.21443142e-01,
       -5.86086488e-01,  3.44364724e+00, -5.97544529e-01, -6.22398421e-01,
       -5.08967531e-01, -5.90689747e-01, -5.75867335e-01, -5.70525262e-01,
        3.45855396e+00, -5.70642308e-01, -5.73637172e-01, -5.77192140e-01,
       -4.77031133e-01, -6.73070958e-01, -5.97544529e-01,  3.46618184e+00,
       -5.41712908e-01, -6.78712041e-01, -4.98110276e-01, -5.68795987e-01,
       -6.08544831e-01, -5.70642308e-01,  3.45989415e+00, -5.45568422e-01,
       -5.98103101e-01, -5.68239505e-01, -5.74714120e-01, -6.18163531e-01,
       -5.45568422e-01,  3.46211972e+00, -5.20398873e-01, -6.26862909e-01,
       -5.76411862e-01, -5.72391123e-01, -6.62426336e-01, -5.20398873e-01,
        3.48188709e+00, -4.92514943e-01, -6.82932861e-01, -5.51222952e-01,
       -5.04870979e-01, -6.70868400e-01, -5.21719557e-01, -5.41712908e-01,
        3.46719492e+00, -5.42508461e-01, -6.85514614e-01, -4.88564585e-01,
       -8.09513537e-01, -4.92514943e-01,  3.77804506e+00, -9.18855624e-01,
       -1.06859637e+00, -6.09872190e-01, -5.19137899e-01, -6.75409653e-01,
        3.50185728e+00, -3.90012610e-01, -6.51974710e-01, -6.55450213e-01,
       -5.05000531e-01, -6.05323175e-01, -6.61688709e-01,  3.47185041e+00,
       -5.74602647e-01, -5.68527489e-01, -5.56707861e-01, -3.90058113e-02,
       -3.85206117e-01, -9.42420181e-01, -3.90012610e-01,  3.60531950e+00,
       -6.73061750e-01, -5.73588339e-01, -6.02024694e-01, -5.83469667e-01,
       -6.33702715e-01, -5.51167778e-01,  3.49582814e+00, -5.15739306e-01,
       -6.05552343e-01, -6.06196329e-01, -2.72057260e-01, -5.24015921e-01,
       -9.18855624e-01,  3.53882142e+00, -6.71775529e-01, -1.13632973e-01,
       -5.77577645e-01, -4.60906472e-01, -6.32323628e-01, -5.68510756e-01,
        3.46045557e+00, -5.02889471e-01, -5.14459046e-01, -6.05540050e-01,
       -6.36732619e-01, -6.08167751e-01, -6.07678003e-01, -5.02889471e-01,
        3.43286218e+00, -4.86384887e-01, -5.88990767e-01, -6.38751296e-01,
       -5.82868294e-01, -1.00930109e+00, -1.03121809e-01, -4.86384887e-01,
        3.75310062e+00, -9.56133806e-01, -6.15290734e-01, -7.33046335e-01,
       -4.12364058e-01,  3.51780608e+00, -6.45109846e-01, -6.22086692e-01,
       -6.77035884e-01, -4.28163268e-01, -5.61916846e-01, -6.55704821e-01,
       -5.14459046e-01,  3.49612527e+00, -6.52675654e-01, -5.36720603e-01,
       -5.74648296e-01, -6.34962142e-01, -4.97974559e-01, -6.45109846e-01,
        3.47922807e+00, -6.21572601e-01, -4.59880702e-01, -6.19728222e-01,
       -6.06599669e-01, -5.32941096e-01, -6.21572601e-01,  3.46817771e+00,
       -6.05232481e-01, -5.15614086e-01, -5.86217774e-01, -4.45741604e-01,
       -6.47607813e-01, -6.52675654e-01,  3.48409556e+00, -6.34447693e-01,
       -6.25518067e-01, -4.78104730e-01, -5.96041433e-01, -6.09603491e-01,
        3.47056471e+00, -5.23442020e-01, -4.97571911e-01, -6.27710607e-01,
       -6.16195244e-01, -5.90243627e-01, -6.14164638e-01, -5.23442020e-01,
        3.46824652e+00, -5.30459783e-01, -6.29314247e-01, -5.80622204e-01,
       -5.48942663e-01, -6.61128897e-01, -5.30459783e-01,  3.47978315e+00,
       -5.55639983e-01, -6.86667368e-01, -4.96944461e-01, -4.56082627e-01,
       -5.56047380e-01, -5.55639983e-01,  3.47503370e+00, -7.30443883e-01,
       -5.73769094e-01, -6.03050734e-01, -6.29890805e-01, -5.77063808e-01,
       -4.97571911e-01,  3.50042871e+00, -5.22033541e-01, -5.98820501e-01,
       -6.75048142e-01, -5.05729606e-01, -4.98902278e-01, -7.30443883e-01,
        3.50879608e+00, -7.40686377e-01, -4.67188247e-01, -5.65845693e-01,
       -6.61628307e-01, -5.82671332e-01, -5.22033541e-01,  3.50533723e+00,
       -5.08240595e-01, -5.95878570e-01, -6.34884885e-01, -4.57946505e-01,
       -6.93754461e-01, -7.40686377e-01,  3.51337510e+00, -5.64743097e-01,
       -6.36520779e-01, -4.19723881e-01, -6.36949047e-01, -5.54399644e-01,
       -5.08240595e-01,  3.48164214e+00, -5.88142685e-01, -6.83291345e-01,
       -5.10618820e-01, -4.21208748e-01, -6.63120495e-01, -5.64743097e-01,
        3.49658739e+00, -6.31434425e-01, -6.52912204e-01, -5.63168423e-01,
       -5.70881026e-01, -5.08275704e-01, -6.34447693e-01, -6.31434425e-01,
        3.49197939e+00, -6.22425032e-01, -5.24515514e-01, -6.55680080e-01,
       -5.40867576e-01, -5.88142685e-01,  3.54691292e+00, -6.81111364e-01,
       -7.82220035e-01, -2.98891177e-01, -4.21092208e-01, -5.22741075e-01,
       -6.81111364e-01,  3.75619764e+00, -1.24056175e+00, -8.90691241e-01,
       -4.44360450e-01, -6.33587971e-01, -6.12782032e-01,  3.49300223e+00,
       -6.04620412e-01, -6.28707212e-01, -5.68944148e-01, -5.97290595e-01,
       -5.40193943e-01, -5.96525431e-01, -6.05232481e-01,  3.47025585e+00,
       -5.91316320e-01, -5.39697081e-01, -4.91351415e-01, -6.90487565e-01,
       -6.04620412e-01,  3.47638561e+00, -5.74329416e-01, -5.87409375e-01,
       -5.28187429e-01, -5.08998687e-01, -5.68181323e-01, -6.74421071e-01,
        3.51406547e+00, -6.46953828e-01, -5.08610704e-01, -6.06899858e-01,
       -5.70464881e-01, -5.55176585e-01, -6.46953828e-01,  3.53964004e+00,
       -5.03111403e-01, -5.14542390e-01, -7.49390952e-01, -9.44658065e-01,
       -5.40495452e-01, -3.59641742e-01,  3.57854446e+00, -4.13331425e-01,
       -6.60260760e-01, -6.60157018e-01, -2.96139456e-01, -5.13918672e-01,
       -5.94869373e-01, -4.13331425e-01,  3.62241054e+00, -5.89971565e-01,
       -6.19926489e-01, -5.94253562e-01, -9.59445858e-01, -5.04319427e-01,
        3.79435702e+00, -8.92534863e-01, -4.24863553e-01, -1.01319332e+00,
       -1.17285845e-01, -5.60795963e-01, -8.92534863e-01,  3.56814502e+00,
       -7.73829496e-01, -2.18183702e-01, -4.04849385e-01, -6.00665769e-01,
       -6.16494781e-01, -4.92398581e-01,  3.42686637e+00, -6.36242103e-01,
       -5.75202562e-01, -5.72299599e-01, -5.34228743e-01, -5.06784847e-01,
       -6.08509838e-01, -6.36242103e-01,  3.47551715e+00, -6.07104916e-01,
       -5.53036682e-01, -5.63838762e-01, -6.55948472e-01, -5.09495734e-01,
       -5.75202562e-01,  3.48789398e+00, -5.56978064e-01, -5.48468357e-01,
       -6.41800788e-01, -6.45688583e-01, -5.66738166e-01, -5.30743455e-01,
       -5.56978064e-01,  3.47723921e+00, -6.11356590e-01, -5.65734357e-01,
       -5.19983939e-01, -6.09005826e-01, -6.07104916e-01,  3.47648067e+00,
       -6.16142449e-01, -5.78020503e-01, -5.46223038e-01, -3.35327396e-01,
       -7.42806106e-01,  3.58654124e+00, -7.22358081e-01, -7.17571589e-01,
       -7.11421927e-01, -3.57056142e-01, -6.75791606e-01, -4.16913403e-01,
       -7.22358081e-01,  3.52042261e+00, -6.86591400e-01, -3.63041763e-01,
       -6.55726359e-01, -6.78833015e-01, -5.12345083e-01, -6.86591400e-01,
        3.50774858e+00, -5.74832162e-01, -3.56810318e-01, -6.98336596e-01,
       -7.82824691e-02, -9.66960837e-01, -6.11356590e-01,  3.83714876e+00,
       -1.03833745e+00, -5.84670011e-01, -5.57541400e-01, -6.97565773e-01,
       -5.89137051e-01, -5.74832162e-01,  3.70228747e+00, -4.40604151e-01,
       -2.27328078e-01, -1.17282025e+00, -6.84741638e-01, -6.03367586e-01,
       -4.40604151e-01,  3.47455276e+00, -5.75460765e-01, -5.51116403e-01,
       -6.19262217e-01, -5.56533308e-01, -6.37394793e-01, -5.75460765e-01,
        3.45551660e+00, -5.23302269e-01, -5.66886526e-01, -5.95938935e-01,
       -5.46875550e-01, -6.51974710e-01, -5.74602647e-01,  3.48225393e+00,
       -5.03121197e-01, -6.04674616e-01, -6.01005210e-01, -6.56184473e-01,
       -8.01700203e-01, -4.81191539e-01,  3.64755023e+00, -4.69993362e-01,
       -7.04053812e-01, -5.34426842e-01, -5.17179024e-01, -6.73061750e-01,
       -4.69993362e-01,  3.42660756e+00, -5.53792273e-01, -6.89269009e-01,
       -5.23312145e-01, -5.22024206e-01, -6.09572489e-01, -7.04053812e-01,
        3.52182000e+00, -5.89513656e-01, -5.39611995e-01, -5.57043846e-01,
       -5.06211667e-01, -5.94611674e-01, -5.89513656e-01,  3.30208962e+00,
       -5.86206094e-01, -5.60201765e-01, -4.65344761e-01, -5.81964831e-01,
       -5.08482042e-01, -5.86206094e-01,  3.52713609e+00, -6.36540626e-01,
       -6.34474282e-01, -5.79468212e-01, -5.72401008e-01, -5.15739306e-01,
       -6.36540626e-01,  3.45527337e+00, -6.26067909e-01, -5.30100523e-01,
       -5.74423996e-01, -5.17208528e-01, -5.68527489e-01, -6.71775529e-01,
        3.47233252e+00, -5.91885698e-01, -5.64130898e-01, -5.58804375e-01,
       -2.78682752e-01, -9.56133806e-01, -6.22086692e-01,  3.61676121e+00,
       -4.54971457e-01, -7.83325772e-01, -5.21560732e-01, -3.28963488e-01,
       -1.24056175e+00, -6.28707212e-01,  3.79629509e+00, -1.65761240e-01,
       -6.25533593e-01, -8.06767805e-01, -4.15869984e-01, -5.74329416e-01,
       -7.73829496e-01,  3.52378859e+00, -6.86395686e-01, -6.58345773e-01,
       -4.15018237e-01, -5.22916906e-01, -5.58759798e-01, -5.89971565e-01,
        3.48519184e+00, -6.45380135e-01, -5.22130323e-01, -6.46033112e-01,
       -5.59918445e-01, -5.08610704e-01, -6.45380135e-01,  3.50748095e+00,
       -6.19654388e-01, -5.15559140e-01, -6.58358136e-01, -6.03133856e-01,
       -5.34672059e-01, -5.97118838e-01,  3.46442176e+00, -5.74465994e-01,
       -4.61425741e-01, -6.93605274e-01, -6.13347644e-01, -6.00282611e-01,
       -5.74465994e-01,  3.49296999e+00, -5.76729915e-01, -5.52927339e-01,
       -5.75216482e-01, -5.52644971e-01, -6.11440023e-01, -5.76729915e-01,
        3.51736404e+00, -5.86882850e-01, -6.04127845e-01, -5.85538438e-01,
       -7.18959085e-01, -5.03111403e-01, -4.61425741e-01,  3.56409410e+00,
       -2.74709123e-01, -4.47588948e-01, -6.29446959e-01, -5.28852842e-01,
       -5.60579577e-01, -6.05552343e-01, -5.86882850e-01,  3.48832869e+00,
       -5.72644271e-01, -5.52609090e-01, -6.10060559e-01, -7.41715046e-01,
       -6.60260760e-01, -4.24863553e-01,  3.50759017e+00, -4.40996407e-01,
       -5.52192611e-01, -6.87561797e-01, -5.24648520e-01, -5.91316320e-01,
       -6.16142449e-01,  3.47153319e+00, -6.18044507e-01, -5.77165882e-01,
       -5.44215511e-01, -1.54501831e-01, -7.17571589e-01, -1.03833745e+00,
        3.69746615e+00, -8.18488884e-01, -5.63881994e-01, -4.04684399e-01,
       -6.48598188e-01, -5.23647292e-01, -5.42508461e-01,  3.46844162e+00,
       -5.54531488e-01, -5.27717804e-01, -6.71438387e-01, -6.16483686e-01,
       -5.52055198e-01, -5.54531488e-01,  3.46104118e+00, -5.54843722e-01,
       -5.48256615e-01, -6.34870470e-01, -5.84149402e-01, -5.85974796e-01,
       -5.54843722e-01,  3.44756226e+00, -5.49373724e-01, -5.71120911e-01,
       -6.02099707e-01, -5.53043900e-01, -6.43754368e-01, -5.49373724e-01,
        3.46058756e+00, -5.62611561e-01, -6.14873799e-01, -5.36930206e-01,
       -4.80717040e-01, -6.89556239e-01, -5.62611561e-01,  3.50051138e+00,
       -6.52765378e-01, -7.41316413e-01, -3.73544750e-01, -3.87328872e-01,
       -5.34115375e-01, -6.52765378e-01,  3.70624944e+00, -8.00423694e-01,
       -1.12597815e+00, -2.05637964e-01, -5.19750278e-01, -5.23302269e-01,
       -8.00423694e-01,  3.50100496e+00, -6.32911981e-01, -4.10350815e-01,
       -6.14265920e-01, -5.40008104e-01, -6.22398421e-01,  3.45432440e+00,
       -5.80002028e-01, -5.78921775e-01, -5.18053720e-01, -6.14940352e-01,
       -6.01021298e-01, -5.56475487e-01, -5.80002028e-01,  3.45786895e+00,
       -5.77221625e-01, -5.53300253e-01, -5.89848258e-01, -5.92872078e-01,
       -5.73637172e-01, -5.77221625e-01,  3.45430472e+00, -5.64633383e-01,
       -5.63451144e-01, -5.82489321e-01, -5.08967531e-01, -6.78712041e-01,
       -5.78921775e-01,  3.47142659e+00, -5.49193255e-01, -6.71143243e-01,
       -4.84488743e-01, -5.77192140e-01, -5.98103101e-01, -5.64633383e-01,
        3.46076517e+00, -5.51474743e-01, -5.95957224e-01, -5.73404583e-01,
       -4.98110276e-01, -6.85514614e-01, -5.27717804e-01, -5.49193255e-01,
        3.46776352e+00, -5.08432545e-01, -6.98795026e-01, -5.68239505e-01,
       -6.26862909e-01, -5.51474743e-01,  3.46260356e+00, -5.05971358e-01,
       -6.27085138e-01, -5.82969902e-01, -5.76411862e-01, -6.82932861e-01,
       -5.05971358e-01,  3.46365159e+00, -4.33808831e-01, -6.58020636e-01,
       -6.06506045e-01, -5.51222952e-01, -1.06859637e+00, -1.13632973e-01,
       -4.33808831e-01,  3.72053064e+00, -8.36149157e-01, -7.17120361e-01,
       -6.55450213e-01, -5.73588339e-01, -5.03121197e-01,  3.47355416e+00,
       -5.14444243e-01, -6.15534904e-01, -6.11415266e-01, -5.56707861e-01,
       -6.04674616e-01, -5.91885698e-01,  3.46680024e+00, -5.42272524e-01,
       -5.94338824e-01, -5.76920713e-01, -6.02024694e-01, -5.53792273e-01,
       -5.14444243e-01,  3.47653954e+00, -6.58293150e-01, -7.33202389e-01,
       -4.14782794e-01, -6.06196329e-01, -6.26067909e-01, -5.72644271e-01,
        3.53113934e+00, -5.23514726e-01, -5.97595118e-01, -6.05120986e-01,
       -5.77577645e-01, -5.64130898e-01,  3.47624838e+00, -5.95647599e-01,
       -5.91829739e-01, -5.90408653e-01, -5.56653846e-01, -6.05540050e-01,
       -5.88990767e-01,  3.47960693e+00, -5.33197740e-01, -5.23850188e-01,
       -5.97370080e-01, -6.30658107e-01, -6.36732619e-01, -5.36720603e-01,
       -5.33197740e-01,  3.44736199e+00, -5.36762771e-01, -5.58813301e-01,
       -6.45134959e-01, -6.38751296e-01, -6.15290734e-01, -4.54971457e-01,
       -5.23850188e-01,  3.48546437e+00, -6.92381103e-01, -5.60219594e-01,
       -6.77035884e-01, -4.59880702e-01,  3.46200470e+00, -5.98540713e-01,
       -6.21290935e-01, -4.75134071e-01, -6.30122395e-01, -4.28163268e-01,
       -7.83325772e-01, -5.98540713e-01,  3.50575766e+00, -5.09110422e-01,
       -6.86873093e-01, -4.99744388e-01, -5.74648296e-01, -6.25518067e-01,
       -5.36762771e-01,  3.50304009e+00, -6.50928418e-01, -5.23402437e-01,
       -5.91780097e-01, -6.19728222e-01, -5.15614086e-01, -6.21290935e-01,
        3.47834501e+00, -6.27190039e-01, -5.05270391e-01, -5.89251343e-01,
       -5.86217774e-01, -5.39697081e-01, -6.18044507e-01, -6.27190039e-01,
        3.46994514e+00, -5.69533157e-01, -5.29262579e-01, -4.78104730e-01,
       -6.22425032e-01, -6.50928418e-01,  3.47699685e+00, -6.07126166e-01,
       -5.93083420e-01, -5.25329087e-01, -6.27710607e-01, -5.98820501e-01,
        3.47705933e+00, -4.79240063e-01, -4.96736056e-01, -6.35932736e-01,
       -6.38619361e-01, -6.16195244e-01, -6.29314247e-01, -4.79240063e-01,
        3.47621932e+00, -4.95980383e-01, -6.50711625e-01, -6.04777754e-01,
       -5.80622204e-01, -6.86667368e-01, -4.95980383e-01,  3.48275054e+00,
       -4.86530510e-01, -6.84542853e-01, -5.48407218e-01, -4.96944461e-01,
       -5.73769094e-01, -4.86530510e-01,  3.46021646e+00, -6.60408958e-01,
       -5.99020283e-01, -6.43543152e-01, -6.03050734e-01, -4.67188247e-01,
       -6.60408958e-01,  3.48021510e+00, -6.73769028e-01, -5.01049488e-01,
       -5.74748648e-01, -6.75048142e-01, -5.95878570e-01, -4.96736056e-01,
        3.44619584e+00, -4.26234360e-01, -6.33197394e-01, -6.19101313e-01,
       -5.65845693e-01, -6.36520779e-01, -6.73769028e-01,  3.50841992e+00,
       -5.58390023e-01, -6.70271719e-01, -4.03622674e-01, -6.34884885e-01,
       -6.83291345e-01, -4.26234360e-01,  3.48493688e+00, -4.64382188e-01,
       -7.06637701e-01, -5.69506402e-01, -4.19723881e-01, -6.52912204e-01,
       -5.58390023e-01,  3.49267249e+00, -6.44545708e-01, -6.69818184e-01,
       -5.47282491e-01, -5.10618820e-01, -7.82220035e-01, -4.64382188e-01,
        3.45330097e+00, -5.01137046e-01, -6.89910112e-01, -5.05032769e-01,
       -5.63168423e-01, -5.24515514e-01, -6.07126166e-01, -6.44545708e-01,
        3.47509088e+00, -6.04143918e-01, -5.31591147e-01, -2.98891177e-01,
       -8.90691241e-01, -1.65761240e-01, -5.01137046e-01,  3.50448577e+00,
       -3.56936266e-01, -7.52704126e-01, -5.38364674e-01, -5.68944148e-01,
       -5.87409375e-01, -6.25533593e-01,  3.47346950e+00, -6.10435937e-01,
       -4.85498680e-01, -5.95647768e-01, -5.28187429e-01, -6.86395686e-01,
       -6.10435937e-01,  3.48064915e+00, -5.02355449e-01, -5.40588416e-01,
       -6.12686237e-01, -6.06899858e-01, -5.14542390e-01, -6.19654388e-01,
        3.51717101e+00, -5.19100125e-01, -5.24756701e-01, -7.32217551e-01,
       -7.49390952e-01, -2.74709123e-01, -5.19100125e-01,  3.79770926e+00,
       -9.25444887e-01, -6.80088045e-01, -6.48976129e-01, -6.60157018e-01,
       -6.19926489e-01, -4.40996407e-01,  3.45947600e+00, -5.40652662e-01,
       -5.89577068e-01, -6.08166360e-01, -1.01319332e+00, -2.18183702e-01,
       -5.52192611e-01,  3.62859304e+00, -4.46924719e-01, -7.00911003e-01,
       -6.97187686e-01, -4.04849385e-01, -6.58345773e-01,  3.97639192e+00,
       -8.79168040e-01, -5.47626753e-01, -1.48640197e+00, -5.72299599e-01,
       -5.48468357e-01,  3.44175161e+00, -6.45842501e-01, -5.89838960e-01,
       -5.52558085e-01, -5.32744107e-01, -5.34228743e-01, -5.53036682e-01,
       -6.45842501e-01,  3.44640904e+00, -6.09836405e-01, -5.15848958e-01,
       -5.87615750e-01, -5.63838762e-01, -5.78020503e-01, -6.09836405e-01,
        3.46206497e+00, -5.89388557e-01, -5.30872703e-01, -5.90108039e-01,
       -6.41800788e-01, -5.65734357e-01, -5.84670011e-01, -5.89838960e-01,
        3.49864143e+00, -5.88880956e-01, -5.27716357e-01, -5.46223038e-01,
       -5.77165882e-01, -5.89388557e-01,  3.43424580e+00, -6.00769465e-01,
       -5.50475068e-01, -5.70223788e-01, -7.11421927e-01, -3.63041763e-01,
        3.51445974e+00, -6.68348941e-01, -7.25667706e-01, -4.68523864e-01,
       -5.77455536e-01, -3.57056142e-01, -8.18488884e-01, -6.68348941e-01,
        3.53368896e+00, -6.08083817e-01, -6.76504075e-01, -4.05207099e-01,
       -6.55726359e-01, -3.56810318e-01, -7.25667706e-01,  3.54893232e+00,
       -8.22817654e-01, -5.21588530e-01, -4.66321751e-01, -6.98336596e-01,
       -2.27328078e-01, -8.22817654e-01,  3.58523714e+00, -8.30819023e-01,
       -5.59689823e-01, -4.46245969e-01, -5.57541400e-01, -5.63881994e-01,
       -5.88880956e-01,  3.53487193e+00, -6.28956934e-01, -6.75978568e-01,
       -5.19632076e-01, -1.17282025e+00, -5.51116403e-01, -8.30819023e-01,
        3.75554390e+00, -5.24841123e-01, -6.75947095e-01, -6.19262217e-01,
       -5.66886526e-01, -5.24841123e-01,  3.46053747e+00, -5.57761348e-01,
       -5.59146462e-01, -6.32639793e-01, -5.95938935e-01, -6.32911981e-01,
       -5.57761348e-01,  3.45495985e+00, -5.09212067e-01, -5.53966803e-01,
       -6.05168719e-01, -4.60906472e-01, -8.36149157e-01, -5.95647599e-01,
        3.52097742e+00, -4.16245584e-01, -6.04021737e-01, -6.08006874e-01,
       -6.01005210e-01, -6.15534904e-01, -5.42272524e-01,  3.47565405e+00,
       -4.80583410e-01, -5.68839795e-01, -6.67418211e-01, -5.34426842e-01,
       -6.89269009e-01, -5.39611995e-01,  3.48186329e+00, -5.53164348e-01,
       -6.81922579e-01, -4.83468520e-01, -5.23312145e-01, -6.58293150e-01,
       -5.53164348e-01,  3.51253297e+00, -6.41022437e-01, -6.82496782e-01,
       -4.54244110e-01, -5.57043846e-01, -5.60201765e-01, -6.81922579e-01,
        3.55921670e+00, -7.22961369e-01, -5.95731271e-01, -4.41355873e-01,
       -4.65344761e-01, -6.34474282e-01, -7.22961369e-01,  3.50627190e+00,
       -5.93831189e-01, -6.38482415e-01, -4.51177881e-01, -5.79468212e-01,
       -5.30100523e-01, -5.93831189e-01,  3.50550547e+00, -6.25140889e-01,
       -6.00047681e-01, -5.76916973e-01, -5.74423996e-01, -5.23514726e-01,
       -6.25140889e-01,  3.46707249e+00, -6.23714048e-01, -5.47082141e-01,
       -5.73196696e-01, -5.58804375e-01, -5.94338824e-01, -5.91829739e-01,
        3.47354067e+00, -5.76649399e-01, -5.94082548e-01, -5.57835789e-01,
       -5.21560732e-01, -6.92381103e-01, -5.09110422e-01,  3.48394991e+00,
       -5.24958503e-01, -6.98024080e-01, -5.37915072e-01, -8.06767805e-01,
       -3.56936266e-01, -4.85498680e-01,  3.50385572e+00, -6.82251705e-01,
       -6.28013765e-01, -5.44387496e-01, -4.15018237e-01, -5.02355449e-01,
       -8.79168040e-01,  3.51571138e+00, -6.33620122e-01, -3.74546685e-02,
       -5.96109297e-01, -4.51985561e-01, -5.94253562e-01, -5.22130323e-01,
       -5.40652662e-01,  3.34294991e+00, -4.40833666e-01, -6.30475014e-01,
       -6.14604686e-01, -6.46033112e-01, -5.15559140e-01, -4.40833666e-01,
        3.43635033e+00, -5.18287788e-01, -6.00138065e-01, -7.15498563e-01,
       -6.58358136e-01, -5.24756701e-01, -5.18287788e-01,  3.47482311e+00,
       -4.80827777e-01, -5.13181212e-01, -7.79411499e-01, -6.93605274e-01,
       -5.52927339e-01, -4.47588948e-01,  3.49274027e+00, -5.41007208e-01,
       -6.80993935e-01, -5.76617562e-01, -5.75216482e-01, -6.04127845e-01,
       -5.41007208e-01,  3.49139401e+00, -5.57945330e-01, -6.20480595e-01,
       -5.92616551e-01, -5.85538438e-01, -5.52609090e-01, -5.57945330e-01,
        3.48577031e+00, -5.35110041e-01, -5.56671145e-01, -6.97896265e-01,
       -6.29446959e-01, -9.25444887e-01,  3.70014570e+00, -5.78452997e-01,
       -2.74656385e-01, -4.85686649e-01, -8.06457818e-01, -6.10060559e-01,
       -5.97595118e-01, -5.35110041e-01,  3.48848608e+00, -5.16773742e-01,
       -5.23032571e-01, -7.05914049e-01, -6.87561797e-01, -5.89577068e-01,
       -4.46924719e-01,  3.48165459e+00, -5.45384653e-01, -5.71065457e-01,
       -6.41140897e-01, -6.00665769e-01, -7.00911003e-01, -5.47626753e-01,
        3.52359774e+00, -4.31102824e-01, -4.92109408e-01, -7.51181982e-01,
       -5.44215511e-01, -5.69533157e-01, -6.00769465e-01,  3.45128321e+00,
       -6.25710209e-01, -5.59789338e-01, -5.51265532e-01, -4.04684399e-01,
       -6.08083817e-01, -6.28956934e-01,  3.60180830e+00, -8.99918335e-01,
       -8.92037021e-01, -1.68127796e-01, -6.71438387e-01, -5.48256615e-01,
       -5.08432545e-01,  3.46874580e+00, -5.28190104e-01, -5.42540119e-01,
       -6.69888030e-01, -6.34870470e-01, -5.71120911e-01, -5.28190104e-01,
        3.46224604e+00, -5.30279906e-01, -5.56898149e-01, -6.40886501e-01,
       -6.02099707e-01, -6.14873799e-01, -5.30279906e-01,  3.45808368e+00,
       -5.19424504e-01, -5.82645513e-01, -6.08760254e-01, -5.36930206e-01,
       -7.41316413e-01, -5.19424504e-01,  3.47707817e+00, -4.93103919e-01,
       -6.56629789e-01, -5.29673344e-01, -3.73544750e-01, -1.12597815e+00,
       -4.93103919e-01,  3.82842432e+00, -8.16751239e-01, -1.01904626e+00,
       -2.05637964e-01, -4.10350815e-01, -8.16751239e-01,  3.52135292e+00,
       -6.34894349e-01, -2.71393191e-01, -6.29222349e-01, -5.53103011e-01,
       -6.14265920e-01, -5.09212067e-01, -6.34894349e-01,  3.48629008e+00,
       -5.95707528e-01, -5.19933120e-01, -6.12277099e-01, -5.18053720e-01,
       -6.71143243e-01,  3.46948963e+00, -5.72108058e-01, -5.55941046e-01,
       -5.06227173e-01, -6.46016385e-01, -6.14940352e-01, -5.53300253e-01,
       -5.72108058e-01,  3.46558520e+00, -5.82385361e-01, -5.34845359e-01,
       -6.08005821e-01, -5.89848258e-01, -5.63451144e-01, -5.82385361e-01,
        3.45171174e+00, -5.73586078e-01, -5.60447022e-01, -5.81993877e-01,
       -5.82489321e-01, -5.95957224e-01, -5.73586078e-01,  3.44500928e+00,
       -5.47153684e-01, -5.80527700e-01, -5.65295278e-01, -4.84488743e-01,
       -6.98795026e-01, -5.42540119e-01, -5.55941046e-01,  3.47046407e+00,
       -5.08054876e-01, -6.80644261e-01, -5.73404583e-01, -6.27085138e-01,
       -5.47153684e-01,  3.46470766e+00, -5.17319007e-01, -6.22309279e-01,
       -5.77435972e-01, -5.82969902e-01, -6.58020636e-01, -5.17319007e-01,
        3.47458254e+00, -4.74075501e-01, -6.42013522e-01, -6.00183969e-01,
       -6.06506045e-01, -7.17120361e-01, -4.16245584e-01, -4.74075501e-01,
        3.48078400e+00, -6.29685579e-01, -6.37150926e-01, -6.11415266e-01,
       -7.33202389e-01, -4.80583410e-01,  3.86125531e+00, -1.61886142e-01,
       -4.58347576e-01, -1.41582053e+00, -5.76920713e-01, -5.68839795e-01,
       -5.76649399e-01,  3.46895244e+00, -5.55268663e-01, -5.74752361e-01,
       -6.16521509e-01, -4.14782794e-01, -6.41022437e-01, -1.61886142e-01,
        3.55924083e+00, -6.85637600e-01, -8.54905491e-01, -4.58010792e-01,
       -3.42995569e-01, -6.05120986e-01, -6.23714048e-01, -5.16773742e-01,
        3.42555101e+00, -5.10592020e-01, -5.29797800e-01, -6.39552411e-01,
       -5.90408653e-01, -6.04021737e-01,  3.48634968e+00, -5.75578373e-01,
       -5.33858909e-01, -6.49811776e-01, -5.32670232e-01, -5.56653846e-01,
       -5.94082548e-01, -5.75578373e-01,  3.46712617e+00, -5.93772252e-01,
       -6.19469872e-01, -5.27569284e-01, -5.97370080e-01, -5.58813301e-01,
        3.45362322e+00, -5.39486170e-01, -5.62136288e-01, -6.09317270e-01,
       -5.86500111e-01, -6.30658107e-01, -5.60219594e-01, -5.24958503e-01,
       -5.39486170e-01,  3.46934145e+00, -6.45372037e-01, -5.68647037e-01,
       -6.45134959e-01, -5.23402437e-01, -5.62136288e-01,  3.45216618e+00,
       -5.51564009e-01, -5.24684364e-01, -6.45244122e-01, -4.75134071e-01,
       -6.86873093e-01,  3.47979422e+00, -5.99602231e-01, -5.69334489e-01,
       -5.06888651e-01, -6.41961684e-01, -6.30122395e-01, -5.05270391e-01,
       -5.99602231e-01,  3.48089544e+00, -6.44033984e-01, -5.07412361e-01,
       -5.94454076e-01, -4.99744388e-01, -6.98024080e-01, -5.69334489e-01,
        3.47890849e+00, -5.46277408e-01, -6.41332479e-01, -5.24195648e-01,
       -5.91780097e-01, -5.93083420e-01, -5.51564009e-01,  3.45519615e+00,
       -5.94903487e-01, -5.09369482e-01, -6.14495654e-01, -5.89251343e-01,
       -5.29262579e-01, -6.25710209e-01, -6.44033984e-01,  3.46562287e+00,
       -5.61952844e-01, -5.15411910e-01, -5.25329087e-01, -6.04143918e-01,
       -5.94903487e-01,  3.47571268e+00, -6.22243552e-01, -5.80642669e-01,
       -5.48449962e-01, -6.35932736e-01, -6.50711625e-01,  3.49055763e+00,
       -4.36115118e-01, -4.72755966e-01, -7.15411765e-01, -5.79630421e-01,
       -6.38619361e-01, -6.33197394e-01, -4.36115118e-01,  3.37880800e+00,
       -4.31966557e-01, -6.26475588e-01, -6.12433984e-01, -6.04777754e-01,
       -6.84542853e-01, -4.72755966e-01,  3.48337786e+00, -4.72922215e-01,
       -6.66959201e-01, -5.81419868e-01, -5.48407218e-01, -5.99020283e-01,
       -4.72922215e-01,  3.43235311e+00, -5.86482171e-01, -6.08186957e-01,
       -6.17334268e-01, -6.43543152e-01, -5.01049488e-01, -5.86482171e-01,
        3.47371967e+00, -6.14811998e-01, -5.36862078e-01, -5.90970788e-01,
       -5.74748648e-01, -6.70271719e-01, -6.14811998e-01,  3.53513084e+00,
       -5.25490113e-01, -7.36299616e-01, -4.13508746e-01, -6.19101313e-01,
       -7.06637701e-01, -4.31966557e-01,  3.51383093e+00, -4.86551409e-01,
       -6.53826001e-01, -6.15747945e-01, -4.03622674e-01, -6.69818184e-01,
       -5.25490113e-01,  3.48798351e+00, -6.55515310e-01, -7.06413434e-01,
       -5.27123799e-01, -5.69506402e-01, -6.89910112e-01, -4.86551409e-01,
        3.48367382e+00, -5.67153918e-01, -5.81524380e-01, -5.89027599e-01,
       -5.47282491e-01, -5.31591147e-01, -6.22243552e-01, -6.55515310e-01,
        3.47335457e+00, -5.72043512e-01, -5.44678558e-01, -5.05032769e-01,
       -7.52704126e-01, -5.67153918e-01,  3.52082741e+00, -5.19703159e-01,
       -5.86278634e-01, -5.89954802e-01, -5.38364674e-01, -6.82251705e-01,
       -5.19703159e-01,  3.50382832e+00, -6.42478417e-01, -6.40578138e-01,
       -4.80452223e-01, -5.95647768e-01, -5.40588416e-01, -6.28013765e-01,
        3.48445266e+00, -6.12423675e-01, -5.30655327e-01, -5.77123705e-01,
       -6.12686237e-01, -6.33620122e-01, -6.12423675e-01,  3.48576286e+00,
       -4.80240129e-01, -4.92378840e-01, -6.54413854e-01, -7.32217551e-01,
       -6.80088045e-01, -4.80827777e-01,  3.62695733e+00, -4.90818367e-01,
       -4.83706623e-01, -7.59298970e-01, -5.28852842e-01, -6.80993935e-01,
       -5.78452997e-01,  3.52226192e+00, -5.78281999e-01, -7.20629654e-01,
       -4.35050497e-01, -6.48976129e-01, -2.74656385e-01, -4.90818367e-01,
        3.54522646e+00, -4.65415210e-01, -4.00952663e-01, -7.74140426e-01,
       -4.90267280e-01, -6.08166360e-01, -6.30475014e-01, -5.45384653e-01,
        3.47241892e+00, -5.03063080e-01, -5.90474911e-01, -5.94854898e-01,
       -6.97187686e-01, -5.71065457e-01, -4.31102824e-01,  3.49089980e+00,
       -5.07415926e-01, -5.67341800e-01, -7.16786107e-01, -1.48640197e+00,
       -3.74546685e-02, -4.92109408e-01,  3.96873222e+00, -3.98733626e-01,
       -7.75072749e-01, -7.78959800e-01, -5.52558085e-01, -5.15848958e-01,
        3.47677527e+00, -6.85154189e-01, -6.41653926e-01, -6.02991280e-01,
       -4.78568831e-01, -5.32744107e-01, -5.27716357e-01, -6.75978568e-01,
       -6.85154189e-01,  3.48659922e+00, -5.00110821e-01, -5.64895175e-01,
       -5.87615750e-01, -5.30872703e-01, -6.41653926e-01,  3.47090640e+00,
       -6.04139329e-01, -5.09075428e-01, -5.97549262e-01, -5.90108039e-01,
       -5.50475068e-01, -6.04139329e-01,  3.47836297e+00, -6.08444128e-01,
       -5.55928135e-01, -5.69268268e-01, -5.70223788e-01, -5.59789338e-01,
       -6.08444128e-01,  3.46778944e+00, -6.21622966e-01, -5.57138817e-01,
       -5.50570405e-01, -4.68523864e-01, -6.76504075e-01,  3.47442646e+00,
       -6.33701403e-01, -5.92349444e-01, -5.39613242e-01, -5.63734435e-01,
       -5.77455536e-01, -5.21588530e-01, -6.33701403e-01,  3.47936683e+00,
       -6.11132040e-01, -5.98920891e-01, -5.36568432e-01, -4.05207099e-01,
       -8.99918335e-01, -5.92349444e-01,  3.57417161e+00, -3.68254131e-01,
       -7.05176512e-01, -6.03266087e-01, -4.66321751e-01, -5.59689823e-01,
       -6.11132040e-01,  3.37817260e+00, -5.54662310e-01, -6.47361700e-01,
       -5.39004976e-01, -4.46245969e-01, -6.75947095e-01, -5.59146462e-01,
       -5.54662310e-01,  3.47373689e+00, -5.57410275e-01, -6.80324777e-01,
       -5.19632076e-01, -8.92037021e-01, -5.00110821e-01,  3.75349154e+00,
       -1.00925468e+00, -8.32456940e-01, -6.32639793e-01, -5.53966803e-01,
       -5.57410275e-01,  3.46702022e+00, -5.56733754e-01, -5.06605249e-01,
       -6.59664344e-01, -6.05168719e-01, -5.95707528e-01, -5.56733754e-01,
        3.47441550e+00, -5.59922302e-01, -5.35277751e-01, -6.21605445e-01,
       -6.08006874e-01, -6.29685579e-01, -5.33858909e-01,  3.47453710e+00,
       -5.02265616e-01, -5.51613835e-01, -6.49106291e-01, -6.67418211e-01,
       -4.58347576e-01, -5.55268663e-01,  3.51083270e+00, -5.66001732e-01,
       -5.12394322e-01, -7.51402196e-01, -4.83468520e-01, -6.82496782e-01,
       -5.95731271e-01,  3.48838555e+00, -6.10962085e-01, -6.64101846e-01,
       -4.51625050e-01, -4.54244110e-01, -6.85637600e-01, -6.10962085e-01,
        3.51150604e+00, -5.84335040e-01, -6.55948659e-01, -5.20378550e-01,
       -4.41355873e-01, -6.38482415e-01, -6.64101846e-01,  3.52245420e+00,
       -6.27287377e-01, -6.19304271e-01, -5.31922416e-01, -4.51177881e-01,
       -6.00047681e-01, -6.27287377e-01,  3.32817607e+00, -5.82210063e-01,
       -5.69451085e-01, -4.98001986e-01, -5.76916973e-01, -5.47082141e-01,
       -5.82210063e-01,  3.46590494e+00, -6.04662535e-01, -6.20324774e-01,
       -5.34708459e-01, -5.73196696e-01, -5.10592020e-01, -6.04662535e-01,
        3.43714629e+00, -6.07486839e-01, -5.54749564e-01, -5.86458641e-01,
       -5.57835789e-01, -5.74752361e-01, -5.93772252e-01,  3.47572016e+00,
       -5.91038664e-01, -5.96327958e-01, -5.61993133e-01, -5.37915072e-01,
       -6.45372037e-01, -5.46277408e-01,  3.47607745e+00, -5.58777996e-01,
       -6.28620539e-01, -5.59114401e-01, -5.44387496e-01, -6.42478417e-01,
       -5.30655327e-01,  3.51839637e+00, -5.80248258e-01, -7.93462656e-01,
       -4.27164215e-01, -5.96109297e-01, -4.80240129e-01,  3.55799509e+00,
       -8.56307434e-01, -5.90373400e-01, -7.14063510e-01, -3.20901324e-01,
       -6.14604686e-01, -6.00138065e-01, -5.03063080e-01,  3.31768799e+00,
       -3.35813787e-01, -7.53119165e-01, -5.10949207e-01, -7.15498563e-01,
       -5.13181212e-01, -3.35813787e-01,  3.37119787e+00, -5.24488561e-01,
       -7.11494488e-01, -5.70721262e-01, -7.79411499e-01, -4.83706623e-01,
       -5.24488561e-01,  3.47584658e+00, -4.64516676e-01, -4.52452643e-01,
       -7.71270575e-01, -5.76617562e-01, -6.20480595e-01, -5.78281999e-01,
        3.55719107e+00, -5.53409819e-01, -6.56281201e-01, -5.72119897e-01,
       -5.92616551e-01, -5.56671145e-01, -5.53409819e-01,  3.52379981e+00,
       -5.05812654e-01, -5.89583597e-01, -7.25706047e-01, -6.97896265e-01,
       -5.23032571e-01, -5.05812654e-01,  3.50021685e+00, -5.08876429e-01,
       -5.41654569e-01, -7.22944363e-01, -4.85686649e-01, -7.20629654e-01,
        3.62524132e+00, -5.65482055e-01, -7.23756179e-01, -2.63246176e-01,
       -8.66440602e-01, -8.06457818e-01, -4.65415210e-01, -5.65482055e-01,
        3.65040669e+00, -5.79728654e-01, -4.32835215e-01, -8.00487740e-01,
       -7.05914049e-01, -5.29797800e-01, -5.08876429e-01,  3.48921797e+00,
       -5.11750742e-01, -5.32232194e-01, -7.00646756e-01, -6.41140897e-01,
       -5.90474911e-01, -5.07415926e-01,  3.46732836e+00, -5.36488667e-01,
       -5.46156619e-01, -6.45651341e-01, -7.51181982e-01, -5.67341800e-01,
       -3.98733626e-01,  3.50631885e+00, -5.02149350e-01, -5.49964225e-01,
       -7.36947863e-01, -4.51985561e-01, -7.75072749e-01, -8.56307434e-01,
        3.77000452e+00, -3.94039163e-01, -1.91423210e-01, -1.10117641e+00,
       -5.51265532e-01, -5.61952844e-01, -6.21622966e-01,  3.46439698e+00,
       -6.47820161e-01, -5.52223524e-01, -5.29511952e-01, -1.68127796e-01,
       -3.68254131e-01, -1.00925468e+00,  3.62067869e+00, -2.15652624e-01,
       -6.97969992e-01, -6.79292566e-01, -4.82126897e-01, -6.69888030e-01,
       -5.56898149e-01, -5.08054876e-01,  3.46836896e+00, -5.22333385e-01,
       -5.53001217e-01, -6.58193304e-01, -6.40886501e-01, -5.82645513e-01,
       -5.22333385e-01,  3.46369629e+00, -5.06544692e-01, -5.68555694e-01,
       -6.42730509e-01, -6.08760254e-01, -6.56629789e-01, -5.06544692e-01,
        3.46735840e+00, -4.68697972e-01, -6.01697903e-01, -6.25027788e-01,
       -5.29673344e-01, -1.01904626e+00, -2.71393191e-01, -4.68697972e-01,
        3.61147835e+00, -6.82689470e-01, -6.39978113e-01, -6.29222349e-01,
       -5.19933120e-01,  3.43795022e+00, -5.62132930e-01, -5.51532373e-01,
       -5.95607590e-01, -5.79521855e-01, -6.12277099e-01, -5.59922302e-01,
       -5.62132930e-01,  3.46978384e+00, -5.30966145e-01, -5.88175150e-01,
       -6.16310212e-01, -5.06227173e-01, -6.80644261e-01, -5.53001217e-01,
        3.46716481e+00, -5.62292321e-01, -5.18177769e-01, -6.46822069e-01,
       -6.46016385e-01, -5.34845359e-01, -5.62292321e-01,  3.46875548e+00,
       -5.76992901e-01, -5.26550579e-01, -6.22057930e-01, -6.08005821e-01,
       -5.60447022e-01, -5.76992901e-01,  3.45939773e+00, -5.81585715e-01,
       -5.45366014e-01, -5.87000256e-01, -5.81993877e-01, -5.80527700e-01,
       -5.81585715e-01,  3.44951384e+00, -5.71202827e-01, -5.71868721e-01,
       -5.62335003e-01, -5.65295278e-01, -6.22309279e-01, -5.71202827e-01,
        3.45879107e+00, -5.46919697e-01, -6.05924158e-01, -5.47139833e-01,
       -5.77435972e-01, -6.42013522e-01, -5.46919697e-01,  3.46920971e+00,
       -5.15991291e-01, -6.25961641e-01, -5.60887586e-01, -6.00183969e-01,
       -6.37150926e-01, -5.02265616e-01, -5.15991291e-01,  3.47023987e+00,
       -5.66298284e-01, -6.48349781e-01, -1.41582053e+00, -8.54905491e-01,
       -5.66001732e-01,  3.94264357e+00, -3.47092855e-01, -7.58822966e-01,
       -6.16521509e-01, -5.12394322e-01, -5.91038664e-01,  3.50000963e+00,
       -5.78532397e-01, -5.47998432e-01, -6.53524307e-01, -4.58010792e-01,
       -5.84335040e-01,  3.53008286e+00, -7.82442118e-01, -6.15990689e-01,
       -4.92409582e-01, -5.96894641e-01, -6.39552411e-01, -6.07486839e-01,
       -5.11750742e-01,  3.47210473e+00, -5.11214972e-01, -5.73341942e-01,
       -6.28757824e-01, -6.49811776e-01, -5.51613835e-01,  3.51237644e+00,
       -5.73463954e-01, -5.44922710e-01, -6.74723457e-01, -5.17840704e-01,
       -5.32670232e-01, -6.19469872e-01, -5.73463954e-01,  3.49068636e+00,
       -6.06605496e-01, -6.45170723e-01, -5.13306083e-01, -5.27569284e-01,
       -5.96327958e-01, -6.06605496e-01,  3.47008916e+00, -6.09103086e-01,
       -6.10053073e-01, -5.20430260e-01, -6.09317270e-01, -5.24684364e-01,
        3.45359307e+00, -5.94486929e-01, -5.91981561e-01, -5.81497071e-01,
       -5.51625870e-01, -5.86500111e-01, -5.68647037e-01, -5.58777996e-01,
       -5.94486929e-01,  3.45557163e+00, -5.96213453e-01, -5.50946107e-01,
       -6.45244122e-01, -5.09369482e-01, -5.91981561e-01,  3.47192188e+00,
       -5.97857384e-01, -5.26636849e-01, -6.00832485e-01, -5.06888651e-01,
       -6.41332479e-01,  3.47048426e+00, -6.10525092e-01, -5.78585040e-01,
       -6.05283565e-01, -5.27869438e-01, -6.41961684e-01, -5.07412361e-01,
       -6.10525092e-01,  3.47300334e+00, -6.31380040e-01, -5.02195997e-01,
       -5.79528171e-01, -5.94454076e-01, -5.15411910e-01, -6.47820161e-01,
       -6.31380040e-01,  3.46980786e+00, -5.62820184e-01, -5.17921488e-01,
       -5.24195648e-01, -6.28620539e-01, -5.78585040e-01,  3.44768215e+00,
       -5.64121483e-01, -6.00163889e-01, -5.51995547e-01, -6.14495654e-01,
       -5.80642669e-01, -5.97857384e-01,  3.56793838e+00, -6.26711371e-01,
       -4.97928643e-01, -6.50302658e-01, -5.48449962e-01, -5.72043512e-01,
       -6.26711371e-01,  3.46207154e+00, -6.49047312e-01, -4.64754794e-01,
       -6.01064585e-01, -7.15411765e-01, -6.26475588e-01,  3.40095225e+00,
       -5.18744789e-01, -4.77802178e-01, -5.53187685e-01, -5.09330242e-01,
       -5.79630421e-01, -6.66959201e-01, -5.18744789e-01,  3.37686648e+00,
       -5.28111265e-01, -5.79446075e-01, -5.03974733e-01, -6.12433984e-01,
       -6.53826001e-01, -4.77802178e-01,  3.56382006e+00, -5.71438472e-01,
       -6.23480419e-01, -6.24839002e-01, -5.81419868e-01, -6.08186957e-01,
       -5.28111265e-01,  3.41029599e+00, -5.79021728e-01, -5.85812790e-01,
       -5.27743381e-01, -6.17334268e-01, -5.36862078e-01, -5.79021728e-01,
        3.48191563e+00, -6.22545309e-01, -5.94825016e-01, -5.31327230e-01,
       -5.90970788e-01, -7.36299616e-01, -6.22545309e-01,  3.54256615e+00,
       -4.28049088e-01, -7.05823681e-01, -4.58877673e-01, -4.13508746e-01,
       -7.06413434e-01, -4.28049088e-01,  3.85236926e+00, -3.68844612e-01,
       -5.31057277e-01, -1.40449611e+00, -6.15747945e-01, -5.81524380e-01,
       -5.71438472e-01,  3.50848209e+00, -6.78959823e-01, -6.08544194e-01,
       -4.52267278e-01, -5.27123799e-01, -5.44678558e-01, -6.49047312e-01,
       -3.68844612e-01,  3.47530797e+00, -5.21556824e-01, -6.66066822e-01,
       -1.97990045e-01, -5.89027599e-01, -5.86278634e-01, -6.78959823e-01,
        3.54943207e+00, -5.29095998e-01, -3.24835569e-01, -8.41234450e-01,
       -5.89954802e-01, -6.40578138e-01, -5.29095998e-01,  3.47109668e+00,
       -5.49973302e-01, -5.40579354e-01, -6.20915082e-01, -4.80452223e-01,
       -5.80248258e-01, -5.49973302e-01,  3.68229073e+00, -1.11914505e+00,
       -7.24016104e-01, -2.28455798e-01, -5.77123705e-01, -4.92378840e-01,
       -7.93462656e-01,  3.51126217e+00, -6.68574575e-01, -3.79319570e-01,
       -6.00402826e-01, -6.54413854e-01, -5.90373400e-01, -6.68574575e-01,
        3.50476311e+00, -4.40690761e-01, -4.13322906e-01, -7.37387612e-01,
       -7.59298970e-01, -4.00952663e-01, -4.64516676e-01,  3.46123740e+00,
       -5.80478023e-01, -7.21104768e-01, -5.34886301e-01, -4.35050497e-01,
       -6.56281201e-01, -7.23756179e-01,  3.62715446e+00, -6.04938935e-01,
       -5.82431493e-01, -6.24696157e-01, -7.74140426e-01, -5.79728654e-01,
        3.59285332e+00, -4.83879749e-01, -4.47862522e-01, -7.15682874e-01,
       -5.91559097e-01, -5.94854898e-01, -7.53119165e-01, -5.36488667e-01,
        3.48546380e+00, -4.62511995e-01, -4.74723400e-01, -6.63765676e-01,
       -7.16786107e-01, -5.46156619e-01, -5.02149350e-01,  3.49410360e+00,
       -5.25860825e-01, -4.62464145e-01, -7.40686552e-01, -7.78959800e-01,
       -5.49964225e-01, -3.94039163e-01,  3.51981850e+00, -4.93645965e-01,
       -5.24973888e-01, -7.78235455e-01, -6.02991280e-01, -5.09075428e-01,
        3.46264019e+00, -6.32667487e-01, -5.92857715e-01, -5.12034168e-01,
       -6.13014108e-01, -4.78568831e-01, -5.64895175e-01, -8.32456940e-01,
       -2.15652624e-01, -6.32667487e-01,  3.48007167e+00, -7.55830609e-01,
       -5.97549262e-01, -5.55928135e-01, -5.92857715e-01,  3.46980830e+00,
       -5.90576842e-01, -5.44506217e-01, -5.88390125e-01, -5.69268268e-01,
       -5.57138817e-01, -5.90576842e-01,  3.47030567e+00, -6.30602441e-01,
       -5.73953609e-01, -5.48765694e-01, -5.50570405e-01, -5.52223524e-01,
       -6.30602441e-01,  3.47540341e+00, -6.65431240e-01, -5.56380150e-01,
       -5.20195645e-01, -5.39613242e-01, -7.05176512e-01,  3.52053045e+00,
       -5.75468224e-01, -5.17766841e-01, -6.22539140e-01, -5.59966492e-01,
       -5.63734435e-01, -5.98920891e-01, -5.75468224e-01,  3.46933340e+00,
       -5.49066820e-01, -6.43746193e-01, -5.38396835e-01, -5.36568432e-01,
       -6.47361700e-01, -5.49066820e-01,  3.51178266e+00, -5.10280882e-01,
       -6.91226240e-01, -5.77278580e-01, -6.03266087e-01, -6.97969992e-01,
       -5.17766841e-01,  3.53005910e+00, -4.04826153e-01, -6.07339833e-01,
       -6.98890190e-01, -5.39004976e-01, -6.80324777e-01, -5.06605249e-01,
       -5.10280882e-01,  3.46404746e+00, -5.49865158e-01, -6.77966416e-01,
       -6.59664344e-01, -5.35277751e-01, -5.49865158e-01,  3.47346807e+00,
       -5.84884419e-01, -5.21431327e-01, -6.22345076e-01, -5.53103011e-01,
       -6.82689470e-01, -5.51532373e-01,  3.47620704e+00, -4.82207426e-01,
       -6.04480999e-01, -6.02193756e-01, -6.21605445e-01, -5.30966145e-01,
       -5.84884419e-01,  3.46876963e+00, -6.08246921e-01, -5.19689836e-01,
       -6.03376867e-01, -6.49106291e-01, -5.66298284e-01, -5.44922710e-01,
        3.47397271e+00, -5.30954501e-01, -5.15559247e-01, -6.67131678e-01,
       -7.51402196e-01, -3.47092855e-01, -5.78532397e-01,  3.47442859e+00,
       -6.69242477e-01, -4.86736298e-01, -6.41422372e-01, -4.51625050e-01,
       -6.55948659e-01, -6.19304271e-01,  3.50416329e+00, -6.40484856e-01,
       -6.15663615e-01, -5.21136839e-01, -3.42995569e-01, -7.58822966e-01,
       -7.82442118e-01, -6.69242477e-01,  3.55985519e+00, -5.42200155e-01,
       -4.64151909e-01, -5.20378550e-01, -6.15990689e-01, -6.40484856e-01,
        3.50519708e+00, -5.72494980e-01, -5.53934143e-01, -6.01913858e-01,
       -5.31922416e-01, -5.69451085e-01, -6.15663615e-01,  3.51167897e+00,
       -6.30360021e-01, -5.68838363e-01, -5.95443466e-01, -4.98001986e-01,
       -6.20324774e-01, -6.30360021e-01,  3.46811798e+00, -6.29347778e-01,
       -5.76288916e-01, -5.13794507e-01, -5.34708459e-01, -5.54749564e-01,
       -6.29347778e-01,  3.44768653e+00, -6.31885552e-01, -5.57217258e-01,
       -5.39777924e-01, -5.86458641e-01, -5.11214972e-01, -6.31885552e-01,
        3.48298804e+00, -6.54432240e-01, -5.43294107e-01, -5.55702526e-01,
       -5.61993133e-01, -5.47998432e-01, -6.09103086e-01,  3.47240364e+00,
       -5.80447803e-01, -5.93213685e-01, -5.79647500e-01, -5.59114401e-01,
       -5.96213453e-01, -5.64121483e-01,  3.46769757e+00, -6.05217414e-01,
       -5.98021043e-01, -5.45009780e-01, -4.27164215e-01, -1.11914505e+00,
       -3.79319570e-01,  3.89167487e+00, -8.34029508e-01, -1.13201652e+00,
       -7.14063510e-01, -4.40690761e-01,  4.14352861e+00, -6.96454548e-01,
       -4.67230322e-01, -5.86758670e-01, -6.02891836e-01, -6.35438960e-01,
       -3.20901324e-01, -1.91423210e-01, -6.96454548e-01,  3.87397718e+00,
       -7.85524865e-01, -7.54221087e-01, -8.28094016e-01, -2.97442492e-01,
        8.43588871e-05, -5.10949207e-01, -7.11494488e-01, -4.62511995e-01,
        3.47511103e+00, -6.40750530e-01, -7.76211790e-01, -3.73193021e-01,
       -5.70721262e-01, -4.52452643e-01, -6.40750530e-01,  3.43241871e+00,
       -6.70702960e-01, -7.08565665e-01, -3.89225646e-01, -7.71270575e-01,
       -5.80478023e-01, -6.70702960e-01,  3.41337157e+00, -6.09600417e-01,
       -7.81319597e-01, -4.90267280e-01, -7.21104768e-01, -4.83879749e-01,
        3.47240257e+00, -5.26327833e-01, -6.79837943e-01, -5.70984996e-01,
       -5.72119897e-01, -5.89583597e-01, -6.04938935e-01,  3.49157922e+00,
       -4.19368221e-01, -6.79194820e-01, -6.26373746e-01, -7.25706047e-01,
       -5.41654569e-01, -4.19368221e-01,  3.50265568e+00, -4.79215578e-01,
       -6.25673469e-01, -7.11037797e-01, -7.22944363e-01, -5.32232194e-01,
       -4.79215578e-01,  3.47780764e+00, -4.81723680e-01, -5.37056921e-01,
       -7.24634909e-01, -2.63246176e-01, -5.82431493e-01,  3.51591654e+00,
       -5.24181526e-01, -6.19650223e-01, -3.11169606e-01, -7.78189259e-01,
       -4.37048255e-01, -8.66440602e-01, -4.32835215e-01, -5.24181526e-01,
        3.60920317e+00, -5.29638366e-01, -4.43066016e-01, -8.13041448e-01,
       -8.00487740e-01, -4.47862522e-01, -5.29638366e-01,  3.56476760e+00,
       -5.69040606e-01, -5.04235498e-01, -7.13502873e-01, -7.00646756e-01,
       -5.73341942e-01, -4.81723680e-01,  3.46848219e+00, -4.72758853e-01,
       -5.61999149e-01, -6.78011815e-01, -6.45651341e-01, -4.74723400e-01,
       -5.25860825e-01,  3.44521034e+00, -6.71029827e-01, -5.30757579e-01,
       -5.97187369e-01, -7.36947863e-01, -4.62464145e-01, -4.93645965e-01,
        3.49942036e+00, -5.92395098e-01, -5.24848573e-01, -6.89118721e-01,
       -1.10117641e+00, -5.24973888e-01, -7.85524865e-01,  3.74592985e+00,
       -5.45875033e-01, -7.88379658e-01, -5.29511952e-01, -5.62820184e-01,
       -6.65431240e-01,  3.48437620e+00, -6.75030411e-01, -5.14791028e-01,
       -5.36791383e-01, -6.79292566e-01, -4.04826153e-01,  3.46382779e+00,
       -5.77953847e-01, -5.53592771e-01, -6.01561335e-01, -6.46601116e-01,
       -4.82126897e-01, -5.12034168e-01, -7.55830609e-01, -5.77953847e-01,
        3.52836078e+00, -6.10236444e-01, -5.90178820e-01, -6.58193304e-01,
       -5.68555694e-01, -5.18177769e-01,  3.46875842e+00, -5.09842246e-01,
       -5.72548137e-01, -6.41441266e-01, -6.42730509e-01, -6.01697903e-01,
       -5.09842246e-01,  3.46680104e+00, -4.92483775e-01, -5.85848120e-01,
       -6.34198487e-01, -6.25027788e-01, -6.39978113e-01, -4.82207426e-01,
       -4.92483775e-01,  3.46907472e+00, -6.22971578e-01, -6.06406041e-01,
       -5.95607590e-01, -5.88175150e-01,  3.44212831e+00, -5.50631664e-01,
       -5.15090026e-01, -6.01563794e-01, -5.91060088e-01, -5.79521855e-01,
       -6.04480999e-01, -5.50631664e-01,  3.44764957e+00, -5.32440524e-01,
       -5.91088348e-01, -5.89486176e-01, -6.16310212e-01, -6.08246921e-01,
       -5.15090026e-01,  3.46610540e+00, -5.15974155e-01, -6.22075448e-01,
       -5.88408640e-01, -6.46822069e-01, -5.26550579e-01, -5.72548137e-01,
        3.46728642e+00, -5.80119805e-01, -5.24752536e-01, -6.16493295e-01,
       -6.22057930e-01, -5.45366014e-01, -5.80119805e-01,  3.46428869e+00,
       -5.87862694e-01, -5.34939281e-01, -5.93942967e-01, -5.87000256e-01,
       -5.71868721e-01, -5.87862694e-01,  3.45971804e+00, -5.88174296e-01,
       -5.57488099e-01, -5.67323978e-01, -5.62335003e-01, -6.05924158e-01,
       -5.88174296e-01,  3.46113445e+00, -5.80111012e-01, -5.90241197e-01,
       -5.34348789e-01, -5.47139833e-01, -6.25961641e-01, -5.80111012e-01,
        3.46702246e+00, -5.49237972e-01, -6.25067191e-01, -5.39504812e-01,
       -5.60887586e-01, -6.48349781e-01, -5.30954501e-01, -5.49237972e-01,
        3.47182994e+00, -5.27463083e-01, -6.54937015e-01, -6.53524307e-01,
       -4.86736298e-01, -5.80447803e-01,  3.48633847e+00, -6.05510825e-01,
       -5.46031772e-01, -6.14087463e-01, -4.92409582e-01, -5.42200155e-01,
        3.34176710e+00, -6.08655090e-01, -6.47222143e-01, -5.49800809e-01,
       -5.01479323e-01, -5.96894641e-01, -5.72494980e-01, -6.08655090e-01,
        3.49596211e+00, -5.76108822e-01, -5.50604430e-01, -5.91204150e-01,
       -6.28757824e-01, -6.54432240e-01, -4.72758853e-01,  3.48119373e+00,
       -5.13492030e-01, -6.27421214e-01, -5.84331572e-01, -6.74723457e-01,
       -5.15559247e-01,  3.49719414e+00, -5.63029515e-01, -5.49171148e-01,
       -6.99152038e-01, -4.95558736e-01, -5.17840704e-01, -6.45170723e-01,
       -5.63029515e-01,  3.50703400e+00, -5.82731561e-01, -6.99616791e-01,
       -4.98644707e-01, -5.13306083e-01, -6.10053073e-01, -5.82731561e-01,
        3.47514689e+00, -6.06874575e-01, -6.31835497e-01, -5.30346102e-01,
       -5.20430260e-01, -5.93213685e-01, -6.06874575e-01,  3.46637973e+00,
       -5.79474994e-01, -6.31852189e-01, -5.34534024e-01, -5.81497071e-01,
       -5.26636849e-01,  3.46867466e+00, -6.32849697e-01, -6.07115373e-01,
       -5.91422985e-01, -5.29152682e-01, -5.51625870e-01, -5.50946107e-01,
       -6.05217414e-01, -6.32849697e-01,  3.48614481e+00, -5.50377095e-01,
       -5.95128629e-01, -6.00832485e-01, -4.97928643e-01, -6.07115373e-01,
        3.48683030e+00, -6.41821095e-01, -5.49985578e-01, -5.89147126e-01,
       -6.05283565e-01, -5.02195997e-01,  3.46958604e+00, -6.15294218e-01,
       -6.53986014e-01, -5.43368418e-01, -5.49457828e-01, -5.27869438e-01,
       -6.00163889e-01, -6.15294218e-01,  3.45307275e+00, -6.01263434e-01,
       -5.69710259e-01, -5.38771517e-01, -5.79528171e-01, -5.17921488e-01,
       -6.75030411e-01, -6.53986014e-01,  3.47655336e+00, -5.13224549e-01,
       -5.36862724e-01, -5.51995547e-01, -5.98021043e-01, -6.01263434e-01,
        3.46663624e+00, -5.84074937e-01, -5.72020877e-01, -5.59260406e-01,
       -6.50302658e-01, -4.64754794e-01, -6.41821095e-01,  3.71098406e+00,
       -4.13275930e-01, -4.91935472e-01, -9.44971956e-01, -1.03922155e-01,
       -6.01064585e-01, -5.21556824e-01, -4.13275930e-01,  3.50439179e+00,
       -7.02823551e-01, -7.68432838e-01, -4.97238059e-01, -5.53187685e-01,
       -6.23480419e-01,  3.51113768e+00, -6.98478753e-01, -5.76740798e-01,
       -5.95822513e-01, -4.63427511e-01, -5.09330242e-01, -5.79446075e-01,
       -6.98478753e-01,  3.52815307e+00, -7.11182634e-01, -4.98891141e-01,
       -5.30824229e-01, -5.03974733e-01, -5.85812790e-01, -7.11182634e-01,
        3.51879036e+00, -6.02002765e-01, -4.99615149e-01, -6.16202290e-01,
       -6.24839002e-01, -6.08544194e-01, -5.76740798e-01,  3.48558340e+00,
       -6.57715669e-01, -5.23364840e-01, -4.94378892e-01, -5.27743381e-01,
       -5.94825016e-01, -6.02002765e-01,  3.55464136e+00, -6.28789661e-01,
       -6.10370617e-01, -5.90909916e-01, -5.31327230e-01, -7.05823681e-01,
       -6.28789661e-01,  3.49600881e+00, -5.08004164e-01, -6.42987860e-01,
       -4.79076218e-01, -4.58877673e-01, -5.31057277e-01, -5.08004164e-01,
        3.58478146e+00, -6.59747500e-01, -6.62373049e-01, -7.64721795e-01,
       -1.40449611e+00, -6.66066822e-01, -6.59747500e-01,  4.48077227e+00,
       -1.71215923e+00, -3.83026134e-02, -4.52267278e-01, -3.24835569e-01,
       -6.57715669e-01,  3.55331310e+00, -9.18915650e-01, -6.74855509e-01,
       -1.67278643e-01, -3.57444786e-01, -8.41234450e-01, -5.40579354e-01,
       -9.18915650e-01,  3.73815642e+00, -4.87485160e-01, -9.49941803e-01,
       -6.20915082e-01, -7.24016104e-01, -4.87485160e-01,  3.49625644e+00,
       -4.58399721e-01, -5.91420903e-01, -6.14019474e-01, -2.28455798e-01,
       -8.34029508e-01, -4.58399721e-01,  3.50836059e+00, -2.22473029e-01,
       -5.64815593e-01, -5.66913846e-01, -6.33273092e-01, -6.00402826e-01,
       -4.13322906e-01, -1.13201652e+00, -2.22473029e-01,  3.68075191e+00,
       -6.00176283e-01, -7.12360347e-01, -7.37387612e-01, -4.67230322e-01,
       -6.00176283e-01,  3.48882378e+00, -5.76582348e-01, -4.92901389e-01,
       -6.14545820e-01, -5.34886301e-01, -6.09600417e-01, -5.26327833e-01,
        3.39279279e+00, -4.64044016e-01, -6.03697589e-01, -6.54236634e-01,
       -6.24696157e-01, -6.79194820e-01, -6.19650223e-01,  3.55312651e+00,
       -4.61707750e-01, -7.44409014e-01, -4.23468549e-01, -7.15682874e-01,
       -5.69040606e-01,  3.53089605e+00, -4.44831927e-01, -5.59335855e-01,
       -6.05619472e-01, -6.36385318e-01, -5.91559097e-01, -6.79837943e-01,
       -4.44831927e-01,  3.47574729e+00, -4.88779007e-01, -6.38625344e-01,
       -6.32113968e-01, -6.63765676e-01, -7.76211790e-01, -6.71029827e-01,
        3.19678865e+00, -6.70230998e-01, -4.15550357e-01, -7.40686552e-01,
       -5.30757579e-01, -5.92395098e-01,  3.51043302e+00, -5.19880519e-01,
       -3.60064166e-01, -7.66649104e-01, -7.78235455e-01, -5.24848573e-01,
       -5.45875033e-01,  3.59245555e+00, -4.11844449e-01, -6.27758755e-01,
       -7.03893283e-01, -6.13014108e-01, -5.44506217e-01, -6.10236444e-01,
        3.48324925e+00, -5.81015113e-01, -5.31228021e-01, -6.03249343e-01,
       -5.88390125e-01, -5.73953609e-01, -5.81015113e-01,  3.46092494e+00,
       -5.91726609e-01, -5.39486728e-01, -5.86352757e-01, -5.48765694e-01,
       -5.56380150e-01, -5.91726609e-01,  3.50753879e+00, -7.51338311e-01,
       -6.83699179e-01, -3.75628850e-01, -5.20195645e-01, -5.14791028e-01,
       -7.51338311e-01,  3.54224128e+00, -7.60872981e-01, -7.08716689e-01,
       -2.86326629e-01, -6.22539140e-01, -6.07339833e-01,  3.44385181e+00,
       -5.35266101e-01, -4.44365194e-01, -6.71270790e-01, -5.63070749e-01,
       -5.59966492e-01, -6.43746193e-01, -5.35266101e-01,  3.46288198e+00,
       -5.23287763e-01, -6.39431103e-01, -5.61184327e-01, -5.38396835e-01,
       -6.91226240e-01, -5.23287763e-01,  3.47829235e+00, -4.89095540e-01,
       -6.65389321e-01, -5.70896653e-01, -5.77278580e-01, -6.77966416e-01,
       -5.21431327e-01, -4.89095540e-01,  3.51732629e+00, -6.14262445e-01,
       -6.37291980e-01, -6.98890190e-01, -5.53592771e-01, -4.44365194e-01,
        3.49356437e+00, -5.41771794e-01, -5.62269904e-01, -6.92674516e-01,
       -6.22345076e-01, -5.19689836e-01, -6.14262445e-01,  3.47735879e+00,
       -6.05098902e-01, -4.95167082e-01, -6.20795451e-01, -6.02193756e-01,
       -6.22971578e-01, -5.32440524e-01,  3.46406878e+00, -5.12811044e-01,
       -5.93624599e-01, -6.00027282e-01, -6.03376867e-01, -5.15974155e-01,
       -6.05098902e-01,  3.46993776e+00, -6.45525759e-01, -5.32201328e-01,
       -5.67760746e-01, -6.67131678e-01, -5.27463083e-01, -5.49171148e-01,
        3.47836258e+00, -5.48766695e-01, -5.00392666e-01, -6.85437309e-01,
       -6.41422372e-01, -4.64151909e-01, -6.05510825e-01, -6.47222143e-01,
        3.49350172e+00, -5.28844589e-01, -6.06349881e-01, -5.21136839e-01,
       -5.53934143e-01, -5.68838363e-01,  3.30156643e+00, -5.68641490e-01,
       -5.53563424e-01, -5.35452176e-01, -6.01913858e-01, -5.76108822e-01,
       -5.68641490e-01,  3.48048564e+00, -5.44136807e-01, -5.99294837e-01,
       -5.90389832e-01, -5.95443466e-01, -5.76288916e-01, -5.53563424e-01,
        3.51627254e+00, -6.17037260e-01, -6.38785711e-01, -5.35153761e-01,
       -5.13794507e-01, -5.57217258e-01, -6.17037260e-01,  3.49698980e+00,
       -6.66405502e-01, -5.88817009e-01, -5.53718262e-01, -5.39777924e-01,
       -5.43294107e-01, -6.66405502e-01,  3.47741837e+00, -6.44356066e-01,
       -5.37707547e-01, -5.45877228e-01, -5.55702526e-01, -5.13492030e-01,
       -6.44356066e-01,  3.47430315e+00, -6.50759918e-01, -5.49322310e-01,
       -5.60670302e-01, -5.79647500e-01, -5.46031772e-01, -5.79474994e-01,
        3.47068565e+00, -6.01862561e-01, -6.39481933e-01, -5.24186894e-01,
       -5.45009780e-01, -5.50377095e-01, -5.84074937e-01,  3.42802994e+00,
       -6.12857298e-01, -5.53295979e-01, -5.82414848e-01, -1.97990045e-01,
       -7.02823551e-01, -1.71215923e+00,  4.13953670e+00, -4.87590237e-01,
       -7.56709341e-01, -2.82264300e-01, -5.86758670e-01, -7.54221087e-01,
        3.65019638e+00, -6.42350649e-01, -4.50488012e-01, -6.61588703e-01,
       -5.54789258e-01, -6.02891836e-01, -5.76582348e-01,  3.49867647e+00,
       -6.20365738e-01, -5.69889456e-01, -6.40981819e-01, -4.87965271e-01,
       -8.28094016e-01, -7.88379658e-01, -4.11844449e-01,  3.71546443e+00,
       -1.14134934e+00, -5.45796968e-01, -3.73193021e-01, -7.08565665e-01,
       -6.70230998e-01,  3.45698665e+00, -5.36903065e-01, -7.01587462e-01,
       -4.66506436e-01, -3.89225646e-01, -7.81319597e-01, -4.64044016e-01,
       -5.36903065e-01,  3.49949787e+00, -5.62997477e-01, -7.65008066e-01,
       -5.70984996e-01, -6.03697589e-01, -4.88779007e-01,  3.41293423e+00,
       -5.26006567e-01, -5.59748956e-01, -6.63717113e-01, -6.26373746e-01,
       -6.25673469e-01, -4.61707750e-01,  3.48498287e+00, -4.67690689e-01,
       -7.06810714e-01, -5.96726500e-01, -7.11037797e-01, -5.37056921e-01,
       -4.67690689e-01,  3.48168101e+00, -4.87093902e-01, -5.84301520e-01,
       -6.94500184e-01, -7.24634909e-01, -5.61999149e-01, -4.87093902e-01,
        3.49017522e+00, -4.37602648e-01, -5.40862976e-01, -7.37981641e-01,
       -3.11169606e-01, -7.44409014e-01,  3.55005316e+00, -6.79136244e-01,
       -8.17388977e-01, -5.37921124e-01, -4.60028195e-01, -7.78189259e-01,
       -4.43066016e-01,  3.52501123e+00, -4.96362130e-01, -5.84182493e-01,
       -5.92261865e-01, -6.30949463e-01, -8.13041448e-01, -5.04235498e-01,
       -4.96362130e-01,  3.59207750e+00, -3.80640746e-01, -5.99907867e-01,
       -7.97889813e-01, -7.13502873e-01, -5.59335855e-01, -3.80640746e-01,
        3.56312947e+00, -6.14549476e-01, -8.29952608e-01, -4.65147909e-01,
       -6.78011815e-01, -6.27421214e-01, -4.37602648e-01,  3.47384357e+00,
       -4.72604947e-01, -6.21495667e-01, -6.36707280e-01, -5.97187369e-01,
       -4.15550357e-01, -5.19880519e-01, -7.01587462e-01,  3.27250508e+00,
       -5.28101738e-01, -5.10197633e-01, -6.89118721e-01, -3.60064166e-01,
       -6.27758755e-01,  3.35782970e+00, -6.48769634e-01, -4.28853433e-01,
       -6.03264992e-01, -5.36791383e-01, -5.13224549e-01, -7.60872981e-01,
        3.49459870e+00, -6.96106211e-01, -4.58764827e-01, -5.28838750e-01,
       -6.01561335e-01, -5.90178820e-01, -5.31228021e-01,  3.43837140e+00,
       -5.18645563e-01, -6.07242541e-01, -5.89515122e-01, -6.46601116e-01,
       -5.41771794e-01, -5.18645563e-01,  3.45953234e+00, -5.19780626e-01,
       -6.23333039e-01, -6.09400205e-01, -6.41441266e-01, -5.85848120e-01,
       -5.24752536e-01,  3.46807863e+00, -5.00638248e-01, -5.94849465e-01,
       -6.20548995e-01, -6.34198487e-01, -6.06406041e-01, -5.12811044e-01,
       -5.00638248e-01,  3.46647355e+00, -6.06785243e-01, -6.05634487e-01,
       -6.01563794e-01, -6.22075448e-01,  3.46569808e+00, -5.46182127e-01,
       -5.23371579e-01, -6.08302636e-01, -5.64202501e-01, -5.91060088e-01,
       -5.91088348e-01, -5.46182127e-01,  3.45317059e+00, -5.64391258e-01,
       -5.65767933e-01, -5.94680832e-01, -5.89486176e-01, -5.93624599e-01,
       -5.64391258e-01,  3.45608911e+00, -5.49548765e-01, -5.61993152e-01,
       -5.97045156e-01, -5.88408640e-01, -6.45525759e-01, -5.23371579e-01,
        3.48301172e+00, -5.53427043e-01, -6.04704344e-01, -5.67574360e-01,
       -6.16493295e-01, -5.34939281e-01, -5.94849465e-01,  3.46494766e+00,
       -5.99560834e-01, -5.30882651e-01, -5.88222137e-01, -5.93942967e-01,
       -5.57488099e-01, -5.99560834e-01,  3.46341647e+00, -5.99870905e-01,
       -5.50710681e-01, -5.61842979e-01, -5.67323978e-01, -5.90241197e-01,
       -5.99870905e-01,  3.45948149e+00, -5.93548998e-01, -5.80141418e-01,
       -5.28354997e-01, -5.34348789e-01, -6.25067191e-01, -5.93548998e-01,
        3.46590672e+00, -5.67942408e-01, -6.28484985e-01, -5.16514354e-01,
       -5.39504812e-01, -6.54937015e-01, -5.48766695e-01, -5.67942408e-01,
        3.47457886e+00, -5.11971694e-01, -6.51456234e-01, -6.14087463e-01,
       -5.28844589e-01, -6.01862561e-01,  3.47278773e+00, -6.10217705e-01,
       -5.78287351e-01, -5.39488067e-01, -5.49800809e-01, -5.50604430e-01,
        3.50929966e+00, -7.14071236e-01, -6.45412930e-01, -4.88436466e-01,
       -5.60973791e-01, -5.01479323e-01, -6.06349881e-01, -6.10217705e-01,
       -7.14071236e-01,  3.48016035e+00, -5.96340268e-01, -4.51701942e-01,
       -5.91204150e-01, -5.44136807e-01, -6.45412930e-01,  3.49052447e+00,
       -6.06660393e-01, -4.38596203e-01, -6.64513990e-01, -5.84331572e-01,
       -6.50759918e-01, -4.72604947e-01,  3.47373440e+00, -5.46999113e-01,
       -7.09683498e-01, -5.09355352e-01, -6.99152038e-01, -5.00392666e-01,
        3.47332874e+00, -5.28788943e-01, -5.29597669e-01, -7.01046895e-01,
       -5.14350532e-01, -4.95558736e-01, -6.99616791e-01, -5.28788943e-01,
        3.48074508e+00, -5.64671588e-01, -6.98967230e-01, -4.93141796e-01,
       -4.98644707e-01, -6.31835497e-01, -5.64671588e-01,  3.48419540e+00,
       -5.90059246e-01, -6.69497464e-01, -5.29486899e-01, -5.30346102e-01,
       -6.31852189e-01, -5.90059246e-01,  3.51338786e+00, -3.93318205e-01,
       -5.11320733e-01, -8.56491384e-01, -5.34534024e-01, -6.39481933e-01,
       -3.93318205e-01,  3.51273580e+00, -6.23584014e-01, -7.97789893e-01,
       -3.44176698e-01, -1.79851037e-01, -5.91422985e-01, -5.49985578e-01,
        3.47168471e+00, -6.10294536e-01, -6.16185192e-01, -5.55173804e-01,
       -5.48622617e-01, -5.29152682e-01, -5.95128629e-01, -6.12857298e-01,
       -6.10294536e-01,  3.46589362e+00, -5.37211567e-01, -5.81248905e-01,
       -5.89147126e-01, -4.91935472e-01, -6.16185192e-01,  3.51416979e+00,
       -6.81094451e-01, -5.96157428e-01, -5.39650124e-01, -5.43368418e-01,
       -5.69710259e-01,  3.47343697e+00, -6.33302997e-01, -6.34851265e-01,
       -5.46634741e-01, -5.45569294e-01, -5.49457828e-01, -5.36862724e-01,
       -6.96106211e-01, -6.33302997e-01,  3.48075932e+00, -4.62097620e-01,
       -6.02931940e-01, -5.38771517e-01, -5.72020877e-01, -6.34851265e-01,
        3.47033656e+00, -6.24256851e-01, -5.50688345e-01, -5.49747702e-01,
       -5.59260406e-01, -5.53295979e-01, -6.24256851e-01,  3.47363857e+00,
       -6.35180035e-01, -5.22357568e-01, -5.79287729e-01, -9.44971956e-01,
       -7.68432838e-01,  3.77131043e+00, -5.89991817e-01, -1.01536221e+00,
       -4.52551605e-01, -4.97238059e-01, -4.87590237e-01, -5.89991817e-01,
        3.49413998e+00, -7.29134044e-01, -6.83599722e-01, -5.06586096e-01,
       -5.95822513e-01, -5.23364840e-01,  3.55774660e+00, -7.76322322e-01,
       -8.73434654e-01, -4.56326351e-01, -3.32475919e-01, -4.63427511e-01,
       -4.98891141e-01, -7.76322322e-01,  3.62907944e+00, -9.98117142e-01,
       -6.05233589e-01, -2.87087731e-01, -5.30824229e-01, -4.99615149e-01,
       -9.98117142e-01,  3.92551655e+00, -5.61202584e-01, -1.88095643e-02,
       -1.31694788e+00, -6.16202290e-01, -6.10370617e-01, -5.61202584e-01,
        3.49555344e+00, -5.23527851e-01, -5.10380498e-01, -6.73869596e-01,
       -4.94378892e-01, -6.74855509e-01, -8.73434654e-01,  3.63581414e+00,
       -5.47196747e-01, -2.22080098e-01, -8.23868241e-01, -5.90909916e-01,
       -6.42987860e-01, -5.23527851e-01,  3.49261127e+00, -5.27791724e-01,
       -5.38038345e-01, -6.69355578e-01, -4.79076218e-01, -6.62373049e-01,
       -5.27791724e-01,  3.44298834e+00, -5.54194902e-01, -6.22029073e-01,
       -5.97523376e-01, -7.64721795e-01, -3.83026134e-02, -7.56709341e-01,
       -5.54194902e-01,  3.41997153e+00, -8.44922122e-01, -4.61120753e-01,
       -1.67278643e-01, -9.49941803e-01, -5.91420903e-01,  3.65164508e+00,
       -5.53053698e-01, -8.42342897e-01, -5.47607132e-01, -6.14019474e-01,
       -5.64815593e-01, -5.53053698e-01,  3.47246985e+00, -6.40528612e-01,
       -5.68813150e-01, -5.31239322e-01, -5.66913846e-01, -7.12360347e-01,
       -4.92901389e-01,  3.48358671e+00, -6.05871423e-01, -4.83411061e-01,
       -6.22128643e-01, -2.97442492e-01, -6.42350649e-01,  3.57273814e+00,
       -9.28643422e-01, -6.56079164e-01, -4.55022932e-01, -5.93199485e-01,
       -6.35438960e-01, -4.50488012e-01, -6.20365738e-01,  3.51352878e+00,
       -6.84641372e-01, -5.62879750e-01, -5.59714948e-01, -6.14545820e-01,
       -5.69889456e-01, -6.05871423e-01,  3.48082223e+00, -5.47247784e-01,
       -5.28798952e-01, -6.14468791e-01,  8.43588871e-05, -1.14134934e+00,
       -9.28643422e-01,  3.81149412e+00, -7.11931297e-01, -5.32716226e-01,
       -4.96938196e-01, -6.54236634e-01, -5.62997477e-01, -5.26006567e-01,
        3.43843158e+00, -4.55670844e-01, -4.95984516e-01, -7.43535541e-01,
       -4.23468549e-01, -7.06810714e-01, -6.79136244e-01,  3.49058657e+00,
       -5.67232553e-01, -6.09823335e-01, -5.04115175e-01, -4.37048255e-01,
       -8.17388977e-01, -5.84182493e-01,  3.51738207e+00, -5.02832366e-01,
       -6.34885188e-01, -5.41044790e-01, -6.05619472e-01, -6.14549476e-01,
        3.44499193e+00, -4.64107742e-01, -6.74683267e-01, -6.34531171e-01,
       -4.51500807e-01, -6.36385318e-01, -6.38625344e-01, -6.56079164e-01,
       -4.64107742e-01,  3.47755021e+00, -5.04295687e-01, -5.78056953e-01,
       -6.32113968e-01, -5.59748956e-01, -4.55022932e-01, -7.11931297e-01,
       -5.04295687e-01,  3.43613144e+00, -5.73018604e-01, -7.66649104e-01,
       -5.28101738e-01, -6.48769634e-01,  3.17035586e+00, -7.17527839e-01,
       -5.09307542e-01, -7.03893283e-01, -5.45796968e-01, -4.28853433e-01,
       -5.32716226e-01,  3.37290308e+00, -6.45795637e-01, -5.15847533e-01,
       -6.03249343e-01, -5.39486728e-01, -6.07242541e-01,  3.43288474e+00,
       -5.60225558e-01, -5.24042574e-01, -5.98637994e-01, -5.86352757e-01,
       -6.83699179e-01, -5.60225558e-01,  3.46956513e+00, -4.14155811e-01,
       -5.52172777e-01, -6.72959050e-01, -3.75628850e-01, -7.08716689e-01,
       -4.14155811e-01,  3.72748115e+00, -1.20638229e+00, -6.55111770e-01,
       -3.67485741e-01, -2.86326629e-01, -4.58764827e-01, -1.20638229e+00,
        3.99306911e+00, -1.25088526e+00, -7.90710109e-01, -6.71270790e-01,
       -5.62269904e-01,  3.55750887e+00, -5.77278725e-01, -4.82181564e-01,
       -6.48861179e-01, -6.15646711e-01, -5.63070749e-01, -6.39431103e-01,
       -5.77278725e-01,  3.47755517e+00, -5.22275183e-01, -5.83262103e-01,
       -5.92237303e-01, -5.61184327e-01, -6.65389321e-01, -5.22275183e-01,
        3.45053598e+00, -4.99142678e-01, -6.31660303e-01, -5.70884164e-01,
       -5.70896653e-01, -6.37291980e-01, -4.95167082e-01, -4.99142678e-01,
        3.41296549e+00, -5.77497810e-01, -6.32969292e-01, -6.92674516e-01,
       -5.19780626e-01, -4.82181564e-01,  3.48824108e+00, -5.70076720e-01,
       -5.45833776e-01, -6.77693879e-01, -6.20795451e-01, -5.32201328e-01,
       -5.77497810e-01,  3.47473782e+00, -6.07138689e-01, -5.37551316e-01,
       -5.99553225e-01, -6.00027282e-01, -6.06785243e-01, -5.49548765e-01,
        3.46337316e+00, -5.19987097e-01, -5.69209329e-01, -6.17815446e-01,
       -5.67760746e-01, -5.53427043e-01, -6.07138689e-01,  3.46752900e+00,
       -6.18269017e-01, -5.51093116e-01, -5.69840387e-01, -6.85437309e-01,
       -5.11971694e-01, -5.29597669e-01,  3.50578653e+00, -5.59866932e-01,
       -5.56136965e-01, -6.62775965e-01, -5.35452176e-01, -5.99294837e-01,
       -6.38785711e-01,  3.49306587e+00, -5.64266725e-01, -5.80023599e-01,
       -5.75242825e-01, -5.90389832e-01, -6.06660393e-01, -5.64266725e-01,
        3.47640568e+00, -5.12014869e-01, -5.91677190e-01, -6.11396672e-01,
       -5.35153761e-01, -5.88817009e-01, -5.80023599e-01,  3.44929658e+00,
       -5.64650946e-01, -5.88379779e-01, -5.92271483e-01, -5.53718262e-01,
       -5.37707547e-01, -5.64650946e-01,  3.42682194e+00, -6.09921598e-01,
       -5.71887496e-01, -5.88936089e-01, -5.45877228e-01, -5.49322310e-01,
       -6.09921598e-01,  3.40535825e+00, -6.16961037e-01, -5.56337969e-01,
       -5.26938112e-01, -5.60670302e-01, -5.46999113e-01, -6.16961037e-01,
        3.44737960e+00, -6.61595220e-01, -5.84901875e-01, -4.76252054e-01,
       -5.24186894e-01, -5.78287351e-01, -6.23584014e-01,  3.48834734e+00,
       -6.39578865e-01, -6.45269694e-01, -4.77440527e-01, -1.03922155e-01,
       -6.81094451e-01, -1.01536221e+00,  3.86144940e+00, -5.96001692e-01,
       -6.49356512e-01, -4.02706672e-01, -4.13005711e-01, -5.82414848e-01,
       -5.37211567e-01, -6.35180035e-01,  3.45642058e+00, -6.13192471e-01,
       -5.07743099e-01, -5.80678556e-01, -2.82264300e-01, -7.29134044e-01,
       -8.44922122e-01,  3.57020991e+00, -5.88402610e-01, -5.46015104e-01,
       -5.79471729e-01, -3.57444786e-01, -5.47196747e-01, -8.42342897e-01,
        3.55977798e+00, -7.33878862e-01, -5.71847664e-01, -5.07067021e-01,
       -6.33273092e-01, -6.40528612e-01, -4.83411061e-01,  3.48147741e+00,
       -4.75340650e-01, -6.76610962e-01, -5.72313037e-01, -6.61588703e-01,
       -5.93199485e-01, -6.74683267e-01, -5.78056953e-01,  3.65022247e+00,
       -5.27098834e-01, -6.15595227e-01, -5.54789258e-01, -6.84641372e-01,
       -5.27098834e-01,  3.57578829e+00, -4.66491326e-01, -6.52675753e-01,
       -6.90091751e-01, -6.40981819e-01, -5.47247784e-01,  3.43493229e+00,
       -6.06994701e-01, -5.23939996e-01, -4.46644099e-01, -6.69123894e-01,
       -4.87965271e-01, -5.62879750e-01, -6.06994701e-01,  3.46991739e+00,
       -6.74936067e-01, -6.32736794e-01, -5.04404803e-01, -4.66506436e-01,
       -7.65008066e-01, -5.10197633e-01, -4.55670844e-01, -7.17527839e-01,
        3.46853328e+00, -5.53622465e-01, -6.63717113e-01, -4.96938196e-01,
       -4.95984516e-01, -5.73018604e-01, -6.45795637e-01,  3.43470571e+00,
       -5.59251644e-01, -5.96726500e-01, -5.84301520e-01, -5.67232553e-01,
        3.45085872e+00, -5.13018372e-01, -6.09797895e-01, -5.79781882e-01,
       -6.94500184e-01, -5.40862976e-01, -5.13018372e-01,  3.47582361e+00,
       -4.85367732e-01, -5.69765805e-01, -6.72308541e-01, -7.37981641e-01,
       -6.21495667e-01, -4.85367732e-01,  3.50391808e+00, -3.83761275e-01,
       -5.09389544e-01, -7.65922218e-01, -5.37921124e-01, -5.02832366e-01,
        3.47884109e+00, -7.10175192e-01, -6.76489120e-01, -5.36943523e-01,
       -5.14479766e-01, -4.60028195e-01, -6.09823335e-01, -7.10175192e-01,
        3.48119226e+00, -6.51822614e-01, -5.26292072e-01, -5.23050856e-01,
       -5.92261865e-01, -6.34885188e-01,  3.50760330e+00, -5.70441440e-01,
       -5.25036459e-01, -6.25018863e-01, -5.59959490e-01, -6.30949463e-01,
       -5.99907867e-01, -5.70441440e-01,  3.52795278e+00, -3.67513343e-01,
       -6.87143729e-01, -6.71996934e-01, -7.97889813e-01, -8.29952608e-01,
       -3.67513343e-01,  3.82385107e+00, -5.30699112e-02, -6.38777593e-01,
       -1.13664780e+00, -4.65147909e-01, -6.34531171e-01, -5.30699112e-02,
        3.66147241e+00, -6.80196855e-01, -1.09984483e+00, -4.12843912e-01,
       -3.15837817e-01, -6.36707280e-01, -7.09683498e-01, -3.83761275e-01,
        3.49961354e+00, -5.18249191e-01, -7.10378175e-01, -5.40834126e-01,
       -6.03264992e-01, -7.43535541e-01, -5.09307542e-01, -5.15847533e-01,
       -5.53622465e-01, -5.59251644e-01,  3.48482972e+00, -5.28838750e-01,
       -4.62097620e-01, -1.25088526e+00,  3.74817827e+00, -5.94418678e-01,
       -1.95273474e-01, -7.16664495e-01, -5.89515122e-01, -6.23333039e-01,
       -5.24042574e-01,  3.46399212e+00, -5.23888297e-01, -6.30520289e-01,
       -5.72692804e-01, -6.09400205e-01, -5.70076720e-01, -5.23888297e-01,
        3.44990840e+00, -5.35790054e-01, -6.31096877e-01, -5.79656248e-01,
       -6.20548995e-01, -6.05634487e-01, -5.30882651e-01, -5.19987097e-01,
        3.46608691e+00, -6.13786617e-01, -5.75247069e-01, -6.08302636e-01,
       -5.65767933e-01,  3.48654886e+00, -5.61653291e-01, -5.88299416e-01,
       -5.29969437e-01, -6.32556142e-01, -5.64202501e-01, -6.04704344e-01,
       -5.61653291e-01,  3.44646756e+00, -5.49818518e-01, -6.15226570e-01,
       -5.50862339e-01, -5.94680832e-01, -5.61993152e-01, -5.88299416e-01,
        3.47055479e+00, -5.82502196e-01, -5.12623274e-01, -6.30455923e-01,
       -5.97045156e-01, -5.69209329e-01, -5.82502196e-01,  3.46457765e+00,
       -5.55642105e-01, -5.26394133e-01, -6.33784734e-01, -5.67574360e-01,
       -6.18269017e-01, -5.49818518e-01,  3.45702958e+00, -5.69374837e-01,
       -6.05718900e-01, -5.46273946e-01, -5.88222137e-01, -5.50710681e-01,
       -6.13786617e-01,  3.46479443e+00, -6.10397548e-01, -5.56139412e-01,
       -5.45538032e-01, -5.61842979e-01, -5.80141418e-01, -6.10397548e-01,
        3.46201676e+00, -6.08188446e-01, -5.85384612e-01, -5.16061754e-01,
       -5.28354997e-01, -6.28484985e-01, -6.08188446e-01,  3.46863660e+00,
       -5.85910825e-01, -6.23995273e-01, -4.93702073e-01, -5.16514354e-01,
       -6.51456234e-01, -5.59866932e-01, -5.85910825e-01,  3.47951507e+00,
       -5.11547009e-01, -6.54219720e-01, -5.39488067e-01, -5.96340268e-01,
       -6.39578865e-01,  3.47866885e+00, -6.52129423e-01, -6.12380438e-01,
       -4.38751786e-01, -4.88436466e-01, -4.51701942e-01, -6.52129423e-01,
        3.97699662e+00, -1.52406582e+00, -8.60662980e-01, -5.60973791e-01,
       -4.38596203e-01, -1.52406582e+00,  4.02984265e+00, -5.60177869e-01,
       -1.58809378e-02, -9.30148030e-01, -6.64513990e-01, -5.12014869e-01,
       -5.60177869e-01,  3.46572601e+00, -5.80397010e-01, -3.99263520e-01,
       -7.49358754e-01, -5.09355352e-01, -6.61595220e-01, -5.18249191e-01,
        3.48895339e+00, -6.14472856e-01, -7.49059661e-01, -4.36221108e-01,
       -7.01046895e-01, -5.56136965e-01,  3.52520720e+00, -5.39902294e-01,
       -5.36039132e-01, -5.37412150e-01, -6.54669767e-01, -5.14350532e-01,
       -6.98967230e-01, -5.39902294e-01,  3.49214040e+00, -5.36694967e-01,
       -6.64213325e-01, -5.38012053e-01, -4.93141796e-01, -6.69497464e-01,
       -5.36694967e-01,  3.46500696e+00, -6.26728577e-01, -7.27654836e-01,
       -4.11289324e-01, -5.29486899e-01, -5.11320733e-01, -6.26728577e-01,
        3.43878195e+00, -6.84790468e-01, -5.16061700e-01, -5.70393571e-01,
       -8.56491384e-01, -7.97789893e-01, -6.84790468e-01,  3.83092808e+00,
       -1.17694087e+00, -3.14915462e-01, -3.44176698e-01, -6.45269694e-01,
        3.62016900e+00, -7.99487258e-01, -6.57418081e-01, -5.53056855e-01,
       -6.20760412e-01, -5.55173804e-01, -5.81248905e-01, -6.13192471e-01,
        3.46388786e+00, -6.03386372e-01, -5.46209093e-01, -5.64677215e-01,
       -5.48622617e-01, -5.96157428e-01, -6.03386372e-01,  3.46813221e+00,
       -6.08807032e-01, -5.75786851e-01, -5.35371908e-01, -5.39650124e-01,
       -5.96001692e-01, -6.08807032e-01,  3.52934316e+00, -6.73653615e-01,
       -6.03882548e-01, -5.07348150e-01, -5.46634741e-01, -6.02931940e-01,
       -5.94418678e-01,  3.47061976e+00, -6.29504437e-01, -5.31468393e-01,
       -5.65661567e-01, -5.45569294e-01, -5.50688345e-01, -6.29504437e-01,
        3.45767414e+00, -6.37843784e-01, -5.61495814e-01, -5.32572468e-01,
       -5.49747702e-01, -5.22357568e-01, -6.37843784e-01,  3.46316244e+00,
       -6.68705348e-01, -5.23351183e-01, -5.61156852e-01, -5.79287729e-01,
       -5.07743099e-01, -6.68705348e-01,  3.48352463e+00, -6.59232343e-01,
       -4.86397760e-01, -5.82158350e-01, -4.52551605e-01, -6.83599722e-01,
       -6.49356512e-01,  3.51749169e+00, -5.71994549e-01, -6.64140125e-01,
       -4.95849180e-01, -5.06586096e-01, -5.88402610e-01, -5.71994549e-01,
        3.39763375e+00, -5.98066142e-01, -6.03181400e-01, -5.29402958e-01,
       -4.56326351e-01, -6.05233589e-01,  3.91620220e+00, -8.06973541e-01,
       -8.48531170e-01, -5.88515619e-01, -6.10621934e-01, -3.32475919e-01,
       -2.22080098e-01, -8.06973541e-01,  4.06057457e+00, -1.07981689e+00,
       -6.03042576e-01, -4.20104090e-01, -5.96081461e-01, -2.87087731e-01,
       -1.88095643e-02, -8.48531170e-01,  3.64237145e+00, -8.76305190e-01,
       -5.91979481e-01, -4.65140284e-01, -5.54518025e-01, -1.31694788e+00,
       -5.10380498e-01, -8.76305190e-01,  3.88236212e+00, -5.29334139e-01,
       -6.49394410e-01, -6.73869596e-01, -5.38038345e-01, -5.29334139e-01,
        3.48500777e+00, -5.97696159e-01, -4.76707609e-01, -6.69361922e-01,
       -8.23868241e-01, -7.33878862e-01, -1.07981689e+00,  3.73175694e+00,
       -5.12090062e-01, -5.82102886e-01, -6.69355578e-01, -6.22029073e-01,
       -5.97696159e-01,  3.55499994e+00, -5.55749843e-01, -4.89498197e-01,
       -6.20671086e-01, -5.97523376e-01, -4.61120753e-01, -5.46015104e-01,
       -5.55749843e-01,  3.08193457e+00, -4.59536283e-01, -4.61989215e-01,
       -5.47607132e-01, -5.68813150e-01, -5.71847664e-01,  3.47995642e+00,
       -6.91513226e-01, -6.44563259e-01, -4.55611990e-01, -5.31239322e-01,
       -4.75340650e-01, -6.91513226e-01,  3.51625265e+00, -8.06407559e-01,
       -6.18351314e-01, -3.93400578e-01, -6.22128643e-01, -5.28798952e-01,
       -6.76610962e-01,  3.48328654e+00, -5.95268479e-01, -4.29042261e-01,
       -6.31437241e-01, -5.59714948e-01, -4.66491326e-01, -6.74936067e-01,
        3.47785884e+00, -6.46859375e-01, -4.83289075e-01, -6.46568046e-01,
       -6.14468791e-01, -5.23939996e-01, -5.95268479e-01,  3.43342229e+00,
       -6.33971899e-01, -5.56187663e-01, -5.09585462e-01, -5.04115175e-01,
       -6.09797895e-01, -6.51822614e-01,  3.46823131e+00, -5.93085980e-01,
       -5.69828641e-01, -5.39581004e-01, -5.41044790e-01, -6.76489120e-01,
       -5.25036459e-01,  3.47987339e+00, -5.20693182e-01, -6.32992717e-01,
       -5.83617123e-01, -4.51500807e-01, -6.15595227e-01, -6.52675753e-01,
       -6.80196855e-01,  3.44637611e+00, -4.02547463e-01, -6.43860004e-01,
       -5.98637994e-01, -5.52172777e-01, -6.30520289e-01,  3.46453007e+00,
       -5.67070432e-01, -5.43280821e-01, -5.72847758e-01, -6.72959050e-01,
       -6.55111770e-01, -5.67070432e-01,  3.50040504e+00, -3.84810850e-01,
       -5.54184578e-01, -6.66268359e-01, -3.67485741e-01, -7.90710109e-01,
       -1.95273474e-01, -3.84810850e-01,  3.51239732e+00, -4.86466765e-01,
       -5.95241106e-01, -6.92409279e-01, -6.48861179e-01, -5.45833776e-01,
        3.49047017e+00, -5.47857890e-01, -4.31774755e-01, -7.49836461e-01,
       -5.66306110e-01, -6.15646711e-01, -5.83262103e-01, -5.47857890e-01,
        3.48958624e+00, -5.53645201e-01, -5.47440367e-01, -6.41733967e-01,
       -5.92237303e-01, -6.31660303e-01, -5.53645201e-01,  3.46888559e+00,
       -5.35006805e-01, -5.98464858e-01, -5.57871124e-01, -5.70884164e-01,
       -6.32969292e-01, -5.37551316e-01, -5.35006805e-01,  3.47631797e+00,
       -5.76327649e-01, -6.23578747e-01, -6.77693879e-01, -5.35790054e-01,
       -4.31774755e-01,  3.48244261e+00, -5.70250327e-01, -6.73685463e-01,
       -5.93248136e-01, -5.99553225e-01, -5.51093116e-01, -5.76327649e-01,
        3.47066538e+00, -5.90159682e-01, -5.61031048e-01, -5.92500664e-01,
       -6.17815446e-01, -5.75247069e-01, -5.55642105e-01, -5.56139412e-01,
        3.46323237e+00, -5.34378837e-01, -6.24009502e-01, -5.69840387e-01,
       -5.69374837e-01, -5.90159682e-01,  3.46224021e+00, -6.04426458e-01,
       -5.81592232e-01, -5.46846610e-01, -6.62775965e-01, -5.11547009e-01,
       -5.36039132e-01,  3.49580333e+00, -5.72731309e-01, -5.53154848e-01,
       -6.59555068e-01, -5.75242825e-01, -5.91677190e-01, -5.88379779e-01,
        3.47558647e+00, -5.32748649e-01, -5.58193385e-01, -6.29344646e-01,
       -6.11396672e-01, -5.80397010e-01, -5.32748649e-01,  3.45922372e+00,
       -4.75015864e-01, -5.73609759e-01, -6.86055765e-01, -5.92271483e-01,
       -5.71887496e-01, -5.58193385e-01,  3.43490060e+00, -5.65696744e-01,
       -5.50032582e-01, -5.96818907e-01, -5.88936089e-01, -5.56337969e-01,
       -5.65696744e-01,  3.45168875e+00, -6.14472957e-01, -5.72299096e-01,
       -5.53945890e-01, -5.26938112e-01, -5.84901875e-01, -6.14472957e-01,
        3.42162608e+00, -6.38609930e-01, -5.81815325e-01, -4.74887882e-01,
       -4.76252054e-01, -6.14472856e-01, -6.38609930e-01,  3.47370967e+00,
       -6.17107478e-01, -6.46536747e-01, -4.80730600e-01, -1.79851037e-01,
       -1.17694087e+00, -7.99487258e-01,  3.78595069e+00, -6.09607785e-01,
       -4.43886482e-01, -5.76177256e-01, -4.77440527e-01, -6.12380438e-01,
       -6.57418081e-01,  3.49807530e+00, -5.77675616e-01, -6.04827077e-01,
       -5.68333560e-01, -4.02706672e-01, -6.73653615e-01,  3.65064601e+00,
       -8.46343888e-01, -6.36982347e-01, -5.82864938e-01, -5.08094554e-01,
       -5.80678556e-01, -5.46209093e-01, -6.59232343e-01,  3.46744947e+00,
       -6.21443239e-01, -5.07192629e-01, -5.52693612e-01, -4.13005711e-01,
       -6.64140125e-01, -8.46343888e-01,  3.65774837e+00, -7.12468238e-01,
       -4.69145643e-01, -5.52644760e-01, -5.79471729e-01, -5.98066142e-01,
       -4.59536283e-01,  3.39070328e+00, -5.60198774e-01, -6.84772393e-01,
       -5.08657956e-01, -5.07067021e-01, -5.12090062e-01, -6.44563259e-01,
        3.48399103e+00, -6.98751192e-01, -6.21467192e-01, -5.00052299e-01,
       -5.72313037e-01, -8.06407559e-01, -4.29042261e-01,  3.53670416e+00,
       -5.12840722e-01, -8.00889022e-01, -4.15211555e-01, -6.90091751e-01,
       -6.46859375e-01, -4.02547463e-01,  3.50909404e+00, -6.55185952e-01,
       -4.13465284e-01, -7.00944217e-01, -4.46644099e-01, -6.32736794e-01,
        3.38295587e+00, -5.14010366e-01, -6.14258742e-01, -6.98525682e-01,
       -4.76780190e-01, -6.69123894e-01, -6.33971899e-01, -5.14010366e-01,
        3.08057957e+00, -6.85085659e-01, -5.78387754e-01, -5.04404803e-01,
       -4.83289075e-01, -6.14258742e-01,  3.39355640e+00, -6.64736135e-01,
       -5.57333153e-01, -5.69534494e-01, -5.79781882e-01, -5.69765805e-01,
       -5.93085980e-01,  3.45369790e+00, -5.70388159e-01, -5.65673073e-01,
       -5.75002999e-01, -6.72308541e-01, -5.09389544e-01, -5.70388159e-01,
        3.47007468e+00, -5.44705931e-01, -5.40648483e-01, -6.32634019e-01,
       -7.65922218e-01, -7.10378175e-01, -5.44705931e-01,  3.56533232e+00,
       -2.66715067e-01, -4.66728309e-01, -8.10882619e-01, -5.36943523e-01,
       -5.20693182e-01,  3.46823557e+00, -6.75783476e-01, -6.44521093e-01,
       -5.39842096e-01, -5.50452200e-01, -5.14479766e-01, -5.26292072e-01,
       -6.75783476e-01,  3.46161916e+00, -6.79266497e-01, -5.07740555e-01,
       -5.58056797e-01, -5.23050856e-01, -5.69828641e-01, -6.79266497e-01,
        3.47141653e+00, -6.46274087e-01, -5.14664169e-01, -5.38332277e-01,
       -6.25018863e-01, -6.32992717e-01,  3.51939644e+00, -4.51641141e-01,
       -4.98065460e-01, -6.32228362e-01, -6.79449895e-01, -5.59959490e-01,
       -6.87143729e-01, -4.51641141e-01,  3.38524682e+00, -4.14287982e-01,
       -6.38432197e-01, -6.33782285e-01, -6.71996934e-01, -6.38777593e-01,
       -4.14287982e-01,  3.36844957e+00, -3.26462099e-01, -5.78983838e-01,
       -7.37941129e-01, -1.13664780e+00, -1.09984483e+00, -3.26462099e-01,
        3.46218467e+00, -8.99229943e-01, -4.12843912e-01, -6.43860004e-01,
       -6.55185952e-01,  3.50099905e+00, -7.18830961e-01, -4.45380690e-01,
       -6.24897533e-01, -5.40834126e-01, -7.49059661e-01, -2.66715067e-01,
        3.58347489e+00, -6.60155221e-01, -9.17616614e-01, -4.49094201e-01,
       -7.16664495e-01, -5.31468393e-01, -4.86466765e-01,  3.48827757e+00,
       -5.95477951e-01, -5.61582796e-01, -5.96617174e-01, -5.72692804e-01,
       -6.31096877e-01, -5.43280821e-01,  3.46320389e+00, -5.37297820e-01,
       -6.30090371e-01, -5.48745201e-01, -5.79656248e-01, -5.70250327e-01,
       -5.37297820e-01,  3.44577649e+00, -5.85141897e-01, -6.30261973e-01,
       -5.43168221e-01, -5.29969437e-01, -6.15226570e-01,  3.47853940e+00,
       -5.99793366e-01, -5.81813561e-01, -5.06562917e-01, -6.45173550e-01,
       -6.32556142e-01, -5.12623274e-01, -5.99793366e-01,  3.47977818e+00,
       -5.97579247e-01, -4.90135850e-01, -6.47090306e-01, -5.50862339e-01,
       -6.05718900e-01, -5.81813561e-01,  3.47277137e+00, -5.85996681e-01,
       -6.29951679e-01, -5.18428214e-01, -6.30455923e-01, -5.26394133e-01,
       -5.97579247e-01,  3.47616604e+00, -5.90438772e-01, -4.89846331e-01,
       -6.41451637e-01, -6.33784734e-01, -5.34378837e-01, -5.90438772e-01,
        3.47253319e+00, -5.86540725e-01, -4.96768833e-01, -6.30621294e-01,
       -5.46273946e-01, -6.04426458e-01, -5.85996681e-01,  3.48369843e+00,
       -5.98044923e-01, -6.14588797e-01, -5.34367624e-01, -5.45538032e-01,
       -5.85384612e-01, -6.24009502e-01, -5.86540725e-01,  3.46806168e+00,
       -6.19906712e-01, -5.06682092e-01, -5.16061754e-01, -6.23995273e-01,
       -6.19906712e-01,  3.47183682e+00, -6.06910524e-01, -6.17442500e-01,
       -4.87520058e-01, -4.93702073e-01, -6.54219720e-01, -5.72731309e-01,
       -6.06910524e-01,  3.48033939e+00, -5.04064183e-01, -6.48711580e-01,
       -4.38751786e-01, -8.60662980e-01, -1.58809378e-02, -5.77675616e-01,
        3.61036617e+00, -4.53399341e-01, -6.43114037e-01, -6.20881470e-01,
       -9.30148030e-01, -3.99263520e-01, -4.53399341e-01,  3.59983150e+00,
       -5.58833105e-01, -5.71339729e-01, -6.86847775e-01, -7.49358754e-01,
       -4.75015864e-01, -5.58833105e-01,  3.51321735e+00, -6.13431168e-01,
       -3.83795369e-01, -7.32783093e-01, -4.36221108e-01, -6.17107478e-01,
       -6.60155221e-01,  3.47673533e+00, -6.36804116e-01, -5.79620210e-01,
       -5.46827199e-01, -5.37412150e-01, -6.64213325e-01,  3.47923224e+00,
       -5.45336803e-01, -5.51028213e-01, -6.12223852e-01, -5.69017902e-01,
       -6.54669767e-01, -5.53154848e-01, -5.45336803e-01,  3.50536260e+00,
       -5.67335427e-01, -5.46812973e-01, -6.38052781e-01, -5.38012053e-01,
       -7.27654836e-01, -5.51028213e-01,  3.47920528e+00, -5.54590606e-01,
       -6.35991189e-01, -4.71928379e-01, -4.11289324e-01, -5.16061700e-01,
       -5.54590606e-01,  3.55217029e+00, -6.85879471e-01, -6.61827332e-01,
       -3.05124139e-01, -4.17397718e-01, -5.70393571e-01, -3.14915462e-01,
       -6.09607785e-01, -6.85879471e-01,  3.50067905e+00, -5.12798267e-01,
       -8.07084494e-01, -5.53056855e-01, -6.04827077e-01,  3.50080492e+00,
       -6.22318693e-01, -5.63841257e-01, -5.70671446e-01, -5.86089595e-01,
       -6.20760412e-01, -4.43886482e-01, -6.22318693e-01,  3.49769462e+00,
       -6.75921804e-01, -5.35753864e-01, -5.99053365e-01, -5.64677215e-01,
       -5.75786851e-01, -6.21443239e-01,  3.46547458e+00, -6.06239978e-01,
       -5.60392105e-01, -5.36935193e-01, -5.35371908e-01, -6.03882548e-01,
       -6.06239978e-01,  3.47956961e+00, -6.17870651e-01, -5.89414780e-01,
       -5.26789747e-01, -5.07348150e-01, -6.36982347e-01, -6.17870651e-01,
        3.48223292e+00, -5.96191314e-01, -6.11978065e-01, -5.11862391e-01,
       -5.65661567e-01, -5.61495814e-01, -5.95477951e-01,  3.45337042e+00,
       -6.52016535e-01, -5.64409339e-01, -5.14309216e-01, -5.32572468e-01,
       -5.23351183e-01, -6.52016535e-01,  3.45168639e+00, -6.78538742e-01,
       -5.25736222e-01, -5.39471244e-01, -5.61156852e-01, -4.86397760e-01,
       -6.78538742e-01,  3.46597062e+00, -6.82229567e-01, -4.88754595e-01,
       -5.68893106e-01, -5.82158350e-01, -5.07192629e-01, -6.82229567e-01,
        3.48442888e+00, -6.51873873e-01, -4.74378695e-01, -5.86595770e-01,
       -4.95849180e-01, -6.03181400e-01, -7.12468238e-01,  3.47918685e+00,
       -6.50846136e-01, -4.77711128e-01, -5.39130772e-01, -5.29402958e-01,
       -5.60198774e-01, -6.50846136e-01,  3.43871621e+00, -6.68483503e-01,
       -5.26116737e-01, -5.03668100e-01, -5.88515619e-01, -5.91979481e-01,
        3.46459287e+00, -5.30217545e-01, -5.39513063e-01, -5.55944514e-01,
       -6.58422647e-01, -6.10621934e-01, -6.03042576e-01, -5.30217545e-01,
        3.42159637e+00, -4.90913748e-01, -6.02945844e-01, -5.83854725e-01,
       -4.20104090e-01, -5.82102886e-01, -6.98751192e-01,  3.40711588e+00,
       -5.59111403e-01, -6.07296626e-01, -5.39749682e-01, -4.65140284e-01,
       -6.49394410e-01, -4.76707609e-01,  3.35431988e+00, -6.20880658e-01,
       -5.95385129e-01, -5.46811791e-01, -6.69361922e-01, -4.89498197e-01,
       -6.20880658e-01,  3.00095361e+00, -5.94182961e-01, -6.27029876e-01,
       -6.20671086e-01, -4.61989215e-01, -6.84772393e-01, -5.94182961e-01,
        3.49903223e+00, -6.21310000e-01, -5.16106574e-01, -4.55611990e-01,
       -6.18351314e-01, -6.21467192e-01,  3.48463719e+00, -6.42749814e-01,
       -6.63369359e-01, -4.83087518e-01, -3.93400578e-01, -5.12840722e-01,
       -6.42749814e-01,  3.64762816e+00, -7.38648958e-01, -8.51203545e-01,
       -5.08784542e-01, -6.31437241e-01, -5.56187663e-01, -8.00889022e-01,
        3.60234286e+00, -5.18268780e-01, -2.26226706e-01, -8.69333449e-01,
       -6.46568046e-01, -4.13465284e-01, -6.64736135e-01,  3.49223142e+00,
       -6.21876238e-01, -4.31910550e-01, -7.13675164e-01, -5.09585462e-01,
       -6.85085659e-01, -5.18268780e-01,  3.32929590e+00, -3.45090094e-01,
       -5.48329618e-01, -7.22936283e-01, -5.39581004e-01, -5.65673073e-01,
       -6.46274087e-01,  3.46772455e+00, -6.27371901e-01, -5.45647075e-01,
       -5.43177414e-01, -5.83617123e-01, -6.44521093e-01, -4.98065460e-01,
        3.44979971e+00, -5.48936969e-01, -5.84438740e-01, -5.90220322e-01,
       -3.15837817e-01, -5.78983838e-01, -8.99229943e-01, -7.18830961e-01,
        3.59752855e+00, -4.00543701e-01, -6.84102290e-01, -5.72847758e-01,
       -5.54184578e-01, -6.30090371e-01,  3.47224543e+00, -6.21656579e-01,
       -5.37624383e-01, -5.55841757e-01, -6.66268359e-01, -5.95241106e-01,
       -6.21656579e-01,  3.55479971e+00, -4.02479089e-01, -3.98306622e-01,
       -8.70847956e-01, -6.92409279e-01, -5.61582796e-01, -4.02479089e-01,
        3.49400492e+00, -5.60115139e-01, -7.50951799e-01, -5.26466816e-01,
       -7.49836461e-01, -5.47440367e-01,  3.48327618e+00, -5.13826785e-01,
       -4.79022717e-01, -5.95532995e-01, -5.97616850e-01, -5.66306110e-01,
       -6.73685463e-01, -5.13826785e-01,  3.50897433e+00, -5.19477228e-01,
       -6.86293400e-01, -5.49385348e-01, -6.41733967e-01, -5.98464858e-01,
       -4.79022717e-01,  3.47548739e+00, -5.53249666e-01, -6.45574006e-01,
       -5.57442173e-01, -5.57871124e-01, -6.23578747e-01, -5.61031048e-01,
       -5.53249666e-01,  3.47145981e+00, -5.58117489e-01, -6.17611736e-01,
       -5.93248136e-01, -5.85141897e-01, -5.19477228e-01,  3.46369251e+00,
       -5.85618539e-01, -6.12599021e-01, -5.67607691e-01, -5.92500664e-01,
       -5.81592232e-01, -5.58117489e-01,  3.46969337e+00, -5.89310587e-01,
       -5.79482162e-01, -5.68690240e-01, -5.46846610e-01, -5.98044923e-01,
       -5.89310587e-01,  3.46226844e+00, -5.87308381e-01, -5.92904440e-01,
       -5.47853496e-01, -6.59555068e-01, -5.04064183e-01, -5.67335427e-01,
        3.48540213e+00, -5.90407222e-01, -5.24983108e-01, -6.39057119e-01,
       -6.29344646e-01, -5.73609759e-01, -5.50032582e-01,  3.48026830e+00,
       -4.98836734e-01, -5.74780097e-01, -6.53664478e-01, -6.86055765e-01,
       -6.13431168e-01, -4.98836734e-01,  3.51610955e+00, -3.67577575e-01,
       -5.33427387e-01, -8.16780925e-01, -5.96818907e-01, -5.72299096e-01,
       -5.74780097e-01,  3.47718882e+00, -5.91923522e-01, -5.50151675e-01,
       -5.91215523e-01, -5.53945890e-01, -5.81815325e-01, -5.91923522e-01,
        3.48260837e+00, -6.24861797e-01, -6.01862152e-01, -5.28199682e-01,
       -4.74887882e-01, -6.46536747e-01, -6.24861797e-01,  3.48950902e+00,
       -5.90608270e-01, -6.24389036e-01, -5.28225285e-01, -4.80730600e-01,
       -6.36804116e-01, -5.90608270e-01,  3.46825029e+00, -5.68599118e-01,
       -6.61867344e-01, -5.29640839e-01, -5.76177256e-01, -5.12798267e-01,
       -6.75921804e-01,  3.50395134e+00, -6.09920199e-01, -4.87902220e-01,
       -6.41231599e-01, -5.68333560e-01, -6.43114037e-01, -5.63841257e-01,
        3.53985371e+00, -4.99955723e-01, -6.15950317e-01, -6.48658812e-01,
       -5.82864938e-01, -4.69145643e-01,  3.49705587e+00, -6.33576909e-01,
       -7.34325300e-01, -6.44381986e-01, -4.32761097e-01, -5.08094554e-01,
       -5.96191314e-01, -6.33576909e-01,  3.48650560e+00, -6.43986516e-01,
       -6.20832503e-01, -4.83823809e-01, -5.52693612e-01, -5.60392105e-01,
       -6.51873873e-01,  3.46919898e+00, -6.26462819e-01, -5.26845512e-01,
       -5.50931055e-01, -5.52644760e-01, -4.77711128e-01, -7.34325300e-01,
        3.53349214e+00, -7.36723646e-01, -3.72149360e-01, -6.59937942e-01,
       -5.08657956e-01, -6.68483503e-01, -6.21310000e-01,  3.47074068e+00,
       -5.40108212e-01, -6.07330078e-01, -5.24850931e-01, -5.54518025e-01,
       -5.39513063e-01, -5.95385129e-01,  3.45254500e+00, -5.43465651e-01,
       -5.59552510e-01, -6.60110626e-01, -5.96081461e-01, -4.90913748e-01,
       -5.59111403e-01,  3.45390725e+00, -5.61608645e-01, -6.34183847e-01,
       -6.12008149e-01, -5.00052299e-01, -6.07296626e-01, -6.63369359e-01,
        3.37201467e+00, -5.06203029e-01, -6.27743622e-01, -4.67349735e-01,
       -4.15211555e-01, -7.38648958e-01, -2.26226706e-01,  3.42484021e+00,
       -6.98686896e-01, -8.78203492e-01, -4.67862607e-01, -7.00944217e-01,
       -4.45380690e-01, -6.21876238e-01,  3.53377292e+00, -5.92903822e-01,
       -4.19694004e-01, -7.52973950e-01, -6.98525682e-01, -5.78387754e-01,
       -3.45090094e-01,  3.44705006e+00, -6.43217227e-01, -8.20893366e-01,
       -3.60935942e-01, -4.76780190e-01, -5.57333153e-01, -6.43217227e-01,
        3.41616707e+00, -6.85014172e-01, -5.40648324e-01, -5.13174002e-01,
       -5.69534494e-01, -4.31910550e-01, -6.85014172e-01,  3.40493960e+00,
       -6.53097466e-01, -4.56370348e-01, -6.09012572e-01, -5.75002999e-01,
       -5.40648483e-01, -6.27371901e-01,  3.46204106e+00, -6.03049520e-01,
       -5.42624836e-01, -5.73343319e-01, -6.32634019e-01, -4.66728309e-01,
       -6.03049520e-01,  3.43650608e+00, -6.09932695e-01, -5.55241874e-01,
       -5.68919665e-01, -8.10882619e-01, -9.17616614e-01, -6.09932695e-01,
        3.67222412e+00, -7.27084609e-01, -6.06707585e-01, -5.39842096e-01,
       -5.07740555e-01,  3.45259380e+00, -6.34943787e-01, -6.65624163e-01,
       -5.45499747e-01, -5.58943448e-01, -5.50452200e-01, -5.48936969e-01,
       -6.34943787e-01,  3.45987529e+00, -5.94942254e-01, -5.59838877e-01,
       -5.70761204e-01, -5.58056797e-01, -5.14664169e-01, -6.65624163e-01,
        3.47404278e+00, -6.62111896e-01, -5.09708037e-01, -5.63877721e-01,
       -5.38332277e-01, -5.45647075e-01, -6.62111896e-01,  3.47828177e+00,
       -6.57349708e-01, -5.40257592e-01, -5.34583221e-01, -6.32228362e-01,
       -5.84438740e-01,  3.51698072e+00, -5.75252738e-01, -5.23002301e-01,
       -4.95676678e-01, -7.06381898e-01, -6.79449895e-01, -6.38432197e-01,
       -5.75252738e-01,  3.60109372e+00, -5.23140498e-01, -6.06504426e-01,
       -5.78313966e-01, -6.33782285e-01, -7.37941129e-01, -4.00543701e-01,
       -5.23140498e-01,  3.39486829e+00, -3.98593129e-01, -7.00867547e-01,
       -6.24897533e-01, -6.84102290e-01, -5.92903822e-01, -3.98593129e-01,
        3.52652648e+00, -5.73447659e-01, -6.52582049e-01, -4.49094201e-01,
       -5.79620210e-01, -7.27084609e-01,  3.50210483e+00, -5.96029900e-01,
       -5.01566753e-01, -6.48709157e-01, -5.96617174e-01, -5.64409339e-01,
       -5.60115139e-01,  3.47217855e+00, -6.45927622e-01, -5.79757689e-01,
       -5.25351589e-01, -5.48745201e-01, -6.30261973e-01, -5.37624383e-01,
        3.45049140e+00, -5.53539898e-01, -6.58180846e-01, -5.22139098e-01,
       -5.43168221e-01, -5.85618539e-01, -5.53539898e-01,  3.43980356e+00,
       -6.03982226e-01, -6.19769376e-01, -5.33725304e-01, -5.06562917e-01,
       -6.29951679e-01,  3.47920996e+00, -6.09404519e-01, -5.92186956e-01,
       -4.96944469e-01, -6.44159417e-01, -6.45173550e-01, -4.90135850e-01,
       -6.09404519e-01,  3.48007358e+00, -6.06490981e-01, -4.78761689e-01,
       -6.50106988e-01, -6.47090306e-01, -4.89846331e-01, -6.06490981e-01,
        3.47956377e+00, -6.13930769e-01, -4.80866245e-01, -6.41339142e-01,
       -5.18428214e-01, -6.14588797e-01, -5.92186956e-01,  3.46780834e+00,
       -6.04552685e-01, -6.27447420e-01, -5.10604271e-01, -6.41451637e-01,
       -4.96768833e-01, -6.13930769e-01,  3.47682239e+00, -6.06373644e-01,
       -4.79453413e-01, -6.38844093e-01, -6.30621294e-01, -5.06682092e-01,
       -6.17442500e-01, -6.06373644e-01,  3.47180132e+00, -6.23733613e-01,
       -4.86948182e-01, -5.34367624e-01, -5.87308381e-01, -6.04552685e-01,
        3.46766344e+00, -6.04630275e-01, -6.04147773e-01, -5.32656705e-01,
       -4.87520058e-01, -6.48711580e-01, -5.90407222e-01, -6.23733613e-01,
        3.47970769e+00, -4.95433599e-01, -6.33901623e-01, -6.20881470e-01,
       -5.71339729e-01, -4.99955723e-01,  3.60349586e+00, -4.79981839e-01,
       -6.54207364e-01, -7.77129735e-01, -6.86847775e-01, -3.83795369e-01,
       -4.79981839e-01,  3.34257897e+00, -4.70762851e-01, -5.47095742e-01,
       -7.74095397e-01, -7.32783093e-01, -3.67577575e-01, -4.70762851e-01,
        3.71985025e+00, -2.49101323e-01, -7.25002333e-01, -7.25616485e-01,
       -4.49006589e-01, -5.46827199e-01, -5.68599118e-01, -5.96029900e-01,
        3.46347297e+00, -6.09577576e-01, -4.92115933e-01, -6.50323240e-01,
       -6.12223852e-01, -5.46812973e-01,  3.45441635e+00, -5.73633651e-01,
       -5.70308231e-01, -5.77846459e-01, -5.73591182e-01, -5.69017902e-01,
       -6.35991189e-01, -5.73633651e-01,  3.47407546e+00, -5.55118490e-01,
       -5.75023105e-01, -5.65291118e-01, -6.38052781e-01, -5.24983108e-01,
       -5.70308231e-01,  3.48035755e+00, -5.99098724e-01, -5.35248551e-01,
       -6.12666159e-01, -4.71928379e-01, -6.61827332e-01, -5.55118490e-01,
        3.54109057e+00, -5.93882993e-01, -6.44784563e-01, -6.13548810e-01,
       -3.05124139e-01, -8.07084494e-01, -6.09920199e-01,  3.65557572e+00,
       -7.94711519e-01, -4.89039839e-01, -6.49695532e-01, -5.70671446e-01,
       -6.15950317e-01,  3.47587652e+00, -5.81549615e-01, -5.28465054e-01,
       -5.58621662e-01, -6.20618424e-01, -5.86089595e-01, -5.35753864e-01,
       -5.81549615e-01,  3.45410434e+00, -5.89536219e-01, -5.68804886e-01,
       -5.92370160e-01, -5.99053365e-01, -4.87902220e-01, -5.89536219e-01,
        3.43394448e+00, -6.17615514e-01, -5.40589741e-01, -5.99247418e-01,
       -5.36935193e-01, -5.89414780e-01, -6.26462819e-01,  3.47239733e+00,
       -6.11820524e-01, -5.67858900e-01, -5.39905118e-01, -5.26789747e-01,
       -6.11978065e-01, -6.11820524e-01,  3.46815121e+00, -5.99219578e-01,
       -5.89261514e-01, -5.29081785e-01, -5.11862391e-01, -6.43986516e-01,
       -5.99219578e-01,  3.47302757e+00, -5.66686780e-01, -6.16825876e-01,
       -5.34446429e-01, -5.14309216e-01, -5.25736222e-01, -6.45927622e-01,
        3.43857514e+00, -7.01888032e-01, -5.35853890e-01, -5.14860163e-01,
       -5.39471244e-01, -4.88754595e-01, -7.01888032e-01,  3.46426945e+00,
       -7.19559737e-01, -5.08988733e-01, -5.05607104e-01, -5.68893106e-01,
       -4.74378695e-01, -7.19559737e-01,  3.46817992e+00, -6.79820927e-01,
       -4.87491122e-01, -5.38036333e-01, -5.86595770e-01, -5.26845512e-01,
       -6.79820927e-01,  3.48493382e+00, -6.32573854e-01, -4.96141012e-01,
       -5.62956744e-01, -5.39130772e-01, -5.26116737e-01, -7.36723646e-01,
        3.48623788e+00, -7.44476760e-01, -4.08578687e-01, -5.31211274e-01,
       -5.03668100e-01, -5.40108212e-01, -7.44476760e-01,  3.48811394e+00,
       -7.24400752e-01, -4.73315921e-01, -5.02144199e-01, -5.55944514e-01,
       -6.02945844e-01,  3.59411670e+00, -5.54226351e-01, -6.08704863e-01,
       -5.25530422e-01, -7.46764706e-01, -6.58422647e-01, -5.43465651e-01,
       -5.54226351e-01,  3.48513606e+00, -5.28807254e-01, -7.90207132e-01,
       -4.10007028e-01, -5.83854725e-01, -5.61608645e-01, -6.08704863e-01,
        3.53050708e+00, -6.07270142e-01, -5.56870770e-01, -6.12197932e-01,
       -5.39749682e-01, -6.34183847e-01, -5.06203029e-01,  3.47998837e+00,
       -4.97265556e-01, -7.09031297e-01, -5.93554959e-01, -5.46811791e-01,
       -6.27029876e-01, -5.16106574e-01, -6.07330078e-01, -5.59552510e-01,
        3.40969924e+00, -5.52868411e-01, -4.83087518e-01, -8.51203545e-01,
       -6.27743622e-01,  3.29525020e+00, -5.35160693e-01, -5.47118789e-01,
       -2.50936033e-01, -5.08784542e-01, -6.98686896e-01, -5.35160693e-01,
        3.45874561e+00, -5.47325876e-01, -8.19286096e-01, -3.49501506e-01,
       -8.69333449e-01, -5.48329618e-01, -8.78203492e-01,  3.43554033e+00,
       -5.34457290e-01, -6.05216477e-01, -7.13675164e-01, -4.19694004e-01,
       -6.53097466e-01,  3.56137285e+00, -6.53450315e-01, -4.23344723e-01,
       -6.98111182e-01, -7.22936283e-01, -8.20893366e-01, -5.34457290e-01,
        3.59818648e+00, -5.87857455e-01, -9.32042084e-01, -5.43177414e-01,
       -5.42624836e-01, -6.57349708e-01,  3.45735103e+00, -6.03651538e-01,
       -5.30783416e-01, -5.79764121e-01, -5.90220322e-01, -5.94942254e-01,
       -5.23002301e-01,  3.45862992e+00, -5.64667451e-01, -5.66455234e-01,
       -6.19342358e-01, -5.55841757e-01, -3.98306622e-01, -6.58180846e-01,
        3.51075924e+00, -8.54480182e-01, -5.66701258e-01, -4.77248578e-01,
       -8.70847956e-01, -7.50951799e-01, -8.54480182e-01,  3.64931145e+00,
       -5.28854515e-01, -6.44176998e-01, -5.26466816e-01, -5.79757689e-01,
       -5.28854515e-01,  3.45838015e+00, -6.48727113e-01, -6.05961728e-01,
       -5.68612291e-01, -5.95532995e-01, -6.86293400e-01,  3.50571769e+00,
       -4.72100185e-01, -5.76800923e-01, -4.84231441e-01, -6.90758745e-01,
       -5.97616850e-01, -6.45574006e-01, -4.72100185e-01,  3.49823407e+00,
       -5.64319142e-01, -6.91796809e-01, -5.26827081e-01, -5.49385348e-01,
       -6.12599021e-01, -5.76800923e-01,  3.48036345e+00, -5.45378833e-01,
       -5.79359722e-01, -6.16839604e-01, -5.57442173e-01, -6.17611736e-01,
       -5.79482162e-01, -5.64319142e-01,  3.47780436e+00, -5.58476887e-01,
       -6.00472257e-01, -5.67607691e-01, -6.03982226e-01, -5.45378833e-01,
        3.45647892e+00, -5.63368255e-01, -5.70797742e-01, -6.05344175e-01,
       -5.68690240e-01, -5.92904440e-01, -5.58476887e-01,  3.45519354e+00,
       -5.74504289e-01, -6.00151331e-01, -5.60466352e-01, -5.47853496e-01,
       -6.04630275e-01, -5.74504289e-01,  3.47189606e+00, -5.85182121e-01,
       -6.08581216e-01, -5.51144660e-01, -6.39057119e-01, -4.95433599e-01,
       -5.99098724e-01,  3.48452963e+00, -6.17968851e-01, -4.99274725e-01,
       -6.33696610e-01, -6.53664478e-01, -5.33427387e-01, -5.50151675e-01,
        3.45382228e+00, -5.12341614e-01, -5.40512188e-01, -6.63724933e-01,
       -8.16780925e-01, -2.49101323e-01, -5.12341614e-01,  3.66911946e+00,
       -1.00282440e+00, -4.81026348e-01, -6.07044848e-01, -5.91215523e-01,
       -6.01862152e-01, -5.40512188e-01,  3.45808616e+00, -5.50277949e-01,
       -5.83714187e-01, -5.90504163e-01, -5.28199682e-01, -6.24389036e-01,
       -5.50277949e-01,  3.40572099e+00, -5.47009148e-01, -6.17040092e-01,
       -5.38805083e-01, -5.28225285e-01, -6.61867344e-01, -5.47009148e-01,
        3.47249403e+00, -5.36097147e-01, -6.48505102e-01, -5.50790006e-01,
       -5.29640839e-01, -6.09577576e-01, -5.36097147e-01,  3.40334071e+00,
       -5.29509515e-01, -6.34066951e-01, -5.64448677e-01, -4.17397718e-01,
       -5.93882993e-01, -7.94711519e-01,  3.60212886e+00, -6.62333309e-01,
       -4.70620010e-01, -6.63183316e-01, -6.41231599e-01, -4.89039839e-01,
       -6.17615514e-01,  3.47699471e+00, -6.15092656e-01, -5.22478076e-01,
       -5.91537023e-01, -6.48658812e-01, -6.54207364e-01, -5.28465054e-01,
        3.54696719e+00, -4.04871656e-01, -5.05008100e-01, -8.05756203e-01,
       -6.44381986e-01, -3.72149360e-01,  3.51758911e+00, -7.33652642e-01,
       -7.06558573e-01, -4.33481526e-01, -6.27365025e-01, -4.32761097e-01,
       -6.20832503e-01, -7.33652642e-01,  3.54743621e+00, -7.00111825e-01,
       -6.93471414e-01, -3.66606724e-01, -4.83823809e-01, -5.66686780e-01,
       -7.00111825e-01,  3.47730780e+00, -6.65001180e-01, -5.53439885e-01,
       -5.08244318e-01, -5.50931055e-01, -5.67858900e-01, -6.32573854e-01,
        3.45799132e+00, -6.15431832e-01, -5.37238221e-01, -5.53957456e-01,
       -6.59937942e-01, -4.08578687e-01, -7.06558573e-01,  3.51492065e+00,
       -7.25171253e-01, -4.53041496e-01, -5.61632698e-01, -5.24850931e-01,
       -6.60110626e-01, -7.24400752e-01, -5.28807254e-01, -5.52868411e-01,
        3.46764216e+00, -4.76604182e-01, -6.12008149e-01, -6.07270142e-01,
       -4.97265556e-01,  3.48118571e+00, -4.81302814e-01, -6.49299820e-01,
       -6.34039229e-01, -4.67349735e-01, -7.09031297e-01, -5.47118789e-01,
        3.49174090e+00, -5.17483494e-01, -7.66607195e-01, -4.84150394e-01,
       -4.67862607e-01, -5.47325876e-01, -6.05216477e-01, -5.87857455e-01,
        3.44289046e+00, -4.21312716e-01, -8.13315324e-01, -7.52973950e-01,
       -5.73447659e-01, -6.53450315e-01,  3.69421195e+00, -4.64590634e-01,
       -4.13183076e-01, -8.36566312e-01, -3.60935942e-01, -5.40648324e-01,
       -9.32042084e-01, -4.21312716e-01,  3.53812225e+00, -7.47953173e-01,
       -5.35230016e-01, -5.13174002e-01, -4.56370348e-01, -7.47953173e-01,
        3.45348322e+00, -6.97886606e-01, -4.13407956e-01, -6.24691136e-01,
       -6.09012572e-01, -4.23344723e-01, -6.97886606e-01,  3.49034245e+00,
       -6.51975894e-01, -4.58688453e-01, -6.49434203e-01, -5.73343319e-01,
       -5.55241874e-01, -6.03651538e-01,  3.43980572e+00, -5.45550200e-01,
       -5.47440249e-01, -6.14578541e-01, -5.68919665e-01, -6.06707585e-01,
       -5.01566753e-01, -5.45550200e-01,  3.48936659e+00, -6.08314282e-01,
       -6.58308100e-01, -5.45499747e-01, -5.59838877e-01,  3.43895965e+00,
       -6.23635947e-01, -5.91013636e-01, -5.60126527e-01, -5.58844915e-01,
       -5.58943448e-01, -5.09708037e-01, -6.23635947e-01,  3.41173186e+00,
       -6.48491145e-01, -5.42020260e-01, -5.28933022e-01, -5.70761204e-01,
       -5.64667451e-01, -5.91013636e-01,  3.45142595e+00, -5.61202712e-01,
       -5.93112443e-01, -5.70668500e-01, -5.63877721e-01, -5.40257592e-01,
       -6.48491145e-01,  3.48824140e+00, -6.58309938e-01, -5.54111799e-01,
       -5.23193200e-01, -5.34583221e-01, -5.30783416e-01, -6.58309938e-01,
        3.45690067e+00, -6.19016094e-01, -5.39500522e-01, -5.74707476e-01,
       -4.95676678e-01, -6.06504426e-01,  3.49094512e+00, -4.41131992e-01,
       -6.48638408e-01, -4.85057892e-01, -8.13935721e-01, -7.06381898e-01,
       -5.66455234e-01, -4.41131992e-01,  3.49891296e+00, -5.29841314e-01,
       -6.46866578e-01, -6.08235946e-01, -5.78313966e-01, -7.00867547e-01,
       -6.52582049e-01, -4.64590634e-01, -6.48638408e-01,  3.66098073e+00,
       -6.15988124e-01, -6.48709157e-01, -4.92115933e-01, -6.08314282e-01,
        3.50241793e+00, -4.89744524e-01, -3.80837704e-01, -8.05378267e-01,
       -7.73180605e-02, -5.25351589e-01, -5.35853890e-01, -6.48727113e-01,
        3.46315132e+00, -7.10218241e-01, -5.45997834e-01, -4.97002659e-01,
       -5.22139098e-01, -6.19769376e-01, -5.66701258e-01,  3.44215853e+00,
       -5.65613548e-01, -6.12599398e-01, -5.55335855e-01, -5.33725304e-01,
       -5.63368255e-01, -5.65613548e-01,  3.39942593e+00, -5.95326884e-01,
       -5.62176583e-01, -5.79215360e-01, -4.96944469e-01, -6.27447420e-01,
        3.47729282e+00, -6.17358323e-01, -6.13715932e-01, -6.32283783e-01,
       -4.89542894e-01, -6.44159417e-01, -4.78761689e-01, -6.17358323e-01,
        3.48507632e+00, -6.19549893e-01, -4.80628958e-01, -6.44618040e-01,
       -6.50106988e-01, -4.80866245e-01, -6.19549893e-01,  3.48809745e+00,
       -6.18774735e-01, -4.71085342e-01, -6.47714242e-01, -6.41339142e-01,
       -4.79453413e-01, -6.18774735e-01,  3.48417114e+00, -6.23957954e-01,
       -6.44426223e-01, -4.76219676e-01, -5.10604271e-01, -6.04147773e-01,
       -6.13715932e-01,  3.47279712e+00, -6.13267544e-01, -6.22003755e-01,
       -5.09057841e-01, -6.38844093e-01, -4.86948182e-01, -6.33901623e-01,
       -6.17968851e-01, -6.23957954e-01,  3.48404118e+00, -4.82420474e-01,
       -5.32656705e-01, -5.85182121e-01, -6.13267544e-01,  3.47418926e+00,
       -6.24313853e-01, -6.03546741e-01, -5.15222293e-01, -7.77129735e-01,
       -5.47095742e-01, -4.04871656e-01,  3.54875154e+00, -5.26907186e-01,
        2.82972242e-03, -4.48015740e-01, -8.47561201e-01, -7.74095397e-01,
       -7.25002333e-01, -5.26907186e-01,  3.78509164e+00, -5.62939744e-01,
       -6.94045902e-01, -5.02101074e-01, -7.25616485e-01, -1.00282440e+00,
        3.80042613e+00, -9.79134482e-01, -6.91723282e-01, -4.01127477e-01,
       -6.50323240e-01, -5.29509515e-01, -4.89744524e-01,  3.46604555e+00,
       -5.89703424e-01, -6.11681173e-01, -5.95083675e-01, -5.77846459e-01,
       -5.75023105e-01,  3.47144982e+00, -6.01907443e-01, -5.82022707e-01,
       -5.34962225e-01, -5.99687879e-01, -5.73591182e-01, -5.35248551e-01,
       -6.01907443e-01,  3.45951570e+00, -6.14974332e-01, -5.58247476e-01,
       -5.75546718e-01, -5.65291118e-01, -6.44784563e-01, -5.82022707e-01,
        3.49714890e+00, -5.32631821e-01, -5.71780646e-01, -6.00638047e-01,
       -6.12666159e-01, -4.99274725e-01, -6.14974332e-01,  3.49234461e+00,
       -6.34901458e-01, -5.05802297e-01, -6.24725637e-01, -6.13548810e-01,
       -6.62333309e-01, -5.32631821e-01,  3.53017620e+00, -4.44703244e-01,
       -6.54512891e-01, -6.22446124e-01, -6.49695532e-01, -4.70620010e-01,
       -6.15092656e-01,  3.47774823e+00, -6.02104442e-01, -5.35222758e-01,
       -6.05012829e-01, -5.58621662e-01, -5.68804886e-01,  3.42072539e+00,
       -6.01781590e-01, -5.69493740e-01, -5.74801149e-01, -5.47222362e-01,
       -6.20618424e-01, -5.05008100e-01, -6.01781590e-01,  3.45267252e+00,
       -6.13212368e-01, -5.17865337e-01, -5.94186701e-01, -5.92370160e-01,
       -5.40589741e-01, -5.69493740e-01,  3.45768161e+00, -5.86561407e-01,
       -5.88921770e-01, -5.79744794e-01, -5.99247418e-01, -5.22478076e-01,
       -5.86561407e-01,  3.44436374e+00, -5.86077412e-01, -5.90136452e-01,
       -5.59862978e-01, -5.39905118e-01, -5.89261514e-01, -6.15431832e-01,
        3.46584434e+00, -6.03217760e-01, -5.67396489e-01, -5.50631628e-01,
       -5.29081785e-01, -6.16825876e-01, -6.03217760e-01,  3.46391620e+00,
       -5.69155942e-01, -5.83120958e-01, -5.62513878e-01, -5.34446429e-01,
       -6.65001180e-01, -5.69155942e-01,  3.47893823e+00, -5.39960602e-01,
       -6.01207171e-01, -5.69166911e-01, -5.14860163e-01, -5.08988733e-01,
       -7.10218241e-01,  3.49011723e+00, -7.49632665e-01, -5.32211469e-01,
       -4.74205953e-01, -5.05607104e-01, -4.87491122e-01, -7.49632665e-01,
        3.49454457e+00, -7.67907466e-01, -5.47037691e-01, -4.36868518e-01,
       -5.38036333e-01, -4.96141012e-01, -7.67907466e-01,  3.48105323e+00,
       -6.92930385e-01, -4.73644163e-01, -5.12393869e-01, -5.62956744e-01,
       -5.37238221e-01, -6.92930385e-01,  3.49481787e+00, -6.42930780e-01,
       -4.51942612e-01, -6.06819131e-01, -5.31211274e-01, -4.73315921e-01,
       -7.25171253e-01,  3.69499119e+00, -1.13829605e+00, -8.26996699e-01,
       -5.02144199e-01, -7.90207132e-01, -4.76604182e-01, -1.13829605e+00,
        3.77106036e+00, -8.33445103e-01, -3.03637015e-02, -5.25530422e-01,
       -5.56870770e-01,  3.50214949e+00, -6.27343669e-01, -6.42926964e-01,
       -6.33484465e-01, -5.15993203e-01, -7.46764706e-01, -4.10007028e-01,
       -8.33445103e-01, -6.27343669e-01,  3.62362602e+00, -4.75107763e-01,
       -5.30957749e-01, -6.12197932e-01, -4.81302814e-01, -6.42926964e-01,
        3.49383263e+00, -6.75576418e-01, -4.41601543e-01, -6.40226959e-01,
       -5.93554959e-01, -6.49299820e-01, -5.17483494e-01,  3.48817605e+00,
       -4.50195009e-01, -6.68466873e-01, -6.09175899e-01, -2.50936033e-01,
       -8.19286096e-01, -7.66607195e-01,  3.48459013e+00, -6.71471750e-01,
       -5.56418119e-01, -4.19870935e-01, -3.49501506e-01, -8.13315324e-01,
       -5.35230016e-01, -4.13407956e-01, -6.71471750e-01,  3.47588903e+00,
       -6.92962474e-01, -6.98111182e-01, -4.13183076e-01, -6.51975894e-01,
        3.82520880e+00, -1.52315149e-02, -6.06114295e-01, -1.10976180e+00,
       -3.30831041e-01, -5.79764121e-01, -5.47440249e-01, -6.19016094e-01,
        3.46108025e+00, -5.49919814e-01, -5.22769057e-01, -6.42170913e-01,
       -6.19342358e-01, -5.61202712e-01, -5.29841314e-01,  3.46764868e+00,
       -6.53381409e-01, -5.77428538e-01, -5.26452350e-01, -4.77248578e-01,
       -6.44176998e-01, -6.05961728e-01, -6.12599398e-01,  3.47429915e+00,
       -5.51233878e-01, -5.83078571e-01, -5.68612291e-01, -5.45997834e-01,
       -5.51233878e-01,  3.44816671e+00, -6.51785404e-01, -5.65905739e-01,
       -5.64631562e-01, -4.84231441e-01, -6.91796809e-01,  3.46637741e+00,
       -5.09505640e-01, -5.74136986e-01, -7.05989380e-01, -5.00717157e-01,
       -6.90758745e-01, -5.79359722e-01, -5.09505640e-01,  3.53750965e+00,
       -5.93897193e-01, -4.43682827e-01, -7.20305523e-01, -5.26827081e-01,
       -6.00472257e-01, -6.00151331e-01, -5.74136986e-01,  3.44414562e+00,
       -5.47968810e-01, -5.94589154e-01, -6.16839604e-01, -5.70797742e-01,
       -5.93897193e-01,  3.49108110e+00, -5.50405614e-01, -4.65730682e-01,
       -6.93410264e-01, -6.05344175e-01, -5.95326884e-01, -5.50405614e-01,
        3.44260447e+00, -5.46457344e-01, -5.02329321e-01, -6.42741135e-01,
       -5.60466352e-01, -6.08581216e-01, -5.47968810e-01,  3.46786255e+00,
       -5.73296109e-01, -6.34513911e-01, -5.43036158e-01, -5.51144660e-01,
       -6.24313853e-01, -5.73296109e-01,  3.48879417e+00, -5.86349972e-01,
       -6.38688516e-01, -5.15001063e-01, -6.33696610e-01, -6.44426223e-01,
       -4.82420474e-01, -6.34901458e-01,  3.49733874e+00, -6.34226500e-01,
       -4.67667474e-01, -6.63724933e-01, -4.81026348e-01, -5.83714187e-01,
        3.46030365e+00, -5.59729466e-01, -5.16667447e-01, -6.55441266e-01,
       -4.49006589e-01, -5.62939744e-01, -9.79134482e-01,  3.53296455e+00,
       -5.44676491e-01, -5.80115031e-01, -4.17092217e-01, -6.07044848e-01,
       -6.91723282e-01, -5.59729466e-01,  3.57583516e+00, -5.07423439e-01,
       -4.60540829e-01, -7.49373295e-01, -5.90504163e-01, -6.17040092e-01,
       -5.16667447e-01,  3.46306558e+00, -5.54412294e-01, -6.16513521e-01,
       -5.67928060e-01, -5.38805083e-01, -6.48505102e-01, -5.54412294e-01,
        3.46823416e+00, -5.48981285e-01, -6.47478662e-01, -5.30051739e-01,
       -5.50790006e-01, -6.34066951e-01, -5.48981285e-01,  3.47339024e+00,
       -5.51090721e-01, -6.55164111e-01, -5.33297170e-01, -5.64448677e-01,
       -5.89703424e-01, -5.51090721e-01,  3.46542883e+00, -5.94922387e-01,
       -6.17510420e-01, -5.47753206e-01, -6.63183316e-01, -4.44703244e-01,
       -6.02104442e-01,  3.47906910e+00, -6.16843652e-01, -5.34907142e-01,
       -6.17327306e-01, -5.91537023e-01, -5.35222758e-01, -5.86077412e-01,
        3.44958322e+00, -5.66255323e-01, -6.17250935e-01, -5.53239768e-01,
       -8.05756203e-01,  2.82972242e-03, -6.13212368e-01,  4.37369288e+00,
       -4.12559611e-01, -2.07136036e+00, -4.73634052e-01, -4.33481526e-01,
       -6.93471414e-01,  3.15628624e+00, -5.05376872e-01, -8.01959885e-01,
       -7.21996540e-01, -6.27365025e-01, -4.53041496e-01, -5.05376872e-01,
        3.26004717e+00, -4.86457535e-01, -6.04186005e-01, -5.83620234e-01,
       -3.66606724e-01, -5.53439885e-01, -8.01959885e-01,  3.38588785e+00,
       -7.32694572e-01, -3.77173263e-01, -5.54013527e-01, -5.08244318e-01,
       -5.39960602e-01, -7.32694572e-01,  3.38463347e+00, -6.76680239e-01,
       -4.64849433e-01, -4.62204308e-01, -5.53957456e-01, -5.67396489e-01,
       -6.42930780e-01,  3.47236232e+00, -6.24375931e-01, -5.25590790e-01,
       -5.58110876e-01, -5.61632698e-01, -8.26996699e-01, -3.03637015e-02,
       -4.75107763e-01, -4.86457535e-01,  3.77183840e+00, -6.74674884e-01,
       -7.16605120e-01, -6.34039229e-01, -6.75576418e-01, -4.50195009e-01,
        3.51577041e+00, -4.32372888e-01, -7.10876127e-01, -6.12710744e-01,
       -4.84150394e-01, -6.68466873e-01, -5.56418119e-01,  3.46125453e+00,
       -7.08531840e-01, -5.25735483e-01, -5.17951824e-01, -8.36566312e-01,
       -4.85057892e-01, -6.15988124e-01, -1.52315149e-02,  3.88841520e+00,
       -3.22965997e-01, -9.67247375e-01, -6.45357985e-01, -6.24691136e-01,
       -4.58688453e-01, -4.19870935e-01, -6.92962474e-01, -7.08531840e-01,
        3.48558374e+00, -5.80838899e-01, -6.49434203e-01, -6.06114295e-01,
       -5.25735483e-01, -5.80838899e-01,  3.51448679e+00, -6.84975192e-01,
       -4.67388717e-01, -6.14578541e-01, -6.58308100e-01, -3.80837704e-01,
       -5.49919814e-01,  3.51950048e+00, -4.93693718e-01, -8.22162600e-01,
       -5.60126527e-01, -5.42020260e-01,  3.46977828e+00, -6.04155802e-01,
       -6.46355413e-01, -5.42255587e-01, -5.74864687e-01, -5.58844915e-01,
       -5.93112443e-01, -6.04155802e-01,  3.44449964e+00, -5.31676297e-01,
       -5.63819692e-01, -5.92890488e-01, -5.28933022e-01, -5.54111799e-01,
       -6.46355413e-01,  3.49309195e+00, -6.59646528e-01, -5.72345412e-01,
       -5.31699775e-01, -5.70668500e-01, -6.53381409e-01, -5.31676297e-01,
        3.47769245e+00, -5.66152274e-01, -7.74484510e-01, -3.81329456e-01,
       -5.23193200e-01, -5.39500522e-01, -6.59646528e-01,  3.46706974e+00,
       -6.44980964e-01, -5.34126500e-01, -5.65622028e-01, -5.74707476e-01,
       -5.22769057e-01, -6.44980964e-01,  3.46878557e+00, -6.16533877e-01,
       -5.22775999e-01, -5.87018192e-01, -8.13935721e-01, -6.46866578e-01,
       -3.22965997e-01,  3.71222584e+00, -5.31424527e-01, -6.74203503e-01,
       -7.22829511e-01, -6.08235946e-01, -5.77428538e-01, -5.31424527e-01,
        3.49138019e+00, -6.55990903e-01, -6.08797435e-01, -5.09502839e-01,
       -8.05378267e-01, -6.11681173e-01,  4.06335396e+00, -4.65479783e-01,
       -1.71995875e+00, -4.60855992e-01, -4.97002659e-01, -5.32211469e-01,
       -6.51785404e-01,  3.41106417e+00, -6.87382030e-01, -5.29100241e-01,
       -5.13582368e-01, -5.55335855e-01, -5.62176583e-01, -5.83078571e-01,
       -5.65905739e-01,  3.46109460e+00, -6.11661911e-01, -5.82935936e-01,
       -5.79215360e-01, -5.46457344e-01, -6.11661911e-01,  3.41649010e+00,
       -5.32024666e-01, -5.11312233e-01, -6.35818585e-01, -6.32283783e-01,
       -4.80628958e-01,  3.49028696e+00, -6.31954404e-01, -6.35709291e-01,
       -4.92564722e-01, -6.17145797e-01, -4.89542894e-01, -6.22003755e-01,
       -6.31954404e-01,  3.47671186e+00, -6.14248416e-01, -5.98336498e-01,
       -5.20625897e-01, -6.44618040e-01, -4.71085342e-01, -6.35709291e-01,
        3.49921068e+00, -6.39781641e-01, -4.73508090e-01, -6.34508274e-01,
       -6.47714242e-01, -4.76219676e-01, -6.34226500e-01, -6.39781641e-01,
        3.49366877e+00, -6.40339409e-01, -4.55387300e-01, -5.09057841e-01,
       -6.03546741e-01, -6.14248416e-01,  3.46947384e+00, -6.19103862e-01,
       -6.03842230e-01, -5.19674755e-01, -5.15222293e-01, -5.86349972e-01,
       -6.19103862e-01,  3.46358502e+00, -6.32102273e-01, -6.01397083e-01,
       -5.09409540e-01, -4.48015740e-01, -6.94045902e-01,  3.58858915e+00,
       -6.74559216e-01, -6.27259316e-01, -3.32969761e-01, -8.11739218e-01,
       -5.02101074e-01, -5.44676491e-01, -6.74559216e-01,  3.55007196e+00,
       -6.87401650e-01, -7.08067214e-01, -4.33266316e-01, -4.01127477e-01,
       -5.80115031e-01, -5.07423439e-01,  3.64076523e+00, -6.22086207e-01,
       -3.73158149e-01, -6.63611478e-01, -4.93243448e-01, -5.95083675e-01,
       -5.94922387e-01, -4.65479783e-01,  3.45500747e+00, -5.47631159e-01,
       -7.13741174e-01, -5.38149289e-01, -5.34962225e-01, -5.58247476e-01,
        3.47019998e+00, -6.09323292e-01, -6.34317595e-01, -5.29714782e-01,
       -6.03634614e-01, -5.99687879e-01, -5.71780646e-01, -6.09323292e-01,
        3.48522852e+00, -5.47321739e-01, -5.12862787e-01, -6.44252181e-01,
       -5.75546718e-01, -5.05802297e-01, -6.34317595e-01,  3.46375303e+00,
       -6.39346039e-01, -5.27118294e-01, -5.81622085e-01, -6.00638047e-01,
       -6.54512891e-01, -5.47321739e-01,  3.51649372e+00, -4.64172832e-01,
       -6.38947473e-01, -6.10900742e-01, -6.24725637e-01, -4.67667474e-01,
       -6.40339409e-01, -6.39346039e-01,  3.50316884e+00, -6.48734310e-01,
       -4.82355974e-01, -6.22446124e-01, -6.16843652e-01, -4.64172832e-01,
        3.47902785e+00, -5.34384569e-01, -6.16932805e-01, -6.24247872e-01,
       -6.05012829e-01, -5.34907142e-01, -5.66255323e-01,  3.45755410e+00,
       -5.72299266e-01, -6.16841855e-01, -5.62237682e-01, -5.74801149e-01,
       -5.17865337e-01,  3.46479994e+00, -6.02155279e-01, -6.71097993e-01,
       -5.99725828e-01, -4.99154358e-01, -5.47222362e-01, -5.88921770e-01,
       -6.02155279e-01,  3.45878161e+00, -5.48343973e-01, -5.97259807e-01,
       -5.74878421e-01, -5.94186701e-01, -4.12559611e-01, -6.71097993e-01,
        3.56777567e+00, -7.46010241e-01, -6.44095014e-01, -4.99826109e-01,
       -5.79744794e-01, -5.90136452e-01, -5.48343973e-01,  3.46015241e+00,
       -5.52766709e-01, -6.67385889e-01, -5.21774591e-01, -5.59862978e-01,
       -6.17250935e-01, -5.52766709e-01,  3.48555424e+00, -5.50543265e-01,
       -6.97455558e-01, -5.07674798e-01, -5.50631628e-01, -5.83120958e-01,
       -6.24375931e-01,  3.46580356e+00, -5.85181389e-01, -5.47067316e-01,
       -5.75426342e-01, -5.62513878e-01, -6.01207171e-01, -5.85181389e-01,
        3.45904321e+00, -5.85369900e-01, -5.51980148e-01, -5.72790724e-01,
       -5.69166911e-01, -6.76680239e-01, -5.85369900e-01,  3.49222061e+00,
       -5.50870233e-01, -6.49833869e-01, -4.60299457e-01, -4.74205953e-01,
       -5.47037691e-01, -6.87382030e-01,  3.47878081e+00, -7.73428652e-01,
       -6.05200161e-01, -3.91526319e-01, -4.36868518e-01, -4.73644163e-01,
       -7.73428652e-01,  3.87916258e+00, -1.35959537e+00, -8.35625879e-01,
       -5.12393869e-01, -4.51942612e-01, -1.35959537e+00,  3.83837496e+00,
       -5.96215080e-01, -5.17353895e-02, -8.66492636e-01, -6.06819131e-01,
       -5.25590790e-01, -5.96215080e-01,  3.46477336e+00, -6.38400178e-01,
       -4.90512756e-01, -6.07235426e-01, -6.33484465e-01, -4.41601543e-01,
        3.51428218e+00, -6.89288669e-01, -6.90694505e-01, -4.15519629e-01,
       -6.43693369e-01, -5.15993203e-01, -5.30957749e-01, -6.74674884e-01,
       -6.89288669e-01,  3.48548555e+00, -4.14264018e-01, -6.60307025e-01,
       -6.40226959e-01, -4.32372888e-01, -6.90694505e-01,  3.54189907e+00,
       -7.67416997e-01, -4.17020788e-01, -5.94166933e-01, -6.09175899e-01,
       -7.10876127e-01, -5.17951824e-01, -6.84975192e-01,  3.58152624e+00,
       -6.90135046e-01, -3.68412150e-01, -1.10976180e+00, -9.67247375e-01,
        3.86015203e+00, -6.18713596e-01, -5.73838409e-01, -5.90590846e-01,
       -3.30831041e-01, -4.67388717e-01, -6.90135046e-01, -6.18713596e-01,
        3.65877528e+00, -7.48863165e-01, -8.02843714e-01, -6.42170913e-01,
       -4.93693718e-01, -6.16533877e-01,  3.47107908e+00, -5.54965744e-01,
       -4.99282730e-01, -6.64432101e-01, -7.73180605e-02, -8.22162600e-01,
       -1.71995875e+00, -5.54965744e-01,  4.10007209e+00, -5.42898751e-01,
       -3.82768183e-01, -5.26452350e-01, -5.66152274e-01, -6.55990903e-01,
        3.50601151e+00, -7.61217848e-01, -5.67165986e-01, -4.29032147e-01,
       -5.64631562e-01, -5.29100241e-01, -5.82935936e-01, -5.32024666e-01,
        3.39984729e+00, -6.42026226e-01, -5.49128658e-01, -7.05989380e-01,
       -4.43682827e-01,  3.48353037e+00, -5.93498887e-01, -5.96874793e-01,
       -6.43430963e-01, -5.00053524e-01, -5.00717157e-01, -5.94589154e-01,
       -6.34513911e-01, -5.93498887e-01,  3.43512776e+00, -5.40076813e-01,
       -5.71731835e-01, -7.20305523e-01, -4.65730682e-01, -5.96874793e-01,
        3.52916077e+00, -6.51006664e-01, -4.37860617e-01, -6.57382492e-01,
       -6.93410264e-01, -5.02329321e-01, -6.51006664e-01,  3.59187680e+00,
       -7.78489740e-01, -5.35314961e-01, -4.31325847e-01, -6.42741135e-01,
       -5.11312233e-01, -7.78489740e-01,  3.67470606e+00, -5.41802870e-01,
       -1.35795128e-01, -1.06456496e+00, -5.43036158e-01, -6.38688516e-01,
       -5.40076813e-01,  3.47743782e+00, -5.79903402e-01, -6.59576881e-01,
       -5.16156051e-01, -5.15001063e-01, -6.32102273e-01, -5.79903402e-01,
        3.45481880e+00, -5.85030135e-01, -6.42137596e-01, -5.00644332e-01,
       -8.47561201e-01, -2.07136036e+00, -6.27259316e-01,  4.41928692e+00,
       -5.64230220e-01, -3.08875816e-01, -6.55441266e-01, -4.60540829e-01,
       -6.16513521e-01,  3.51488547e+00, -6.72452357e-01, -5.13340389e-01,
       -5.96597103e-01, -4.17092217e-01, -6.87401650e-01, -6.22086207e-01,
        3.48558795e+00, -6.68061666e-01, -5.72096720e-01, -5.18849491e-01,
       -7.49373295e-01, -3.73158149e-01, -6.72452357e-01,  3.86171911e+00,
       -1.18915137e+00, -5.51323628e-01, -3.26260313e-01, -5.67928060e-01,
       -6.47478662e-01, -5.13340389e-01,  3.46997871e+00, -5.40547291e-01,
       -6.65284903e-01, -5.35399407e-01, -5.30051739e-01, -6.55164111e-01,
       -5.40547291e-01,  3.47110278e+00, -5.67345641e-01, -6.68750996e-01,
       -5.09243008e-01, -5.33297170e-01, -6.17510420e-01, -5.67345641e-01,
        3.47262518e+00, -5.99010258e-01, -6.39581507e-01, -5.15880187e-01,
       -5.47753206e-01, -5.47631159e-01, -5.99010258e-01,  3.47366643e+00,
       -6.54302777e-01, -5.84918251e-01, -5.40050777e-01, -6.17327306e-01,
       -5.34384569e-01, -5.72299266e-01,  3.47468827e+00, -5.52264889e-01,
       -5.81522796e-01, -6.16889445e-01, -5.53239768e-01, -6.16841855e-01,
       -5.50543265e-01,  3.48005768e+00, -5.68320921e-01, -6.62789809e-01,
       -5.28322062e-01, -4.73634052e-01, -7.46010241e-01, -5.64230220e-01,
        3.63766805e+00, -6.97381181e-01, -5.52142890e-01, -6.04269466e-01,
       -7.21996540e-01, -6.04186005e-01, -3.77173263e-01,  3.46680948e+00,
       -4.72708329e-01, -7.26491890e-01, -5.64253458e-01, -5.83620234e-01,
       -7.16605120e-01, -4.14264018e-01, -4.72708329e-01,  3.67045005e+00,
       -6.89555398e-01, -7.93696949e-01, -5.54013527e-01, -4.64849433e-01,
       -7.26491890e-01,  3.45499320e+00, -7.38937173e-01, -4.59780073e-01,
       -5.10921104e-01, -4.62204308e-01, -5.50870233e-01, -7.38937173e-01,
        3.46750924e+00, -7.08033199e-01, -5.44443331e-01, -4.63021000e-01,
       -5.58110876e-01, -5.47067316e-01, -6.38400178e-01,  3.47953677e+00,
       -6.63219442e-01, -5.41001800e-01, -5.31737161e-01, -6.12710744e-01,
       -7.67416997e-01, -3.68412150e-01, -7.48863165e-01,  3.60134222e+00,
       -4.61744348e-01, -2.79341138e-01, -3.62853683e-01, -6.45357985e-01,
       -6.74203503e-01, -5.73838409e-01,  3.55945108e+00, -3.80601219e-01,
       -5.79969182e-01, -7.05480786e-01, -5.42255587e-01, -5.72345412e-01,
        3.46955297e+00, -6.00519621e-01, -6.07910558e-01, -5.65234985e-01,
       -5.81286802e-01, -5.74864687e-01, -5.63819692e-01, -6.00519621e-01,
        3.46564758e+00, -5.37396586e-01, -5.47843087e-01, -6.41203903e-01,
       -5.92890488e-01, -7.74484510e-01, -5.37396586e-01,  3.56926087e+00,
       -2.49243313e-01, -5.43579717e-01, -8.71666260e-01, -5.31699775e-01,
       -5.34126500e-01, -6.07910558e-01,  3.42177011e+00, -6.40565695e-01,
       -5.41740303e-01, -5.65727278e-01, -3.81329456e-01, -7.61217848e-01,
       -2.49243313e-01,  3.58146930e+00, -5.94211394e-01, -9.65589342e-01,
       -4.30350242e-01, -1.99527710e-01, -5.65622028e-01, -5.22775999e-01,
       -6.40565695e-01,  3.46448905e+00, -6.30881431e-01, -5.27796631e-01,
       -5.76847265e-01, -5.87018192e-01, -4.99282730e-01, -6.30881431e-01,
        3.47866938e+00, -6.51695293e-01, -5.44190355e-01, -5.65601382e-01,
       -7.22829511e-01, -6.08797435e-01, -3.80601219e-01,  3.51224840e+00,
       -6.17120659e-01, -5.87390439e-01, -5.95509133e-01, -5.09502839e-01,
       -5.67165986e-01, -6.17120659e-01,  3.52863171e+00, -8.19508165e-01,
       -6.50685864e-01, -3.64648201e-01, -4.60855992e-01, -7.13741174e-01,
       -5.42898751e-01,  3.48635459e+00, -5.19945763e-01, -7.41751049e-01,
       -5.07161859e-01, -5.13582368e-01, -6.05200161e-01, -6.42026226e-01,
        3.43917101e+00, -5.42995407e-01, -5.99961479e-01, -5.35405367e-01,
       -6.35818585e-01, -5.49128658e-01, -5.41802870e-01, -5.42995407e-01,
        3.44372037e+00, -4.49360188e-01, -7.24614664e-01, -4.92564722e-01,
       -5.98336498e-01,  3.50020792e+00, -6.64189834e-01, -6.32132663e-01,
       -5.58114735e-01, -5.54869469e-01, -6.17145797e-01, -4.73508090e-01,
       -6.64189834e-01,  3.52674391e+00, -6.58379133e-01, -5.22553280e-01,
       -5.90967773e-01, -5.20625897e-01, -6.03842230e-01, -6.32132663e-01,
        3.48834602e+00, -6.06643787e-01, -5.72189780e-01, -5.52911663e-01,
       -6.34508274e-01, -4.55387300e-01, -6.48734310e-01, -6.58379133e-01,
        3.51249640e+00, -6.28809372e-01, -4.86678015e-01, -5.19674755e-01,
       -6.01397083e-01, -6.06643787e-01,  3.48121782e+00, -6.12935102e-01,
       -6.05713581e-01, -5.34853508e-01, -5.09409540e-01, -5.85030135e-01,
       -6.12935102e-01,  3.46301794e+00, -6.32391090e-01, -6.13455975e-01,
       -5.09796098e-01, -3.32969761e-01, -7.08067214e-01,  3.44644779e+00,
       -6.44904626e-01, -5.94838490e-01, -6.53939168e-01, -5.11728532e-01,
       -8.11739218e-01, -3.08875816e-01, -6.97381181e-01, -6.44904626e-01,
        3.53467903e+00, -5.50110335e-01, -5.21667853e-01, -4.33266316e-01,
       -6.68061666e-01, -5.94838490e-01,  3.44564577e+00, -5.32011314e-01,
       -6.62817777e-01, -5.54650205e-01, -6.63611478e-01, -5.72096720e-01,
        3.49404545e+00, -3.96773374e-01, -6.28023329e-01, -6.76061833e-01,
       -5.57478712e-01, -4.93243448e-01, -1.18915137e+00, -3.96773374e-01,
        4.35866892e+00, -1.74743214e-01, -5.08945442e-01, -1.59581207e+00,
       -5.38149289e-01, -6.54302777e-01, -5.19945763e-01,  3.47493089e+00,
       -5.33624509e-01, -6.75288169e-01, -5.53620378e-01, -5.29714782e-01,
       -5.27118294e-01,  3.47710516e+00, -6.69302879e-01, -6.70660095e-01,
       -5.20278613e-01, -5.60030500e-01, -6.03634614e-01, -5.12862787e-01,
       -6.69302879e-01,  3.49699576e+00, -5.60582866e-01, -4.44738214e-01,
       -7.05874400e-01, -6.44252181e-01, -6.38947473e-01, -5.60582866e-01,
        3.54769554e+00, -4.83086522e-01, -5.68982105e-01, -6.51844392e-01,
       -5.81622085e-01, -4.82355974e-01, -6.28809372e-01, -6.70660095e-01,
        3.46815427e+00, -6.16429993e-01, -4.88276753e-01, -6.10900742e-01,
       -6.16932805e-01, -4.83086522e-01,  3.48347995e+00, -5.33354132e-01,
       -6.11534382e-01, -6.27671373e-01, -6.24247872e-01, -5.52264889e-01,
       -5.33354132e-01,  3.48658735e+00, -5.68642928e-01, -5.67637097e-01,
       -6.40440432e-01, -5.62237682e-01, -5.81522796e-01, -5.68320921e-01,
        3.46544204e+00, -5.54849600e-01, -6.11761856e-01, -5.86749185e-01,
       -5.99725828e-01, -5.97259807e-01,  3.47716386e+00, -5.59670386e-01,
       -4.82212030e-01, -6.61653888e-01, -5.76641919e-01, -4.99154358e-01,
       -6.44095014e-01, -5.59670386e-01,  3.47477045e+00, -5.97253806e-01,
       -6.52798456e-01, -5.21798430e-01, -5.74878421e-01, -6.67385889e-01,
       -4.82212030e-01,  3.53169667e+00, -5.11138363e-01, -3.70525621e-01,
       -5.90366862e-02, -8.66519662e-01, -4.99826109e-01, -5.52142890e-01,
       -5.97253806e-01,  3.51557007e+00, -5.78921830e-01, -6.46589733e-01,
       -6.40835705e-01, -5.21774591e-01, -6.97455558e-01, -5.11138363e-01,
        3.57420249e+00, -5.48333362e-01, -8.60466702e-01, -4.35033915e-01,
       -5.07674798e-01, -6.62789809e-01, -5.48333362e-01,  3.52107668e+00,
       -5.85704258e-01, -6.99274354e-01, -5.17300096e-01, -5.75426342e-01,
       -5.51980148e-01, -6.63219442e-01,  3.48300948e+00, -6.21821490e-01,
       -5.17424454e-01, -5.53137601e-01, -5.72790724e-01, -6.49833869e-01,
       -6.21821490e-01,  3.49708526e+00, -5.74492618e-01, -6.66817617e-01,
       -4.11328937e-01, -4.60299457e-01, -7.08033199e-01, -5.74492618e-01,
        3.49090745e+00, -5.89590450e-01, -7.15876963e-01, -4.42614766e-01,
       -3.91526319e-01, -8.35625879e-01, -5.17353895e-02, -5.99961479e-01,
        3.48954288e+00, -4.06455395e-01, -5.74473630e-01, -6.29764787e-01,
       -8.66492636e-01, -4.90512756e-01, -4.06455395e-01,  3.52557905e+00,
       -5.01516303e-01, -6.62534191e-01, -5.98067765e-01, -6.07235426e-01,
       -5.41001800e-01, -5.01516303e-01,  3.52048693e+00, -7.16550540e-01,
       -6.74451556e-01, -4.79731304e-01, -4.15519629e-01, -6.60307025e-01,
       -6.89555398e-01,  3.53102750e+00, -6.89128053e-01, -3.93226933e-01,
       -6.83290460e-01, -6.43693369e-01, -4.17020788e-01, -6.89128053e-01,
        3.52670896e+00, -7.11788207e-01, -4.50366110e-01, -6.14712433e-01,
       -5.94166933e-01, -4.61744348e-01, -7.11788207e-01,  3.54525857e+00,
       -7.76531640e-01, -4.77953194e-01, -5.23074250e-01, -5.90590846e-01,
       -8.02843714e-01, -2.79341138e-01, -5.79969182e-01,  3.60866132e+00,
       -8.14050105e-01, -5.41866331e-01, -6.64432101e-01, -3.82768183e-01,
       -6.51695293e-01, -7.41751049e-01,  3.50900909e+00, -5.12088946e-01,
       -5.56273521e-01, -4.29032147e-01, -5.94211394e-01, -8.19508165e-01,
        3.55718496e+00, -5.75075248e-01, -2.22394457e-01, -9.16963550e-01,
       -6.43430963e-01, -4.37860617e-01,  3.48242414e+00, -6.37694090e-01,
       -6.77346582e-01, -5.43610230e-01, -5.42481657e-01, -5.00053524e-01,
       -5.71731835e-01, -6.59576881e-01, -6.37694090e-01,  3.47695186e+00,
       -5.81617759e-01, -5.26277775e-01, -6.57382492e-01, -5.35314961e-01,
       -6.77346582e-01,  3.56156027e+00, -6.91627683e-01, -5.08316320e-01,
       -4.91572228e-01, -4.31325847e-01, -1.35795128e-01, -6.91627683e-01,
        3.57213247e+00, -9.10788472e-01, -5.70419301e-01, -3.25938312e-01,
       -5.06237723e-01, -1.06456496e+00, -4.49360188e-01, -9.10788472e-01,
        3.79323487e+00, -4.78747353e-01, -8.89773901e-01, -5.16156051e-01,
       -6.42137596e-01, -5.81617759e-01,  3.49106161e+00, -5.94762346e-01,
       -6.54254495e-01, -5.02133363e-01, -5.00644332e-01, -6.32391090e-01,
       -5.94762346e-01,  3.48186522e+00, -6.07430444e-01, -6.37593176e-01,
       -5.09043828e-01, -5.96597103e-01, -5.51323628e-01, -6.65284903e-01,
        3.48422326e+00, -5.40974778e-01, -5.20868199e-01, -6.09174645e-01,
       -5.18849491e-01, -5.32011314e-01, -6.28023329e-01,  3.44166053e+00,
       -6.56415429e-01, -5.06596983e-01, -5.99763988e-01, -3.26260313e-01,
       -1.74743214e-01, -5.40974778e-01,  3.73074944e+00, -9.42716562e-01,
       -3.94495972e-01, -3.78035205e-01, -9.91311369e-01,  1.77879708e-02,
       -5.35399407e-01, -6.68750996e-01, -5.20868199e-01,  3.46718855e+00,
       -5.46926318e-01, -6.51788302e-01, -5.43455329e-01, -5.09243008e-01,
       -6.39581507e-01, -5.46926318e-01,  3.45467950e+00, -5.94796697e-01,
       -6.25429386e-01, -5.38702581e-01, -5.15880187e-01, -5.84918251e-01,
       -5.94796697e-01,  3.46154073e+00, -6.16302218e-01, -5.65151490e-01,
       -5.84491886e-01, -5.40050777e-01, -5.33624509e-01, -6.16302218e-01,
        3.47465353e+00, -6.52502884e-01, -5.28228205e-01, -6.03944937e-01,
       -6.16889445e-01, -5.68642928e-01, -5.54849600e-01,  3.47880901e+00,
       -5.15551209e-01, -5.90272114e-01, -6.32603719e-01, -5.28322062e-01,
       -6.11761856e-01, -5.85704258e-01,  3.47508132e+00, -5.52180357e-01,
       -6.50138157e-01, -5.46974627e-01, -6.04269466e-01, -5.50110335e-01,
       -5.78921830e-01,  3.44105149e+00, -4.35541582e-01, -3.63201372e-01,
       -7.40576662e-01, -1.68430241e-01, -5.64253458e-01, -7.93696949e-01,
       -4.59780073e-01, -3.93226933e-01,  3.55200425e+00, -7.42043726e-01,
       -5.99003106e-01, -5.10921104e-01, -5.44443331e-01, -7.42043726e-01,
        3.50835829e+00, -7.10062994e-01, -4.83986690e-01, -5.16900448e-01,
       -4.63021000e-01, -5.89590450e-01, -7.10062994e-01,  3.50798487e+00,
       -6.72529226e-01, -5.37927034e-01, -5.34854166e-01, -5.31737161e-01,
       -5.17424454e-01, -7.16550540e-01,  3.51566916e+00, -7.41180119e-01,
       -5.32675306e-01, -4.76101583e-01, -3.62853683e-01, -7.76531640e-01,
       -8.14050105e-01,  3.60077996e+00, -5.84090667e-01, -5.41329840e-01,
       -5.21924021e-01, -7.05480786e-01, -5.87390439e-01, -5.41866331e-01,
       -5.84090667e-01,  3.54886359e+00, -5.07388417e-01, -6.22646945e-01,
       -5.65234985e-01, -5.41740303e-01,  3.46215774e+00, -5.92522502e-01,
       -6.34947495e-01, -5.77231929e-01, -5.50480525e-01, -5.81286802e-01,
       -5.47843087e-01, -5.92522502e-01,  3.46630370e+00, -5.73100083e-01,
       -5.47335688e-01, -6.24215541e-01, -6.41203903e-01, -5.43579717e-01,
       -5.73100083e-01,  3.47262376e+00, -5.18432921e-01, -5.20440758e-01,
       -6.75866380e-01, -8.71666260e-01, -9.65589342e-01, -5.18432921e-01,
        3.73845018e+00, -9.01955345e-01, -4.80806313e-01, -5.65727278e-01,
       -5.27796631e-01, -6.34947495e-01,  3.47032285e+00, -6.35670573e-01,
       -5.34404215e-01, -5.71776661e-01, -4.30350242e-01, -5.75075248e-01,
        3.53405881e+00, -7.83613147e-01, -5.91995701e-01, -4.34938613e-01,
       -7.18085856e-01, -5.76847265e-01, -5.44190355e-01, -6.35670573e-01,
        3.47104361e+00, -6.13381142e-01, -5.43277885e-01, -5.57676387e-01,
       -5.65601382e-01, -5.12088946e-01, -6.13381142e-01,  3.44087455e+00,
       -6.44557632e-01, -5.54640199e-01, -5.50605247e-01, -5.95509133e-01,
       -6.50685864e-01, -5.07388417e-01,  3.58949232e+00, -7.06674847e-01,
       -6.28606491e-01, -5.00627566e-01, -3.64648201e-01, -2.22394457e-01,
       -7.06674847e-01,  3.55801821e+00, -9.29416446e-01, -6.25699898e-01,
       -2.65480316e-01, -4.43704041e-01, -5.07161859e-01, -6.75288169e-01,
       -5.56273521e-01, -6.44557632e-01,  3.47655001e+00, -5.28574529e-01,
       -5.64694299e-01, -5.35405367e-01, -7.24614664e-01, -5.74473630e-01,
       -4.78747353e-01,  3.49247201e+00, -5.89896284e-01, -5.89334707e-01,
       -5.58114735e-01, -5.72189780e-01,  3.46679507e+00, -5.97321777e-01,
       -5.90871258e-01, -6.03319786e-01, -5.44977729e-01, -5.54869469e-01,
       -5.22553280e-01, -5.97321777e-01,  3.42250709e+00, -6.02347342e-01,
       -5.62894898e-01, -5.82520324e-01, -5.90967773e-01, -4.86678015e-01,
       -6.16429993e-01, -6.02347342e-01,  3.44032478e+00, -6.23022515e-01,
       -5.20879138e-01, -5.52911663e-01, -6.05713581e-01, -5.90871258e-01,
        3.47724288e+00, -5.64098051e-01, -6.17192477e-01, -5.46455852e-01,
       -5.34853508e-01, -6.13455975e-01, -5.64098051e-01,  3.48830498e+00,
       -5.89390991e-01, -6.51341539e-01, -5.35164921e-01, -5.09796098e-01,
       -6.07430444e-01, -5.89390991e-01,  3.45668276e+00, -6.03540642e-01,
       -6.20565158e-01, -5.25959423e-01, -6.53939168e-01, -5.21667853e-01,
       -4.35541582e-01,  3.32139274e+00, -6.40929585e-01, -6.94504509e-01,
       -3.74810044e-01, -5.11728532e-01, -6.62817777e-01, -6.40929585e-01,
        3.42492516e+00, -4.11196780e-01, -4.16778156e-01, -7.81474333e-01,
       -5.54650205e-01, -6.56415429e-01, -4.11196780e-01,  3.39076133e+00,
       -4.74715734e-01, -7.28574437e-01, -5.65208744e-01, -6.76061833e-01,
       -5.08945442e-01,  3.52004295e+00, -6.20254836e-01, -5.70417788e-01,
       -6.27677266e-01, -5.16685782e-01, -5.57478712e-01, -5.06596983e-01,
       -6.20254836e-01,  3.46363523e+00, -6.79802801e-01, -4.80587993e-01,
       -6.18913907e-01, -1.59581207e+00, -9.42716562e-01, -5.70417788e-01,
        4.12154279e+00, -5.40222375e-01, -4.72373997e-01, -5.53620378e-01,
       -6.52502884e-01, -5.28574529e-01,  3.48081270e+00, -4.99983627e-01,
       -6.23226620e-01, -6.22904659e-01, -5.20278613e-01, -4.44738214e-01,
        3.56038231e+00, -7.55865844e-01, -5.88479203e-01, -8.15925064e-01,
       -4.35095373e-01, -5.60030500e-01, -4.88276753e-01, -6.23022515e-01,
       -7.55865844e-01,  3.49720370e+00, -6.30568350e-01, -4.39439741e-01,
       -7.05874400e-01, -5.68982105e-01, -5.88479203e-01,  3.55692232e+00,
       -4.45781124e-01, -6.46390096e-01, -6.01415390e-01, -6.51844392e-01,
       -6.11534382e-01, -4.45781124e-01,  3.49973657e+00, -5.15320653e-01,
       -6.90533148e-01, -5.84722868e-01, -6.27671373e-01, -5.67637097e-01,
       -5.15320653e-01,  3.49701116e+00, -5.56611024e-01, -5.96836329e-01,
       -6.32934682e-01, -6.40440432e-01, -5.15551209e-01, -5.56611024e-01,
        3.48313255e+00, -6.45203221e-01, -5.42937851e-01, -5.82388817e-01,
       -5.86749185e-01, -5.90272114e-01, -5.52180357e-01,  3.47155897e+00,
       -5.08561930e-01, -6.19776166e-01, -6.14019218e-01, -6.61653888e-01,
       -3.70525621e-01,  3.57345343e+00, -4.87195078e-01, -9.84603786e-01,
       -5.39711116e-01, -5.29763941e-01, -5.76641919e-01, -6.52798456e-01,
       -4.87195078e-01,  3.47024448e+00, -5.21339368e-01, -6.56727561e-01,
       -5.75542099e-01, -5.21798430e-01, -6.46589733e-01, -5.21339368e-01,
        3.47240058e+00, -5.51944483e-01, -6.64708324e-01, -5.66020241e-01,
       -5.90366862e-02, -8.60466702e-01,  3.70864863e+00, -6.45834397e-01,
       -1.14099086e+00, -3.55323059e-01, -6.46996930e-01, -6.40835705e-01,
       -3.63201372e-01, -5.51944483e-01,  3.50258780e+00, -7.48389820e-01,
       -6.39195294e-01, -5.59021128e-01, -4.35033915e-01, -6.99274354e-01,
       -6.45834397e-01,  3.55982639e+00, -5.44628600e-01, -7.43968548e-01,
       -4.91086576e-01, -5.17300096e-01, -6.50138157e-01, -5.44628600e-01,
        3.48950894e+00, -4.90721156e-01, -6.98178294e-01, -5.88542636e-01,
       -5.53137601e-01, -6.66817617e-01, -7.41180119e-01,  3.54020563e+00,
       -5.90727415e-01, -6.29298420e-01, -3.59044459e-01, -4.11328937e-01,
       -7.15876963e-01, -5.90727415e-01,  3.50476058e+00, -6.09567924e-01,
       -7.16084237e-01, -4.61175102e-01, -4.42614766e-01, -6.72529226e-01,
       -6.09567924e-01,  3.46537463e+00, -5.78003064e-01, -6.36239593e-01,
       -5.26420053e-01, -6.29764787e-01, -6.62534191e-01, -5.89896284e-01,
        3.47088471e+00, -4.86675728e-01, -5.31753984e-01, -5.70259735e-01,
       -5.98067765e-01, -6.74451556e-01, -4.86675728e-01,  3.51351217e+00,
       -5.84835881e-01, -7.19839221e-01, -4.49642015e-01, -4.79731304e-01,
       -5.32675306e-01, -5.84835881e-01,  3.52144204e+00, -8.02569847e-01,
       -6.40052300e-01, -4.81577397e-01, -6.83290460e-01, -4.50366110e-01,
       -5.99003106e-01, -4.83986690e-01,  3.51347467e+00, -6.37304702e-01,
       -6.59523601e-01, -6.14712433e-01, -4.77953194e-01, -6.37304702e-01,
        3.49375516e+00, -6.86027717e-01, -5.10915857e-01, -5.66841257e-01,
       -5.23074250e-01, -5.41329840e-01, -6.86027717e-01,  3.48780095e+00,
       -6.35768170e-01, -5.43537381e-01, -5.58063595e-01, -1.99527710e-01,
       -9.01955345e-01, -7.83613147e-01,  3.61583943e+00, -5.67178540e-01,
       -6.85948267e-01, -4.77616421e-01, -9.16963550e-01, -5.91995701e-01,
       -9.29416446e-01,  3.72131508e+00, -4.27037084e-01, -8.55902300e-01,
       -5.43610230e-01, -5.08316320e-01,  3.47675394e+00, -6.58071858e-01,
       -6.84877842e-01, -5.18885661e-01, -5.62992028e-01, -5.42481657e-01,
       -5.26277775e-01, -6.54254495e-01, -6.58071858e-01,  3.47613064e+00,
       -5.77409063e-01, -5.17635792e-01, -4.91572228e-01, -5.70419301e-01,
       -6.84877842e-01,  3.49847404e+00, -6.31514410e-01, -5.79065073e-01,
       -5.41025180e-01, -3.25938312e-01, -8.89773901e-01, -5.89334707e-01,
       -5.31753984e-01,  3.64951727e+00, -7.06312917e-01, -6.06403443e-01,
       -5.02133363e-01, -6.37593176e-01, -5.77409063e-01,  3.47568655e+00,
       -6.01347022e-01, -6.46543953e-01, -5.10659976e-01, -5.09043828e-01,
       -6.03540642e-01, -6.01347022e-01,  3.48124519e+00, -6.18220869e-01,
       -6.23395689e-01, -5.25697145e-01, -6.09174645e-01, -3.94495972e-01,
       -6.51788302e-01,  3.54100444e+00, -5.28863921e-01, -8.40368665e-01,
       -5.16312937e-01, -5.99763988e-01, -4.74715734e-01, -6.79802801e-01,
        3.50689233e+00, -6.34760028e-01, -4.75411889e-01, -6.42437893e-01,
       -3.78035205e-01, -5.40222375e-01,  3.65680822e+00, -7.83434184e-01,
       -9.67551767e-01, -3.84625823e-01, -6.02938865e-01, -5.43455329e-01,
       -6.25429386e-01, -5.28863921e-01,  3.46755075e+00, -5.87318379e-01,
       -6.48705789e-01, -5.33777943e-01, -5.38702581e-01, -5.65151490e-01,
       -5.87318379e-01,  3.45138313e+00, -6.12991806e-01, -5.47494232e-01,
       -5.99724641e-01, -5.84491886e-01, -5.28228205e-01, -6.12991806e-01,
        3.46008405e+00, -5.79886206e-01, -4.56749586e-01, -6.97736361e-01,
       -6.03944937e-01, -4.99983627e-01, -5.79886206e-01,  3.40135778e+00,
       -5.65037120e-01, -4.79514113e-01, -6.72991778e-01, -6.32603719e-01,
       -6.45203221e-01, -5.08561930e-01,  3.49403915e+00, -4.39932836e-01,
       -6.37382649e-01, -6.30354793e-01, -5.46974627e-01, -6.19776166e-01,
       -4.90721156e-01,  3.46588477e+00, -5.12259098e-01, -7.25742996e-01,
       -5.70410729e-01, -8.66519662e-01, -9.84603786e-01, -1.14099086e+00,
        3.89013036e+00, -5.09571991e-01, -3.88444067e-01, -7.40576662e-01,
       -7.48389820e-01,  3.70814190e+00, -1.07434550e+00, -5.74992587e-01,
       -5.69837330e-01, -5.16900448e-01, -5.37927034e-01, -6.59523601e-01,
       -5.10915857e-01,  3.46575414e+00, -6.19916303e-01, -6.20570899e-01,
       -5.34854166e-01, -5.78003064e-01, -6.19916303e-01,  3.48841501e+00,
       -6.05016743e-01, -5.45353514e-01, -6.05271220e-01, -4.76101583e-01,
       -6.29298420e-01, -8.02569847e-01,  3.53559228e+00, -6.68592363e-01,
       -5.62592745e-01, -3.96437324e-01, -5.21924021e-01, -6.22646945e-01,
       -6.28606491e-01, -6.35768170e-01,  3.54735480e+00, -5.68218399e-01,
       -5.70190773e-01, -5.77231929e-01, -5.34404215e-01,  3.45913614e+00,
       -6.07772053e-01, -6.14953684e-01, -5.80918433e-01, -5.43855826e-01,
       -5.50480525e-01, -5.47335688e-01, -6.07772053e-01,  3.45364996e+00,
       -5.93539003e-01, -5.69042686e-01, -5.85480001e-01, -6.24215541e-01,
       -5.20440758e-01, -5.93539003e-01,  3.47122079e+00, -5.77876177e-01,
       -5.28575027e-01, -6.26574281e-01, -6.75866380e-01, -4.80806313e-01,
       -5.67178540e-01, -5.77876177e-01,  3.47618671e+00, -6.62275344e-01,
       -5.12183954e-01, -5.71776661e-01, -5.43277885e-01, -6.14953684e-01,
        3.47133908e+00, -6.13542285e-01, -5.58158380e-01, -5.69630180e-01,
       -4.34938613e-01, -6.85948267e-01,  3.49598557e+00, -6.07698851e-01,
       -5.97411837e-01, -6.63174090e-01, -5.06813911e-01, -7.18085856e-01,
       -4.27037084e-01, -6.07698851e-01,  3.49286736e+00, -6.00892688e-01,
       -4.74782750e-01, -6.64370126e-01, -5.57676387e-01, -5.54640199e-01,
       -6.13542285e-01,  3.44077126e+00, -5.96236774e-01, -5.64817602e-01,
       -5.53858015e-01, -5.50605247e-01, -5.64694299e-01, -6.23226620e-01,
       -5.96236774e-01,  3.47174858e+00, -5.22606577e-01, -6.14379058e-01,
       -5.00627566e-01, -6.25699898e-01, -5.68218399e-01,  3.48730351e+00,
       -5.65243312e-01, -6.45531900e-01, -5.81982436e-01, -2.65480316e-01,
       -8.55902300e-01, -6.00892688e-01,  3.58830039e+00, -7.40561587e-01,
       -4.53777981e-01, -6.71685519e-01, -5.06237723e-01, -6.31514410e-01,
       -7.06312917e-01,  3.62518876e+00, -5.67352112e-01, -5.83219261e-01,
       -6.30552334e-01, -6.03319786e-01, -5.62894898e-01,  3.43948387e+00,
       -5.40370410e-01, -5.71522583e-01, -5.69609364e-01, -5.91766831e-01,
       -5.44977729e-01, -6.17192477e-01, -5.40370410e-01,  3.45715768e+00,
       -5.37033753e-01, -6.54704164e-01, -5.62879151e-01, -5.82520324e-01,
       -5.20879138e-01, -6.30568350e-01, -5.71522583e-01,  3.43614236e+00,
       -5.60772821e-01, -5.69879148e-01, -5.46455852e-01, -6.51341539e-01,
       -5.37033753e-01,  3.46940561e+00, -5.38026369e-01, -6.61869027e-01,
       -5.34679069e-01, -5.35164921e-01, -6.20565158e-01, -5.38026369e-01,
        3.47052534e+00, -5.63160306e-01, -6.77986150e-01, -5.35622439e-01,
       -5.25959423e-01, -6.18220869e-01, -5.63160306e-01,  3.48748558e+00,
       -5.73504431e-01, -6.44652940e-01, -5.61987604e-01, -1.68430241e-01,
       -6.94504509e-01, -1.07434550e+00,  3.55754852e+00, -8.54642284e-01,
       -4.20666523e-01, -3.44959457e-01, -9.91311369e-01, -8.40368665e-01,
        4.01644605e+00, -1.31776876e+00, -5.46104880e-01, -3.20892369e-01,
       -3.74810044e-01, -4.16778156e-01, -8.54642284e-01,  3.54959064e+00,
       -9.15747564e-01, -6.04968287e-01, -3.82644302e-01, -7.81474333e-01,
       -7.28574437e-01, -9.15747564e-01,  3.55548502e+00, -5.02243273e-01,
       -6.27445415e-01, -5.65208744e-01, -6.34760028e-01, -5.02243273e-01,
        3.45449266e+00, -5.34857766e-01, -6.12058199e-01, -6.05364650e-01,
       -6.27677266e-01, -4.80587993e-01,  3.46332345e+00, -6.30597125e-01,
       -5.86963959e-01, -4.40999276e-01, -6.96497835e-01, -5.16685782e-01,
       -4.72373997e-01, -7.83434184e-01, -6.30597125e-01,  3.50823270e+00,
       -4.53528809e-01, -6.51612800e-01, -6.18913907e-01, -4.75411889e-01,
       -5.86963959e-01,  3.42125048e+00, -5.79473726e-01, -4.97910846e-01,
       -6.62576156e-01,  1.77879708e-02, -9.67551767e-01, -1.31776876e+00,
        4.01206208e+00, -6.59708615e-01, -8.52865469e-01, -2.31955436e-01,
       -6.22904659e-01, -5.65037120e-01, -5.22606577e-01,  3.46600831e+00,
       -5.52041437e-01, -5.66007301e-01, -6.37411219e-01, -8.15925064e-01,
       -6.46390096e-01,  3.61037547e+00, -3.93378555e-01, -4.45583633e-01,
       -4.83299434e-01, -8.25798685e-01, -4.35095373e-01, -4.39439741e-01,
       -5.60772821e-01, -3.93378555e-01,  3.45656846e+00, -6.22569751e-01,
       -4.14253318e-01, -5.91058899e-01, -6.01415390e-01, -6.90533148e-01,
       -4.45583633e-01,  3.51587925e+00, -4.88420077e-01, -7.24500089e-01,
       -5.65426915e-01, -5.84722868e-01, -5.96836329e-01, -4.88420077e-01,
        3.49844661e+00, -5.92204894e-01, -6.91247446e-01, -5.45015000e-01,
       -6.32934682e-01, -5.42937851e-01, -5.92204894e-01,  3.60739326e+00,
       -5.51265126e-01, -4.51851868e-01, -8.36198842e-01, -5.82388817e-01,
       -4.39932836e-01, -5.51265126e-01,  3.52558711e+00, -8.38917817e-01,
       -5.78744876e-01, -5.34337635e-01, -6.14019218e-01, -6.37382649e-01,
       -5.12259098e-01,  3.49648469e+00, -4.23706626e-01, -6.12801543e-01,
       -6.96315552e-01, -5.39711116e-01, -5.09571991e-01,  3.41857515e+00,
       -5.56530822e-01, -7.14894525e-01, -4.99927461e-01, -5.97939237e-01,
       -5.29763941e-01, -6.56727561e-01, -5.56530822e-01,  3.45220992e+00,
       -5.06672274e-01, -6.01864392e-01, -6.00650933e-01, -5.75542099e-01,
       -6.64708324e-01, -5.06672274e-01,  3.48015403e+00, -4.92491118e-01,
       -6.35728380e-01, -6.05011838e-01, -5.66020241e-01, -6.39195294e-01,
       -4.92491118e-01,  3.49741313e+00, -5.46391122e-01, -6.79316922e-01,
       -5.73998431e-01, -3.55323059e-01, -7.43968548e-01,  3.52321989e+00,
       -6.94037471e-01, -5.68989892e-01, -4.93178042e-01, -6.67722873e-01,
       -6.46996930e-01, -3.88444067e-01, -7.14894525e-01, -6.94037471e-01,
        3.52622790e+00, -6.02482117e-01, -4.79372784e-01, -5.59021128e-01,
       -5.74992587e-01, -5.46391122e-01,  3.48003630e+00, -6.31585610e-01,
       -6.43809929e-01, -5.24235928e-01, -4.91086576e-01, -6.98178294e-01,
       -5.68989892e-01,  3.50096869e+00, -4.36765633e-01, -6.59953457e-01,
       -6.45994843e-01, -5.88542636e-01, -7.25742996e-01, -4.36765633e-01,
        3.77373366e+00, -1.88193497e-01, -5.61388885e-01, -1.27310001e+00,
       -3.59044459e-01, -7.16084237e-01, -6.68592363e-01,  3.50230453e+00,
       -6.31213368e-01, -6.62232998e-01, -4.65137104e-01, -4.61175102e-01,
       -6.36239593e-01, -6.31213368e-01,  3.47062508e+00, -6.08351179e-01,
       -6.14818607e-01, -5.18827233e-01, -5.26420053e-01, -6.05016743e-01,
       -6.08351179e-01,  3.47046346e+00, -5.60936971e-01, -5.51841499e-01,
       -6.17897010e-01, -5.70259735e-01, -7.19839221e-01, -6.06403443e-01,
       -5.67352112e-01,  3.52146435e+00, -5.79236193e-01, -4.78373646e-01,
       -4.49642015e-01, -6.40052300e-01, -5.79236193e-01,  3.49392704e+00,
       -6.66647565e-01, -6.44704473e-01, -5.13644495e-01, -4.81577397e-01,
       -5.62592745e-01, -6.66647565e-01,  3.48833017e+00, -6.99001485e-01,
       -5.71184430e-01, -5.07326551e-01, -5.66841257e-01, -5.43537381e-01,
       -6.20570899e-01, -5.45353514e-01,  3.47096359e+00, -6.15526576e-01,
       -5.79133967e-01, -5.58063595e-01, -5.70190773e-01, -5.65243312e-01,
       -6.15526576e-01,  3.49760070e+00, -5.64970972e-01, -6.23605469e-01,
       -4.77616421e-01, -6.62275344e-01, -5.97411837e-01,  3.49126494e+00,
       -5.95811302e-01, -6.22392999e-01, -5.35757034e-01, -4.43704041e-01,
       -6.45531900e-01, -7.40561587e-01,  3.52338095e+00, -4.82852847e-01,
       -4.61913325e-01, -7.48817245e-01, -5.18885661e-01, -5.79065073e-01,
        3.46876481e+00, -6.36261003e-01, -6.03895583e-01, -5.81546360e-01,
       -5.49111126e-01, -5.62992028e-01, -5.17635792e-01, -6.46543953e-01,
       -6.36261003e-01,  3.47077612e+00, -5.82775954e-01, -5.24567387e-01,
       -5.41025180e-01, -5.83219261e-01, -6.03895583e-01,  3.48482777e+00,
       -5.42697303e-01, -6.04056766e-01, -6.09933672e-01, -5.10659976e-01,
       -6.23395689e-01, -5.82775954e-01,  3.46818398e+00, -6.01264757e-01,
       -6.26861420e-01, -5.23226180e-01, -5.25697145e-01, -5.73504431e-01,
       -6.01264757e-01,  3.45958067e+00, -6.03871582e-01, -6.07234877e-01,
       -5.48007873e-01, -5.16312937e-01, -6.48705789e-01, -5.46104880e-01,
        3.47249761e+00, -5.95872241e-01, -6.21419343e-01, -5.44082423e-01,
       -6.42437893e-01, -5.34857766e-01, -5.79473726e-01,  3.50721770e+00,
       -5.78421312e-01, -5.06195240e-01, -6.65831766e-01, -3.84625823e-01,
       -6.59708615e-01,  3.49626714e+00, -6.74461456e-01, -7.70260317e-01,
       -3.94503209e-01, -6.12707722e-01, -6.02938865e-01, -4.53528809e-01,
       -6.74461456e-01,  3.47680776e+00, -6.44707792e-01, -4.80095795e-01,
       -6.21075042e-01, -5.33777943e-01, -5.47494232e-01, -5.95872241e-01,
        3.45671107e+00, -6.82356101e-01, -5.66541412e-01, -5.30669139e-01,
       -5.99724641e-01, -4.56749586e-01, -6.82356101e-01,  3.50460102e+00,
       -6.30050298e-01, -3.68383388e-01, -7.67337000e-01, -6.97736361e-01,
       -4.79514113e-01, -6.30050298e-01,  3.47416163e+00, -5.44243368e-01,
       -3.72437443e-01, -7.50180044e-01, -6.72991778e-01, -5.52041437e-01,
       -5.44243368e-01,  3.46500748e+00, -5.19651083e-01, -5.17666944e-01,
       -6.58412875e-01, -6.30354793e-01, -8.38917817e-01, -4.23706626e-01,
        3.62542178e+00, -2.74858257e-01, -5.99790638e-01, -8.57793648e-01,
       -5.70410729e-01, -6.12801543e-01, -1.88193497e-01,  3.46437207e+00,
       -4.91916551e-01, -7.47612628e-01, -6.23857220e-01, -2.29579905e-01,
       -5.69837330e-01, -4.20666523e-01, -6.31585610e-01,  3.45407179e+00,
       -8.00699997e-01, -5.24516939e-01, -5.06765389e-01, -6.05271220e-01,
       -5.60936971e-01, -5.79133967e-01, -5.64970972e-01,  3.49672493e+00,
       -2.36623008e-01, -2.48107408e-01, -7.01681382e-01, -3.96437324e-01,
       -6.62232998e-01, -6.99001485e-01,  3.50049577e+00, -6.54199471e-01,
       -5.90402678e-01, -4.98221812e-01, -5.80918433e-01, -5.58158380e-01,
        3.45844219e+00, -6.12342545e-01, -6.02283382e-01, -5.63386340e-01,
       -5.41353115e-01, -5.43855826e-01, -5.69042686e-01, -6.12342545e-01,
        3.43891266e+00, -6.04240666e-01, -5.67891890e-01, -5.41539047e-01,
       -5.85480001e-01, -5.28575027e-01, -6.04240666e-01,  3.45088804e+00,
       -5.81758949e-01, -5.90897256e-01, -5.59936140e-01, -6.26574281e-01,
       -5.12183954e-01, -5.95811302e-01, -5.81758949e-01,  3.47733234e+00,
       -5.67244712e-01, -5.93759146e-01, -5.69630180e-01, -5.64817602e-01,
       -6.02283382e-01,  3.42927962e+00, -5.88142252e-01, -5.47165603e-01,
       -5.57240604e-01, -6.63174090e-01, -4.74782750e-01,  3.47377165e+00,
       -5.59852944e-01, -5.82452921e-01, -6.02232996e-01, -5.91275953e-01,
       -5.06813911e-01, -6.22392999e-01, -5.59852944e-01,  3.44597793e+00,
       -5.49237054e-01, -5.74469741e-01, -6.33211284e-01, -6.64370126e-01,
       -4.53777981e-01, -5.82452921e-01,  3.47270841e+00, -5.96505898e-01,
       -5.45141299e-01, -6.30460179e-01, -5.53858015e-01, -6.14379058e-01,
       -5.66007301e-01, -5.88142252e-01,  3.47844819e+00, -5.55425313e-01,
       -6.00636247e-01, -5.81982436e-01, -6.23605469e-01, -4.82852847e-01,
       -2.36623008e-01,  3.82216569e+00, -5.21796647e-01, -1.37530528e+00,
       -6.71685519e-01, -4.61913325e-01, -5.96505898e-01,  3.47372193e+00,
       -5.68709845e-01, -5.00090684e-01, -6.74816654e-01, -6.30552334e-01,
       -4.78373646e-01, -6.44704473e-01, -5.42697303e-01,  3.49661287e+00,
       -6.30103444e-01, -5.70181672e-01, -5.69609364e-01, -6.54704164e-01,
        3.44965679e+00, -5.47435891e-01, -5.04514130e-01, -5.64864692e-01,
       -6.08528547e-01, -5.91766831e-01, -5.69879148e-01, -6.22569751e-01,
       -5.47435891e-01,  3.44884808e+00, -5.08958667e-01, -6.08237791e-01,
       -5.62879151e-01, -6.61869027e-01, -5.04514130e-01,  3.45244068e+00,
       -5.25291209e-01, -6.51534818e-01, -5.46352339e-01, -5.34679069e-01,
       -6.77986150e-01, -5.25291209e-01,  3.48029090e+00, -5.22251780e-01,
       -6.73883900e-01, -5.46198795e-01, -5.35622439e-01, -6.44652940e-01,
       -5.22251780e-01,  3.45881533e+00, -5.38254999e-01, -6.63521733e-01,
       -5.54511442e-01, -5.61987604e-01, -6.03871582e-01, -5.38254999e-01,
        3.47826342e+00, -5.51964653e-01, -6.26527737e-01, -5.95656845e-01,
       -3.44959457e-01, -6.04968287e-01, -8.00699997e-01,  3.47723912e+00,
       -7.41846418e-01, -5.16390256e-01, -4.68374707e-01, -3.20892369e-01,
       -8.52865469e-01, -6.21419343e-01,  3.65696691e+00, -8.17767043e-01,
       -6.31685076e-01, -4.12337606e-01, -3.82644302e-01, -6.27445415e-01,
       -6.12058199e-01, -7.41846418e-01,  3.45133287e+00, -5.01095447e-01,
       -5.86243086e-01, -6.05364650e-01, -5.78421312e-01, -5.01095447e-01,
        3.50188410e+00, -5.50044127e-01, -6.37704053e-01, -6.29254509e-01,
       -4.40999276e-01, -6.51612800e-01, -6.44707792e-01,  3.45386652e+00,
       -5.68272086e-01, -4.26710573e-01, -7.21563992e-01, -6.96497835e-01,
       -4.97910846e-01, -5.68272086e-01,  3.44310674e+00, -5.12688240e-01,
       -3.84834034e-01, -7.82903699e-01, -6.62576156e-01, -5.06195240e-01,
       -5.12688240e-01,  3.38045968e+00, -5.15576323e-01, -5.29564126e-01,
       -6.53859592e-01, -2.31955436e-01, -7.70260317e-01, -8.17767043e-01,
        3.61433144e+00, -7.21205603e-01, -7.30773975e-01, -3.42369063e-01,
       -6.37411219e-01, -5.19651083e-01, -5.55425313e-01,  3.45262482e+00,
       -5.74357093e-01, -5.55947589e-01, -6.09832524e-01, -4.83299434e-01,
       -7.24500089e-01,  3.51541763e+00, -6.01366550e-01, -5.24855805e-01,
       -6.65197185e-01, -5.16198568e-01, -8.25798685e-01, -4.14253318e-01,
       -6.01366550e-01,  3.67668481e+00, -7.15910394e-01, -4.87251695e-01,
       -6.32104168e-01, -5.65426915e-01, -6.91247446e-01, -5.24855805e-01,
        3.52727787e+00, -5.43185701e-01, -6.46299956e-01, -5.56262047e-01,
       -5.45015000e-01, -4.51851868e-01, -5.43185701e-01,  3.59842693e+00,
       -4.58321501e-01, -6.26273632e-01, -5.73108954e-01, -4.00670271e-01,
       -8.36198842e-01, -5.78744876e-01, -4.58321501e-01,  3.71799348e+00,
       -6.24165347e-01, -6.17512608e-01, -6.03050309e-01, -5.34337635e-01,
       -2.74858257e-01, -6.24165347e-01,  3.49257271e+00, -5.63847756e-01,
       -4.34383372e-01, -6.81987225e-01, -3.78993120e-01, -6.96315552e-01,
       -5.99790638e-01, -4.91916551e-01,  3.53016415e+00, -4.48215131e-01,
       -5.46095616e-01, -7.47830664e-01, -4.99927461e-01, -6.02482117e-01,
        3.48491080e+00, -5.88701507e-01, -6.01853148e-01, -5.15823733e-01,
       -6.76122837e-01, -5.97939237e-01, -6.01864392e-01, -5.88701507e-01,
        3.52635905e+00, -4.99116687e-01, -5.48212091e-01, -6.90525131e-01,
       -6.00650933e-01, -6.35728380e-01, -4.99116687e-01,  3.46051129e+00,
       -4.75376579e-01, -6.08433033e-01, -6.41205676e-01, -6.05011838e-01,
       -6.79316922e-01, -4.75376579e-01,  3.49004699e+00, -4.63552820e-01,
       -6.60301370e-01, -6.06487461e-01, -5.73998431e-01, -6.43809929e-01,
       -4.63552820e-01,  3.49208660e+00, -5.40974427e-01, -6.90267136e-01,
       -5.79483858e-01, -4.93178042e-01, -6.59953457e-01,  3.45399441e+00,
       -5.63023471e-01, -4.78461257e-01, -6.16386414e-01, -6.42991765e-01,
       -6.67722873e-01, -4.79372784e-01, -6.01853148e-01, -5.63023471e-01,
        3.45035720e+00, -5.96189973e-01, -5.42194946e-01, -5.24235928e-01,
       -5.24516939e-01, -5.40974427e-01,  3.50035232e+00, -7.49454187e-01,
       -6.19304943e-01, -5.41865898e-01, -6.45994843e-01, -5.61388885e-01,
       -4.78461257e-01,  3.46847015e+00, -4.10413589e-01, -5.46358201e-01,
       -8.25853380e-01, -1.27310001e+00, -7.47612628e-01, -4.10413589e-01,
        4.02989428e+00, -1.28263713e+00, -3.16130919e-01, -4.65137104e-01,
       -6.14818607e-01, -6.54199471e-01,  3.48450485e+00, -6.54189043e-01,
       -5.84252633e-01, -5.11907990e-01, -5.18827233e-01, -5.51841499e-01,
       -6.54189043e-01,  3.49720625e+00, -6.57077480e-01, -5.38503185e-01,
       -5.76767814e-01, -6.17897010e-01, -2.48107408e-01, -6.57077480e-01,
        3.85864176e+00, -1.39381977e+00, -3.84387273e-01, -5.57352825e-01,
       -5.13644495e-01, -5.71184430e-01, -6.30103444e-01,  3.48057940e+00,
       -6.35031628e-01, -5.56591300e-01, -5.74024103e-01, -5.07326551e-01,
       -5.90402678e-01, -6.35031628e-01,  3.47792181e+00, -6.27912464e-01,
       -5.53720540e-01, -5.63527952e-01, -5.35757034e-01, -5.67244712e-01,
       -5.49237054e-01,  3.36038248e+00, -5.42083628e-01, -5.48054966e-01,
       -6.18005088e-01, -7.48817245e-01, -5.21796647e-01, -5.68709845e-01,
        3.52379656e+00, -3.74172040e-01, -4.60240037e-01, -8.50060750e-01,
       -5.81546360e-01, -5.24567387e-01, -6.26861420e-01,  3.47431445e+00,
       -6.01401155e-01, -5.90890616e-01, -5.49047516e-01, -5.49111126e-01,
       -6.04056766e-01, -6.01401155e-01,  3.46953358e+00, -5.32597061e-01,
       -6.05760794e-01, -5.76606681e-01, -6.09933672e-01, -5.70181672e-01,
       -5.56591300e-01, -5.32597061e-01,  3.48573366e+00, -6.06114949e-01,
       -6.10315011e-01, -5.23226180e-01, -6.07234877e-01, -5.90890616e-01,
        3.47105690e+00, -5.87752231e-01, -6.00198971e-01, -5.61754023e-01,
       -5.48007873e-01, -5.51964653e-01, -5.87752231e-01,  3.43270384e+00,
       -5.88830983e-01, -5.87140044e-01, -5.69008051e-01, -5.44082423e-01,
       -5.66541412e-01, -6.31685076e-01,  3.48785431e+00, -6.77516754e-01,
       -5.63713666e-01, -5.04314976e-01, -6.65831766e-01, -5.50044127e-01,
       -5.15576323e-01,  3.44663498e+00, -5.08542284e-01, -5.81576605e-01,
       -6.25063876e-01, -3.94503209e-01, -7.21205603e-01,  3.55408285e+00,
       -6.33939095e-01, -7.04377484e-01, -6.66620287e-01, -4.33437169e-01,
       -6.12707722e-01, -4.80095795e-01, -6.33939095e-01,  3.46693268e+00,
       -6.48884134e-01, -4.97072762e-01, -5.94233167e-01, -6.21075042e-01,
       -4.26710573e-01, -6.48884134e-01,  3.46547830e+00, -6.70281944e-01,
       -4.75170832e-01, -6.23355777e-01, -5.30669139e-01, -3.68383388e-01,
       -6.77516754e-01,  3.51733058e+00, -8.78093402e-01, -5.97243939e-01,
       -8.55189849e-02, -3.79904972e-01, -7.67337000e-01, -3.72437443e-01,
       -8.78093402e-01,  3.86541445e+00, -5.60538678e-01, -1.28700793e+00,
       -7.50180044e-01, -5.17666944e-01, -5.60538678e-01,  3.48849184e+00,
       -5.31275512e-01, -4.54722009e-01, -6.74108657e-01, -6.58412875e-01,
       -5.74357093e-01, -5.31275512e-01,  3.46692478e+00, -5.36042200e-01,
       -5.37679821e-01, -6.29157285e-01, -5.91058899e-01, -5.08958667e-01,
       -7.15910394e-01,  3.55493940e+00, -6.02993987e-01, -5.17789613e-01,
       -6.18227839e-01, -8.57793648e-01, -5.63847756e-01, -4.48215131e-01,
        3.62776409e+00, -5.33876365e-01, -4.58689596e-01, -7.65341590e-01,
       -6.23857220e-01, -5.46095616e-01,  3.54559307e+00, -5.81363763e-01,
       -5.35195907e-01, -6.95000687e-01, -5.64079882e-01, -5.06765389e-01,
       -5.16390256e-01, -7.49454187e-01,  3.53783970e+00, -7.59579089e-01,
       -4.92945405e-01, -5.12705379e-01, -2.29579905e-01, -1.28263713e+00,
       -5.81363763e-01,  3.74228349e+00, -5.75803270e-01, -6.24480111e-01,
       -4.48419316e-01, -7.01681382e-01, -1.37530528e+00, -1.39381977e+00,
       -3.74172040e-01,  4.14338343e+00, -2.98404959e-01, -4.98221812e-01,
       -5.84252633e-01, -6.27912464e-01,  3.47705003e+00, -6.32293931e-01,
       -5.67460598e-01, -5.66908596e-01, -5.63386340e-01, -5.47165603e-01,
        3.48329493e+00, -6.34719885e-01, -6.19895266e-01, -5.68986453e-01,
       -5.49141382e-01, -5.41353115e-01, -5.67891890e-01, -6.34719885e-01,
        3.47479720e+00, -6.16728597e-01, -5.57788273e-01, -5.56315444e-01,
       -5.41539047e-01, -5.90897256e-01, -6.16728597e-01,  3.47027942e+00,
       -5.78160778e-01, -5.87868352e-01, -5.55085394e-01, -5.59936140e-01,
       -5.93759146e-01, -5.42083628e-01, -5.78160778e-01,  3.47519981e+00,
       -5.73654458e-01, -6.27605661e-01, -5.57240604e-01, -6.00636247e-01,
       -5.55947589e-01, -6.19895266e-01,  3.47416984e+00, -5.87943576e-01,
       -5.52506555e-01, -6.02232996e-01, -5.45141299e-01,  3.45598549e+00,
       -5.61928944e-01, -5.77180633e-01, -6.13165264e-01, -5.56336358e-01,
       -5.91275953e-01, -5.74469741e-01, -5.61928944e-01,  3.44962999e+00,
       -5.62893830e-01, -5.21251076e-01, -6.37810443e-01, -6.33211284e-01,
       -5.48054966e-01, -5.62893830e-01,  3.47534265e+00, -5.58776571e-01,
       -5.06503234e-01, -6.65902765e-01, -6.30460179e-01, -5.00090684e-01,
       -5.77180633e-01,  3.45364430e+00, -5.54416969e-01, -5.77703982e-01,
       -6.13791852e-01, -6.74816654e-01, -4.60240037e-01, -5.54416969e-01,
        3.45711455e+00, -5.26108618e-01, -5.56893932e-01, -6.84638343e-01,
       -5.64864692e-01, -6.51534818e-01,  3.44971630e+00, -5.43361010e-01,
       -5.39580413e-01, -5.54274291e-01, -5.96101078e-01, -6.08528547e-01,
       -6.08237791e-01, -6.02993987e-01, -5.43361010e-01,  3.46401999e+00,
       -5.01388313e-01, -5.99510340e-01, -5.46352339e-01, -6.73883900e-01,
       -5.39580413e-01,  3.48001155e+00, -5.25278998e-01, -6.34289349e-01,
       -5.60626547e-01, -5.46198795e-01, -6.63521733e-01, -5.25278998e-01,
        3.45148192e+00, -5.27658437e-01, -6.36849433e-01, -5.51974528e-01,
       -5.54511442e-01, -6.26527737e-01, -5.27658437e-01,  3.46028376e+00,
       -5.35563417e-01, -6.49009516e-01, -5.67013208e-01, -5.95656845e-01,
       -5.88830983e-01, -5.35563417e-01,  3.48195954e+00, -5.47045445e-01,
       -6.07081796e-01, -6.07781055e-01, -4.68374707e-01, -5.86243086e-01,
       -6.37704053e-01, -7.59579089e-01,  3.51879795e+00, -5.34273416e-01,
       -5.32623599e-01, -4.12337606e-01, -7.30773975e-01, -5.63713666e-01,
        3.52034181e+00, -6.89943880e-01, -6.76860679e-01, -4.46712009e-01,
       -6.29254509e-01, -5.08542284e-01, -5.34273416e-01,  3.48633846e+00,
       -5.93773409e-01, -6.01109273e-01, -6.19385573e-01, -7.21563992e-01,
       -3.84834034e-01, -6.70281944e-01,  3.44197001e+00, -6.03929347e-01,
       -2.91069548e-01, -7.70291147e-01, -7.82903699e-01, -5.29564126e-01,
       -6.03929347e-01,  3.49067455e+00, -4.70840412e-01, -3.28784685e-01,
       -7.74652276e-01, -6.53859592e-01, -5.81576605e-01, -4.70840412e-01,
        3.37301836e+00, -4.92453913e-01, -5.87916011e-01, -5.86371822e-01,
       -3.42369063e-01, -7.04377484e-01, -6.89943880e-01,  3.52718792e+00,
       -6.48596902e-01, -7.04246682e-01, -4.37653907e-01, -6.09832524e-01,
       -5.36042200e-01, -5.87943576e-01,  3.42465354e+00, -5.56103818e-01,
       -5.65951135e-01, -5.68780288e-01, -6.65197185e-01, -4.87251695e-01,
        3.48535830e+00, -5.75563662e-01, -6.08498472e-01, -6.10445278e-01,
       -5.38402003e-01, -5.16198568e-01, -6.46299956e-01, -5.75563662e-01,
        3.47760307e+00, -5.69467017e-01, -6.53505211e-01, -5.16568660e-01,
       -6.32104168e-01, -5.17789613e-01, -6.08498472e-01,  3.51738923e+00,
       -6.32448495e-01, -5.43633161e-01, -5.82915320e-01, -5.56262047e-01,
       -6.26273632e-01, -5.69467017e-01,  3.57601508e+00, -7.08329171e-01,
       -7.15631215e-01, -4.00052003e-01, -5.73108954e-01, -6.17512608e-01,
        3.74162534e+00, -6.45794765e-01, -8.55959497e-01, -5.57288856e-01,
       -4.91960665e-01, -6.03050309e-01, -4.34383372e-01, -6.45794765e-01,
        3.58060404e+00, -8.47635393e-01, -5.30395022e-01, -5.19345179e-01,
       -6.81987225e-01, -5.33876365e-01,  3.52454653e+00, -6.11098067e-01,
       -5.06096543e-01, -4.22648376e-01, -7.68839951e-01, -7.47830664e-01,
       -4.58689596e-01, -5.35195907e-01,  3.54787668e+00, -5.88869181e-01,
       -5.75715910e-01, -6.41575419e-01, -5.15823733e-01, -5.96189973e-01,
        3.39460924e+00, -5.43775810e-01, -5.70674142e-01, -6.06298651e-01,
       -5.61846933e-01, -6.76122837e-01, -5.48212091e-01, -5.43775810e-01,
        3.52350902e+00, -5.07925207e-01, -5.83151404e-01, -6.64321665e-01,
       -6.90525131e-01, -6.08433033e-01, -5.07925207e-01,  3.50058186e+00,
       -4.20653182e-01, -5.37157149e-01, -7.35888156e-01, -6.41205676e-01,
       -6.60301370e-01, -4.20653182e-01,  3.48420117e+00, -4.12189845e-01,
       -6.37171770e-01, -7.12679322e-01, -6.06487461e-01, -6.90267136e-01,
       -4.12189845e-01,  3.47013370e+00, -4.54857375e-01, -7.13570755e-01,
       -5.92761131e-01, -5.79483858e-01, -6.19304943e-01, -4.54857375e-01,
        3.45423143e+00, -5.29775256e-01, -6.72170541e-01, -5.98639454e-01,
       -6.16386414e-01, -5.42194946e-01, -5.70674142e-01,  3.46745141e+00,
       -5.12420810e-01, -5.39532166e-01, -6.86242934e-01, -6.42991765e-01,
       -5.46358201e-01, -5.12420810e-01,  3.36194513e+00, -4.70185410e-01,
       -4.69772899e-01, -7.20216046e-01, -5.41865898e-01, -4.92945405e-01,
       -5.29775256e-01,  3.36747187e+00, -6.68973974e-01, -5.88191700e-01,
       -5.45719634e-01, -8.25853380e-01, -3.16130919e-01, -5.75803270e-01,
       -4.70185410e-01,  3.35750825e+00, -7.35854633e-01, -4.33680640e-01,
       -5.11907990e-01, -5.38503185e-01, -6.32293931e-01,  3.47911950e+00,
       -6.82273418e-01, -5.59042614e-01, -5.55098362e-01, -5.76767814e-01,
       -3.84387273e-01, -6.82273418e-01,  3.51286036e+00, -8.18437655e-01,
       -5.42927633e-01, -5.08066562e-01, -5.57352825e-01, -8.50060750e-01,
       -2.98404959e-01, -5.26108618e-01, -8.18437655e-01,  3.53493721e+00,
       -4.84572405e-01, -5.74024103e-01, -5.53720540e-01, -6.06114949e-01,
        3.49061938e+00, -6.19379150e-01, -5.18506165e-01, -6.18874472e-01,
       -5.63527952e-01, -5.67460598e-01, -6.19379150e-01,  3.47699806e+00,
       -5.08149165e-01, -4.77667621e-01, -7.40813572e-01, -6.18005088e-01,
       -5.73654458e-01, -5.58776571e-01,  3.46124523e+00, -5.31403609e-01,
       -5.22099143e-01, -6.57306363e-01, -5.49047516e-01, -6.05760794e-01,
       -6.00198971e-01,  3.48582291e+00, -5.22748018e-01, -5.64301061e-01,
       -6.43766551e-01, -5.76606681e-01, -6.10315011e-01, -5.18506165e-01,
       -5.22748018e-01,  3.47451338e+00, -6.61146265e-01, -5.85191241e-01,
       -5.61754023e-01, -5.87140044e-01, -5.64301061e-01,  3.45601981e+00,
       -5.87514254e-01, -5.37984674e-01, -6.17325750e-01, -5.69008051e-01,
       -5.47045445e-01, -5.87514254e-01,  3.42700416e+00, -5.56709476e-01,
       -5.80014669e-01, -5.86712270e-01, -5.04314976e-01, -5.97243939e-01,
       -6.76860679e-01,  3.50354772e+00, -5.96693314e-01, -7.06263601e-01,
       -4.22171216e-01, -6.25063876e-01, -5.93773409e-01, -4.92453913e-01,
        3.49148161e+00, -5.20268904e-01, -6.02667696e-01, -6.57253815e-01,
       -6.66620287e-01, -4.97072762e-01,  3.59639810e+00, -6.25553557e-01,
       -6.00697232e-01, -6.72056087e-01, -5.34398178e-01, -4.33437169e-01,
       -6.48596902e-01, -6.25553557e-01,  3.48242673e+00, -6.41443514e-01,
       -6.03943984e-01, -5.29451603e-01, -5.94233167e-01, -4.75170832e-01,
       -6.00697232e-01,  3.44925587e+00, -6.60543409e-01, -5.29143426e-01,
       -5.89467801e-01, -6.23355777e-01, -2.91069548e-01, -6.60543409e-01,
        3.48019359e+00, -5.14562383e-01, -5.62466601e-01, -7.11004035e-01,
       -1.17191837e-01, -8.55189849e-02, -1.28700793e+00, -4.54722009e-01,
        3.80053688e+00, -5.23099738e-01, -8.18951026e-01, -6.31237190e-01,
       -6.74108657e-01, -5.37679821e-01, -5.23099738e-01,  3.46652504e+00,
       -5.78374968e-01, -5.46513098e-01, -6.06748756e-01, -6.29157285e-01,
       -5.56103818e-01, -5.78374968e-01,  3.47137397e+00, -5.92115971e-01,
       -5.33257354e-01, -5.82364575e-01, -6.18227839e-01, -5.01388313e-01,
       -6.32448495e-01,  3.48397883e+00, -5.93685867e-01, -5.44514726e-01,
       -5.93713592e-01, -4.00670271e-01, -7.08329171e-01, -8.55959497e-01,
        3.69570858e+00, -6.97348306e-01, -3.78310340e-01, -6.55090994e-01,
       -3.78993120e-01, -8.47635393e-01, -6.11098067e-01,  3.60355556e+00,
       -5.10320512e-01, -8.52943273e-01, -4.02565193e-01, -7.65341590e-01,
       -5.06096543e-01, -5.88869181e-01,  3.64433340e+00, -4.93999646e-01,
       -4.98525007e-01, -7.91501435e-01, -6.95000687e-01, -5.75715910e-01,
        3.58369159e+00, -4.73126883e-01, -4.87415348e-01, -6.30621742e-01,
       -7.21811023e-01, -5.64079882e-01, -6.24480111e-01, -4.73126883e-01,
        3.41264930e+00, -6.14852598e-01, -5.65103003e-01, -5.71006819e-01,
       -5.12705379e-01, -5.32623599e-01, -6.01109273e-01, -6.68973974e-01,
        3.37256423e+00, -4.99317208e-01, -5.57834802e-01, -4.48419316e-01,
       -7.35854633e-01, -6.14852598e-01,  3.50187849e+00, -5.83155676e-01,
       -5.98370136e-01, -5.21226127e-01, -5.66908596e-01, -5.59042614e-01,
       -5.08149165e-01,  3.45455077e+00, -5.53491390e-01, -6.48178421e-01,
       -6.18780585e-01, -5.68986453e-01, -5.52506555e-01, -5.65951135e-01,
        3.41301849e+00, -5.66234617e-01, -5.94783298e-01, -5.64556433e-01,
       -5.49141382e-01, -5.57788273e-01, -5.66234617e-01,  3.43717784e+00,
       -5.61808919e-01, -6.31056610e-01, -5.71148037e-01, -5.56315444e-01,
       -5.87868352e-01, -5.61808919e-01,  3.46304052e+00, -5.52452727e-01,
       -6.29129564e-01, -5.75465510e-01, -5.55085394e-01, -6.27605661e-01,
       -5.31403609e-01, -5.52452727e-01,  3.46003548e+00, -5.52880190e-01,
       -6.40607903e-01, -6.13165264e-01, -5.21251076e-01,  3.45883605e+00,
       -6.06556938e-01, -6.25473950e-01, -4.92571441e-01, -5.99817386e-01,
       -5.56336358e-01, -5.77703982e-01, -6.06556938e-01,  3.44441793e+00,
       -5.66962212e-01, -5.76746624e-01, -5.60111812e-01, -6.37810443e-01,
       -5.06503234e-01, -6.25473950e-01,  3.48130309e+00, -5.86383930e-01,
       -4.43751351e-01, -6.81380182e-01, -6.65902765e-01, -5.22099143e-01,
       -5.86383930e-01,  3.48717071e+00, -5.54272240e-01, -4.46873077e-01,
       -7.11639551e-01, -6.13791852e-01, -5.56893932e-01, -5.66962212e-01,
        3.45303967e+00, -5.48709341e-01, -5.68709259e-01, -5.97973073e-01,
       -6.84638343e-01, -5.42927633e-01, -4.84572405e-01, -5.48709341e-01,
        3.45739985e+00, -6.51590271e-01, -5.44961859e-01, -5.54274291e-01,
       -6.34289349e-01,  3.45647714e+00, -5.73799411e-01, -5.48896304e-01,
       -5.78125371e-01, -5.67092414e-01, -5.96101078e-01, -5.99510340e-01,
       -5.93685867e-01, -5.73799411e-01,  3.48804121e+00, -5.19189955e-01,
       -6.05754554e-01, -5.60626547e-01, -6.36849433e-01, -5.48896304e-01,
        3.45485905e+00, -5.35130061e-01, -5.92371575e-01, -5.80985134e-01,
       -5.51974528e-01, -6.49009516e-01, -5.35130061e-01,  3.46440185e+00,
       -5.36373279e-01, -6.23749882e-01, -5.68164581e-01, -5.67013208e-01,
       -6.07081796e-01, -5.36373279e-01,  3.46300528e+00, -5.48414362e-01,
       -6.35498503e-01, -5.68624136e-01, -6.07781055e-01, -5.56709476e-01,
       -5.48414362e-01,  3.47900002e+00, -5.73367957e-01, -5.86746427e-01,
       -6.05980747e-01, -4.46712009e-01, -7.04246682e-01, -5.96693314e-01,
        3.46286491e+00, -5.95362969e-01, -7.23691307e-01, -3.96158627e-01,
       -6.19385573e-01, -5.20268904e-01, -4.99317208e-01,  3.44896235e+00,
       -6.04140290e-01, -5.71166195e-01, -6.34684181e-01, -7.70291147e-01,
       -3.28784685e-01, -5.14562383e-01,  3.88563886e+00, -8.47993690e-01,
       -1.42400695e+00, -7.74652276e-01, -5.87916011e-01, -8.47993690e-01,
        3.54249391e+00, -4.99464659e-01, -1.11236110e-01, -7.21231162e-01,
       -5.86371822e-01, -6.02667696e-01, -4.99464659e-01,  3.40194914e+00,
       -5.15885966e-01, -5.76284392e-01, -6.21274607e-01, -4.37653907e-01,
       -6.41443514e-01, -5.95362969e-01,  3.47484542e+00, -5.99354149e-01,
       -6.67890831e-01, -5.33140048e-01, -3.79904972e-01, -7.06263601e-01,
       -8.18951026e-01,  3.59663103e+00, -6.04500526e-01, -5.63011861e-01,
       -5.23999048e-01, -5.68780288e-01, -5.92115971e-01, -5.94783298e-01,
        3.46852643e+00, -5.71487958e-01, -5.61854229e-01, -5.79504690e-01,
       -6.10445278e-01, -5.43633161e-01,  3.47102723e+00, -5.58022588e-01,
       -5.96920254e-01, -5.80237530e-01, -5.81768423e-01, -5.38402003e-01,
       -6.53505211e-01, -5.58022588e-01,  3.47880903e+00, -5.55225702e-01,
       -6.32341284e-01, -5.41312237e-01, -5.16568660e-01, -7.15631215e-01,
       -5.55225702e-01,  3.54820391e+00, -5.73898698e-01, -6.99050092e-01,
       -4.87829542e-01, -5.82915320e-01, -5.44514726e-01, -5.96920254e-01,
        3.47861995e+00, -6.24268116e-01, -5.80115144e-01, -5.49886390e-01,
       -4.00052003e-01, -6.97348306e-01, -5.73898698e-01,  3.59167564e+00,
       -4.48011596e-01, -5.04786095e-01, -5.17581495e-01, -4.49997443e-01,
       -5.57288856e-01, -3.78310340e-01,  3.58148287e+00, -7.33781514e-01,
       -8.86557434e-01, -4.43853254e-01, -5.81691468e-01, -4.91960665e-01,
       -5.30395022e-01, -7.33781514e-01,  3.48553961e+00, -6.64086731e-01,
       -5.02446194e-01, -5.62869483e-01, -5.19345179e-01, -5.10320512e-01,
       -6.64086731e-01,  3.50174204e+00, -7.89064811e-01, -4.96401669e-01,
       -5.22523143e-01, -4.22648376e-01, -8.52943273e-01,  3.60701033e+00,
       -5.14352202e-01, -5.58757261e-01, -8.30848232e-01, -4.27460988e-01,
       -7.68839951e-01, -4.93999646e-01, -5.14352202e-01,  3.47743633e+00,
       -4.52603219e-01, -4.52746656e-01, -7.94894654e-01, -6.41575419e-01,
       -4.98525007e-01, -4.87415348e-01,  3.68451312e+00, -7.97232788e-01,
       -7.99892898e-01, -4.59871663e-01, -6.06298651e-01, -5.39532166e-01,
        3.46608420e+00, -5.71771887e-01, -5.82531831e-01, -5.11870329e-01,
       -6.54079333e-01, -5.61846933e-01, -5.83151404e-01, -5.71771887e-01,
        3.46222929e+00, -5.75093654e-01, -5.98767160e-01, -5.71598247e-01,
       -6.64321665e-01, -5.37157149e-01, -5.75093654e-01,  3.46141005e+00,
       -5.26280602e-01, -5.24649455e-01, -6.33907525e-01, -7.35888156e-01,
       -6.37171770e-01, -5.26280602e-01,  3.52982808e+00, -3.38198550e-01,
       -4.54701524e-01, -8.37587480e-01, -7.12679322e-01, -7.13570755e-01,
       -3.38198550e-01,  3.52756684e+00, -4.15741381e-01, -6.43308534e-01,
       -7.04068297e-01, -5.92761131e-01, -6.72170541e-01, -4.15741381e-01,
        3.45245170e+00, -4.91378048e-01, -6.83934421e-01, -5.96466173e-01,
       -5.98639454e-01, -5.88191700e-01, -4.91378048e-01,  3.42529099e+00,
       -5.83075011e-01, -5.98269115e-01, -5.65737661e-01, -6.86242934e-01,
       -4.69772899e-01, -5.82531831e-01,  3.40851928e+00, -5.99323379e-01,
       -4.80986595e-01, -5.89661639e-01, -7.20216046e-01, -4.33680640e-01,
       -5.83155676e-01, -5.99323379e-01,  3.46124864e+00, -6.00621287e-01,
       -5.24251607e-01, -5.45719634e-01, -5.57834802e-01, -6.04140290e-01,
       -5.83075011e-01,  3.40015283e+00, -5.22030112e-01, -5.87352979e-01,
       -5.55098362e-01, -5.08066562e-01, -5.53491390e-01, -6.51590271e-01,
        3.46277908e+00, -6.10742270e-01, -5.83790227e-01, -6.18874472e-01,
       -4.77667621e-01, -6.61146265e-01,  3.52185441e+00, -2.28025525e-01,
       -4.41465491e-01, -8.74186271e-01, -2.20488768e-01, -7.40813572e-01,
       -6.48178421e-01, -2.28025525e-01,  3.61009920e+00, -4.35841650e-01,
       -9.81776149e-01, -5.75463886e-01, -6.57306363e-01, -5.52880190e-01,
       -5.54272240e-01,  3.48022687e+00, -5.19395603e-01, -5.01597479e-01,
       -6.94774991e-01, -6.43766551e-01, -5.85191241e-01, -5.37984674e-01,
       -4.41465491e-01,  3.50892969e+00, -5.24796543e-01, -7.75725189e-01,
       -6.17325750e-01, -5.80014669e-01, -5.24796543e-01,  3.47763277e+00,
       -5.66647494e-01, -5.91958456e-01, -5.96889853e-01, -5.86712270e-01,
       -5.73367957e-01, -5.66647494e-01,  3.45336548e+00, -5.61177877e-01,
       -5.94197284e-01, -5.71262602e-01, -4.22171216e-01, -7.23691307e-01,
       -6.04500526e-01,  3.67692234e+00, -8.22718902e-01, -7.29209852e-01,
       -3.74630536e-01, -6.57253815e-01, -5.71166195e-01, -5.15885966e-01,
        3.49361566e+00, -5.42223432e-01, -5.80599576e-01, -6.26486675e-01,
       -6.72056087e-01, -5.29143426e-01,  3.57697063e+00, -5.40913793e-01,
       -5.85306432e-01, -6.66906091e-01, -5.82644803e-01, -5.34398178e-01,
       -6.03943984e-01, -5.40913793e-01,  3.44340107e+00, -5.84255597e-01,
       -5.83478720e-01, -5.96410802e-01, -5.29451603e-01, -5.99354149e-01,
       -5.84255597e-01,  3.43854767e+00, -5.64108074e-01, -5.32761691e-01,
       -6.28616556e-01, -5.89467801e-01, -5.62466601e-01, -5.85306432e-01,
        3.46807046e+00, -6.23812846e-01, -5.50164951e-01, -5.56851831e-01,
       -7.11004035e-01, -1.42400695e+00, -1.11236110e-01,  4.33365946e+00,
       -5.31223425e-01, -1.55618894e+00, -6.31237190e-01, -5.46513098e-01,
       -5.63011861e-01,  3.45073852e+00, -6.01467521e-01, -4.97584803e-01,
       -6.10924044e-01, -6.06748756e-01, -5.33257354e-01, -6.01467521e-01,
        3.45217731e+00, -5.91166622e-01, -5.06485328e-01, -6.13051730e-01,
       -5.82364575e-01, -5.71487958e-01, -5.91166622e-01,  3.45384020e+00,
       -5.68169809e-01, -5.40912326e-01, -5.99738908e-01, -5.93713592e-01,
       -5.19189955e-01, -6.24268116e-01,  3.49836338e+00, -6.02195350e-01,
       -5.73415949e-01, -5.85580420e-01, -6.55090994e-01, -4.48011596e-01,
       -8.86557434e-01,  3.62041335e+00, -7.68365193e-01, -3.85447179e-01,
       -4.76940956e-01, -4.02565193e-01, -7.89064811e-01, -5.58757261e-01,
        3.57041933e+00, -5.54209031e-01, -8.84003932e-01, -3.81819105e-01,
       -7.91501435e-01, -4.52603219e-01, -7.97232788e-01,  3.77737679e+00,
       -5.01079977e-01, -3.27521878e-01, -9.07437498e-01, -6.30621742e-01,
       -7.99892898e-01,  3.56033690e+00, -3.94980533e-01, -5.64589437e-01,
       -4.38989454e-01, -7.31262834e-01, -7.21811023e-01, -5.65103003e-01,
       -3.94980533e-01,  3.50451509e+00, -5.27458728e-01, -6.15104726e-01,
       -6.80057080e-01, -5.71006819e-01, -5.98370136e-01, -5.27458728e-01,
        3.46005269e+00, -5.97592890e-01, -5.67228486e-01, -5.98395628e-01,
       -5.21226127e-01, -6.00621287e-01, -5.97592890e-01,  3.46029785e+00,
       -6.27515813e-01, -5.63691135e-01, -5.49650599e-01, -6.18780585e-01,
       -6.10742270e-01, -4.35841650e-01,  3.47026081e+00, -5.23861996e-01,
       -6.98612367e-01, -5.82421945e-01, -5.64556433e-01, -6.31056610e-01,
       -5.61854229e-01,  3.46932397e+00, -5.37191635e-01, -5.98051914e-01,
       -5.76613151e-01, -5.71148037e-01, -6.29129564e-01, -5.37191635e-01,
        3.44359219e+00, -5.30197463e-01, -6.13787770e-01, -5.62137726e-01,
       -5.75465510e-01, -6.40607903e-01, -5.19395603e-01, -5.30197463e-01,
        3.46064206e+00, -5.44679327e-01, -6.50296253e-01, -4.92571441e-01,
       -5.76746624e-01,  3.45579776e+00, -7.15770292e-01, -6.25466891e-01,
       -5.23601060e-01, -5.21641457e-01, -5.99817386e-01, -4.43751351e-01,
       -7.15770292e-01,  3.52333946e+00, -8.42931507e-01, -5.20049403e-01,
       -4.01019522e-01, -5.60111812e-01, -5.68709259e-01, -6.25466891e-01,
        3.45301824e+00, -5.92361517e-01, -5.29651679e-01, -5.76717086e-01,
       -6.81380182e-01, -4.46873077e-01, -8.42931507e-01,  3.63707848e+00,
       -5.57116903e-01, -1.45594608e-01, -9.63182203e-01, -7.11639551e-01,
       -5.01597479e-01, -5.57116903e-01,  3.47992526e+00, -5.09822730e-01,
       -4.53537614e-01, -7.46210979e-01, -5.97973073e-01, -5.44961859e-01,
       -5.83790227e-01, -5.23861996e-01, -5.92361517e-01,  3.45289058e+00,
       -6.09941909e-01, -5.78125371e-01, -5.92371575e-01,  3.46181485e+00,
       -6.00586410e-01, -5.57375491e-01, -6.43283125e-01, -4.90072877e-01,
       -5.67092414e-01, -6.05754554e-01, -6.02195350e-01, -6.00586410e-01,
        3.52530087e+00, -5.20148461e-01, -6.29523685e-01, -5.80985134e-01,
       -6.23749882e-01, -5.57375491e-01,  3.45845686e+00, -5.26656401e-01,
       -5.39226211e-01, -6.30463741e-01, -5.68164581e-01, -6.35498503e-01,
       -5.26656401e-01,  3.45252502e+00, -5.37833723e-01, -6.12239507e-01,
       -5.72132302e-01, -5.68624136e-01, -5.86746427e-01, -5.37833723e-01,
        3.45275753e+00, -5.42565646e-01, -6.46179653e-01, -5.70807943e-01,
       -6.05980747e-01, -5.61177877e-01, -5.42565646e-01,  3.48568096e+00,
       -5.91625065e-01, -5.88349574e-01, -5.95982048e-01, -3.96158627e-01,
       -6.67890831e-01, -8.22718902e-01,  3.63500382e+00, -5.97516894e-01,
       -6.44787344e-01, -5.05931224e-01, -6.34684181e-01, -5.22030112e-01,
       -5.42223432e-01,  3.42929576e+00, -5.92914538e-01, -5.60737244e-01,
       -5.76706259e-01, -7.21231162e-01, -5.76284392e-01, -5.31223425e-01,
        3.45034606e+00, -5.53341758e-01, -4.15422302e-01, -6.52843026e-01,
       -6.21274607e-01, -5.80599576e-01, -5.53341758e-01,  3.45232719e+00,
       -5.57582746e-01, -5.45194487e-01, -5.94334021e-01, -5.33140048e-01,
       -5.64108074e-01, -5.97516894e-01,  3.36995680e+00, -5.23432752e-01,
       -5.52877757e-01, -5.98881271e-01, -1.17191837e-01, -6.23812846e-01,
       -1.55618894e+00, -4.15422302e-01,  3.87350487e+00, -6.29524615e-01,
       -5.31364325e-01, -5.23999048e-01, -7.29209852e-01, -4.97584803e-01,
        3.55481566e+00, -5.20650451e-01, -6.97684044e-01, -5.85687458e-01,
       -5.79504690e-01, -5.68169809e-01, -5.98051914e-01,  3.46460221e+00,
       -5.82690243e-01, -5.58016652e-01, -5.78168901e-01, -5.80237530e-01,
       -5.80115144e-01,  3.46550213e+00, -5.41916035e-01, -5.80694666e-01,
       -5.76907204e-01, -6.05631554e-01, -5.81768423e-01, -6.32341284e-01,
       -5.41916035e-01,  3.47425511e+00, -4.95060398e-01, -6.11911039e-01,
       -6.11257935e-01, -5.41312237e-01, -6.99050092e-01, -4.95060398e-01,
        3.65666191e+00, -3.02016152e-01, -5.49075127e-01, -1.07014791e+00,
       -4.87829542e-01, -5.04786095e-01, -3.02016152e-01,  3.71186035e+00,
       -1.23903697e+00, -7.97043412e-01, -9.78679976e-02, -2.83280185e-01,
       -5.49886390e-01, -5.73415949e-01, -5.80694666e-01,  3.47269213e+00,
       -6.12693340e-01, -5.86176934e-01, -5.69824851e-01, -5.17581495e-01,
       -7.68365193e-01,  3.58635613e+00, -4.83811020e-01, -5.08822931e-01,
       -7.30477598e-01, -5.77297896e-01, -4.43853254e-01, -5.02446194e-01,
        3.47347283e+00, -7.86492955e-01, -6.64115211e-01, -5.39663982e-01,
       -5.36901235e-01, -5.81691468e-01, -3.85447179e-01, -7.86492955e-01,
        3.56458895e+00, -8.45041865e-01, -4.81446773e-01, -4.84468704e-01,
       -5.62869483e-01, -4.96401669e-01, -6.64115211e-01,  3.47012864e+00,
       -6.37035159e-01, -4.99068410e-01, -6.10638711e-01, -5.22523143e-01,
       -5.54209031e-01, -6.37035159e-01,  3.50333202e+00, -7.25128606e-01,
       -4.96787552e-01, -5.67648532e-01, -8.30848232e-01, -4.52746656e-01,
        3.60999861e+00, -4.33510749e-01, -5.18028103e-01, -5.81297599e-01,
       -7.93567273e-01, -4.27460988e-01, -8.84003932e-01, -4.33510749e-01,
        3.54048953e+00, -6.22570132e-01, -6.90628694e-01, -4.82315036e-01,
       -7.94894654e-01, -5.01079977e-01, -5.18028103e-01,  3.54118146e+00,
       -4.45101541e-01, -5.36685283e-01, -7.45391906e-01, -4.59871663e-01,
       -3.27521878e-01, -5.64589437e-01,  3.77733882e+00, -3.72224653e-01,
       -1.07854768e+00, -7.82814763e-01, -1.91768751e-01, -5.11870329e-01,
       -5.98767160e-01,  3.44625993e+00, -6.08655956e-01, -6.03937293e-01,
       -6.03899574e-01, -5.19129616e-01, -6.54079333e-01, -4.80986595e-01,
       -6.08655956e-01,  3.47576255e+00, -6.40729662e-01, -5.12389794e-01,
       -5.78921211e-01, -5.71598247e-01, -5.24649455e-01, -6.03937293e-01,
        3.46972497e+00, -6.17414956e-01, -5.91633802e-01, -5.60491213e-01,
       -6.33907525e-01, -4.54701524e-01, -6.17414956e-01,  3.45132255e+00,
       -5.70883746e-01, -5.19551214e-01, -6.54863590e-01, -8.37587480e-01,
       -6.43308534e-01, -5.70883746e-01,  3.55282233e+00, -3.44486454e-01,
       -3.28119562e-01, -8.28436557e-01, -7.04068297e-01, -6.83934421e-01,
       -3.44486454e-01,  3.52268675e+00, -5.40961161e-01, -7.59004720e-01,
       -4.90231698e-01, -5.96466173e-01, -5.98269115e-01, -5.40961161e-01,
        3.44096369e+00, -5.61340119e-01, -6.01271577e-01, -5.42655542e-01,
       -5.65737661e-01, -5.87352979e-01, -5.92914538e-01, -5.61340119e-01,
        3.44154565e+00, -5.59102616e-01, -5.75097736e-01, -5.89661639e-01,
       -5.24251607e-01, -6.27515813e-01, -6.40729662e-01,  3.46658279e+00,
       -5.65395532e-01, -5.19028539e-01, -8.74186271e-01, -9.81776149e-01,
        3.75127990e+00, -4.58324429e-01, -8.99351592e-01, -5.37641458e-01,
       -5.75463886e-01, -6.98612367e-01, -4.58324429e-01,  3.48523134e+00,
       -5.01231084e-01, -6.65413432e-01, -5.86186140e-01, -6.94774991e-01,
       -5.44679327e-01, -5.09822730e-01,  3.48803316e+00, -5.14405215e-01,
       -5.39432949e-01, -6.84917949e-01, -2.20488768e-01, -7.75725189e-01,
       -5.91958456e-01, -8.99351592e-01,  3.56462894e+00, -5.71413635e-01,
       -5.05691305e-01, -5.96889853e-01, -5.94197284e-01, -5.71413635e-01,
        3.45966480e+00, -5.51648914e-01, -6.42314670e-01, -5.03200448e-01,
       -5.71262602e-01, -5.91625065e-01, -5.51648914e-01,  3.46122338e+00,
       -5.58595811e-01, -6.33917328e-01, -5.54173657e-01, -3.74630536e-01,
       -6.44787344e-01, -5.20650451e-01,  3.74012394e+00, -7.23774022e-01,
       -1.14277121e+00, -3.33510380e-01, -6.26486675e-01, -5.60737244e-01,
       -5.57582746e-01,  3.47819072e+00, -5.75479354e-01, -5.77940940e-01,
       -5.79963757e-01, -6.66906091e-01, -5.50164951e-01,  3.53026316e+00,
       -5.01300907e-01, -5.60662695e-01, -6.48313581e-01, -6.02914931e-01,
       -5.82644803e-01, -5.83478720e-01, -5.01300907e-01,  3.45758114e+00,
       -5.50276611e-01, -6.02130078e-01, -6.37750023e-01, -5.96410802e-01,
       -5.32761691e-01, -5.50276611e-01,  3.42242941e+00, -5.61452360e-01,
       -5.26092744e-01, -6.55435200e-01, -6.28616556e-01, -5.23432752e-01,
       -5.61452360e-01,  3.44320687e+00, -5.82589068e-01, -4.94668191e-01,
       -6.52447944e-01, -5.56851831e-01, -6.29524615e-01, -5.60662695e-01,
        3.47039895e+00, -5.38309779e-01, -5.67942597e-01, -6.17107437e-01,
       -6.10924044e-01, -5.06485328e-01, -6.97684044e-01,  3.50635634e+00,
       -6.17165919e-01, -3.60897712e-01, -7.13199294e-01, -6.13051730e-01,
       -5.40912326e-01, -6.17165919e-01,  3.45756641e+00, -5.90384269e-01,
       -4.81101062e-01, -6.14951107e-01, -5.99738908e-01, -5.82690243e-01,
       -5.90384269e-01,  3.46201275e+00, -5.74708914e-01, -5.30824158e-01,
       -5.83666258e-01, -4.49997443e-01, -1.23903697e+00, -4.83811020e-01,
        3.96520613e+00, -8.49347160e-01, -9.43013540e-01, -5.85580420e-01,
       -5.20148461e-01, -6.12693340e-01,  3.51749011e+00, -6.19843781e-01,
       -5.52697888e-01, -6.26526223e-01, -4.76940956e-01, -5.08822931e-01,
       -8.45041865e-01,  3.54487095e+00, -7.25759741e-01, -5.41438083e-01,
       -4.46867375e-01, -3.81819105e-01, -7.25128606e-01, -6.22570132e-01,
        3.54840419e+00, -6.32396458e-01, -7.84659330e-01, -4.01830562e-01,
       -9.07437498e-01, -4.45101541e-01, -3.72224653e-01,  3.55639857e+00,
       -5.62378601e-01, -5.61608493e-01, -7.07647788e-01, -4.38989454e-01,
       -1.07854768e+00,  3.84404474e+00, -4.45017703e-01, -9.41711468e-01,
       -9.39778430e-01, -7.31262834e-01, -6.15104726e-01, -4.45017703e-01,
        3.51466858e+00, -4.49046195e-01, -6.11928705e-01, -6.62308423e-01,
       -6.80057080e-01, -5.67228486e-01, -4.49046195e-01,  3.47733141e+00,
       -5.40973474e-01, -6.19732820e-01, -6.20293353e-01, -5.98395628e-01,
       -5.63691135e-01, -5.40973474e-01,  3.46233866e+00, -5.99990282e-01,
       -5.89639795e-01, -5.69648352e-01, -5.49650599e-01, -5.65395532e-01,
       -5.99990282e-01,  3.45731291e+00, -6.36248672e-01, -5.68851901e-01,
       -5.37175921e-01, -5.82421945e-01, -5.29651679e-01, -6.09941909e-01,
       -5.01231084e-01,  3.46741763e+00, -6.10405256e-01, -6.33765757e-01,
       -5.76613151e-01, -6.13787770e-01, -5.58016652e-01,  3.46655995e+00,
       -5.41080452e-01, -5.96956090e-01, -5.80105838e-01, -5.62137726e-01,
       -6.50296253e-01, -5.14405215e-01, -5.41080452e-01,  3.45224128e+00,
       -5.49251489e-01, -6.35070150e-01, -5.23601060e-01, -5.20049403e-01,
        3.48935089e+00, -6.70984263e-01, -7.36288921e-01, -5.56483630e-01,
       -4.81943614e-01, -5.21641457e-01, -5.76717086e-01, -6.10405256e-01,
       -6.70984263e-01,  3.44945635e+00, -5.07905768e-01, -5.61802521e-01,
       -4.01019522e-01, -1.45594608e-01, -7.36288921e-01,  3.57412818e+00,
       -1.00070736e+00, -6.29423234e-01, -2.49881717e-01, -4.11212813e-01,
       -9.63182203e-01, -4.53537614e-01, -1.00070736e+00,  3.74453830e+00,
       -4.72472736e-01, -8.54638383e-01, -7.46210979e-01, -5.39432949e-01,
       -4.72472736e-01,  3.50080167e+00, -4.81054143e-01, -5.69233769e-01,
       -6.92397091e-01, -6.43283125e-01, -5.39226211e-01,  3.48833779e+00,
       -6.23511956e-01, -6.04413302e-01, -4.00049842e-01, -6.77853351e-01,
       -4.90072877e-01, -6.29523685e-01, -6.19843781e-01, -6.23511956e-01,
        3.50230706e+00, -4.55128900e-01, -6.84225860e-01, -6.30463741e-01,
       -6.12239507e-01, -6.04413302e-01,  3.49926464e+00, -5.11959571e-01,
       -4.15494028e-01, -7.24694489e-01, -5.72132302e-01, -6.46179653e-01,
       -5.11959571e-01,  3.47408173e+00, -5.34040031e-01, -6.60419128e-01,
       -5.49351040e-01, -5.70807943e-01, -5.88349574e-01, -5.34040031e-01,
        3.44838067e+00, -5.47711661e-01, -6.40979073e-01, -5.66492391e-01,
       -5.95982048e-01, -5.58595811e-01, -5.47711661e-01,  3.47057123e+00,
       -5.90873536e-01, -5.85159179e-01, -5.92248992e-01, -5.05931224e-01,
       -5.52877757e-01, -7.23774022e-01,  3.52580853e+00, -5.62769072e-01,
       -5.42337428e-01, -6.38119028e-01, -5.76706259e-01, -5.59102616e-01,
       -5.75479354e-01,  3.44394672e+00, -6.05774982e-01, -5.67796526e-01,
       -5.59086982e-01, -6.52843026e-01, -5.45194487e-01, -5.31364325e-01,
       -5.38309779e-01,  3.43827365e+00, -5.94514490e-01, -5.76047540e-01,
       -5.94334021e-01, -5.77940940e-01, -5.94514490e-01,  3.47328845e+00,
       -5.93542698e-01, -5.43310021e-01, -5.69646277e-01, -5.98881271e-01,
       -5.82589068e-01, -5.62769072e-01,  3.42406117e+00, -4.84598446e-01,
       -5.43144313e-01, -6.52078999e-01, -5.85687458e-01, -1.14277121e+00,
       -3.60897712e-01,  3.71866929e+00, -2.37916999e-01, -5.96064636e-01,
       -7.95331277e-01, -5.78168901e-01, -5.74708914e-01, -5.96956090e-01,
        3.44677104e+00, -5.65770000e-01, -5.61444678e-01, -5.69722459e-01,
       -5.76907204e-01, -5.86176934e-01,  3.46689775e+00, -5.27525332e-01,
       -5.80559956e-01, -5.62954855e-01, -6.32773469e-01, -6.05631554e-01,
       -6.11911039e-01, -5.27525332e-01,  3.47220298e+00, -4.96327193e-01,
       -6.08564612e-01, -6.22243250e-01, -6.11257935e-01, -5.49075127e-01,
       -4.96327193e-01,  3.50755677e+00, -5.18195029e-01, -5.94738185e-01,
       -7.37963299e-01, -1.07014791e+00, -7.97043412e-01, -5.18195029e-01,
        3.89076778e+00, -1.16858040e+00, -3.36801030e-01, -9.78679976e-02,
       -8.49347160e-01,  3.76899989e+00, -4.03564210e-01, -5.96602709e-01,
       -8.07566426e-01, -3.70304129e-01, -6.43747262e-01, -5.69824851e-01,
       -5.52697888e-01, -5.80559956e-01,  3.47175740e+00, -6.01375844e-01,
       -5.85265356e-01, -5.82033505e-01, -7.30477598e-01, -9.43013540e-01,
       -4.03564210e-01,  3.73617394e+00, -4.26168922e-01, -6.90146656e-01,
       -5.42803016e-01, -5.77297896e-01, -7.25759741e-01, -4.26168922e-01,
        3.54228993e+00, -6.10994519e-01, -7.36807196e-01, -4.65261656e-01,
       -5.39663982e-01, -4.81446773e-01,  3.48666680e+00, -7.02737851e-01,
       -7.32578213e-01, -4.97091379e-01, -5.33148604e-01, -5.36901235e-01,
       -4.99068410e-01, -7.02737851e-01,  3.47048680e+00, -6.13551678e-01,
       -5.18213812e-01, -6.00013814e-01, -4.84468704e-01, -5.41438083e-01,
       -7.32578213e-01,  3.49947981e+00, -7.09248452e-01, -5.64221445e-01,
       -4.67524914e-01, -6.10638711e-01, -4.96787552e-01, -6.13551678e-01,
        3.48320091e+00, -5.96784581e-01, -4.95269302e-01, -6.70169087e-01,
       -5.67648532e-01, -6.32396458e-01, -5.96784581e-01,  3.50399007e+00,
       -5.82482605e-01, -4.83084040e-01, -6.41593858e-01, -5.81297599e-01,
       -6.90628694e-01,  3.52027135e+00, -4.26047698e-01, -4.23590296e-01,
       -6.18003060e-01, -7.80704001e-01, -7.93567273e-01, -5.36685283e-01,
       -4.26047698e-01,  3.54090741e+00, -4.11691526e-01, -6.81988958e-01,
       -6.90926668e-01, -4.82315036e-01, -7.84659330e-01, -4.23590296e-01,
        3.43821403e+00, -6.80247696e-01, -5.58857758e-01, -5.08543912e-01,
       -7.45391906e-01, -5.62378601e-01, -4.11691526e-01,  3.51199318e+00,
       -4.76738531e-01, -6.52773566e-01, -6.63019051e-01, -7.82814763e-01,
       -5.61608493e-01,  3.50219618e+00, -4.14415398e-01, -5.38439366e-01,
       -5.27969631e-01, -6.76948527e-01, -1.91768751e-01, -9.41711468e-01,
       -4.14415398e-01,  3.71200192e+00, -3.24238048e-01, -7.39310267e-01,
       -5.47860830e-01, -5.52697158e-01, -6.03899574e-01, -5.12389794e-01,
        3.45678737e+00, -6.01506210e-01, -6.34156300e-01, -5.92822547e-01,
       -5.12012945e-01, -5.19129616e-01, -5.91633802e-01, -6.01506210e-01,
        3.46352013e+00, -5.96049386e-01, -6.31281845e-01, -5.23919275e-01,
       -5.78921211e-01, -5.19028539e-01, -6.36248672e-01, -6.34156300e-01,
        3.46411724e+00, -5.69374453e-01, -5.26388062e-01, -5.60491213e-01,
       -5.19551214e-01, -5.96049386e-01,  3.46394102e+00, -6.40857681e-01,
       -6.49057669e-01, -4.97933856e-01, -6.54863590e-01, -3.28119562e-01,
       -6.40857681e-01,  3.50203751e+00, -7.53431426e-01, -6.30657557e-01,
       -4.94107691e-01, -8.28436557e-01, -7.59004720e-01, -7.53431426e-01,
        3.62223777e+00, -6.29850200e-01, -6.51514862e-01, -4.90231698e-01,
       -6.01271577e-01, -6.29850200e-01,  3.47358361e+00, -6.29112965e-01,
       -5.36673759e-01, -5.86443414e-01, -5.42655542e-01, -5.75097736e-01,
       -6.05774982e-01, -6.29112965e-01,  3.45835184e+00, -5.70768437e-01,
       -5.34942175e-01, -5.37641458e-01, -6.65413432e-01, -5.05691305e-01,
       -6.42314670e-01,  3.46860038e+00, -5.01508282e-01, -6.16031229e-01,
       -5.86186140e-01, -6.33765757e-01, -5.07905768e-01, -5.01508282e-01,
        3.48692002e+00, -6.28155863e-01, -6.29398212e-01, -6.84917949e-01,
       -5.49251489e-01, -4.81054143e-01,  3.44254665e+00, -5.22677373e-01,
       -5.48556255e-01, -6.56089441e-01, -5.03200448e-01, -6.33917328e-01,
       -6.16031229e-01, -6.28155863e-01,  3.47936663e+00, -5.94181219e-01,
       -5.03880548e-01, -5.54173657e-01, -5.90873536e-01, -5.94181219e-01,
        3.46180710e+00, -5.78538222e-01, -5.79072402e-01, -5.64968062e-01,
       -3.33510380e-01, -5.42337428e-01, -2.37916999e-01,  3.48451210e+00,
       -6.33648918e-01, -8.14893779e-01, -6.12104834e-01, -3.10099761e-01,
       -5.79963757e-01, -5.67796526e-01, -5.93542698e-01,  3.47721773e+00,
       -5.83410255e-01, -5.59283336e-01, -5.93221161e-01, -6.48313581e-01,
       -5.67942597e-01,  3.49313514e+00, -5.10086113e-01, -5.51396222e-01,
       -5.95220685e-01, -6.20175945e-01, -6.02914931e-01, -6.02130078e-01,
       -5.10086113e-01,  3.46665534e+00, -5.25186095e-01, -6.12200610e-01,
       -6.14137519e-01, -6.37750023e-01, -5.26092744e-01, -5.25186095e-01,
        3.43022829e+00, -5.69704451e-01, -5.52119415e-01, -6.19375561e-01,
       -6.55435200e-01, -4.94668191e-01, -5.69704451e-01,  3.45858114e+00,
       -5.98020962e-01, -5.19638065e-01, -6.21114268e-01, -6.52447944e-01,
       -4.84598446e-01, -5.98020962e-01,  3.45668003e+00, -6.04776177e-01,
       -5.07978149e-01, -6.08858347e-01, -6.17107437e-01, -5.76047540e-01,
       -5.43310021e-01, -5.51396222e-01,  3.46894684e+00, -6.14266652e-01,
       -5.66818968e-01, -7.13199294e-01, -4.81101062e-01, -5.96064636e-01,
        3.51679476e+00, -6.02264299e-01, -4.05921752e-01, -7.18243722e-01,
       -6.14951107e-01, -5.30824158e-01, -6.02264299e-01,  3.47068137e+00,
       -6.12725951e-01, -5.00846442e-01, -6.09069414e-01, -5.83666258e-01,
       -5.65770000e-01, -6.12725951e-01,  3.46586129e+00, -5.98976964e-01,
       -5.33708764e-01, -5.71013356e-01, -2.83280185e-01, -1.16858040e+00,
       -5.96602709e-01,  3.91418435e+00, -6.75753791e-01, -3.82762068e-01,
       -4.41074528e-01, -3.66130666e-01, -6.26526223e-01, -4.55128900e-01,
       -6.01375844e-01,  3.48726285e+00, -6.35940873e-01, -5.28313912e-01,
       -6.39977097e-01, -4.46867375e-01, -6.10994519e-01, -7.09248452e-01,
        3.49905022e+00, -6.58472958e-01, -6.35560993e-01, -4.37905921e-01,
       -4.01830562e-01, -5.82482605e-01, -6.80247696e-01,  3.74255854e+00,
       -5.52949679e-01, -1.06945494e+00, -4.55593064e-01, -7.07647788e-01,
       -4.76738531e-01, -5.38439366e-01,  3.47239057e+00, -6.21840390e-01,
       -5.08025102e-01, -6.19699389e-01, -9.39778430e-01, -6.11928705e-01,
       -3.24238048e-01,  3.58709078e+00, -4.52964603e-01, -6.55991582e-01,
       -6.02189415e-01, -6.62308423e-01, -6.19732820e-01, -4.52964603e-01,
        3.47855432e+00, -5.17461505e-01, -6.15750775e-01, -6.10336195e-01,
       -6.20293353e-01, -5.89639795e-01, -5.17461505e-01,  3.46596151e+00,
       -5.68333709e-01, -5.89225918e-01, -5.81007231e-01, -5.69648352e-01,
       -5.68851901e-01, -5.68333709e-01,  3.45400537e+00, -6.13128483e-01,
       -5.72519888e-01, -5.61523037e-01, -5.37175921e-01, -5.69374453e-01,
       -6.13128483e-01,  3.46079734e+00, -6.12638130e-01, -5.70910108e-01,
       -5.57570250e-01, -5.80105838e-01, -6.35070150e-01, -5.61444678e-01,
       -5.22677373e-01,  3.45000839e+00, -5.95952690e-01, -5.54757658e-01,
       -5.56483630e-01, -5.61802521e-01, -6.29398212e-01, -5.03880548e-01,
       -5.78538222e-01,  3.46413717e+00, -6.34034037e-01, -4.81943614e-01,
       -6.29423234e-01, -5.79072402e-01, -6.34034037e-01,  3.47639935e+00,
       -5.57412523e-01, -5.94513541e-01, -2.49881717e-01, -8.54638383e-01,
       -5.69233769e-01,  3.58786065e+00, -4.60479130e-01, -7.95299450e-01,
       -6.58328202e-01, -6.92397091e-01, -5.48556255e-01, -4.60479130e-01,
        3.46551379e+00, -5.33955164e-01, -5.68243757e-01, -6.61882394e-01,
       -4.00049842e-01, -6.84225860e-01, -6.35940873e-01,  3.52231066e+00,
       -7.04370725e-01, -4.15252689e-01, -6.82470671e-01, -6.77853351e-01,
       -4.15494028e-01, -7.04370725e-01,  3.56813520e+00, -2.70813704e-01,
       -3.50867127e-01, -9.24740132e-01, -2.23996129e-01, -7.24694489e-01,
       -6.60419128e-01, -2.70813704e-01,  3.58410834e+00, -5.09007417e-01,
       -8.94519475e-01, -5.24654132e-01, -5.49351040e-01, -6.40979073e-01,
       -5.09007417e-01,  3.47303664e+00, -5.54181040e-01, -6.85624443e-01,
       -5.33893625e-01, -5.66492391e-01, -5.85159179e-01, -5.54181040e-01,
        3.47355795e+00, -5.71493427e-01, -6.19899745e-01, -5.76332166e-01,
       -5.92248992e-01, -5.64968062e-01, -5.57412523e-01, -5.71493427e-01,
        3.46699540e+00, -6.38537949e-01, -5.42334451e-01, -6.38119028e-01,
       -5.43144313e-01, -6.33648918e-01,  3.43607488e+00, -5.06666517e-01,
       -4.36788900e-01, -6.77707208e-01, -5.59086982e-01, -5.70768437e-01,
       -5.83410255e-01,  3.45581012e+00, -6.08997986e-01, -5.62703902e-01,
       -5.70842558e-01, -5.69646277e-01, -5.59283336e-01, -6.14266652e-01,
        3.47956806e+00, -5.81947333e-01, -5.46420643e-01, -6.08003821e-01,
       -6.52078999e-01, -6.04776177e-01, -5.06666517e-01,  3.46193501e+00,
       -5.26264846e-01, -5.58557531e-01, -6.13590942e-01, -7.95331277e-01,
       -8.14893779e-01, -4.05921752e-01,  3.56713156e+00, -4.31768717e-01,
       -5.41719845e-01, -5.77496192e-01, -5.69722459e-01, -5.98976964e-01,
       -5.95952690e-01,  3.46590055e+00, -5.73351548e-01, -5.61215950e-01,
       -5.66680940e-01, -5.62954855e-01, -5.85265356e-01,  3.45905000e+00,
       -5.39996016e-01, -5.72032954e-01, -6.28420823e-01, -5.70379992e-01,
       -6.32773469e-01, -6.08564612e-01, -5.39996016e-01,  3.47271738e+00,
       -4.65550166e-01, -5.50784727e-01, -6.75048386e-01, -6.22243250e-01,
       -5.94738185e-01, -4.65550166e-01,  3.51070877e+00, -5.46948225e-01,
       -5.75288049e-01, -7.05940895e-01, -7.37963299e-01, -3.36801030e-01,
       -6.75753791e-01, -5.46948225e-01,  3.71810929e+00, -9.19209097e-01,
       -5.01433848e-01, -8.07566426e-01, -3.82762068e-01,  3.57995626e+00,
       -7.93673583e-01, -5.00435441e-01, -5.46612221e-01, -5.48906526e-01,
       -3.70304129e-01, -6.90146656e-01,  3.81907038e+00, -4.71188805e-01,
       -4.07264544e-01, -6.30118827e-01, -6.52073265e-01, -5.97974158e-01,
       -5.82033505e-01, -5.28313912e-01, -5.72032954e-01,  3.46671476e+00,
       -6.20961711e-01, -5.94636743e-01, -5.68735934e-01, -5.42803016e-01,
       -7.36807196e-01, -4.71188805e-01,  3.60893951e+00, -5.94276292e-01,
       -7.79097916e-01, -4.84766285e-01, -4.65261656e-01, -6.58472958e-01,
       -5.94276292e-01,  3.49243700e+00, -6.52085929e-01, -6.63246983e-01,
       -4.59093181e-01, -4.97091379e-01, -5.64221445e-01,  3.47769683e+00,
       -6.83268261e-01, -6.83353152e-01, -4.91278514e-01, -5.58484084e-01,
       -5.33148604e-01, -5.18213812e-01, -6.83268261e-01,  3.47423114e+00,
       -6.44341694e-01, -5.26876903e-01, -5.68381865e-01, -6.00013814e-01,
       -4.95269302e-01, -6.44341694e-01,  3.48336700e+00, -5.91158742e-01,
       -5.08904956e-01, -6.43678495e-01, -4.67524914e-01, -6.35560993e-01,
       -6.83353152e-01,  3.48981691e+00, -6.33821848e-01, -6.34584239e-01,
       -4.34971765e-01, -6.70169087e-01, -4.83084040e-01, -5.91158742e-01,
        3.49666562e+00, -5.85619737e-01, -4.73215912e-01, -6.93418104e-01,
       -6.41593858e-01, -5.52949679e-01, -5.85619737e-01,  3.51645250e+00,
       -5.92496342e-01, -4.59360609e-01, -6.84432275e-01, -6.18003060e-01,
       -6.81988958e-01,  3.47520037e+00, -4.21215699e-01, -3.91791455e-01,
       -7.55081120e-01, -6.07120080e-01, -7.80704001e-01, -5.58857758e-01,
       -4.21215699e-01,  3.55496022e+00, -4.34512965e-01, -4.75711825e-01,
       -8.83957974e-01, -6.90926668e-01, -6.52773566e-01, -3.91791455e-01,
        3.50186321e+00, -4.15681169e-01, -7.08338485e-01, -6.42351867e-01,
       -5.08543912e-01, -1.06945494e+00, -4.34512965e-01,  3.86825209e+00,
       -4.31521431e-01, -4.32277733e-01,  1.36793470e-01, -1.12873458e+00,
       -6.63019051e-01, -6.21840390e-01, -4.15681169e-01,  3.49344936e+00,
       -4.75302250e-01, -6.99220112e-01, -6.18386390e-01, -5.27969631e-01,
       -7.39310267e-01, -4.07264544e-01, -7.79097916e-01,  3.65055572e+00,
       -5.45569760e-01, -6.51343601e-01, -6.76948527e-01, -5.08025102e-01,
       -4.84766285e-01, -6.52085929e-01, -5.45569760e-01,  3.49758519e+00,
       -6.30189585e-01, -5.47860830e-01, -6.55991582e-01,  3.51900763e+00,
       -5.65400421e-01, -5.25639693e-01, -5.78689226e-01, -6.45425878e-01,
       -5.92822547e-01, -5.26388062e-01, -6.12638130e-01,  3.45849137e+00,
       -5.79986227e-01, -5.80223283e-01, -5.66433121e-01, -5.12012945e-01,
       -6.31281845e-01, -5.79986227e-01,  3.45372048e+00, -5.44682671e-01,
       -6.36764704e-01, -5.48992084e-01, -5.23919275e-01, -6.49057669e-01,
       -5.44682671e-01,  3.49292270e+00, -5.87476995e-01, -7.88008454e-01,
       -3.99777636e-01, -4.97933856e-01, -6.30657557e-01, -5.87476995e-01,
        3.48043192e+00, -5.70219104e-01, -7.14809262e-01, -4.79335149e-01,
       -4.94107691e-01, -6.51514862e-01, -5.36673759e-01, -5.70219104e-01,
        3.47629534e+00, -5.72160853e-01, -6.51619070e-01, -5.86443414e-01,
       -5.34942175e-01, -6.08997986e-01, -5.72160853e-01,  3.46264032e+00,
       -6.11576931e-01, -5.48518960e-01, -6.56089441e-01, -5.54757658e-01,
       -5.33955164e-01, -5.73351548e-01,  3.46789045e+00, -5.43681652e-01,
       -6.06054990e-01, -4.11212813e-01, -5.94513541e-01, -7.95299450e-01,
       -6.38537949e-01,  3.50727631e+00, -4.83086841e-01, -5.84625712e-01,
       -6.12104834e-01, -4.36788900e-01,  3.75259244e+00, -1.30411073e+00,
       -5.60118413e-01, -6.71840257e-01, -1.67629304e-01, -5.93221161e-01,
       -5.62703902e-01, -5.81947333e-01,  3.45517670e+00, -5.50367960e-01,
       -5.20411170e-01, -6.46525172e-01, -5.95220685e-01, -6.12200610e-01,
        3.47219012e+00, -5.39014941e-01, -5.41426785e-01, -6.08837115e-01,
       -5.75489979e-01, -6.20175945e-01, -5.66818968e-01, -5.46420643e-01,
       -5.39014941e-01,  3.47675208e+00, -6.06989089e-01, -5.97332492e-01,
       -6.14137519e-01, -5.52119415e-01, -5.41426785e-01,  3.45089303e+00,
       -5.74854795e-01, -5.84735617e-01, -5.83618897e-01, -6.19375561e-01,
       -5.19638065e-01, -5.74854795e-01,  3.44762491e+00, -5.91548007e-01,
       -5.67593620e-01, -5.74614862e-01, -6.21114268e-01, -5.07978149e-01,
       -5.91548007e-01,  3.45788372e+00, -6.08845035e-01, -5.69517502e-01,
       -5.58880755e-01, -6.08858347e-01, -5.26264846e-01, -6.08845035e-01,
        3.46094520e+00, -6.44179168e-01, -5.69960630e-01, -5.02837176e-01,
       -3.10099761e-01, -4.31768717e-01, -1.30411073e+00,  4.00254097e+00,
       -1.17138750e+00, -7.85174263e-01, -7.18243722e-01, -5.00846442e-01,
       -5.41719845e-01,  3.50919282e+00, -5.89388474e-01, -5.47738925e-01,
       -6.11255411e-01, -6.09069414e-01, -5.33708764e-01, -5.89388474e-01,
        3.45468421e+00, -6.24936525e-01, -5.07524447e-01, -5.90056587e-01,
       -5.71013356e-01, -5.61215950e-01, -6.24936525e-01,  3.45639150e+00,
       -6.06261681e-01, -5.27678348e-01, -5.65285642e-01, -4.41074528e-01,
       -9.19209097e-01,  3.77666353e+00, -8.18690739e-01, -3.59815804e-01,
       -8.00821894e-01, -4.37051470e-01, -3.66130666e-01, -7.93673583e-01,
       -8.18690739e-01,  3.67788265e+00, -5.65557027e-01, -7.63018248e-01,
       -3.70812389e-01, -6.43747262e-01, -5.00435441e-01, -6.30118827e-01,
        3.62589347e+00, -6.65282728e-01, -5.28909495e-01, -6.57399718e-01,
       -6.39977097e-01, -4.15252689e-01, -6.20961711e-01,  3.48941231e+00,
       -6.64205732e-01, -5.37383953e-01, -6.11631133e-01, -4.37905921e-01,
       -6.63246983e-01, -6.33821848e-01,  3.49267118e+00, -6.27769675e-01,
       -6.84185739e-01, -4.45741014e-01, -4.55593064e-01, -5.92496342e-01,
       -4.31521431e-01,  3.56820220e+00, -4.35185208e-01, -1.27119887e+00,
       -3.82207277e-01, -6.19699389e-01, -4.59093181e-01, -4.75302250e-01,
       -6.30189585e-01, -6.27769675e-01,  3.47576410e+00, -6.63710022e-01,
       -5.52697158e-01, -6.52073265e-01, -6.51343601e-01, -5.65400421e-01,
        3.64456124e+00, -6.30064389e-01, -5.92982402e-01, -6.02189415e-01,
       -6.15750775e-01, -5.25639693e-01,  3.47298330e+00, -5.24184907e-01,
       -5.39367942e-01, -6.65850572e-01, -6.10336195e-01, -5.89225918e-01,
       -5.24184907e-01,  3.46547687e+00, -5.58899750e-01, -5.58134242e-01,
       -6.24695857e-01, -5.81007231e-01, -5.72519888e-01, -5.58899750e-01,
        3.46415591e+00, -5.90601388e-01, -5.64702861e-01, -5.96424789e-01,
       -5.61523037e-01, -5.70910108e-01, -5.90601388e-01,  3.46058069e+00,
       -5.85863794e-01, -5.60539275e-01, -5.91143090e-01, -5.57570250e-01,
       -5.80223283e-01, -5.85863794e-01,  3.45761489e+00, -5.61072432e-01,
       -5.78298051e-01, -5.94587081e-01, -6.58328202e-01, -5.68243757e-01,
       -4.83086841e-01,  3.44601450e+00, -5.27711490e-01, -5.96373215e-01,
       -6.12270998e-01, -6.61882394e-01, -5.43681652e-01, -5.27711490e-01,
        3.46918663e+00, -5.82960893e-01, -5.41598588e-01, -6.11351612e-01,
       -6.82470671e-01, -3.50867127e-01, -6.64205732e-01,  3.60184000e+00,
       -8.51817649e-01, -4.73876131e-01, -5.78602689e-01, -9.24740132e-01,
       -8.94519475e-01,  3.73865050e+00, -4.91198580e-01, -9.16306003e-01,
       -5.11886314e-01, -5.24654132e-01, -6.85624443e-01, -4.91198580e-01,
        3.48574866e+00, -5.59624955e-01, -7.11217195e-01, -5.13429355e-01,
       -5.33893625e-01, -6.19899745e-01, -5.59624955e-01,  3.47037247e+00,
       -5.75191712e-01, -5.39848137e-01, -6.41914298e-01, -5.76332166e-01,
       -5.42334451e-01, -5.84625712e-01, -5.96373215e-01, -5.75191712e-01,
        3.45938370e+00, -5.84526446e-01, -6.77707208e-01, -5.58557531e-01,
       -5.60118413e-01,  3.47234465e+00, -5.51679195e-01, -4.97778929e-01,
       -6.26503371e-01, -5.70842558e-01, -6.11576931e-01, -5.50367960e-01,
        3.43993403e+00, -5.31010177e-01, -5.58306197e-01, -6.17830210e-01,
       -6.08003821e-01, -5.20411170e-01, -6.06989089e-01,  3.47491292e+00,
       -5.74710976e-01, -4.98921979e-01, -6.65875885e-01, -6.13590942e-01,
       -6.44179168e-01, -5.51679195e-01,  3.45965460e+00, -5.23477339e-01,
       -5.88145759e-01, -5.38582200e-01, -5.77496192e-01, -1.17138750e+00,
       -5.47738925e-01,  3.79189047e+00, -2.95421171e-01, -6.30145191e-01,
       -5.69701492e-01, -5.66680940e-01, -6.06054990e-01, -6.06261681e-01,
       -5.82960893e-01,  3.46366400e+00, -5.59051625e-01, -5.42653870e-01,
       -6.28420823e-01, -5.50784727e-01,  3.46012110e+00, -5.37731788e-01,
       -5.22923100e-01, -5.77087984e-01, -6.43172676e-01, -5.70379992e-01,
       -5.94636743e-01, -5.37731788e-01,  3.46031023e+00, -5.61836088e-01,
       -6.14886400e-01, -5.80839221e-01, -6.75048386e-01, -5.75288049e-01,
       -5.22923100e-01,  3.53123860e+00, -4.59924004e-01, -5.22190275e-01,
       -7.75864791e-01, -7.05940895e-01, -5.01433848e-01, -3.59815804e-01,
       -4.59924004e-01,  3.70384432e+00, -5.12709520e-01, -4.33818190e-01,
       -7.30202056e-01, -5.46612221e-01, -5.65557027e-01,  3.50137004e+00,
       -5.96936203e-01, -7.08279425e-01, -4.69203673e-01, -6.14781495e-01,
       -5.48906526e-01, -6.65282728e-01, -5.96936203e-01,  3.53249676e+00,
       -5.12127715e-01, -5.45732232e-01, -6.63511356e-01, -5.97974158e-01,
       -5.28909495e-01, -6.30064389e-01,  3.51770061e+00, -5.26415906e-01,
       -6.26510265e-01, -6.07826393e-01, -5.68735934e-01, -5.37383953e-01,
       -5.61836088e-01,  3.45663161e+00, -6.07174194e-01, -6.12862258e-01,
       -5.68639178e-01, -4.91278514e-01, -6.34584239e-01,  3.47187204e+00,
       -6.57644025e-01, -6.11398964e-01, -6.42168684e-01, -4.34797611e-01,
       -5.58484084e-01, -5.26876903e-01, -6.57644025e-01,  3.46879481e+00,
       -6.59497947e-01, -5.07715571e-01, -5.58576282e-01, -5.68381865e-01,
       -5.08904956e-01, -6.59497947e-01,  3.47025881e+00, -5.96672567e-01,
       -5.46083419e-01, -5.90718060e-01, -6.43678495e-01, -4.73215912e-01,
       -5.96672567e-01,  3.47955299e+00, -5.84097811e-01, -5.13968834e-01,
       -6.67919369e-01, -4.34971765e-01, -6.84185739e-01, -6.11398964e-01,
        3.49472524e+00, -6.22406007e-01, -7.23941721e-01, -4.17821050e-01,
       -6.93418104e-01, -4.59360609e-01, -5.84097811e-01,  3.49507304e+00,
       -5.99979491e-01, -4.71209325e-01, -6.87007698e-01, -6.84432275e-01,
       -4.35185208e-01, -5.99979491e-01,  3.48632134e+00, -6.71173502e-01,
       -4.47201629e-01, -6.48349239e-01, -7.55081120e-01, -4.75711825e-01,
        3.46611513e+00, -4.42511362e-01, -4.87502473e-01, -7.33368226e-01,
       -5.71940119e-01, -6.07120080e-01, -7.08338485e-01, -4.42511362e-01,
        3.47876043e+00, -3.88257992e-01, -7.47020673e-01, -5.85511839e-01,
       -8.83957974e-01, -4.32277733e-01, -4.87502473e-01,  3.65376133e+00,
       -4.55006726e-01, -3.75398631e-01, -9.85912306e-01, -3.37054873e-02,
       -6.42351867e-01, -6.99220112e-01, -3.88257992e-01,  3.50964306e+00,
       -4.23899896e-01, -5.97775776e-01, -7.58137421e-01,  1.36793470e-01,
       -1.27119887e+00,  4.05457943e+00, -6.58601110e-01, -1.22280448e+00,
       -3.60551753e-01, -6.78216689e-01, -6.18386390e-01, -4.45741014e-01,
       -6.63710022e-01, -6.22406007e-01, -4.23899896e-01,  3.48484076e+00,
       -7.10697436e-01, -5.78689226e-01, -5.92982402e-01, -5.26415906e-01,
        3.49784357e+00, -5.85732466e-01, -6.33359338e-01, -5.80664233e-01,
       -6.45425878e-01, -5.39367942e-01, -5.85732466e-01,  3.51760820e+00,
       -5.18557453e-01, -4.79808096e-01, -7.48716365e-01, -5.66433121e-01,
       -6.36764704e-01, -5.61072432e-01,  3.47202178e+00, -4.66839638e-01,
       -5.79618673e-01, -6.61293208e-01, -5.48992084e-01, -7.88008454e-01,
       -4.66839638e-01,  3.67411792e+00, -1.95933244e-01, -5.88151285e-01,
       -1.08619321e+00, -3.99777636e-01, -7.14809262e-01, -1.95933244e-01,
        3.52568922e+00, -6.04557365e-01, -4.22994183e-01, -8.83468904e-01,
       -3.04148621e-01, -4.79335149e-01, -6.51619070e-01, -5.48518960e-01,
       -5.31010177e-01, -6.04557365e-01,  3.46979087e+00, -6.54750153e-01,
       -2.23996129e-01, -8.51817649e-01, -9.16306003e-01,  3.64618651e+00,
       -5.10427893e-01, -5.15572822e-01, -6.28066015e-01, -6.71840257e-01,
       -4.97778929e-01,  3.44300812e+00, -6.14683393e-01, -6.31589436e-01,
       -5.11029372e-01, -5.16086735e-01, -1.67629304e-01, -7.85174263e-01,
       -2.95421171e-01, -6.14683393e-01,  3.49374494e+00, -5.41395365e-01,
       -5.81687176e-01, -5.07754263e-01, -6.46525172e-01, -5.58306197e-01,
       -5.74710976e-01,  3.47219856e+00, -5.01837825e-01, -4.56719038e-01,
       -7.34099353e-01, -6.08837115e-01, -5.97332492e-01, -4.98921979e-01,
        3.46995436e+00, -5.46573023e-01, -6.11103058e-01, -6.07186698e-01,
       -5.75489979e-01, -5.84735617e-01, -5.46573023e-01,  3.46819003e+00,
       -5.53939102e-01, -6.38004838e-01, -5.69447469e-01, -5.83618897e-01,
       -5.67593620e-01, -5.53939102e-01,  3.46774860e+00, -6.06365079e-01,
       -6.61347734e-01, -4.94884164e-01, -5.74614862e-01, -5.69517502e-01,
       -6.06365079e-01,  3.45854489e+00, -5.84402404e-01, -6.06214117e-01,
       -5.17430921e-01, -5.58880755e-01, -5.69960630e-01, -5.84402404e-01,
        3.51078363e+00, -4.60194604e-01, -7.14851965e-01, -6.22493269e-01,
       -5.02837176e-01, -5.23477339e-01, -4.60194604e-01,  4.09297152e+00,
       -7.95725976e-01, -1.72360710e+00, -8.71293245e-02, -6.11255411e-01,
       -5.07524447e-01, -6.30145191e-01,  3.51445902e+00, -6.60921379e-01,
       -6.66088214e-01, -4.38524374e-01, -5.90056587e-01, -5.27678348e-01,
       -6.60921379e-01,  3.48124961e+00, -6.39362686e-01, -5.66355769e-01,
       -4.96874847e-01, -5.65285642e-01, -5.59051625e-01, -6.39362686e-01,
        3.46183702e+00, -6.47707004e-01, -4.95885207e-01, -5.54544852e-01,
       -8.00821894e-01, -5.12709520e-01,  3.64237271e+00, -5.46614878e-01,
       -5.86631307e-01, -6.07041613e-01, -5.88553499e-01, -4.37051470e-01,
       -7.63018248e-01, -5.46614878e-01,  3.48479856e+00, -6.88968880e-01,
       -5.64944914e-01, -4.84200175e-01, -3.70812389e-01, -7.08279425e-01,
       -6.88968880e-01,  3.52025697e+00, -6.91078890e-01, -6.36313127e-01,
       -4.24804257e-01, -6.57399718e-01, -5.12127715e-01, -6.26510265e-01,
        3.54981885e+00, -4.84863269e-01, -5.29755620e-01, -7.39162265e-01,
       -6.11631133e-01, -4.73876131e-01, -6.07174194e-01,  3.48045911e+00,
       -6.55653407e-01, -5.56123280e-01, -5.76000961e-01, -3.82207277e-01,
       -6.71173502e-01, -6.58601110e-01,  3.34194653e+00, -4.10608273e-01,
       -8.57978202e-01, -3.61378161e-01, -1.12873458e+00, -4.55006726e-01,
       -1.22280448e+00,  3.93843371e+00, -5.58025290e-01, -5.73862639e-01,
       -6.65850572e-01, -5.58134242e-01, -5.18557453e-01,  3.47712487e+00,
       -5.23984344e-01, -5.22540565e-01, -6.88057694e-01, -6.24695857e-01,
       -5.64702861e-01, -5.23984344e-01,  3.45727688e+00, -5.60756856e-01,
       -5.43706460e-01, -6.39430502e-01, -5.96424789e-01, -5.60539275e-01,
       -5.60756856e-01,  3.46438020e+00, -5.56579668e-01, -5.41688366e-01,
       -6.48391249e-01, -5.91143090e-01, -5.78298051e-01, -5.56579668e-01,
        3.45735767e+00, -5.55446963e-01, -5.58488198e-01, -6.17401696e-01,
       -5.94587081e-01, -5.79618673e-01, -5.55446963e-01,  3.45225158e+00,
       -5.25374907e-01, -5.95666025e-01, -6.01557933e-01, -6.12270998e-01,
       -5.41598588e-01, -5.39848137e-01, -5.84526446e-01,  3.46024888e+00,
       -5.96421523e-01, -5.85583189e-01, -6.11351612e-01, -5.42653870e-01,
       -6.47707004e-01, -5.96421523e-01,  3.46903836e+00, -5.37508919e-01,
       -5.33395426e-01, -5.78602689e-01, -5.10427893e-01, -6.55653407e-01,
        3.54276899e+00, -6.00866200e-01, -4.97444026e-01, -6.99774775e-01,
       -5.11886314e-01, -7.11217195e-01, -5.15572822e-01,  3.47661931e+00,
       -5.29643114e-01, -6.21233615e-01, -5.87066250e-01, -5.13429355e-01,
       -6.41914298e-01, -5.85583189e-01, -5.37508919e-01, -5.29643114e-01,
        3.46392382e+00, -6.55844947e-01, -6.26503371e-01, -5.88145759e-01,
       -6.31589436e-01,  3.48301145e+00, -5.90759006e-01, -5.04305552e-01,
       -5.41708329e-01, -6.17830210e-01, -4.22994183e-01, -6.54750153e-01,
       -5.01837825e-01,  3.48995838e+00, -5.51428151e-01, -7.41117861e-01,
       -6.65875885e-01, -4.56719038e-01, -6.11103058e-01,  3.56197484e+00,
       -3.52793808e-01, -4.57027746e-01, -8.70326189e-01, -1.48129118e-01,
       -5.38582200e-01, -7.95725976e-01, -5.90759006e-01,  3.54038125e+00,
       -3.52905893e-01, -6.38383739e-01, -6.24024435e-01, -5.69701492e-01,
       -5.41395365e-01, -6.66088214e-01,  3.56944251e+00, -7.19211327e-01,
       -7.34547671e-01, -3.38498443e-01, -5.77087984e-01, -6.14886400e-01,
        3.44874042e+00, -5.28463654e-01, -5.25364542e-01, -5.89106293e-01,
       -6.13831543e-01, -6.43172676e-01, -5.22190275e-01, -5.28463654e-01,
        3.47832029e+00, -5.60925400e-01, -6.01271766e-01, -6.22296521e-01,
       -5.80839221e-01, -6.12862258e-01, -5.25364542e-01,  3.46560718e+00,
       -5.30307319e-01, -6.14068942e-01, -6.02164897e-01, -7.75864791e-01,
       -4.33818190e-01, -5.60925400e-01,  3.58391466e+00, -6.14996415e-01,
       -5.15511699e-01, -6.82798164e-01, -7.30202056e-01, -5.86631307e-01,
       -6.14996415e-01,  3.68003291e+00, -4.68116513e-01, -5.35268159e-01,
       -7.44818461e-01, -4.69203673e-01, -6.91078890e-01,  3.48852229e+00,
       -5.57205862e-01, -6.00742959e-01, -6.45639766e-01, -5.24651136e-01,
       -6.14781495e-01, -5.45732232e-01, -5.57205862e-01,  3.48463843e+00,
       -5.38400400e-01, -5.59912881e-01, -6.68605560e-01, -6.63511356e-01,
       -4.84863269e-01, -5.38400400e-01,  3.48934598e+00, -5.71989361e-01,
       -5.14080291e-01, -7.16501305e-01, -6.07826393e-01, -6.33359338e-01,
       -5.29755620e-01,  3.57045678e+00, -5.20240824e-01, -3.97568410e-01,
       -8.81706192e-01, -5.68639178e-01, -5.56123280e-01, -5.30307319e-01,
        3.44527317e+00, -5.88554460e-01, -6.15154119e-01, -5.86494817e-01,
       -6.42168684e-01, -5.07715571e-01,  3.48951910e+00, -5.91829800e-01,
       -6.39563038e-01, -4.49582558e-01, -6.58659451e-01, -4.34797611e-01,
       -7.23941721e-01, -5.91829800e-01,  3.50309758e+00, -5.86134125e-01,
       -7.43435689e-01, -4.22958638e-01, -5.58576282e-01, -5.46083419e-01,
       -6.39563038e-01,  3.46966240e+00, -6.26060938e-01, -5.14631277e-01,
       -5.84747444e-01, -5.90718060e-01, -5.13968834e-01, -6.26060938e-01,
        3.46143350e+00, -5.67396306e-01, -5.36470261e-01, -6.26819107e-01,
       -6.67919369e-01, -4.71209325e-01, -5.67396306e-01,  3.47310182e+00,
       -5.85819881e-01, -5.20648872e-01, -6.60108064e-01, -4.17821050e-01,
       -5.97775776e-01, -7.10697436e-01, -5.86134125e-01,  3.49324861e+00,
       -4.07229281e-01, -7.73590942e-01, -6.87007698e-01, -4.47201629e-01,
       -5.85819881e-01,  3.48434062e+00, -6.43849400e-01, -4.91402746e-01,
       -6.29059264e-01, -6.48349239e-01, -4.10608273e-01, -6.43849400e-01,
        3.59984972e+00, -1.06925552e+00, -6.11163754e-01, -2.16623540e-01,
       -7.33368226e-01, -3.75398631e-01,  3.55364314e+00, -5.20749723e-01,
       -8.47188572e-01, -4.34698776e-01, -6.42239210e-01, -5.71940119e-01,
       -7.47020673e-01, -5.20749723e-01,  3.52135268e+00, -3.90779980e-01,
       -7.00199260e-01, -5.90662926e-01, -5.85511839e-01, -7.58137421e-01,
       -4.07229281e-01, -3.90779980e-01,  3.50016914e+00, -5.50001125e-01,
       -8.08509492e-01, -9.85912306e-01, -5.58025290e-01,  3.95973274e+00,
       -4.29388103e-01, -1.45762785e+00, -5.28779194e-01, -3.60551753e-01,
       -8.57978202e-01,  3.58773030e+00, -5.33834320e-01, -5.48502656e-01,
       -4.19311326e-01, -8.67552040e-01, -6.78216689e-01, -5.73862639e-01,
       -4.29388103e-01, -5.33834320e-01,  3.51593972e+00, -7.20393968e-01,
       -5.80244004e-01, -5.80664233e-01, -4.79808096e-01, -5.20240824e-01,
        3.49938632e+00, -5.53180500e-01, -4.02663441e-01, -5.61033693e-01,
       -4.01795531e-01, -7.48716365e-01, -5.22540565e-01, -5.53180500e-01,
        3.54347882e+00, -4.77261293e-01, -4.57133487e-01, -7.84646613e-01,
       -6.61293208e-01, -5.88151285e-01, -5.25374907e-01,  3.47492013e+00,
       -4.50445711e-01, -5.78080708e-01, -6.71574310e-01, -1.08619321e+00,
       -8.83468904e-01, -4.50445711e-01,  3.77572909e+00, -8.59117561e-01,
       -4.96503703e-01, -6.28066015e-01, -6.00866200e-01, -6.21233615e-01,
        3.54760764e+00, -8.53557964e-01, -4.16181580e-01, -4.27702264e-01,
       -5.11029372e-01, -5.04305552e-01,  3.50765384e+00, -7.13490922e-01,
       -8.47770104e-01, -6.72148013e-01, -2.58909874e-01, -5.16086735e-01,
       -5.81687176e-01, -7.13490922e-01,  3.43523084e+00, -6.55471197e-01,
       -5.31264468e-01, -4.37230345e-01, -7.34099353e-01, -5.51428151e-01,
       -3.52793808e-01,  3.58154115e+00, -5.36472151e-01, -8.47728989e-01,
       -5.59018694e-01, -6.07186698e-01, -6.38004838e-01, -4.57027746e-01,
        3.42572300e+00, -5.11863780e-01, -6.64555124e-01, -5.47084817e-01,
       -5.69447469e-01, -6.61347734e-01, -5.11863780e-01,  3.66003052e+00,
       -3.61914485e-01, -5.29979742e-01, -1.02547731e+00, -4.94884164e-01,
       -6.06214117e-01, -3.61914485e-01,  3.58574965e+00, -6.58545674e-01,
       -8.66918225e-01, -5.12248246e-01, -8.50247405e-02, -5.17430921e-01,
       -7.14851965e-01, -6.58545674e-01,  3.51178572e+00, -3.79244615e-01,
       -5.49934137e-01, -6.91778405e-01, -6.22493269e-01, -1.72360710e+00,
       -3.79244615e-01,  4.08707938e+00, -8.78785460e-01, -4.82948931e-01,
       -8.71293245e-02, -3.52905893e-01, -8.78785460e-01,  3.53246475e+00,
       -6.63276370e-01, -5.14780873e-01, -6.53671683e-01, -3.81915143e-01,
       -5.07754263e-01, -7.19211327e-01, -6.55471197e-01,  3.49706202e+00,
       -5.84274817e-01, -4.59589941e-01, -5.70760478e-01, -4.38524374e-01,
       -5.66355769e-01, -7.34547671e-01,  3.81232782e+00, -1.10299287e+00,
       -9.69907131e-01, -4.96874847e-01, -4.95885207e-01, -8.53557964e-01,
       -1.10299287e+00,  3.76594217e+00, -7.38302379e-01, -7.83289013e-02,
       -5.54544852e-01, -5.33395426e-01, -5.87066250e-01, -6.55844947e-01,
       -4.16181580e-01, -7.38302379e-01,  3.48533543e+00, -6.07041613e-01,
       -4.68116513e-01,  3.46850934e+00, -5.86472173e-01, -6.51615602e-01,
       -5.30837811e-01, -6.24425624e-01, -5.88553499e-01, -5.64944914e-01,
       -5.86472173e-01,  3.49227533e+00, -6.16927343e-01, -4.91215216e-01,
       -6.44162183e-01, -4.84200175e-01, -6.36313127e-01, -6.16927343e-01,
        3.51651615e+00, -7.33720984e-01, -5.07220523e-01, -5.38133994e-01,
       -4.24804257e-01, -6.00742959e-01, -7.33720984e-01,  3.50547641e+00,
       -6.91756768e-01, -5.66744305e-01, -4.87707138e-01, -7.39162265e-01,
       -5.71989361e-01, -3.97568410e-01,  3.51879934e+00, -4.37239717e-01,
       -6.49294311e-01, -7.23545281e-01, -5.76000961e-01, -4.97444026e-01,
       -5.88554460e-01, -5.84274817e-01,  3.47675703e+00, -6.28046651e-01,
       -6.02436118e-01, -3.61378161e-01, -1.06925552e+00, -5.48502656e-01,
        3.80036328e+00, -6.79494373e-01, -1.14173257e+00, -3.37054873e-02,
       -8.47188572e-01, -1.45762785e+00,  3.96891285e+00, -7.12483928e-01,
       -5.07134616e-01, -4.10772397e-01, -6.88057694e-01, -5.43706460e-01,
       -4.77261293e-01,  3.48001167e+00, -5.37648796e-01, -5.34355543e-01,
       -6.98981883e-01, -6.39430502e-01, -5.41688366e-01, -5.37648796e-01,
        3.47139477e+00, -5.95460459e-01, -5.20938057e-01, -6.36228590e-01,
       -6.48391249e-01, -5.58488198e-01, -5.95460459e-01,  3.51915526e+00,
       -4.71962328e-01, -4.25363985e-01, -8.19489044e-01, -6.17401696e-01,
       -5.95666025e-01, -4.71962328e-01,  3.46464357e+00, -5.32901813e-01,
       -6.70123199e-01, -5.76588510e-01, -3.04148621e-01, -7.41117861e-01,
       -8.59117561e-01, -5.36472151e-01,  3.55500763e+00, -5.56611567e-01,
       -5.57539872e-01, -6.01557933e-01, -5.78080708e-01, -5.32901813e-01,
        3.45231114e+00, -5.24150241e-01, -6.20965524e-01, -5.94654919e-01,
       -6.99774775e-01, -3.38498443e-01, -4.27702264e-01, -4.59589941e-01,
       -9.69907131e-01, -7.83289013e-02, -6.28046651e-01,  3.60184811e+00,
       -5.41708329e-01, -6.38383739e-01, -8.47770104e-01,  3.69259744e+00,
       -6.24267043e-01, -1.04046823e+00, -8.70326189e-01, -8.47728989e-01,
        3.81135583e+00, -5.16582860e-01, -1.12594529e+00, -4.50772494e-01,
       -6.24024435e-01, -6.63276370e-01, -6.24267043e-01,  3.53342580e+00,
       -4.16179266e-01, -4.89450351e-01, -7.16228334e-01, -5.89106293e-01,
       -6.01271766e-01,  3.47597834e+00, -5.21479842e-01, -5.50648786e-01,
       -6.49247155e-01, -5.64224494e-01, -6.13831543e-01, -6.14068942e-01,
       -5.21479842e-01,  3.47914928e+00, -5.01098185e-01, -5.62612174e-01,
       -6.66058592e-01, -6.22296521e-01, -5.15511699e-01, -5.50648786e-01,
        3.48563405e+00, -5.87614764e-01, -5.60231522e-01, -6.49330754e-01,
       -6.02164897e-01, -6.15154119e-01, -5.01098185e-01,  3.46784487e+00,
       -5.02222450e-01, -6.32756263e-01, -6.14448951e-01, -6.82798164e-01,
       -5.35268159e-01, -5.87614764e-01,  3.55815753e+00, -3.45226743e-01,
       -4.38383766e-01, -3.85410738e-01, -5.83455195e-01, -7.44818461e-01,
       -6.51615602e-01, -3.45226743e-01,  3.68454550e+00, -4.69695268e-01,
       -8.69705610e-01, -6.03483813e-01, -6.45639766e-01, -5.59912881e-01,
        3.46638691e+00, -5.06061388e-01, -5.14721860e-01, -6.10806864e-01,
       -6.29244153e-01, -5.24651136e-01, -6.91756768e-01, -5.06061388e-01,
        3.47726795e+00, -5.47684670e-01, -6.24225580e-01, -5.82888405e-01,
       -6.68605560e-01, -5.14080291e-01, -5.14721860e-01,  3.47835439e+00,
       -5.56963887e-01, -5.61307257e-01, -6.62675536e-01, -7.16501305e-01,
       -4.37239717e-01, -5.56963887e-01,  3.48512391e+00, -6.20302847e-01,
       -5.10415652e-01, -6.43700501e-01, -8.81706192e-01, -4.02663441e-01,
       -6.49294311e-01,  3.61311706e+00, -4.86607325e-01, -7.08663634e-01,
       -4.84182155e-01, -5.86494817e-01, -5.31264468e-01, -5.70760478e-01,
       -6.02436118e-01, -5.02222450e-01,  3.47920738e+00, -6.86029050e-01,
       -4.49582558e-01, -7.43435689e-01,  3.48735172e+00, -5.50647410e-01,
       -5.49040706e-01, -7.36462091e-01, -4.58183266e-01, -6.58659451e-01,
       -5.14631277e-01, -5.50647410e-01,  3.45893370e+00, -6.20252331e-01,
       -4.78977483e-01, -6.35765749e-01, -4.22958638e-01, -7.73590942e-01,
       -5.50001125e-01, -5.49040706e-01,  3.49713541e+00, -4.11516941e-01,
       -7.90027055e-01, -5.84747444e-01, -5.36470261e-01, -6.20252331e-01,
        3.45184245e+00, -5.97450377e-01, -5.11514472e-01, -6.01407563e-01,
       -6.26819107e-01, -5.20648872e-01, -5.97450377e-01,  3.46172304e+00,
       -5.53364742e-01, -5.12836318e-01, -6.50603624e-01, -6.60108064e-01,
       -4.91402746e-01, -5.53364742e-01,  3.47238602e+00, -6.11673531e-01,
       -5.45339774e-01, -6.10497166e-01, -6.29059264e-01, -6.11163754e-01,
       -6.11673531e-01,  3.50391683e+00, -4.99118072e-01, -5.79876001e-01,
       -5.73026207e-01, -2.16623540e-01, -6.79494373e-01, -4.99118072e-01,
        3.49467223e+00, -3.13440193e-01, -7.02976721e-01, -5.97934210e-01,
       -4.85085120e-01, -4.34698776e-01, -7.12483928e-01,  3.51578514e+00,
       -6.02478629e-01, -5.79548101e-01, -4.95528293e-01, -6.91047408e-01,
       -6.42239210e-01, -7.00199260e-01, -6.02478629e-01,  3.51083271e+00,
       -3.29534070e-01, -6.99763731e-01, -5.36617806e-01, -5.90662926e-01,
       -8.08509492e-01, -4.11516941e-01, -3.29534070e-01,  3.54503410e+00,
       -5.36745527e-01, -8.68065139e-01, -5.28779194e-01, -7.20393968e-01,
       -5.07134616e-01,  3.52853488e+00, -3.62354732e-01, -7.93570362e-01,
       -6.16302008e-01, -4.19311326e-01, -1.14173257e+00, -3.13440193e-01,
        3.71157748e+00, -3.68766117e-01, -7.33052692e-01, -7.35274581e-01,
       -8.67552040e-01, -5.80244004e-01, -3.62354732e-01, -3.68766117e-01,
        3.55851799e+00, -6.80344213e-01, -6.99256889e-01, -5.61033693e-01,
       -4.57133487e-01,  4.46140067e+00, -4.75720512e-01, -8.63861756e-01,
       -2.10365122e+00, -7.84646613e-01, -5.34355543e-01, -4.75720512e-01,
        3.59269649e+00, -4.95649689e-01, -5.26416926e-01, -7.75907204e-01,
       -6.71574310e-01, -4.96503703e-01, -5.56611567e-01, -5.24150241e-01,
        3.45554437e+00, -6.14052730e-01, -5.92651818e-01, -6.72148013e-01,
       -4.37230345e-01, -6.32756263e-01, -6.86029050e-01,  3.47784383e+00,
       -6.46621680e-01, -4.03058476e-01, -2.58909874e-01, -1.04046823e+00,
       -4.16179266e-01, -6.46621680e-01,  3.58359058e+00, -4.74300740e-01,
       -7.47110794e-01, -5.59018694e-01, -5.57539872e-01, -5.16582860e-01,
       -6.14052730e-01,  3.47539351e+00, -5.12374172e-01, -7.15825178e-01,
       -1.48129118e-01, -6.64555124e-01, -1.12594529e+00,  3.68440355e+00,
       -7.44396312e-01, -5.18869507e-01, -4.82508195e-01, -5.47084817e-01,
       -5.29979742e-01, -7.44396312e-01,  3.52156689e+00, -5.32643836e-01,
       -4.55290275e-01, -7.12171910e-01, -1.02547731e+00, -8.66918225e-01,
       -5.32643836e-01,  3.92275240e+00, -1.21593045e+00, -2.81782581e-01,
       -5.12248246e-01, -5.49934137e-01,  3.55657676e+00, -6.19612113e-01,
       -8.60792077e-01, -3.58059411e-01, -6.55930780e-01, -6.91778405e-01,
       -4.82948931e-01, -5.14780873e-01, -6.19612113e-01,  3.50273773e+00,
       -7.30971423e-01, -4.62645988e-01, -6.53671683e-01, -4.89450351e-01,
        3.47162204e+00, -6.26712041e-01, -5.44765488e-01, -5.02742532e-01,
       -6.54279941e-01, -5.30837811e-01, -4.69695268e-01,  3.52259398e+00,
       -6.23840589e-01, -7.05420986e-01, -5.95223467e-01, -5.97575860e-01,
       -6.24425624e-01, -4.91215216e-01, -6.23840589e-01,  3.46584407e+00,
       -5.59407532e-01, -4.63782682e-01, -7.03172422e-01, -6.44162183e-01,
       -5.07220523e-01, -5.59407532e-01,  3.46691614e+00, -5.82793535e-01,
       -4.97742865e-01, -6.75589506e-01, -5.38133994e-01, -5.66744305e-01,
       -5.82793535e-01,  3.47877739e+00, -6.99760229e-01, -5.11436651e-01,
       -5.79908678e-01, -4.87707138e-01, -5.47684670e-01, -6.99760229e-01,
        3.47546520e+00, -6.70164013e-01, -5.28462663e-01, -5.41686487e-01,
       -7.23545281e-01, -6.20302847e-01, -4.86607325e-01,  3.50620302e+00,
       -5.81107702e-01, -5.92735431e-01, -5.01904436e-01, -4.10772397e-01,
       -5.79548101e-01, -7.93570362e-01,  3.51035067e+00, -6.93026022e-01,
       -4.77188186e-01, -5.56245607e-01, -4.01795531e-01, -7.08663634e-01,
       -8.63861756e-01,  3.67552752e+00, -5.92315683e-01,  1.83127251e-02,
       -5.81780769e-01, -5.45422874e-01, -6.98981883e-01, -5.20938057e-01,
       -4.95649689e-01,  3.48155513e+00, -5.95857686e-01, -5.67090667e-01,
       -6.03037143e-01, -6.36228590e-01, -4.25363985e-01, -5.95857686e-01,
        3.49072846e+00, -9.28237343e-02, -5.40195281e-01, -7.99407912e-01,
       -4.00851268e-01, -8.19489044e-01, -6.70123199e-01, -9.28237343e-02,
        3.78715977e+00, -4.66225784e-01, -1.24711692e+00, -4.91381094e-01,
       -5.76588510e-01, -6.20965524e-01, -4.66225784e-01,  3.45586236e+00,
       -5.18824982e-01, -6.73390509e-01, -5.99867049e-01, -5.94654919e-01,
       -5.92651818e-01, -5.12374172e-01, -5.18824982e-01,  3.44604079e+00,
       -5.80307722e-01, -6.47227176e-01, -4.50772494e-01, -7.15825178e-01,
       -5.18869507e-01, -5.80307722e-01,  3.47964173e+00, -7.21694126e-01,
       -4.92172704e-01, -8.50247405e-02, -1.21593045e+00, -8.60792077e-01,
        3.87826512e+00, -7.28564895e-01, -6.65727639e-01, -3.22225312e-01,
       -3.81915143e-01, -7.30971423e-01, -6.26712041e-01,  3.47851658e+00,
       -5.80590593e-01, -6.33053457e-01, -5.25273919e-01, -7.16228334e-01,
       -4.74300740e-01, -5.44765488e-01,  3.49389037e+00, -6.84894734e-01,
       -5.39187965e-01, -5.34513106e-01, -6.49247155e-01, -5.62612174e-01,
       -6.84894734e-01,  3.49487189e+00, -5.67629926e-01, -4.93415724e-01,
       -5.37072181e-01, -5.64224494e-01, -5.60231522e-01, -5.67629926e-01,
        3.46544510e+00, -5.94337490e-01, -5.99182211e-01, -5.79839459e-01,
       -6.66058592e-01, -6.14448951e-01, -4.03058476e-01, -7.47110794e-01,
       -5.39187965e-01, -4.93415724e-01,  3.46328050e+00, -6.49330754e-01,
       -4.38383766e-01, -5.94337490e-01,  3.50999495e+00, -7.61538494e-01,
       -5.07305000e-01, -5.59099448e-01, -3.85410738e-01, -8.69705610e-01,
        3.60021298e+00, -4.18383047e-01, -6.95629286e-01, -5.32174713e-01,
       -6.98909582e-01, -6.03483813e-01, -7.05420986e-01, -4.18383047e-01,
        3.54200221e+00, -4.33437391e-01, -7.50373676e-01, -6.30903300e-01,
       -6.10806864e-01, -6.24225580e-01,  3.44876264e+00, -4.83705194e-01,
       -4.81513146e-01, -6.50481348e-01, -5.98030510e-01, -6.29244153e-01,
       -5.61307257e-01, -4.83705194e-01,  3.46176683e+00, -5.46932126e-01,
       -6.09136954e-01, -6.31441146e-01, -5.82888405e-01, -6.70164013e-01,
       -4.81513146e-01,  3.47069854e+00, -5.03289785e-01, -6.09722972e-01,
       -6.23120217e-01, -6.62675536e-01, -5.10415652e-01, -5.46932126e-01,
        3.47037344e+00, -5.86574485e-01, -5.59315738e-01, -6.04459902e-01,
       -6.43700501e-01, -5.81107702e-01, -5.86574485e-01,  3.47254624e+00,
       -5.33358309e-01, -6.59603833e-01, -4.68201412e-01, -4.84182155e-01,
       -5.92735431e-01, -5.92315683e-01,  3.50650303e+00, -6.32402512e-01,
       -6.69812495e-01, -5.35054752e-01, -7.36462091e-01, -4.78977483e-01,
        3.48642836e+00, -5.35029774e-01, -5.84514325e-01, -6.86085901e-01,
       -4.65358781e-01, -4.58183266e-01, -7.90027055e-01, -5.36745527e-01,
       -5.35029774e-01,  3.50403892e+00, -3.97982077e-01, -7.86071224e-01,
       -6.35765749e-01, -5.11514472e-01, -5.84514325e-01,  3.46993689e+00,
       -6.42750835e-01, -5.11969162e-01, -5.83422347e-01, -6.01407563e-01,
       -5.12836318e-01, -6.42750835e-01,  3.46966459e+00, -5.66292872e-01,
       -5.30096286e-01, -6.16280712e-01, -6.50603624e-01, -5.45339774e-01,
       -5.66292872e-01,  3.47335042e+00, -5.60809708e-01, -5.20779131e-01,
       -6.29525312e-01, -6.10497166e-01, -5.79876001e-01, -5.60809708e-01,
        3.48374297e+00, -5.45160113e-01, -5.73808432e-01, -6.13591552e-01,
       -5.73026207e-01, -7.02976721e-01, -5.45160113e-01,  3.44284913e+00,
       -5.40389826e-01, -5.39722066e-01, -5.41574196e-01, -5.97934210e-01,
       -7.33052692e-01,  3.52365244e+00, -3.58719909e-01, -5.06380176e-01,
       -5.81681443e-01, -7.45884006e-01, -4.95528293e-01, -6.93026022e-01,
        3.48296554e+00, -5.41131802e-01, -5.09107755e-01, -5.80048709e-01,
       -6.64122959e-01, -6.91047408e-01, -6.99763731e-01, -5.41131802e-01,
        3.51085571e+00, -3.51759573e-01, -7.09647776e-01, -5.17505424e-01,
       -5.36617806e-01, -8.68065139e-01, -3.97982077e-01, -3.51759573e-01,
        3.55925483e+00, -5.34233785e-01, -8.70596452e-01, -6.16302008e-01,
       -6.80344213e-01, -4.77188186e-01,  3.51140174e+00, -3.65299244e-01,
       -6.68970180e-01, -7.03297907e-01, -7.35274581e-01, -6.99256889e-01,
       -3.58719909e-01, -3.65299244e-01,  3.53450027e+00, -7.79765433e-01,
       -5.96184214e-01, -2.10365122e+00, -5.26416926e-01,  1.83127251e-02,
        4.45699693e+00, -3.25296340e-01, -8.37557869e-01, -6.82387295e-01,
       -7.75907204e-01, -5.67090667e-01, -3.25296340e-01,  3.49404221e+00,
       -5.58602053e-01, -6.11580100e-01, -6.55565841e-01, -4.82508195e-01,
       -4.55290275e-01, -7.21694126e-01,  3.44588067e+00, -6.96239738e-01,
       -5.55139527e-01, -5.35008809e-01, -7.12171910e-01, -2.81782581e-01,
       -7.28564895e-01, -6.96239738e-01,  3.52901562e+00, -6.67571190e-01,
       -4.42685309e-01, -3.58059411e-01, -6.65727639e-01,  3.52546002e+00,
       -6.52695654e-01, -7.14335437e-01, -7.31121210e-01, -4.03520669e-01,
       -6.55930780e-01, -4.62645988e-01, -5.80590593e-01, -6.52695654e-01,
        3.50332339e+00, -6.84240445e-01, -4.67219933e-01, -5.02742532e-01,
       -6.33053457e-01,  3.43114250e+00, -5.69848436e-01, -5.49093045e-01,
       -5.75434747e-01, -6.00970286e-01, -6.54279941e-01, -5.34513106e-01,
       -5.37072181e-01, -5.99182211e-01, -5.69848436e-01,  3.47317031e+00,
       -5.78274436e-01, -5.83455195e-01, -7.61538494e-01, -6.95629286e-01,
        3.58658909e+00, -4.71257539e-01, -5.40002738e-01, -5.34705839e-01,
       -5.95223467e-01, -4.63782682e-01,  3.45901330e+00, -6.05820405e-01,
       -5.50792608e-01, -5.76725050e-01, -6.66669090e-01, -5.97575860e-01,
       -4.33437391e-01, -6.05820405e-01,  3.49783002e+00, -6.70732318e-01,
       -5.03231501e-01, -6.87032541e-01, -7.03172422e-01, -4.97742865e-01,
       -5.50792608e-01,  3.52443741e+00, -5.26906519e-01, -5.00201514e-01,
       -7.45621483e-01, -6.75589506e-01, -5.11436651e-01, -5.26906519e-01,
        3.48866383e+00, -5.85510579e-01, -4.98915598e-01, -6.90304982e-01,
       -5.79908678e-01, -5.28462663e-01, -5.85510579e-01,  3.47248134e+00,
       -6.97317869e-01, -4.81249993e-01, -6.00031560e-01, -5.41686487e-01,
       -5.03289785e-01, -6.97317869e-01,  3.47604262e+00, -6.70879512e-01,
       -4.93470282e-01, -5.69398687e-01, -5.01904436e-01, -5.33358309e-01,
       -6.32402512e-01,  3.49732660e+00, -7.28124062e-01, -6.17114862e-01,
       -4.84422419e-01, -4.85085120e-01, -5.40389826e-01, -5.06380176e-01,
        3.13695369e+00, -5.09450132e-01, -5.84345674e-01, -5.11302760e-01,
       -5.56245607e-01, -5.09107755e-01, -6.68970180e-01,  3.47972402e+00,
       -6.31008770e-01, -4.65026990e-01, -6.49364718e-01, -5.81780769e-01,
       -6.69812495e-01,  3.56215310e+00, -5.77245323e-01, -6.57576919e-01,
       -5.55466381e-01, -5.20271210e-01, -5.45422874e-01, -8.37557869e-01,
       -5.77245323e-01,  3.63138111e+00, -6.84636292e-01, -3.82729901e-01,
       -6.03788852e-01, -6.03037143e-01, -5.40195281e-01, -5.58602053e-01,
        3.48695233e+00, -5.22119967e-01, -5.87332039e-01, -6.75665848e-01,
       -7.99407912e-01, -1.24711692e+00,  3.80088305e+00, -4.74601981e-01,
       -8.22905587e-01, -4.56850653e-01, -4.91381094e-01, -6.73390509e-01,
       -4.74601981e-01,  3.39630636e+00, -4.76962488e-01, -5.67050734e-01,
       -7.12919554e-01, -5.99867049e-01, -6.47227176e-01, -4.92172704e-01,
       -5.55139527e-01, -4.76962488e-01,  3.45759852e+00, -6.86229578e-01,
       -3.22225312e-01, -6.67571190e-01, -7.14335437e-01,  3.49779917e+00,
       -6.52811164e-01, -6.11301783e-01, -5.29554283e-01, -5.25273919e-01,
       -6.84240445e-01, -5.49093045e-01,  3.50493398e+00, -5.27956488e-01,
       -5.99338082e-01, -6.19032001e-01, -5.79839459e-01, -5.07305000e-01,
       -5.75434747e-01, -5.78274436e-01,  3.49776373e+00, -6.93510418e-01,
       -5.63399671e-01, -5.59099448e-01, -4.71257539e-01, -6.93510418e-01,
        3.49786523e+00, -4.91525683e-01, -7.35347282e-01, -5.47124860e-01,
       -5.32174713e-01, -7.50373676e-01, -6.57576919e-01,  3.53472712e+00,
       -5.01277286e-01, -4.43765748e-01, -6.49558780e-01, -6.98909582e-01,
       -5.40002738e-01, -5.55466381e-01, -6.84636292e-01, -5.01277286e-01,
        3.54499999e+00, -5.64707714e-01, -6.30903300e-01, -6.70732318e-01,
       -4.43765748e-01,  3.50211920e+00, -6.40477519e-01, -4.32206487e-01,
       -6.84033826e-01, -6.50481348e-01, -6.09722972e-01,  3.46582723e+00,
       -4.88907959e-01, -4.56902133e-01, -6.70113582e-01, -5.89699235e-01,
       -5.98030510e-01, -6.09136954e-01, -4.88907959e-01,  3.45993714e+00,
       -5.20896331e-01, -6.30829725e-01, -6.12135658e-01, -6.31441146e-01,
       -5.59315738e-01, -5.20896331e-01,  3.45709739e+00, -5.49830433e-01,
       -5.96273077e-01, -5.99340661e-01, -6.23120217e-01, -6.70879512e-01,
       -4.56902133e-01,  3.47796962e+00, -4.76106959e-01, -6.23477824e-01,
       -6.27482979e-01, -6.04459902e-01, -6.59603833e-01, -5.49830433e-01,
        3.48352222e+00, -4.98205655e-01, -7.18192127e-01, -4.53230268e-01,
       -4.68201412e-01, -7.28124062e-01, -4.98205655e-01,  3.50264090e+00,
       -5.65351566e-01, -8.25461193e-01, -4.17297009e-01, -5.35054752e-01,
       -6.17114862e-01, -5.20271210e-01, -6.49558780e-01, -6.40477519e-01,
        3.50317989e+00, -5.40702764e-01, -6.86085901e-01, -5.11969162e-01,
        3.47635994e+00, -5.53275196e-01, -6.10476314e-01, -4.90453699e-01,
       -6.24099671e-01, -4.65358781e-01, -7.86071224e-01, -5.34233785e-01,
       -5.53275196e-01,  3.51331373e+00, -4.20572742e-01, -7.53802002e-01,
       -5.83422347e-01, -5.30096286e-01, -6.10476314e-01,  3.46525260e+00,
       -6.21731471e-01, -5.66516060e-01, -5.53010120e-01, -6.16280712e-01,
       -5.20779131e-01, -6.21731471e-01,  3.47219789e+00, -5.65850286e-01,
       -5.64076320e-01, -5.83479967e-01, -6.29525312e-01, -5.73808432e-01,
       -5.65850286e-01,  3.47468154e+00, -5.63510693e-01, -5.52192222e-01,
       -5.89794594e-01, -6.13591552e-01, -5.39722066e-01, -5.63510693e-01,
        3.53176416e+00, -7.72176294e-01, -6.75451585e-01, -3.67311969e-01,
       -5.41574196e-01, -5.09450132e-01, -7.72176294e-01,  3.79334549e+00,
       -1.09008408e+00, -8.80060783e-01, -5.81681443e-01, -7.79765433e-01,
        3.51888343e+00, -4.27284254e-01, -4.16859534e-01, -7.74111549e-01,
       -5.39181214e-01, -7.45884006e-01, -5.84345674e-01, -4.27284254e-01,
        3.52312130e+00, -5.16423620e-01, -5.10950022e-01, -7.38233722e-01,
       -5.80048709e-01, -6.31008770e-01,  3.47382182e+00, -5.26915611e-01,
       -4.89219910e-01, -6.11749400e-01, -6.34879417e-01, -6.64122959e-01,
       -7.09647776e-01, -5.26915611e-01,  3.48316645e+00, -3.81241275e-01,
       -6.80209663e-01, -5.21029167e-01, -5.17505424e-01, -8.70596452e-01,
       -4.20572742e-01, -3.81241275e-01,  3.55816149e+00, -5.38722943e-01,
       -8.29522651e-01, -7.03297907e-01, -5.96184214e-01, -4.65026990e-01,
       -4.16859534e-01,  3.50224014e+00, -6.02988284e-01, -7.17883208e-01,
       -6.82387295e-01, -6.11580100e-01, -3.82729901e-01,  3.53468201e+00,
       -4.92363946e-01, -6.30583268e-01, -7.35037497e-01, -6.55565841e-01,
       -5.22119967e-01, -4.92363946e-01,  3.44927886e+00, -5.43298012e-01,
       -7.14759342e-01, -5.21171756e-01, -4.00851268e-01, -5.87332039e-01,
       -8.22905587e-01,  3.46399856e+00, -5.30650985e-01, -6.15931572e-01,
       -5.06327109e-01, -5.35008809e-01, -4.42685309e-01, -5.67050734e-01,
       -6.86229578e-01, -6.52811164e-01,  3.39932997e+00, -5.15544373e-01,
       -7.31121210e-01, -4.67219933e-01, -5.27956488e-01,  3.54331322e+00,
       -5.10724985e-01, -6.81796501e-01, -6.24494103e-01, -4.03520669e-01,
       -6.11301783e-01, -5.30650985e-01, -5.10724985e-01,  3.25955954e+00,
       -5.55003056e-01, -6.48358064e-01, -6.00970286e-01, -5.99338082e-01,
       -5.63399671e-01, -4.91525683e-01,  3.48738369e+00, -5.21668775e-01,
       -7.10481191e-01, -5.34705839e-01, -6.03788852e-01, -7.35347282e-01,
       -5.64707714e-01, -6.30583268e-01,  3.51787864e+00, -4.48745684e-01,
       -5.76725050e-01, -5.03231501e-01,  3.52859979e+00, -4.38582714e-01,
       -6.42048179e-01, -7.01601484e-01, -6.66410861e-01, -6.66669090e-01,
       -5.00201514e-01, -4.38582714e-01,  3.44193503e+00, -4.80906518e-01,
       -2.20749394e-01, -3.17072973e-01, -8.17752823e-01, -6.87032541e-01,
       -4.32206487e-01, -6.42048179e-01,  3.49887892e+00, -6.14618945e-01,
       -6.81186371e-01, -4.41786395e-01, -7.45621483e-01, -4.98915598e-01,
       -4.80906518e-01,  3.53424478e+00, -5.29215360e-01, -6.39335487e-01,
       -6.40250330e-01, -6.90304982e-01, -4.81249993e-01, -5.29215360e-01,
        3.48813771e+00, -6.16273321e-01, -4.89107228e-01, -6.81986829e-01,
       -6.00031560e-01, -4.93470282e-01, -6.16273321e-01,  3.46570566e+00,
       -6.95167069e-01, -4.52262093e-01, -6.08501338e-01, -5.69398687e-01,
       -4.76106959e-01, -6.95167069e-01,  3.47473890e+00, -6.83516691e-01,
       -4.70612258e-01, -5.79937241e-01, -4.84422419e-01, -6.84033826e-01,
       -5.65351566e-01, -5.40702764e-01, -6.14618945e-01,  3.50232449e+00,
       -6.13194967e-01, -5.11302760e-01, -1.09008408e+00, -5.16423620e-01,
        3.66985595e+00, -1.99307892e-01, -5.05247418e-01, -8.47490180e-01,
       -6.49364718e-01, -4.89219910e-01, -6.02988284e-01,  3.47314407e+00,
       -4.95935603e-01, -5.94430733e-01, -6.41204827e-01, -6.75665848e-01,
       -5.43298012e-01, -6.15931572e-01, -6.81796501e-01, -5.55003056e-01,
        3.55402626e+00, -4.82331276e-01, -4.56850653e-01, -7.12919554e-01,
       -5.29554283e-01, -5.06327109e-01, -5.15544373e-01, -6.48358064e-01,
        3.36955404e+00, -6.19032001e-01, -7.14759342e-01, -6.24494103e-01,
       -5.21668775e-01, -4.82331276e-01,  3.48591124e+00, -5.23625745e-01,
       -5.47124860e-01, -7.35037497e-01, -5.21171756e-01, -7.10481191e-01,
       -4.48745684e-01, -5.23625745e-01,  3.48618673e+00, -6.70113582e-01,
       -6.23477824e-01,  3.47912941e+00, -4.68342964e-01, -4.37601023e-01,
       -6.84945711e-01, -5.94648306e-01, -5.89699235e-01, -6.30829725e-01,
       -4.68342964e-01,  3.46755331e+00, -5.27440058e-01, -6.37196088e-01,
       -6.14045242e-01, -6.12135658e-01, -5.96273077e-01, -5.27440058e-01,
        3.45916314e+00, -5.25478432e-01, -5.88072825e-01, -6.09763094e-01,
       -5.99340661e-01, -7.18192127e-01, -5.25478432e-01,  3.49041054e+00,
       -4.35057786e-01, -7.02834073e-01, -5.09507456e-01, -6.27482979e-01,
       -6.83516691e-01, -4.37601023e-01,  3.47476798e+00, -4.73910922e-01,
       -6.08271371e-01, -6.43984998e-01, -4.53230268e-01, -8.25461193e-01,
       -4.35057786e-01,  3.53620442e+00, -4.92078055e-01, -8.80957155e-01,
       -4.49419966e-01, -4.17297009e-01, -6.81186371e-01, -6.13194967e-01,
       -4.92078055e-01,  3.47294756e+00, -6.18189664e-01, -6.51001497e-01,
       -4.90453699e-01, -7.53802002e-01, -5.38722943e-01,  3.50196348e+00,
       -5.70275021e-01, -4.37803528e-01, -7.10906291e-01, -6.24099671e-01,
       -5.66516060e-01, -5.70275021e-01,  3.46861753e+00, -6.13451769e-01,
       -5.20174875e-01, -5.74100130e-01, -5.53010120e-01, -5.64076320e-01,
       -6.13451769e-01,  3.47351111e+00, -6.19781658e-01, -5.91554361e-01,
       -5.31636886e-01, -5.83479967e-01, -5.52192222e-01, -6.19781658e-01,
        3.48063350e+00, -5.70597245e-01, -5.95111824e-01, -5.59470584e-01,
       -5.89794594e-01, -6.75451585e-01, -5.70597245e-01,  3.48799170e+00,
       -4.73182652e-01, -5.92217565e-01, -5.86748056e-01, -3.67311969e-01,
       -8.80060783e-01, -1.99307892e-01, -4.73182652e-01,  3.54741658e+00,
       -3.25399433e-01, -6.23557017e-01, -6.78596837e-01, -7.74111549e-01,
       -5.10950022e-01,  3.50328130e+00, -4.70206775e-01, -4.97123299e-01,
       -5.39550335e-01, -7.11339316e-01, -5.39181214e-01, -7.17883208e-01,
       -4.95935603e-01, -4.70206775e-01,  3.49461180e+00, -5.86396087e-01,
       -6.85008914e-01, -7.38233722e-01, -5.05247418e-01, -4.97123299e-01,
        3.47513492e+00, -5.27481070e-01, -5.09372253e-01, -6.97677159e-01,
       -6.11749400e-01, -6.80209663e-01,  3.43668754e+00, -5.28125226e-01,
       -4.29299476e-01, -6.44800596e-01, -5.42503179e-01, -6.34879417e-01,
       -5.94430733e-01, -5.28125226e-01,  3.47088193e+00, -5.09372498e-01,
       -5.79210780e-01, -6.24863279e-01, -5.21029167e-01, -8.29522651e-01,
       -4.37803528e-01, -4.29299476e-01,  3.52699715e+00, -5.54745611e-01,
       -7.54596723e-01, -7.01601484e-01, -4.41786395e-01, -6.18189664e-01,
        3.49908185e+00, -5.84042684e-01, -6.45156075e-01, -5.08305551e-01,
       -6.66410861e-01, -2.20749394e-01, -5.84042684e-01,  3.70515021e+00,
       -1.13608882e+00, -4.42016262e-01, -6.55842191e-01, -3.17072973e-01,
       -6.39335487e-01,  3.61674528e+00, -5.60169693e-01, -1.10773522e+00,
       -4.45445970e-01, -5.46985937e-01, -6.40250330e-01, -4.89107228e-01,
       -5.60169693e-01,  3.49140504e+00, -5.89107098e-01, -6.50957894e-01,
       -5.61812799e-01, -6.81986829e-01, -4.52262093e-01, -5.89107098e-01,
        3.48246360e+00, -6.47283504e-01, -4.35921654e-01, -6.75902424e-01,
       -6.08501338e-01, -4.70612258e-01, -6.47283504e-01,  3.48656209e+00,
       -6.90996804e-01, -4.62204983e-01, -6.06963205e-01, -5.79937241e-01,
       -4.73910922e-01, -6.90996804e-01,  3.48027251e+00, -6.61871250e-01,
       -4.73428815e-01, -6.00127482e-01, -8.47490180e-01, -3.25399433e-01,
       -5.27481070e-01,  3.37187216e+00, -5.68585217e-01, -4.77110145e-01,
       -6.25806111e-01, -6.41204827e-01, -5.86396087e-01, -5.09372498e-01,
        3.47019061e+00, -5.20276636e-01, -6.00169847e-01, -6.12770716e-01,
       -8.17752823e-01, -1.13608882e+00, -1.10773522e+00,  3.88989480e+00,
       -3.58839232e-01, -4.69478705e-01, -6.84945711e-01, -6.08271371e-01,
        3.47818120e+00, -4.65944504e-01, -4.62276700e-01, -6.85670920e-01,
       -5.71071990e-01, -5.94648306e-01, -6.37196088e-01, -4.65944504e-01,
        3.47107697e+00, -5.20917093e-01, -6.49349088e-01, -6.03021889e-01,
       -6.14045242e-01, -5.88072825e-01, -5.20917093e-01,  3.46339993e+00,
       -5.42379188e-01, -6.02529325e-01, -5.95456259e-01, -6.09763094e-01,
       -7.02834073e-01, -5.42379188e-01,  3.49491367e+00, -4.35902683e-01,
       -6.89428076e-01, -5.14606557e-01, -5.09507456e-01, -8.80957155e-01,
       -4.35902683e-01,  3.58528300e+00, -3.70377545e-01, -8.39723077e-01,
       -5.48815082e-01, -6.43984998e-01, -6.61871250e-01, -4.62276700e-01,
        3.47631486e+00, -4.99542151e-01, -5.87605160e-01, -6.21034596e-01,
       -4.49419966e-01, -6.51001497e-01, -6.45156075e-01, -3.70377545e-01,
        3.62378157e+00, -7.84675086e-01, -7.23151397e-01, -7.10906291e-01,
       -5.20174875e-01, -5.54745611e-01,  3.48861098e+00, -5.95937404e-01,
       -4.73116433e-01, -6.33730370e-01, -5.74100130e-01, -5.91554361e-01,
       -5.95937404e-01,  3.47635881e+00, -6.32808496e-01, -5.57058359e-01,
       -5.24900064e-01, -5.31636886e-01, -5.95111824e-01, -6.32808496e-01,
        3.48166448e+00, -6.25418858e-01, -5.91751741e-01, -5.04936672e-01,
       -5.59470584e-01, -5.92217565e-01, -6.25418858e-01,  3.48300371e+00,
       -5.83374277e-01, -5.72904168e-01, -5.49618258e-01, -5.86748056e-01,
       -6.23557017e-01, -5.83374277e-01,  3.48339979e+00, -5.62888722e-01,
       -5.35642008e-01, -5.91189711e-01, -6.78596837e-01, -5.68585217e-01,
       -5.62888722e-01,  3.75366456e+00, -1.03357364e+00, -8.26795170e-01,
       -8.32249765e-02, -5.39550335e-01, -6.85008914e-01, -5.20276636e-01,
        3.48375335e+00, -5.26493379e-01, -5.95568341e-01, -6.16855748e-01,
       -7.11339316e-01, -5.09372253e-01, -5.26493379e-01,  3.49586373e+00,
       -5.50329456e-01, -5.35331323e-01, -6.62998001e-01, -6.97677159e-01,
       -4.77110145e-01, -5.50329456e-01,  3.44010356e+00, -6.46337788e-01,
       -4.37806509e-01, -6.30842508e-01, -6.44800596e-01, -5.79210780e-01,
        3.41121855e+00, -4.89017156e-01, -5.65902252e-01, -5.63918322e-01,
       -5.68369449e-01, -5.42503179e-01, -7.54596723e-01, -4.73116433e-01,
       -4.89017156e-01,  3.50061420e+00, -5.72562393e-01, -6.68818317e-01,
       -6.24863279e-01, -6.00169847e-01, -5.65902252e-01,  3.47103928e+00,
       -5.35603639e-01, -5.52477276e-01, -5.92022990e-01, -5.08305551e-01,
       -4.42016262e-01, -7.84675086e-01,  3.52898113e+00, -7.30152954e-01,
       -4.18887154e-01, -6.44944121e-01, -6.55842191e-01, -3.58839232e-01,
       -7.30152954e-01,  3.48926474e+00, -7.65693524e-01, -3.97414556e-01,
       -5.81322282e-01, -4.45445970e-01, -6.50957894e-01,  3.52664933e+00,
       -6.79120207e-01, -6.10647735e-01, -6.29779748e-01, -5.10697779e-01,
       -5.46985937e-01, -4.69478705e-01, -7.65693524e-01, -6.79120207e-01,
        3.51754708e+00, -5.86205051e-01, -4.70063656e-01, -5.61812799e-01,
       -4.35921654e-01, -6.10647735e-01,  3.36383465e+00, -5.82507731e-01,
       -6.12916765e-01, -5.60027962e-01, -6.75902424e-01, -4.62204983e-01,
       -5.82507731e-01,  3.51472023e+00, -6.38453978e-01, -5.12490843e-01,
       -6.43160268e-01, -6.06963205e-01, -4.73428815e-01, -6.38453978e-01,
        3.49376125e+00, -6.50474641e-01, -5.07179448e-01, -6.17261162e-01,
       -6.00127482e-01, -4.99542151e-01, -6.50474641e-01,  3.47894987e+00,
       -6.30701520e-01, -5.19848751e-01, -5.78255329e-01, -6.25806111e-01,
       -1.03357364e+00, -6.46337788e-01,  3.44208103e+00, -8.45456424e-01,
       -2.90907073e-01, -6.12770716e-01, -5.95568341e-01, -5.35603639e-01,
        3.47550083e+00, -5.77087443e-01, -6.26001585e-01, -5.28469105e-01,
       -6.85670920e-01, -5.87605160e-01,  3.47733404e+00, -4.87345898e-01,
       -5.08589970e-01, -5.71232860e-01, -6.36889231e-01, -5.71071990e-01,
       -6.49349088e-01, -4.87345898e-01,  3.46676334e+00, -5.26170044e-01,
       -6.38665915e-01, -5.94160409e-01, -6.03021889e-01, -6.02529325e-01,
       -5.26170044e-01,  3.45330767e+00, -5.38541733e-01, -5.96115971e-01,
       -5.86928707e-01, -5.95456259e-01, -6.89428076e-01, -5.38541733e-01,
        3.47629158e+00, -4.64478046e-01, -6.92933825e-01, -4.95453645e-01,
       -5.14606557e-01, -8.39723077e-01, -4.64478046e-01,  3.53023647e+00,
       -3.88694491e-01, -8.31608511e-01, -4.91125790e-01, -5.48815082e-01,
       -7.23151397e-01, -4.18887154e-01, -3.88694491e-01,  3.51647235e+00,
       -6.34823387e-01, -8.02100840e-01, -6.21034596e-01, -6.30701520e-01,
       -5.08589970e-01,  3.47728387e+00, -5.64780211e-01, -5.94275805e-01,
       -5.57901767e-01, -6.33730370e-01, -5.57058359e-01, -5.72562393e-01,
        3.48377493e+00, -6.30717735e-01, -5.48890567e-01, -5.40815504e-01,
       -5.24900064e-01, -5.91751741e-01, -6.30717735e-01,  3.49306785e+00,
       -6.79545385e-01, -6.27477539e-01, -4.38675388e-01, -5.04936672e-01,
       -5.72904168e-01, -6.79545385e-01,  3.50079251e+00, -6.71096212e-01,
       -6.13607849e-01, -4.58702221e-01, -5.49618258e-01, -5.35642008e-01,
       -6.71096212e-01,  3.50136331e+00, -6.71330580e-01, -5.34046428e-01,
       -5.39629818e-01, -5.91189711e-01, -8.26795170e-01, -6.71330580e-01,
        3.65960522e+00, -1.92494211e-01, -4.96781743e-01, -8.81013808e-01,
       -8.32249765e-02, -8.45456424e-01, -1.92494211e-01,  3.56683460e+00,
       -9.82956218e-01, -1.00716087e+00, -4.55541899e-01, -6.16855748e-01,
       -5.35331323e-01, -5.77087443e-01,  3.48391910e+00, -6.31769916e-01,
       -5.98057205e-01, -5.24817461e-01, -6.62998001e-01, -4.37806509e-01,
       -6.31769916e-01,  3.53694059e+00, -8.10435094e-01, -5.65610827e-01,
       -4.28320241e-01, -6.30842508e-01, -2.90907073e-01, -9.82956218e-01,
       -8.10435094e-01,  3.63050161e+00, -2.30612277e-01, -6.84748440e-01,
       -5.63918322e-01, -5.52477276e-01,  3.24780100e+00, -5.56580639e-01,
       -6.39831271e-01, -3.25530759e-01, -6.09462736e-01, -5.68369449e-01,
       -6.68818317e-01, -5.48890567e-01, -5.56580639e-01,  3.49309197e+00,
       -5.81609744e-01, -5.68823258e-01, -5.92022990e-01, -6.26001585e-01,
       -6.39831271e-01,  3.49764545e+00, -5.95813121e-01, -5.56159032e-01,
       -4.87817453e-01, -6.44944121e-01, -3.97414556e-01, -6.34823387e-01,
        3.48433439e+00, -6.26325182e-01, -4.36929253e-01, -7.43897886e-01,
       -5.81322282e-01, -5.86205051e-01, -6.26325182e-01,  3.51524632e+00,
       -6.63162436e-01, -4.18731325e-01, -6.39500045e-01, -6.29779748e-01,
       -4.70063656e-01, -6.63162436e-01,  3.48188384e+00, -6.20688838e-01,
       -5.24429367e-01, -5.73759797e-01, -5.10697779e-01, -6.12916765e-01,
       -6.20688838e-01,  3.47748604e+00, -5.93091694e-01, -5.44625750e-01,
       -5.95465212e-01, -5.60027962e-01, -5.12490843e-01, -5.93091694e-01,
        3.27841298e+00, -5.41218261e-01, -5.24356330e-01, -5.47227893e-01,
       -6.43160268e-01, -5.07179448e-01, -5.41218261e-01,  3.42548821e+00,
       -6.06545638e-01, -5.68038844e-01, -5.59345749e-01, -6.17261162e-01,
       -5.19848751e-01, -6.06545638e-01,  3.47695108e+00, -5.97107686e-01,
       -5.57839723e-01, -5.78348124e-01, -5.78255329e-01, -5.64780211e-01,
       -5.97107686e-01,  3.47814472e+00, -5.88587820e-01, -5.94585119e-01,
       -5.54828555e-01, -5.28469105e-01, -5.98057205e-01, -5.95813121e-01,
        3.48899767e+00, -6.69266146e-01, -6.52157673e-01, -4.45234422e-01,
       -5.71232860e-01, -6.38665915e-01,  3.47755205e+00, -5.14180591e-01,
       -5.45488991e-01, -6.20348473e-01, -5.87635220e-01, -6.36889231e-01,
       -5.94275805e-01, -5.14180591e-01,  3.48212822e+00, -5.89575778e-01,
       -5.85812326e-01, -5.61394490e-01, -5.94160409e-01, -5.96115971e-01,
       -5.45488991e-01,  3.46341831e+00, -5.62134167e-01, -5.93181436e-01,
       -5.72337331e-01, -5.86928707e-01, -6.92933825e-01, -5.62134167e-01,
        3.47281019e+00, -4.80741852e-01, -6.63236896e-01, -4.86834746e-01,
       -4.95453645e-01, -8.31608511e-01, -4.80741852e-01,  3.53784469e+00,
       -4.35264722e-01, -8.47542837e-01, -4.47233124e-01, -4.91125790e-01,
       -8.02100840e-01, -4.36929253e-01, -4.35264722e-01,  3.53211514e+00,
       -5.65966280e-01, -8.00728255e-01, -5.57901767e-01, -5.88587820e-01,
       -5.89575778e-01,  3.48256572e+00, -6.28114133e-01, -6.25989475e-01,
       -4.92396748e-01, -5.40815504e-01, -6.27477539e-01, -5.81609744e-01,
        3.52053016e+00, -6.85300321e-01, -7.07925148e-01, -3.77401904e-01,
       -4.38675388e-01, -6.13607849e-01, -6.85300321e-01,  3.54407132e+00,
       -7.61853839e-01, -6.87220307e-01, -3.57413615e-01, -4.58702221e-01,
       -5.34046428e-01, -7.61853839e-01,  3.52014367e+00, -7.24430658e-01,
       -5.06049077e-01, -5.35061442e-01, -5.39629818e-01, -4.96781743e-01,
       -7.24430658e-01,  3.48725270e+00, -6.58565828e-01, -3.98530180e-01,
       -6.69314479e-01, -8.81013808e-01, -1.00716087e+00, -6.58565828e-01,
        3.75217706e+00, -7.83513678e-01, -4.21922878e-01, -4.55541899e-01,
       -2.30612277e-01, -7.83513678e-01,  3.24911772e+00, -6.45816720e-01,
       -5.95745687e-01, -5.37887465e-01, -5.24817461e-01, -5.65610827e-01,
       -6.69266146e-01,  3.49563602e+00, -6.61106484e-01, -5.88194878e-01,
       -4.86640222e-01, -4.28320241e-01, -6.84748440e-01, -6.45816720e-01,
       -6.61106484e-01,  3.53145203e+00, -3.56170803e-01, -7.55289347e-01,
       -3.25530759e-01, -5.56159032e-01,  3.49030560e+00, -8.79210361e-01,
       -1.14267115e+00, -4.18748324e-01, -1.67985970e-01, -6.09462736e-01,
       -5.68823258e-01, -7.07925148e-01, -8.79210361e-01,  3.56103837e+00,
       -7.95616867e-01, -4.87817453e-01, -6.52157673e-01, -1.14267115e+00,
        3.79135062e+00, -6.80945099e-01, -8.27759248e-01, -7.43897886e-01,
       -4.18731325e-01, -5.65966280e-01,  3.51037344e+00, -5.76854065e-01,
       -4.75911964e-01, -7.29011917e-01, -6.39500045e-01, -5.24429367e-01,
       -5.76854065e-01,  3.50104049e+00, -6.24331768e-01, -4.34169035e-01,
       -7.01756215e-01, -5.73759797e-01, -5.44625750e-01, -6.24331768e-01,
        3.47662772e+00, -6.20147631e-01, -4.82982706e-01, -6.30780070e-01,
       -5.95465212e-01, -5.24356330e-01, -6.20147631e-01,  3.47949858e+00,
       -6.21213736e-01, -4.88707651e-01, -6.29608015e-01, -5.47227893e-01,
       -5.68038844e-01, -6.21213736e-01,  3.37552627e+00, -5.90042654e-01,
       -4.99813710e-01, -5.49189429e-01, -5.59345749e-01, -5.57839723e-01,
       -5.90042654e-01,  3.41079143e+00, -5.93408585e-01, -5.92508022e-01,
       -5.17646700e-01, -5.78348124e-01, -5.94585119e-01, -5.93408585e-01,
        3.47223574e+00, -5.55997748e-01, -5.87147292e-01, -5.62748866e-01,
       -5.54828555e-01, -6.28114133e-01, -5.55997748e-01,  3.47611353e+00,
       -5.92238110e-01, -6.37256095e-01, -5.07678891e-01, -4.45234422e-01,
       -5.88194878e-01, -6.80945099e-01,  3.50322970e+00, -6.83455528e-01,
       -5.62954447e-01, -5.42445326e-01, -6.20348473e-01, -5.85812326e-01,
        3.48254308e+00, -5.23470235e-01, -5.47329427e-01, -5.85789905e-01,
       -6.19792712e-01, -5.87635220e-01, -5.93181436e-01, -5.23470235e-01,
        3.47681842e+00, -5.74101368e-01, -5.72036676e-01, -6.26393484e-01,
       -5.61394490e-01, -6.25989475e-01, -5.47329427e-01,  3.48775328e+00,
       -6.49150098e-01, -5.98033944e-01, -5.05855846e-01, -5.72337331e-01,
       -6.63236896e-01, -5.74101368e-01,  3.47956249e+00, -5.03969623e-01,
       -6.01285107e-01, -5.64632170e-01, -4.86834746e-01, -8.47542837e-01,
       -5.03969623e-01,  3.54212050e+00, -4.34072389e-01, -8.18252329e-01,
       -4.51448576e-01, -4.47233124e-01, -8.00728255e-01, -4.75911964e-01,
       -4.34072389e-01,  3.52596646e+00, -5.92743143e-01, -7.75277585e-01,
       -4.92396748e-01, -5.92238110e-01, -6.49150098e-01,  3.49908923e+00,
       -6.89232083e-01, -6.53226972e-01, -4.22845215e-01, -3.77401904e-01,
       -6.87220307e-01, -4.18748324e-01, -7.95616867e-01,  3.61449476e+00,
       -8.62173164e-01, -4.73334197e-01, -3.57413615e-01, -5.06049077e-01,
       -8.62173164e-01,  3.52908540e+00, -8.29138500e-01, -4.81614222e-01,
       -4.92696820e-01, -5.35061442e-01, -3.98530180e-01, -8.29138500e-01,
        3.49234612e+00, -7.70060937e-01, -3.77505133e-01, -5.82049930e-01,
       -6.69314479e-01, -4.21922878e-01, -5.95745687e-01, -7.70060937e-01,
        3.72652657e+00, -1.17497236e+00, -9.45102296e-02, -5.37887465e-01,
       -3.56170803e-01, -1.17497236e+00,  3.39443012e+00, -3.86268067e-01,
       -9.39131431e-01, -4.86640222e-01, -7.55289347e-01, -6.83455528e-01,
       -3.86268067e-01,  3.51330246e+00, -4.72654026e-01, -7.28995273e-01,
       -1.67985970e-01, -8.27759248e-01, -5.62954447e-01, -4.73334197e-01,
       -4.81614222e-01,  3.08303924e+00, -5.69391154e-01, -7.29011917e-01,
       -4.34169035e-01, -5.92743143e-01,  3.50768362e+00, -5.69564279e-01,
       -4.57902359e-01, -7.24292883e-01, -7.01756215e-01, -4.82982706e-01,
       -5.69564279e-01,  3.51218600e+00, -6.25643695e-01, -4.11685687e-01,
       -7.20553414e-01, -6.30780070e-01, -4.88707651e-01, -6.25643695e-01,
        3.48819034e+00, -6.34239128e-01, -4.47090154e-01, -6.61729644e-01,
       -6.29608015e-01, -4.99813710e-01, -6.34239128e-01,  3.48429252e+00,
       -6.05006617e-01, -4.60278704e-01, -6.55346345e-01, -5.49189429e-01,
       -5.92508022e-01, -6.05006617e-01,  3.42805653e+00, -6.07989670e-01,
       -5.19554912e-01, -5.53807883e-01, -5.17646700e-01, -5.87147292e-01,
       -6.07989670e-01,  3.30305003e+00, -5.28717357e-01, -6.09727897e-01,
       -4.51821110e-01, -5.62748866e-01, -6.37256095e-01, -5.28717357e-01,
        3.62358352e+00, -6.05692727e-01, -1.04101961e+00, -2.48148866e-01,
       -5.07678891e-01, -6.89232083e-01, -6.05692727e-01,  3.49979311e+00,
       -6.48945165e-01, -6.52296811e-01, -3.95947429e-01, -5.42445326e-01,
       -4.92696820e-01, -3.77505133e-01, -4.72654026e-01, -5.69391154e-01,
        3.20379045e+00, -7.49097986e-01, -5.85789905e-01, -5.98033944e-01,
        3.48690813e+00, -5.34588461e-01, -5.72657132e-01, -5.49100594e-01,
       -6.46738091e-01, -6.19792712e-01, -5.72036676e-01, -5.34588461e-01,
        3.48870629e+00, -5.35618164e-01, -5.36629255e-01, -6.90041020e-01,
       -6.26393484e-01, -6.01285107e-01, -5.35618164e-01,  3.49903279e+00,
       -5.18332193e-01, -5.42407461e-01, -6.74996385e-01, -5.05855846e-01,
       -6.53226972e-01, -5.72657132e-01,  3.50066767e+00, -6.73089600e-01,
       -6.21280617e-01, -4.74557504e-01, -5.64632170e-01, -8.18252329e-01,
       -5.18332193e-01,  3.58003075e+00, -4.06232641e-01, -7.38688953e-01,
       -5.33892464e-01, -4.51448576e-01, -7.75277585e-01, -4.57902359e-01,
       -4.06232641e-01,  3.48748117e+00, -6.33122052e-01, -7.63497960e-01,
       -4.22845215e-01, -6.48945165e-01, -6.73089600e-01,  3.50846774e+00,
       -6.61023137e-01, -6.88317780e-01, -4.14246842e-01, -5.82049930e-01,
       -9.45102296e-02, -9.39131431e-01, -7.28995273e-01, -7.49097986e-01,
        3.09378485e+00, -7.24292883e-01, -4.11685687e-01, -6.33122052e-01,
        3.52490288e+00, -5.91747380e-01, -3.96927783e-01, -7.67127094e-01,
       -7.20553414e-01, -4.47090154e-01, -5.91747380e-01,  3.51543433e+00,
       -6.38047260e-01, -3.93238430e-01, -7.24757690e-01, -6.61729644e-01,
       -4.60278704e-01, -6.38047260e-01,  3.49680384e+00, -6.33301696e-01,
       -4.20326769e-01, -6.83119767e-01, -6.55346345e-01, -5.19554912e-01,
       -6.33301696e-01,  3.48767218e+00, -5.78176381e-01, -4.34902129e-01,
       -6.66390717e-01, -5.53807883e-01, -6.09727897e-01, -5.78176381e-01,
        3.40125232e+00, -6.29583850e-01, -5.20034752e-01, -5.09921553e-01,
       -4.51821110e-01, -1.04101961e+00, -6.29583850e-01,  3.65300222e+00,
       -8.82571091e-01, -6.48006561e-01, -2.48148866e-01, -6.52296811e-01,
       -8.82571091e-01,  3.50963252e+00, -7.24073737e-01, -4.96971137e-01,
       -5.05570877e-01, -3.95947429e-01, -6.61023137e-01, -7.24073737e-01,
        3.57494622e+00, -7.30400129e-01, -5.35057061e-01, -5.28444733e-01,
       -5.49100594e-01, -6.21280617e-01,  3.48808093e+00, -5.50852798e-01,
       -5.75026373e-01, -6.42223775e-01, -5.49596769e-01, -6.46738091e-01,
       -5.36629255e-01, -5.50852798e-01,  3.50064248e+00, -5.30110558e-01,
       -5.05868669e-01, -7.30443109e-01, -6.90041020e-01, -5.42407461e-01,
       -5.30110558e-01,  3.51872330e+00, -4.93153340e-01, -4.75706159e-01,
       -7.87304767e-01, -6.74996385e-01, -7.38688953e-01, -4.93153340e-01,
        3.53445677e+00, -3.51227052e-01, -6.75038279e-01, -6.01352758e-01,
       -4.74557504e-01, -6.88317780e-01, -5.75026373e-01,  3.49105301e+00,
       -5.65965524e-01, -6.74200894e-01, -5.12984934e-01, -5.33892464e-01,
       -7.63497960e-01, -3.96927783e-01, -3.51227052e-01,  3.45816058e+00,
       -6.22225856e-01, -7.90389464e-01, -4.14246842e-01, -7.30400129e-01,
       -5.65965524e-01,  3.50139695e+00, -4.99534949e-01, -7.99821738e-01,
       -4.91427765e-01, -7.67127094e-01, -3.93238430e-01, -6.22225856e-01,
        3.53726384e+00, -6.04588785e-01, -3.72086895e-01, -7.77996776e-01,
       -7.24757690e-01, -4.20326769e-01, -6.04588785e-01,  3.51572747e+00,
       -6.43200242e-01, -4.22900182e-01, -6.99953807e-01, -6.83119767e-01,
       -4.34902129e-01, -6.43200242e-01,  3.50184239e+00, -6.29931629e-01,
       -4.50135377e-01, -6.60553248e-01, -6.66390717e-01, -5.20034752e-01,
       -6.29931629e-01,  3.49122451e+00, -5.73326464e-01, -4.46520106e-01,
       -6.55020845e-01, -5.09921553e-01, -6.48006561e-01, -4.96971137e-01,
       -5.73326464e-01,  3.35323542e+00, -6.14172708e-01, -5.10837001e-01,
       -5.05570877e-01, -5.35057061e-01, -6.14172708e-01,  3.35043886e+00,
       -6.58649662e-01, -5.99084894e-01, -4.37903658e-01, -5.28444733e-01,
       -4.99534949e-01, -6.58649662e-01,  3.13843102e+00, -7.12950141e-01,
       -7.38851534e-01, -6.42223775e-01, -5.05868669e-01,  3.51843126e+00,
       -5.76154095e-01, -5.60133356e-01, -4.99904689e-01, -7.34146676e-01,
       -5.49596769e-01, -6.74200894e-01, -5.76154095e-01,  3.49152680e+00,
       -4.98284927e-01, -4.95527277e-01, -6.97762838e-01, -7.30443109e-01,
       -4.75706159e-01, -5.60133356e-01,  3.50443197e+00, -4.99491451e-01,
       -4.16462806e-01, -8.22195094e-01, -7.87304767e-01, -6.75038279e-01,
       -4.99491451e-01,  3.52923238e+00, -3.06127094e-01, -6.04267048e-01,
       -6.57003743e-01, -6.01352758e-01, -7.90389464e-01, -3.72086895e-01,
       -3.06127094e-01,  3.46463438e+00, -5.86524781e-01, -8.08153388e-01,
       -5.12984934e-01, -7.99821738e-01, -4.98284927e-01,  3.51335227e+00,
       -3.31262076e-01, -5.96892679e-01, -7.74105920e-01, -4.91427765e-01,
       -7.12950141e-01, -3.31262076e-01,  3.11950270e+00, -4.05391973e-01,
       -7.54200155e-01, -4.24270594e-01, -7.77996776e-01, -4.22900182e-01,
       -5.86524781e-01,  3.53572781e+00, -5.63678286e-01, -3.95125767e-01,
       -7.89502019e-01, -6.99953807e-01, -4.50135377e-01, -5.63678286e-01,
        3.49399416e+00, -5.94344765e-01, -5.23802802e-01, -6.62079121e-01,
       -6.60553248e-01, -4.46520106e-01, -5.94344765e-01,  3.48779329e+00,
       -6.37404083e-01, -5.76664819e-01, -5.72306264e-01, -6.55020845e-01,
       -5.10837001e-01, -5.99084894e-01, -6.37404083e-01,  3.50331101e+00,
       -5.63032186e-01, -5.37932006e-01, -4.37903658e-01, -7.38851534e-01,
       -4.05391973e-01, -5.63032186e-01,  3.47760422e+00, -6.71286583e-01,
       -6.61138285e-01, -4.99904689e-01, -4.95527277e-01,  3.32051001e+00,
       -6.61252267e-01, -5.74222450e-01, -4.62979688e-01, -6.26623638e-01,
       -7.34146676e-01, -4.16462806e-01, -6.61252267e-01,  3.58217737e+00,
       -5.26914431e-01, -4.04038139e-01, -8.39363051e-01, -6.97762838e-01,
       -5.96892679e-01, -5.74222450e-01,  3.43007707e+00, -4.11519938e-01,
       -2.40497254e-01, -9.09181915e-01, -8.22195094e-01, -6.04267048e-01,
       -5.26914431e-01,  3.49328037e+00, -2.83568923e-01, -4.66549594e-01,
       -7.89785285e-01, -6.57003743e-01, -8.08153388e-01, -3.95125767e-01,
       -2.83568923e-01,  3.45810631e+00, -4.76868868e-01, -8.37385621e-01,
       -7.74105920e-01, -7.54200155e-01, -4.11519938e-01,  3.09249541e+00,
       -6.63038396e-01, -4.89630996e-01, -4.24270594e-01, -6.71286583e-01,
       -6.63038396e-01,  3.48170690e+00, -4.29663607e-01, -6.80776636e-01,
       -6.12671082e-01, -7.89502019e-01, -5.23802802e-01, -4.76868868e-01,
        3.53776195e+00, -4.47394410e-01, -4.98024572e-01, -8.02169282e-01,
       -6.62079121e-01, -5.76664819e-01, -4.47394410e-01,  3.49460238e+00,
       -5.49052451e-01, -5.52999649e-01, -7.06411932e-01, -5.72306264e-01,
       -5.37932006e-01, -6.61138285e-01, -4.29663607e-01, -5.49052451e-01,
        3.50342544e+00, -7.53332821e-01, -4.62979688e-01, -4.04038139e-01,
        3.31480265e+00, -4.26318795e-01, -8.60695960e-01, -1.16077007e+00,
       -6.26623638e-01, -2.40497254e-01, -4.26318795e-01,  3.05896867e+00,
       -1.11167871e+00, -6.53850281e-01, -8.39363051e-01, -4.66549594e-01,
       -8.60695960e-01,  3.79158435e+00, -7.53365237e-01, -8.21862811e-01,
       -4.97476992e-02, -9.09181915e-01, -4.89630996e-01, -6.80776636e-01,
       -1.11167871e+00,  3.80273083e+00, -3.36443509e-01, -2.75019067e-01,
       -7.89785285e-01, -8.37385621e-01, -4.98024572e-01, -7.53365237e-01,
        3.13026683e+00, -2.51706115e-01, -6.12671082e-01, -5.52999649e-01,
       -7.53332821e-01, -3.36443509e-01,  3.62814641e+00, -3.90617221e-01,
       -9.82082123e-01, -8.02169282e-01, -7.06411932e-01, -8.21862811e-01,
       -2.51706115e-01, -3.90617221e-01,  3.66635374e+00, -6.93586382e-01,
       -1.16077007e+00, -6.53850281e-01, -4.97476992e-02, -2.75019067e-01,
       -9.82082123e-01, -6.93586382e-01,  3.81505562e+00])
        indices = np.array([   0,    1,    2,    3,    5,    6,    9,    0,    1,    2,    3,
          4,    7,    0,    1,    2,    4,    5,    8,   10,   14,    0,
          1,    3,    6,    7,   11,   12,    1,    2,    4,    7,    8,
         13,   15,    0,    2,    5,    9,   10,   16,   17,    0,    3,
          6,    9,   11,   18,   19,    1,    3,    4,    7,   12,   13,
         20,    2,    4,    8,   14,   15,   21,   22,    0,    5,    6,
          9,   16,   19,   23,    2,    5,   10,   14,   17,   24,   25,
          3,    6,   11,   12,   18,   26,   27,    3,    7,   11,   12,
         20,   27,   28,    4,    7,   13,   15,   20,   29,   30,    2,
          8,   10,   14,   21,   24,   31,    4,    8,   13,   15,   22,
         30,   32,    5,    9,   16,   17,   23,   33,   34,    5,   10,
         16,   17,   25,   34,   35,    6,   11,   18,   19,   26,   36,
         37,    6,    9,   18,   19,   23,   37,   38,    7,   12,   13,
         20,   28,   29,   39,    8,   14,   21,   22,   31,   40,   41,
          8,   15,   21,   22,   32,   40,   42,    9,   16,   19,   23,
         33,   38,   43,   10,   14,   24,   25,   31,   44,   45,   10,
         17,   24,   25,   35,   45,   46,   11,   18,   26,   27,   36,
         47,   48,   11,   12,   26,   27,   28,   48,   49,   12,   20,
         27,   28,   39,   49,   50,   13,   20,   29,   30,   39,   51,
         52,   13,   15,   29,   30,   32,   52,   53,   14,   21,   24,
         31,   41,   44,   54,   15,   22,   30,   32,   42,   53,   55,
         16,   23,   33,   34,   43,   56,   57,   16,   17,   33,   34,
         35,   56,   58,   17,   25,   34,   35,   46,   58,   59,   18,
         26,   36,   37,   47,   60,   61,   18,   19,   36,   37,   38,
         60,   62,   19,   23,   37,   38,   43,   62,   63,   20,   28,
         29,   39,   50,   51,   64,   21,   22,   40,   41,   42,   65,
         66,   21,   31,   40,   41,   54,   65,   67,   22,   32,   40,
         42,   55,   66,   68,   23,   33,   38,   43,   57,   63,   69,
         24,   31,   44,   45,   54,   70,   71,   24,   25,   44,   45,
         46,   70,   72,   25,   35,   45,   46,   59,   72,   73,   26,
         36,   47,   48,   61,   74,   75,   26,   27,   47,   48,   49,
         74,   76,   27,   28,   48,   49,   50,   76,   77,   28,   39,
         49,   50,   64,   77,   78,   29,   39,   51,   52,   64,   79,
         80,   29,   30,   51,   52,   53,   80,   81,   30,   32,   52,
         53,   55,   81,   82,   31,   41,   44,   54,   67,   71,   83,
         32,   42,   53,   55,   68,   82,   84,   33,   34,   56,   57,
         58,   85,   86,  100,   33,   43,   56,   57,   69,   85,   87,
         34,   35,   56,   58,   59,   86,   88,   35,   46,   58,   59,
         73,   88,   89,   36,   37,   60,   61,   62,   90,   91,   36,
         47,   60,   61,   75,   91,   92,   37,   38,   60,   62,   63,
         90,   38,   43,   62,   63,   69,   90,   93,   39,   50,   51,
         64,   78,   79,   94,   95,   40,   41,   65,   66,   67,   96,
         97,  118,   40,   42,   65,   66,   68,   96,   41,   54,   65,
         67,   83,   97,   98,   42,   55,   66,   68,   84,   96,   99,
         43,   57,   63,   69,   87,   93,  101,   44,   45,   70,   71,
         72,  102,  103,   44,   54,   70,   71,   83,  103,  104,   45,
         46,   70,   72,   73,  102,  105,   46,   59,   72,   73,   89,
        105,  106,   47,   48,   74,   75,   76,  107,  108,   47,   61,
         74,   75,   92,  107,  109,   48,   49,   74,   76,   77,  108,
        110,   49,   50,   76,   77,   78,  110,  111,   50,   64,   77,
         78,   94,  111,  112,   51,   64,   79,   80,   95,  113,  114,
         51,   52,   79,   80,   81,  113,  115,   52,   53,   80,   81,
         82,  115,  116,   53,   55,   81,   82,   84,  116,  117,   54,
         67,   71,   83,   98,  104,  119,   55,   68,   82,   84,   99,
        117,  120,   56,   57,   85,   87,  100,  121,   56,   58,   86,
         88,  100,  122,  123,   57,   69,   85,   87,  101,  121,  124,
         58,   59,   86,   88,   89,  122,  125,   59,   73,   88,   89,
        106,  125,  126,   60,   62,   63,   90,   91,   93,  127,  130,
         60,   61,   90,   91,   92,  127,  128,   61,   75,   91,   92,
        109,  128,  129,   63,   69,   90,   93,  101,  130,  131,   64,
         78,   94,   95,  112,  132,   64,   79,   94,   95,  114,  132,
        133,   65,   66,   68,   96,   99,  118,  134,   65,   67,   97,
         98,  118,  135,  136,   67,   83,   97,   98,  119,  136,  137,
         68,   84,   96,   99,  120,  134,  138,   56,   85,   86,  100,
        121,  123,  139,   69,   87,   93,  101,  124,  131,  140,   70,
         72,  102,  103,  105,  141,  142,   70,   71,  102,  103,  104,
        141,  143,   71,   83,  103,  104,  119,  143,  144,   72,   73,
        102,  105,  106,  142,  145,   73,   89,  105,  106,  126,  145,
        146,   74,   75,  107,  108,  109,  147,  148,   74,   76,  107,
        108,  110,  148,   75,   92,  107,  109,  129,  147,  149,   76,
         77,  108,  110,  111,  148,  150,   77,   78,  110,  111,  112,
        150,  151,   78,   94,  111,  112,  132,  151,  152,   79,   80,
        113,  114,  115,  153,  154,   79,   95,  113,  114,  133,  153,
        155,   80,   81,  113,  115,  116,  154,  156,   81,   82,  115,
        116,  117,  156,  157,   82,   84,  116,  117,  120,  157,  158,
         65,   96,   97,  118,  134,  135,  159,   83,   98,  104,  119,
        137,  144,  160,   84,   99,  117,  120,  138,  158,  161,   85,
         87,  100,  121,  124,  139,  162,   86,   88,  122,  123,  125,
        163,  164,   86,  100,  122,  123,  139,  163,  165,   87,  101,
        121,  124,  140,  162,  166,   88,   89,  122,  125,  126,  164,
        167,   89,  106,  125,  126,  146,  167,  168,   90,   91,  127,
        128,  130,  169,  170,   91,   92,  127,  128,  129,  170,  171,
         92,  109,  128,  129,  149,  171,  172,   90,   93,  127,  130,
        131,  169,  173,   93,  101,  130,  131,  140,  173,  174,   94,
         95,  112,  132,  133,  152,  175,  176,   95,  114,  132,  133,
        155,  176,  177,   96,   99,  118,  134,  138,  159,  178,   97,
        118,  135,  136,  159,  179,  180,   97,   98,  135,  136,  137,
        179,  181,   98,  119,  136,  137,  160,  181,  182,   99,  120,
        134,  138,  161,  178,  183,  100,  121,  123,  139,  162,  165,
        184,  101,  124,  131,  140,  166,  174,  185,  102,  103,  141,
        142,  143,  186,  187,  102,  105,  141,  142,  145,  187,  188,
        103,  104,  141,  143,  144,  186,  189,  104,  119,  143,  144,
        160,  189,  190,  105,  106,  142,  145,  146,  188,  191,  106,
        126,  145,  146,  168,  191,  192,  107,  109,  147,  148,  149,
        193,  107,  108,  110,  147,  148,  150,  193,  194,  109,  129,
        147,  149,  172,  193,  195,  110,  111,  148,  150,  151,  194,
        196,  111,  112,  150,  151,  152,  196,  197,  112,  132,  151,
        152,  175,  197,  198,  113,  114,  153,  154,  155,  199,  200,
        113,  115,  153,  154,  156,  200,  201,  114,  133,  153,  155,
        177,  199,  202,  115,  116,  154,  156,  157,  201,  203,  116,
        117,  156,  157,  158,  203,  204,  117,  120,  157,  158,  161,
        204,  205,  118,  134,  135,  159,  178,  180,  206,  119,  137,
        144,  160,  182,  190,  207,  120,  138,  158,  161,  183,  205,
        208,  121,  124,  139,  162,  166,  184,  209,  122,  123,  163,
        164,  165,  210,  211,  122,  125,  163,  164,  167,  210,  212,
        123,  139,  163,  165,  184,  211,  213,  124,  140,  162,  166,
        185,  209,  214,  125,  126,  164,  167,  168,  212,  215,  126,
        146,  167,  168,  192,  215,  216,  127,  130,  169,  170,  173,
        217,  218,  127,  128,  169,  170,  171,  218,  219,  128,  129,
        170,  171,  172,  219,  220,  129,  149,  171,  172,  195,  220,
        221,  130,  131,  169,  173,  174,  217,  222,  131,  140,  173,
        174,  185,  222,  223,  132,  152,  175,  176,  198,  224,  225,
        132,  133,  175,  176,  177,  224,  226,  133,  155,  176,  177,
        202,  226,  227,  134,  138,  159,  178,  183,  206,  228,  135,
        136,  179,  180,  181,  229,  230,  135,  159,  179,  180,  206,
        229,  231,  136,  137,  179,  181,  182,  230,  232,  137,  160,
        181,  182,  207,  232,  233,  138,  161,  178,  183,  208,  228,
        234,  139,  162,  165,  184,  209,  213,  235,  140,  166,  174,
        185,  214,  223,  236,  141,  143,  186,  187,  189,  237,  238,
        141,  142,  186,  187,  188,  238,  239,  142,  145,  187,  188,
        191,  239,  240,  143,  144,  186,  189,  190,  237,  241,  144,
        160,  189,  190,  207,  241,  242,  145,  146,  188,  191,  192,
        240,  243,  146,  168,  191,  192,  216,  243,  244,  147,  148,
        149,  193,  194,  195,  245,  148,  150,  193,  194,  196,  245,
        246,  149,  172,  193,  195,  221,  245,  247,  150,  151,  194,
        196,  197,  246,  248,  151,  152,  196,  197,  198,  248,  249,
        152,  175,  197,  198,  225,  249,  250,  153,  155,  199,  200,
        202,  251,  252,  153,  154,  199,  200,  201,  251,  253,  154,
        156,  200,  201,  203,  253,  254,  155,  177,  199,  202,  227,
        252,  255,  156,  157,  201,  203,  204,  254,  256,  157,  158,
        203,  204,  205,  256,  257,  158,  161,  204,  205,  208,  257,
        258,  159,  178,  180,  206,  228,  231,  259,  160,  182,  190,
        207,  233,  242,  260,  161,  183,  205,  208,  234,  258,  261,
        162,  166,  184,  209,  214,  235,  262,  163,  164,  210,  211,
        212,  263,  264,  163,  165,  210,  211,  213,  264,  265,  164,
        167,  210,  212,  215,  263,  266,  165,  184,  211,  213,  235,
        265,  267,  166,  185,  209,  214,  236,  262,  268,  167,  168,
        212,  215,  216,  266,  269,  168,  192,  215,  216,  244,  269,
        270,  169,  173,  217,  218,  222,  271,  272,  169,  170,  217,
        218,  219,  272,  273,  170,  171,  218,  219,  220,  273,  274,
        171,  172,  219,  220,  221,  274,  275,  172,  195,  220,  221,
        247,  275,  276,  173,  174,  217,  222,  223,  271,  277,  174,
        185,  222,  223,  236,  277,  278,  175,  176,  224,  225,  226,
        279,  280,  175,  198,  224,  225,  250,  279,  281,  176,  177,
        224,  226,  227,  280,  282,  177,  202,  226,  227,  255,  282,
        283,  178,  183,  206,  228,  234,  259,  284,  179,  180,  229,
        230,  231,  285,  286,  179,  181,  229,  230,  232,  286,  287,
        180,  206,  229,  231,  259,  285,  288,  181,  182,  230,  232,
        233,  287,  289,  182,  207,  232,  233,  260,  289,  290,  183,
        208,  228,  234,  261,  284,  291,  184,  209,  213,  235,  262,
        267,  292,  185,  214,  223,  236,  268,  278,  293,  186,  189,
        237,  238,  241,  294,  295,  186,  187,  237,  238,  239,  295,
        296,  187,  188,  238,  239,  240,  296,  297,  188,  191,  239,
        240,  243,  297,  298,  189,  190,  237,  241,  242,  294,  299,
        190,  207,  241,  242,  260,  299,  300,  191,  192,  240,  243,
        244,  298,  301,  329,  192,  216,  243,  244,  270,  301,  302,
        193,  194,  195,  245,  246,  247,  303,  194,  196,  245,  246,
        248,  303,  195,  221,  245,  247,  276,  303,  304,  196,  197,
        246,  248,  249,  303,  305,  197,  198,  248,  249,  250,  305,
        306,  198,  225,  249,  250,  281,  306,  307,  199,  200,  251,
        252,  253,  308,  309,  199,  202,  251,  252,  255,  308,  310,
        200,  201,  251,  253,  254,  309,  311,  201,  203,  253,  254,
        256,  311,  312,  202,  227,  252,  255,  283,  310,  313,  203,
        204,  254,  256,  257,  312,  314,  204,  205,  256,  257,  258,
        314,  315,  205,  208,  257,  258,  261,  315,  316,  206,  228,
        231,  259,  284,  288,  317,  207,  233,  242,  260,  290,  300,
        318,  208,  234,  258,  261,  291,  316,  319,  209,  214,  235,
        262,  268,  292,  320,  210,  212,  263,  264,  266,  321,  322,
        210,  211,  263,  264,  265,  321,  323,  211,  213,  264,  265,
        267,  323,  324,  212,  215,  263,  266,  269,  322,  325,  213,
        235,  265,  267,  292,  324,  326,  214,  236,  262,  268,  293,
        320,  327,  215,  216,  266,  269,  270,  325,  328,  216,  244,
        269,  270,  302,  328,  330,  217,  222,  271,  272,  277,  331,
        332,  217,  218,  271,  272,  273,  331,  333,  218,  219,  272,
        273,  274,  333,  334,  219,  220,  273,  274,  275,  334,  335,
        220,  221,  274,  275,  276,  335,  336,  221,  247,  275,  276,
        304,  336,  337,  222,  223,  271,  277,  278,  332,  338,  223,
        236,  277,  278,  293,  338,  339,  224,  225,  279,  280,  281,
        340,  341,  224,  226,  279,  280,  282,  340,  342,  225,  250,
        279,  281,  307,  341,  343,  226,  227,  280,  282,  283,  342,
        344,  227,  255,  282,  283,  313,  344,  345,  228,  234,  259,
        284,  291,  317,  346,  229,  231,  285,  286,  288,  347,  348,
        229,  230,  285,  286,  287,  347,  349,  230,  232,  286,  287,
        289,  349,  350,  231,  259,  285,  288,  317,  348,  351,  232,
        233,  287,  289,  290,  350,  352,  233,  260,  289,  290,  318,
        352,  353,  234,  261,  284,  291,  319,  346,  354,  235,  262,
        267,  292,  320,  326,  355,  236,  268,  278,  293,  327,  339,
        356,  237,  241,  294,  295,  299,  357,  358,  237,  238,  294,
        295,  296,  358,  359,  238,  239,  295,  296,  297,  359,  360,
        239,  240,  296,  297,  298,  360,  240,  243,  297,  298,  329,
        360,  361,  241,  242,  294,  299,  300,  357,  362,  242,  260,
        299,  300,  318,  362,  363,  243,  244,  301,  302,  329,  364,
        244,  270,  301,  302,  330,  364,  365,  245,  246,  247,  248,
        303,  304,  305,  247,  276,  303,  304,  305,  337,  366,  248,
        249,  303,  304,  305,  306,  366,  249,  250,  305,  306,  307,
        366,  367,  250,  281,  306,  307,  343,  367,  368,  251,  252,
        308,  309,  310,  369,  370,  251,  253,  308,  309,  311,  370,
        371,  252,  255,  308,  310,  313,  369,  372,  253,  254,  309,
        311,  312,  371,  373,  254,  256,  311,  312,  314,  373,  374,
        255,  283,  310,  313,  345,  372,  375,  256,  257,  312,  314,
        315,  374,  376,  257,  258,  314,  315,  316,  376,  377,  258,
        261,  315,  316,  319,  377,  378,  259,  284,  288,  317,  346,
        351,  379,  260,  290,  300,  318,  353,  363,  380,  261,  291,
        316,  319,  354,  378,  381,  262,  268,  292,  320,  327,  355,
        382,  263,  264,  321,  322,  323,  383,  384,  263,  266,  321,
        322,  325,  384,  385,  264,  265,  321,  323,  324,  383,  386,
        265,  267,  323,  324,  326,  386,  387,  266,  269,  322,  325,
        328,  385,  388,  420,  267,  292,  324,  326,  355,  387,  389,
        268,  293,  320,  327,  356,  382,  390,  269,  270,  325,  328,
        330,  388,  391,  243,  298,  301,  329,  361,  364,  392,  270,
        302,  328,  330,  365,  391,  393,  271,  272,  331,  332,  333,
        394,  395,  271,  277,  331,  332,  338,  395,  396,  272,  273,
        331,  333,  334,  394,  273,  274,  333,  334,  335,  394,  397,
        274,  275,  334,  335,  336,  397,  398,  275,  276,  335,  336,
        337,  398,  399,  276,  304,  336,  337,  366,  399,  400,  277,
        278,  332,  338,  339,  396,  401,  278,  293,  338,  339,  356,
        401,  402,  279,  280,  340,  341,  342,  403,  404,  279,  281,
        340,  341,  343,  403,  405,  432,  280,  282,  340,  342,  344,
        404,  406,  281,  307,  341,  343,  368,  405,  407,  282,  283,
        342,  344,  345,  406,  408,  283,  313,  344,  345,  375,  408,
        409,  284,  291,  317,  346,  354,  379,  410,  285,  286,  347,
        348,  349,  411,  285,  288,  347,  348,  351,  411,  412,  286,
        287,  347,  349,  350,  411,  413,  446,  287,  289,  349,  350,
        352,  413,  414,  288,  317,  348,  351,  379,  412,  415,  289,
        290,  350,  352,  353,  414,  416,  290,  318,  352,  353,  380,
        416,  417,  291,  319,  346,  354,  381,  410,  418,  292,  320,
        326,  355,  382,  389,  419,  293,  327,  339,  356,  390,  402,
        421,  294,  299,  357,  358,  362,  422,  423,  294,  295,  357,
        358,  359,  422,  424,  295,  296,  358,  359,  360,  424,  425,
        296,  297,  298,  359,  360,  361,  425,  298,  329,  360,  361,
        392,  425,  426,  299,  300,  357,  362,  363,  423,  427,  300,
        318,  362,  363,  380,  427,  428,  301,  302,  329,  364,  365,
        392,  429,  302,  330,  364,  365,  393,  429,  430,  304,  305,
        306,  337,  366,  367,  400,  306,  307,  366,  367,  368,  400,
        431,  307,  343,  367,  368,  407,  431,  433,  308,  310,  369,
        370,  372,  434,  435,  308,  309,  369,  370,  371,  435,  436,
        309,  311,  370,  371,  373,  436,  437,  310,  313,  369,  372,
        375,  434,  438,  311,  312,  371,  373,  374,  437,  439,  312,
        314,  373,  374,  376,  439,  440,  313,  345,  372,  375,  409,
        438,  441,  314,  315,  374,  376,  377,  440,  442,  315,  316,
        376,  377,  378,  442,  443,  316,  319,  377,  378,  381,  443,
        444,  317,  346,  351,  379,  410,  415,  445,  318,  353,  363,
        380,  417,  428,  447,  319,  354,  378,  381,  418,  444,  448,
        320,  327,  355,  382,  390,  419,  449,  321,  323,  383,  384,
        386,  450,  451,  321,  322,  383,  384,  385,  450,  452,  322,
        325,  384,  385,  420,  452,  453,  323,  324,  383,  386,  387,
        451,  454,  324,  326,  386,  387,  389,  454,  455,  325,  328,
        388,  391,  420,  456,  457,  326,  355,  387,  389,  419,  455,
        458,  327,  356,  382,  390,  421,  449,  459,  328,  330,  388,
        391,  393,  456,  460,  329,  361,  364,  392,  426,  429,  461,
        330,  365,  391,  393,  430,  460,  462,  331,  333,  334,  394,
        395,  397,  463,  331,  332,  394,  395,  396,  463,  464,  332,
        338,  395,  396,  401,  464,  465,  334,  335,  394,  397,  398,
        463,  466,  335,  336,  397,  398,  399,  466,  467,  336,  337,
        398,  399,  400,  467,  468,  337,  366,  367,  399,  400,  431,
        468,  338,  339,  396,  401,  402,  465,  469,  339,  356,  401,
        402,  421,  469,  470,  340,  341,  403,  404,  432,  471,  472,
        340,  342,  403,  404,  406,  471,  473,  341,  343,  405,  407,
        432,  474,  475,  342,  344,  404,  406,  408,  473,  476,  343,
        368,  405,  407,  433,  474,  477,  344,  345,  406,  408,  409,
        476,  478,  345,  375,  408,  409,  441,  478,  479,  346,  354,
        379,  410,  418,  445,  480,  347,  348,  349,  411,  412,  446,
        481,  348,  351,  411,  412,  415,  481,  482,  349,  350,  413,
        414,  446,  483,  484,  350,  352,  413,  414,  416,  483,  485,
        351,  379,  412,  415,  445,  482,  486,  352,  353,  414,  416,
        417,  485,  487,  353,  380,  416,  417,  447,  487,  488,  354,
        381,  410,  418,  448,  480,  489,  355,  382,  389,  419,  449,
        458,  490,  325,  385,  388,  420,  453,  457,  356,  390,  402,
        421,  459,  470,  357,  358,  422,  423,  424,  491,  492,  357,
        362,  422,  423,  427,  491,  493,  358,  359,  422,  424,  425,
        492,  494,  359,  360,  361,  424,  425,  426,  494,  361,  392,
        425,  426,  461,  494,  495,  362,  363,  423,  427,  428,  493,
        496,  363,  380,  427,  428,  447,  496,  497,  364,  365,  392,
        429,  430,  461,  498,  365,  393,  429,  430,  462,  498,  499,
        367,  368,  400,  431,  433,  468,  500,  341,  403,  405,  432,
        472,  475,  368,  407,  431,  433,  477,  500,  501,  369,  372,
        434,  435,  438,  502,  503,  369,  370,  434,  435,  436,  502,
        504,  370,  371,  435,  436,  437,  504,  505,  371,  373,  436,
        437,  439,  505,  506,  372,  375,  434,  438,  441,  503,  507,
        373,  374,  437,  439,  440,  506,  508,  374,  376,  439,  440,
        442,  508,  509,  375,  409,  438,  441,  479,  507,  510,  376,
        377,  440,  442,  443,  509,  511,  377,  378,  442,  443,  444,
        511,  378,  381,  443,  444,  448,  511,  512,  546,  379,  410,
        415,  445,  480,  486,  349,  411,  413,  446,  481,  484,  513,
        380,  417,  428,  447,  488,  497,  514,  381,  418,  444,  448,
        489,  512,  515,  382,  390,  419,  449,  459,  490,  516,  383,
        384,  450,  451,  452,  517,  518,  383,  386,  450,  451,  454,
        518,  519,  384,  385,  450,  452,  453,  517,  520,  385,  420,
        452,  453,  457,  520,  521,  386,  387,  451,  454,  455,  519,
        522,  387,  389,  454,  455,  458,  522,  523,  388,  391,  456,
        457,  460,  524,  525,  388,  420,  453,  456,  457,  521,  525,
        389,  419,  455,  458,  490,  523,  526,  390,  421,  449,  459,
        470,  516,  527,  391,  393,  456,  460,  462,  524,  528,  392,
        426,  429,  461,  495,  498,  529,  393,  430,  460,  462,  499,
        528,  530,  394,  395,  397,  463,  464,  466,  531,  395,  396,
        463,  464,  465,  531,  532,  396,  401,  464,  465,  469,  532,
        533,  397,  398,  463,  466,  467,  531,  534,  398,  399,  466,
        467,  468,  534,  399,  400,  431,  467,  468,  500,  534,  568,
        401,  402,  465,  469,  470,  533,  535,  402,  421,  459,  469,
        470,  527,  535,  558,  403,  404,  471,  472,  473,  536,  537,
        403,  432,  471,  472,  475,  536,  538,  404,  406,  471,  473,
        476,  537,  539,  405,  407,  474,  475,  477,  540,  541,  405,
        432,  472,  474,  475,  538,  541,  406,  408,  473,  476,  478,
        539,  542,  407,  433,  474,  477,  501,  540,  543,  408,  409,
        476,  478,  479,  542,  544,  409,  441,  478,  479,  510,  544,
        545,  410,  418,  445,  480,  486,  489,  547,  411,  412,  446,
        481,  482,  513,  548,  412,  415,  481,  482,  486,  548,  549,
        413,  414,  483,  484,  485,  550,  551,  413,  446,  483,  484,
        513,  551,  552,  414,  416,  483,  485,  487,  550,  553,  415,
        445,  480,  482,  486,  547,  549,  582,  416,  417,  485,  487,
        488,  553,  554,  417,  447,  487,  488,  514,  554,  555,  418,
        448,  480,  489,  515,  547,  556,  419,  449,  458,  490,  516,
        526,  557,  422,  423,  491,  492,  493,  559,  560,  422,  424,
        491,  492,  494,  560,  561,  423,  427,  491,  493,  496,  559,
        562,  424,  425,  426,  492,  494,  495,  561,  426,  461,  494,
        495,  529,  561,  563,  427,  428,  493,  496,  497,  562,  564,
        428,  447,  496,  497,  514,  564,  565,  429,  430,  461,  498,
        499,  529,  566,  430,  462,  498,  499,  530,  566,  567,  431,
        433,  468,  500,  501,  568,  433,  477,  500,  501,  543,  568,
        569,  607,  434,  435,  502,  503,  504,  570,  571,  434,  438,
        502,  503,  507,  570,  572,  435,  436,  502,  504,  505,  571,
        573,  436,  437,  504,  505,  506,  573,  574,  437,  439,  505,
        506,  508,  574,  575,  438,  441,  503,  507,  510,  572,  576,
        439,  440,  506,  508,  509,  575,  577,  440,  442,  508,  509,
        511,  577,  578,  441,  479,  507,  510,  545,  576,  579,  442,
        443,  444,  509,  511,  546,  578,  444,  448,  512,  515,  546,
        580,  581,  446,  481,  484,  513,  548,  552,  583,  447,  488,
        497,  514,  555,  565,  584,  448,  489,  512,  515,  556,  581,
        585,  449,  459,  490,  516,  527,  557,  586,  587,  450,  452,
        517,  518,  520,  588,  589,  450,  451,  517,  518,  519,  589,
        590,  451,  454,  518,  519,  522,  590,  591,  452,  453,  517,
        520,  521,  588,  592,  453,  457,  520,  521,  525,  592,  593,
        454,  455,  519,  522,  523,  591,  594,  455,  458,  522,  523,
        526,  594,  595,  456,  460,  524,  525,  528,  596,  597,  456,
        457,  521,  524,  525,  593,  597,  458,  490,  523,  526,  557,
        595,  598,  459,  470,  516,  527,  558,  586,  599,  460,  462,
        524,  528,  530,  596,  600,  461,  495,  498,  529,  563,  566,
        601,  462,  499,  528,  530,  567,  600,  602,  463,  464,  466,
        531,  532,  534,  603,  464,  465,  531,  532,  533,  603,  604,
        465,  469,  532,  533,  535,  604,  605,  466,  467,  468,  531,
        534,  568,  603,  469,  470,  533,  535,  558,  605,  606,  471,
        472,  536,  537,  538,  608,  609,  471,  473,  536,  537,  539,
        609,  610,  472,  475,  536,  538,  541,  608,  611,  473,  476,
        537,  539,  542,  610,  612,  474,  477,  540,  541,  543,  613,
        614,  474,  475,  538,  540,  541,  611,  614,  476,  478,  539,
        542,  544,  612,  615,  477,  501,  540,  543,  569,  613,  616,
        478,  479,  542,  544,  545,  615,  617,  479,  510,  544,  545,
        579,  617,  618,  444,  511,  512,  546,  578,  580,  619,  480,
        486,  489,  547,  556,  582,  620,  481,  482,  513,  548,  549,
        583,  621,  482,  486,  548,  549,  582,  621,  622,  483,  485,
        550,  551,  553,  623,  624,  483,  484,  550,  551,  552,  623,
        625,  484,  513,  551,  552,  583,  625,  626,  485,  487,  550,
        553,  554,  624,  627,  487,  488,  553,  554,  555,  627,  488,
        514,  554,  555,  584,  627,  628,  489,  515,  547,  556,  585,
        620,  629,  490,  516,  526,  557,  587,  598,  630,  470,  527,
        535,  558,  599,  606,  631,  491,  493,  559,  560,  562,  632,
        633,  491,  492,  559,  560,  561,  632,  634,  492,  494,  495,
        560,  561,  563,  634,  493,  496,  559,  562,  564,  633,  635,
        495,  529,  561,  563,  601,  634,  636,  496,  497,  562,  564,
        565,  635,  637,  497,  514,  564,  565,  584,  637,  638,  498,
        499,  529,  566,  567,  601,  639,  499,  530,  566,  567,  602,
        639,  640,  468,  500,  501,  534,  568,  603,  607,  501,  543,
        569,  607,  616,  641,  642,  502,  503,  570,  571,  572,  643,
        644,  502,  504,  570,  571,  573,  644,  645,  503,  507,  570,
        572,  576,  643,  646,  504,  505,  571,  573,  574,  645,  647,
        505,  506,  573,  574,  575,  647,  648,  506,  508,  574,  575,
        577,  648,  649,  507,  510,  572,  576,  579,  646,  650,  508,
        509,  575,  577,  578,  649,  651,  509,  511,  546,  577,  578,
        619,  651,  510,  545,  576,  579,  618,  650,  652,  512,  546,
        580,  581,  619,  653,  654,  512,  515,  580,  581,  585,  653,
        655,  486,  547,  549,  582,  620,  622,  656,  513,  548,  552,
        583,  621,  626,  657,  514,  555,  565,  584,  628,  638,  658,
        515,  556,  581,  585,  629,  655,  659,  516,  527,  586,  587,
        599,  660,  661,  516,  557,  586,  587,  630,  661,  517,  520,
        588,  589,  592,  662,  663,  517,  518,  588,  589,  590,  662,
        664,  518,  519,  589,  590,  591,  664,  665,  519,  522,  590,
        591,  594,  665,  666,  520,  521,  588,  592,  593,  663,  667,
        521,  525,  592,  593,  597,  667,  668,  522,  523,  591,  594,
        595,  666,  669,  523,  526,  594,  595,  598,  669,  670,  524,
        528,  596,  597,  600,  671,  672,  524,  525,  593,  596,  597,
        668,  672,  526,  557,  595,  598,  630,  670,  673,  705,  527,
        558,  586,  599,  631,  660,  674,  528,  530,  596,  600,  602,
        671,  675,  529,  563,  566,  601,  636,  639,  676,  530,  567,
        600,  602,  640,  675,  677,  707,  531,  532,  534,  568,  603,
        604,  607,  532,  533,  603,  604,  605,  607,  641,  533,  535,
        604,  605,  606,  641,  678,  535,  558,  605,  606,  631,  678,
        679,  501,  568,  569,  603,  604,  607,  641,  536,  538,  608,
        609,  611,  680,  681,  536,  537,  608,  609,  610,  681,  682,
        537,  539,  609,  610,  612,  682,  683,  538,  541,  608,  611,
        614,  680,  684,  539,  542,  610,  612,  615,  683,  685,  540,
        543,  613,  614,  616,  686,  687,  540,  541,  611,  613,  614,
        684,  686,  542,  544,  612,  615,  617,  685,  688,  543,  569,
        613,  616,  642,  687,  689,  544,  545,  615,  617,  618,  688,
        690,  545,  579,  617,  618,  652,  690,  691,  546,  578,  580,
        619,  651,  654,  692,  547,  556,  582,  620,  629,  656,  693,
        548,  549,  583,  621,  622,  657,  694,  549,  582,  621,  622,
        656,  694,  695,  550,  551,  623,  624,  625,  696,  697,  550,
        553,  623,  624,  627,  697,  698,  551,  552,  623,  625,  626,
        696,  699,  552,  583,  625,  626,  657,  699,  700,  553,  554,
        555,  624,  627,  628,  698,  701,  555,  584,  627,  628,  658,
        701,  702,  556,  585,  620,  629,  659,  693,  703,  557,  587,
        598,  630,  661,  673,  704,  558,  599,  606,  631,  674,  679,
        706,  559,  560,  632,  633,  634,  708,  709,  559,  562,  632,
        633,  635,  708,  710,  560,  561,  563,  632,  634,  636,  709,
        562,  564,  633,  635,  637,  710,  711,  563,  601,  634,  636,
        676,  709,  712,  564,  565,  635,  637,  638,  711,  713,  754,
        565,  584,  637,  638,  658,  713,  714,  566,  567,  601,  639,
        640,  676,  715,  567,  602,  639,  640,  677,  715,  569,  604,
        605,  607,  641,  642,  678,  569,  616,  641,  642,  678,  689,
        716,  570,  572,  643,  644,  646,  717,  718,  570,  571,  643,
        644,  645,  718,  719,  571,  573,  644,  645,  647,  719,  720,
        572,  576,  643,  646,  650,  717,  721,  573,  574,  645,  647,
        648,  720,  722,  574,  575,  647,  648,  649,  722,  723,  575,
        577,  648,  649,  651,  723,  724,  576,  579,  646,  650,  652,
        721,  725,  577,  578,  619,  649,  651,  692,  724,  579,  618,
        650,  652,  691,  725,  726,  580,  581,  653,  654,  655,  727,
        728,  580,  619,  653,  654,  692,  727,  729,  581,  585,  653,
        655,  659,  728,  730,  582,  620,  622,  656,  693,  695,  731,
        583,  621,  626,  657,  694,  700,  732,  584,  628,  638,  658,
        702,  714,  733,  585,  629,  655,  659,  703,  730,  734,  586,
        599,  660,  661,  674,  735,  736,  586,  587,  630,  660,  661,
        704,  735,  737,  588,  589,  662,  663,  664,  738,  739,  588,
        592,  662,  663,  667,  738,  740,  589,  590,  662,  664,  665,
        739,  741,  590,  591,  664,  665,  666,  741,  742,  591,  594,
        665,  666,  669,  742,  743,  592,  593,  663,  667,  668,  740,
        744,  593,  597,  667,  668,  672,  744,  745,  594,  595,  666,
        669,  670,  743,  746,  595,  598,  669,  670,  705,  746,  747,
        596,  600,  671,  672,  675,  748,  749,  596,  597,  668,  671,
        672,  745,  748,  598,  630,  673,  704,  705,  750,  751,  599,
        631,  660,  674,  706,  736,  752,  600,  602,  671,  675,  707,
        749,  753,  601,  636,  639,  676,  712,  715,  755,  602,  640,
        677,  707,  715,  756,  757,  605,  606,  641,  642,  678,  679,
        716,  758,  606,  631,  678,  679,  706,  758,  608,  611,  680,
        681,  684,  759,  760,  608,  609,  680,  681,  682,  759,  761,
        609,  610,  681,  682,  683,  761,  762,  610,  612,  682,  683,
        685,  762,  763,  611,  614,  680,  684,  686,  760,  764,  612,
        615,  683,  685,  688,  763,  765,  613,  614,  684,  686,  687,
        764,  766,  613,  616,  686,  687,  689,  766,  767,  615,  617,
        685,  688,  690,  765,  768,  616,  642,  687,  689,  716,  767,
        769,  617,  618,  688,  690,  691,  768,  770,  618,  652,  690,
        691,  726,  770,  771,  619,  651,  654,  692,  724,  729,  772,
        620,  629,  656,  693,  703,  731,  773,  621,  622,  657,  694,
        695,  732,  774,  622,  656,  694,  695,  731,  774,  775,  623,
        625,  696,  697,  699,  776,  623,  624,  696,  697,  698,  776,
        777,  624,  627,  697,  698,  701,  777,  778,  625,  626,  696,
        699,  700,  776,  779,  626,  657,  699,  700,  732,  779,  780,
        627,  628,  698,  701,  702,  778,  781,  628,  658,  701,  702,
        733,  781,  782,  629,  659,  693,  703,  734,  773,  783,  630,
        661,  673,  704,  737,  750,  598,  670,  673,  705,  747,  751,
        631,  674,  679,  706,  752,  758,  784,  602,  675,  677,  707,
        753,  757,  785,  632,  633,  708,  709,  710,  786,  787,  632,
        634,  636,  708,  709,  712,  786,  633,  635,  708,  710,  711,
        787,  788,  635,  637,  710,  711,  754,  788,  789,  636,  676,
        709,  712,  755,  786,  790,  637,  638,  713,  714,  754,  791,
        792,  638,  658,  713,  714,  733,  791,  793,  639,  640,  676,
        677,  715,  755,  756,  642,  678,  689,  716,  758,  769,  794,
        643,  646,  717,  718,  721,  795,  796,  643,  644,  717,  718,
        719,  795,  797,  644,  645,  718,  719,  720,  797,  798,  645,
        647,  719,  720,  722,  798,  799,  646,  650,  717,  721,  725,
        796,  800,  647,  648,  720,  722,  723,  799,  801,  648,  649,
        722,  723,  724,  801,  802,  649,  651,  692,  723,  724,  772,
        802,  650,  652,  721,  725,  726,  800,  803,  652,  691,  725,
        726,  771,  803,  804,  653,  654,  727,  728,  729,  805,  806,
        653,  655,  727,  728,  730,  806,  807,  654,  692,  727,  729,
        772,  805,  808,  655,  659,  728,  730,  734,  807,  809,  656,
        693,  695,  731,  773,  775,  657,  694,  700,  732,  774,  780,
        810,  658,  702,  714,  733,  782,  793,  811,  659,  703,  730,
        734,  783,  809,  812,  660,  661,  735,  736,  737,  813,  814,
        660,  674,  735,  736,  752,  814,  815,  661,  704,  735,  737,
        750,  813,  816,  662,  663,  738,  739,  740,  817,  818,  662,
        664,  738,  739,  741,  817,  819,  663,  667,  738,  740,  744,
        818,  820,  664,  665,  739,  741,  742,  819,  821,  665,  666,
        741,  742,  743,  821,  822,  862,  666,  669,  742,  743,  746,
        822,  823,  667,  668,  740,  744,  745,  820,  824,  668,  672,
        744,  745,  748,  824,  825,  669,  670,  743,  746,  747,  823,
        826,  670,  705,  746,  747,  751,  826,  827,  671,  672,  745,
        748,  749,  825,  828,  671,  675,  748,  749,  753,  828,  829,
        673,  704,  737,  750,  751,  816,  830,  673,  705,  747,  750,
        751,  827,  830,  674,  706,  736,  752,  784,  815,  831,  675,
        707,  749,  753,  785,  829,  832,  637,  711,  713,  754,  789,
        792,  833,  676,  712,  715,  755,  756,  790,  834,  677,  715,
        755,  756,  757,  834,  835,  677,  707,  756,  757,  785,  835,
        836,  678,  679,  706,  716,  758,  784,  794,  680,  681,  759,
        760,  761,  837,  838,  680,  684,  759,  760,  764,  838,  839,
        681,  682,  759,  761,  762,  837,  840,  682,  683,  761,  762,
        763,  840,  841,  683,  685,  762,  763,  765,  841,  842,  684,
        686,  760,  764,  766,  839,  843,  685,  688,  763,  765,  768,
        842,  844,  686,  687,  764,  766,  767,  843,  845,  687,  689,
        766,  767,  769,  845,  846,  688,  690,  765,  768,  770,  844,
        847,  689,  716,  767,  769,  794,  846,  848,  690,  691,  768,
        770,  771,  847,  849,  691,  726,  770,  771,  804,  849,  850,
        692,  724,  729,  772,  802,  808,  851,  693,  703,  731,  773,
        775,  783,  852,  694,  695,  732,  774,  775,  810,  853,  695,
        731,  773,  774,  775,  852,  853,  890,  696,  697,  699,  776,
        777,  779,  854,  855,  697,  698,  776,  777,  778,  855,  856,
        698,  701,  777,  778,  781,  856,  857,  699,  700,  776,  779,
        780,  854,  858,  892,  700,  732,  779,  780,  810,  858,  701,
        702,  778,  781,  782,  857,  859,  702,  733,  781,  782,  811,
        859,  860,  703,  734,  773,  783,  812,  852,  861,  706,  752,
        758,  784,  794,  831,  863,  707,  753,  757,  785,  832,  836,
        864,  708,  709,  712,  786,  787,  790,  865,  708,  710,  786,
        787,  788,  865,  866,  710,  711,  787,  788,  789,  866,  867,
        711,  754,  788,  789,  833,  867,  868,  712,  755,  786,  790,
        834,  865,  869,  713,  714,  791,  792,  793,  870,  871,  713,
        754,  791,  792,  833,  870,  872,  714,  733,  791,  793,  811,
        871,  873,  716,  758,  769,  784,  794,  848,  863,  717,  718,
        795,  796,  797,  874,  875,  717,  721,  795,  796,  800,  875,
        876,  718,  719,  795,  797,  798,  874,  877,  719,  720,  797,
        798,  799,  877,  878,  720,  722,  798,  799,  801,  878,  879,
        721,  725,  796,  800,  803,  876,  880,  722,  723,  799,  801,
        802,  879,  881,  723,  724,  772,  801,  802,  851,  881,  725,
        726,  800,  803,  804,  880,  882,  726,  771,  803,  804,  850,
        882,  883,  727,  729,  805,  806,  808,  884,  885,  727,  728,
        805,  806,  807,  885,  886,  728,  730,  806,  807,  809,  886,
        887,  729,  772,  805,  808,  851,  884,  888,  730,  734,  807,
        809,  812,  887,  889,  732,  774,  780,  810,  853,  858,  891,
        733,  782,  793,  811,  860,  873,  893,  734,  783,  809,  812,
        861,  889,  894,  735,  737,  813,  814,  816,  895,  896,  735,
        736,  813,  814,  815,  895,  897,  736,  752,  814,  815,  831,
        897,  898,  737,  750,  813,  816,  830,  896,  899,  738,  739,
        817,  818,  819,  900,  901,  738,  740,  817,  818,  820,  900,
        902,  739,  741,  817,  819,  821,  901,  903,  740,  744,  818,
        820,  824,  902,  904,  741,  742,  819,  821,  862,  903,  905,
        742,  743,  822,  823,  862,  906,  907,  743,  746,  822,  823,
        826,  907,  908,  744,  745,  820,  824,  825,  904,  909,  745,
        748,  824,  825,  828,  909,  910,  746,  747,  823,  826,  827,
        908,  911,  747,  751,  826,  827,  830,  911,  912,  748,  749,
        825,  828,  829,  910,  913,  749,  753,  828,  829,  832,  913,
        914,  750,  751,  816,  827,  830,  899,  912,  752,  784,  815,
        831,  863,  898,  915,  753,  785,  829,  832,  864,  914,  916,
        754,  789,  792,  833,  868,  872,  917,  755,  756,  790,  834,
        835,  869,  918,  756,  757,  834,  835,  836,  918,  919,  757,
        785,  835,  836,  864,  919,  920,  759,  761,  837,  838,  840,
        921,  922,  759,  760,  837,  838,  839,  921,  923,  760,  764,
        838,  839,  843,  923,  924,  761,  762,  837,  840,  841,  922,
        925,  762,  763,  840,  841,  842,  925,  926,  763,  765,  841,
        842,  844,  926,  927,  764,  766,  839,  843,  845,  924,  928,
        765,  768,  842,  844,  847,  927,  929,  766,  767,  843,  845,
        846,  928,  930,  767,  769,  845,  846,  848,  930,  931,  768,
        770,  844,  847,  849,  929,  932,  769,  794,  846,  848,  863,
        931,  933,  770,  771,  847,  849,  850,  932,  934,  771,  804,
        849,  850,  883,  934,  935,  772,  802,  808,  851,  881,  888,
        936,  773,  775,  783,  852,  861,  890,  937,  774,  775,  810,
        853,  890,  891,  938,  776,  779,  854,  855,  892,  939,  940,
        776,  777,  854,  855,  856,  940,  777,  778,  855,  856,  857,
        940,  941,  778,  781,  856,  857,  859,  941,  942,  779,  780,
        810,  858,  891,  892,  943,  781,  782,  857,  859,  860,  942,
        944,  782,  811,  859,  860,  893,  944,  945,  783,  812,  852,
        861,  894,  937,  946,  742,  821,  822,  862,  905,  906,  784,
        794,  831,  848,  863,  915,  933,  785,  832,  836,  864,  916,
        920,  947,  786,  787,  790,  865,  866,  869,  948,  787,  788,
        865,  866,  867,  948,  949,  788,  789,  866,  867,  868,  949,
        950,  789,  833,  867,  868,  917,  950,  951,  790,  834,  865,
        869,  918,  948,  952,  791,  792,  870,  871,  872,  953,  954,
        791,  793,  870,  871,  873,  953,  955,  792,  833,  870,  872,
        917,  954,  956,  793,  811,  871,  873,  893,  955,  957,  795,
        797,  874,  875,  877,  958,  959,  795,  796,  874,  875,  876,
        959,  960,  796,  800,  875,  876,  880,  960,  797,  798,  874,
        877,  878,  958,  961,  798,  799,  877,  878,  879,  961,  962,
        799,  801,  878,  879,  881,  962,  963,  800,  803,  876,  880,
        882,  960,  964,  801,  802,  851,  879,  881,  936,  963,  803,
        804,  880,  882,  883,  964,  965,  804,  850,  882,  883,  935,
        965,  966,  805,  808,  884,  885,  888,  967,  968,  805,  806,
        884,  885,  886,  967,  969,  806,  807,  885,  886,  887,  969,
        970,  807,  809,  886,  887,  889,  970,  971,  808,  851,  884,
        888,  936,  968,  972,  809,  812,  887,  889,  894,  971,  973,
        775,  852,  853,  890,  937,  938,  974,  810,  853,  858,  891,
        938,  943,  975,  779,  854,  858,  892,  939,  943,  976,  811,
        860,  873,  893,  945,  957,  977,  812,  861,  889,  894,  946,
        973,  978,  813,  814,  895,  896,  897,  979,  980,  813,  816,
        895,  896,  899,  979,  981,  814,  815,  895,  897,  898,  980,
        982,  815,  831,  897,  898,  915,  982,  983,  816,  830,  896,
        899,  912,  981,  984,  817,  818,  900,  901,  902,  985,  986,
        817,  819,  900,  901,  903,  985,  987,  818,  820,  900,  902,
        904,  986,  988,  819,  821,  901,  903,  905,  987,  989,  820,
        824,  902,  904,  909,  988,  990,  821,  862,  903,  905,  906,
        989,  991,  822,  862,  905,  906,  907,  991,  992,  822,  823,
        906,  907,  908,  992,  993,  823,  826,  907,  908,  911,  993,
        994,  824,  825,  904,  909,  910,  990,  995,  825,  828,  909,
        910,  913,  995,  996,  826,  827,  908,  911,  912,  994,  997,
        827,  830,  899,  911,  912,  984,  997,  828,  829,  910,  913,
        914,  996,  998,  829,  832,  913,  914,  916,  998,  999,  831,
        863,  898,  915,  933,  983, 1000,  832,  864,  914,  916,  947,
        999, 1001,  833,  868,  872,  917,  951,  956, 1002,  834,  835,
        869,  918,  919,  952, 1003,  835,  836,  918,  919,  920, 1003,
       1004,  836,  864,  919,  920,  947, 1004, 1005,  837,  838,  921,
        922,  923, 1006, 1007,  837,  840,  921,  922,  925, 1007, 1008,
        838,  839,  921,  923,  924, 1006, 1009,  839,  843,  923,  924,
        928, 1009, 1010,  840,  841,  922,  925,  926, 1008, 1011,  841,
        842,  925,  926,  927, 1011,  842,  844,  926,  927,  929, 1011,
       1012, 1041,  843,  845,  924,  928,  930, 1010, 1013,  844,  847,
        927,  929,  932, 1012, 1014,  845,  846,  928,  930,  931, 1013,
       1015,  846,  848,  930,  931,  933, 1015, 1016,  847,  849,  929,
        932,  934, 1014, 1017,  848,  863,  915,  931,  933, 1000, 1016,
        849,  850,  932,  934,  935, 1017, 1018,  850,  883,  934,  935,
        966, 1018, 1019,  851,  881,  888,  936,  963,  972, 1020,  852,
        861,  890,  937,  946,  974,  853,  890,  891,  938,  974,  975,
       1021,  854,  892,  939,  940,  976, 1022, 1023,  854,  855,  856,
        939,  940,  941, 1022,  856,  857,  940,  941,  942, 1022, 1024,
        857,  859,  941,  942,  944, 1024, 1025,  858,  891,  892,  943,
        975,  976, 1026,  859,  860,  942,  944,  945, 1025, 1027,  860,
        893,  944,  945,  977, 1027, 1028,  861,  894,  937,  946,  974,
        978, 1029,  864,  916,  920,  947, 1001, 1005, 1030,  865,  866,
        869,  948,  949,  952, 1031, 1034,  866,  867,  948,  949,  950,
       1031,  867,  868,  949,  950,  951, 1031, 1032,  868,  917,  950,
        951, 1002, 1032, 1033,  869,  918,  948,  952, 1003, 1034, 1035,
        870,  871,  953,  954,  955, 1036, 1037,  870,  872,  953,  954,
        956, 1036,  871,  873,  953,  955,  957, 1037, 1038,  872,  917,
        954,  956, 1002, 1036, 1039,  873,  893,  955,  957,  977, 1038,
       1040,  874,  877,  958,  959,  961, 1042, 1043,  874,  875,  958,
        959,  960, 1042, 1044,  875,  876,  880,  959,  960,  964, 1044,
       1045,  877,  878,  958,  961,  962, 1043, 1046,  878,  879,  961,
        962,  963, 1046, 1047,  879,  881,  936,  962,  963, 1020, 1047,
        880,  882,  960,  964,  965, 1045, 1048,  882,  883,  964,  965,
        966, 1048, 1049,  883,  935,  965,  966, 1019, 1049, 1050,  884,
        885,  967,  968,  969, 1051, 1052,  884,  888,  967,  968,  972,
       1051, 1053,  885,  886,  967,  969,  970, 1052, 1054,  886,  887,
        969,  970,  971, 1054, 1055,  887,  889,  970,  971,  973, 1055,
       1056,  888,  936,  968,  972, 1020, 1053, 1057,  889,  894,  971,
        973,  978, 1056, 1058,  890,  937,  938,  946,  974, 1021, 1029,
       1060,  891,  938,  943,  975, 1021, 1026, 1059,  892,  939,  943,
        976, 1023, 1026, 1061,  893,  945,  957,  977, 1028, 1040, 1062,
        894,  946,  973,  978, 1029, 1058, 1063,  895,  896,  979,  980,
        981, 1064, 1065,  895,  897,  979,  980,  982, 1065, 1066,  896,
        899,  979,  981,  984, 1064, 1067,  897,  898,  980,  982,  983,
       1066, 1068,  898,  915,  982,  983, 1000, 1068, 1069,  899,  912,
        981,  984,  997, 1067, 1070,  900,  901,  985,  986,  987, 1071,
       1072,  900,  902,  985,  986,  988, 1071, 1073,  901,  903,  985,
        987,  989, 1072, 1074,  902,  904,  986,  988,  990, 1073, 1075,
        903,  905,  987,  989,  991, 1074, 1076,  904,  909,  988,  990,
        995, 1075, 1077,  905,  906,  989,  991,  992, 1076, 1078,  906,
        907,  991,  992,  993, 1078, 1079,  907,  908,  992,  993,  994,
       1079, 1080,  908,  911,  993,  994,  997, 1080, 1081,  909,  910,
        990,  995,  996, 1077, 1082,  910,  913,  995,  996,  998, 1082,
       1083,  911,  912,  984,  994,  997, 1070, 1081,  913,  914,  996,
        998,  999, 1083, 1084,  914,  916,  998,  999, 1001, 1084, 1085,
        915,  933,  983, 1000, 1016, 1069, 1086,  916,  947,  999, 1001,
       1030, 1085, 1087,  917,  951,  956, 1002, 1033, 1039, 1088,  918,
        919,  952, 1003, 1004, 1035,  919,  920, 1003, 1004, 1005, 1035,
       1089, 1117,  920,  947, 1004, 1005, 1030, 1089, 1090,  921,  923,
       1006, 1007, 1009, 1091, 1092,  921,  922, 1006, 1007, 1008, 1092,
       1093,  922,  925, 1007, 1008, 1011, 1093, 1094,  923,  924, 1006,
       1009, 1010, 1091, 1095,  924,  928, 1009, 1010, 1013, 1095, 1096,
       1123,  925,  926,  927, 1008, 1011, 1041, 1094,  927,  929, 1012,
       1014, 1041, 1097, 1098,  928,  930, 1010, 1013, 1015, 1096,  929,
        932, 1012, 1014, 1017, 1097, 1099,  930,  931, 1013, 1015, 1016,
       1096, 1100,  931,  933, 1000, 1015, 1016, 1086, 1100,  932,  934,
       1014, 1017, 1018, 1099, 1101,  934,  935, 1017, 1018, 1019, 1101,
        935,  966, 1018, 1019, 1050, 1101, 1102,  936,  963,  972, 1020,
       1047, 1057, 1103,  938,  974,  975, 1021, 1059, 1060, 1104,  939,
        940,  941, 1022, 1023, 1024, 1105,  939,  976, 1022, 1023, 1061,
       1105, 1106,  941,  942, 1022, 1024, 1025, 1105, 1107,  942,  944,
       1024, 1025, 1027, 1107, 1108,  943,  975,  976, 1026, 1059, 1061,
       1109,  944,  945, 1025, 1027, 1028, 1108, 1110,  945,  977, 1027,
       1028, 1062, 1110, 1111,  946,  974,  978, 1029, 1060, 1063, 1112,
        947, 1001, 1005, 1030, 1087, 1090, 1113,  948,  949,  950, 1031,
       1032, 1034,  950,  951, 1031, 1032, 1033, 1034, 1114, 1116,  951,
       1002, 1032, 1033, 1088, 1114, 1115,  948,  952, 1031, 1032, 1034,
       1035, 1116,  952, 1003, 1004, 1034, 1035, 1116, 1117,  953,  954,
        956, 1036, 1037, 1039, 1118,  953,  955, 1036, 1037, 1038, 1118,
       1119,  955,  957, 1037, 1038, 1040, 1119, 1120,  956, 1002, 1036,
       1039, 1088, 1118, 1121,  957,  977, 1038, 1040, 1062, 1120, 1122,
        927, 1011, 1012, 1041, 1094, 1098, 1124,  958,  959, 1042, 1043,
       1044, 1125, 1126,  958,  961, 1042, 1043, 1046, 1125, 1127,  959,
        960, 1042, 1044, 1045, 1126, 1128,  960,  964, 1044, 1045, 1048,
       1128, 1129,  961,  962, 1043, 1046, 1047, 1127,  962,  963, 1020,
       1046, 1047, 1103, 1127,  964,  965, 1045, 1048, 1049, 1129, 1130,
        965,  966, 1048, 1049, 1050, 1130, 1131,  966, 1019, 1049, 1050,
       1102, 1131, 1132,  967,  968, 1051, 1052, 1053, 1133, 1134,  967,
        969, 1051, 1052, 1054, 1133, 1135,  968,  972, 1051, 1053, 1057,
       1134, 1136,  969,  970, 1052, 1054, 1055, 1135, 1137,  970,  971,
       1054, 1055, 1056, 1137, 1138,  971,  973, 1055, 1056, 1058, 1138,
       1139,  972, 1020, 1053, 1057, 1103, 1136, 1140,  973,  978, 1056,
       1058, 1063, 1139, 1141,  975, 1021, 1026, 1059, 1104, 1109, 1142,
        974, 1021, 1029, 1060, 1104, 1112, 1143,  976, 1023, 1026, 1061,
       1106, 1109, 1144,  977, 1028, 1040, 1062, 1111, 1122, 1145,  978,
       1029, 1058, 1063, 1112, 1141, 1146,  979,  981, 1064, 1065, 1067,
       1147, 1148,  979,  980, 1064, 1065, 1066, 1148, 1149,  980,  982,
       1065, 1066, 1068, 1149, 1150,  981,  984, 1064, 1067, 1070, 1147,
       1151,  982,  983, 1066, 1068, 1069, 1150, 1152,  983, 1000, 1068,
       1069, 1086, 1152, 1153,  984,  997, 1067, 1070, 1081, 1151, 1154,
        985,  986, 1071, 1072, 1073, 1155, 1156,  985,  987, 1071, 1072,
       1074, 1156, 1157,  986,  988, 1071, 1073, 1075, 1155, 1158,  987,
        989, 1072, 1074, 1076, 1157, 1159,  988,  990, 1073, 1075, 1077,
       1158, 1160,  989,  991, 1074, 1076, 1078, 1159, 1161,  990,  995,
       1075, 1077, 1082, 1160, 1162,  991,  992, 1076, 1078, 1079, 1161,
       1163,  992,  993, 1078, 1079, 1080, 1163, 1164,  993,  994, 1079,
       1080, 1081, 1164, 1165,  994,  997, 1070, 1080, 1081, 1154, 1165,
        995,  996, 1077, 1082, 1083, 1162, 1166,  996,  998, 1082, 1083,
       1084, 1166, 1167,  998,  999, 1083, 1084, 1085, 1167, 1168,  999,
       1001, 1084, 1085, 1087, 1168, 1169, 1000, 1016, 1069, 1086, 1100,
       1153, 1170, 1001, 1030, 1085, 1087, 1113, 1169, 1171, 1002, 1033,
       1039, 1088, 1115, 1121, 1172, 1173, 1004, 1005, 1089, 1090, 1117,
       1174, 1175, 1005, 1030, 1089, 1090, 1113, 1175, 1176, 1208, 1006,
       1009, 1091, 1092, 1095, 1177, 1178, 1006, 1007, 1091, 1092, 1093,
       1178, 1179, 1007, 1008, 1092, 1093, 1094, 1179, 1180, 1008, 1011,
       1041, 1093, 1094, 1124, 1180, 1009, 1010, 1091, 1095, 1123, 1177,
       1181, 1010, 1013, 1015, 1096, 1100, 1123, 1182, 1012, 1014, 1097,
       1098, 1099, 1183, 1184, 1012, 1041, 1097, 1098, 1124, 1183, 1185,
       1014, 1017, 1097, 1099, 1101, 1184, 1186, 1015, 1016, 1086, 1096,
       1100, 1170, 1182, 1017, 1018, 1019, 1099, 1101, 1102, 1186, 1187,
       1019, 1050, 1101, 1102, 1132, 1187, 1188, 1020, 1047, 1057, 1103,
       1127, 1140, 1189, 1021, 1059, 1060, 1104, 1142, 1143, 1190, 1022,
       1023, 1024, 1105, 1106, 1107, 1191, 1023, 1061, 1105, 1106, 1144,
       1191, 1192, 1024, 1025, 1105, 1107, 1108, 1191, 1193, 1025, 1027,
       1107, 1108, 1110, 1193, 1194, 1026, 1059, 1061, 1109, 1142, 1144,
       1195, 1027, 1028, 1108, 1110, 1111, 1194, 1196, 1028, 1062, 1110,
       1111, 1145, 1196, 1197, 1029, 1060, 1063, 1112, 1143, 1146, 1198,
       1030, 1087, 1090, 1113, 1171, 1176, 1199, 1032, 1033, 1114, 1115,
       1116, 1200, 1201, 1033, 1088, 1114, 1115, 1172, 1200, 1202, 1032,
       1034, 1035, 1114, 1116, 1117, 1201, 1004, 1035, 1089, 1116, 1117,
       1174, 1201, 1036, 1037, 1039, 1118, 1119, 1121, 1203, 1037, 1038,
       1118, 1119, 1120, 1203, 1204, 1038, 1040, 1119, 1120, 1122, 1204,
       1205, 1039, 1088, 1118, 1121, 1173, 1203, 1206, 1040, 1062, 1120,
       1122, 1145, 1205, 1207, 1010, 1095, 1096, 1123, 1181, 1182, 1209,
       1041, 1094, 1098, 1124, 1180, 1185, 1210, 1042, 1043, 1125, 1126,
       1127, 1189, 1211, 1042, 1044, 1125, 1126, 1128, 1211, 1212, 1043,
       1046, 1047, 1103, 1125, 1127, 1189, 1044, 1045, 1126, 1128, 1129,
       1212, 1213, 1045, 1048, 1128, 1129, 1130, 1213, 1214, 1048, 1049,
       1129, 1130, 1131, 1214, 1215, 1049, 1050, 1130, 1131, 1132, 1215,
       1216, 1050, 1102, 1131, 1132, 1188, 1216, 1217, 1051, 1052, 1133,
       1134, 1135, 1218, 1219, 1051, 1053, 1133, 1134, 1136, 1218, 1220,
       1052, 1054, 1133, 1135, 1137, 1219, 1221, 1053, 1057, 1134, 1136,
       1140, 1220, 1222, 1054, 1055, 1135, 1137, 1138, 1221, 1223, 1055,
       1056, 1137, 1138, 1139, 1223, 1224, 1056, 1058, 1138, 1139, 1141,
       1224, 1225, 1057, 1103, 1136, 1140, 1189, 1222, 1226, 1058, 1063,
       1139, 1141, 1146, 1225, 1227, 1059, 1104, 1109, 1142, 1190, 1195,
       1228, 1060, 1104, 1112, 1143, 1190, 1198, 1229, 1061, 1106, 1109,
       1144, 1192, 1195, 1230, 1062, 1111, 1122, 1145, 1197, 1207, 1231,
       1063, 1112, 1141, 1146, 1198, 1227, 1232, 1064, 1067, 1147, 1148,
       1151, 1233, 1234, 1064, 1065, 1147, 1148, 1149, 1234, 1235, 1065,
       1066, 1148, 1149, 1150, 1235, 1066, 1068, 1149, 1150, 1152, 1235,
       1236, 1283, 1067, 1070, 1147, 1151, 1154, 1233, 1237, 1068, 1069,
       1150, 1152, 1153, 1236, 1238, 1069, 1086, 1152, 1153, 1170, 1238,
       1239, 1070, 1081, 1151, 1154, 1165, 1237, 1240, 1071, 1073, 1155,
       1156, 1158, 1241, 1242, 1071, 1072, 1155, 1156, 1157, 1242, 1243,
       1072, 1074, 1156, 1157, 1159, 1243, 1244, 1073, 1075, 1155, 1158,
       1160, 1241, 1245, 1074, 1076, 1157, 1159, 1161, 1244, 1246, 1075,
       1077, 1158, 1160, 1162, 1245, 1247, 1076, 1078, 1159, 1161, 1163,
       1246, 1248, 1077, 1082, 1160, 1162, 1166, 1247, 1249, 1078, 1079,
       1161, 1163, 1164, 1248, 1250, 1079, 1080, 1163, 1164, 1165, 1250,
       1251, 1080, 1081, 1154, 1164, 1165, 1240, 1251, 1082, 1083, 1162,
       1166, 1167, 1249, 1252, 1083, 1084, 1166, 1167, 1168, 1252, 1253,
       1084, 1085, 1167, 1168, 1169, 1253, 1254, 1284, 1085, 1087, 1168,
       1169, 1171, 1254, 1086, 1100, 1153, 1170, 1182, 1239, 1255, 1087,
       1113, 1169, 1171, 1199, 1254, 1256, 1088, 1115, 1172, 1173, 1202,
       1257, 1088, 1121, 1172, 1173, 1206, 1257, 1258, 1089, 1117, 1174,
       1175, 1201, 1259, 1260, 1089, 1090, 1174, 1175, 1208, 1259, 1090,
       1113, 1176, 1199, 1208, 1261, 1262, 1091, 1095, 1177, 1178, 1181,
       1263, 1264, 1091, 1092, 1177, 1178, 1179, 1263, 1265, 1092, 1093,
       1178, 1179, 1180, 1265, 1266, 1093, 1094, 1124, 1179, 1180, 1210,
       1266, 1095, 1123, 1177, 1181, 1209, 1264, 1267, 1096, 1100, 1123,
       1170, 1182, 1209, 1255, 1097, 1098, 1183, 1184, 1185, 1268, 1269,
       1097, 1099, 1183, 1184, 1186, 1269, 1270, 1098, 1124, 1183, 1185,
       1210, 1268, 1271, 1295, 1099, 1101, 1184, 1186, 1187, 1270, 1272,
       1101, 1102, 1186, 1187, 1188, 1272, 1273, 1102, 1132, 1187, 1188,
       1217, 1273, 1274, 1103, 1125, 1127, 1140, 1189, 1211, 1226, 1104,
       1142, 1143, 1190, 1228, 1229, 1275, 1105, 1106, 1107, 1191, 1192,
       1193, 1276, 1106, 1144, 1191, 1192, 1230, 1276, 1277, 1107, 1108,
       1191, 1193, 1194, 1276, 1278, 1108, 1110, 1193, 1194, 1196, 1278,
       1279, 1109, 1142, 1144, 1195, 1228, 1230, 1110, 1111, 1194, 1196,
       1197, 1279, 1280, 1111, 1145, 1196, 1197, 1231, 1280, 1281, 1112,
       1143, 1146, 1198, 1229, 1232, 1282, 1113, 1171, 1176, 1199, 1256,
       1262, 1285, 1114, 1115, 1200, 1201, 1202, 1260, 1286, 1114, 1116,
       1117, 1174, 1200, 1201, 1260, 1115, 1172, 1200, 1202, 1257, 1286,
       1287, 1118, 1119, 1121, 1203, 1204, 1206, 1288, 1119, 1120, 1203,
       1204, 1205, 1288, 1289, 1120, 1122, 1204, 1205, 1207, 1289, 1290,
       1121, 1173, 1203, 1206, 1258, 1288, 1291, 1122, 1145, 1205, 1207,
       1231, 1290, 1292, 1090, 1175, 1176, 1208, 1259, 1261, 1293, 1123,
       1181, 1182, 1209, 1255, 1267, 1294, 1124, 1180, 1185, 1210, 1266,
       1271, 1125, 1126, 1189, 1211, 1212, 1226, 1296, 1126, 1128, 1211,
       1212, 1213, 1296, 1297, 1128, 1129, 1212, 1213, 1214, 1297, 1298,
       1129, 1130, 1213, 1214, 1215, 1298, 1299, 1130, 1131, 1214, 1215,
       1216, 1299, 1300, 1131, 1132, 1215, 1216, 1217, 1300, 1301, 1132,
       1188, 1216, 1217, 1274, 1301, 1302, 1133, 1134, 1218, 1219, 1220,
       1303, 1304, 1133, 1135, 1218, 1219, 1221, 1304, 1305, 1134, 1136,
       1218, 1220, 1222, 1303, 1306, 1135, 1137, 1219, 1221, 1223, 1305,
       1307, 1136, 1140, 1220, 1222, 1226, 1306, 1308, 1137, 1138, 1221,
       1223, 1224, 1307, 1309, 1138, 1139, 1223, 1224, 1225, 1309, 1310,
       1139, 1141, 1224, 1225, 1227, 1310, 1311, 1140, 1189, 1211, 1222,
       1226, 1296, 1308, 1141, 1146, 1225, 1227, 1232, 1311, 1142, 1190,
       1195, 1228, 1230, 1275, 1312, 1143, 1190, 1198, 1229, 1275, 1282,
       1313, 1144, 1192, 1195, 1228, 1230, 1277, 1312, 1314, 1145, 1197,
       1207, 1231, 1281, 1292, 1315, 1146, 1198, 1227, 1232, 1282, 1311,
       1316, 1358, 1147, 1151, 1233, 1234, 1237, 1317, 1318, 1147, 1148,
       1233, 1234, 1235, 1317, 1319, 1148, 1149, 1150, 1234, 1235, 1283,
       1319, 1150, 1152, 1236, 1238, 1283, 1320, 1321, 1151, 1154, 1233,
       1237, 1240, 1318, 1322, 1152, 1153, 1236, 1238, 1239, 1320, 1323,
       1153, 1170, 1238, 1239, 1255, 1323, 1324, 1154, 1165, 1237, 1240,
       1251, 1322, 1325, 1155, 1158, 1241, 1242, 1245, 1326, 1327, 1155,
       1156, 1241, 1242, 1243, 1327, 1328, 1156, 1157, 1242, 1243, 1244,
       1328, 1329, 1157, 1159, 1243, 1244, 1246, 1329, 1330, 1158, 1160,
       1241, 1245, 1247, 1326, 1331, 1159, 1161, 1244, 1246, 1248, 1330,
       1332, 1160, 1162, 1245, 1247, 1249, 1331, 1333, 1161, 1163, 1246,
       1248, 1250, 1332, 1334, 1162, 1166, 1247, 1249, 1252, 1333, 1335,
       1163, 1164, 1248, 1250, 1251, 1334, 1336, 1164, 1165, 1240, 1250,
       1251, 1325, 1336, 1166, 1167, 1249, 1252, 1253, 1335, 1337, 1167,
       1168, 1252, 1253, 1284, 1337, 1168, 1169, 1171, 1254, 1256, 1284,
       1338, 1170, 1182, 1209, 1239, 1255, 1294, 1324, 1171, 1199, 1254,
       1256, 1285, 1338, 1339, 1172, 1173, 1202, 1257, 1258, 1287, 1340,
       1173, 1206, 1257, 1258, 1291, 1340, 1341, 1174, 1175, 1208, 1259,
       1260, 1293, 1342, 1174, 1200, 1201, 1259, 1260, 1286, 1342, 1370,
       1176, 1208, 1261, 1262, 1293, 1343, 1176, 1199, 1261, 1262, 1285,
       1343, 1344, 1379, 1177, 1178, 1263, 1264, 1265, 1345, 1346, 1177,
       1181, 1263, 1264, 1267, 1346, 1347, 1178, 1179, 1263, 1265, 1266,
       1345, 1348, 1179, 1180, 1210, 1265, 1266, 1271, 1348, 1181, 1209,
       1264, 1267, 1294, 1347, 1349, 1183, 1185, 1268, 1269, 1295, 1350,
       1351, 1183, 1184, 1268, 1269, 1270, 1350, 1352, 1184, 1186, 1269,
       1270, 1272, 1352, 1353, 1185, 1210, 1266, 1271, 1295, 1348, 1354,
       1186, 1187, 1270, 1272, 1273, 1353, 1355, 1187, 1188, 1272, 1273,
       1274, 1355, 1356, 1188, 1217, 1273, 1274, 1302, 1356, 1357, 1190,
       1228, 1229, 1275, 1312, 1313, 1359, 1191, 1192, 1193, 1276, 1277,
       1278, 1360, 1192, 1230, 1276, 1277, 1314, 1360, 1361, 1193, 1194,
       1276, 1278, 1279, 1360, 1362, 1194, 1196, 1278, 1279, 1280, 1362,
       1363, 1196, 1197, 1279, 1280, 1281, 1363, 1364, 1197, 1231, 1280,
       1281, 1315, 1364, 1365, 1198, 1229, 1232, 1282, 1313, 1316, 1366,
       1150, 1235, 1236, 1283, 1319, 1321, 1367, 1168, 1253, 1254, 1284,
       1337, 1338, 1368, 1199, 1256, 1262, 1285, 1339, 1344, 1369, 1200,
       1202, 1260, 1286, 1287, 1370, 1371, 1202, 1257, 1286, 1287, 1340,
       1371, 1372, 1203, 1204, 1206, 1288, 1289, 1291, 1373, 1204, 1205,
       1288, 1289, 1290, 1373, 1374, 1205, 1207, 1289, 1290, 1292, 1374,
       1375, 1206, 1258, 1288, 1291, 1341, 1373, 1376, 1427, 1207, 1231,
       1290, 1292, 1315, 1375, 1377, 1208, 1259, 1261, 1293, 1342, 1343,
       1378, 1209, 1255, 1267, 1294, 1324, 1349, 1380, 1185, 1268, 1271,
       1295, 1351, 1354, 1381, 1211, 1212, 1226, 1296, 1297, 1308, 1382,
       1212, 1213, 1296, 1297, 1298, 1382, 1383, 1213, 1214, 1297, 1298,
       1299, 1383, 1384, 1214, 1215, 1298, 1299, 1300, 1384, 1385, 1215,
       1216, 1299, 1300, 1301, 1385, 1386, 1216, 1217, 1300, 1301, 1302,
       1386, 1387, 1217, 1274, 1301, 1302, 1357, 1387, 1388, 1218, 1220,
       1303, 1304, 1306, 1389, 1390, 1218, 1219, 1303, 1304, 1305, 1390,
       1391, 1219, 1221, 1304, 1305, 1307, 1391, 1392, 1220, 1222, 1303,
       1306, 1308, 1389, 1393, 1221, 1223, 1305, 1307, 1309, 1392, 1394,
       1222, 1226, 1296, 1306, 1308, 1382, 1393, 1223, 1224, 1307, 1309,
       1310, 1394, 1395, 1224, 1225, 1309, 1310, 1311, 1395, 1396, 1225,
       1227, 1232, 1310, 1311, 1358, 1396, 1228, 1230, 1275, 1312, 1314,
       1359, 1397, 1229, 1275, 1282, 1313, 1359, 1366, 1398, 1230, 1277,
       1312, 1314, 1361, 1397, 1399, 1231, 1281, 1292, 1315, 1365, 1377,
       1400, 1232, 1282, 1316, 1358, 1366, 1401, 1402, 1233, 1234, 1317,
       1318, 1319, 1403, 1404, 1233, 1237, 1317, 1318, 1322, 1403, 1405,
       1234, 1235, 1283, 1317, 1319, 1367, 1404, 1236, 1238, 1320, 1321,
       1323, 1406, 1407, 1236, 1283, 1320, 1321, 1367, 1406, 1408, 1237,
       1240, 1318, 1322, 1325, 1405, 1409, 1238, 1239, 1320, 1323, 1324,
       1407, 1410, 1239, 1255, 1294, 1323, 1324, 1380, 1410, 1240, 1251,
       1322, 1325, 1336, 1409, 1411, 1241, 1245, 1326, 1327, 1331, 1412,
       1413, 1241, 1242, 1326, 1327, 1328, 1412, 1414, 1242, 1243, 1327,
       1328, 1329, 1414, 1415, 1243, 1244, 1328, 1329, 1330, 1415, 1416,
       1244, 1246, 1329, 1330, 1332, 1416, 1417, 1245, 1247, 1326, 1331,
       1333, 1413, 1418, 1246, 1248, 1330, 1332, 1334, 1417, 1419, 1247,
       1249, 1331, 1333, 1335, 1418, 1420, 1248, 1250, 1332, 1334, 1336,
       1419, 1421, 1249, 1252, 1333, 1335, 1337, 1420, 1422, 1250, 1251,
       1325, 1334, 1336, 1411, 1421, 1252, 1253, 1284, 1335, 1337, 1368,
       1422, 1423, 1254, 1256, 1284, 1338, 1339, 1368, 1424, 1256, 1285,
       1338, 1339, 1369, 1424, 1425, 1257, 1258, 1287, 1340, 1341, 1372,
       1426, 1258, 1291, 1340, 1341, 1376, 1426, 1428, 1259, 1260, 1293,
       1342, 1370, 1378, 1429, 1261, 1262, 1293, 1343, 1378, 1379, 1430,
       1262, 1285, 1344, 1369, 1379, 1431, 1263, 1265, 1345, 1346, 1348,
       1432, 1433, 1263, 1264, 1345, 1346, 1347, 1432, 1434, 1264, 1267,
       1346, 1347, 1349, 1434, 1435, 1265, 1266, 1271, 1345, 1348, 1354,
       1433, 1267, 1294, 1347, 1349, 1380, 1435, 1436, 1268, 1269, 1350,
       1351, 1352, 1437, 1438, 1268, 1295, 1350, 1351, 1381, 1437, 1439,
       1269, 1270, 1350, 1352, 1353, 1438, 1440, 1270, 1272, 1352, 1353,
       1355, 1440, 1441, 1271, 1295, 1348, 1354, 1381, 1433, 1442, 1272,
       1273, 1353, 1355, 1356, 1441, 1273, 1274, 1355, 1356, 1357, 1441,
       1443, 1274, 1302, 1356, 1357, 1388, 1443, 1444, 1232, 1311, 1316,
       1358, 1396, 1401, 1445, 1275, 1312, 1313, 1359, 1397, 1398, 1446,
       1276, 1277, 1278, 1360, 1361, 1362, 1447, 1277, 1314, 1360, 1361,
       1399, 1447, 1448, 1278, 1279, 1360, 1362, 1363, 1447, 1449, 1279,
       1280, 1362, 1363, 1364, 1449, 1450, 1280, 1281, 1363, 1364, 1365,
       1450, 1451, 1281, 1315, 1364, 1365, 1400, 1451, 1452, 1282, 1313,
       1316, 1366, 1398, 1402, 1453, 1283, 1319, 1321, 1367, 1404, 1408,
       1454, 1284, 1337, 1338, 1368, 1423, 1424, 1455, 1285, 1339, 1344,
       1369, 1425, 1431, 1456, 1468, 1260, 1286, 1342, 1370, 1371, 1429,
       1457, 1286, 1287, 1370, 1371, 1372, 1457, 1458, 1287, 1340, 1371,
       1372, 1426, 1458, 1459, 1288, 1289, 1291, 1373, 1374, 1427, 1460,
       1289, 1290, 1373, 1374, 1375, 1460, 1461, 1290, 1292, 1374, 1375,
       1377, 1461, 1462, 1291, 1341, 1376, 1427, 1428, 1463, 1464, 1292,
       1315, 1375, 1377, 1400, 1462, 1465, 1293, 1342, 1343, 1378, 1429,
       1430, 1466, 1262, 1343, 1344, 1379, 1430, 1431, 1467, 1294, 1324,
       1349, 1380, 1410, 1436, 1469, 1295, 1351, 1354, 1381, 1439, 1442,
       1470, 1296, 1297, 1308, 1382, 1383, 1393, 1471, 1297, 1298, 1382,
       1383, 1384, 1471, 1472, 1298, 1299, 1383, 1384, 1385, 1472, 1473,
       1299, 1300, 1384, 1385, 1386, 1473, 1474, 1300, 1301, 1385, 1386,
       1387, 1474, 1301, 1302, 1386, 1387, 1388, 1474, 1475, 1531, 1302,
       1357, 1387, 1388, 1444, 1475, 1476, 1303, 1306, 1389, 1390, 1393,
       1477, 1478, 1303, 1304, 1389, 1390, 1391, 1478, 1479, 1304, 1305,
       1390, 1391, 1392, 1479, 1480, 1305, 1307, 1391, 1392, 1394, 1480,
       1481, 1306, 1308, 1382, 1389, 1393, 1471, 1477, 1307, 1309, 1392,
       1394, 1395, 1481, 1482, 1309, 1310, 1394, 1395, 1396, 1482, 1483,
       1310, 1311, 1358, 1395, 1396, 1445, 1483, 1312, 1314, 1359, 1397,
       1399, 1446, 1484, 1313, 1359, 1366, 1398, 1446, 1453, 1485, 1314,
       1361, 1397, 1399, 1448, 1484, 1486, 1536, 1315, 1365, 1377, 1400,
       1452, 1465, 1487, 1316, 1358, 1401, 1402, 1445, 1488, 1489, 1316,
       1366, 1401, 1402, 1453, 1489, 1490, 1317, 1318, 1403, 1404, 1405,
       1491, 1492, 1317, 1319, 1367, 1403, 1404, 1454, 1492, 1318, 1322,
       1403, 1405, 1409, 1491, 1493, 1320, 1321, 1406, 1407, 1408, 1494,
       1495, 1320, 1323, 1406, 1407, 1410, 1495, 1496, 1321, 1367, 1406,
       1408, 1454, 1494, 1497, 1322, 1325, 1405, 1409, 1411, 1493, 1498,
       1323, 1324, 1380, 1407, 1410, 1469, 1496, 1325, 1336, 1409, 1411,
       1421, 1498, 1499, 1326, 1327, 1412, 1413, 1414, 1500, 1501, 1326,
       1331, 1412, 1413, 1418, 1500, 1502, 1327, 1328, 1412, 1414, 1415,
       1501, 1503, 1328, 1329, 1414, 1415, 1416, 1503, 1504, 1329, 1330,
       1415, 1416, 1417, 1504, 1505, 1330, 1332, 1416, 1417, 1419, 1505,
       1506, 1331, 1333, 1413, 1418, 1420, 1502, 1507, 1332, 1334, 1417,
       1419, 1421, 1506, 1508, 1333, 1335, 1418, 1420, 1422, 1507, 1509,
       1334, 1336, 1411, 1419, 1421, 1499, 1508, 1335, 1337, 1420, 1422,
       1423, 1509, 1510, 1337, 1368, 1422, 1423, 1455, 1510, 1511, 1338,
       1339, 1368, 1424, 1425, 1455, 1512, 1339, 1369, 1424, 1425, 1456,
       1512, 1513, 1340, 1341, 1372, 1426, 1428, 1459, 1514, 1291, 1373,
       1376, 1427, 1460, 1463, 1515, 1341, 1376, 1426, 1428, 1464, 1514,
       1516, 1550, 1342, 1370, 1378, 1429, 1457, 1466, 1517, 1343, 1378,
       1379, 1430, 1466, 1467, 1518, 1344, 1369, 1379, 1431, 1467, 1468,
       1519, 1345, 1346, 1432, 1433, 1434, 1520, 1521, 1345, 1348, 1354,
       1432, 1433, 1442, 1521, 1346, 1347, 1432, 1434, 1435, 1520, 1522,
       1347, 1349, 1434, 1435, 1436, 1522, 1523, 1349, 1380, 1435, 1436,
       1469, 1523, 1524, 1350, 1351, 1437, 1438, 1439, 1525, 1526, 1350,
       1352, 1437, 1438, 1440, 1526, 1527, 1351, 1381, 1437, 1439, 1470,
       1525, 1528, 1352, 1353, 1438, 1440, 1441, 1527, 1529, 1353, 1355,
       1356, 1440, 1441, 1443, 1529, 1354, 1381, 1433, 1442, 1470, 1521,
       1356, 1357, 1441, 1443, 1444, 1529, 1530, 1357, 1388, 1443, 1444,
       1476, 1530, 1532, 1358, 1396, 1401, 1445, 1483, 1488, 1533, 1359,
       1397, 1398, 1446, 1484, 1485, 1534, 1360, 1361, 1362, 1447, 1448,
       1449, 1535, 1361, 1399, 1447, 1448, 1486, 1535, 1537, 1362, 1363,
       1447, 1449, 1450, 1535, 1538, 1363, 1364, 1449, 1450, 1451, 1538,
       1539, 1364, 1365, 1450, 1451, 1452, 1539, 1540, 1365, 1400, 1451,
       1452, 1487, 1540, 1541, 1366, 1398, 1402, 1453, 1485, 1490, 1542,
       1367, 1404, 1408, 1454, 1492, 1497, 1543, 1368, 1423, 1424, 1455,
       1511, 1512, 1544, 1369, 1425, 1456, 1468, 1513, 1545, 1546, 1370,
       1371, 1429, 1457, 1458, 1517, 1547, 1371, 1372, 1457, 1458, 1459,
       1547, 1548, 1372, 1426, 1458, 1459, 1514, 1548, 1549, 1373, 1374,
       1427, 1460, 1461, 1515, 1551, 1374, 1375, 1460, 1461, 1462, 1551,
       1552, 1375, 1377, 1461, 1462, 1465, 1552, 1553, 1376, 1427, 1463,
       1464, 1515, 1554, 1555, 1376, 1428, 1463, 1464, 1516, 1555, 1556,
       1377, 1400, 1462, 1465, 1487, 1553, 1557, 1378, 1429, 1430, 1466,
       1517, 1518, 1558, 1379, 1430, 1431, 1467, 1518, 1519, 1559, 1369,
       1431, 1456, 1468, 1519, 1546, 1560, 1380, 1410, 1436, 1469, 1496,
       1524, 1561, 1381, 1439, 1442, 1470, 1521, 1528, 1562, 1563, 1382,
       1383, 1393, 1471, 1472, 1477, 1564, 1383, 1384, 1471, 1472, 1473,
       1564, 1565, 1384, 1385, 1472, 1473, 1474, 1565, 1566, 1385, 1386,
       1387, 1473, 1474, 1531, 1566, 1387, 1388, 1475, 1476, 1531, 1567,
       1568, 1388, 1444, 1475, 1476, 1532, 1567, 1569, 1389, 1393, 1471,
       1477, 1478, 1564, 1570, 1389, 1390, 1477, 1478, 1479, 1570, 1571,
       1390, 1391, 1478, 1479, 1480, 1571, 1572, 1391, 1392, 1479, 1480,
       1481, 1572, 1573, 1392, 1394, 1480, 1481, 1482, 1573, 1574, 1394,
       1395, 1481, 1482, 1483, 1574, 1575, 1395, 1396, 1445, 1482, 1483,
       1533, 1575, 1397, 1399, 1446, 1484, 1534, 1536, 1398, 1446, 1453,
       1485, 1534, 1542, 1576, 1399, 1448, 1486, 1536, 1537, 1577, 1578,
       1400, 1452, 1465, 1487, 1541, 1557, 1579, 1401, 1445, 1488, 1489,
       1533, 1580, 1581, 1401, 1402, 1488, 1489, 1490, 1581, 1582, 1402,
       1453, 1489, 1490, 1542, 1582, 1583, 1403, 1405, 1491, 1492, 1493,
       1584, 1585, 1403, 1404, 1454, 1491, 1492, 1543, 1585, 1405, 1409,
       1491, 1493, 1498, 1584, 1586, 1406, 1408, 1494, 1495, 1497, 1587,
       1588, 1406, 1407, 1494, 1495, 1496, 1587, 1589, 1407, 1410, 1469,
       1495, 1496, 1561, 1589, 1408, 1454, 1494, 1497, 1543, 1588, 1590,
       1409, 1411, 1493, 1498, 1499, 1586, 1591, 1411, 1421, 1498, 1499,
       1508, 1591, 1592, 1412, 1413, 1500, 1501, 1502, 1593, 1594, 1412,
       1414, 1500, 1501, 1503, 1594, 1595, 1413, 1418, 1500, 1502, 1507,
       1593, 1596, 1414, 1415, 1501, 1503, 1504, 1595, 1597, 1415, 1416,
       1503, 1504, 1505, 1597, 1598, 1416, 1417, 1504, 1505, 1506, 1598,
       1599, 1417, 1419, 1505, 1506, 1508, 1599, 1600, 1418, 1420, 1502,
       1507, 1509, 1596, 1601, 1419, 1421, 1499, 1506, 1508, 1592, 1600,
       1636, 1420, 1422, 1507, 1509, 1510, 1601, 1602, 1422, 1423, 1509,
       1510, 1511, 1602, 1603, 1423, 1455, 1510, 1511, 1544, 1603, 1604,
       1424, 1425, 1455, 1512, 1513, 1544, 1605, 1425, 1456, 1512, 1513,
       1545, 1605, 1606, 1426, 1428, 1459, 1514, 1549, 1550, 1607, 1427,
       1460, 1463, 1515, 1551, 1554, 1608, 1428, 1464, 1516, 1550, 1556,
       1609, 1610, 1429, 1457, 1466, 1517, 1547, 1558, 1611, 1430, 1466,
       1467, 1518, 1558, 1559, 1612, 1431, 1467, 1468, 1519, 1559, 1560,
       1613, 1432, 1434, 1520, 1521, 1522, 1563, 1614, 1432, 1433, 1442,
       1470, 1520, 1521, 1563, 1434, 1435, 1520, 1522, 1523, 1614, 1615,
       1435, 1436, 1522, 1523, 1524, 1615, 1616, 1436, 1469, 1523, 1524,
       1561, 1616, 1617, 1437, 1439, 1525, 1526, 1528, 1618, 1619, 1437,
       1438, 1525, 1526, 1527, 1619, 1620, 1438, 1440, 1526, 1527, 1529,
       1620, 1621, 1439, 1470, 1525, 1528, 1562, 1618, 1622, 1440, 1441,
       1443, 1527, 1529, 1530, 1621, 1443, 1444, 1529, 1530, 1532, 1621,
       1623, 1387, 1474, 1475, 1531, 1566, 1568, 1624, 1444, 1476, 1530,
       1532, 1569, 1623, 1625, 1445, 1483, 1488, 1533, 1575, 1580, 1626,
       1446, 1484, 1485, 1534, 1536, 1576, 1627, 1447, 1448, 1449, 1535,
       1537, 1538, 1628, 1399, 1484, 1486, 1534, 1536, 1577, 1627, 1448,
       1486, 1535, 1537, 1578, 1628, 1629, 1449, 1450, 1535, 1538, 1539,
       1628, 1630, 1450, 1451, 1538, 1539, 1540, 1630, 1631, 1451, 1452,
       1539, 1540, 1541, 1631, 1632, 1452, 1487, 1540, 1541, 1579, 1632,
       1633, 1453, 1485, 1490, 1542, 1576, 1583, 1634, 1454, 1492, 1497,
       1543, 1585, 1590, 1635, 1455, 1511, 1512, 1544, 1604, 1605, 1456,
       1513, 1545, 1546, 1606, 1637, 1638, 1698, 1456, 1468, 1545, 1546,
       1560, 1637, 1639, 1697, 1700, 1457, 1458, 1517, 1547, 1548, 1611,
       1640, 1458, 1459, 1547, 1548, 1549, 1640, 1641, 1459, 1514, 1548,
       1549, 1607, 1641, 1428, 1514, 1516, 1550, 1607, 1610, 1642, 1460,
       1461, 1515, 1551, 1552, 1608, 1643, 1461, 1462, 1551, 1552, 1553,
       1643, 1644, 1462, 1465, 1552, 1553, 1557, 1644, 1645, 1463, 1515,
       1554, 1555, 1608, 1646, 1647, 1703, 1463, 1464, 1554, 1555, 1556,
       1647, 1648, 1464, 1516, 1555, 1556, 1609, 1648, 1649, 1465, 1487,
       1553, 1557, 1579, 1645, 1650, 1466, 1517, 1518, 1558, 1611, 1612,
       1651, 1467, 1518, 1519, 1559, 1612, 1613, 1652, 1468, 1519, 1546,
       1560, 1613, 1639, 1469, 1496, 1524, 1561, 1589, 1617, 1653, 1470,
       1528, 1562, 1563, 1622, 1654, 1655, 1470, 1520, 1521, 1562, 1563,
       1614, 1654, 1471, 1472, 1477, 1564, 1565, 1570, 1656, 1472, 1473,
       1564, 1565, 1566, 1656, 1657, 1473, 1474, 1531, 1565, 1566, 1624,
       1657, 1475, 1476, 1567, 1568, 1569, 1658, 1659, 1475, 1531, 1567,
       1568, 1624, 1659, 1660, 1476, 1532, 1567, 1569, 1625, 1658, 1661,
       1477, 1478, 1564, 1570, 1571, 1656, 1662, 1478, 1479, 1570, 1571,
       1572, 1662, 1663, 1479, 1480, 1571, 1572, 1573, 1663, 1664, 1480,
       1481, 1572, 1573, 1574, 1664, 1665, 1481, 1482, 1573, 1574, 1575,
       1665, 1666, 1482, 1483, 1533, 1574, 1575, 1626, 1666, 1485, 1534,
       1542, 1576, 1627, 1634, 1667, 1486, 1536, 1577, 1578, 1627, 1668,
       1669, 1486, 1537, 1577, 1578, 1629, 1668, 1670, 1487, 1541, 1557,
       1579, 1633, 1650, 1671, 1488, 1533, 1580, 1581, 1626, 1672, 1673,
       1488, 1489, 1580, 1581, 1582, 1673, 1674, 1489, 1490, 1581, 1582,
       1583, 1674, 1675, 1490, 1542, 1582, 1583, 1634, 1675, 1676, 1491,
       1493, 1584, 1585, 1586, 1677, 1678, 1491, 1492, 1543, 1584, 1585,
       1635, 1678, 1493, 1498, 1584, 1586, 1591, 1677, 1679, 1494, 1495,
       1587, 1588, 1589, 1680, 1681, 1494, 1497, 1587, 1588, 1590, 1680,
       1682, 1495, 1496, 1561, 1587, 1589, 1653, 1681, 1497, 1543, 1588,
       1590, 1635, 1682, 1683, 1498, 1499, 1586, 1591, 1592, 1679, 1684,
       1729, 1499, 1508, 1591, 1592, 1636, 1684, 1685, 1500, 1502, 1593,
       1594, 1596, 1686, 1687, 1500, 1501, 1593, 1594, 1595, 1687, 1688,
       1501, 1503, 1594, 1595, 1597, 1688, 1689, 1502, 1507, 1593, 1596,
       1601, 1686, 1690, 1503, 1504, 1595, 1597, 1598, 1689, 1691, 1504,
       1505, 1597, 1598, 1599, 1691, 1692, 1505, 1506, 1598, 1599, 1600,
       1692, 1693, 1506, 1508, 1599, 1600, 1636, 1693, 1507, 1509, 1596,
       1601, 1602, 1690, 1694, 1732, 1509, 1510, 1601, 1602, 1603, 1694,
       1510, 1511, 1602, 1603, 1604, 1694, 1695, 1511, 1544, 1603, 1604,
       1605, 1695, 1696, 1733, 1512, 1513, 1544, 1604, 1605, 1606, 1696,
       1513, 1545, 1605, 1606, 1638, 1696, 1699, 1514, 1549, 1550, 1607,
       1641, 1642, 1701, 1515, 1551, 1554, 1608, 1643, 1646, 1702, 1516,
       1556, 1609, 1610, 1649, 1704, 1705, 1516, 1550, 1609, 1610, 1642,
       1705, 1706, 1517, 1547, 1558, 1611, 1640, 1651, 1518, 1558, 1559,
       1612, 1651, 1652, 1707, 1519, 1559, 1560, 1613, 1639, 1652, 1708,
       1520, 1522, 1563, 1614, 1615, 1654, 1709, 1522, 1523, 1614, 1615,
       1616, 1709, 1710, 1523, 1524, 1615, 1616, 1617, 1710, 1711, 1524,
       1561, 1616, 1617, 1653, 1711, 1712, 1525, 1528, 1618, 1619, 1622,
       1713, 1714, 1525, 1526, 1618, 1619, 1620, 1714, 1715, 1526, 1527,
       1619, 1620, 1621, 1715, 1716, 1527, 1529, 1530, 1620, 1621, 1623,
       1716, 1528, 1562, 1618, 1622, 1655, 1713, 1717, 1530, 1532, 1621,
       1623, 1625, 1716, 1718, 1531, 1566, 1568, 1624, 1657, 1660, 1719,
       1532, 1569, 1623, 1625, 1661, 1718, 1720, 1533, 1575, 1580, 1626,
       1666, 1672, 1721, 1534, 1536, 1576, 1577, 1627, 1667, 1669, 1535,
       1537, 1538, 1628, 1629, 1630, 1722, 1537, 1578, 1628, 1629, 1670,
       1722, 1723, 1538, 1539, 1628, 1630, 1631, 1722, 1724, 1539, 1540,
       1630, 1631, 1632, 1724, 1725, 1540, 1541, 1631, 1632, 1633, 1725,
       1726, 1541, 1579, 1632, 1633, 1671, 1726, 1727, 1542, 1576, 1583,
       1634, 1667, 1676, 1728, 1543, 1585, 1590, 1635, 1678, 1683, 1730,
       1508, 1592, 1600, 1636, 1685, 1693, 1731, 1545, 1546, 1637, 1697,
       1698, 1734, 1735, 1545, 1606, 1638, 1698, 1699, 1736, 1737, 1546,
       1560, 1613, 1639, 1700, 1708, 1547, 1548, 1611, 1640, 1641, 1651,
       1738, 1548, 1549, 1607, 1640, 1641, 1701, 1738, 1550, 1607, 1610,
       1642, 1701, 1706, 1739, 1551, 1552, 1608, 1643, 1644, 1702, 1740,
       1552, 1553, 1643, 1644, 1645, 1740, 1741, 1553, 1557, 1644, 1645,
       1650, 1741, 1742, 1554, 1608, 1646, 1702, 1703, 1743, 1744, 1554,
       1555, 1647, 1648, 1703, 1745, 1746, 1555, 1556, 1647, 1648, 1649,
       1746, 1747, 1556, 1609, 1648, 1649, 1704, 1747, 1748, 1557, 1579,
       1645, 1650, 1671, 1742, 1749, 1558, 1611, 1612, 1640, 1651, 1707,
       1738, 1559, 1612, 1613, 1652, 1707, 1708, 1750, 1561, 1589, 1617,
       1653, 1681, 1712, 1751, 1562, 1563, 1614, 1654, 1655, 1709, 1752,
       1562, 1622, 1654, 1655, 1717, 1752, 1753, 1564, 1565, 1570, 1656,
       1657, 1662, 1754, 1565, 1566, 1624, 1656, 1657, 1719, 1754, 1567,
       1569, 1658, 1659, 1661, 1755, 1756, 1567, 1568, 1658, 1659, 1660,
       1755, 1757, 1568, 1624, 1659, 1660, 1719, 1757, 1758, 1569, 1625,
       1658, 1661, 1720, 1756, 1759, 1570, 1571, 1656, 1662, 1663, 1754,
       1760, 1571, 1572, 1662, 1663, 1664, 1760, 1761, 1572, 1573, 1663,
       1664, 1665, 1761, 1762, 1573, 1574, 1664, 1665, 1666, 1762, 1763,
       1574, 1575, 1626, 1665, 1666, 1721, 1763, 1576, 1627, 1634, 1667,
       1669, 1728, 1764, 1577, 1578, 1668, 1669, 1670, 1765, 1766, 1577,
       1627, 1667, 1668, 1669, 1764, 1765, 1578, 1629, 1668, 1670, 1723,
       1766, 1767, 1579, 1633, 1650, 1671, 1727, 1749, 1768, 1580, 1626,
       1672, 1673, 1721, 1769, 1770, 1580, 1581, 1672, 1673, 1674, 1770,
       1771, 1581, 1582, 1673, 1674, 1675, 1771, 1772, 1582, 1583, 1674,
       1675, 1676, 1772, 1773, 1583, 1634, 1675, 1676, 1728, 1773, 1774,
       1818, 1584, 1586, 1677, 1678, 1679, 1775, 1776, 1584, 1585, 1635,
       1677, 1678, 1730, 1775, 1586, 1591, 1677, 1679, 1729, 1776, 1777,
       1587, 1588, 1680, 1681, 1682, 1778, 1779, 1587, 1589, 1653, 1680,
       1681, 1751, 1778, 1588, 1590, 1680, 1682, 1683, 1779, 1780, 1590,
       1635, 1682, 1683, 1730, 1780, 1781, 1591, 1592, 1684, 1685, 1729,
       1782, 1592, 1636, 1684, 1685, 1731, 1782, 1783, 1593, 1596, 1686,
       1687, 1690, 1784, 1785, 1593, 1594, 1686, 1687, 1688, 1784, 1786,
       1594, 1595, 1687, 1688, 1689, 1786, 1787, 1595, 1597, 1688, 1689,
       1691, 1787, 1788, 1596, 1601, 1686, 1690, 1732, 1785, 1789, 1597,
       1598, 1689, 1691, 1692, 1788, 1790, 1598, 1599, 1691, 1692, 1693,
       1790, 1791, 1599, 1600, 1636, 1692, 1693, 1731, 1791, 1601, 1602,
       1603, 1694, 1695, 1732, 1792, 1603, 1604, 1694, 1695, 1733, 1792,
       1793, 1604, 1605, 1606, 1696, 1699, 1733, 1794, 1546, 1637, 1697,
       1700, 1705, 1706, 1734, 1545, 1637, 1638, 1698, 1735, 1737, 1795,
       1606, 1638, 1696, 1699, 1736, 1794, 1796, 1546, 1639, 1697, 1700,
       1706, 1708, 1739, 1607, 1641, 1642, 1701, 1738, 1739, 1750, 1608,
       1643, 1646, 1702, 1740, 1744, 1797, 1554, 1646, 1647, 1703, 1743,
       1745, 1798, 1609, 1649, 1704, 1705, 1734, 1748, 1799, 1609, 1610,
       1697, 1704, 1705, 1706, 1734, 1610, 1642, 1697, 1700, 1705, 1706,
       1739, 1612, 1651, 1652, 1707, 1738, 1750, 1613, 1639, 1652, 1700,
       1708, 1739, 1750, 1614, 1615, 1654, 1709, 1710, 1752, 1800, 1615,
       1616, 1709, 1710, 1711, 1800, 1801, 1616, 1617, 1710, 1711, 1712,
       1801, 1802, 1617, 1653, 1711, 1712, 1751, 1802, 1618, 1622, 1713,
       1714, 1717, 1803, 1804, 1618, 1619, 1713, 1714, 1715, 1804, 1805,
       1619, 1620, 1714, 1715, 1716, 1805, 1806, 1620, 1621, 1623, 1715,
       1716, 1718, 1806, 1622, 1655, 1713, 1717, 1753, 1803, 1807, 1623,
       1625, 1716, 1718, 1720, 1806, 1808, 1624, 1657, 1660, 1719, 1754,
       1758, 1809, 1625, 1661, 1718, 1720, 1759, 1808, 1810, 1626, 1666,
       1672, 1721, 1763, 1769, 1811, 1628, 1629, 1630, 1722, 1723, 1724,
       1812, 1629, 1670, 1722, 1723, 1767, 1812, 1813, 1630, 1631, 1722,
       1724, 1725, 1812, 1814, 1631, 1632, 1724, 1725, 1726, 1814, 1815,
       1632, 1633, 1725, 1726, 1727, 1815, 1816, 1633, 1671, 1726, 1727,
       1768, 1816, 1817, 1634, 1667, 1676, 1728, 1764, 1774, 1819, 1591,
       1679, 1684, 1729, 1777, 1782, 1820, 1822, 1635, 1678, 1683, 1730,
       1775, 1781, 1821, 1636, 1685, 1693, 1731, 1783, 1791, 1823, 1601,
       1690, 1694, 1732, 1789, 1792, 1824, 1604, 1695, 1696, 1733, 1793,
       1794, 1825, 1637, 1697, 1704, 1705, 1734, 1735, 1799, 1637, 1698,
       1734, 1735, 1795, 1799, 1826, 1638, 1699, 1736, 1737, 1796, 1827,
       1828, 1638, 1698, 1736, 1737, 1795, 1827, 1829, 1640, 1641, 1651,
       1701, 1707, 1738, 1750, 1642, 1700, 1701, 1706, 1708, 1739, 1750,
       1643, 1644, 1702, 1740, 1741, 1797, 1830, 1644, 1645, 1740, 1741,
       1742, 1830, 1831, 1645, 1650, 1741, 1742, 1749, 1831, 1832, 1646,
       1703, 1743, 1744, 1798, 1833, 1834, 1646, 1702, 1743, 1744, 1797,
       1834, 1835, 1647, 1703, 1745, 1746, 1798, 1836, 1837, 1647, 1648,
       1745, 1746, 1747, 1837, 1838, 1648, 1649, 1746, 1747, 1748, 1838,
       1839, 1649, 1704, 1747, 1748, 1799, 1839, 1840, 1887, 1650, 1671,
       1742, 1749, 1768, 1832, 1841, 1652, 1701, 1707, 1708, 1738, 1739,
       1750, 1653, 1681, 1712, 1751, 1778, 1802, 1842, 1654, 1655, 1709,
       1752, 1753, 1800, 1843, 1655, 1717, 1752, 1753, 1807, 1843, 1844,
       1656, 1657, 1662, 1719, 1754, 1760, 1809, 1658, 1659, 1755, 1756,
       1757, 1845, 1846, 1658, 1661, 1755, 1756, 1759, 1845, 1847, 1659,
       1660, 1755, 1757, 1758, 1846, 1848, 1660, 1719, 1757, 1758, 1809,
       1848, 1849, 1661, 1720, 1756, 1759, 1810, 1847, 1850, 1662, 1663,
       1754, 1760, 1761, 1809, 1851, 1663, 1664, 1760, 1761, 1762, 1851,
       1852, 1664, 1665, 1761, 1762, 1763, 1852, 1853, 1665, 1666, 1721,
       1762, 1763, 1811, 1853, 1667, 1669, 1728, 1764, 1765, 1819, 1854,
       1668, 1669, 1764, 1765, 1766, 1854, 1668, 1670, 1765, 1766, 1767,
       1854, 1855, 1670, 1723, 1766, 1767, 1813, 1855, 1856, 1671, 1727,
       1749, 1768, 1817, 1841, 1857, 1672, 1721, 1769, 1770, 1811, 1858,
       1859, 1672, 1673, 1769, 1770, 1771, 1858, 1860, 1673, 1674, 1770,
       1771, 1772, 1860, 1861, 1674, 1675, 1771, 1772, 1773, 1861, 1862,
       1675, 1676, 1772, 1773, 1818, 1862, 1676, 1728, 1774, 1818, 1819,
       1863, 1864, 1677, 1678, 1730, 1775, 1776, 1821, 1865, 1677, 1679,
       1775, 1776, 1777, 1865, 1866, 1679, 1729, 1776, 1777, 1820, 1866,
       1867, 1680, 1681, 1751, 1778, 1779, 1842, 1868, 1680, 1682, 1778,
       1779, 1780, 1868, 1869, 1682, 1683, 1779, 1780, 1781, 1869, 1870,
       1683, 1730, 1780, 1781, 1821, 1870, 1871, 1684, 1685, 1729, 1782,
       1783, 1822, 1872, 1685, 1731, 1782, 1783, 1823, 1872, 1873, 1686,
       1687, 1784, 1785, 1786, 1874, 1875, 1686, 1690, 1784, 1785, 1789,
       1875, 1876, 1913, 1687, 1688, 1784, 1786, 1787, 1874, 1877, 1912,
       1688, 1689, 1786, 1787, 1788, 1877, 1689, 1691, 1787, 1788, 1790,
       1877, 1878, 1690, 1732, 1785, 1789, 1824, 1876, 1691, 1692, 1788,
       1790, 1791, 1878, 1879, 1692, 1693, 1731, 1790, 1791, 1823, 1879,
       1694, 1695, 1732, 1792, 1793, 1824, 1880, 1695, 1733, 1792, 1793,
       1825, 1880, 1881, 1696, 1699, 1733, 1794, 1796, 1825, 1882, 1698,
       1735, 1737, 1795, 1826, 1829, 1883, 1699, 1736, 1794, 1796, 1828,
       1882, 1884, 1702, 1740, 1744, 1797, 1830, 1835, 1885, 1703, 1743,
       1745, 1798, 1833, 1836, 1886, 1704, 1734, 1735, 1748, 1799, 1826,
       1840, 1709, 1710, 1752, 1800, 1801, 1843, 1888, 1710, 1711, 1800,
       1801, 1802, 1888, 1889, 1711, 1712, 1751, 1801, 1802, 1842, 1889,
       1890, 1713, 1717, 1803, 1804, 1807, 1891, 1892, 1713, 1714, 1803,
       1804, 1805, 1891, 1893, 1714, 1715, 1804, 1805, 1806, 1893, 1894,
       1715, 1716, 1718, 1805, 1806, 1808, 1894, 1717, 1753, 1803, 1807,
       1844, 1892, 1895, 1718, 1720, 1806, 1808, 1810, 1894, 1896, 1719,
       1754, 1758, 1760, 1809, 1849, 1851, 1720, 1759, 1808, 1810, 1850,
       1896, 1897, 1721, 1763, 1769, 1811, 1853, 1859, 1898, 1722, 1723,
       1724, 1812, 1813, 1814, 1899, 1723, 1767, 1812, 1813, 1856, 1899,
       1900, 1724, 1725, 1812, 1814, 1815, 1899, 1901, 1725, 1726, 1814,
       1815, 1816, 1901, 1902, 1726, 1727, 1815, 1816, 1817, 1902, 1903,
       1727, 1768, 1816, 1817, 1857, 1903, 1904, 1676, 1773, 1774, 1818,
       1862, 1864, 1905, 1728, 1764, 1774, 1819, 1854, 1863, 1906, 1729,
       1777, 1820, 1822, 1867, 1907, 1908, 1730, 1775, 1781, 1821, 1865,
       1871, 1909, 1729, 1782, 1820, 1822, 1872, 1907, 1910, 1731, 1783,
       1791, 1823, 1873, 1879, 1911, 1732, 1789, 1792, 1824, 1876, 1880,
       1914, 1733, 1793, 1794, 1825, 1881, 1882, 1915, 1735, 1795, 1799,
       1826, 1840, 1883, 1916, 1736, 1737, 1827, 1828, 1829, 1917, 1918,
       1736, 1796, 1827, 1828, 1884, 1917, 1737, 1795, 1827, 1829, 1883,
       1918, 1919, 1740, 1741, 1797, 1830, 1831, 1885, 1920, 1741, 1742,
       1830, 1831, 1832, 1920, 1921, 1742, 1749, 1831, 1832, 1841, 1921,
       1922, 1743, 1798, 1833, 1834, 1886, 1923, 1924, 1743, 1744, 1833,
       1834, 1835, 1923, 1925, 1744, 1797, 1834, 1835, 1885, 1925, 1926,
       1745, 1798, 1836, 1837, 1886, 1927, 1928, 1745, 1746, 1836, 1837,
       1838, 1928, 1929, 1746, 1747, 1837, 1838, 1839, 1887, 1929, 1747,
       1748, 1838, 1839, 1887, 1748, 1799, 1826, 1840, 1887, 1916, 1930,
       1749, 1768, 1832, 1841, 1857, 1922, 1931, 1751, 1778, 1802, 1842,
       1868, 1890, 1932, 1752, 1753, 1800, 1843, 1844, 1888, 1933, 1753,
       1807, 1843, 1844, 1895, 1933, 1934, 1755, 1756, 1845, 1846, 1847,
       1935, 1936, 1755, 1757, 1845, 1846, 1848, 1936, 1937, 1756, 1759,
       1845, 1847, 1850, 1935, 1938, 1757, 1758, 1846, 1848, 1849, 1937,
       1939, 1758, 1809, 1848, 1849, 1851, 1939, 1940, 1759, 1810, 1847,
       1850, 1897, 1938, 1941, 1760, 1761, 1809, 1849, 1851, 1852, 1940,
       1761, 1762, 1851, 1852, 1853, 1940, 1942, 1762, 1763, 1811, 1852,
       1853, 1898, 1942, 1764, 1765, 1766, 1819, 1854, 1855, 1906, 1943,
       1766, 1767, 1854, 1855, 1856, 1943, 1944, 1767, 1813, 1855, 1856,
       1900, 1944, 1945, 1768, 1817, 1841, 1857, 1904, 1931, 1946, 1769,
       1770, 1858, 1859, 1860, 1947, 1948, 1769, 1811, 1858, 1859, 1898,
       1947, 1949, 1770, 1771, 1858, 1860, 1861, 1948, 1950, 1771, 1772,
       1860, 1861, 1862, 1950, 1951, 1993, 1772, 1773, 1818, 1861, 1862,
       1905, 1951, 1774, 1819, 1863, 1864, 1906, 1952, 1953, 1774, 1818,
       1863, 1864, 1905, 1953, 1954, 1775, 1776, 1821, 1865, 1866, 1909,
       1955, 1776, 1777, 1865, 1866, 1867, 1955, 1956, 1777, 1820, 1866,
       1867, 1908, 1956, 1957, 1778, 1779, 1842, 1868, 1869, 1932, 1958,
       1779, 1780, 1868, 1869, 1870, 1958, 1959, 1780, 1781, 1869, 1870,
       1871, 1959, 1960, 1781, 1821, 1870, 1871, 1909, 1960, 1961, 1782,
       1783, 1822, 1872, 1873, 1910, 1962, 1783, 1823, 1872, 1873, 1911,
       1962, 1963, 1784, 1786, 1874, 1875, 1912, 1964, 1965, 1784, 1785,
       1874, 1875, 1913, 1964, 1966, 1785, 1789, 1824, 1876, 1913, 1914,
       1967, 1786, 1787, 1788, 1877, 1878, 1912, 1968, 1788, 1790, 1877,
       1878, 1879, 1968, 1790, 1791, 1823, 1878, 1879, 1911, 1968, 1792,
       1793, 1824, 1880, 1881, 1914, 1969, 1793, 1825, 1880, 1881, 1915,
       1969, 1970, 1794, 1796, 1825, 1882, 1884, 1915, 1971, 1795, 1826,
       1829, 1883, 1916, 1919, 1972, 1796, 1828, 1882, 1884, 1917, 1971,
       1973, 1797, 1830, 1835, 1885, 1920, 1926, 1974, 1798, 1833, 1836,
       1886, 1924, 1927, 1975, 1748, 1838, 1839, 1840, 1887, 1929, 1930,
       1800, 1801, 1843, 1888, 1889, 1933, 1976, 1801, 1802, 1888, 1889,
       1890, 1976, 1977, 1802, 1842, 1889, 1890, 1932, 1977, 1978, 1803,
       1804, 1891, 1892, 1893, 1979, 1980, 1803, 1807, 1891, 1892, 1895,
       1979, 1981, 1804, 1805, 1891, 1893, 1894, 1980, 1982, 1805, 1806,
       1808, 1893, 1894, 1896, 1982, 1807, 1844, 1892, 1895, 1934, 1981,
       1983, 1808, 1810, 1894, 1896, 1897, 1982, 1984, 1810, 1850, 1896,
       1897, 1941, 1984, 1985, 1811, 1853, 1859, 1898, 1942, 1949, 1986,
       1812, 1813, 1814, 1899, 1900, 1901, 1987, 1813, 1856, 1899, 1900,
       1945, 1987, 1988, 1814, 1815, 1899, 1901, 1902, 1987, 1989, 1815,
       1816, 1901, 1902, 1903, 1989, 1990, 1816, 1817, 1902, 1903, 1904,
       1990, 1991, 1817, 1857, 1903, 1904, 1946, 1991, 1992, 1818, 1862,
       1864, 1905, 1951, 1954, 1994, 1819, 1854, 1863, 1906, 1943, 1952,
       1995, 1820, 1822, 1907, 1908, 1910, 1996, 1997, 1820, 1867, 1907,
       1908, 1957, 1997, 1998, 1821, 1865, 1871, 1909, 1955, 1961, 1999,
       1822, 1872, 1907, 1910, 1962, 1996, 2000, 1823, 1873, 1879, 1911,
       1963, 1968, 2001, 1786, 1874, 1877, 1912, 1965, 1968, 2001, 1785,
       1875, 1876, 1913, 1966, 1967, 2002, 1824, 1876, 1880, 1914, 1967,
       1969, 2003, 1825, 1881, 1882, 1915, 1970, 1971, 2004, 1826, 1840,
       1883, 1916, 1930, 1972, 2005, 1827, 1828, 1884, 1917, 1918, 1973,
       2006, 1827, 1829, 1917, 1918, 1919, 2006, 2007, 1829, 1883, 1918,
       1919, 1972, 2007, 2008, 1830, 1831, 1885, 1920, 1921, 1974, 2009,
       1831, 1832, 1920, 1921, 1922, 2009, 2010, 1832, 1841, 1921, 1922,
       1931, 2010, 1833, 1834, 1923, 1924, 1925, 2011, 2012, 1833, 1886,
       1923, 1924, 1975, 2011, 2013, 1834, 1835, 1923, 1925, 1926, 2012,
       2014, 1835, 1885, 1925, 1926, 1974, 2014, 2015, 1836, 1886, 1927,
       1928, 1975, 2016, 2017, 1836, 1837, 1927, 1928, 1929, 2016, 2018,
       1837, 1838, 1887, 1928, 1929, 1930, 2018, 1840, 1887, 1916, 1929,
       1930, 2005, 2018, 1841, 1857, 1922, 1931, 1946, 2010, 2019, 1842,
       1868, 1890, 1932, 1958, 1978, 2020, 1843, 1844, 1888, 1933, 1934,
       1976, 2021, 1844, 1895, 1933, 1934, 1983, 2021, 2022, 1845, 1847,
       1935, 1936, 1938, 2023, 2024, 1845, 1846, 1935, 1936, 1937, 2024,
       2025, 1846, 1848, 1936, 1937, 1939, 2025, 2026, 1847, 1850, 1935,
       1938, 1941, 2023, 2027, 1848, 1849, 1937, 1939, 1940, 2026, 2028,
       1849, 1851, 1852, 1939, 1940, 1942, 2028, 1850, 1897, 1938, 1941,
       1985, 2027, 2029, 1852, 1853, 1898, 1940, 1942, 1986, 2028, 1854,
       1855, 1906, 1943, 1944, 1995, 2030, 1855, 1856, 1943, 1944, 1945,
       2030, 2031, 1856, 1900, 1944, 1945, 1988, 2031, 2032, 2073, 1857,
       1904, 1931, 1946, 1992, 2019, 2033, 1858, 1859, 1947, 1948, 1949,
       2034, 2035, 1858, 1860, 1947, 1948, 1950, 2034, 2036, 1859, 1898,
       1947, 1949, 1986, 2035, 2037, 1860, 1861, 1948, 1950, 1993, 2036,
       2038, 1861, 1862, 1905, 1951, 1993, 1994, 2039, 1863, 1906, 1952,
       1953, 1995, 2040, 2041, 1863, 1864, 1952, 1953, 1954, 2040, 2042,
       1864, 1905, 1953, 1954, 1994, 2042, 2043, 1865, 1866, 1909, 1955,
       1956, 1999, 2044, 1866, 1867, 1955, 1956, 1957, 2044, 2045, 1867,
       1908, 1956, 1957, 1998, 2045, 2046, 1868, 1869, 1932, 1958, 1959,
       2020, 2047, 1869, 1870, 1958, 1959, 1960, 2047, 2048, 1870, 1871,
       1959, 1960, 1961, 2048, 2049, 1871, 1909, 1960, 1961, 1999, 2049,
       2050, 1872, 1873, 1910, 1962, 1963, 2000, 2051, 1873, 1911, 1962,
       1963, 2001, 2051, 2052, 1874, 1875, 1964, 1965, 1966, 2053, 2054,
       1874, 1912, 1964, 1965, 2001, 2052, 2054, 1875, 1913, 1964, 1966,
       2002, 2053, 2055, 1876, 1913, 1914, 1967, 2002, 2003, 2056, 1877,
       1878, 1879, 1911, 1912, 1968, 2001, 1880, 1881, 1914, 1969, 1970,
       2003, 2057, 1881, 1915, 1969, 1970, 2004, 2057, 2058, 1882, 1884,
       1915, 1971, 1973, 2004, 1883, 1916, 1919, 1972, 2005, 2008, 2059,
       1884, 1917, 1971, 1973, 2004, 2006, 1885, 1920, 1926, 1974, 2009,
       2015, 2060, 1886, 1924, 1927, 1975, 2013, 2017, 2061, 1888, 1889,
       1933, 1976, 1977, 2021, 2062, 1889, 1890, 1976, 1977, 1978, 2062,
       1890, 1932, 1977, 1978, 2020, 2062, 2063, 1891, 1892, 1979, 1980,
       1981, 2064, 2065, 1891, 1893, 1979, 1980, 1982, 2064, 2066, 1892,
       1895, 1979, 1981, 1983, 2065, 2067, 1893, 1894, 1896, 1980, 1982,
       1984, 2066, 1895, 1934, 1981, 1983, 2022, 2067, 2068, 1896, 1897,
       1982, 1984, 1985, 2066, 2069, 1897, 1941, 1984, 1985, 2029, 2069,
       2070, 1898, 1942, 1949, 1986, 2028, 2037, 2071, 1899, 1900, 1901,
       1987, 1988, 1989, 2072, 1900, 1945, 1987, 1988, 2032, 2072, 2074,
       1901, 1902, 1987, 1989, 1990, 2072, 2075, 1902, 1903, 1989, 1990,
       1991, 2075, 2076, 1903, 1904, 1990, 1991, 1992, 2076, 2077, 1904,
       1946, 1991, 1992, 2033, 2077, 2078, 1861, 1950, 1951, 1993, 2038,
       2039, 2079, 1905, 1951, 1954, 1994, 2039, 2043, 2080, 1906, 1943,
       1952, 1995, 2030, 2041, 2081, 1907, 1910, 1996, 1997, 2000, 2082,
       2083, 1907, 1908, 1996, 1997, 1998, 2082, 2084, 1908, 1957, 1997,
       1998, 2046, 2084, 2085, 1909, 1955, 1961, 1999, 2044, 2050, 2086,
       1910, 1962, 1996, 2000, 2051, 2083, 2087, 1911, 1912, 1963, 1965,
       1968, 2001, 2052, 1913, 1966, 1967, 2002, 2055, 2056, 2088, 1914,
       1967, 1969, 2003, 2056, 2057, 2089, 1915, 1970, 1971, 1973, 2004,
       2006, 2058, 1916, 1930, 1972, 2005, 2018, 2059, 2090, 1917, 1918,
       1973, 2004, 2006, 2007, 2058, 1918, 1919, 2006, 2007, 2008, 2058,
       2091, 1919, 1972, 2007, 2008, 2059, 2091, 2092, 1920, 1921, 1974,
       2009, 2010, 2060, 2093, 1921, 1922, 1931, 2009, 2010, 2019, 2093,
       1923, 1924, 2011, 2012, 2013, 2094, 2095, 1923, 1925, 2011, 2012,
       2014, 2094, 2096, 1924, 1975, 2011, 2013, 2061, 2095, 2097, 1925,
       1926, 2012, 2014, 2015, 2096, 2098, 1926, 1974, 2014, 2015, 2060,
       2098, 2099, 1927, 1928, 2016, 2017, 2018, 2090, 2100, 1927, 1975,
       2016, 2017, 2061, 2100, 2101, 1928, 1929, 1930, 2005, 2016, 2018,
       2090, 1931, 1946, 2010, 2019, 2033, 2093, 2102, 2142, 1932, 1958,
       1978, 2020, 2047, 2063, 2103, 1933, 1934, 1976, 2021, 2022, 2062,
       2104, 1934, 1983, 2021, 2022, 2068, 2104, 2105, 1935, 1938, 2023,
       2024, 2027, 2106, 2107, 1935, 1936, 2023, 2024, 2025, 2106, 2108,
       1936, 1937, 2024, 2025, 2026, 2108, 2109, 1937, 1939, 2025, 2026,
       2028, 2071, 2109, 1938, 1941, 2023, 2027, 2029, 2107, 2110, 1939,
       1940, 1942, 1986, 2026, 2028, 2071, 1941, 1985, 2027, 2029, 2070,
       2110, 2111, 1943, 1944, 1995, 2030, 2031, 2081, 2112, 2152, 1944,
       1945, 2030, 2031, 2073, 2112, 2113, 1945, 1988, 2032, 2073, 2074,
       2114, 1946, 1992, 2019, 2033, 2078, 2102, 2115, 1947, 1948, 2034,
       2035, 2036, 2116, 2117, 1947, 1949, 2034, 2035, 2037, 2116, 2118,
       1948, 1950, 2034, 2036, 2038, 2117, 2119, 1949, 1986, 2035, 2037,
       2071, 2118, 2120, 1950, 1993, 2036, 2038, 2079, 2119, 2121, 1951,
       1993, 1994, 2039, 2079, 2080, 2122, 1952, 1953, 2040, 2041, 2042,
       2123, 2124, 1952, 1995, 2040, 2041, 2081, 2123, 2125, 1953, 1954,
       2040, 2042, 2043, 2124, 2126, 1954, 1994, 2042, 2043, 2080, 2126,
       2127, 1955, 1956, 1999, 2044, 2045, 2086, 2128, 1956, 1957, 2044,
       2045, 2046, 2128, 2129, 1957, 1998, 2045, 2046, 2085, 2129, 2130,
       1958, 1959, 2020, 2047, 2048, 2103, 2131, 1959, 1960, 2047, 2048,
       2049, 2131, 2132, 1960, 1961, 2048, 2049, 2050, 2132, 2133, 1961,
       1999, 2049, 2050, 2086, 2133, 2134, 1962, 1963, 2000, 2051, 2052,
       2087, 1963, 1965, 2001, 2051, 2052, 2054, 2087, 1964, 1966, 2053,
       2054, 2055, 2135, 2136, 1964, 1965, 2052, 2053, 2054, 2087, 2136,
       1966, 2002, 2053, 2055, 2088, 2135, 2137, 1967, 2002, 2003, 2056,
       2088, 2089, 2138, 1969, 1970, 2003, 2057, 2058, 2089, 2091, 1970,
       2004, 2006, 2007, 2057, 2058, 2091, 1972, 2005, 2008, 2059, 2090,
       2092, 2139, 2140, 1974, 2009, 2015, 2060, 2093, 2099, 2141, 1975,
       2013, 2017, 2061, 2097, 2101, 2143, 1976, 1977, 1978, 2021, 2062,
       2063, 2104, 1978, 2020, 2062, 2063, 2103, 2104, 2144, 1979, 1980,
       2064, 2065, 2066, 2145, 2146, 1979, 1981, 2064, 2065, 2067, 2145,
       2147, 1980, 1982, 1984, 2064, 2066, 2069, 2146, 1981, 1983, 2065,
       2067, 2068, 2147, 2148, 1983, 2022, 2067, 2068, 2105, 2148, 2149,
       1984, 1985, 2066, 2069, 2070, 2146, 2150, 1985, 2029, 2069, 2070,
       2111, 2150, 2151, 1986, 2026, 2028, 2037, 2071, 2109, 2120, 1987,
       1988, 1989, 2072, 2074, 2075, 2153, 1945, 2031, 2032, 2073, 2113,
       2114, 2154, 1988, 2032, 2072, 2074, 2114, 2153, 2155, 1989, 1990,
       2072, 2075, 2076, 2153, 2156, 1990, 1991, 2075, 2076, 2077, 2156,
       2157, 1991, 1992, 2076, 2077, 2078, 2157, 2158, 1992, 2033, 2077,
       2078, 2115, 2158, 2159, 1993, 2038, 2039, 2079, 2121, 2122, 2160,
       1994, 2039, 2043, 2080, 2122, 2127, 2161, 1995, 2030, 2041, 2081,
       2125, 2152, 2162, 1996, 1997, 2082, 2083, 2084, 2163, 1996, 2000,
       2082, 2083, 2087, 2163, 2164, 1997, 1998, 2082, 2084, 2085, 2163,
       2165, 1998, 2046, 2084, 2085, 2130, 2165, 2166, 1999, 2044, 2050,
       2086, 2128, 2134, 2167, 2000, 2051, 2052, 2054, 2083, 2087, 2136,
       2164, 2002, 2055, 2056, 2088, 2137, 2138, 2168, 2003, 2056, 2057,
       2089, 2091, 2092, 2138, 2005, 2016, 2018, 2059, 2090, 2100, 2139,
       2169, 2007, 2008, 2057, 2058, 2089, 2091, 2092, 2008, 2059, 2089,
       2091, 2092, 2138, 2140, 2009, 2010, 2019, 2060, 2093, 2141, 2142,
       2011, 2012, 2094, 2095, 2096, 2170, 2171, 2011, 2013, 2094, 2095,
       2097, 2171, 2172, 2012, 2014, 2094, 2096, 2098, 2170, 2173, 2013,
       2061, 2095, 2097, 2143, 2172, 2174, 2014, 2015, 2096, 2098, 2099,
       2173, 2175, 2015, 2060, 2098, 2099, 2141, 2175, 2176, 2016, 2017,
       2090, 2100, 2101, 2169, 2177, 2017, 2061, 2100, 2101, 2143, 2177,
       2178, 2019, 2033, 2102, 2115, 2142, 2179, 2020, 2047, 2063, 2103,
       2131, 2144, 2180, 2021, 2022, 2062, 2063, 2104, 2105, 2144, 2022,
       2068, 2104, 2105, 2144, 2149, 2181, 2023, 2024, 2106, 2107, 2108,
       2182, 2183, 2023, 2027, 2106, 2107, 2110, 2182, 2184, 2024, 2025,
       2106, 2108, 2109, 2183, 2185, 2025, 2026, 2071, 2108, 2109, 2120,
       2185, 2027, 2029, 2107, 2110, 2111, 2184, 2186, 2029, 2070, 2110,
       2111, 2151, 2186, 2187, 2030, 2031, 2112, 2113, 2152, 2188, 2189,
       2031, 2073, 2112, 2113, 2154, 2188, 2190, 2032, 2073, 2074, 2114,
       2154, 2155, 2191, 2192, 2033, 2078, 2102, 2115, 2159, 2179, 2193,
       2034, 2035, 2116, 2117, 2118, 2194, 2195, 2034, 2036, 2116, 2117,
       2119, 2195, 2196, 2035, 2037, 2116, 2118, 2120, 2194, 2197, 2036,
       2038, 2117, 2119, 2121, 2196, 2198, 2037, 2071, 2109, 2118, 2120,
       2185, 2197, 2038, 2079, 2119, 2121, 2160, 2198, 2199, 2039, 2079,
       2080, 2122, 2160, 2161, 2200, 2040, 2041, 2123, 2124, 2125, 2201,
       2202, 2040, 2042, 2123, 2124, 2126, 2201, 2203, 2041, 2081, 2123,
       2125, 2162, 2202, 2204, 2042, 2043, 2124, 2126, 2127, 2203, 2205,
       2043, 2080, 2126, 2127, 2161, 2205, 2206, 2044, 2045, 2086, 2128,
       2129, 2167, 2207, 2045, 2046, 2128, 2129, 2130, 2207, 2208, 2046,
       2085, 2129, 2130, 2166, 2208, 2209, 2047, 2048, 2103, 2131, 2132,
       2180, 2210, 2048, 2049, 2131, 2132, 2133, 2210, 2049, 2050, 2132,
       2133, 2134, 2210, 2211, 2050, 2086, 2133, 2134, 2167, 2211, 2212,
       2053, 2055, 2135, 2136, 2137, 2213, 2214, 2053, 2054, 2087, 2135,
       2136, 2164, 2213, 2055, 2088, 2135, 2137, 2168, 2214, 2215, 2056,
       2088, 2089, 2092, 2138, 2140, 2168, 2059, 2090, 2139, 2140, 2169,
       2216, 2059, 2092, 2138, 2139, 2140, 2168, 2216, 2060, 2093, 2099,
       2141, 2142, 2176, 2217, 2019, 2093, 2102, 2141, 2142, 2179, 2217,
       2061, 2097, 2101, 2143, 2174, 2178, 2218, 2063, 2103, 2104, 2105,
       2144, 2180, 2181, 2064, 2065, 2145, 2146, 2147, 2219, 2220, 2064,
       2066, 2069, 2145, 2146, 2150, 2220, 2065, 2067, 2145, 2147, 2148,
       2219, 2221, 2067, 2068, 2147, 2148, 2149, 2221, 2222, 2068, 2105,
       2148, 2149, 2181, 2222, 2223, 2069, 2070, 2146, 2150, 2151, 2220,
       2224, 2070, 2111, 2150, 2151, 2187, 2224, 2225, 2030, 2081, 2112,
       2152, 2162, 2189, 2072, 2074, 2075, 2153, 2155, 2156, 2226, 2073,
       2113, 2114, 2154, 2190, 2191, 2227, 2074, 2114, 2153, 2155, 2192,
       2226, 2228, 2075, 2076, 2153, 2156, 2157, 2226, 2229, 2076, 2077,
       2156, 2157, 2158, 2229, 2230, 2077, 2078, 2157, 2158, 2159, 2230,
       2231, 2078, 2115, 2158, 2159, 2193, 2231, 2232, 2079, 2121, 2122,
       2160, 2199, 2200, 2233, 2080, 2122, 2127, 2161, 2200, 2206, 2234,
       2081, 2125, 2152, 2162, 2189, 2204, 2235, 2082, 2083, 2084, 2163,
       2164, 2165, 2236, 2083, 2087, 2136, 2163, 2164, 2213, 2236, 2084,
       2085, 2163, 2165, 2166, 2236, 2237, 2085, 2130, 2165, 2166, 2209,
       2237, 2238, 2086, 2128, 2134, 2167, 2207, 2212, 2239, 2088, 2137,
       2138, 2140, 2168, 2215, 2216, 2240, 2090, 2100, 2139, 2169, 2177,
       2216, 2241, 2094, 2096, 2170, 2171, 2173, 2242, 2243, 2094, 2095,
       2170, 2171, 2172, 2243, 2244, 2095, 2097, 2171, 2172, 2174, 2244,
       2245, 2096, 2098, 2170, 2173, 2175, 2242, 2246, 2097, 2143, 2172,
       2174, 2218, 2245, 2247, 2290, 2098, 2099, 2173, 2175, 2176, 2246,
       2248, 2099, 2141, 2175, 2176, 2217, 2248, 2249, 2100, 2101, 2169,
       2177, 2178, 2241, 2250, 2101, 2143, 2177, 2178, 2218, 2250, 2251,
       2102, 2115, 2142, 2179, 2193, 2217, 2252, 2103, 2131, 2144, 2180,
       2181, 2210, 2253, 2105, 2144, 2149, 2180, 2181, 2223, 2253, 2106,
       2107, 2182, 2183, 2184, 2254, 2255, 2106, 2108, 2182, 2183, 2185,
       2255, 2256, 2107, 2110, 2182, 2184, 2186, 2254, 2257, 2108, 2109,
       2120, 2183, 2185, 2197, 2256, 2110, 2111, 2184, 2186, 2187, 2257,
       2258, 2111, 2151, 2186, 2187, 2225, 2258, 2259, 2112, 2113, 2188,
       2189, 2190, 2260, 2261, 2112, 2152, 2162, 2188, 2189, 2235, 2260,
       2113, 2154, 2188, 2190, 2227, 2261, 2262, 2114, 2154, 2191, 2192,
       2227, 2263, 2264, 2114, 2155, 2191, 2192, 2228, 2263, 2265, 2115,
       2159, 2179, 2193, 2232, 2252, 2266, 2116, 2118, 2194, 2195, 2197,
       2267, 2268, 2116, 2117, 2194, 2195, 2196, 2267, 2269, 2117, 2119,
       2195, 2196, 2198, 2269, 2270, 2118, 2120, 2185, 2194, 2197, 2256,
       2268, 2119, 2121, 2196, 2198, 2199, 2270, 2271, 2121, 2160, 2198,
       2199, 2233, 2271, 2272, 2122, 2160, 2161, 2200, 2233, 2234, 2273,
       2123, 2124, 2201, 2202, 2203, 2274, 2275, 2123, 2125, 2201, 2202,
       2204, 2275, 2276, 2124, 2126, 2201, 2203, 2205, 2274, 2277, 2307,
       2125, 2162, 2202, 2204, 2235, 2276, 2278, 2126, 2127, 2203, 2205,
       2206, 2277, 2279, 2127, 2161, 2205, 2206, 2234, 2279, 2280, 2128,
       2129, 2167, 2207, 2208, 2239, 2281, 2129, 2130, 2207, 2208, 2209,
       2281, 2282, 2130, 2166, 2208, 2209, 2238, 2282, 2283, 2131, 2132,
       2133, 2180, 2210, 2211, 2253, 2284, 2133, 2134, 2210, 2211, 2212,
       2284, 2285, 2134, 2167, 2211, 2212, 2239, 2285, 2286, 2135, 2136,
       2164, 2213, 2214, 2236, 2287, 2135, 2137, 2213, 2214, 2215, 2287,
       2288, 2137, 2168, 2214, 2215, 2240, 2288, 2289, 2139, 2140, 2168,
       2169, 2216, 2240, 2241, 2141, 2142, 2176, 2179, 2217, 2249, 2252,
       2143, 2174, 2178, 2218, 2247, 2251, 2291, 2145, 2147, 2219, 2220,
       2221, 2292, 2293, 2145, 2146, 2150, 2219, 2220, 2224, 2293, 2147,
       2148, 2219, 2221, 2222, 2292, 2294, 2148, 2149, 2221, 2222, 2223,
       2294, 2295, 2324, 2149, 2181, 2222, 2223, 2253, 2295, 2150, 2151,
       2220, 2224, 2225, 2293, 2296, 2151, 2187, 2224, 2225, 2259, 2296,
       2297, 2153, 2155, 2156, 2226, 2228, 2229, 2298, 2154, 2190, 2191,
       2227, 2262, 2264, 2299, 2155, 2192, 2226, 2228, 2265, 2298, 2300,
       2332, 2339, 2156, 2157, 2226, 2229, 2230, 2298, 2301, 2157, 2158,
       2229, 2230, 2231, 2301, 2302, 2158, 2159, 2230, 2231, 2232, 2302,
       2303, 2159, 2193, 2231, 2232, 2266, 2303, 2304, 2160, 2199, 2200,
       2233, 2272, 2273, 2305, 2161, 2200, 2206, 2234, 2273, 2280, 2306,
       2162, 2189, 2204, 2235, 2260, 2278, 2308, 2331, 2163, 2164, 2165,
       2213, 2236, 2237, 2287, 2165, 2166, 2236, 2237, 2238, 2287, 2309,
       2166, 2209, 2237, 2238, 2283, 2309, 2310, 2167, 2207, 2212, 2239,
       2281, 2286, 2311, 2168, 2215, 2216, 2240, 2241, 2289, 2312, 2169,
       2177, 2216, 2240, 2241, 2250, 2312, 2170, 2173, 2242, 2243, 2246,
       2313, 2314, 2170, 2171, 2242, 2243, 2244, 2314, 2315, 2171, 2172,
       2243, 2244, 2245, 2315, 2316, 2172, 2174, 2244, 2245, 2290, 2316,
       2173, 2175, 2242, 2246, 2248, 2313, 2317, 2174, 2218, 2247, 2290,
       2291, 2318, 2319, 2175, 2176, 2246, 2248, 2249, 2317, 2320, 2176,
       2217, 2248, 2249, 2252, 2320, 2321, 2177, 2178, 2241, 2250, 2251,
       2312, 2322, 2178, 2218, 2250, 2251, 2291, 2322, 2323, 2366, 2179,
       2193, 2217, 2249, 2252, 2266, 2321, 2180, 2181, 2210, 2223, 2253,
       2284, 2295, 2182, 2184, 2254, 2255, 2257, 2325, 2326, 2182, 2183,
       2254, 2255, 2256, 2325, 2327, 2183, 2185, 2197, 2255, 2256, 2268,
       2327, 2184, 2186, 2254, 2257, 2258, 2326, 2328, 2186, 2187, 2257,
       2258, 2259, 2328, 2329, 2187, 2225, 2258, 2259, 2297, 2329, 2330,
       2188, 2189, 2235, 2260, 2261, 2331, 2333, 2188, 2190, 2260, 2261,
       2262, 2333, 2334, 2190, 2227, 2261, 2262, 2299, 2334, 2335, 2191,
       2192, 2263, 2264, 2265, 2336, 2337, 2191, 2227, 2263, 2264, 2299,
       2336, 2338, 2192, 2228, 2263, 2265, 2300, 2337, 2193, 2232, 2252,
       2266, 2304, 2321, 2340, 2194, 2195, 2267, 2268, 2269, 2341, 2342,
       2194, 2197, 2256, 2267, 2268, 2327, 2342, 2195, 2196, 2267, 2269,
       2270, 2341, 2343, 2196, 2198, 2269, 2270, 2271, 2343, 2344, 2198,
       2199, 2270, 2271, 2272, 2344, 2345, 2199, 2233, 2271, 2272, 2305,
       2345, 2346, 2200, 2233, 2234, 2273, 2305, 2306, 2347, 2201, 2203,
       2274, 2275, 2307, 2348, 2349, 2201, 2202, 2274, 2275, 2276, 2349,
       2350, 2202, 2204, 2275, 2276, 2278, 2350, 2351, 2203, 2205, 2277,
       2279, 2307, 2352, 2353, 2204, 2235, 2276, 2278, 2308, 2351, 2354,
       2205, 2206, 2277, 2279, 2280, 2352, 2355, 2206, 2234, 2279, 2280,
       2306, 2355, 2356, 2207, 2208, 2239, 2281, 2282, 2311, 2357, 2208,
       2209, 2281, 2282, 2283, 2357, 2358, 2209, 2238, 2282, 2283, 2310,
       2358, 2359, 2210, 2211, 2253, 2284, 2285, 2295, 2360, 2211, 2212,
       2284, 2285, 2286, 2360, 2361, 2212, 2239, 2285, 2286, 2311, 2361,
       2362, 2213, 2214, 2236, 2237, 2287, 2288, 2309, 2214, 2215, 2287,
       2288, 2289, 2309, 2363, 2215, 2240, 2288, 2289, 2312, 2363, 2364,
       2174, 2245, 2247, 2290, 2316, 2318, 2365, 2218, 2247, 2251, 2291,
       2319, 2323, 2219, 2221, 2292, 2293, 2294, 2367, 2368, 2219, 2220,
       2224, 2292, 2293, 2296, 2368, 2221, 2222, 2292, 2294, 2324, 2367,
       2369, 2222, 2223, 2253, 2284, 2295, 2324, 2360, 2224, 2225, 2293,
       2296, 2297, 2368, 2370, 2225, 2259, 2296, 2297, 2330, 2370, 2371,
       2226, 2228, 2229, 2298, 2301, 2332, 2372, 2227, 2262, 2264, 2299,
       2335, 2338, 2373, 2228, 2265, 2300, 2337, 2339, 2374, 2375, 2229,
       2230, 2298, 2301, 2302, 2372, 2376, 2230, 2231, 2301, 2302, 2303,
       2376, 2377, 2231, 2232, 2302, 2303, 2304, 2377, 2378, 2232, 2266,
       2303, 2304, 2340, 2378, 2379, 2233, 2272, 2273, 2305, 2346, 2347,
       2380, 2234, 2273, 2280, 2306, 2347, 2356, 2381, 2203, 2274, 2277,
       2307, 2348, 2353, 2235, 2278, 2308, 2331, 2354, 2382, 2237, 2238,
       2287, 2288, 2309, 2310, 2363, 2238, 2283, 2309, 2310, 2359, 2363,
       2383, 2239, 2281, 2286, 2311, 2357, 2362, 2384, 2240, 2241, 2250,
       2289, 2312, 2322, 2364, 2242, 2246, 2313, 2314, 2317, 2385, 2386,
       2242, 2243, 2313, 2314, 2315, 2386, 2387, 2243, 2244, 2314, 2315,
       2316, 2387, 2388, 2244, 2245, 2290, 2315, 2316, 2365, 2388, 2246,
       2248, 2313, 2317, 2320, 2385, 2389, 2247, 2290, 2318, 2319, 2365,
       2390, 2391, 2247, 2291, 2318, 2319, 2323, 2390, 2392, 2248, 2249,
       2317, 2320, 2321, 2389, 2393, 2249, 2252, 2266, 2320, 2321, 2340,
       2393, 2250, 2251, 2312, 2322, 2364, 2366, 2394, 2251, 2291, 2319,
       2323, 2366, 2392, 2395, 2222, 2294, 2295, 2324, 2360, 2369, 2396,
       2254, 2255, 2325, 2326, 2327, 2397, 2398, 2254, 2257, 2325, 2326,
       2328, 2397, 2399, 2255, 2256, 2268, 2325, 2327, 2342, 2398, 2257,
       2258, 2326, 2328, 2329, 2399, 2400, 2258, 2259, 2328, 2329, 2330,
       2400, 2401, 2259, 2297, 2329, 2330, 2371, 2401, 2402, 2235, 2260,
       2308, 2331, 2333, 2382, 2403, 2228, 2298, 2332, 2339, 2372, 2404,
       2260, 2261, 2331, 2333, 2334, 2403, 2405, 2261, 2262, 2333, 2334,
       2335, 2405, 2262, 2299, 2334, 2335, 2373, 2405, 2406, 2263, 2264,
       2336, 2337, 2338, 2407, 2408, 2263, 2265, 2300, 2336, 2337, 2375,
       2407, 2264, 2299, 2336, 2338, 2373, 2408, 2409, 2228, 2300, 2332,
       2339, 2374, 2404, 2410, 2266, 2304, 2321, 2340, 2379, 2393, 2411,
       2267, 2269, 2341, 2342, 2343, 2412, 2413, 2267, 2268, 2327, 2341,
       2342, 2398, 2413, 2450, 2269, 2270, 2341, 2343, 2344, 2412, 2414,
       2270, 2271, 2343, 2344, 2345, 2414, 2415, 2271, 2272, 2344, 2345,
       2346, 2415, 2416, 2272, 2305, 2345, 2346, 2380, 2416, 2417, 2273,
       2305, 2306, 2347, 2380, 2381, 2418, 2274, 2307, 2348, 2349, 2353,
       2419, 2420, 2274, 2275, 2348, 2349, 2350, 2420, 2421, 2275, 2276,
       2349, 2350, 2351, 2421, 2422, 2276, 2278, 2350, 2351, 2354, 2422,
       2423, 2277, 2279, 2352, 2353, 2355, 2424, 2425, 2277, 2307, 2348,
       2352, 2353, 2419, 2425, 2278, 2308, 2351, 2354, 2382, 2423, 2426,
       2279, 2280, 2352, 2355, 2356, 2424, 2427, 2280, 2306, 2355, 2356,
       2381, 2427, 2428, 2281, 2282, 2311, 2357, 2358, 2384, 2429, 2282,
       2283, 2357, 2358, 2359, 2429, 2430, 2283, 2310, 2358, 2359, 2383,
       2430, 2431, 2284, 2285, 2295, 2324, 2360, 2361, 2396, 2285, 2286,
       2360, 2361, 2362, 2396, 2432, 2286, 2311, 2361, 2362, 2384, 2432,
       2433, 2288, 2289, 2309, 2310, 2363, 2364, 2383, 2289, 2312, 2322,
       2363, 2364, 2383, 2394, 2290, 2316, 2318, 2365, 2388, 2391, 2434,
       2251, 2322, 2323, 2366, 2394, 2395, 2435, 2292, 2294, 2367, 2368,
       2369, 2436, 2437, 2292, 2293, 2296, 2367, 2368, 2370, 2436, 2294,
       2324, 2367, 2369, 2396, 2437, 2438, 2296, 2297, 2368, 2370, 2371,
       2436, 2439, 2297, 2330, 2370, 2371, 2402, 2439, 2440, 2298, 2301,
       2332, 2372, 2376, 2404, 2441, 2299, 2335, 2338, 2373, 2406, 2409,
       2442, 2300, 2339, 2374, 2375, 2410, 2443, 2444, 2300, 2337, 2374,
       2375, 2407, 2444, 2445, 2301, 2302, 2372, 2376, 2377, 2441, 2446,
       2302, 2303, 2376, 2377, 2378, 2446, 2447, 2303, 2304, 2377, 2378,
       2379, 2447, 2448, 2304, 2340, 2378, 2379, 2411, 2448, 2449, 2305,
       2346, 2347, 2380, 2417, 2418, 2451, 2306, 2347, 2356, 2381, 2418,
       2428, 2452, 2454, 2308, 2331, 2354, 2382, 2403, 2426, 2453, 2310,
       2359, 2363, 2364, 2383, 2394, 2431, 2455, 2311, 2357, 2362, 2384,
       2429, 2433, 2456, 2313, 2317, 2385, 2386, 2389, 2457, 2458, 2313,
       2314, 2385, 2386, 2387, 2458, 2459, 2314, 2315, 2386, 2387, 2388,
       2459, 2460, 2315, 2316, 2365, 2387, 2388, 2434, 2460, 2317, 2320,
       2385, 2389, 2393, 2457, 2461, 2318, 2319, 2390, 2391, 2392, 2462,
       2463, 2318, 2365, 2390, 2391, 2434, 2463, 2464, 2319, 2323, 2390,
       2392, 2395, 2462, 2465, 2320, 2321, 2340, 2389, 2393, 2411, 2461,
       2322, 2364, 2366, 2383, 2394, 2435, 2455, 2323, 2366, 2392, 2395,
       2435, 2465, 2466, 2324, 2360, 2361, 2369, 2396, 2432, 2438, 2325,
       2326, 2397, 2398, 2399, 2467, 2468, 2325, 2327, 2342, 2397, 2398,
       2450, 2468, 2326, 2328, 2397, 2399, 2400, 2467, 2469, 2328, 2329,
       2399, 2400, 2401, 2469, 2470, 2329, 2330, 2400, 2401, 2402, 2470,
       2471, 2330, 2371, 2401, 2402, 2440, 2471, 2472, 2331, 2333, 2382,
       2403, 2405, 2453, 2473, 2332, 2339, 2372, 2404, 2410, 2441, 2474,
       2333, 2334, 2335, 2403, 2405, 2406, 2473, 2335, 2373, 2405, 2406,
       2442, 2473, 2475, 2336, 2337, 2375, 2407, 2408, 2445, 2476, 2336,
       2338, 2407, 2408, 2409, 2476, 2477, 2338, 2373, 2408, 2409, 2442,
       2477, 2478, 2339, 2374, 2404, 2410, 2443, 2474, 2479, 2340, 2379,
       2393, 2411, 2449, 2461, 2480, 2341, 2343, 2412, 2413, 2414, 2481,
       2482, 2341, 2342, 2412, 2413, 2450, 2481, 2483, 2343, 2344, 2412,
       2414, 2415, 2482, 2484, 2344, 2345, 2414, 2415, 2416, 2484, 2485,
       2519, 2345, 2346, 2415, 2416, 2417, 2485, 2486, 2346, 2380, 2416,
       2417, 2451, 2486, 2487, 2520, 2347, 2380, 2381, 2418, 2451, 2452,
       2488, 2348, 2353, 2419, 2420, 2425, 2489, 2490, 2348, 2349, 2419,
       2420, 2421, 2490, 2491, 2349, 2350, 2420, 2421, 2422, 2491, 2492,
       2350, 2351, 2421, 2422, 2423, 2492, 2493, 2351, 2354, 2422, 2423,
       2426, 2493, 2494, 2352, 2355, 2424, 2425, 2427, 2495, 2496, 2352,
       2353, 2419, 2424, 2425, 2489, 2495, 2354, 2382, 2423, 2426, 2453,
       2494, 2497, 2355, 2356, 2424, 2427, 2428, 2496, 2498, 2356, 2381,
       2427, 2428, 2454, 2498, 2357, 2358, 2384, 2429, 2430, 2456, 2499,
       2358, 2359, 2429, 2430, 2431, 2499, 2500, 2359, 2383, 2430, 2431,
       2455, 2500, 2501, 2361, 2362, 2396, 2432, 2433, 2438, 2502, 2362,
       2384, 2432, 2433, 2456, 2502, 2503, 2365, 2388, 2391, 2434, 2460,
       2464, 2504, 2366, 2394, 2395, 2435, 2455, 2466, 2501, 2367, 2368,
       2370, 2436, 2437, 2439, 2505, 2367, 2369, 2436, 2437, 2438, 2505,
       2506, 2369, 2396, 2432, 2437, 2438, 2502, 2506, 2370, 2371, 2436,
       2439, 2440, 2505, 2507, 2371, 2402, 2439, 2440, 2472, 2507, 2508,
       2372, 2376, 2404, 2441, 2446, 2474, 2509, 2373, 2406, 2409, 2442,
       2475, 2478, 2510, 2374, 2410, 2443, 2444, 2479, 2511, 2512, 2374,
       2375, 2443, 2444, 2445, 2511, 2513, 2375, 2407, 2444, 2445, 2476,
       2513, 2514, 2376, 2377, 2441, 2446, 2447, 2509, 2515, 2549, 2377,
       2378, 2446, 2447, 2448, 2515, 2378, 2379, 2447, 2448, 2449, 2515,
       2516, 2379, 2411, 2448, 2449, 2480, 2516, 2517, 2342, 2398, 2413,
       2450, 2468, 2483, 2518, 2380, 2417, 2418, 2451, 2487, 2488, 2521,
       2381, 2418, 2452, 2454, 2488, 2522, 2523, 2382, 2403, 2426, 2453,
       2473, 2497, 2524, 2381, 2428, 2452, 2454, 2498, 2523, 2525, 2383,
       2394, 2431, 2435, 2455, 2501, 2384, 2429, 2433, 2456, 2499, 2503,
       2526, 2385, 2389, 2457, 2458, 2461, 2527, 2528, 2385, 2386, 2457,
       2458, 2459, 2528, 2529, 2386, 2387, 2458, 2459, 2460, 2529, 2530,
       2387, 2388, 2434, 2459, 2460, 2504, 2530, 2389, 2393, 2411, 2457,
       2461, 2480, 2527, 2390, 2392, 2462, 2463, 2465, 2531, 2532, 2390,
       2391, 2462, 2463, 2464, 2531, 2533, 2391, 2434, 2463, 2464, 2504,
       2533, 2534, 2392, 2395, 2462, 2465, 2466, 2532, 2535, 2395, 2435,
       2465, 2466, 2501, 2535, 2536, 2397, 2399, 2467, 2468, 2469, 2537,
       2538, 2397, 2398, 2450, 2467, 2468, 2518, 2538, 2399, 2400, 2467,
       2469, 2470, 2537, 2539, 2400, 2401, 2469, 2470, 2471, 2539, 2540,
       2401, 2402, 2470, 2471, 2472, 2540, 2541, 2402, 2440, 2471, 2472,
       2508, 2541, 2542, 2403, 2405, 2406, 2453, 2473, 2475, 2524, 2404,
       2410, 2441, 2474, 2479, 2509, 2543, 2406, 2442, 2473, 2475, 2510,
       2524, 2544, 2407, 2408, 2445, 2476, 2477, 2514, 2545, 2408, 2409,
       2476, 2477, 2478, 2545, 2546, 2409, 2442, 2477, 2478, 2510, 2546,
       2547, 2410, 2443, 2474, 2479, 2512, 2543, 2548, 2411, 2449, 2461,
       2480, 2517, 2527, 2550, 2412, 2413, 2481, 2482, 2483, 2551, 2552,
       2412, 2414, 2481, 2482, 2484, 2552, 2553, 2413, 2450, 2481, 2483,
       2518, 2551, 2554, 2414, 2415, 2482, 2484, 2519, 2553, 2555, 2415,
       2416, 2485, 2486, 2519, 2556, 2557, 2416, 2417, 2485, 2486, 2520,
       2557, 2558, 2417, 2451, 2487, 2520, 2521, 2559, 2560, 2418, 2451,
       2452, 2488, 2521, 2522, 2561, 2419, 2425, 2489, 2490, 2495, 2562,
       2563, 2419, 2420, 2489, 2490, 2491, 2563, 2564, 2420, 2421, 2490,
       2491, 2492, 2564, 2565, 2421, 2422, 2491, 2492, 2493, 2565, 2566,
       2422, 2423, 2492, 2493, 2494, 2566, 2567, 2423, 2426, 2493, 2494,
       2497, 2567, 2568, 2424, 2425, 2489, 2495, 2496, 2562, 2569, 2424,
       2427, 2495, 2496, 2498, 2569, 2570, 2426, 2453, 2494, 2497, 2524,
       2568, 2571, 2427, 2428, 2454, 2496, 2498, 2525, 2570, 2429, 2430,
       2456, 2499, 2500, 2526, 2572, 2430, 2431, 2499, 2500, 2501, 2536,
       2572, 2431, 2435, 2455, 2466, 2500, 2501, 2536, 2432, 2433, 2438,
       2502, 2503, 2506, 2573, 2433, 2456, 2502, 2503, 2526, 2573, 2574,
       2434, 2460, 2464, 2504, 2530, 2534, 2575, 2436, 2437, 2439, 2505,
       2506, 2507, 2576, 2437, 2438, 2502, 2505, 2506, 2573, 2576, 2439,
       2440, 2505, 2507, 2508, 2576, 2577, 2440, 2472, 2507, 2508, 2542,
       2577, 2578, 2441, 2446, 2474, 2509, 2543, 2549, 2579, 2442, 2475,
       2478, 2510, 2544, 2547, 2580, 2443, 2444, 2511, 2512, 2513, 2581,
       2582, 2443, 2479, 2511, 2512, 2548, 2582, 2583, 2444, 2445, 2511,
       2513, 2514, 2581, 2584, 2445, 2476, 2513, 2514, 2545, 2584, 2585,
       2618, 2446, 2447, 2448, 2515, 2516, 2549, 2586, 2448, 2449, 2515,
       2516, 2517, 2586, 2587, 2449, 2480, 2516, 2517, 2550, 2587, 2588,
       2450, 2468, 2483, 2518, 2538, 2554, 2589, 2415, 2484, 2485, 2519,
       2555, 2556, 2590, 2417, 2486, 2487, 2520, 2558, 2559, 2591, 2451,
       2487, 2488, 2521, 2560, 2561, 2592, 2452, 2488, 2522, 2523, 2561,
       2593, 2594, 2452, 2454, 2522, 2523, 2525, 2594, 2595, 2453, 2473,
       2475, 2497, 2524, 2544, 2571, 2454, 2498, 2523, 2525, 2570, 2595,
       2596, 2456, 2499, 2503, 2526, 2572, 2574, 2597, 2457, 2461, 2480,
       2527, 2528, 2550, 2598, 2457, 2458, 2527, 2528, 2529, 2598, 2599,
       2458, 2459, 2528, 2529, 2530, 2599, 2600, 2459, 2460, 2504, 2529,
       2530, 2575, 2600, 2462, 2463, 2531, 2532, 2533, 2601, 2602, 2462,
       2465, 2531, 2532, 2535, 2601, 2603, 2463, 2464, 2531, 2533, 2534,
       2602, 2604, 2464, 2504, 2533, 2534, 2575, 2604, 2605, 2465, 2466,
       2532, 2535, 2536, 2603, 2606, 2466, 2500, 2501, 2535, 2536, 2572,
       2606, 2467, 2469, 2537, 2538, 2539, 2607, 2608, 2467, 2468, 2518,
       2537, 2538, 2589, 2608, 2469, 2470, 2537, 2539, 2540, 2607, 2609,
       2470, 2471, 2539, 2540, 2541, 2609, 2610, 2471, 2472, 2540, 2541,
       2542, 2610, 2611, 2472, 2508, 2541, 2542, 2578, 2611, 2612, 2474,
       2479, 2509, 2543, 2548, 2579, 2613, 2475, 2510, 2524, 2544, 2571,
       2580, 2614, 2476, 2477, 2514, 2545, 2546, 2585, 2477, 2478, 2545,
       2546, 2547, 2585, 2615, 2478, 2510, 2546, 2547, 2580, 2615, 2616,
       2479, 2512, 2543, 2548, 2583, 2613, 2617, 2446, 2509, 2515, 2549,
       2579, 2586, 2619, 2480, 2517, 2527, 2550, 2588, 2598, 2620, 2481,
       2483, 2551, 2552, 2554, 2621, 2622, 2481, 2482, 2551, 2552, 2553,
       2622, 2623, 2482, 2484, 2552, 2553, 2555, 2623, 2624, 2483, 2518,
       2551, 2554, 2589, 2621, 2625, 2484, 2519, 2553, 2555, 2590, 2624,
       2626, 2660, 2485, 2519, 2556, 2557, 2590, 2627, 2628, 2485, 2486,
       2556, 2557, 2558, 2627, 2629, 2486, 2520, 2557, 2558, 2591, 2629,
       2630, 2487, 2520, 2559, 2560, 2591, 2631, 2632, 2487, 2521, 2559,
       2560, 2592, 2631, 2633, 2488, 2521, 2522, 2561, 2592, 2593, 2634,
       2489, 2495, 2562, 2563, 2569, 2635, 2636, 2489, 2490, 2562, 2563,
       2564, 2635, 2637, 2490, 2491, 2563, 2564, 2565, 2637, 2638, 2491,
       2492, 2564, 2565, 2566, 2638, 2639, 2492, 2493, 2565, 2566, 2567,
       2639, 2640, 2493, 2494, 2566, 2567, 2568, 2640, 2641, 2494, 2497,
       2567, 2568, 2571, 2641, 2642, 2495, 2496, 2562, 2569, 2570, 2636,
       2643, 2496, 2498, 2525, 2569, 2570, 2596, 2643, 2497, 2524, 2544,
       2568, 2571, 2614, 2642, 2499, 2500, 2526, 2536, 2572, 2597, 2606,
       2502, 2503, 2506, 2573, 2574, 2576, 2644, 2647, 2503, 2526, 2573,
       2574, 2597, 2644, 2645, 2504, 2530, 2534, 2575, 2600, 2605, 2646,
       2505, 2506, 2507, 2573, 2576, 2577, 2647, 2507, 2508, 2576, 2577,
       2578, 2647, 2648, 2508, 2542, 2577, 2578, 2612, 2648, 2649, 2509,
       2543, 2549, 2579, 2613, 2619, 2650, 2510, 2544, 2547, 2580, 2614,
       2616, 2651, 2511, 2513, 2581, 2582, 2584, 2652, 2653, 2511, 2512,
       2581, 2582, 2583, 2653, 2654, 2512, 2548, 2582, 2583, 2617, 2654,
       2655, 2513, 2514, 2581, 2584, 2618, 2652, 2656, 2514, 2545, 2546,
       2585, 2615, 2618, 2515, 2516, 2549, 2586, 2587, 2619, 2657, 2516,
       2517, 2586, 2587, 2588, 2657, 2658, 2517, 2550, 2587, 2588, 2620,
       2658, 2659, 2518, 2538, 2554, 2589, 2608, 2625, 2661, 2519, 2555,
       2556, 2590, 2626, 2628, 2662, 2520, 2558, 2559, 2591, 2630, 2632,
       2663, 2521, 2560, 2561, 2592, 2633, 2634, 2664, 2522, 2561, 2593,
       2594, 2634, 2665, 2666, 2522, 2523, 2593, 2594, 2595, 2666, 2667,
       2523, 2525, 2594, 2595, 2596, 2667, 2668, 2525, 2570, 2595, 2596,
       2643, 2668, 2669, 2526, 2572, 2574, 2597, 2606, 2645, 2670, 2527,
       2528, 2550, 2598, 2599, 2620, 2671, 2528, 2529, 2598, 2599, 2600,
       2671, 2672, 2529, 2530, 2575, 2599, 2600, 2646, 2672, 2531, 2532,
       2601, 2602, 2603, 2673, 2674, 2531, 2533, 2601, 2602, 2604, 2673,
       2675, 2532, 2535, 2601, 2603, 2606, 2670, 2674, 2533, 2534, 2602,
       2604, 2605, 2675, 2676, 2534, 2575, 2604, 2605, 2646, 2676, 2677,
       2535, 2536, 2572, 2597, 2603, 2606, 2670, 2537, 2539, 2607, 2608,
       2609, 2678, 2679, 2537, 2538, 2589, 2607, 2608, 2661, 2679, 2539,
       2540, 2607, 2609, 2610, 2678, 2680, 2540, 2541, 2609, 2610, 2611,
       2680, 2681, 2541, 2542, 2610, 2611, 2612, 2681, 2682, 2542, 2578,
       2611, 2612, 2649, 2682, 2683, 2543, 2548, 2579, 2613, 2617, 2650,
       2684, 2544, 2571, 2580, 2614, 2642, 2651, 2685, 2546, 2547, 2585,
       2615, 2616, 2618, 2686, 2547, 2580, 2615, 2616, 2651, 2686, 2687,
       2548, 2583, 2613, 2617, 2655, 2684, 2688, 2514, 2584, 2585, 2615,
       2618, 2656, 2686, 2549, 2579, 2586, 2619, 2650, 2657, 2689, 2550,
       2588, 2598, 2620, 2659, 2671, 2690, 2551, 2554, 2621, 2622, 2625,
       2691, 2692, 2551, 2552, 2621, 2622, 2623, 2692, 2693, 2552, 2553,
       2622, 2623, 2624, 2693, 2694, 2553, 2555, 2623, 2624, 2660, 2694,
       2695, 2734, 2554, 2589, 2621, 2625, 2661, 2691, 2696, 2555, 2590,
       2626, 2660, 2662, 2697, 2698, 2556, 2557, 2627, 2628, 2629, 2699,
       2700, 2556, 2590, 2627, 2628, 2662, 2699, 2701, 2557, 2558, 2627,
       2629, 2630, 2700, 2702, 2558, 2591, 2629, 2630, 2663, 2702, 2703,
       2559, 2560, 2631, 2632, 2633, 2704, 2705, 2559, 2591, 2631, 2632,
       2663, 2704, 2706, 2560, 2592, 2631, 2633, 2664, 2705, 2707, 2561,
       2592, 2593, 2634, 2664, 2665, 2708, 2709, 2562, 2563, 2635, 2636,
       2637, 2710, 2711, 2562, 2569, 2635, 2636, 2643, 2710, 2712, 2563,
       2564, 2635, 2637, 2638, 2711, 2713, 2564, 2565, 2637, 2638, 2639,
       2713, 2714, 2565, 2566, 2638, 2639, 2640, 2714, 2715, 2566, 2567,
       2639, 2640, 2641, 2715, 2716, 2567, 2568, 2640, 2641, 2642, 2716,
       2717, 2568, 2571, 2614, 2641, 2642, 2685, 2717, 2569, 2570, 2596,
       2636, 2643, 2669, 2712, 2573, 2574, 2644, 2645, 2647, 2718, 2574,
       2597, 2644, 2645, 2670, 2718, 2719, 2575, 2600, 2605, 2646, 2672,
       2677, 2720, 2573, 2576, 2577, 2644, 2647, 2648, 2718, 2577, 2578,
       2647, 2648, 2649, 2718, 2721, 2578, 2612, 2648, 2649, 2683, 2721,
       2722, 2579, 2613, 2619, 2650, 2684, 2689, 2723, 2580, 2614, 2616,
       2651, 2685, 2687, 2724, 2581, 2584, 2652, 2653, 2656, 2725, 2726,
       2581, 2582, 2652, 2653, 2654, 2726, 2727, 2582, 2583, 2653, 2654,
       2655, 2727, 2728, 2583, 2617, 2654, 2655, 2688, 2728, 2729, 2584,
       2618, 2652, 2656, 2686, 2725, 2730, 2586, 2587, 2619, 2657, 2658,
       2689, 2731, 2587, 2588, 2657, 2658, 2659, 2731, 2732, 2588, 2620,
       2658, 2659, 2690, 2732, 2733, 2555, 2624, 2626, 2660, 2695, 2697,
       2589, 2608, 2625, 2661, 2679, 2696, 2735, 2590, 2626, 2628, 2662,
       2698, 2701, 2736, 2591, 2630, 2632, 2663, 2703, 2706, 2737, 2592,
       2633, 2634, 2664, 2707, 2708, 2738, 2593, 2634, 2665, 2666, 2709,
       2739, 2593, 2594, 2665, 2666, 2667, 2739, 2740, 2594, 2595, 2666,
       2667, 2668, 2740, 2741, 2595, 2596, 2667, 2668, 2669, 2741, 2742,
       2596, 2643, 2668, 2669, 2712, 2742, 2743, 2597, 2603, 2606, 2645,
       2670, 2674, 2719, 2598, 2599, 2620, 2671, 2672, 2690, 2744, 2599,
       2600, 2646, 2671, 2672, 2720, 2744, 2601, 2602, 2673, 2674, 2675,
       2745, 2746, 2601, 2603, 2670, 2673, 2674, 2719, 2745, 2602, 2604,
       2673, 2675, 2676, 2746, 2747, 2791, 2604, 2605, 2675, 2676, 2677,
       2747, 2605, 2646, 2676, 2677, 2720, 2747, 2748, 2607, 2609, 2678,
       2679, 2680, 2749, 2750, 2607, 2608, 2661, 2678, 2679, 2735, 2749,
       2609, 2610, 2678, 2680, 2681, 2750, 2751, 2610, 2611, 2680, 2681,
       2682, 2751, 2752, 2611, 2612, 2681, 2682, 2683, 2752, 2753, 2612,
       2649, 2682, 2683, 2722, 2753, 2754, 2613, 2617, 2650, 2684, 2688,
       2723, 2755, 2614, 2642, 2651, 2685, 2717, 2724, 2756, 2615, 2616,
       2618, 2656, 2686, 2687, 2730, 2616, 2651, 2686, 2687, 2724, 2730,
       2757, 2617, 2655, 2684, 2688, 2729, 2755, 2758, 2619, 2650, 2657,
       2689, 2723, 2731, 2759, 2620, 2659, 2671, 2690, 2733, 2744, 2760,
       2621, 2625, 2691, 2692, 2696, 2761, 2762, 2621, 2622, 2691, 2692,
       2693, 2762, 2763, 2622, 2623, 2692, 2693, 2694, 2763, 2764, 2623,
       2624, 2693, 2694, 2734, 2764, 2624, 2660, 2695, 2697, 2734, 2765,
       2766, 2806, 2625, 2661, 2691, 2696, 2735, 2761, 2767, 2626, 2660,
       2695, 2697, 2698, 2766, 2768, 2626, 2662, 2697, 2698, 2736, 2768,
       2769, 2627, 2628, 2699, 2700, 2701, 2770, 2771, 2627, 2629, 2699,
       2700, 2702, 2771, 2772, 2628, 2662, 2699, 2701, 2736, 2770, 2773,
       2629, 2630, 2700, 2702, 2703, 2772, 2774, 2630, 2663, 2702, 2703,
       2737, 2774, 2775, 2631, 2632, 2704, 2705, 2706, 2776, 2777, 2631,
       2633, 2704, 2705, 2707, 2776, 2778, 2632, 2663, 2704, 2706, 2737,
       2777, 2779, 2633, 2664, 2705, 2707, 2738, 2778, 2780, 2634, 2664,
       2708, 2709, 2738, 2781, 2782, 2634, 2665, 2708, 2709, 2739, 2781,
       2783, 2811, 2635, 2636, 2710, 2711, 2712, 2784, 2785, 2635, 2637,
       2710, 2711, 2713, 2785, 2786, 2636, 2643, 2669, 2710, 2712, 2743,
       2784, 2637, 2638, 2711, 2713, 2714, 2786, 2787, 2638, 2639, 2713,
       2714, 2715, 2787, 2788, 2639, 2640, 2714, 2715, 2716, 2788, 2640,
       2641, 2715, 2716, 2717, 2788, 2789, 2641, 2642, 2685, 2716, 2717,
       2756, 2789, 2644, 2645, 2647, 2648, 2718, 2719, 2721, 2645, 2670,
       2674, 2718, 2719, 2721, 2745, 2646, 2672, 2677, 2720, 2744, 2748,
       2790, 2648, 2649, 2718, 2719, 2721, 2722, 2745, 2649, 2683, 2721,
       2722, 2745, 2746, 2754, 2650, 2684, 2689, 2723, 2755, 2759, 2792,
       2800, 2651, 2685, 2687, 2724, 2756, 2757, 2793, 2652, 2656, 2725,
       2726, 2730, 2794, 2795, 2652, 2653, 2725, 2726, 2727, 2794, 2796,
       2653, 2654, 2726, 2727, 2728, 2796, 2797, 2654, 2655, 2727, 2728,
       2729, 2797, 2798, 2655, 2688, 2728, 2729, 2758, 2798, 2799, 2656,
       2686, 2687, 2725, 2730, 2757, 2795, 2657, 2658, 2689, 2731, 2732,
       2759, 2801, 2658, 2659, 2731, 2732, 2733, 2801, 2802, 2659, 2690,
       2732, 2733, 2760, 2802, 2803, 2624, 2694, 2695, 2734, 2764, 2765,
       2804, 2805, 2661, 2679, 2696, 2735, 2749, 2767, 2807, 2662, 2698,
       2701, 2736, 2769, 2773, 2808, 2663, 2703, 2706, 2737, 2775, 2779,
       2809, 2664, 2707, 2708, 2738, 2780, 2782, 2810, 2665, 2666, 2709,
       2739, 2740, 2783, 2812, 2666, 2667, 2739, 2740, 2741, 2812, 2813,
       2667, 2668, 2740, 2741, 2742, 2813, 2814, 2668, 2669, 2741, 2742,
       2743, 2814, 2815, 2669, 2712, 2742, 2743, 2784, 2815, 2816, 2671,
       2672, 2690, 2720, 2744, 2760, 2790, 2673, 2674, 2719, 2721, 2722,
       2745, 2746, 2673, 2675, 2722, 2745, 2746, 2754, 2791, 2675, 2676,
       2677, 2747, 2748, 2791, 2817, 2677, 2720, 2747, 2748, 2790, 2817,
       2818, 2678, 2679, 2735, 2749, 2750, 2807, 2819, 2678, 2680, 2749,
       2750, 2751, 2819, 2820, 2857, 2680, 2681, 2750, 2751, 2752, 2820,
       2821, 2681, 2682, 2751, 2752, 2753, 2821, 2822, 2682, 2683, 2752,
       2753, 2754, 2822, 2823, 2683, 2722, 2746, 2753, 2754, 2791, 2823,
       2684, 2688, 2723, 2755, 2758, 2792, 2824, 2685, 2717, 2724, 2756,
       2789, 2793, 2825, 2687, 2724, 2730, 2757, 2793, 2795, 2826, 2688,
       2729, 2755, 2758, 2799, 2824, 2827, 2689, 2723, 2731, 2759, 2800,
       2801, 2828, 2690, 2733, 2744, 2760, 2790, 2803, 2829, 2691, 2696,
       2761, 2762, 2767, 2830, 2831, 2691, 2692, 2761, 2762, 2763, 2830,
       2832, 2692, 2693, 2762, 2763, 2764, 2832, 2833, 2693, 2694, 2734,
       2763, 2764, 2804, 2833, 2695, 2734, 2765, 2805, 2806, 2834, 2835,
       2695, 2697, 2766, 2768, 2781, 2806, 2811, 2836, 2696, 2735, 2761,
       2767, 2807, 2831, 2837, 2697, 2698, 2766, 2768, 2769, 2781, 2782,
       2698, 2736, 2768, 2769, 2782, 2808, 2810, 2699, 2701, 2770, 2771,
       2773, 2838, 2839, 2699, 2700, 2770, 2771, 2772, 2839, 2840, 2700,
       2702, 2771, 2772, 2774, 2840, 2841, 2701, 2736, 2770, 2773, 2808,
       2838, 2842, 2702, 2703, 2772, 2774, 2775, 2841, 2843, 2703, 2737,
       2774, 2775, 2809, 2843, 2844, 2704, 2705, 2776, 2777, 2778, 2845,
       2846, 2704, 2706, 2776, 2777, 2779, 2845, 2847, 2705, 2707, 2776,
       2778, 2780, 2846, 2848, 2706, 2737, 2777, 2779, 2809, 2847, 2849,
       2876, 2707, 2738, 2778, 2780, 2810, 2848, 2850, 2708, 2709, 2766,
       2768, 2781, 2782, 2811, 2708, 2738, 2768, 2769, 2781, 2782, 2810,
       2709, 2739, 2783, 2811, 2812, 2851, 2852, 2710, 2712, 2743, 2784,
       2785, 2816, 2853, 2710, 2711, 2784, 2785, 2786, 2853, 2854, 2711,
       2713, 2785, 2786, 2787, 2854, 2855, 2713, 2714, 2786, 2787, 2788,
       2855, 2856, 2714, 2715, 2716, 2787, 2788, 2789, 2856, 2716, 2717,
       2756, 2788, 2789, 2825, 2856, 2720, 2744, 2748, 2760, 2790, 2818,
       2829, 2675, 2746, 2747, 2754, 2791, 2817, 2823, 2723, 2755, 2792,
       2800, 2824, 2858, 2859, 2724, 2756, 2757, 2793, 2825, 2826, 2860,
       2725, 2726, 2794, 2795, 2796, 2861, 2862, 2725, 2730, 2757, 2794,
       2795, 2826, 2861, 2726, 2727, 2794, 2796, 2797, 2862, 2863, 2727,
       2728, 2796, 2797, 2798, 2863, 2864, 2728, 2729, 2797, 2798, 2799,
       2864, 2865, 2729, 2758, 2798, 2799, 2827, 2865, 2866, 2723, 2759,
       2792, 2800, 2828, 2859, 2731, 2732, 2759, 2801, 2802, 2828, 2867,
       2732, 2733, 2801, 2802, 2803, 2867, 2868, 2733, 2760, 2802, 2803,
       2829, 2868, 2869, 2734, 2764, 2804, 2805, 2833, 2870, 2871, 2734,
       2765, 2804, 2805, 2834, 2871, 2872, 2695, 2765, 2766, 2806, 2835,
       2836, 2873, 2735, 2749, 2767, 2807, 2819, 2837, 2874, 2736, 2769,
       2773, 2808, 2810, 2842, 2850, 2737, 2775, 2779, 2809, 2844, 2849,
       2875, 2738, 2769, 2780, 2782, 2808, 2810, 2850, 2709, 2766, 2781,
       2783, 2811, 2836, 2851, 2739, 2740, 2783, 2812, 2813, 2852, 2877,
       2740, 2741, 2812, 2813, 2814, 2877, 2878, 2741, 2742, 2813, 2814,
       2815, 2878, 2879, 2742, 2743, 2814, 2815, 2816, 2879, 2880, 2743,
       2784, 2815, 2816, 2853, 2880, 2881, 2747, 2748, 2791, 2817, 2818,
       2823, 2882, 2748, 2790, 2817, 2818, 2829, 2882, 2883, 2749, 2750,
       2807, 2819, 2857, 2874, 2884, 2750, 2751, 2820, 2821, 2857, 2885,
       2751, 2752, 2820, 2821, 2822, 2885, 2886, 2752, 2753, 2821, 2822,
       2823, 2882, 2886, 2753, 2754, 2791, 2817, 2822, 2823, 2882, 2755,
       2758, 2792, 2824, 2827, 2858, 2887, 2756, 2789, 2793, 2825, 2856,
       2860, 2888, 2757, 2793, 2795, 2826, 2860, 2861, 2889, 2758, 2799,
       2824, 2827, 2866, 2887, 2890, 2759, 2800, 2801, 2828, 2859, 2867,
       2891, 2760, 2790, 2803, 2818, 2829, 2869, 2883, 2761, 2762, 2830,
       2831, 2832, 2892, 2893, 2761, 2767, 2830, 2831, 2837, 2892, 2894,
       2762, 2763, 2830, 2832, 2833, 2893, 2895, 2763, 2764, 2804, 2832,
       2833, 2870, 2895, 2896, 2765, 2805, 2834, 2835, 2872, 2897, 2898,
       2765, 2806, 2834, 2835, 2873, 2898, 2899, 2766, 2806, 2811, 2836,
       2851, 2873, 2900, 2767, 2807, 2831, 2837, 2874, 2894, 2901, 2770,
       2773, 2838, 2839, 2842, 2902, 2903, 2770, 2771, 2838, 2839, 2840,
       2902, 2904, 2771, 2772, 2839, 2840, 2841, 2904, 2905, 2772, 2774,
       2840, 2841, 2843, 2905, 2906, 2773, 2808, 2838, 2842, 2850, 2903,
       2907, 2774, 2775, 2841, 2843, 2844, 2906, 2908, 2775, 2809, 2843,
       2844, 2875, 2908, 2909, 2776, 2777, 2845, 2846, 2847, 2910, 2911,
       2776, 2778, 2845, 2846, 2848, 2911, 2912, 2777, 2779, 2845, 2847,
       2876, 2910, 2913, 2941, 2778, 2780, 2846, 2848, 2850, 2907, 2912,
       2779, 2809, 2849, 2875, 2876, 2914, 2915, 2780, 2808, 2810, 2842,
       2848, 2850, 2907, 2783, 2811, 2836, 2851, 2852, 2900, 2916, 2783,
       2812, 2851, 2852, 2877, 2916, 2917, 2784, 2785, 2816, 2853, 2854,
       2881, 2918, 2785, 2786, 2853, 2854, 2855, 2918, 2919, 2786, 2787,
       2854, 2855, 2856, 2888, 2919, 2946, 2787, 2788, 2789, 2825, 2855,
       2856, 2888, 2750, 2819, 2820, 2857, 2884, 2885, 2920, 2792, 2824,
       2858, 2859, 2887, 2921, 2922, 2792, 2800, 2828, 2858, 2859, 2891,
       2922, 2930, 2793, 2825, 2826, 2860, 2888, 2889, 2923, 2794, 2795,
       2826, 2861, 2862, 2889, 2924, 2794, 2796, 2861, 2862, 2863, 2924,
       2925, 2796, 2797, 2862, 2863, 2864, 2925, 2926, 2797, 2798, 2863,
       2864, 2865, 2926, 2927, 2798, 2799, 2864, 2865, 2866, 2927, 2928,
       2799, 2827, 2865, 2866, 2890, 2928, 2929, 2801, 2802, 2828, 2867,
       2868, 2891, 2931, 2802, 2803, 2867, 2868, 2869, 2931, 2932, 2803,
       2829, 2868, 2869, 2883, 2932, 2933, 2804, 2833, 2870, 2871, 2896,
       2934, 2935, 2804, 2805, 2870, 2871, 2872, 2935, 2936, 2805, 2834,
       2871, 2872, 2897, 2936, 2937, 2806, 2835, 2836, 2873, 2899, 2900,
       2938, 2807, 2819, 2837, 2874, 2884, 2901, 2939, 2809, 2844, 2849,
       2875, 2909, 2914, 2940, 2779, 2847, 2849, 2876, 2913, 2915, 2812,
       2813, 2852, 2877, 2878, 2917, 2942, 2813, 2814, 2877, 2878, 2879,
       2942, 2943, 2814, 2815, 2878, 2879, 2880, 2943, 2944, 2815, 2816,
       2879, 2880, 2881, 2944, 2945, 2816, 2853, 2880, 2881, 2918, 2945,
       2947, 2817, 2818, 2822, 2823, 2882, 2883, 2886, 2818, 2829, 2869,
       2882, 2883, 2886, 2933, 2819, 2857, 2874, 2884, 2920, 2939, 2948,
       2820, 2821, 2857, 2885, 2886, 2920, 2933, 2821, 2822, 2882, 2883,
       2885, 2886, 2933, 2824, 2827, 2858, 2887, 2890, 2921, 2949, 2825,
       2855, 2856, 2860, 2888, 2923, 2946, 2826, 2860, 2861, 2889, 2923,
       2924, 2950, 2984, 2827, 2866, 2887, 2890, 2929, 2949, 2951, 2828,
       2859, 2867, 2891, 2930, 2931, 2948, 2830, 2831, 2892, 2893, 2894,
       2952, 2953, 2830, 2832, 2892, 2893, 2895, 2952, 2954, 2831, 2837,
       2892, 2894, 2901, 2953, 2955, 2832, 2833, 2893, 2895, 2896, 2954,
       2956, 2833, 2870, 2895, 2896, 2934, 2956, 2957, 2834, 2872, 2897,
       2898, 2937, 2958, 2959, 2834, 2835, 2897, 2898, 2899, 2958, 2960,
       2835, 2873, 2898, 2899, 2938, 2960, 2961, 2836, 2851, 2873, 2900,
       2916, 2938, 2962, 2837, 2874, 2894, 2901, 2939, 2955, 2963, 2838,
       2839, 2902, 2903, 2904, 2964, 2965, 2838, 2842, 2902, 2903, 2907,
       2964, 2966, 2839, 2840, 2902, 2904, 2905, 2965, 2967, 2840, 2841,
       2904, 2905, 2906, 2967, 2968, 2841, 2843, 2905, 2906, 2908, 2968,
       2969, 2842, 2848, 2850, 2903, 2907, 2912, 2966, 2843, 2844, 2906,
       2908, 2909, 2969, 2970, 2844, 2875, 2908, 2909, 2940, 2970, 2971,
       2845, 2847, 2910, 2911, 2941, 2972, 2973, 2845, 2846, 2910, 2911,
       2912, 2973, 2974, 2846, 2848, 2907, 2911, 2912, 2966, 2974, 2847,
       2876, 2913, 2915, 2941, 2975, 2849, 2875, 2914, 2915, 2940, 2976,
       2977, 2849, 2876, 2913, 2914, 2915, 2975, 2977, 2851, 2852, 2900,
       2916, 2917, 2962, 2978, 2997, 2852, 2877, 2916, 2917, 2942, 2978,
       2979, 2853, 2854, 2881, 2918, 2919, 2947, 2980, 2854, 2855, 2918,
       2919, 2946, 2980, 2857, 2884, 2885, 2920, 2932, 2933, 2948, 2858,
       2887, 2921, 2922, 2949, 2981, 2982, 2858, 2859, 2921, 2922, 2930,
       2963, 2981, 2860, 2888, 2889, 2923, 2946, 2950, 2983, 2861, 2862,
       2889, 2924, 2925, 2984, 2985, 2862, 2863, 2924, 2925, 2926, 2985,
       2986, 2863, 2864, 2925, 2926, 2927, 2986, 2987, 3004, 2864, 2865,
       2926, 2927, 2928, 2987, 2988, 2865, 2866, 2927, 2928, 2929, 2988,
       2866, 2890, 2928, 2929, 2951, 2988, 2989, 3005, 2859, 2891, 2922,
       2930, 2939, 2948, 2963, 2867, 2868, 2891, 2931, 2932, 2948, 2868,
       2869, 2920, 2931, 2932, 2933, 2948, 2869, 2883, 2885, 2886, 2920,
       2932, 2933, 2870, 2896, 2934, 2935, 2957, 2990, 2991, 2870, 2871,
       2934, 2935, 2936, 2991, 2992, 2871, 2872, 2935, 2936, 2937, 2992,
       2993, 2872, 2897, 2936, 2937, 2959, 2993, 2994, 2873, 2899, 2900,
       2938, 2961, 2962, 2995, 2874, 2884, 2901, 2930, 2939, 2948, 2963,
       2875, 2909, 2914, 2940, 2971, 2976, 2847, 2910, 2913, 2941, 2972,
       2975, 2996, 2877, 2878, 2917, 2942, 2943, 2979, 2998, 2878, 2879,
       2942, 2943, 2944, 2998, 2999, 2879, 2880, 2943, 2944, 2945, 2999,
       3000, 2880, 2881, 2944, 2945, 2947, 3000, 3001, 2855, 2888, 2919,
       2923, 2946, 2980, 2983, 2881, 2918, 2945, 2947, 2980, 3001, 3002,
       2884, 2891, 2920, 2930, 2931, 2932, 2939, 2948, 2887, 2890, 2921,
       2949, 2951, 2982, 2889, 2923, 2950, 2983, 2984, 3003, 2890, 2929,
       2949, 2951, 2982, 2989, 3006, 2892, 2893, 2952, 2953, 2954, 3007,
       3008, 2892, 2894, 2952, 2953, 2955, 3007, 3009, 2893, 2895, 2952,
       2954, 2956, 3008, 3010, 2894, 2901, 2953, 2955, 2963, 2981, 3009,
       2895, 2896, 2954, 2956, 2957, 3010, 3011, 3040, 2896, 2934, 2956,
       2957, 2990, 3011, 3012, 2897, 2898, 2958, 2959, 2960, 3013, 3014,
       2897, 2937, 2958, 2959, 2994, 3013, 3015, 2898, 2899, 2958, 2960,
       2961, 3014, 3016, 2899, 2938, 2960, 2961, 2995, 3016, 3017, 2900,
       2916, 2938, 2962, 2995, 2997, 3018, 2901, 2922, 2930, 2939, 2955,
       2963, 2981, 2902, 2903, 2964, 2965, 2966, 3019, 3020, 2902, 2904,
       2964, 2965, 2967, 3019, 3021, 2903, 2907, 2912, 2964, 2966, 2974,
       3020, 2904, 2905, 2965, 2967, 2968, 3021, 3022, 2905, 2906, 2967,
       2968, 2969, 3022, 3023, 2906, 2908, 2968, 2969, 2970, 3023, 3024,
       2908, 2909, 2969, 2970, 2971, 3024, 3025, 2909, 2940, 2970, 2971,
       2976, 3025, 3026, 3048, 2910, 2941, 2972, 2973, 2996, 3027, 3028,
       2910, 2911, 2972, 2973, 2974, 3028, 3029, 2911, 2912, 2966, 2973,
       2974, 3020, 3029, 2913, 2915, 2941, 2975, 2977, 2996, 3030, 2914,
       2940, 2971, 2976, 2977, 3026, 3031, 2914, 2915, 2975, 2976, 2977,
       3030, 3031, 2916, 2917, 2978, 2979, 2997, 3032, 2917, 2942, 2978,
       2979, 2998, 3032, 3033, 2918, 2919, 2946, 2947, 2980, 2983, 3002,
       2921, 2922, 2955, 2963, 2981, 2982, 3009, 2921, 2949, 2951, 2981,
       2982, 3006, 3009, 2923, 2946, 2950, 2980, 2983, 3002, 3003, 2889,
       2924, 2950, 2984, 2985, 3003, 3034, 2924, 2925, 2984, 2985, 2986,
       3034, 3035, 2925, 2926, 2985, 2986, 3004, 3035, 2926, 2927, 2987,
       2988, 3004, 3036, 3037, 2927, 2928, 2929, 2987, 2988, 3005, 3037,
       2929, 2951, 2989, 3005, 3006, 3038, 3039, 2934, 2957, 2990, 2991,
       3012, 3041, 3042, 2934, 2935, 2990, 2991, 2992, 3041, 3043, 2935,
       2936, 2991, 2992, 2993, 3043, 3044, 2936, 2937, 2992, 2993, 2994,
       3044, 3045, 2937, 2959, 2993, 2994, 3015, 3045, 3046, 2938, 2961,
       2962, 2995, 3017, 3018, 3047, 2941, 2972, 2975, 2996, 3027, 3030,
       3049, 2916, 2962, 2978, 2997, 3018, 3032, 3050, 3051, 2942, 2943,
       2979, 2998, 2999, 3033, 3052, 2943, 2944, 2998, 2999, 3000, 3052,
       3053, 3085, 2944, 2945, 2999, 3000, 3001, 3053, 3054, 2945, 2947,
       3000, 3001, 3002, 3054, 3055, 2947, 2980, 2983, 3001, 3002, 3003,
       3055, 2950, 2983, 2984, 3002, 3003, 3034, 3055, 2926, 2986, 2987,
       3004, 3035, 3036, 3056, 2929, 2988, 2989, 3005, 3037, 3038, 3057,
       2951, 2982, 2989, 3006, 3007, 3009, 3039, 2952, 2953, 3006, 3007,
       3008, 3009, 3039, 2952, 2954, 3007, 3008, 3010, 3039, 3058, 2953,
       2955, 2981, 2982, 3006, 3007, 3009, 2954, 2956, 3008, 3010, 3040,
       3058, 3059, 2956, 2957, 3011, 3012, 3040, 3060, 3061, 2957, 2990,
       3011, 3012, 3042, 3060, 3062, 2958, 2959, 3013, 3014, 3015, 3063,
       3064, 2958, 2960, 3013, 3014, 3016, 3064, 3065, 2959, 2994, 3013,
       3015, 3046, 3063, 3066, 2960, 2961, 3014, 3016, 3017, 3065, 3067,
       2961, 2995, 3016, 3017, 3047, 3067, 3068, 2962, 2995, 2997, 3018,
       3047, 3050, 3069, 2964, 2965, 3019, 3020, 3021, 3070, 3071, 2964,
       2966, 2974, 3019, 3020, 3029, 3071, 2965, 2967, 3019, 3021, 3022,
       3070, 3072, 2967, 2968, 3021, 3022, 3023, 3072, 3073, 2968, 2969,
       3022, 3023, 3024, 3073, 3074, 2969, 2970, 3023, 3024, 3025, 3074,
       3075, 2970, 2971, 3024, 3025, 3048, 3075, 3076, 2971, 2976, 3026,
       3031, 3048, 3077, 3078, 2972, 2996, 3027, 3028, 3049, 3079, 3080,
       2972, 2973, 3027, 3028, 3029, 3080, 3081, 2973, 2974, 3020, 3028,
       3029, 3071, 3081, 2975, 2977, 2996, 3030, 3031, 3049, 3082, 2976,
       2977, 3026, 3030, 3031, 3077, 3082, 2978, 2979, 2997, 3032, 3033,
       3051, 3083, 2979, 2998, 3032, 3033, 3052, 3083, 3084, 2984, 2985,
       3003, 3034, 3035, 3055, 3086, 2985, 2986, 3004, 3034, 3035, 3056,
       3086, 2987, 3004, 3036, 3037, 3056, 3087, 3088, 2987, 2988, 3005,
       3036, 3037, 3057, 3087, 2989, 3005, 3038, 3039, 3057, 3058, 3089,
       2989, 3006, 3007, 3008, 3038, 3039, 3058, 2956, 3010, 3011, 3040,
       3059, 3061, 3090, 2990, 2991, 3041, 3042, 3043, 3091, 3092, 2990,
       3012, 3041, 3042, 3062, 3091, 3093, 2991, 2992, 3041, 3043, 3044,
       3092, 3094, 2992, 2993, 3043, 3044, 3045, 3094, 3095, 2993, 2994,
       3044, 3045, 3046, 3095, 3096, 2994, 3015, 3045, 3046, 3066, 3096,
       3097, 2995, 3017, 3018, 3047, 3068, 3069, 3098, 2971, 3025, 3026,
       3048, 3076, 3078, 3099, 2996, 3027, 3030, 3049, 3079, 3082, 3100,
       2997, 3018, 3050, 3051, 3060, 3061, 3069, 2997, 3032, 3050, 3051,
       3061, 3083, 3090, 2998, 2999, 3033, 3052, 3084, 3085, 3101, 2999,
       3000, 3053, 3054, 3085, 3102, 3000, 3001, 3053, 3054, 3055, 3086,
       3102, 3001, 3002, 3003, 3034, 3054, 3055, 3086, 3004, 3035, 3036,
       3056, 3086, 3088, 3102, 3005, 3037, 3038, 3057, 3087, 3089, 3103,
       3008, 3010, 3038, 3039, 3058, 3059, 3089, 3010, 3040, 3058, 3059,
       3089, 3090, 3104, 3011, 3012, 3050, 3060, 3061, 3062, 3069, 3011,
       3040, 3050, 3051, 3060, 3061, 3090, 3012, 3042, 3060, 3062, 3069,
       3093, 3098, 3013, 3015, 3063, 3064, 3066, 3105, 3106, 3013, 3014,
       3063, 3064, 3065, 3106, 3107, 3014, 3016, 3064, 3065, 3067, 3107,
       3108, 3015, 3046, 3063, 3066, 3097, 3105, 3109, 3016, 3017, 3065,
       3067, 3068, 3108, 3110, 3017, 3047, 3067, 3068, 3098, 3110, 3111,
       3018, 3047, 3050, 3060, 3062, 3069, 3098, 3019, 3021, 3070, 3071,
       3072, 3112, 3113, 3019, 3020, 3029, 3070, 3071, 3081, 3112, 3021,
       3022, 3070, 3072, 3073, 3113, 3114, 3022, 3023, 3072, 3073, 3074,
       3114, 3115, 3023, 3024, 3073, 3074, 3075, 3115, 3116, 3024, 3025,
       3074, 3075, 3076, 3116, 3117, 3025, 3048, 3075, 3076, 3099, 3117,
       3026, 3031, 3077, 3078, 3082, 3118, 3119, 3026, 3048, 3077, 3078,
       3099, 3118, 3120, 3027, 3049, 3079, 3080, 3100, 3121, 3122, 3027,
       3028, 3079, 3080, 3081, 3121, 3123, 3028, 3029, 3071, 3080, 3081,
       3112, 3123, 3030, 3031, 3049, 3077, 3082, 3100, 3119, 3032, 3033,
       3051, 3083, 3084, 3090, 3104, 3033, 3052, 3083, 3084, 3101, 3103,
       3104, 2999, 3052, 3053, 3085, 3088, 3101, 3102, 3034, 3035, 3054,
       3055, 3056, 3086, 3102, 3036, 3037, 3057, 3087, 3088, 3101, 3103,
       3036, 3056, 3085, 3087, 3088, 3101, 3102, 3038, 3057, 3058, 3059,
       3089, 3103, 3104, 3040, 3051, 3059, 3061, 3083, 3090, 3104, 3041,
       3042, 3091, 3092, 3093, 3124, 3125, 3041, 3043, 3091, 3092, 3094,
       3125, 3126, 3133, 3042, 3062, 3091, 3093, 3098, 3111, 3124, 3043,
       3044, 3092, 3094, 3095, 3126, 3127, 3044, 3045, 3094, 3095, 3096,
       3127, 3128, 3045, 3046, 3095, 3096, 3097, 3128, 3129, 3046, 3066,
       3096, 3097, 3109, 3129, 3130, 3047, 3062, 3068, 3069, 3093, 3098,
       3111, 3048, 3076, 3078, 3099, 3117, 3120, 3131, 3049, 3079, 3082,
       3100, 3119, 3122, 3132, 3052, 3084, 3085, 3087, 3088, 3101, 3103,
       3053, 3054, 3056, 3085, 3086, 3088, 3102, 3057, 3084, 3087, 3089,
       3101, 3103, 3104, 3059, 3083, 3084, 3089, 3090, 3103, 3104, 3063,
       3066, 3105, 3106, 3109, 3134, 3135, 3063, 3064, 3105, 3106, 3107,
       3135, 3136, 3064, 3065, 3106, 3107, 3108, 3136, 3137, 3065, 3067,
       3107, 3108, 3110, 3137, 3138, 3066, 3097, 3105, 3109, 3130, 3134,
       3139, 3067, 3068, 3108, 3110, 3111, 3138, 3140, 3068, 3093, 3098,
       3110, 3111, 3124, 3140, 3070, 3071, 3081, 3112, 3113, 3123, 3141,
       3070, 3072, 3112, 3113, 3114, 3141, 3142, 3072, 3073, 3113, 3114,
       3115, 3142, 3143, 3073, 3074, 3114, 3115, 3116, 3143, 3144, 3074,
       3075, 3115, 3116, 3117, 3144, 3145, 3075, 3076, 3099, 3116, 3117,
       3131, 3145, 3146, 3077, 3078, 3118, 3119, 3120, 3147, 3148, 3077,
       3082, 3100, 3118, 3119, 3132, 3147, 3078, 3099, 3118, 3120, 3131,
       3148, 3149, 3079, 3080, 3121, 3122, 3123, 3150, 3151, 3079, 3100,
       3121, 3122, 3132, 3150, 3152, 3080, 3081, 3112, 3121, 3123, 3141,
       3151, 3091, 3093, 3111, 3124, 3125, 3140, 3153, 3091, 3092, 3124,
       3125, 3133, 3153, 3154, 3092, 3094, 3126, 3127, 3133, 3155, 3156,
       3094, 3095, 3126, 3127, 3128, 3155, 3157, 3095, 3096, 3127, 3128,
       3129, 3157, 3158, 3096, 3097, 3128, 3129, 3130, 3158, 3159, 3097,
       3109, 3129, 3130, 3139, 3159, 3160, 3099, 3117, 3120, 3131, 3146,
       3149, 3161, 3100, 3119, 3122, 3132, 3147, 3152, 3162, 3092, 3125,
       3126, 3133, 3154, 3156, 3105, 3109, 3134, 3135, 3139, 3163, 3164,
       3105, 3106, 3134, 3135, 3136, 3164, 3165, 3106, 3107, 3135, 3136,
       3137, 3165, 3166, 3107, 3108, 3136, 3137, 3138, 3166, 3167, 3108,
       3110, 3137, 3138, 3140, 3167, 3168, 3109, 3130, 3134, 3139, 3160,
       3163, 3169, 3110, 3111, 3124, 3138, 3140, 3153, 3168, 3112, 3113,
       3123, 3141, 3142, 3151, 3170, 3113, 3114, 3141, 3142, 3143, 3170,
       3171, 3114, 3115, 3142, 3143, 3144, 3171, 3172, 3115, 3116, 3143,
       3144, 3145, 3172, 3173, 3116, 3117, 3144, 3145, 3146, 3173, 3174,
       3117, 3131, 3145, 3146, 3161, 3174, 3175, 3118, 3119, 3132, 3147,
       3148, 3162, 3176, 3118, 3120, 3147, 3148, 3149, 3176, 3177, 3120,
       3131, 3148, 3149, 3161, 3177, 3178, 3121, 3122, 3150, 3151, 3152,
       3179, 3180, 3121, 3123, 3141, 3150, 3151, 3170, 3180, 3122, 3132,
       3150, 3152, 3162, 3179, 3181, 3124, 3125, 3140, 3153, 3154, 3168,
       3182, 3125, 3133, 3153, 3154, 3156, 3182, 3183, 3126, 3127, 3155,
       3156, 3157, 3184, 3185, 3126, 3133, 3154, 3155, 3156, 3183, 3184,
       3127, 3128, 3155, 3157, 3158, 3185, 3186, 3128, 3129, 3157, 3158,
       3159, 3186, 3187, 3129, 3130, 3158, 3159, 3160, 3187, 3188, 3130,
       3139, 3159, 3160, 3169, 3188, 3189, 3131, 3146, 3149, 3161, 3175,
       3178, 3132, 3147, 3152, 3162, 3176, 3181, 3190, 3134, 3139, 3163,
       3164, 3169, 3191, 3192, 3134, 3135, 3163, 3164, 3165, 3191, 3193,
       3135, 3136, 3164, 3165, 3166, 3193, 3194, 3136, 3137, 3165, 3166,
       3167, 3194, 3195, 3137, 3138, 3166, 3167, 3168, 3195, 3196, 3138,
       3140, 3153, 3167, 3168, 3182, 3196, 3139, 3160, 3163, 3169, 3189,
       3192, 3197, 3141, 3142, 3151, 3170, 3171, 3180, 3198, 3142, 3143,
       3170, 3171, 3172, 3198, 3199, 3143, 3144, 3171, 3172, 3173, 3199,
       3200, 3144, 3145, 3172, 3173, 3174, 3200, 3201, 3145, 3146, 3173,
       3174, 3175, 3201, 3202, 3146, 3161, 3174, 3175, 3178, 3202, 3203,
       3147, 3148, 3162, 3176, 3177, 3190, 3204, 3148, 3149, 3176, 3177,
       3178, 3204, 3205, 3149, 3161, 3175, 3177, 3178, 3203, 3205, 3150,
       3152, 3179, 3180, 3181, 3206, 3207, 3150, 3151, 3170, 3179, 3180,
       3198, 3207, 3152, 3162, 3179, 3181, 3190, 3206, 3208, 3153, 3154,
       3168, 3182, 3183, 3196, 3209, 3154, 3156, 3182, 3183, 3184, 3209,
       3210, 3155, 3156, 3183, 3184, 3185, 3210, 3211, 3155, 3157, 3184,
       3185, 3186, 3211, 3212, 3157, 3158, 3185, 3186, 3187, 3212, 3213,
       3158, 3159, 3186, 3187, 3188, 3213, 3214, 3159, 3160, 3187, 3188,
       3189, 3214, 3215, 3160, 3169, 3188, 3189, 3197, 3215, 3216, 3162,
       3176, 3181, 3190, 3204, 3208, 3217, 3163, 3164, 3191, 3192, 3193,
       3218, 3219, 3163, 3169, 3191, 3192, 3197, 3218, 3220, 3164, 3165,
       3191, 3193, 3194, 3219, 3221, 3165, 3166, 3193, 3194, 3195, 3221,
       3222, 3166, 3167, 3194, 3195, 3196, 3222, 3223, 3167, 3168, 3182,
       3195, 3196, 3209, 3223, 3169, 3189, 3192, 3197, 3216, 3220, 3224,
       3170, 3171, 3180, 3198, 3199, 3207, 3225, 3171, 3172, 3198, 3199,
       3200, 3225, 3226, 3172, 3173, 3199, 3200, 3201, 3226, 3227, 3173,
       3174, 3200, 3201, 3202, 3227, 3228, 3174, 3175, 3201, 3202, 3203,
       3228, 3175, 3178, 3202, 3203, 3205, 3228, 3229, 3176, 3177, 3190,
       3204, 3205, 3217, 3230, 3177, 3178, 3203, 3204, 3205, 3229, 3230,
       3179, 3181, 3206, 3207, 3208, 3225, 3231, 3179, 3180, 3198, 3206,
       3207, 3225, 3181, 3190, 3206, 3208, 3217, 3231, 3182, 3183, 3196,
       3209, 3210, 3223, 3232, 3183, 3184, 3209, 3210, 3211, 3232, 3233,
       3184, 3185, 3210, 3211, 3212, 3233, 3234, 3185, 3186, 3211, 3212,
       3213, 3234, 3235, 3186, 3187, 3212, 3213, 3214, 3235, 3236, 3187,
       3188, 3213, 3214, 3215, 3236, 3237, 3188, 3189, 3214, 3215, 3216,
       3237, 3238, 3189, 3197, 3215, 3216, 3224, 3238, 3239, 3190, 3204,
       3208, 3217, 3230, 3231, 3240, 3191, 3192, 3218, 3219, 3220, 3241,
       3242, 3191, 3193, 3218, 3219, 3221, 3242, 3243, 3192, 3197, 3218,
       3220, 3224, 3241, 3244, 3193, 3194, 3219, 3221, 3222, 3243, 3245,
       3194, 3195, 3221, 3222, 3223, 3245, 3246, 3195, 3196, 3209, 3222,
       3223, 3232, 3246, 3197, 3216, 3220, 3224, 3239, 3244, 3247, 3198,
       3199, 3206, 3207, 3225, 3226, 3231, 3199, 3200, 3225, 3226, 3227,
       3231, 3240, 3200, 3201, 3226, 3227, 3228, 3240, 3248, 3201, 3202,
       3203, 3227, 3228, 3229, 3248, 3203, 3205, 3228, 3229, 3230, 3248,
       3204, 3205, 3217, 3229, 3230, 3240, 3248, 3206, 3208, 3217, 3225,
       3226, 3231, 3240, 3209, 3210, 3223, 3232, 3233, 3246, 3249, 3210,
       3211, 3232, 3233, 3234, 3249, 3250, 3211, 3212, 3233, 3234, 3235,
       3250, 3251, 3212, 3213, 3234, 3235, 3236, 3251, 3252, 3213, 3214,
       3235, 3236, 3237, 3252, 3253, 3214, 3215, 3236, 3237, 3238, 3253,
       3254, 3215, 3216, 3237, 3238, 3239, 3254, 3255, 3216, 3224, 3238,
       3239, 3247, 3255, 3256, 3217, 3226, 3227, 3230, 3231, 3240, 3248,
       3218, 3220, 3241, 3242, 3244, 3257, 3258, 3218, 3219, 3241, 3242,
       3243, 3258, 3259, 3219, 3221, 3242, 3243, 3245, 3259, 3260, 3220,
       3224, 3241, 3244, 3247, 3257, 3261, 3221, 3222, 3243, 3245, 3246,
       3260, 3262, 3222, 3223, 3232, 3245, 3246, 3249, 3262, 3224, 3239,
       3244, 3247, 3256, 3261, 3263, 3227, 3228, 3229, 3230, 3240, 3248,
       3232, 3233, 3246, 3249, 3250, 3262, 3264, 3233, 3234, 3249, 3250,
       3251, 3264, 3265, 3234, 3235, 3250, 3251, 3252, 3265, 3266, 3235,
       3236, 3251, 3252, 3253, 3266, 3267, 3236, 3237, 3252, 3253, 3254,
       3267, 3268, 3237, 3238, 3253, 3254, 3255, 3268, 3238, 3239, 3254,
       3255, 3256, 3268, 3269, 3239, 3247, 3255, 3256, 3263, 3269, 3270,
       3241, 3244, 3257, 3258, 3261, 3271, 3272, 3241, 3242, 3257, 3258,
       3259, 3271, 3273, 3242, 3243, 3258, 3259, 3260, 3273, 3274, 3243,
       3245, 3259, 3260, 3262, 3274, 3275, 3244, 3247, 3257, 3261, 3263,
       3272, 3276, 3245, 3246, 3249, 3260, 3262, 3264, 3275, 3247, 3256,
       3261, 3263, 3270, 3276, 3277, 3249, 3250, 3262, 3264, 3265, 3275,
       3278, 3250, 3251, 3264, 3265, 3266, 3278, 3279, 3251, 3252, 3265,
       3266, 3267, 3279, 3280, 3252, 3253, 3266, 3267, 3268, 3280, 3281,
       3253, 3254, 3255, 3267, 3268, 3269, 3281, 3255, 3256, 3268, 3269,
       3270, 3281, 3282, 3256, 3263, 3269, 3270, 3277, 3282, 3257, 3258,
       3271, 3272, 3273, 3283, 3284, 3257, 3261, 3271, 3272, 3276, 3283,
       3285, 3258, 3259, 3271, 3273, 3274, 3284, 3286, 3259, 3260, 3273,
       3274, 3275, 3286, 3287, 3260, 3262, 3264, 3274, 3275, 3278, 3287,
       3261, 3263, 3272, 3276, 3277, 3285, 3288, 3263, 3270, 3276, 3277,
       3282, 3288, 3289, 3264, 3265, 3275, 3278, 3279, 3287, 3290, 3265,
       3266, 3278, 3279, 3280, 3290, 3291, 3266, 3267, 3279, 3280, 3281,
       3291, 3292, 3267, 3268, 3269, 3280, 3281, 3282, 3292, 3269, 3270,
       3277, 3281, 3282, 3289, 3292, 3271, 3272, 3283, 3284, 3285, 3293,
       3294, 3271, 3273, 3283, 3284, 3286, 3293, 3295, 3272, 3276, 3283,
       3285, 3288, 3294, 3296, 3273, 3274, 3284, 3286, 3287, 3295, 3297,
       3274, 3275, 3278, 3286, 3287, 3290, 3297, 3276, 3277, 3285, 3288,
       3289, 3296, 3277, 3282, 3288, 3289, 3292, 3296, 3298, 3278, 3279,
       3287, 3290, 3291, 3297, 3299, 3279, 3280, 3290, 3291, 3292, 3298,
       3299, 3280, 3281, 3282, 3289, 3291, 3292, 3298, 3283, 3284, 3293,
       3294, 3295, 3300, 3283, 3285, 3293, 3294, 3296, 3300, 3284, 3286,
       3293, 3295, 3297, 3299, 3300, 3285, 3288, 3289, 3294, 3296, 3298,
       3300, 3286, 3287, 3290, 3295, 3297, 3299, 3289, 3291, 3292, 3296,
       3298, 3299, 3300, 3290, 3291, 3295, 3297, 3298, 3299, 3300, 3293,
       3294, 3295, 3296, 3298, 3299, 3300], dtype=np.int32)
        indptr = np.array([    0,     7,    13,    21,    28,    35,    42,    49,    56,
          63,    70,    77,    84,    91,    98,   105,   112,   119,
         126,   133,   140,   147,   154,   161,   168,   175,   182,
         189,   196,   203,   210,   217,   224,   231,   238,   245,
         252,   259,   266,   273,   280,   287,   294,   301,   308,
         315,   322,   329,   336,   343,   350,   357,   364,   371,
         378,   385,   392,   400,   407,   414,   421,   428,   435,
         441,   448,   456,   464,   470,   477,   484,   491,   498,
         505,   512,   519,   526,   533,   540,   547,   554,   561,
         568,   575,   582,   589,   596,   602,   609,   616,   623,
         630,   638,   645,   652,   659,   665,   672,   679,   686,
         693,   700,   707,   714,   721,   728,   735,   742,   749,
         756,   762,   769,   776,   783,   790,   797,   804,   811,
         818,   825,   832,   839,   846,   853,   860,   867,   874,
         881,   888,   895,   902,   909,   916,   923,   931,   938,
         945,   952,   959,   966,   973,   980,   987,   994,  1001,
        1008,  1015,  1022,  1029,  1035,  1043,  1050,  1057,  1064,
        1071,  1078,  1085,  1092,  1099,  1106,  1113,  1120,  1127,
        1134,  1141,  1148,  1155,  1162,  1169,  1176,  1183,  1190,
        1197,  1204,  1211,  1218,  1225,  1232,  1239,  1246,  1253,
        1260,  1267,  1274,  1281,  1288,  1295,  1302,  1309,  1316,
        1323,  1330,  1337,  1344,  1351,  1358,  1365,  1372,  1379,
        1386,  1393,  1400,  1407,  1414,  1421,  1428,  1435,  1442,
        1449,  1456,  1463,  1470,  1477,  1484,  1491,  1498,  1505,
        1512,  1519,  1526,  1533,  1540,  1547,  1554,  1561,  1568,
        1575,  1582,  1589,  1596,  1603,  1610,  1617,  1624,  1631,
        1638,  1645,  1652,  1659,  1666,  1673,  1680,  1687,  1694,
        1701,  1709,  1716,  1723,  1729,  1736,  1743,  1750,  1757,
        1764,  1771,  1778,  1785,  1792,  1799,  1806,  1813,  1820,
        1827,  1834,  1841,  1848,  1855,  1862,  1869,  1876,  1883,
        1890,  1897,  1904,  1911,  1918,  1925,  1932,  1939,  1946,
        1953,  1960,  1967,  1974,  1981,  1988,  1995,  2002,  2009,
        2016,  2023,  2030,  2037,  2044,  2051,  2058,  2065,  2072,
        2079,  2085,  2092,  2099,  2106,  2112,  2119,  2126,  2133,
        2140,  2147,  2154,  2161,  2168,  2175,  2182,  2189,  2196,
        2203,  2210,  2217,  2224,  2231,  2238,  2245,  2252,  2259,
        2266,  2273,  2281,  2288,  2295,  2302,  2309,  2316,  2323,
        2330,  2336,  2343,  2350,  2357,  2364,  2371,  2378,  2385,
        2393,  2400,  2407,  2414,  2421,  2428,  2434,  2441,  2449,
        2456,  2463,  2470,  2477,  2484,  2491,  2498,  2505,  2512,
        2519,  2526,  2533,  2540,  2547,  2554,  2561,  2568,  2575,
        2582,  2589,  2596,  2603,  2610,  2617,  2624,  2631,  2638,
        2645,  2652,  2659,  2666,  2673,  2680,  2687,  2694,  2701,
        2708,  2715,  2722,  2729,  2736,  2743,  2750,  2757,  2764,
        2771,  2778,  2785,  2792,  2799,  2806,  2813,  2820,  2827,
        2834,  2841,  2848,  2855,  2862,  2869,  2876,  2883,  2890,
        2897,  2904,  2911,  2918,  2925,  2932,  2939,  2945,  2951,
        2958,  2965,  2972,  2979,  2986,  2993,  3000,  3007,  3014,
        3021,  3027,  3034,  3041,  3048,  3055,  3062,  3069,  3076,
        3083,  3090,  3097,  3103,  3111,  3117,  3124,  3131,  3138,
        3145,  3152,  3159,  3166,  3173,  3180,  3187,  3194,  3201,
        3208,  3215,  3222,  3229,  3236,  3243,  3250,  3257,  3264,
        3270,  3278,  3285,  3293,  3300,  3307,  3314,  3321,  3328,
        3335,  3342,  3349,  3356,  3363,  3370,  3377,  3384,  3391,
        3398,  3406,  3413,  3420,  3427,  3434,  3441,  3448,  3455,
        3462,  3469,  3476,  3483,  3490,  3497,  3503,  3511,  3518,
        3525,  3532,  3539,  3546,  3553,  3560,  3567,  3574,  3581,
        3588,  3595,  3602,  3609,  3617,  3624,  3631,  3638,  3645,
        3652,  3659,  3666,  3673,  3680,  3687,  3694,  3701,  3708,
        3715,  3722,  3729,  3736,  3743,  3750,  3757,  3764,  3771,
        3778,  3785,  3792,  3799,  3806,  3813,  3820,  3827,  3834,
        3841,  3848,  3855,  3862,  3869,  3876,  3882,  3889,  3896,
        3903,  3910,  3917,  3924,  3931,  3938,  3945,  3952,  3959,
        3966,  3973,  3980,  3987,  3994,  4001,  4008,  4015,  4022,
        4029,  4036,  4043,  4050,  4057,  4064,  4071,  4078,  4085,
        4092,  4099,  4106,  4112,  4119,  4126,  4133,  4140,  4147,
        4154,  4161,  4168,  4175,  4182,  4190,  4197,  4204,  4211,
        4219,  4226,  4233,  4240,  4247,  4254,  4261,  4268,  4275,
        4282,  4289,  4296,  4303,  4310,  4317,  4324,  4331,  4338,
        4345,  4352,  4359,  4366,  4373,  4380,  4387,  4395,  4402,
        4409,  4416,  4423,  4430,  4437,  4444,  4451,  4458,  4466,
        4473,  4480,  4486,  4493,  4500,  4507,  4514,  4521,  4528,
        4535,  4542,  4549,  4556,  4563,  4570,  4577,  4584,  4591,
        4598,  4605,  4612,  4619,  4626,  4634,  4641,  4648,  4655,
        4662,  4669,  4676,  4683,  4690,  4697,  4704,  4711,  4718,
        4725,  4732,  4739,  4746,  4754,  4760,  4767,  4774,  4781,
        4788,  4795,  4802,  4809,  4816,  4823,  4830,  4837,  4844,
        4851,  4858,  4865,  4872,  4878,  4885,  4892,  4899,  4906,
        4913,  4920,  4927,  4933,  4939,  4946,  4953,  4960,  4967,
        4974,  4981,  4988,  4995,  5002,  5009,  5016,  5023,  5030,
        5037,  5044,  5051,  5058,  5065,  5072,  5079,  5086,  5093,
        5100,  5107,  5114,  5120,  5127,  5134,  5141,  5148,  5155,
        5162,  5169,  5176,  5183,  5190,  5198,  5205,  5212,  5219,
        5226,  5233,  5240,  5247,  5254,  5261,  5268,  5275,  5282,
        5289,  5296,  5303,  5310,  5317,  5324,  5331,  5338,  5345,
        5352,  5359,  5366,  5373,  5380,  5387,  5394,  5401,  5408,
        5415,  5422,  5430,  5438,  5445,  5452,  5460,  5466,  5473,
        5480,  5487,  5494,  5501,  5508,  5515,  5522,  5529,  5536,
        5543,  5550,  5557,  5564,  5571,  5578,  5585,  5592,  5599,
        5606,  5613,  5620,  5627,  5634,  5641,  5648,  5655,  5662,
        5669,  5676,  5683,  5690,  5697,  5704,  5711,  5718,  5725,
        5732,  5739,  5746,  5753,  5760,  5767,  5774,  5781,  5788,
        5795,  5802,  5809,  5816,  5823,  5830,  5837,  5844,  5851,
        5858,  5865,  5872,  5879,  5886,  5893,  5900,  5907,  5914,
        5921,  5928,  5935,  5942,  5949,  5956,  5963,  5970,  5977,
        5984,  5990,  5997,  6004,  6011,  6018,  6025,  6032,  6038,
        6045,  6052,  6059,  6066,  6073,  6080,  6087,  6094,  6101,
        6108,  6115,  6122,  6129,  6135,  6142,  6149,  6156,  6163,
        6170,  6177,  6184,  6191,  6198,  6205,  6212,  6219,  6226,
        6233,  6240,  6247,  6254,  6261,  6268,  6275,  6282,  6289,
        6296,  6303,  6310,  6317,  6324,  6331,  6338,  6345,  6352,
        6359,  6366,  6373,  6380,  6387,  6394,  6401,  6408,  6415,
        6422,  6429,  6436,  6443,  6450,  6457,  6464,  6471,  6478,
        6484,  6492,  6499,  6506,  6513,  6520,  6527,  6534,  6541,
        6548,  6555,  6561,  6568,  6575,  6582,  6589,  6596,  6603,
        6610,  6617,  6624,  6631,  6639,  6645,  6652,  6659,  6666,
        6673,  6679,  6686,  6693,  6700,  6707,  6714,  6722,  6729,
        6736,  6743,  6750,  6757,  6764,  6771,  6778,  6785,  6792,
        6799,  6806,  6813,  6821,  6828,  6835,  6842,  6849,  6856,
        6863,  6870,  6877,  6884,  6891,  6898,  6905,  6912,  6919,
        6926,  6933,  6940,  6947,  6954,  6961,  6968,  6975,  6982,
        6989,  6996,  7003,  7010,  7017,  7023,  7031,  7038,  7045,
        7052,  7059,  7066,  7074,  7081,  7088,  7094,  7101,  7108,
        7115,  7122,  7128,  7135,  7142,  7149,  7156,  7163,  7170,
        7177,  7184,  7191,  7198,  7205,  7212,  7218,  7226,  7233,
        7240,  7247,  7254,  7261,  7268,  7275,  7282,  7289,  7296,
        7303,  7310,  7317,  7323,  7330,  7337,  7344,  7351,  7358,
        7365,  7372,  7379,  7386,  7393,  7400,  7407,  7414,  7421,
        7428,  7435,  7442,  7449,  7456,  7463,  7470,  7477,  7484,
        7491,  7498,  7505,  7512,  7519,  7526,  7533,  7540,  7547,
        7554,  7561,  7568,  7575,  7582,  7589,  7596,  7603,  7610,
        7618,  7625,  7633,  7640,  7647,  7654,  7661,  7668,  7675,
        7682,  7689,  7696,  7703,  7711,  7718,  7725,  7732,  7739,
        7746,  7753,  7760,  7767,  7774,  7781,  7788,  7795,  7802,
        7809,  7816,  7823,  7830,  7837,  7844,  7851,  7858,  7865,
        7872,  7879,  7886,  7893,  7900,  7907,  7914,  7921,  7928,
        7935,  7942,  7949,  7956,  7963,  7970,  7977,  7984,  7991,
        7998,  8005,  8012,  8019,  8026,  8033,  8040,  8046,  8054,
        8061,  8068,  8075,  8082,  8089,  8096,  8103,  8110,  8117,
        8124,  8131,  8138,  8145,  8152,  8159,  8166,  8173,  8181,
        8187,  8194,  8201,  8207,  8214,  8221,  8227,  8234,  8241,
        8248,  8255,  8262,  8269,  8276,  8283,  8290,  8298,  8305,
        8312,  8319,  8326,  8333,  8340,  8347,  8354,  8361,  8367,
        8374,  8381,  8388,  8395,  8402,  8409,  8416,  8423,  8430,
        8437,  8444,  8451,  8458,  8465,  8471,  8478,  8485,  8492,
        8499,  8506,  8513,  8520,  8527,  8534,  8541,  8548,  8555,
        8562,  8569,  8576,  8583,  8589,  8596,  8603,  8611,  8618,
        8626,  8633,  8640,  8647,  8654,  8661,  8668,  8675,  8682,
        8689,  8696,  8703,  8710,  8717,  8724,  8731,  8738,  8745,
        8752,  8759,  8766,  8772,  8779,  8786,  8793,  8800,  8807,
        8814,  8822,  8828,  8836,  8843,  8850,  8857,  8864,  8871,
        8878,  8885,  8892,  8899,  8906,  8913,  8920,  8927,  8934,
        8941,  8948,  8955,  8962,  8969,  8976,  8983,  8990,  8997,
        9004,  9011,  9018,  9025,  9032,  9040,  9047,  9054,  9061,
        9068,  9075,  9082,  9089,  9096,  9103,  9110,  9117,  9124,
        9131,  9138,  9145,  9152,  9159,  9166,  9173,  9180,  9187,
        9194,  9201,  9208,  9215,  9222,  9229,  9236,  9243,  9250,
        9257,  9264,  9271,  9278,  9285,  9292,  9299,  9306,  9313,
        9320,  9327,  9334,  9341,  9348,  9355,  9363,  9370,  9377,
        9384,  9391,  9398,  9405,  9411,  9418,  9425,  9432,  9439,
        9446,  9453,  9460,  9467,  9474,  9481,  9487,  9494,  9501,
        9508,  9515,  9522,  9529,  9536,  9543,  9550,  9557,  9564,
        9571,  9578,  9586,  9593,  9600,  9607,  9614,  9621,  9628,
        9635,  9642,  9649,  9656,  9663,  9670,  9677,  9684,  9691,
        9698,  9704,  9712,  9719,  9726,  9733,  9740,  9747,  9754,
        9761,  9768,  9775,  9782,  9789,  9797,  9804,  9811,  9818,
        9825,  9832,  9839,  9846,  9853,  9860,  9867,  9874,  9881,
        9888,  9895,  9902,  9909,  9916,  9923,  9930,  9937,  9944,
        9951,  9958,  9965,  9972,  9979,  9986,  9993, 10001, 10008,
       10015, 10022, 10029, 10036, 10043, 10050, 10057, 10064, 10071,
       10078, 10085, 10092, 10098, 10105, 10112, 10119, 10126, 10133,
       10140, 10147, 10154, 10161, 10168, 10175, 10182, 10189, 10196,
       10203, 10210, 10217, 10224, 10231, 10238, 10245, 10252, 10259,
       10266, 10273, 10280, 10287, 10295, 10302, 10309, 10316, 10323,
       10330, 10337, 10344, 10351, 10358, 10365, 10372, 10379, 10386,
       10392, 10399, 10406, 10413, 10420, 10427, 10434, 10441, 10448,
       10455, 10462, 10469, 10476, 10483, 10490, 10497, 10504, 10511,
       10518, 10525, 10532, 10539, 10546, 10553, 10561, 10568, 10575,
       10582, 10589, 10596, 10603, 10610, 10617, 10624, 10631, 10638,
       10645, 10652, 10659, 10666, 10673, 10680, 10687, 10694, 10701,
       10708, 10715, 10722, 10729, 10736, 10743, 10750, 10757, 10764,
       10771, 10778, 10785, 10792, 10799, 10806, 10812, 10820, 10829,
       10836, 10843, 10849, 10856, 10863, 10870, 10877, 10885, 10892,
       10899, 10906, 10913, 10920, 10926, 10933, 10940, 10947, 10954,
       10961, 10968, 10975, 10982, 10989, 10996, 11003, 11010, 11017,
       11024, 11031, 11038, 11045, 11052, 11059, 11066, 11073, 11080,
       11087, 11094, 11101, 11108, 11115, 11122, 11129, 11136, 11144,
       11151, 11158, 11165, 11172, 11179, 11186, 11193, 11200, 11206,
       11214, 11220, 11227, 11235, 11242, 11249, 11256, 11263, 11270,
       11277, 11283, 11290, 11297, 11304, 11311, 11318, 11325, 11332,
       11339, 11346, 11353, 11360, 11367, 11374, 11381, 11388, 11395,
       11402, 11409, 11416, 11423, 11430, 11437, 11444, 11451, 11458,
       11465, 11472, 11478, 11485, 11492, 11499, 11506, 11513, 11520,
       11527, 11534, 11541, 11548, 11555, 11562, 11569, 11576, 11583,
       11590, 11597, 11604, 11611, 11618, 11625, 11632, 11639, 11646,
       11653, 11660, 11667, 11674, 11681, 11688, 11695, 11702, 11709,
       11716, 11723, 11730, 11738, 11745, 11752, 11759, 11766, 11773,
       11780, 11787, 11793, 11800, 11807, 11814, 11821, 11828, 11835,
       11842, 11849, 11856, 11863, 11870, 11877, 11884, 11891, 11898,
       11905, 11912, 11919, 11926, 11933, 11940, 11947, 11953, 11960,
       11967, 11974, 11981, 11987, 11994, 12001, 12008, 12015, 12022,
       12029, 12036, 12043, 12050, 12057, 12064, 12071, 12078, 12085,
       12092, 12099, 12107, 12114, 12121, 12128, 12135, 12142, 12149,
       12156, 12163, 12170, 12177, 12184, 12191, 12198, 12205, 12212,
       12219, 12226, 12233, 12241, 12248, 12255, 12262, 12269, 12276,
       12283, 12290, 12297, 12304, 12311, 12318, 12325, 12332, 12339,
       12346, 12353, 12359, 12366, 12373, 12380, 12387, 12394, 12401,
       12408, 12414, 12421, 12428, 12435, 12442, 12449, 12456, 12463,
       12470, 12477, 12484, 12491, 12499, 12507, 12513, 12520, 12526,
       12533, 12540, 12547, 12554, 12561, 12568, 12575, 12582, 12589,
       12596, 12603, 12610, 12618, 12625, 12632, 12639, 12646, 12653,
       12660, 12667, 12674, 12681, 12688, 12695, 12702, 12709, 12716,
       12723, 12730, 12737, 12744, 12751, 12758, 12765, 12772, 12779,
       12786, 12793, 12799, 12806, 12813, 12820, 12827, 12834, 12841,
       12848, 12855, 12862, 12869, 12874, 12881, 12888, 12895, 12902,
       12909, 12916, 12923, 12930, 12937, 12944, 12951, 12958, 12965,
       12972, 12980, 12987, 12994, 13001, 13008, 13015, 13022, 13030,
       13037, 13044, 13051, 13058, 13065, 13072, 13079, 13086, 13093,
       13100, 13107, 13114, 13121, 13128, 13135, 13142, 13148, 13155,
       13162, 13169, 13176, 13183, 13190, 13197, 13204, 13211, 13218,
       13225, 13232, 13239, 13246, 13253, 13260, 13267, 13274, 13281,
       13288, 13295, 13302, 13309, 13316, 13323, 13330, 13337, 13344,
       13351, 13358, 13365, 13372, 13379, 13386, 13393, 13400, 13407,
       13414, 13421, 13428, 13435, 13442, 13449, 13455, 13462, 13469,
       13476, 13483, 13490, 13497, 13504, 13511, 13518, 13525, 13532,
       13539, 13546, 13553, 13560, 13567, 13574, 13581, 13588, 13595,
       13602, 13609, 13617, 13624, 13631, 13638, 13645, 13652, 13659,
       13666, 13673, 13680, 13687, 13694, 13701, 13708, 13715, 13722,
       13729, 13736, 13743, 13750, 13757, 13764, 13771, 13778, 13785,
       13792, 13798, 13805, 13811, 13818, 13825, 13832, 13838, 13845,
       13852, 13859, 13866, 13873, 13880, 13887, 13894, 13901, 13908,
       13915, 13922, 13929, 13936, 13943, 13950, 13957, 13964, 13971,
       13978, 13985, 13992, 13999, 14006, 14013, 14020, 14027, 14034,
       14041, 14048, 14055, 14062, 14069, 14076, 14083, 14090, 14097,
       14104, 14111, 14118, 14125, 14133, 14140, 14147, 14154, 14161,
       14168, 14175, 14182, 14189, 14196, 14203, 14211, 14218, 14224,
       14231, 14238, 14245, 14252, 14259, 14266, 14273, 14280, 14287,
       14294, 14301, 14308, 14315, 14322, 14329, 14336, 14343, 14350,
       14356, 14363, 14370, 14377, 14384, 14391, 14398, 14405, 14413,
       14420, 14427, 14434, 14441, 14448, 14455, 14462, 14469, 14476,
       14483, 14490, 14497, 14504, 14511, 14518, 14525, 14532, 14539,
       14546, 14553, 14560, 14567, 14573, 14580, 14587, 14594, 14601,
       14609, 14616, 14623, 14631, 14638, 14645, 14652, 14659, 14666,
       14673, 14680, 14687, 14694, 14701, 14708, 14714, 14721, 14728,
       14735, 14742, 14749, 14756, 14763, 14770, 14777, 14784, 14791,
       14799, 14806, 14813, 14820, 14827, 14834, 14841, 14848, 14855,
       14862, 14869, 14876, 14883, 14890, 14897, 14904, 14911, 14918,
       14924, 14931, 14938, 14945, 14952, 14959, 14966, 14972, 14979,
       14986, 14993, 15000, 15007, 15014, 15021, 15028, 15035, 15042,
       15049, 15056, 15062, 15069, 15076, 15083, 15090, 15097, 15104,
       15111, 15118, 15125, 15132, 15139, 15146, 15153, 15160, 15167,
       15175, 15182, 15189, 15196, 15203, 15210, 15218, 15225, 15232,
       15239, 15246, 15253, 15260, 15267, 15274, 15281, 15288, 15295,
       15302, 15309, 15316, 15323, 15330, 15337, 15344, 15351, 15358,
       15365, 15372, 15379, 15386, 15393, 15400, 15407, 15414, 15422,
       15429, 15436, 15443, 15450, 15457, 15464, 15472, 15479, 15486,
       15493, 15500, 15507, 15514, 15521, 15528, 15535, 15542, 15549,
       15557, 15563, 15570, 15577, 15584, 15591, 15600, 15607, 15614,
       15621, 15628, 15635, 15642, 15650, 15657, 15664, 15671, 15678,
       15685, 15692, 15699, 15706, 15713, 15719, 15726, 15733, 15740,
       15747, 15754, 15762, 15769, 15776, 15783, 15790, 15797, 15804,
       15811, 15818, 15825, 15832, 15839, 15846, 15853, 15859, 15866,
       15873, 15880, 15887, 15894, 15901, 15908, 15915, 15922, 15929,
       15936, 15943, 15950, 15957, 15964, 15971, 15978, 15985, 15992,
       15999, 16006, 16013, 16020, 16027, 16034, 16040, 16047, 16054,
       16061, 16068, 16075, 16082, 16089, 16096, 16103, 16110, 16117,
       16124, 16131, 16138, 16145, 16151, 16157, 16164, 16171, 16178,
       16185, 16192, 16199, 16206, 16213, 16220, 16227, 16234, 16241,
       16248, 16255, 16262, 16269, 16276, 16283, 16290, 16297, 16304,
       16311, 16318, 16324, 16331, 16337, 16344, 16351, 16358, 16365,
       16372, 16379, 16386, 16394, 16401, 16408, 16415, 16422, 16429,
       16436, 16443, 16450, 16457, 16464, 16471, 16478, 16485, 16492,
       16499, 16506, 16513, 16520, 16527, 16534, 16541, 16548, 16555,
       16562, 16569, 16576, 16583, 16590, 16597, 16604, 16611, 16618,
       16625, 16632, 16639, 16646, 16653, 16660, 16668, 16675, 16683,
       16690, 16697, 16704, 16711, 16718, 16725, 16732, 16739, 16746,
       16753, 16760, 16767, 16774, 16781, 16788, 16795, 16802, 16809,
       16816, 16823, 16830, 16837, 16844, 16851, 16858, 16865, 16872,
       16879, 16886, 16893, 16900, 16908, 16915, 16923, 16930, 16937,
       16944, 16951, 16958, 16965, 16972, 16979, 16986, 16993, 16999,
       17006, 17013, 17020, 17027, 17034, 17041, 17048, 17055, 17062,
       17069, 17076, 17083, 17090, 17097, 17104, 17111, 17118, 17126,
       17132, 17139, 17146, 17153, 17160, 17167, 17174, 17181, 17187,
       17194, 17201, 17208, 17215, 17222, 17229, 17236, 17243, 17250,
       17257, 17264, 17271, 17278, 17285, 17292, 17299, 17306, 17313,
       17320, 17327, 17334, 17341, 17348, 17355, 17362, 17369, 17376,
       17383, 17390, 17397, 17404, 17411, 17418, 17425, 17432, 17439,
       17446, 17453, 17460, 17467, 17474, 17481, 17488, 17495, 17502,
       17509, 17516, 17523, 17530, 17537, 17544, 17551, 17558, 17565,
       17572, 17579, 17586, 17593, 17601, 17608, 17615, 17622, 17629,
       17636, 17643, 17650, 17657, 17664, 17671, 17678, 17685, 17692,
       17699, 17706, 17713, 17720, 17727, 17734, 17741, 17748, 17755,
       17762, 17769, 17776, 17783, 17790, 17797, 17804, 17811, 17817,
       17824, 17831, 17838, 17845, 17852, 17859, 17866, 17873, 17880,
       17888, 17895, 17902, 17909, 17916, 17923, 17930, 17937, 17944,
       17951, 17958, 17965, 17972, 17979, 17986, 17993, 18000, 18007,
       18015, 18022, 18029, 18036, 18043, 18050, 18057, 18064, 18071,
       18078, 18085, 18092, 18098, 18105, 18112, 18119, 18126, 18133,
       18140, 18147, 18154, 18161, 18168, 18175, 18182, 18189, 18196,
       18203, 18210, 18217, 18224, 18231, 18238, 18245, 18252, 18259,
       18266, 18273, 18280, 18287, 18294, 18301, 18308, 18315, 18322,
       18329, 18336, 18343, 18350, 18357, 18364, 18372, 18379, 18386,
       18393, 18400, 18407, 18414, 18421, 18428, 18435, 18443, 18450,
       18457, 18464, 18471, 18478, 18485, 18492, 18499, 18506, 18512,
       18519, 18526, 18533, 18540, 18547, 18554, 18561, 18568, 18575,
       18582, 18589, 18596, 18603, 18610, 18617, 18623, 18630, 18637,
       18644, 18651, 18657, 18664, 18671, 18678, 18685, 18692, 18699,
       18706, 18713, 18720, 18728, 18734, 18741, 18748, 18755, 18762,
       18769, 18776, 18783, 18790, 18797, 18804, 18811, 18818, 18825,
       18832, 18839, 18846, 18853, 18859, 18867, 18874, 18881, 18888,
       18895, 18902, 18909, 18916, 18923, 18930, 18937, 18944, 18951,
       18958, 18966, 18973, 18980, 18987, 18994, 19001, 19007, 19014,
       19021, 19028, 19035, 19042, 19049, 19056, 19064, 19071, 19078,
       19085, 19092, 19099, 19106, 19113, 19120, 19127, 19134, 19142,
       19149, 19156, 19163, 19170, 19177, 19184, 19191, 19198, 19205,
       19212, 19219, 19226, 19233, 19240, 19247, 19255, 19262, 19269,
       19276, 19283, 19290, 19297, 19304, 19311, 19318, 19325, 19332,
       19339, 19346, 19353, 19360, 19368, 19375, 19382, 19389, 19396,
       19403, 19410, 19417, 19424, 19431, 19438, 19445, 19452, 19460,
       19467, 19474, 19481, 19488, 19495, 19502, 19509, 19516, 19523,
       19530, 19537, 19544, 19551, 19558, 19565, 19572, 19579, 19586,
       19593, 19600, 19606, 19613, 19620, 19627, 19634, 19641, 19648,
       19655, 19662, 19669, 19676, 19683, 19690, 19697, 19704, 19711,
       19718, 19725, 19732, 19739, 19745, 19752, 19759, 19766, 19773,
       19780, 19787, 19794, 19801, 19808, 19815, 19822, 19829, 19837,
       19844, 19851, 19858, 19865, 19872, 19879, 19886, 19893, 19900,
       19907, 19914, 19921, 19928, 19936, 19943, 19950, 19957, 19964,
       19971, 19978, 19985, 19993, 20000, 20007, 20014, 20022, 20029,
       20036, 20043, 20050, 20057, 20064, 20071, 20078, 20085, 20092,
       20099, 20106, 20113, 20120, 20127, 20134, 20140, 20147, 20154,
       20161, 20168, 20175, 20182, 20189, 20196, 20203, 20210, 20217,
       20224, 20232, 20239, 20246, 20253, 20260, 20267, 20274, 20281,
       20288, 20295, 20302, 20309, 20316, 20323, 20330, 20337, 20344,
       20351, 20358, 20365, 20372, 20379, 20386, 20393, 20399, 20406,
       20413, 20421, 20428, 20435, 20441, 20448, 20455, 20462, 20469,
       20476, 20483, 20491, 20498, 20504, 20512, 20519, 20525, 20532,
       20539, 20546, 20553, 20560, 20567, 20574, 20581, 20587, 20594,
       20601, 20608, 20615, 20622, 20629, 20636, 20644, 20650, 20656,
       20663, 20670, 20677, 20684, 20691, 20699, 20706, 20713, 20720,
       20727, 20734, 20741, 20748, 20755, 20762, 20769, 20776, 20783,
       20790, 20797, 20805, 20812, 20819, 20826, 20833, 20840, 20847,
       20853, 20860, 20867, 20874, 20881, 20888, 20895, 20902, 20908,
       20915, 20922, 20929, 20936, 20943, 20950, 20957, 20964, 20971,
       20978, 20986, 20993, 21001, 21008, 21015, 21022, 21029, 21036,
       21043, 21050, 21057, 21064, 21071, 21078, 21085, 21092, 21099,
       21106, 21113, 21120, 21127, 21134, 21141, 21148, 21155, 21162,
       21169, 21176, 21183, 21190, 21197, 21204, 21211, 21218, 21225,
       21232, 21239, 21246, 21253, 21260, 21267, 21274, 21281, 21288,
       21295, 21302, 21309, 21316, 21323, 21330, 21337, 21344, 21351,
       21358, 21365, 21372, 21378, 21385, 21392, 21399, 21406, 21413,
       21420, 21427, 21434, 21441, 21448, 21455, 21462, 21469, 21476,
       21483, 21490, 21497, 21504, 21511, 21518, 21525, 21532, 21538,
       21545, 21552, 21559, 21566, 21573, 21580, 21587, 21594, 21601,
       21608, 21615, 21622, 21629, 21636, 21643, 21651, 21658, 21665,
       21672, 21679, 21686, 21693, 21700, 21707, 21714, 21721, 21728,
       21735, 21742, 21749, 21756, 21763, 21770, 21777, 21784, 21791,
       21798, 21805, 21812, 21819, 21827, 21834, 21841, 21848, 21855,
       21862, 21869, 21876, 21883, 21890, 21897, 21904, 21911, 21918,
       21925, 21932, 21938, 21945, 21952, 21959, 21966, 21973, 21980,
       21987, 21994, 22001, 22008, 22015, 22022, 22029, 22036, 22043,
       22050, 22057, 22064, 22071, 22078, 22085, 22092, 22099, 22106,
       22113, 22120, 22127, 22133, 22140, 22147, 22154, 22161, 22168,
       22175, 22182, 22189, 22196, 22203, 22210, 22217, 22224, 22231,
       22238, 22245, 22252, 22259, 22266, 22273, 22280, 22287, 22294,
       22301, 22308, 22315, 22322, 22329, 22336, 22343, 22350, 22357,
       22364, 22371, 22378, 22385, 22392, 22399, 22406, 22413, 22419,
       22426, 22433, 22440, 22447, 22453, 22459, 22466, 22473, 22480,
       22487, 22494, 22501, 22508, 22515, 22522, 22529, 22536, 22543,
       22550, 22557, 22564, 22571, 22578, 22585, 22592, 22599, 22605,
       22612, 22619, 22626, 22633, 22640, 22647, 22654, 22661, 22668,
       22675, 22682, 22689, 22696, 22703, 22710, 22717, 22724, 22731,
       22737, 22744, 22751, 22758, 22765, 22772, 22778, 22785, 22792,
       22799, 22806, 22813, 22820, 22827, 22834, 22841, 22848, 22855,
       22862, 22869, 22876, 22883, 22889, 22896, 22903, 22910, 22917,
       22924, 22931, 22938, 22945, 22952, 22959, 22966, 22973, 22980,
       22987, 22994, 23001, 23008, 23014, 23021, 23028, 23035, 23042,
       23048, 23054, 23061, 23068, 23074, 23081, 23088, 23095],
      dtype=np.int32)
        L_gt = sp.sparse.csr_matrix((data, indices, indptr))

        self.assertTrue(np.isclose(sp.sparse.linalg.norm(L-L_gt) / L.nnz, 0.))
        


        


if __name__ == '__main__':
    unittest.main()