from sail import cli, util

import subprocess, time
import click, pathlib
import re, shlex

def _get_extend_filters(paths, prefix=None):
	root = util.find_root()

	extend_filters = []
	if not paths:
		return []

	if prefix:
		prefixed_root = root.rstrip('/') + '/' + prefix.strip('/')

	for entry in paths:
		_entry = pathlib.Path(entry)

		# Requested root directory, remove all filters.
		if str(_entry.resolve()) == root:
			return []

		try:
			relative = _entry.resolve().relative_to(root)
		except ValueError:
			raise click.ClickException('Could not resolve path: %s' % entry)

		if prefix:
			try:
				relative = _entry.resolve().relative_to(prefixed_root)
			except ValueError:
				continue # The item is in root, but not in relative root, skip

			if str(relative) == '.':
				continue

		for parent in reversed(relative.parents):
			if str(parent) == '.':
				continue
			extend_filters.append('+ /%s' % parent)

		if _entry.is_dir():
			extend_filters.append('+ /%s/***' % relative)
		else:
			extend_filters.append('+ /%s' % relative)

	# Skip everything else.
	if len(extend_filters) > 0:
		extend_filters.append('- *')

	return extend_filters

@cli.command()
@click.argument('path', nargs=-1, required=False)
@click.option('--with-uploads', is_flag=True, help='Include the wp-content/uploads directory')
@click.option('--dry-run', is_flag=True, help='Show changes about to be deployed to production')
def deploy(with_uploads, dry_run, path):
	'''Deploy your working copy to production. If path is not specified then all application files are deployed.'''
	root = util.find_root()
	config = util.config()

	app_id = config['app_id']
	release = str(int(time.time()))

	if dry_run:
		click.echo('# Comparing files')

		destination = 'root@%s:/var/www/public/' % config['hostname']
		source = '%s/' % root
		files = _diff(source, destination, _get_extend_filters(path))
		empty = True

		colors = {'created': 'green', 'deleted': 'red', 'updated': 'yellow'}
		labels = {'created': 'New', 'deleted': 'Delete', 'updated': 'Update'}
		for op in ['created', 'updated', 'deleted']:
			for filename in files[op]:
				empty = False
				click.secho('- %s: %s' % (labels[op], filename), fg=colors[op])

		# TODO: Compare uploads if requested --with-uploads

		if empty:
			click.echo('- No changes')

		return

	click.echo('# Deploying to production')
	click.echo('- Preparing release directory')
	c = util.connection()

	c.run('mkdir -p /var/www/releases/%s' % release)
	c.run('rsync -rogtl /var/www/public/ /var/www/releases/%s' % release)

	click.echo('- Uploading application files to production')

	returncode, stdout, stderr = util.rsync(
		args=['-rtl', '--rsync-path', 'sudo -u www-data rsync',
			'--copy-dest', '/var/www/public/', '--delete'],
		source='%s/' % root,
		destination='root@%s:/var/www/releases/%s' % (config['hostname'], release),
		extend_filters=_get_extend_filters(path)
	)

	if returncode != 0:
		raise click.ClickException('An error occurred during upload. Please try again.')

	if with_uploads:
		click.echo('- Uploading wp-content/uploads')

		# Send uploads to production
		returncode, stdout, stderr = util.rsync(
			args=['-rtl', '--rsync-path', 'sudo -u www-data rsync', '--delete'],
			source='%s/wp-content/uploads/' % root,
			destination='root@%s:/var/www/uploads/' % config['hostname'],
			default_filters=False,
			extend_filters=_get_extend_filters(path, 'wp-content/uploads')
		)

		if returncode != 0:
			raise click.ClickException('An error occurred during upload. Please try again.')

	click.echo('- Deploying release: %s' % release)

	click.echo('- Updating symlinks')
	c.run('sudo -u www-data ln -sfn /var/www/uploads /var/www/releases/%s/wp-content/uploads' % release)
	c.run('sudo -u www-data ln -sfn /var/www/releases/%s /var/www/public' % release)

	click.echo('- Reloading services')
	c.run('docker exec sail nginx -s reload')
	c.run('docker exec sail kill -s USR2 $(docker exec sail cat /var/run/php/php7.4-fpm.pid)')

	releases = c.run('ls /var/www/releases')
	releases = re.findall('\d+', releases.stdout)
	releases = [int(i) for i in releases]

	keep = util.get_sail_default('keep')
	if not keep:
		keep = 5

	keep = int(keep)
	keep = max(keep, 2)
	keep = min(keep, 30)

	if len(releases) > keep:
		click.echo('- Removing outdated releases')
		remove = sorted(releases)[:len(releases)-keep]
		for key in remove:
			c.run(shlex.join(['rm', '-rf', '/var/www/releases/%s' % key]))

	click.echo('- Successfully deployed %s' % release)

@cli.command()
@click.argument('release', required=False, type=int, nargs=1)
@click.option('--releases', is_flag=True, help='Get a list of valid releases to rollback to')
def rollback(release=None, releases=False):
	'''Rollback production to a previous release'''
	sail = util.config()
	c = util.connection()

	if releases or not release:
		_releases = c.run('ls /var/www/releases')
		_releases = re.findall('\d+', _releases.stdout)

		if len(_releases) < 1:
			raise click.ClickException('Could not find any releases')

		try:
			_current = c.run('readlink /var/www/public').stdout.strip().split('/')[-1]
		except:
			_current = '0'

		click.echo('# Available releases:')
		for r in _releases:
			flags = '(current)' if r == _current else ''
			click.echo('- %s %s' % (r, flags))

		click.echo()
		click.echo('Rollback with: sail rollback <release>')
		return

	if release:
		release = str(release)

	click.echo('# Rolling back to %s' % release)

	_releases = c.run('ls /var/www/releases').stdout.strip().split('\n')
	if release not in _releases:
		raise click.ClickException('Invalid release. To get a list run: sail rollback --releases')

	click.echo('- Updating symlinks')
	c.run('ln -sfn /var/www/releases/%s /var/www/public' % release)

	click.echo('- Reloading services')
	c.run('docker exec sail nginx -s reload')
	c.run('docker exec sail kill -s USR2 $(docker exec sail cat /var/run/php/php7.4-fpm.pid)')

	click.echo('- Successfully rolled back to %s' % release)

@cli.command()
@click.argument('path', nargs=-1, required=False)
@click.option('--yes', '-y', is_flag=True, help='Force Y on overwriting local copy')
@click.option('--with-uploads', is_flag=True, help='Include the wp-content/uploads directory')
@click.option('--delete', is_flag=True, help='Delete files from local copy that do not exist on production')
@click.option('--dry-run', is_flag=True, help='Show changes about to be downloaded to the working copy')
def download(path, yes, with_uploads, delete, dry_run):
	'''Download files from production to your working copy'''
	root = util.find_root()
	config = util.config()

	if not yes and not dry_run:
		click.confirm('Downloading files from production may overwrite '
			+ 'your local copy. Continue?',
			abort=True
		)

	app_id = config['app_id']
	delete = ['--delete'] if delete else []

	if dry_run:
		click.echo('# Comparing files')

		source = 'root@%s:/var/www/public/' % config['hostname']
		destination = '%s/' % root
		files = _diff(source, destination, _get_extend_filters(path))
		empty = True

		colors = {'created': 'green', 'deleted': 'red', 'updated': 'yellow'}
		labels = {'created': 'New', 'deleted': 'Delete', 'updated': 'Update'}
		for op in ['created', 'updated', 'deleted']:
			for filename in files[op]:
				empty = False
				click.secho('- %s: %s' % (labels[op], filename), fg=colors[op])

		# TODO: Compare uploads if requested --with-uploads

		if empty:
			click.echo('- No changes')

		return

	click.echo('# Downloading application files from production')

	returncode, stdout, stderr = util.rsync(
		args=['-rtl'] + delete,
		source='root@%s:/var/www/public/' % config['hostname'],
		destination='%s/' % root,
		extend_filters=_get_extend_filters(path)
	)

	if returncode != 0:
		raise click.ClickException('An error occurred during download. Please try again.')

	if with_uploads:
		click.echo('- Downloading wp-content/uploads')

		# Download uploads from production
		returncode, stdout, stderr = util.rsync(
			args=['-rtl'] + delete,
			source='root@%s:/var/www/uploads/' % config['hostname'],
			destination='%s/wp-content/uploads/' % root,
			default_filters=False,
			extend_filters=_get_extend_filters(path, 'wp-content/uploads')
		)

		if returncode != 0:
			raise click.ClickException('An error occurred during download. Please try again.')

	click.echo('- Files download completed')

def _diff(source, destination, extend_filters=[]):
	'''Compare application files between environments'''
	root = util.find_root()

	if source == destination:
		raise click.ClickException('Can not compare apples to apples')

	args = ['-rlci', '--delete', '--dry-run']
	returncode, stdout, stderr = util.rsync(args, source, destination, extend_filters=extend_filters)

	if returncode != 0:
		raise click.ClickException('An error occurred in rsync. Please try again.')

	files = {
		'created': [],
		'deleted': [],
		'updated': [],
	}

	if util.debug():
		util.dlog(stdout)

	for line in stdout.splitlines():
		change = line[:11]
		path = line[12:]

		if len(change) < 11:
			continue

		# See --itemize-changes https://linux.die.net/man/1/rsync
		# Deleted
		if change.strip() == '*deleting':
			files['deleted'].append(path)
			continue

		# Newly created file.
		if change == '<f+++++++++' or change == '>f+++++++++':
			files['created'].append(path)
			continue

		# Newly created directory.
		if change == 'cd+++++++++':
			files['created'].append(path)
			continue

		# Checksum different.
		if change[:3] == '<fc' or change[:3] == '>fc':
			files['updated'].append(path)
			continue

		# Permission change
		if change[5] == 'p':
			files['updated'].append(path)
			continue

	return files
