"use strict";
const assert_1 = require("@aws-cdk/assert");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const config = require("../lib");
module.exports = {
    'access keys rotated'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new config.AccessKeysRotated(stack, 'AccessKeys');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Config::ConfigRule', {
            Source: {
                Owner: 'AWS',
                SourceIdentifier: 'ACCESS_KEYS_ROTATED',
            },
        }));
        test.done();
    },
    'cloudformation stack drift detection check'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new config.CloudFormationStackDriftDetectionCheck(stack, 'Drift');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Config::ConfigRule', {
            Source: {
                Owner: 'AWS',
                SourceIdentifier: 'CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK',
            },
            InputParameters: {
                cloudformationRoleArn: {
                    'Fn::GetAtt': [
                        'DriftRole8A5FB833',
                        'Arn',
                    ],
                },
            },
            Scope: {
                ComplianceResourceTypes: [
                    'AWS::CloudFormation::Stack',
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'config.amazonaws.com',
                        },
                    },
                ],
                Version: '2012-10-17',
            },
            ManagedPolicyArns: [
                {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/ReadOnlyAccess',
                        ],
                    ],
                },
            ],
        }));
        test.done();
    },
    'cloudformation stack notification check'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic1 = new sns.Topic(stack, 'AllowedTopic1');
        const topic2 = new sns.Topic(stack, 'AllowedTopic2');
        // WHEN
        new config.CloudFormationStackNotificationCheck(stack, 'Notification', {
            topics: [topic1, topic2],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Config::ConfigRule', {
            Source: {
                Owner: 'AWS',
                SourceIdentifier: 'CLOUDFORMATION_STACK_NOTIFICATION_CHECK',
            },
            InputParameters: {
                snsTopic1: {
                    Ref: 'AllowedTopic10C9144F9',
                },
                snsTopic2: {
                    Ref: 'AllowedTopic24ECF6C0D',
                },
            },
            Scope: {
                ComplianceResourceTypes: [
                    'AWS::CloudFormation::Stack',
                ],
            },
        }));
        test.done();
    },
    'cloudformation stack notification check throws with more than 5 topics'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'AllowedTopic1');
        // THEN
        test.throws(() => new config.CloudFormationStackNotificationCheck(stack, 'Notification', {
            topics: [topic, topic, topic, topic, topic, topic],
        }), /5 topics/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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