# Vizion

Vizion is a lightweight, notebook-friendly helper for quick exploratory data analysis (EDA).  
It bundles the repetitive parts of dataset inspection—summaries, missing data checks, visualizations,
outlier handling, and walkthrough tips—into a single class so you can focus on insights.

## Installation

```bash
pip install vizion
```

or, for local development:

```bash
pip install -e .
```

## Quick Start

```python
import pandas as pd
import vizion as vz

df = pd.read_csv("data.csv")

# Simple summary in one line
vz.quick_summary(df)

# Visual EDA helpers
vz.plot_numeric_eda(df, plot_types=["hist", "box"])
vz.plot_categorical_eda(df, target="SalePrice")

# Data hygiene helpers
df_clean, report = vz.handle_missing(df)
df_no_outliers = vz.handle_outliers(df_clean)

# Guided workflow
vz.help_steps()
```

## API Reference

All functions live under the `Vizion` class but are also re-exported at module level (`vizion.quick_summary`, etc.).

- `quick_summary(df, show=True)`  
  Prints (or returns) dataset shape, numeric/categorical/datetime column counts, duplicated rows, and top missing columns.

- `missing_value_summary(df)`  
  Returns a DataFrame detailing missing counts, percentages, dtypes, and suggested remediation per column.

- `get_column_types(df)`  
  Identifies numeric, categorical, and missing columns, storing them in globals (`num_col`, `cat_col`, `mis_col`) for quick reuse.

- `plot_numeric_eda(df, columns=None, plot_types=None, ..., show_corr=True, show_pairplot=False)`  
  Generates grid plots (hist, kde, box, violin, scatter, line, area) for numeric columns, with optional correlation heatmap and pairplot.

- `plot_categorical_eda(df, columns=None, plot_types=None, ..., top_n=20)`  
  Creates count/pie plots for categorical features, and supports target-aware bar/box/violin visuals.

- `handle_outliers(df, cols=None, method="auto", threshold=1.5, visualize=True, ...)`  
  Detects numeric outliers via IQR or IsolationForest, optionally caps/removes values and shows before/after boxplots.

- `handle_missing(df, drop_threshold=0.75, numeric_strategy="median", categorical_strategy="mode", datetime_strategy="ffill")`  
  Drops ultra-sparse columns and fills the rest according to strategy, returning the cleaned DataFrame plus a fill/drop report.

- `generate_doc(filename=None)`  
  Auto-builds Markdown docs for every Vizion method; prints to stdout or saves to a file.

- `help_steps()`  
  Prints a recommended, step-by-step EDA workflow that chains Vizion helpers in a sensible order.

## Contributing

Issues and PRs are welcome! See the open tasks in the GitHub repo or propose your own improvements (encoding, scaling, imbalance checks, etc.).

## License

MIT © Milind Chaudhari